"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ComputeEnvironment = exports.AllocationStrategy = exports.ComputeResourceType = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const batch_generated_1 = require("./batch.generated");
/**
 * Property to specify if the compute environment
 * uses On-Demand or SpotFleet compute resources.
 */
var ComputeResourceType;
(function (ComputeResourceType) {
    /**
     * Resources will be EC2 On-Demand resources.
     */
    ComputeResourceType["ON_DEMAND"] = "EC2";
    /**
     * Resources will be EC2 SpotFleet resources.
     */
    ComputeResourceType["SPOT"] = "SPOT";
})(ComputeResourceType = exports.ComputeResourceType || (exports.ComputeResourceType = {}));
/**
 * Properties for how to prepare compute resources
 * that are provisioned for a compute environment.
 */
var AllocationStrategy;
(function (AllocationStrategy) {
    /**
     * Batch will use the best fitting instance type will be used
     * when assigning a batch job in this compute environment.
     */
    AllocationStrategy["BEST_FIT"] = "BEST_FIT";
    /**
     * Batch will select additional instance types that are large enough to
     * meet the requirements of the jobs in the queue, with a preference for
     * instance types with a lower cost per unit vCPU.
     */
    AllocationStrategy["BEST_FIT_PROGRESSIVE"] = "BEST_FIT_PROGRESSIVE";
    /**
     * This is only available for Spot Instance compute resources and will select
     * additional instance types that are large enough to meet the requirements of
     * the jobs in the queue, with a preference for instance types that are less
     * likely to be interrupted.
     */
    AllocationStrategy["SPOT_CAPACITY_OPTIMIZED"] = "SPOT_CAPACITY_OPTIMIZED";
})(AllocationStrategy = exports.AllocationStrategy || (exports.AllocationStrategy = {}));
/**
 * Batch Compute Environment.
 *
 * Defines a batch compute environment to run batch jobs on.
 */
class ComputeEnvironment extends core_1.Resource {
    constructor(scope, id, props = { enabled: true, managed: true }) {
        super(scope, id, {
            physicalName: props.computeEnvironmentName,
        });
        this.validateProps(props);
        const spotFleetRole = this.getSpotFleetRole(props);
        let computeResources;
        // Only allow compute resources to be set when using MANAGED type
        if (props.computeResources && this.isManaged(props)) {
            computeResources = {
                allocationStrategy: props.computeResources.allocationStrategy
                    || (props.computeResources.type === ComputeResourceType.SPOT
                        ? AllocationStrategy.SPOT_CAPACITY_OPTIMIZED
                        : AllocationStrategy.BEST_FIT),
                bidPercentage: props.computeResources.bidPercentage,
                desiredvCpus: props.computeResources.desiredvCpus,
                ec2KeyPair: props.computeResources.ec2KeyPair,
                imageId: props.computeResources.image && props.computeResources.image.getImage(this).imageId,
                instanceRole: props.computeResources.instanceRole
                    ? props.computeResources.instanceRole
                    : new iam.CfnInstanceProfile(this, 'Instance-Profile', {
                        roles: [new iam.Role(this, 'Ecs-Instance-Role', {
                                assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
                                managedPolicies: [
                                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2ContainerServiceforEC2Role'),
                                ],
                            }).roleName],
                    }).attrArn,
                instanceTypes: this.buildInstanceTypes(props.computeResources.instanceTypes),
                launchTemplate: props.computeResources.launchTemplate,
                maxvCpus: props.computeResources.maxvCpus || 256,
                minvCpus: props.computeResources.minvCpus || 0,
                securityGroupIds: this.buildSecurityGroupIds(props.computeResources.vpc, props.computeResources.securityGroups),
                spotIamFleetRole: spotFleetRole ? spotFleetRole.roleArn : undefined,
                subnets: props.computeResources.vpc.selectSubnets(props.computeResources.vpcSubnets).subnetIds,
                tags: props.computeResources.computeResourcesTags,
                type: props.computeResources.type || ComputeResourceType.ON_DEMAND,
            };
        }
        const computeEnvironment = new batch_generated_1.CfnComputeEnvironment(this, 'Resource', {
            computeEnvironmentName: this.physicalName,
            computeResources,
            serviceRole: props.serviceRole
                ? props.serviceRole.roleArn
                : new iam.Role(this, 'Resource-Service-Instance-Role', {
                    managedPolicies: [
                        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSBatchServiceRole'),
                    ],
                    assumedBy: new iam.ServicePrincipal('batch.amazonaws.com'),
                }).roleArn,
            state: props.enabled === undefined ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            type: this.isManaged(props) ? 'MANAGED' : 'UNMANAGED',
        });
        if (props.computeResources && props.computeResources.vpc) {
            this.node.addDependency(props.computeResources.vpc);
        }
        this.computeEnvironmentArn = this.getResourceArnAttribute(computeEnvironment.ref, {
            service: 'batch',
            resource: 'compute-environment',
            resourceName: this.physicalName,
        });
        this.computeEnvironmentName = this.getResourceNameAttribute(computeEnvironment.ref);
    }
    /**
     * Fetches an existing batch compute environment by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param computeEnvironmentArn
     */
    static fromComputeEnvironmentArn(scope, id, computeEnvironmentArn) {
        const stack = core_1.Stack.of(scope);
        const computeEnvironmentName = stack.parseArn(computeEnvironmentArn).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.computeEnvironmentArn = computeEnvironmentArn;
                this.computeEnvironmentName = computeEnvironmentName;
            }
        }
        return new Import(scope, id);
    }
    isManaged(props) {
        return props.managed === undefined ? true : props.managed;
    }
    /**
     * Validates the properties provided for a new batch compute environment.
     */
    validateProps(props) {
        if (props === undefined) {
            return;
        }
        if (!this.isManaged(props) && props.computeResources !== undefined) {
            throw new Error('It is not allowed to set computeResources on an AWS unmanaged compute environment');
        }
        if (this.isManaged(props) && props.computeResources === undefined) {
            throw new Error('computeResources is missing but required on a managed compute environment');
        }
        // Setting a bid percentage is only allowed on SPOT resources +
        // Cannot use SPOT_CAPACITY_OPTIMIZED when using ON_DEMAND
        if (props.computeResources) {
            if (props.computeResources.type === ComputeResourceType.ON_DEMAND) {
                // VALIDATE FOR ON_DEMAND
                // Bid percentage is not allowed
                if (props.computeResources.bidPercentage !== undefined) {
                    throw new Error('Setting the bid percentage is only allowed for SPOT type resources on a batch compute environment');
                }
                // SPOT_CAPACITY_OPTIMIZED allocation is not allowed
                if (props.computeResources.allocationStrategy && props.computeResources.allocationStrategy === AllocationStrategy.SPOT_CAPACITY_OPTIMIZED) {
                    throw new Error('The SPOT_CAPACITY_OPTIMIZED allocation strategy is only allowed if the environment is a SPOT type compute environment');
                }
            }
            else {
                // VALIDATE FOR SPOT
                // Bid percentage must be from 0 - 100
                if (props.computeResources.bidPercentage !== undefined &&
                    (props.computeResources.bidPercentage < 0 || props.computeResources.bidPercentage > 100)) {
                    throw new Error('Bid percentage can only be a value between 0 and 100');
                }
            }
            if (props.computeResources.minvCpus) {
                // minvCpus cannot be less than 0
                if (props.computeResources.minvCpus < 0) {
                    throw new Error('Minimum vCpus for a batch compute environment cannot be less than 0');
                }
                // minvCpus cannot exceed max vCpus
                if (props.computeResources.maxvCpus &&
                    props.computeResources.minvCpus > props.computeResources.maxvCpus) {
                    throw new Error('Minimum vCpus cannot be greater than the maximum vCpus');
                }
            }
        }
    }
    buildInstanceTypes(instanceTypes) {
        if (instanceTypes === undefined) {
            return [
                'optimal',
            ];
        }
        return instanceTypes.map((type) => type.toString());
    }
    buildSecurityGroupIds(vpc, securityGroups) {
        if (securityGroups === undefined) {
            return [
                new ec2.SecurityGroup(this, 'Resource-Security-Group', { vpc }).securityGroupId,
            ];
        }
        return securityGroups.map((group) => group.securityGroupId);
    }
    /**
     * Generates an AWS IAM role for provisioning spotfleet resources
     * if the allocation strategy is set to BEST_FIT or not defined.
     *
     * @param props - the compute environment construct properties
     */
    getSpotFleetRole(props) {
        var _a;
        if (((_a = props.computeResources) === null || _a === void 0 ? void 0 : _a.allocationStrategy) && props.computeResources.allocationStrategy !== AllocationStrategy.BEST_FIT) {
            return undefined;
        }
        if (props.computeResources) {
            if (props.computeResources.spotFleetRole) {
                return props.computeResources.spotFleetRole;
            }
            else if (props.computeResources.type === ComputeResourceType.SPOT) {
                return iam.Role.fromRoleArn(this, 'Resource-SpotFleet-Role', `arn${core_1.Aws.PARTITION}iam::${this.stack.account}:role/aws-service-role/spotfleet.amazonaws.com/AWSServiceRoleForEC2SpotFleet`);
            }
        }
        return undefined;
    }
}
exports.ComputeEnvironment = ComputeEnvironment;
//# sourceMappingURL=data:application/json;base64,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