//----------------------------------------------------------------------
//   Copyright 2005-2022 SyoSil ApS
//   All Rights Reserved Worldwide
//
//   Licensed under the Apache License, Version 2.0 (the
//   "License"); you may not use this file except in
//   compliance with the License.  You may obtain a copy of
//   the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in
//   writing, software distributed under the License is
//   distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
//   CONDITIONS OF ANY KIND, either express or implied.  See
//   the License for the specific language governing
//   permissions and limitations under the License.
//----------------------------------------------------------------------
/// MD5 hash algorithm implementation. The class implements the hash API as defined
/// by the hash base class.
class cl_syoscb_hash_md5 extends cl_syoscb_hash_base#(pk_syoscb::MD5_HASH_DIGEST_WIDTH);
  //Used in the internal hashing functions
  typedef bit [31:0] M;
  //-------------------------------------
  // Non randomizable variables
  //-------------------------------------

  //-------------------------------------
  // UVM Macros
  //-------------------------------------
  `uvm_object_utils_begin(cl_syoscb_hash_md5)
  `uvm_object_utils_end

  //-------------------------------------
  // Constructor
  //-------------------------------------
  extern function new(string name = "cl_syoscb_hash_md5");

  //-------------------------------------
  // Hash API
  //-------------------------------------
  // Basic hash functions
  extern protected virtual function tp_hash_digest do_hash(bit ser []);

  // Support functions
  extern virtual local function bit [511:0] get_next_512_word(int unsigned word_idx,bit ser[]);
  extern virtual local function M [15:0] get_next_32_word(bit [511:0]word);
endclass: cl_syoscb_hash_md5

function cl_syoscb_hash_md5::new(string name = "cl_syoscb_hash_md5");
  cl_syoscb_md5_packer p;
  super.new(name);

  p = new;
  this.packer = p;
endfunction: new

/// <b>Hash API:</b> See cl_syoscb_hash_base#do_hash for more details
/// Expects a bitstream with a length which is a multiple of 512, which
/// follows the below format, specified in RFC 1321:
///  <table>
///  <tr>
///    <th>Bits</th>
///    <th>Information</th>
///  </tr>
///  <tr>
///    <td> 0 - length of the serialized item-1</td>
///    <td> Serialized item</td>
///  </tr>
///  <tr>
///    <td> length of the serialized item - lentgh of the serialized item+6</td>
///    <td> Zeros</td>
///  </tr>
///  <tr>
///    <td>length of the serialized item+7</td>
///    <td> One </td>
///  </tr>
///  <tr>
///    <td>length of the serialized item+8 - (512*x-64)</td>
///    <td> Zeros</td>
///  </tr>
///  <tr>
///    <td> last 64 bits</td>
///    <td> length of the item modulo 2^64</td>
///  </tr>
///  </table>
/// A bitstream of this kind can be generated by packing with cl_syoscb_md5_packer,
/// which automatically appends the required metadata to the end of the bitstream
function cl_syoscb_hash_md5::tp_hash_digest cl_syoscb_hash_md5::do_hash(bit ser[]);
 // implement md5 algorithm
  int unsigned s [64];
  int unsigned  K [64];
  int      i;
  int unsigned     a0;
  int unsigned     b0;
  int unsigned     c0;
  int unsigned     d0;
  bit [31:0] a0d;
  bit [31:0] b0d;
  bit [31:0] c0d;
  bit [31:0] d0d;
  int p;
  int q;
  int unsigned     A;
  int unsigned     B;
  int unsigned     C;
  int unsigned     D;
  bit      [63:0]length;
  int      lmsg;
  int      j;
  int unsigned      F;
  int      g;
  int      l;
  bit    [511:0] word;
  M [15:0] mWords;

   //s specifies the per-round shift amounts
  s[ 0:15] = { 7, 12, 17, 22,  7, 12, 17, 22,  7, 12, 17, 22,  7, 12, 17, 22 };
  s[16:31] = { 5,  9, 14, 20,  5,  9, 14, 20,  5,  9, 14, 20,  5,  9, 14, 20 };
  s[32:47] = { 4, 11, 16, 23,  4, 11, 16, 23,  4, 11, 16, 23,  4, 11, 16, 23 };
  s[48:63] = { 6, 10, 15, 21,  6, 10, 15, 21,  6, 10, 15, 21,  6, 10, 15, 21 };

  K[ 0: 3] = { 'hd76aa478, 'he8c7b756, 'h242070db, 'hc1bdceee };
  K[ 4: 7] = { 'hf57c0faf, 'h4787c62a, 'ha8304613, 'hfd469501 };
  K[ 8:11] = { 'h698098d8, 'h8b44f7af, 'hffff5bb1, 'h895cd7be };
  K[12:15] = { 'h6b901122, 'hfd987193, 'ha679438e, 'h49b40821 };
  K[16:19] = { 'hf61e2562, 'hc040b340, 'h265e5a51, 'he9b6c7aa };
  K[20:23] = { 'hd62f105d, 'h02441453, 'hd8a1e681, 'he7d3fbc8 };
  K[24:27] = { 'h21e1cde6, 'hc33707d6, 'hf4d50d87, 'h455a14ed };
  K[28:31] = { 'ha9e3e905, 'hfcefa3f8, 'h676f02d9, 'h8d2a4c8a };
  K[32:35] = { 'hfffa3942, 'h8771f681, 'h6d9d6122, 'hfde5380c };
  K[36:39] = { 'ha4beea44, 'h4bdecfa9, 'hf6bb4b60, 'hbebfbc70 };
  K[40:43] = { 'h289b7ec6, 'heaa127fa, 'hd4ef3085, 'h04881d05 };
  K[44:47] = { 'hd9d4d039, 'he6db99e5, 'h1fa27cf8, 'hc4ac5665 };
  K[48:51] = { 'hf4292244, 'h432aff97, 'hab9423a7, 'hfc93a039 };
  K[52:55] = { 'h655b59c3, 'h8f0ccc92, 'hffeff47d, 'h85845dd1 };
  K[56:59] = { 'h6fa87e4f, 'hfe2ce6e0, 'ha3014314, 'h4e0811a1 };
  K[60:63] = { 'hf7537e82, 'hbd3af235, 'h2ad7d2bb, 'heb86d391 };

  //Initialize variables:
  a0 = 'h67452301;
  b0 = 'hefcdab89;
  c0 = 'h98badcfe;
  d0 = 'h10325476;

  lmsg=ser.size();
  j=lmsg/512;

  for (i=0;i<j;i=i+1) begin
    word=this.get_next_512_word(i,ser);
    mWords=this.get_next_32_word(word);

    A=a0;
    B=b0;
    C=c0;
    D=d0;

    //Main loop
    for (l=0;l<64;l=l+1)begin
      if(0 <= l && l <= 15) begin
        F=D^(B&(C^D));
        g=l;

      end
      else if(16 <= l && l <= 31) begin
        F=C^(D&(B^C));
        g=(5*l+1)% (16);

      end
      else if(32 <= l && l<= 47) begin
        F = ((B^C)^D);
        g=(3*l+5)%(16);

      end
      else if(48 <= l && l<= 63) begin
        F = (C^ (B | ~D));
        g=(7*l)%(16);

      end

      F=F+A+K[l] + mWords[g];
      A=D;
      D=C;
      C=B;
      B=B+((F<<<s[l])|(F>>>(32-s[l])));

    end // for (l=0;l<64;l=l+1)

    a0=a0+A;
    b0=b0+B;
    c0=c0+C;
    d0=d0+D;

  end // for (i=0;i<j;i=i+1)
  for (p=0;p< 4;p=p+1) begin

    for (q=0;q<8;q=q+1)begin
      a0d[31-p*8-q]=a0[7+(8*p)-q];
      b0d[31-p*8-q]=b0[7+(8*p)-q];
      c0d[31-p*8-q]=c0[7+(8*p)-q];
      d0d[31-p*8-q]=d0[7+(8*p)-q];

    end

  end

  return {a0d,b0d,c0d,d0d};
endfunction : do_hash

// Returns the next 512 bits acording to word_idx of the serialized bits ser.
function bit [511:0] cl_syoscb_hash_md5::get_next_512_word(int unsigned word_idx,bit ser[]);
  int            k;
  bit [511:0]    out;

  for (k=0;k<512;k=k+1)begin
    out[k]=ser[k+512*word_idx];
  end

  return out;
endfunction : get_next_512_word

// Returns the 16 words of 32 bits in which the 512 bits array word is divided
function cl_syoscb_hash_md5::M [15:0] cl_syoscb_hash_md5::get_next_32_word(bit [511:0]word);

  int k;
  int l;
  bit [31:0] out;
  M [15:0] mWords;

  for (k=0;k<16;k=k+1) begin
    for (l=0;l<32;l=l+1)begin
      out[l]=word[k*32+l];
    end

    mWords[k]=out;
  end

  return mWords;
endfunction : get_next_32_word
