import numpy as np
from PIL import Image, ImageDraw
from moviepy import editor
from pylsd import lsd
import os

# get media file paths
pkg_path = os.path.dirname(os.path.realpath(__file__))
video_path = os.path.join(pkg_path, 'media/stick_bug.mp4')
audio_path = os.path.join(pkg_path, 'media/audio.wav')

# end segment points (using the 1280x720 video)
end_segments = np.array([
    [595, 212, 688, 262, 5],
    [688, 262, 791, 263, 5],
    [516, 266, 612, 223, 5],
    [450, 339, 516, 266, 5],
    [548, 351, 600, 228, 5],
    [599, 297, 652, 247, 5],
    [586, 353, 599, 297, 5],
    [644, 360, 678, 260, 5],
    [774, 370, 769, 264, 5],
])


def find_segments(im: Image, num_segments=9, sort_by_width=False) -> np.ndarray:
    """
    Find the largest line segments in an image and return an array of segment points.
    :param im: PIL Image object
    :param num_segments: Number of segments to return
    :param sort_by_width: If true, return segments with the largest width rather than distance
    :return: An array of line segment points [x1, y1, x2, y2, width]
    """
    # resize image to fit in video (1280x720) without cropping
    scale = min(1280 / im.width, 720 / im.height)
    im = im.resize((int(im.width * scale), int(im.height * scale)))
    x_offset = (1280 - im.width) // 2
    y_offset = (720 - im.height) // 2

    im_gray = np.array(im.convert('L'))

    segments = lsd(im_gray)

    # add offset to segment points since the image will be centered in the video
    segments[:, 0:3:2] += x_offset  # add x_offset to column 0 and 2
    segments[:, 1:4:2] += y_offset  # add Y_offset to column 1 and 3

    # sort by distance or width of segments
    if sort_by_width:
        segments = segments[segments[:, 4].argsort()[::-1]]
    else:
        # add a column to store distance
        rows, cols = segments.shape
        segments_d = np.empty((rows, cols + 1))
        segments_d[:, :-1] = segments

        # find length of each line segment
        for i in range(segments_d.shape[0]):
            x1, y1, x2, y2, *_ = segments_d[i]
            segments_d[i, 5] = np.sqrt((x2 - x1) ** 2 + (y2 - y1) ** 2)  # distance formula

        # sort and remove distance column
        segments = segments_d[segments_d[:, 5].argsort()[::-1]][:, :-1]

    return segments[:num_segments]


def interp_segments(segments: np.ndarray) -> np.ndarray:
    """
    Return the linearly interpolated points of each line segment for each frame.
    :param segments: array of segments returned by find_segments()
    :return: a 3-dimensional array [frame, segment, point]
    """
    num_frames = 52

    # array to store the points of each segment in each frame
    segment_frames = np.empty((segments.shape[0], num_frames, 5), float)

    for i in range(segments.shape[0]):
        # array to store the segment points of each frame (each index is a frame)
        segment_frames[i] = np.array([
            np.linspace(segments[i][0], end_segments[i][0], num_frames, dtype=int),
            np.linspace(segments[i][1], end_segments[i][1], num_frames, dtype=int),
            np.linspace(segments[i][2], end_segments[i][2], num_frames, dtype=int),
            np.linspace(segments[i][3], end_segments[i][3], num_frames, dtype=int),
            np.linspace(segments[i][4], end_segments[i][4], num_frames, dtype=int),
        ]).transpose()

    return np.transpose(segment_frames, (1, 0, 2))


def generate_clip(segment_frames: np.ndarray, src_im: Image, line_color=0xd3ffff, bg_color=0x77737d):
    """
    Generate the video with the following format:
    1. The source image
    2. Line segments start appearing
    3. Source image disappears and background changes
    4. Line segments interpolate to form a stick bug
    5. The stick bug video

    :param segment_frames: Array of segments in each frame generated by interp_segments()
    :param src_im: The source PIL Image
    :param line_color: RGB color for the line segments (either int or a tuple of ints)
    :param bg_color: RGB color for the background after the image disappears
    :return: MoviePy video clip
    """
    # list of clips in the video
    clips = []

    # first clip is the source image
    # center the image on a black background
    im = Image.new('RGB', (1280, 720))
    scale = min(im.width / src_im.width, im.height / src_im.height)
    src_im = src_im.resize((int(src_im.width * scale), int(src_im.height * scale)))
    im.paste(src_im, ((im.width - src_im.width) // 2, (im.height - src_im.height) // 2))
    clips.append(editor.ImageClip(np.array(im), duration=1))

    # line segments start appearing
    # draw a line, store the image clip, repeat for each line
    draw = ImageDraw.Draw(im)
    for segment in segment_frames[0]:
        draw.line(tuple(segment[:4]), fill=line_color, width=int(segment[4]))
        clips.append(editor.ImageClip(np.array(im), duration=0.33))

    # one more slightly longer clip after that
    clips.append(editor.ImageClip(np.array(im), duration=1))

    # redraw lines with gray background
    draw.rectangle([0, 0, 1280, 720], bg_color)
    for segment in segment_frames[0]:
        draw.line(tuple(segment[:4]), fill=line_color, width=int(segment[4]))
    clips.append(editor.ImageClip(np.array(im), duration=0.75))

    # use ImageSequenceClip for the line interpolation
    frames = []
    for i in range(segment_frames.shape[0]):
        draw.rectangle([0, 0, 1280, 720], bg_color)
        for segment in segment_frames[i]:
            draw.line(tuple(segment[:4]), fill=line_color, width=int(segment[4]))
        frames.append(np.asarray(im))
    clips.append(editor.ImageSequenceClip(frames, 30))

    # concatenate all of the clips and add the audio
    all_clips = editor.concatenate_videoclips(clips)
    all_clips = all_clips.set_audio(editor.AudioFileClip(audio_path))

    # add stick bug video to the clips and return video
    stick_bug_clip = editor.VideoFileClip(video_path).without_audio()
    final_clip = editor.concatenate_videoclips([all_clips, stick_bug_clip])
    return final_clip


def save_video(clip, filename: str):
    """
    Write a 30 fps moviepy video clip to filename.
    :param clip: a moviepy video clip
    :param filename: path to save the clip as
    """
    clip.write_videofile(filename, fps=30)


def generate_video_from_image_path(filename: str, line_color=(255, 255, 211), bg_color=(125, 115, 119)):
    """
    Create a 'get stick bugged' video from an image file path. This function completes the entire process, aside from
    saving the video.
    :param filename: file path of the image to use
    :param line_color: color of the line segments in the video
    :param bg_color: color of the background in the video after the image disappears
    :return: a moviepy video clip
    """
    im = Image.open(filename)  # open the image with PIL
    segments = find_segments(im)  # find 9 line segments in the image
    segment_frames = interp_segments(segments)  # find the linear interpolation of the line segment points
    return generate_clip(segment_frames, im, line_color, bg_color)  # create the video
