# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['bytechomp', 'bytechomp.datatypes']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'bytechomp',
    'version': '0.3.0',
    'description': 'A pure python, declarative binary protocol parser & generator using dataclasses and type hinting.',
    'long_description': '# bytechomp\n\n[![ci](https://github.com/AndrewSpittlemeister/bytechomp/actions/workflows/ci.yml/badge.svg?branch=main)](https://github.com/AndrewSpittlemeister/bytechomp/actions/workflows/ci.yml)\n[![PyPI Version](https://img.shields.io/pypi/v/bytechomp.svg)](https://pypi.org/project/bytechomp/)\n[![Python Versions](https://img.shields.io/pypi/pyversions/bytechomp.svg)](https://pypi.org/project/bytechomp/)\n![Lines of Code](https://tokei.rs/b1/github/AndrewSpittlemeister/bytechomp?category=code)\n\n> *A pure python, declarative custom binary protocol parser & generator using dataclasses and type hinting.*\n\n`bytechomp` leverages Python\'s type hinting system at runtime to build binary protocol parsing schemas from dataclass implementations. Deserialization/Serialization of the binary data is now abstracted away by `bytechomp`, leaving you to work in the land of typed and structured data.\n\n**Features:**\n- [x] Pure Python\n- [x] Zero Dependencies\n- [x] Uses native type-hinting & dataclasses\n- [x] Supports lower-precision numerics\n- [x] Supports `bytes` fields of known length\n- [x] Supports `list` types for repeated, continuous fields of known length\n- [x] Supports nested structures\n- [x] Supports serialization of populated data structures\n\n## Installation\n\n`bytechomp` is a small, pure python library with zero dependencies. It can be installed via PyPI:\n\n```\npip install bytechomp\n```\n\nor Git for the latest unreleased code:\n\n```\npip install https://github.com/AndrewSpittlemeister/bytechomp.git@main\n```\n\n## Reader API\n\nThe `Reader` class uses Python\'s built-in [generics](https://docs.python.org/3/library/typing.html#generics) determine the dataclass used when parsing. This dataclass is defined by the user to mimic the binary protocol. Once instantiated, the `Reader` class can be fed `bytes` and used to construct the dataclass when ready. There are various ways to accomplish this with the `Reader` class:\n\n```python\nfrom dataclasses import dataclass\n\nfrom bytechomp import Reader\n\n@dataclass\nclass MyStruct:\n    timestamp: float\n    identity: int\n\n# instantiate a reader\nreader = Reader[MyStruct]().allocate()\n\n# add data to the internal buffer\nreader.feed(stream.read(512))\n\n# check if enough data is present to build\nprint(reader.is_complete())\n\n# add via the bitshift method\nreader << stream.read(512)\n\n# check via bool magic method\nprint(bool(reader))\n\n# combine alternative methods\nif reader << stream.read(512):\n    # construct dataclass\n    my_struct = reader.build()\n\n# clear internal byte buffer\nreader.clear()\n\n# use the iterator API\nsimulated_byte_iterator = [b"a"] * 10\nfor my_struct in reader.iter(simulated_byte_iterator):\n    print(my_struct)\n```\n\n## Serialization API\nSimilar to the `Reader`, serialization of data is accomplished through defining dataclasses in the same manner.\n\n```python\nfrom bytechomp import serialize\n\nmy_struct = MyStruct(1.1, 15)\n\nserialized_struct: bytes = serialize(my_struct)\n```\n\n## Supported Type Fields\nFields on the dataclasses can be integers, floats, strings, bytes, lists, or other dataclasses. Python-native `int` and `float` represent 64-bit variants. Other sizes can be imported from `bytechomp`:\n\n```python\nfrom bytechomp.datatypes import (\n    U8,  # 8-bit unsigned integer\n    U16,  # 16-bit unsigned integer\n    U32,  # 32-bit unsigned integer\n    U64,  # 64-bit unsigned integer\n    I8,  # 8-bit signed integer\n    I16,  # 16-bit signed integer\n    I32,  # 32-bit signed integer\n    I64,  # 64-bit signed integer\n    F16,  # 16-bit float\n    F32,  # 32-bit float\n    F64,  # 64-bit float\n)\n```\n\nAlthough these allow a `Reader` to parse a field of a custom size, the resulting value populated in a dataclass field will always be the python-natives `int` or `float`.\n\nRepeated fields like `bytes` and `list` require the use of Python\'s `typing.Annotated` to allow defining a length.\n\n```python\nfrom bytechomp import Annotated, dataclass  # re-exported by bytechomp\n\n@dataclass\nclass Message:\n    name: Annotated[bytes, 10]\n    identity: Annotated[bytes, 10]\n    flags: Annotated[list[int], 5]\n```\n\nFinally, `list` fields can contain any other supported datatype, including other dataclass structures to handle complex, nested protocols.\n\n## Byte Ordering\nByte default the byte-ordering is set to the machine\'s native format, but can be changed:\n\n```python\nfrom bytechomp import Reader, ByteOrder, dataclass, serialize\n\n@dataclass\nclass MyStruct:\n    timestamp: float\n    identity: int\n\n# use native (the default)\nreader = Reader[MyStruct](ByteOrder.NATIVE).allocate()\ndata = serialize(MyStruct(1.1, 15), ByteOrder.NATIVE)\n\n# use little endian\nreader = Reader[MyStruct](ByteOrder.LITTLE).allocate()\ndata = serialize(MyStruct(1.1, 15), ByteOrder.LITTLE)\n\n# use big endian\nreader = Reader[MyStruct](ByteOrder.BIG).allocate()\ndata = serialize(MyStruct(1.1, 15), ByteOrder.BIG)\n```\n\n## A Longer Example\n\n```python\nfrom bytechomp import Reader, dataclass, Annotated, serialize\nfrom bytechomp.datatypes import U16, F32\n\n\n@dataclass\nclass Header:\n    timestamp: float  # native datatypes can be used when assuming full precision\n    message_count: int  # similarly with 64-bit integers\n    message_identity: U16  # custom datatypes are available and will be cast to native when deserialized\n\n\n@dataclass\nclass Body:\n    unique_id: Annotated[bytes, 5]  # use of typing.Annotated to denote length\n    balance: F32\n\n\n@dataclass\nclass Message:\n    header: Header  # nested data structures are allowed\n    body: Body\n\n\n@dataclass\nclass MessageBundle:\n    messages: Annotated[list[Message], 8]  # so are lists of data structures!\n\n\ndef main() -> None:\n    # build Reader object using the MessageBundle class as its generic argument\n    reader = Reader[MessageBundle]().allocate()\n\n    with open("my-binary-data-stream.dat", "rb") as fp:\n        while (data := fp.read(4)):\n            # feed stream into the reader\n            reader.feed(data)\n\n            # check if the structure has been saturated with enough data\n            if reader.is_complete():\n                # parse the stream and create your typed data structure!\n                msg_bundle = reader.build()\n                print(msg_bundle)\n\n                # re-serialize this data\n                print(f"serialized data: {serialize(msg_bundle)}")\n```\n\n## Other Examples\n- See [parse-sqlite-header.py](./examples/parse-sqlite-header.py) for an example of using `bytechomp` to read the header message of an sqlite file. A rough estimate of what this should result in can be found [here](https://docs.fileformat.com/database/sqlite/).\n- See [tcp-client-server.py](./examples/tcp-client-server.py) for an example of using `bytechomp` to serialize/deserialize binary messages across a TCP connection.\n\n\n## How does this work?\n\nWhile a binary stream is usually represented as a flat, continuous data, `bytechomp` can be used as a structural abstraction over this data. Therefore, if there was a message with the following structure for a message called `UserState`:\n\n| Field | Type | Description |\n| ----- | ---- | ----------- |\n| `user_id` | uint64 | user\'s unique identity |\n| `balance` | float32 | user\'s balance |\n\nThe resulting translation to a dataclass would be the following:\n\n```python\nfrom bytechomp import Reader, dataclass\nfrom bytechomp.datatypes import F32\n\n@dataclass\nclass UserState:\n    user_id: int\n    balance: F32\n```\n\nWhen parsing messages that contain other messages, you will need to be aware of how the embedded messages are contained and how the resulting memory layout will look for the container message as whole. Since the container message is still represented as one set of continuous bytes, nested classes in bytechomp are constructed using a depth first search of the contained fields in nested structures to build out a flattened parsing pattern for Python\'s `struct` module.\n\nConsider the following structures:\n\n```python\nfrom bytechomp import Reader, dataclass, Annotated  # using re-export from within bytechomp\nfrom bytechomp.datatypes import F32\n\n@dataclass\nclass UserState:\n    user_id: int\n    balance: F32\n\n@dataclass\nclass Transaction:\n    amount: F32\n    sender: int\n    receiver: int\n\n@dataclass\nclass User:\n    user_state: UserState\n    recent_transactions: Annotated[list[Transaction], 3]\n```\n\nThe `User` message would correspond to the following memory layout:\n\n```\nuint64, float32, float32, int64, int64, float32, int64, int64, float32, int64, int64\n```\n\n## Additional Notes\n\nThis package is based on a mostly undocumented feature in standard implementation of CPython. This is the ability to inspect the type information generic parameters via the `self.__orig_class__.__args__` structures. The information in this structure is only populated after initialization (hence the need for the `allocate()` method when instantiated a `Reader` object). Should this behavior change in future versions of Python, `bytechomp` will adapt accordingly. For now, it will stay away from passing a type object as a argument to initialization because that just seems hacky.\n\n**Future Improvements:**\n- Perhaps allowing for parameterized fields to reference previously declared fields (i.e. allowing a list of size `n` where `n` is the previous field)\n- Allow declaring value restraints on fields\n    - Making use of the `typing.Literal` python class\n- Allow for enums to be generated for integer fields\n',
    'author': 'Andrew Spittlemeister',
    'author_email': 'andrew.spittlemeister@pm.me',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/AndrewSpittlemeister/bytechomp',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.10,<3.11',
}


setup(**setup_kwargs)
