"""
OsonWeb Templates - Jinja2 shablonizator
"""
from pathlib import Path
from typing import Dict, Any, Optional
from fastapi.templating import Jinja2Templates
from fastapi.responses import HTMLResponse
from starlette.requests import Request


# Global shablonizator
_templates: Optional[Jinja2Templates] = None


def init_templates(templates_dir: Path):
    """Shablonizatorni sozlash"""
    global _templates
    _templates = Jinja2Templates(directory=str(templates_dir))


def get_templates() -> Jinja2Templates:
    """Shablonizatorni olish"""
    global _templates
    
    if _templates is None:
        # Avtomatik templates papkasini topish
        templates_dir = Path.cwd() / "templates"
        if not templates_dir.exists():
            templates_dir.mkdir(parents=True, exist_ok=True)
        init_templates(templates_dir)
    
    return _templates


def render_template(
    template_name: str,
    request: Optional[Request] = None,
    **context: Any
) -> HTMLResponse:
    """
    Shablonni render qilish (Flask uslubida)
    
    Misol:
        @app.get("/")
        async def index(request: Request):
            return render_template("index.html", request=request, title="Bosh sahifa")
    """
    templates = get_templates()
    
    # Agar request berilmagan bo'lsa, bo'sh context yaratish
    if request is None:
        # FastAPI uchun request majburiy, shuning uchun xatolik
        raise ValueError("request parametri majburiy!")
    
    return templates.TemplateResponse(
        template_name,
        {"request": request, **context}
    )


def render_string(template_string: str, **context: Any) -> str:
    """
    Matn shablonini render qilish
    
    Misol:
        html = render_string("<h1>{{ title }}</h1>", title="Salom")
    """
    from jinja2 import Template
    template = Template(template_string)
    return template.render(**context)


__all__ = [
    "init_templates",
    "get_templates",
    "render_template",
    "render_string",
]
