"""
OsonWeb Core - Asosiy freymvork klassi
"""
from fastapi import FastAPI, Request as FastAPIRequest, Response as FastAPIResponse
from fastapi.staticfiles import StaticFiles
from fastapi.templating import Jinja2Templates
from pathlib import Path
from typing import Optional


class OsonWeb(FastAPI):
    """
    OsonWeb asosiy klassi - FastAPI asosida qurilgan
    Flask uslubida sodda va tushunarli API
    """
    
    def __init__(
        self,
        name: str,
        templates_dir: Optional[Path] = None,
        static_dir: Optional[Path] = None,
        **kwargs
    ):
        super().__init__(title=name, **kwargs)
        
        self.name = name
        self._templates_dir = templates_dir
        self._static_dir = static_dir
        self._templates = None
        
        # Agar templates va static papkalari ko'rsatilgan bo'lsa, sozlash
        if templates_dir and templates_dir.exists():
            self._templates = Jinja2Templates(directory=str(templates_dir))
        
        if static_dir and static_dir.exists():
            self.mount("/static", StaticFiles(directory=str(static_dir)), name="static")
    
    @property
    def templates(self):
        """Jinja2 shablonizator"""
        if self._templates is None:
            # Avtomatik templates papkasini topish
            templates_dir = Path.cwd() / "templates"
            if templates_dir.exists():
                self._templates = Jinja2Templates(directory=str(templates_dir))
        return self._templates
    
    def route(self, path: str, methods: list = None, **kwargs):
        """
        Flask uslubida marshrutni ro'yxatdan o'tkazish
        
        Misol:
            @app.route("/users", methods=["GET", "POST"])
            async def users():
                return {"users": []}
        """
        if methods is None:
            methods = ["GET"]
        
        def decorator(func):
            for method in methods:
                method = method.upper()
                if method == "GET":
                    self.get(path, **kwargs)(func)
                elif method == "POST":
                    self.post(path, **kwargs)(func)
                elif method == "PUT":
                    self.put(path, **kwargs)(func)
                elif method == "DELETE":
                    self.delete(path, **kwargs)(func)
                elif method == "PATCH":
                    self.patch(path, **kwargs)(func)
            return func
        
        return decorator


# FastAPI Request va Response ni eksport qilish
Request = FastAPIRequest
Response = FastAPIResponse
