"""
OsonWeb Auth Utils - Autentifikatsiya yordamchi funksiyalari
"""
from datetime import datetime, timedelta
from typing import Optional
from fastapi import Depends, HTTPException, status
from fastapi.security import OAuth2PasswordBearer
from jose import JWTError, jwt
from passlib.context import CryptContext
from .models import User


# Parol shifrlash
pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")

# OAuth2 sxemasi
oauth2_scheme = OAuth2PasswordBearer(tokenUrl="token")

# JWT sozlamalari
SECRET_KEY = "osonweb-secret-key-change-this-in-production"
ALGORITHM = "HS256"
ACCESS_TOKEN_EXPIRE_MINUTES = 30


def verify_password(plain_password: str, hashed_password: str) -> bool:
    """Parolni tekshirish"""
    return pwd_context.verify(plain_password, hashed_password)


def get_password_hash(password: str) -> str:
    """Parolni shifrlash"""
    return pwd_context.hash(password)


def create_access_token(data: dict, expires_delta: Optional[timedelta] = None) -> str:
    """
    JWT access token yaratish
    
    Args:
        data: Token ichiga qo'shiladigan ma'lumotlar
        expires_delta: Token amal qilish muddati
    
    Returns:
        JWT token
    """
    to_encode = data.copy()
    
    if expires_delta:
        expire = datetime.utcnow() + expires_delta
    else:
        expire = datetime.utcnow() + timedelta(minutes=ACCESS_TOKEN_EXPIRE_MINUTES)
    
    to_encode.update({"exp": expire})
    encoded_jwt = jwt.encode(to_encode, SECRET_KEY, algorithm=ALGORITHM)
    
    return encoded_jwt


def verify_token(token: str) -> Optional[dict]:
    """
    JWT tokenni tekshirish
    
    Args:
        token: JWT token
    
    Returns:
        Token ichidagi ma'lumotlar yoki None
    """
    try:
        payload = jwt.decode(token, SECRET_KEY, algorithms=[ALGORITHM])
        return payload
    except JWTError:
        return None


async def get_current_user(token: str = Depends(oauth2_scheme)) -> User:
    """
    Joriy foydalanuvchini olish (JWT token orqali)
    
    Args:
        token: JWT token
    
    Returns:
        User obyekti
    
    Raises:
        HTTPException: Agar token noto'g'ri yoki foydalanuvchi topilmasa
    """
    credentials_exception = HTTPException(
        status_code=status.HTTP_401_UNAUTHORIZED,
        detail="Autentifikatsiya ma'lumotlari noto'g'ri",
        headers={"WWW-Authenticate": "Bearer"},
    )
    
    payload = verify_token(token)
    if payload is None:
        raise credentials_exception
    
    username: str = payload.get("sub")
    if username is None:
        raise credentials_exception
    
    user = await User.filter(username=username).first()
    if user is None:
        raise credentials_exception
    
    return user


async def get_current_active_user(current_user: User = Depends(get_current_user)) -> User:
    """
    Faol foydalanuvchini olish
    
    Args:
        current_user: Joriy foydalanuvchi
    
    Returns:
        User obyekti
    
    Raises:
        HTTPException: Agar foydalanuvchi faol bo'lmasa
    """
    if not current_user.is_active:
        raise HTTPException(status_code=400, detail="Foydalanuvchi faol emas")
    
    return current_user


async def authenticate_user(username: str, password: str) -> Optional[User]:
    """
    Foydalanuvchini autentifikatsiya qilish
    
    Args:
        username: Foydalanuvchi nomi
        password: Parol
    
    Returns:
        User obyekti yoki None
    """
    user = await User.filter(username=username).first()
    if not user:
        return None
    
    if not await user.check_password(password):
        return None
    
    return user


__all__ = [
    "verify_password",
    "get_password_hash",
    "create_access_token",
    "verify_token",
    "get_current_user",
    "get_current_active_user",
    "authenticate_user",
]
