"""
OsonWeb Auth - Foydalanuvchi autentifikatsiya modellari
"""
from tortoise import fields
from tortoise.models import Model
from datetime import datetime


class User(Model):
    """
    Foydalanuvchi modeli
    Django User modeliga o'xshash
    """
    id = fields.IntField(pk=True)
    username = fields.CharField(max_length=150, unique=True, description="Foydalanuvchi nomi")
    email = fields.CharField(max_length=255, unique=True, description="Email manzil")
    password = fields.CharField(max_length=255, description="Shifrlangan parol")
    
    first_name = fields.CharField(max_length=150, null=True, description="Ism")
    last_name = fields.CharField(max_length=150, null=True, description="Familiya")
    
    is_active = fields.BooleanField(default=True, description="Faol")
    is_staff = fields.BooleanField(default=False, description="Xodim")
    is_superuser = fields.BooleanField(default=False, description="Superuser")
    
    date_joined = fields.DatetimeField(auto_now_add=True, description="Ro'yxatdan o'tgan sana")
    last_login = fields.DatetimeField(null=True, description="Oxirgi kirish")
    
    class Meta:
        table = "auth_user"
        ordering = ["-date_joined"]
    
    def __str__(self):
        return self.username
    
    @property
    def full_name(self):
        """To'liq ism"""
        if self.first_name and self.last_name:
            return f"{self.first_name} {self.last_name}"
        return self.username
    
    async def set_password(self, raw_password: str):
        """Parolni shifrlash va saqlash"""
        from passlib.context import CryptContext
        pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")
        self.password = pwd_context.hash(raw_password)
        await self.save()
    
    async def check_password(self, raw_password: str) -> bool:
        """Parolni tekshirish"""
        from passlib.context import CryptContext
        pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")
        return pwd_context.verify(raw_password, self.password)
    
    async def update_last_login(self):
        """Oxirgi kirish vaqtini yangilash"""
        self.last_login = datetime.now()
        await self.save()


class Group(Model):
    """
    Guruh modeli
    Foydalanuvchilarni guruhlash uchun
    """
    id = fields.IntField(pk=True)
    name = fields.CharField(max_length=150, unique=True, description="Guruh nomi")
    
    users = fields.ManyToManyField(
        "models.User",
        related_name="groups",
        through="auth_user_groups"
    )
    
    class Meta:
        table = "auth_group"
    
    def __str__(self):
        return self.name


class Permission(Model):
    """
    Ruxsat modeli
    Foydalanuvchi va guruhlar uchun ruxsatlar
    """
    id = fields.IntField(pk=True)
    name = fields.CharField(max_length=255, description="Ruxsat nomi")
    codename = fields.CharField(max_length=100, unique=True, description="Kod nomi")
    
    class Meta:
        table = "auth_permission"
    
    def __str__(self):
        return self.name
