"""
OsonWeb Admin - Admin panel router va views
"""
from typing import Optional, List
from fastapi import APIRouter, Request, Form, HTTPException, Depends
from fastapi.responses import HTMLResponse, RedirectResponse
from fastapi.templating import Jinja2Templates
from pathlib import Path
from tortoise.models import Model
from tortoise.exceptions import DoesNotExist
from .admin import get_admin_site


# Router yaratish
admin_router = APIRouter(tags=["admin"])

# Shablonlar papkasi
templates_dir = Path(__file__).parent / "templates"
templates = Jinja2Templates(directory=str(templates_dir))


@admin_router.get("/", response_class=HTMLResponse)
async def admin_index(request: Request):
    """
    Admin panel bosh sahifasi
    Barcha ro'yxatdan o'tgan modellarni ko'rsatish
    """
    site = get_admin_site()
    registry = site.get_registry()
    
    # Modellar ro'yxatini tayyorlash
    models_list = []
    for model, admin in registry.items():
        model_name = model.__name__
        table_name = model._meta.db_table if hasattr(model._meta, 'db_table') else model_name.lower()
        
        # Modellar sonini olish
        count = await model.all().count()
        
        models_list.append({
            "name": model_name,
            "table": table_name,
            "count": count,
            "verbose_name": model_name,
            "verbose_name_plural": f"{model_name}lar",
        })
    
    return templates.TemplateResponse(
        "admin/index.html",
        {
            "request": request,
            "title": "Admin Panel",
            "models": models_list,
        }
    )


@admin_router.get("/{model_name}/", response_class=HTMLResponse)
async def admin_model_list(
    request: Request,
    model_name: str,
    page: int = 1,
    search: Optional[str] = None
):
    """
    Model ro'yxatini ko'rsatish
    """
    site = get_admin_site()
    
    # Modelni topish
    model = None
    model_admin = None
    
    for m, admin in site.get_registry().items():
        if m.__name__.lower() == model_name.lower():
            model = m
            model_admin = admin
            break
    
    if model is None:
        raise HTTPException(status_code=404, detail="Model topilmadi")
    
    # QuerySet ni olish
    queryset = await model_admin.get_queryset()
    
    # Qidiruv
    if search and model_admin.get_search_fields():
        # Oddiy qidiruv (birinchi maydon bo'yicha)
        search_field = model_admin.get_search_fields()[0]
        queryset = queryset.filter(**{f"{search_field}__icontains": search})
    
    # Pagination
    per_page = model_admin.list_per_page
    offset = (page - 1) * per_page
    
    total = await queryset.count()
    items = await queryset.offset(offset).limit(per_page)
    
    # Maydonlarni olish
    list_display = model_admin.get_list_display()
    
    # Ma'lumotlarni tayyorlash
    items_data = []
    for item in items:
        item_dict = {"id": item.id}
        for field in list_display:
            if field != "id":
                value = getattr(item, field, None)
                item_dict[field] = value
        items_data.append(item_dict)
    
    # Sahifalar soni
    total_pages = (total + per_page - 1) // per_page
    
    return templates.TemplateResponse(
        "admin/model_list.html",
        {
            "request": request,
            "title": f"{model.__name__} ro'yxati",
            "model_name": model_name,
            "list_display": list_display,
            "items": items_data,
            "page": page,
            "total_pages": total_pages,
            "total": total,
            "search": search or "",
            "has_search": bool(model_admin.get_search_fields()),
        }
    )


@admin_router.get("/{model_name}/add/", response_class=HTMLResponse)
async def admin_model_add(request: Request, model_name: str):
    """
    Yangi obyekt qo'shish sahifasi
    """
    site = get_admin_site()
    
    # Modelni topish
    model = None
    model_admin = None
    
    for m, admin in site.get_registry().items():
        if m.__name__.lower() == model_name.lower():
            model = m
            model_admin = admin
            break
    
    if model is None or not model_admin.has_add_permission():
        raise HTTPException(status_code=404, detail="Model topilmadi yoki ruxsat yo'q")
    
    # Maydonlarni olish
    fields = model_admin.get_fields()
    if fields is None:
        # Barcha maydonlarni olish (id dan tashqari)
        fields = [f.model_field_name for f in model._meta.fields_map.values() 
                 if f.model_field_name not in ["id"]]
    
    return templates.TemplateResponse(
        "admin/model_form.html",
        {
            "request": request,
            "title": f"Yangi {model.__name__} qo'shish",
            "model_name": model_name,
            "fields": fields,
            "is_add": True,
        }
    )


@admin_router.post("/{model_name}/add/")
async def admin_model_add_post(request: Request, model_name: str):
    """
    Yangi obyekt qo'shish (POST)
    """
    site = get_admin_site()
    
    # Modelni topish
    model = None
    model_admin = None
    
    for m, admin in site.get_registry().items():
        if m.__name__.lower() == model_name.lower():
            model = m
            model_admin = admin
            break
    
    if model is None or not model_admin.has_add_permission():
        raise HTTPException(status_code=404, detail="Model topilmadi yoki ruxsat yo'q")
    
    # Form ma'lumotlarini olish
    form_data = await request.form()
    data = dict(form_data)
    
    # Obyekt yaratish
    try:
        await model.create(**data)
        return RedirectResponse(
            url=f"/admin/{model_name}/",
            status_code=303
        )
    except Exception as e:
        raise HTTPException(status_code=400, detail=str(e))


@admin_router.get("/{model_name}/{item_id}/", response_class=HTMLResponse)
async def admin_model_edit(request: Request, model_name: str, item_id: int):
    """
    Obyektni tahrirlash sahifasi
    """
    site = get_admin_site()
    
    # Modelni topish
    model = None
    model_admin = None
    
    for m, admin in site.get_registry().items():
        if m.__name__.lower() == model_name.lower():
            model = m
            model_admin = admin
            break
    
    if model is None or not model_admin.has_change_permission():
        raise HTTPException(status_code=404, detail="Model topilmadi yoki ruxsat yo'q")
    
    # Obyektni topish
    try:
        item = await model.get(id=item_id)
    except DoesNotExist:
        raise HTTPException(status_code=404, detail="Obyekt topilmadi")
    
    # Maydonlarni olish
    fields = model_admin.get_fields()
    if fields is None:
        fields = [f.model_field_name for f in model._meta.fields_map.values() 
                 if f.model_field_name not in ["id"]]
    
    # Maydon qiymatlarini olish
    field_values = {}
    for field in fields:
        field_values[field] = getattr(item, field, "")
    
    return templates.TemplateResponse(
        "admin/model_form.html",
        {
            "request": request,
            "title": f"{model.__name__} tahrirlash",
            "model_name": model_name,
            "item_id": item_id,
            "fields": fields,
            "field_values": field_values,
            "is_add": False,
        }
    )


@admin_router.post("/{model_name}/{item_id}/")
async def admin_model_edit_post(request: Request, model_name: str, item_id: int):
    """
    Obyektni tahrirlash (POST)
    """
    site = get_admin_site()
    
    # Modelni topish
    model = None
    model_admin = None
    
    for m, admin in site.get_registry().items():
        if m.__name__.lower() == model_name.lower():
            model = m
            model_admin = admin
            break
    
    if model is None or not model_admin.has_change_permission():
        raise HTTPException(status_code=404, detail="Model topilmadi yoki ruxsat yo'q")
    
    # Obyektni topish
    try:
        item = await model.get(id=item_id)
    except DoesNotExist:
        raise HTTPException(status_code=404, detail="Obyekt topilmadi")
    
    # Form ma'lumotlarini olish
    form_data = await request.form()
    data = dict(form_data)
    
    # Obyektni yangilash
    try:
        for key, value in data.items():
            setattr(item, key, value)
        await item.save()
        
        return RedirectResponse(
            url=f"/admin/{model_name}/",
            status_code=303
        )
    except Exception as e:
        raise HTTPException(status_code=400, detail=str(e))


@admin_router.post("/{model_name}/{item_id}/delete/")
async def admin_model_delete(model_name: str, item_id: int):
    """
    Obyektni o'chirish
    """
    site = get_admin_site()
    
    # Modelni topish
    model = None
    model_admin = None
    
    for m, admin in site.get_registry().items():
        if m.__name__.lower() == model_name.lower():
            model = m
            model_admin = admin
            break
    
    if model is None or not model_admin.has_delete_permission():
        raise HTTPException(status_code=404, detail="Model topilmadi yoki ruxsat yo'q")
    
    # Obyektni topish va o'chirish
    try:
        item = await model.get(id=item_id)
        await item.delete()
        
        return RedirectResponse(
            url=f"/admin/{model_name}/",
            status_code=303
        )
    except DoesNotExist:
        raise HTTPException(status_code=404, detail="Obyekt topilmadi")
    except Exception as e:
        raise HTTPException(status_code=400, detail=str(e))


__all__ = ["admin_router"]
