"""
OsonWeb Admin - Admin panel klasslari
"""
from typing import Type, List, Optional, Dict, Any
from tortoise.models import Model


class ModelAdmin:
    """
    Model uchun admin konfiguratsiyasi
    Django ModelAdmin ga o'xshash
    """
    
    # Ko'rsatiladigan maydonlar
    list_display: List[str] = ["id"]
    
    # Qidiruv maydonlari
    search_fields: List[str] = []
    
    # Filtrlash maydonlari
    list_filter: List[str] = []
    
    # Sahifadagi elementlar soni
    list_per_page: int = 25
    
    # Tahrirlash maydonlari
    fields: Optional[List[str]] = None
    
    # Faqat o'qish uchun maydonlar
    readonly_fields: List[str] = []
    
    # Ordering
    ordering: List[str] = ["-id"]
    
    def __init__(self, model: Type[Model]):
        self.model = model
    
    def get_list_display(self) -> List[str]:
        """Ko'rsatiladigan maydonlar ro'yxatini olish"""
        return self.list_display
    
    def get_search_fields(self) -> List[str]:
        """Qidiruv maydonlari ro'yxatini olish"""
        return self.search_fields
    
    def get_list_filter(self) -> List[str]:
        """Filtrlash maydonlari ro'yxatini olish"""
        return self.list_filter
    
    def get_fields(self) -> Optional[List[str]]:
        """Tahrirlash maydonlari ro'yxatini olish"""
        return self.fields
    
    def get_readonly_fields(self) -> List[str]:
        """Faqat o'qish uchun maydonlar ro'yxatini olish"""
        return self.readonly_fields
    
    def get_ordering(self) -> List[str]:
        """Tartiblash maydonlari ro'yxatini olish"""
        return self.ordering
    
    async def get_queryset(self):
        """QuerySet ni olish"""
        queryset = self.model.all()
        ordering = self.get_ordering()
        if ordering:
            queryset = queryset.order_by(*ordering)
        return queryset
    
    def has_add_permission(self) -> bool:
        """Qo'shish ruxsati"""
        return True
    
    def has_change_permission(self) -> bool:
        """O'zgartirish ruxsati"""
        return True
    
    def has_delete_permission(self) -> bool:
        """O'chirish ruxsati"""
        return True
    
    def has_view_permission(self) -> bool:
        """Ko'rish ruxsati"""
        return True


class AdminSite:
    """
    Admin panel asosiy klassi
    Barcha modellarni ro'yxatdan o'tkazish va boshqarish
    """
    
    def __init__(self):
        self._registry: Dict[Type[Model], ModelAdmin] = {}
    
    def register(
        self,
        model: Type[Model],
        admin_class: Optional[Type[ModelAdmin]] = None
    ):
        """
        Modelni admin panelga ro'yxatdan o'tkazish
        
        Args:
            model: Tortoise Model klassi
            admin_class: ModelAdmin klassi (ixtiyoriy)
        """
        if admin_class is None:
            admin_class = ModelAdmin
        
        self._registry[model] = admin_class(model)
    
    def unregister(self, model: Type[Model]):
        """Modelni ro'yxatdan o'chirish"""
        if model in self._registry:
            del self._registry[model]
    
    def is_registered(self, model: Type[Model]) -> bool:
        """Model ro'yxatdan o'tganligini tekshirish"""
        return model in self._registry
    
    def get_model_admin(self, model: Type[Model]) -> Optional[ModelAdmin]:
        """Model uchun ModelAdmin ni olish"""
        return self._registry.get(model)
    
    def get_registry(self) -> Dict[Type[Model], ModelAdmin]:
        """Barcha ro'yxatdan o'tgan modellarni olish"""
        return self._registry


# Global admin sayt
_site = AdminSite()


def register(model: Type[Model], admin_class: Optional[Type[ModelAdmin]] = None):
    """
    Dekorator: Modelni admin panelga ro'yxatdan o'tkazish
    
    Misol:
        @register(User)
        class UserAdmin(ModelAdmin):
            list_display = ["id", "username", "email"]
    """
    def decorator(admin_cls):
        _site.register(model, admin_cls)
        return admin_cls
    
    # Agar admin_class berilmagan bo'lsa, dekorator sifatida ishlatish
    if admin_class is None:
        return decorator
    
    # Aks holda, to'g'ridan-to'g'ri ro'yxatdan o'tkazish
    _site.register(model, admin_class)
    return admin_class


def get_admin_site() -> AdminSite:
    """Global admin saytni olish"""
    return _site


__all__ = [
    "ModelAdmin",
    "AdminSite",
    "register",
    "get_admin_site",
]
