"""
OsonWeb CLI - Buyruqlar interfeysi
"""
import os
import sys
import shutil
from pathlib import Path
from typing import Optional
import typer
from rich.console import Console
from rich.panel import Panel
from rich.progress import Progress, SpinnerColumn, TextColumn

app = typer.Typer(
    name="osonweb",
    help="OsonWeb - O'zbekcha Web Freymvork CLI",
    add_completion=False,
)
console = Console()


def get_template_dir():
    """Shablon papkasini topish"""
    return Path(__file__).parent / "project_template"


def get_app_template_dir():
    """App shablon papkasini topish"""
    return Path(__file__).parent / "app_template"


@app.command()
def createproject(
    name: str = typer.Argument(..., help="Loyiha nomi"),
    directory: Optional[str] = typer.Option(None, help="Loyiha yaratish uchun papka"),
):
    """
    Yangi OsonWeb loyihasini yaratish
    
    Misol:
        osonweb createproject mening_loyiham
    """
    target_dir = Path(directory) if directory else Path.cwd() / name
    
    if target_dir.exists():
        console.print(f"[red]Xato:[/red] '{target_dir}' papkasi allaqachon mavjud!")
        raise typer.Exit(1)
    
    with Progress(
        SpinnerColumn(),
        TextColumn("[progress.description]{task.description}"),
        console=console,
    ) as progress:
        task = progress.add_task(f"[cyan]Loyiha yaratilmoqda: {name}...", total=None)
        
        # Loyiha papkasini yaratish
        target_dir.mkdir(parents=True, exist_ok=True)
        
        # Shablon fayllarini nusxalash
        template_dir = get_template_dir()
        
        # Asosiy fayllarni yaratish
        create_project_structure(target_dir, name)
        
        progress.update(task, completed=True)
    
    console.print(Panel.fit(
        f"[green]✓[/green] Loyiha muvaffaqiyatli yaratildi: [bold]{name}[/bold]\n\n"
        f"Keyingi qadamlar:\n"
        f"  cd {name}\n"
        f"  osonweb createapp blog\n"
        f"  osonweb migrate\n"
        f"  osonweb runserver",
        title="[bold green]Muvaffaqiyat![/bold green]",
        border_style="green",
    ))


def create_project_structure(target_dir: Path, project_name: str):
    """Loyiha strukturasini yaratish"""
    
    # manage.py
    manage_py = target_dir / "manage.py"
    manage_py.write_text(f'''#!/usr/bin/env python3
"""
{project_name} - OsonWeb Loyihasi
Boshqaruv skripti
"""
import sys
from pathlib import Path

# Loyiha yo'lini qo'shish
sys.path.insert(0, str(Path(__file__).parent))

if __name__ == "__main__":
    from osonweb.cli import app
    app()
''')
    manage_py.chmod(0o755)
    
    # Loyiha papkasi
    project_dir = target_dir / project_name
    project_dir.mkdir(exist_ok=True)
    
    # __init__.py
    (project_dir / "__init__.py").write_text('"""Loyiha paketi"""\n')
    
    # settings.py
    settings_py = project_dir / "settings.py"
    settings_py.write_text(f'''"""
{project_name} - Sozlamalar
"""
from pathlib import Path

# Loyiha asosiy papkasi
BASE_DIR = Path(__file__).resolve().parent.parent

# Xavfsizlik
SECRET_KEY = "osonweb-secret-key-{project_name}-change-this-in-production"
DEBUG = True
ALLOWED_HOSTS = ["*"]

# Ilovalar
INSTALLED_APPS = [
    "osonweb.contrib.admin",
    "osonweb.contrib.auth",
]

# Ma'lumotlar bazasi
DATABASES = {{
    "default": {{
        "engine": "tortoise.backends.sqlite",
        "credentials": {{
            "file_path": str(BASE_DIR / "db.sqlite3"),
        }}
    }}
}}

# Shablonlar
TEMPLATES_DIR = BASE_DIR / "templates"
STATIC_DIR = BASE_DIR / "static"

# Tortoise ORM sozlamalari
TORTOISE_ORM = {{
    "connections": {{"default": "sqlite://db.sqlite3"}},
    "apps": {{
        "models": {{
            "models": ["osonweb.contrib.auth.models", "aerich.models"],
            "default_connection": "default",
        }},
    }},
}}
''')
    
    # urls.py
    urls_py = project_dir / "urls.py"
    urls_py.write_text(f'''"""
{project_name} - URL marshrutlari
"""
from osonweb import OsonWeb

app = OsonWeb(__name__)

@app.get("/")
async def index():
    return {{"message": "Xush kelibsiz! OsonWeb ishlamoqda."}}

# Admin panelni ulash
from osonweb.contrib.admin import admin_router
app.include_router(admin_router, prefix="/admin")
''')
    
    # main.py
    main_py = project_dir / "main.py"
    main_py.write_text(f'''"""
{project_name} - Asosiy ilova
"""
import sys
from pathlib import Path

# Loyiha yo'lini qo'shish
sys.path.insert(0, str(Path(__file__).resolve().parent.parent))

from {project_name}.urls import app
from {project_name} import settings
from tortoise.contrib.fastapi import register_tortoise

# Tortoise ORM ni ro'yxatdan o'tkazish
register_tortoise(
    app,
    config=settings.TORTOISE_ORM,
    generate_schemas=True,
    add_exception_handlers=True,
)

if __name__ == "__main__":
    import uvicorn
    uvicorn.run(app, host="127.0.0.1", port=8000)
''')
    
    # templates va static papkalari
    (target_dir / "templates").mkdir(exist_ok=True)
    (target_dir / "static").mkdir(exist_ok=True)
    
    # .gitignore
    gitignore = target_dir / ".gitignore"
    gitignore.write_text('''__pycache__/
*.py[cod]
*$py.class
*.so
.Python
db.sqlite3
.env
venv/
.venv/
migrations/
''')
    
    # requirements.txt
    requirements = target_dir / "requirements.txt"
    requirements.write_text('''osonweb>=0.1.0
uvicorn[standard]>=0.24.0
''')


@app.command()
def createapp(
    name: str = typer.Argument(..., help="App nomi"),
):
    """
    Loyiha ichida yangi app yaratish
    
    Misol:
        osonweb createapp blog
    """
    target_dir = Path.cwd() / name
    
    if target_dir.exists():
        console.print(f"[red]Xato:[/red] '{name}' papkasi allaqachon mavjud!")
        raise typer.Exit(1)
    
    with Progress(
        SpinnerColumn(),
        TextColumn("[progress.description]{task.description}"),
        console=console,
    ) as progress:
        task = progress.add_task(f"[cyan]App yaratilmoqda: {name}...", total=None)
        
        create_app_structure(target_dir, name)
        
        progress.update(task, completed=True)
    
    console.print(Panel.fit(
        f"[green]✓[/green] App muvaffaqiyatli yaratildi: [bold]{name}[/bold]\n\n"
        f"Keyingi qadamlar:\n"
        f"  1. settings.py fayliga '{name}' ni INSTALLED_APPS ga qo'shing\n"
        f"  2. {name}/models.py da modellarni yarating\n"
        f"  3. osonweb makemigrations\n"
        f"  4. osonweb migrate",
        title="[bold green]Muvaffaqiyat![/bold green]",
        border_style="green",
    ))


def create_app_structure(target_dir: Path, app_name: str):
    """App strukturasini yaratish"""
    target_dir.mkdir(parents=True, exist_ok=True)
    
    # __init__.py
    (target_dir / "__init__.py").write_text(f'""""{app_name}" ilovasi"""\n')
    
    # models.py
    models_py = target_dir / "models.py"
    models_py.write_text(f'''"""
{app_name} - Modellar
"""
from tortoise import fields
from tortoise.models import Model


# Bu yerda modellaringizni yarating
# Misol:
# class Post(Model):
#     id = fields.IntField(pk=True)
#     title = fields.CharField(max_length=200)
#     content = fields.TextField()
#     created_at = fields.DatetimeField(auto_now_add=True)
#
#     class Meta:
#         table = "{app_name}_post"
''')
    
    # views.py
    views_py = target_dir / "views.py"
    views_py.write_text(f'''"""
{app_name} - Ko'rinishlar (Views)
"""
from fastapi import APIRouter

router = APIRouter()


@router.get("/")
async def index():
    return {{"message": "{app_name} ilovasi"}}


# Bu yerda ko'rinishlaringizni yarating
''')
    
    # urls.py
    urls_py = target_dir / "urls.py"
    urls_py.write_text(f'''"""
{app_name} - URL marshrutlari
"""
from .views import router

# Bu routerni asosiy urls.py ga qo'shing:
# from {app_name}.urls import router as {app_name}_router
# app.include_router({app_name}_router, prefix="/{app_name}")
''')
    
    # admin.py
    admin_py = target_dir / "admin.py"
    admin_py.write_text(f'''"""
{app_name} - Admin panel sozlamalari
"""
# from osonweb.contrib.admin import register
# from .models import YourModel

# Bu yerda modellaringizni admin panelga ro'yxatdan o'tkazing
# register(YourModel)
''')


@app.command()
def makemigrations(
    name: Optional[str] = typer.Option(None, help="Migratsiya nomi"),
):
    """
    Modellar asosida migratsiya fayllarini yaratish
    
    Misol:
        osonweb makemigrations
        osonweb makemigrations --name "add_user_model"
    """
    console.print("[cyan]Migratsiyalar tayyorlanmoqda...[/cyan]")
    
    # Aerich ni ishga tushirish
    import subprocess
    
    try:
        # Aerich init (agar birinchi marta bo'lsa)
        aerich_config = Path.cwd() / "pyproject.toml"
        if not aerich_config.exists():
            result = subprocess.run(
                ["aerich", "init", "-t", "settings.TORTOISE_ORM"],
                capture_output=True,
                text=True,
            )
            if result.returncode != 0:
                # Aerich konfiguratsiyasini qo'lda yaratish
                init_aerich_config()
        
        # Migratsiya yaratish
        cmd = ["aerich", "migrate"]
        if name:
            cmd.extend(["--name", name])
        
        result = subprocess.run(cmd, capture_output=True, text=True)
        
        if result.returncode == 0:
            console.print("[green]✓[/green] Migratsiyalar muvaffaqiyatli yaratildi!")
            if result.stdout:
                console.print(result.stdout)
        else:
            console.print(f"[yellow]Ogohlantirish:[/yellow] {result.stderr}")
            
    except FileNotFoundError:
        console.print("[red]Xato:[/red] Aerich o'rnatilmagan. O'rnatish: pip install aerich")
        raise typer.Exit(1)
    except Exception as e:
        console.print(f"[red]Xato:[/red] {str(e)}")
        raise typer.Exit(1)


def init_aerich_config():
    """Aerich konfiguratsiyasini yaratish"""
    config_file = Path.cwd() / "pyproject.toml"
    
    # settings.py ni topish
    settings_files = list(Path.cwd().rglob("settings.py"))
    if not settings_files:
        console.print("[red]Xato:[/red] settings.py fayli topilmadi!")
        raise typer.Exit(1)
    
    settings_path = settings_files[0]
    project_name = settings_path.parent.name
    
    config_content = f'''[tool.aerich]
tortoise_orm = "{project_name}.settings.TORTOISE_ORM"
location = "./migrations"
src_folder = "./."
'''
    
    if config_file.exists():
        # Mavjud faylga qo'shish
        content = config_file.read_text()
        if "[tool.aerich]" not in content:
            config_file.write_text(content + "\n" + config_content)
    else:
        config_file.write_text(config_content)
    
    # Aerich init-db
    import subprocess
    subprocess.run(["aerich", "init-db"], check=True)


@app.command()
def migrate():
    """
    Migratsiyalarni ma'lumotlar bazasiga qo'llash
    
    Misol:
        osonweb migrate
    """
    console.print("[cyan]Migratsiyalar qo'llanmoqda...[/cyan]")
    
    import subprocess
    
    try:
        result = subprocess.run(
            ["aerich", "upgrade"],
            capture_output=True,
            text=True,
        )
        
        if result.returncode == 0:
            console.print("[green]✓[/green] Migratsiyalar muvaffaqiyatli qo'llandi!")
            if result.stdout:
                console.print(result.stdout)
        else:
            console.print(f"[red]Xato:[/red] {result.stderr}")
            raise typer.Exit(1)
            
    except FileNotFoundError:
        console.print("[red]Xato:[/red] Aerich o'rnatilmagan. O'rnatish: pip install aerich")
        raise typer.Exit(1)
    except Exception as e:
        console.print(f"[red]Xato:[/red] {str(e)}")
        raise typer.Exit(1)


@app.command()
def runserver(
    address: str = typer.Argument("127.0.0.1:8000", help="Host:Port (masalan: 0.0.0.0:8000)"),
    reload: bool = typer.Option(True, help="Avtomatik qayta yuklash"),
):
    """
    Rivojlantirish serverini ishga tushirish
    
    Misol:
        osonweb runserver
        osonweb runserver 0.0.0.0:8080
        osonweb runserver 127.0.0.1:8000 --no-reload
    """
    # Host va portni ajratish
    if ":" in address:
        host, port_str = address.split(":", 1)
        port = int(port_str)
    else:
        host = address
        port = 8000
    
    # main.py ni topish
    main_files = list(Path.cwd().rglob("main.py"))
    if not main_files:
        console.print("[red]Xato:[/red] main.py fayli topilmadi!")
        console.print("Loyiha papkasida ekanligingizga ishonch hosil qiling.")
        raise typer.Exit(1)
    
    main_file = main_files[0]
    project_name = main_file.parent.name
    
    console.print(Panel.fit(
        f"[bold cyan]OsonWeb Server[/bold cyan]\n\n"
        f"Loyiha: [bold]{project_name}[/bold]\n"
        f"Manzil: [bold]http://{host}:{port}[/bold]\n"
        f"Admin: [bold]http://{host}:{port}/admin[/bold]\n\n"
        f"[dim]Server to'xtatish uchun CTRL+C ni bosing[/dim]",
        border_style="cyan",
    ))
    
    import subprocess
    
    try:
        cmd = [
            "uvicorn",
            f"{project_name}.main:app",
            "--host", host,
            "--port", str(port),
        ]
        
        if reload:
            cmd.append("--reload")
        
        subprocess.run(cmd)
        
    except KeyboardInterrupt:
        console.print("\n[yellow]Server to'xtatildi.[/yellow]")
    except Exception as e:
        console.print(f"[red]Xato:[/red] {str(e)}")
        raise typer.Exit(1)


@app.command()
def createsuperuser():
    """
    Admin panel uchun superuser yaratish
    
    Misol:
        osonweb createsuperuser
    """
    console.print("[bold cyan]Superuser yaratish[/bold cyan]\n")
    
    username = typer.prompt("Foydalanuvchi nomi")
    email = typer.prompt("Email")
    password = typer.prompt("Parol", hide_input=True)
    password_confirm = typer.prompt("Parolni tasdiqlang", hide_input=True)
    
    if password != password_confirm:
        console.print("[red]Xato:[/red] Parollar mos kelmadi!")
        raise typer.Exit(1)
    
    # Foydalanuvchini yaratish
    import asyncio
    from pathlib import Path
    
    # settings.py ni topish
    settings_files = list(Path.cwd().rglob("settings.py"))
    if not settings_files:
        console.print("[red]Xato:[/red] settings.py fayli topilmadi!")
        raise typer.Exit(1)
    
    async def create_user():
        from tortoise import Tortoise
        import sys
        
        # Loyiha yo'lini qo'shish
        project_dir = settings_files[0].parent.parent
        sys.path.insert(0, str(project_dir))
        
        # Settings ni import qilish
        project_name = settings_files[0].parent.name
        settings_module = __import__(f"{project_name}.settings", fromlist=["TORTOISE_ORM"])
        
        await Tortoise.init(config=settings_module.TORTOISE_ORM)
        await Tortoise.generate_schemas()
        
        # User modelini import qilish
        try:
            from osonweb.contrib.auth.models import User
            
            # Foydalanuvchi mavjudligini tekshirish
            existing_user = await User.filter(username=username).first()
            if existing_user:
                console.print(f"[red]Xato:[/red] '{username}' foydalanuvchisi allaqachon mavjud!")
                await Tortoise.close_connections()
                return False
            
            # Yangi foydalanuvchi yaratish
            from passlib.context import CryptContext
            pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")
            
            user = await User.create(
                username=username,
                email=email,
                password=pwd_context.hash(password),
                is_superuser=True,
                is_staff=True,
                is_active=True,
            )
            
            await Tortoise.close_connections()
            return True
            
        except ImportError:
            console.print("[red]Xato:[/red] Auth modeli topilmadi!")
            await Tortoise.close_connections()
            return False
    
    try:
        success = asyncio.run(create_user())
        if success:
            console.print(f"\n[green]✓[/green] Superuser muvaffaqiyatli yaratildi: [bold]{username}[/bold]")
    except Exception as e:
        console.print(f"[red]Xato:[/red] {str(e)}")
        raise typer.Exit(1)


@app.command()
def version():
    """OsonWeb versiyasini ko'rsatish"""
    from osonweb import __version__
    console.print(f"OsonWeb versiya: [bold cyan]{__version__}[/bold cyan]")


if __name__ == "__main__":
    app()
