"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const s3_assets = require("@aws-cdk/aws-s3-assets");
/**
 * Specifies bucket deployment source.
 *
 * Usage:
 *
 *     Source.bucket(bucket, key)
 *     Source.asset('/local/path/to/directory')
 *     Source.asset('/local/path/to/a/file.zip')
 *
 */
class Source {
    /**
     * Uses a .zip file stored in an S3 bucket as the source for the destination bucket contents.
     * @param bucket The S3 Bucket
     * @param zipObjectKey The S3 object key of the zip file with contents
     */
    static bucket(bucket, zipObjectKey) {
        return {
            bind: () => ({ bucket, zipObjectKey })
        };
    }
    /**
     * Uses a local asset as the deployment source.
     * @param path The path to a local .zip file or a directory
     */
    static asset(path, options) {
        return {
            bind(context) {
                let id = 1;
                while (context.node.tryFindChild(`Asset${id}`)) {
                    id++;
                }
                const asset = new s3_assets.Asset(context, `Asset${id}`, {
                    path,
                    ...options,
                });
                if (!asset.isZipArchive) {
                    throw new Error('Asset path must be either a .zip file or a directory');
                }
                return {
                    bucket: asset.bucket,
                    zipObjectKey: asset.s3ObjectKey
                };
            }
        };
    }
    constructor() { }
}
exports.Source = Source;
//# sourceMappingURL=data:application/json;base64,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