import abc
import json
import os
from datetime import datetime
from pathlib import Path
import pydantic
from typing import Union

from flowui.client.airflow_client import AirflowRestClient
from flowui.logger import get_configured_logger
from flowui.utils.enum_types import DeployModeType
from flowui.exceptions.exceptions import InvalidOperatorOutputError


class BaseOperator(metaclass=abc.ABCMeta):

    @classmethod
    def set_metadata(cls, metadata):
        """
        _summary_

        Args:
            metadata (_type_): _description_
        """
        # Operator name as used by Airflow
        cls.__name__ = metadata.get("name", "BaseOperator")

        # Operator version: update it to auto update the deployed version
        cls.__version__ = metadata.get("version", "0.1.0")

        # Dockerfile to build Image for this function
        cls.__dockerfile__ = metadata.get("dockerfile", "Dockerfile-base")

        # Full metadata
        cls._metadata_ = metadata


    def __init__(
        self,
        deploy_mode: DeployModeType,
        task_id: str,
        dag_id: str,
    ) -> None:
        """
        The base class from which every FlowUI custom Operator should inherit from.
        BaseOperator methods and variables that can be used by inheriting Operators:

        self._metadata_ - Metadata as defined by the user
        self.upstream_tasks - Metadata and XCom data results from previous Tasks
        self.results_path - Path to store results data
        self.logger - Logger functionality

        Args:
            deploy_mode (DeployModeType): _description_
            task_id (str): _description_
            dag_id (str): _description_
        """

        # Operator task attributes
        self.task_id = task_id
        self.dag_id = dag_id
        self.deploy_mode = deploy_mode

        # Clients
        self.airflow_client = AirflowRestClient()

        # Logger
        self.logger = get_configured_logger(f"{self.__class__.__name__ }-{self.task_id}")


    def start_logger(self):
        """
        Start logger.
        """
        self.logger.info(f"Started {self.task_id} of type {self.__class__.__name__} at {str(datetime.now().isoformat())}")


    def generate_paths(self):
        """
        Generates paths and stores them in the attributes `self.volume_mount_path`, `self.run_path`, `self.results_path` and `self.report_path`.
        """
        self.volume_mount_path = os.environ.get("VOLUME_MOUNT_PATH_DOCKER", "/opt/mnt/fs")
        
        self.run_path = f"{self.volume_mount_path}/runs/{self.dag_id}/{self.dag_run_id}"
        if not Path(self.run_path).is_dir():
            Path(self.run_path).mkdir(parents=True, exist_ok=True)
        
        self.results_path = f"{self.run_path}/{self.task_id}/results"
        if not Path(self.results_path).is_dir():
            Path(self.results_path).mkdir(parents=True, exist_ok=True)
        
        self.report_path = f"{self.run_path}/{self.task_id}/report"
        if not Path(self.report_path).is_dir():
            Path(self.report_path).mkdir(parents=True, exist_ok=True)


    def get_upstream_tasks_xcom(self):
        """
        Get XCOM from upstream tasks. Stores this information in the attribute `self.upstream_task_xcom`

        Raises:
            NotImplementedError: _description_
        """
        self.upstream_task_xcom = dict()
        if self.deploy_mode == "local-python":
            for tid in list(self.airflow_context['task'].upstream_task_ids):
                self.upstream_task_xcom[tid] = self.airflow_context['ti'].xcom_pull(task_ids=tid) 
        elif self.deploy_mode == "local-bash":
            with open("/opt/mnt/fs/tmp/xcom_input.json") as f:
                self.upstream_task_xcom = json.load(f)
        elif self.deploy_mode == "local-k8s":
            with open("/opt/mnt/fs/tmp/xcom_input.json") as f:
                self.upstream_task_xcom = json.load(f)
        else:
            raise NotImplementedError(f"Get upstream XCOM not implemented for deploy_mode=={self.deploy_mode}")

    
    def validate_and_get_env_secrets(self, operator_secrets_model: pydantic.BaseModel = None):
        """
        Get secret variables for this Operator from ENV. The necessary secret variables to run the Operator should be defined in the Operator's SecretsModel.

        Args:
            operator_secrets_model (pydantic.BaseModel): _description_
        """
        self.secrets = None
        if operator_secrets_model:
            secrets_names = list(operator_secrets_model.schema()["properties"].keys())
            secrets = dict()
            for s in secrets_names:
                secrets[s] = os.getenv(s, None)
            self.secrets = operator_secrets_model(**secrets)


    def format_xcom(self, xcom_obj: dict) -> dict:
        """
        Adds extra metadata to XCOM dictionary content.

        Args:
            xcom_obj (dict): XCOM dictionary

        Returns:
            dict: XCOM dictionary
        """
        if not isinstance(xcom_obj, dict):
            print(f"Operator {self.__class__.__name__} is not returning a valid XCOM object. Auto-generating a base XCOM for it...")
            self.logger.info(f"Operator {self.__class__.__name__} is not returning a valid XCOM object. Auto-generating a base XCOM for it...")
            xcom_obj = dict()

        xcom_obj.update(
            operator_name=self.__class__.__name__,
            operator_metadata=self._metadata_,
            results_path=self.results_path,
        )
        return xcom_obj


    def push_xcom(self, xcom_obj: dict):
        """
        Push operator's output to XCOM, to be used by downstream operators.

        Args:
            xcom_obj (dict): Formatted XCOM object as a dictionary

        Raises:
            NotImplementedError: _description_
        """
        if self.deploy_mode == "local-python":
            self.airflow_context['ti'].xcom_push(key=self.task_id, value=xcom_obj)
        elif self.deploy_mode == "local-bash":
            # For our extended BashOperator, return XCom must be stored in /opt/mnt/fs/tmp/xcom_output.json
            file_path = Path("/opt/mnt/fs/tmp/xcom_output.json")
            file_path.parent.mkdir(parents=True, exist_ok=True)
            with open(str(file_path), 'w') as fp:
                json.dump(xcom_obj, fp, indent=4)
        elif self.deploy_mode == "kubernetes":
            # In Kubernetes, return XCom must be stored in /airflow/xcom/return.json
            # https://airflow.apache.org/docs/apache-airflow-providers-cncf-kubernetes/stable/operators.html#how-does-xcom-work
            file_path = Path('/airflow/xcom/return.json')
            file_path.parent.mkdir(parents=True, exist_ok=True)
            with open(str(file_path), 'w') as fp:
                json.dump(xcom_obj, fp)
        else:
            raise NotImplementedError("deploy mode not accepted for xcom push")
    

    def run_operator_function(
        self, 
        airflow_context: dict,
        op_kwargs: dict,
        operator_input_model: pydantic.BaseModel,
        operator_output_model: pydantic.BaseModel, 
        operator_secrets_model: pydantic.BaseModel = None,  
    ):
        """
        _summary_

        Args:
            airflow_context (dict): Dictionary containing Airflow context information
            op_kwargs (dict): Dictionary containing Operator's kwargs
            operator_input_model (pydantic.BaseModel): Operator's InputModel
            operator_output_model (pydantic.BaseModel): Operator's OutputModel
            operator_secrets_model (pydantic.BaseModel, optional): Operator's SecretsModel. Defaults to None.

        Raises:
            InvalidOperatorOutputError: _description_
        """
        # Start logger
        self.start_logger()

        # Airflow context dictionary: https://composed.blog/airflow/execute-context
        # For local-bash and kubernetes deploy modes, we assemble this ourselves and the context data is more limited
        self.airflow_context = airflow_context
        self.dag_run_id = airflow_context.get("dag_run_id")

        # Check if Operator's necessary secrets are present in ENV
        self.validate_and_get_env_secrets(operator_secrets_model=operator_secrets_model)

        # Generate paths
        self.generate_paths()

        # Get XCOM from upstream tasks
        self.get_upstream_tasks_xcom()

        # Using pydantic to validate input data
        # operator_model_obj = operator_input_model(**op_kwargs)
        updated_op_kwargs = dict()
        for k, v in op_kwargs.items():
            if isinstance(v, dict) and v.get("type", None) == "FromUpstream":
                upstream_task_id = v.get("upstream_task_id")
                output_arg = v.get("output_arg")
                updated_op_kwargs[k] = self.upstream_task_xcom[upstream_task_id][output_arg]
            else:
                updated_op_kwargs[k] = v
        input_model_obj = operator_input_model(**updated_op_kwargs)

        # Run operator function
        output_model_obj = self.operator_function(input_model=input_model_obj)

        # Validate output data
        if not isinstance(output_model_obj, operator_output_model):
            raise InvalidOperatorOutputError(operator_name=self.__class__.__name__)

        # Push XCom
        xcom_obj = self.format_xcom(xcom_obj=output_model_obj.dict())
        self.push_xcom(xcom_obj=xcom_obj)


    @abc.abstractmethod
    def operator_function(self):
        """
        This function carries the relevant code for the Operator run.
        It should have all the necessary content for auto-generating json schemas.
        All arguments should be type annotated and docstring should carry description for each argument.
        """
        raise NotImplementedError("This method must be implemented in the child class!")        

    
    def generate_report(self):
        """This function carries the relevant code for the Operator report."""
        raise NotImplementedError("This method must be implemented in the child class!")
