"""REST client handling, including rev_ioStream base class."""

import requests
from pathlib import Path
from typing import Any, Dict, Optional, Union, List, Iterable

from memoization import cached

from singer_sdk.helpers.jsonpath import extract_jsonpath
from singer_sdk.streams import RESTStream
from singer_sdk.authenticators import BasicAuthenticator
import singer

LOGGER = singer.get_logger()


SCHEMAS_DIR = Path(__file__).parent / Path("./schemas")


class rev_ioStream(RESTStream):
    """rev_io stream class."""

    # TODO: Set the API's base URL here:
    url_base = "https://restapi.rev.io/v1"

    # OR use a dynamic url_base:
    # @property
    # def url_base(self) -> str:
    #     """Return the API URL root, configurable via tap settings."""
    #     return self.config["api_url"]


    #records_jsonpath = "$[*]"  # Or override `parse_response`.
    records_jsonpath = "$.records[*]"
    next_page_token_jsonpath = "$.has_more"  # Or override `get_next_page_token`.

    @property
    def authenticator(self) -> BasicAuthenticator:
        """Return a new authenticator object."""
        return BasicAuthenticator.create_for_stream(
            self,
            username=self.config.get("username"),
            password=self.config.get("password"),
        )

    @property
    def http_headers(self) -> dict:
        """Return the http headers needed."""
        headers = {}
        if "user_agent" in self.config:
            headers["User-Agent"] = self.config.get("user_agent")
        # If not using an authenticator, you may also provide inline auth headers:
        # headers["Private-Token"] = self.config.get("auth_token")
        return headers

    def get_next_page_token(
        self, response: requests.Response, previous_token: Optional[Any]
    ) -> Optional[Any]:
        """Return a token for identifying next page or None if no more pages."""
        
        if self.next_page_token_jsonpath:
            all_matches = extract_jsonpath(
                self.next_page_token_jsonpath, response.json()
            )
            first_match = next(iter(all_matches), None)
            if first_match: 
                first_match = (previous_token or 1) + 1                 
                 #previous_token + 1 if previous_token else 2 
            next_page_token = first_match 
        else:
            next_page_token = response.headers.get("X-Next-Page", None)

        return next_page_token

    def get_url_params(
        self, context: Optional[dict], next_page_token: Optional[Any]
    ) -> Dict[str, Any]:
        """Return a dictionary of values to be used in URL parameterization."""
        params: dict = {}
        if next_page_token:
            params["search.page"] = next_page_token

        #LOGGER.info("Starting timestamp " + str(self.get_starting_timestamp(context)))
        #LOGGER.info("With context " + str(context))
        #LOGGER.info("Replication key " + str(self.replication_key))
        
        if self.replication_key:
            #params["sort"] = "asc"
            if self.get_starting_timestamp(context):
                # Only implemented for replication keys as dates
                params["search." + self.replication_key + "_start" ] = self.get_starting_timestamp(context).strftime("%Y-%m-%dT%H:%M:%S")
                params["search.sort"] = self.replication_key

        params["search.page_size"] = self.config.get('page_size')

        #LOGGER.info("Passing params " + str(params))
        return params

    def prepare_request_payload(
        self, context: Optional[dict], next_page_token: Optional[Any]
    ) -> Optional[dict]:
        """Prepare the data payload for the REST API request.

        By default, no payload will be sent (return None).
        """
        # TODO: Delete this method if no payload is required. (Most REST APIs.)
        return None

    def parse_response(self, response: requests.Response) -> Iterable[dict]:
        """Parse the response and return an iterator of result rows."""
        # TODO: Parse response body and return a set of records.
        yield from extract_jsonpath(self.records_jsonpath, input=response.json())


    def post_process(self, row: dict, context: Optional[dict]) -> dict:
        """As needed, append or transform raw data to match expected structure."""

        return row
