# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['smoothtransitions',
 'smoothtransitions.subcommands',
 'smoothtransitions.subcommands.utils']

package_data = \
{'': ['*']}

install_requires = \
['PyYAML>=5.4.1,<6.0.0',
 'cclib>=1.7,<2.0',
 'click>=7.1.2,<8.0.0',
 'matplotlib>=3.3.4,<4.0.0',
 'numpy>=1.20.1,<2.0.0',
 'pandas>=1.2.3,<2.0.0',
 'tabulate>=0.8.9,<0.9.0',
 'typer>=0.3.2,<0.4.0']

entry_points = \
{'console_scripts': ['smoothtransitions = '
                     'smoothtransitions.smoothtransitions:app']}

setup_kwargs = {
    'name': 'smoothtransitions',
    'version': '0.4.1',
    'description': '',
    'long_description': '![CI](https://github.com/the-grayson-group/SmoothTransitions/actions/workflows/test.yml/badge.svg)\n![Publish](https://github.com/the-grayson-group/SmoothTransitions/actions/workflows/publish.yml/badge.svg)\n# Smoothtransitions\n## Overview\n\nSmoothtransitions is a commandline tool for running workflows that aims to identify ground state and/or transition state structures in chemical reactions given a set of starting geometries.\n\nFor each of the candidate structures, a typical workflow might include:\n\n- A constrained optimisation\n- A series of unconstrained optimisations that last until the structure has converged or reached a maximum number of optimisation stages\n- A single point energy calculation on the final optimised structure but only if it previously converged previously during optimisation\n\nRunning this workflow manually can be very laborious and error prone, especially when done for many structures at the same time, but smoothtransitions can make this process a lot easier.\nEach stage in the workflow, and all necessary options for it, can be defined in a single configuration file.\nThis includes:\n\n- SLURM settings for each stage\n- Gaussian settings for each stage\n- Number of iterations for each stage\n- etc.\n\nAfter defining these settings, smoothtransitions will handle the rest of the work. Starting from a folder containing a configuration file and the initial structures as mol2 files, smoothtransitions will:\n\n- create subfolders for each stage and iteration and save properly named input and output files into these subfolders\n- generate and submit the necessary SLURM scripts\n- track the progress of the workflow in a log file\n- upon completion of one SLURM job, determine the next action and submit further jobs as necessary\n\nIf no mol2 files are available, smoothtransitions can also handle conversion from most molecular formats into gaussian input files.\n\nSmoothtransitions offers the following advantages:\n\n- no more mistakes when manually modifying gaussian output files to new input files in order to continue a simulation\n- no more wasted time because jobs finish at night or on the weekend when manual submission of the next batch may not be possible\n- consistent structure for each experiment that can be shared across teams\n\nThis has allowed us to run transition state calculations over hundreds of candidate molecules.\n\nAdditional features in the pipeline include:\n\n- automated generation of graphs such as atom distances\n- automated analysis of SPE runs using tools such as goodvibes\n- automated custom analysis after any stage\n\n## Risks\n\nSmoothtransitions will submit jobs in the users name on a HPC system. This bears the same risks as any other automated way of submitting jobs to a HPC system, and hence care should be taken, especially when using smoothtransitions to submit jobs to paid queues. Depending on its configuration, the application can potentially submit many simulations, generating costs equivalent to the run simulations. The authors of this tool take no responsibility for incurred cost by using this tool.\n\n## Installation\n\nThe only requirement is openbabel (either version 2.4.1 or 3.1.1) and the respecitve Python bindings that come with it.\nThe best way to install them is using conda but on a HPC system `openbabel` and its Python bindings might already be\ninstalled as a module. To check this run `module avail` or open a Python prompt via `python` and see if `import pybel` or `from openbabel import pybel` work.\n\nIf the requirements are fullfilled, installing smoothtransitions is as easy as: `pip install smoothtransitions`\n\nUsing a virtual environment for your smoothtransitions setup is recommended.\n\n### Installing openbabel and smoothtransitions together\nThis assumes that anaconda, miniconda oder miniforge are installed on the system.\n1) `conda create --name smoothtransitions python=3.7` # 3.6 and 3.8 are supported as well\n2) `conda activate smoothtransitions` # depending on your conda version this might also be `source activate smoothtransitions`\n3) `conda install -c conda-forge openbabel=3.1.1`\n4) `pip install smoothtranstions`\n\n### Additional instructions only for users of the Bath HPC Cluster:\n\n- Add `module load anaconda3/2.5.0` into your `.bashrc` file and make sure to DELETE `module load openbabel/2.4.1`\n- Follow the instructions above for installing `openbabel` and `smoothtransitions` together\n\n# Running SmoothTransitions:\n\n## General commands\n\n```\nsmoothtransitions\nUsage: smoothtransitions [OPTIONS] COMMAND [ARGS]...\n  --help                          Show this message and exit.\n\nCommands:\n  io            Function that handles creating gjf files from various input...\n  report        Run the reporting logic for all structures provided.\n  runner        This command will do two things: 1) Determine what needs to...\n  workflow      This function will start or continue a workflow defined by...\n  workflow-log  Small command to print the workflow log nicely in the...\n```\n\n- `smoothtransitions workflow` this is the workhorse of the application as it will start or continue a defined workflow and call other commands internally\n- `smoothtransitions io` allows the generation of Gaussian input files (`*.gjf`) from different input formats\n- `smoothtransitions report` allows manual recording of simulation runs in a project logfile\n- `smoothtransitions runner` allows manual running of individual simulation stages\n- `smoothtransitions workflow-log` prints the log of the current experiment nicely to the command line.\n\nFor a detailed example, see the example folder.\n\n## Structure of a configuration file\n\nThe following describes the structure of a configuration file. The format is always a yaml file.\n\nA config file will always have the following general structure:\n\n```\nproject_name: <some name>\nfirst-stage: <first stage to run>\n\n# one or more stages\n<stage-name>:\n  <configuration of the stage>\n\nsmoothtransitions:\n  <configuration of the smoothtransitions steps>\n```\n\nA stage always needs the following structure, here we call the stage `optimisation`:\n```\noptimisation:\n  next-stage: SPE # a further stage name or "abort"\n  # what to do when the maximum number of iterations is reached\n  after-max-iter: optimisation # a stage name or "abort"\n  max-iter: 1 # number of iterations (slurm runs) to run\n  gaussian:\n    <configuration of the gaussian setup>\n  slurm:\n    <configuration of the slurm job>\n```\n\n### Special sections\n\nThere are currently three special stages, the first two are based on our current workflow and can be used or omitted. The third is a special section for the smoothtransitions commands that need to be run to keep the workflow running.\n\n#### constraint\nThis section defines a stage that has additional requirements in terms of gjf files. In particular it allows us to define bond constraints to be added to the gjf file for running constrained optimisations.\n\n#### SPE\nThis section defines a single point energy calculation which in our typical workflow defines the last step and therefore in some cases is treated differently.\n\n#### smoothtransitions\nThis section defines all the commands that need to be run to keep the workflow running. Smoothtransitions achieves the submission of subsequent jobs in a workflow by running dependent commands. After each gaussian run, a report and runner job are run that depend on the previous in order to log the results of the previous run to the project log (report job) and then figure out and run the next jon (runner).\n\nAn example with just one stage is shown below:\n\n```\nproject_name: "peroxide" # basic information\nfirst-stage: constraint # the workflow needs to know which stage is the first in the workflow\n\nconstraint: # first level keys identify stages or general settings\n  # basic workflow information\n  # which stage to run after the current one\n  next-stage: optimisation # a stage name or "abort"\n  # what to do when the maximum number of iterations is reached\n  after-max-iter: optimisation # a stage name or "abort"\n  max-iter: 1 # number of iterations (slurm runs) to run\n  # section that define all the gaussian settings\n  gaussian:\n    # this section will create the following gaussian header:\n    # ---------------------------------------\n    # %nprocshared=16\n    # %mem=48GB\n\n    # B3LYP/6-31G(d), opt=(calcfc, modredundant, maxcycles=80, maxstep=10), freq=noraman, geom=connectivity\n    # ---------------------------------------\n    # define all link 0 details, see gaussian documentation for valid keywords\n    link_0:\n      nprocshared: "16"\n      mem: "48GB"\n    # define all route details, see gaussian documentation for valid keywords\n    route:\n      # text that appear without a keyword before them need to be declared as such\n      # (e.g. maxcycles below is a keyword before the value "80")\n      non_keywords: "B3LYP/6-31G(d)"\n      opt:\n        non_keywords: "calcfc, modredundant"\n        maxcycles: "80"\n        maxstep: "10"\n      freq: noraman\n      geom: connectivity\n  # additional bond constraints to be added (only for constraint sections)\n  # This will add "B 1 2 F" at the end of the gjf file\n  constraint:\n    - "B 1 2 F"\n  # all settings for the slurm job this will create a slurm submission script\n  slurm:\n    settings:\n      job-name: "constraint" # this should be some overall name\n      time: "00:10:00" # walltime for the job\n      mem: "48000"\n      nodes: "1"\n      ntasks: "1"\n      cpus-per-task: "16"\n      partition: "batch-all"\n      account: "free"\n    # actual command to be run on submission, this will differ for your setup\n    command: |\n      export GAUSS_SCRDIR=$SCRATCH\n\n      module purge\n      module load slurm\n      module load gaussian/16\n\n      source $g16profile\n\n      g16 < $gjf_file > $outputfile\n\n# special section for smoothtransitions jobs that need to run after a gaussian job\nsmoothtransitions:\n  slurm:\n    settings:\n      time: "00:05:00"\n      mem: "48000"\n      nodes: "1"\n      ntasks: "1"\n      cpus-per-task: "16"\n      partition: "batch-all"\n      account: "free"\n\n    # this command will record the previous job in the project log\n    report-command: |\n      module purge\n      module load slurm\n      module load anaconda3/2.5.0\n\n      source activate smoothtransitions\n\n      smoothtransitions report $gaussianoutfile --config-file $configfile --project-file $projectfile\n\n    # this command will record the previous job in the project log\n    runner-command: |\n      module purge\n      module load slurm\n      module load anaconda3/2.5.0\n\n      source activate smoothtransitions\n\n      smoothtransitions runner $gaussianoutfile --config-file $configfile --project-file $projectfile\n```\n',
    'author': 'Florian Roessler',
    'author_email': 'smoothtransitions.github@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7.1,<4.0.0',
}


setup(**setup_kwargs)
