"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const deployment_instance_1 = require("../../core/lib/deployment-instance");
const lib_1 = require("../lib");
const secrets_management_1 = require("../lib/secrets-management");
class MockUserData extends aws_ec2_1.UserData {
    constructor() {
        super();
        this.addCommands = jest.fn();
        this.addCommands = jest.fn();
        this.addOnExitCommands = jest.fn();
        this.render = jest.fn();
        this.addExecuteFileCommand = jest.fn();
        this.addS3DownloadCommand = jest.fn();
        this.addSignalOnExitCommand = jest.fn();
    }
}
class MockDeploymentInstance extends deployment_instance_1.DeploymentInstance {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.mockUserData = new MockUserData();
    }
    get userData() {
        return this.mockUserData;
    }
}
function writeSynthedTemplate(stack, outputFile) {
    const template = assert_1.SynthUtils.synthesize(stack).template;
    fs.writeFileSync(outputFile, JSON.stringify(template, null, 2), { encoding: 'utf8' });
}
const DEADLINE_CLIENT_SUBNET_NAME = 'DeadlineClient';
const RENDER_QUEUE_ALB_SUBNET_NAME = 'RenderQueueALB';
describe('SecretsManagementIdentityRegistration', () => {
    let app;
    let dependencyStack;
    let deploymentInstanceStack;
    let stack;
    let vpc;
    let version;
    let repository;
    let deploymentInstance;
    let deploymentInstanceRole;
    let renderQueueSubnets;
    let target;
    // @ts-ignore
    function writeSynthedTemplates() {
        writeSynthedTemplate(deploymentInstanceStack, 'deployment-instance-stack.json');
        writeSynthedTemplate(stack, 'secrets-management-stack.json');
    }
    beforeEach(() => {
        app = new core_1.App();
        dependencyStack = new core_1.Stack(app, 'DependencyStack');
        deploymentInstanceStack = new core_1.Stack(app, 'DeploymentInstanceStack');
        stack = new core_1.Stack(app, 'Stack');
        vpc = new aws_ec2_1.Vpc(dependencyStack, 'Vpc', {
            subnetConfiguration: [
                {
                    name: RENDER_QUEUE_ALB_SUBNET_NAME,
                    subnetType: aws_ec2_1.SubnetType.PRIVATE,
                    cidrMask: 28,
                },
                {
                    name: 'Public',
                    subnetType: aws_ec2_1.SubnetType.PUBLIC,
                    cidrMask: 28,
                },
                {
                    name: DEADLINE_CLIENT_SUBNET_NAME,
                    subnetType: aws_ec2_1.SubnetType.PUBLIC,
                    cidrMask: 28,
                },
            ],
        });
        version = new lib_1.VersionQuery(dependencyStack, 'Version');
        deploymentInstance = new MockDeploymentInstance(deploymentInstanceStack, 'DeploymentInstance', {
            vpc,
        });
        renderQueueSubnets = vpc.selectSubnets({ subnetGroupName: RENDER_QUEUE_ALB_SUBNET_NAME });
    });
    describe('when Repository uses secrets management', () => {
        beforeEach(() => {
            // GIVEN
            repository = new lib_1.Repository(dependencyStack, 'Repository', {
                version,
                vpc,
                secretsManagementSettings: {
                    enabled: true,
                },
            });
            jest.spyOn(repository, 'configureClientInstance');
            // Get a reference to the DeploymentInstance's IAM role L1 resource
            deploymentInstanceRole = (deploymentInstance
                .node.findChild('ASG')
                .node.findChild('InstanceRole')
                .node.defaultChild);
        });
        function createTarget() {
            target = new secrets_management_1.SecretsManagementIdentityRegistration(stack, 'IdentityRegistration', {
                deploymentInstance,
                renderQueueSubnets,
                repository,
                version,
            });
        }
        describe('Deadline Client installer', () => {
            test('grant S3 read to client installer', () => {
                // WHEN
                createTarget();
                // THEN
                assert_1.expect(deploymentInstanceStack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            Effect: 'Allow',
                            Resource: assert_1.arrayWith(...deploymentInstanceStack.resolve([
                                version.linuxInstallers.client.s3Bucket.bucketArn,
                                version.linuxInstallers.client.s3Bucket.arnForObjects(version.linuxInstallers.client.objectKey),
                            ])),
                        }),
                    },
                    Roles: [
                        deploymentInstanceStack.resolve(deploymentInstanceRole.ref),
                    ],
                }));
            });
            test('downloads and executes Client installer', () => {
                // GIVEN
                const clientInstallerLocalFilename = 'clientInstallerLocalFilename';
                const userData = deploymentInstance.userData;
                userData.addS3DownloadCommand.mockReturnValueOnce(clientInstallerLocalFilename);
                // WHEN
                createTarget();
                // THEN
                expect(userData.addS3DownloadCommand).toHaveBeenCalledWith({
                    bucket: version.linuxInstallers.client.s3Bucket,
                    bucketKey: version.linuxInstallers.client.objectKey,
                });
                expect(userData.addCommands).toHaveBeenCalledWith(`chmod +x "${clientInstallerLocalFilename}"`);
                expect(userData.addCommands).toHaveBeenCalledWith([
                    // This is required b/c USER and HOME environment variables are not defined when running
                    // user-data
                    'sudo', '--login',
                    // Run the Deadline Client installer
                    `"${clientInstallerLocalFilename}"`,
                    '--mode', 'unattended',
                    '--connectiontype', 'Remote',
                    '--proxyrootdir', '127.0.0.1:8080',
                    '--noguimode', 'true',
                    '--slavestartup', 'false',
                    '--launcherdaemon', 'false',
                    '--restartstalled', 'true',
                    '--autoupdateoverride', 'False',
                ].join(' '));
            });
        });
        test('grants DeploymentInstance role permissions to describe subnets', () => {
            // WHEN
            createTarget();
            // THEN
            assert_1.expect(deploymentInstanceStack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: 'ec2:DescribeSubnets',
                        Effect: 'Allow',
                        Resource: '*',
                    }),
                },
                Roles: [stack.resolve(deploymentInstanceRole.ref)],
            }));
        });
        test('sets up Python environment', () => {
            // WHEN
            createTarget();
            // THEN
            // The script requires boto3 to query the subnets CIDR. This script runs
            // as the ec2-user so we install this into the user's package directory
            expect(deploymentInstance.userData.addCommands)
                .toHaveBeenCalledWith('sudo -u ec2-user python3 -m pip install --user boto3');
        });
        test('configures direct repository connection', () => {
            // WHEN
            createTarget();
            // THEN
            expect(repository.configureClientInstance).toHaveBeenCalledWith({
                host: deploymentInstance,
                mountPoint: expect.any(String),
            });
        });
        test('grants DeploymentInstance read access to the Deadline Secrets Management admin credentials secret', () => {
            // WHEN
            createTarget();
            // THEN
            assert_1.expect(deploymentInstanceStack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: deploymentInstanceStack.resolve(repository.secretsManagementSettings.credentials.secretArn),
                    }),
                },
                Roles: [
                    deploymentInstanceStack.resolve(deploymentInstanceRole.ref),
                ],
            }));
        });
        describe('Identity registration settings script', () => {
            function getIdentityRegistrationSettingsScript() {
                return target.node.findChild('ConfigureIdentityRegistrationSettingScript');
            }
            test('DeploymentInstance granted S3 read access', () => {
                // WHEN
                createTarget();
                const identityRegistrationSettingsScript = getIdentityRegistrationSettingsScript();
                // THEN
                assert_1.expect(deploymentInstanceStack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            Effect: 'Allow',
                            Resource: deploymentInstanceStack.resolve([
                                identityRegistrationSettingsScript.bucket.bucketArn,
                                identityRegistrationSettingsScript.bucket.arnForObjects('*'),
                            ]),
                        }),
                    },
                    Roles: [deploymentInstanceStack.resolve(deploymentInstanceRole.ref)],
                }));
            });
            test('DeploymentInstance downloads script', () => {
                // GIVEN
                const identityRegistrationSettingsScriptLocalPath = 'identityRegistrationSettingsScriptLocalPath';
                deploymentInstance.userData.addS3DownloadCommand.mockReturnValueOnce('deadlineClientLocalPath');
                deploymentInstance.userData.addS3DownloadCommand.mockReturnValueOnce(identityRegistrationSettingsScriptLocalPath);
                // WHEN
                createTarget();
                const identityRegistrationSettingsScript = getIdentityRegistrationSettingsScript();
                // THEN
                expect(deploymentInstance.userData.addS3DownloadCommand).toHaveBeenCalledWith({
                    bucket: identityRegistrationSettingsScript.bucket,
                    bucketKey: identityRegistrationSettingsScript.s3ObjectKey,
                    localFile: expect.stringMatching(/^\/home\/ec2-user\//),
                });
            });
            test('DeploymentInstance sets ownership and executable permissions for ec2-user', () => {
                // GIVEN
                const identityRegistrationSettingsScriptLocalPath = 'identityRegistrationSettingsScriptLocalPath';
                (deploymentInstance.userData.addS3DownloadCommand
                    .mockReturnValueOnce('deadlineClientInstallerLocalPath')
                    .mockReturnValueOnce('efsMountScriptLocalPath')
                    .mockReturnValueOnce('directRepoConnectionConfigScriptLocalPath')
                    .mockReturnValueOnce(identityRegistrationSettingsScriptLocalPath));
                // WHEN
                createTarget();
                // THEN
                expect(deploymentInstance.userData.addCommands).toHaveBeenCalledWith(`chmod +x ${identityRegistrationSettingsScriptLocalPath}`, `chown ec2-user:ec2-user ${identityRegistrationSettingsScriptLocalPath}`);
            });
        });
        describe('.addSubnetIdentityRegistrationSetting(...)', () => {
            describe.each([
                [lib_1.SecretsManagementRole.SERVER],
                [lib_1.SecretsManagementRole.CLIENT],
            ])('identityRole=%s', (identityRole) => {
                describe.each([
                    [lib_1.SecretsManagementRegistrationStatus.PENDING],
                    [lib_1.SecretsManagementRegistrationStatus.REGISTERED],
                    [lib_1.SecretsManagementRegistrationStatus.REVOKED],
                ])('registrationStatus=%s', (registrationStatus) => {
                    test('executes identity registration settings configuration script with proper arguments', () => {
                        // GIVEN
                        const identityRegistrationSettingsScriptLocalPath = 'identityRegistrationSettingsScriptLocalPath';
                        (deploymentInstance.userData.addS3DownloadCommand
                            .mockReturnValueOnce('deadlineClientInstallerLocalPath')
                            .mockReturnValueOnce('efsMountScriptLocalPath')
                            .mockReturnValueOnce('directRepoConnectionConfigScriptLocalPath')
                            .mockReturnValueOnce(identityRegistrationSettingsScriptLocalPath));
                        const clientStack = new core_1.Stack(app, 'ClientStack');
                        createTarget();
                        // WHEN
                        target.addSubnetIdentityRegistrationSetting({
                            dependent: new core_1.Construct(clientStack, 'DeadlineClient'),
                            registrationStatus,
                            role: identityRole,
                            vpc,
                            vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                        });
                        // THEN
                        const resolvedCalls = deploymentInstance.userData.addCommands.mock.calls.map(call => {
                            return deploymentInstanceStack.resolve(call);
                        });
                        const expectedCall = [{
                                'Fn::Join': [
                                    '',
                                    [
                                        // Command is run as "ec2-user" which has the database credentials stored
                                        `sudo --login -u ec2-user ${identityRegistrationSettingsScriptLocalPath} `,
                                        stack.resolve(core_1.Fn.join(' ', [
                                            '--region',
                                            stack.region,
                                            // The Deadline Secrets Management admin credentials secret ARN is passed
                                            '--credentials',
                                            `"${repository.secretsManagementSettings.credentials.secretArn}"`,
                                            // The Render Queue's ALB subnets are passed as --connection-subnet args
                                            ...(vpc.selectSubnets({ subnetGroupName: RENDER_QUEUE_ALB_SUBNET_NAME })
                                                .subnetIds.map(subnetID => `--connection-subnet "${subnetID}"`)),
                                            // The Deadline Client's subnets, desired role, and registration status are passed as --source-subnet args
                                            ...(vpc.selectSubnets({ subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME })
                                                .subnetIds.map(subnetID => {
                                                return `--source-subnet "${subnetID},${identityRole},${registrationStatus}"`;
                                            })),
                                        ])),
                                    ],
                                ],
                            }];
                        expect(resolvedCalls).toContainEqual(expectedCall);
                    });
                });
            });
            test('throws execption when using Administrator role', () => {
                // GIVEN
                createTarget();
                // WHEN
                function when() {
                    target.addSubnetIdentityRegistrationSetting({
                        dependent: new core_1.Construct(stack, 'Dependent'),
                        registrationStatus: lib_1.SecretsManagementRegistrationStatus.REGISTERED,
                        role: lib_1.SecretsManagementRole.ADMINISTRATOR,
                        vpc,
                        vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                    });
                }
                // THEN
                expect(when)
                    .toThrowError('The Administrator role cannot be set using a Deadline identity registration setting');
            });
            test('throws when two rules for same source subnet with different roles', () => {
                // GIVEN
                const client1 = new core_1.Construct(stack, 'client1');
                const client2 = new core_1.Construct(stack, 'client2');
                const existingRole = lib_1.SecretsManagementRole.SERVER;
                const newRole = lib_1.SecretsManagementRole.CLIENT;
                createTarget();
                target.addSubnetIdentityRegistrationSetting({
                    dependent: client1,
                    registrationStatus: lib_1.SecretsManagementRegistrationStatus.REGISTERED,
                    role: existingRole,
                    vpc,
                    vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                });
                // WHEN
                function when() {
                    target.addSubnetIdentityRegistrationSetting({
                        dependent: client2,
                        registrationStatus: lib_1.SecretsManagementRegistrationStatus.REGISTERED,
                        role: newRole,
                        vpc,
                        vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                    });
                }
                // THEN
                expect(when)
                    .toThrowError(`Subnet is already registered with role "${existingRole}" but another caller requested "${newRole}"`);
            });
            test('throws when two rules for same source subnet with different registration statuses', () => {
                // GIVEN
                const client1 = new core_1.Construct(stack, 'client1');
                const client2 = new core_1.Construct(stack, 'client2');
                const role = lib_1.SecretsManagementRole.CLIENT;
                const existingStatus = lib_1.SecretsManagementRegistrationStatus.REGISTERED;
                const newStatus = lib_1.SecretsManagementRegistrationStatus.PENDING;
                createTarget();
                target.addSubnetIdentityRegistrationSetting({
                    dependent: client1,
                    registrationStatus: existingStatus,
                    role,
                    vpc,
                    vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                });
                // WHEN
                function when() {
                    target.addSubnetIdentityRegistrationSetting({
                        dependent: client2,
                        registrationStatus: newStatus,
                        role,
                        vpc,
                        vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                    });
                }
                // THEN
                expect(when)
                    .toThrowError(`Subnet is already registered with registrationStatus "${existingStatus}" but another caller requested "${newStatus}"`);
            });
            test('de-duplicates subnets', () => {
                // GIVEN
                const identityRegistrationSettingsScriptLocalPath = 'identityRegistrationSettingsScriptLocalPath';
                (deploymentInstance.userData.addS3DownloadCommand
                    .mockReturnValueOnce('deadlineClientInstallerLocalPath')
                    .mockReturnValueOnce('efsMountScriptLocalPath')
                    .mockReturnValueOnce('directRepoConnectionConfigScriptLocalPath')
                    .mockReturnValueOnce(identityRegistrationSettingsScriptLocalPath));
                const clientStack = new core_1.Stack(app, 'ClientStack');
                const client1 = new core_1.Construct(clientStack, 'client1');
                const client2 = new core_1.Construct(clientStack, 'client2');
                createTarget();
                const baseProps = {
                    registrationStatus: lib_1.SecretsManagementRegistrationStatus.REGISTERED,
                    role: lib_1.SecretsManagementRole.CLIENT,
                    vpc,
                    vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                };
                target.addSubnetIdentityRegistrationSetting({
                    ...baseProps,
                    dependent: client1,
                });
                // WHEN
                target.addSubnetIdentityRegistrationSetting({
                    ...baseProps,
                    dependent: client2,
                });
                // THEN
                const resolvedCalls = deploymentInstance.userData.addCommands.mock.calls.map(call => {
                    return deploymentInstanceStack.resolve(call);
                });
                const expectedCall = [{
                        'Fn::Join': [
                            '',
                            [
                                // Command is run as "ec2-user" which has the database credentials stored
                                `sudo --login -u ec2-user ${identityRegistrationSettingsScriptLocalPath} `,
                                stack.resolve(core_1.Fn.join(' ', [
                                    '--region',
                                    stack.region,
                                    // The Deadline Secrets Management admin credentials secret ARN is passed
                                    '--credentials',
                                    `"${repository.secretsManagementSettings.credentials.secretArn}"`,
                                    // The Render Queue's ALB subnets are passed as --connection-subnet args
                                    ...(vpc.selectSubnets({ subnetGroupName: RENDER_QUEUE_ALB_SUBNET_NAME })
                                        .subnetIds.map(subnetID => `--connection-subnet "${subnetID}"`)),
                                    // The Deadline Client's subnets, desired role, and registration status are passed as --source-subnet args
                                    ...(vpc.selectSubnets({ subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME })
                                        .subnetIds.map(subnetID => {
                                        return `--source-subnet "${subnetID},${baseProps.role},${baseProps.registrationStatus}"`;
                                    })),
                                ])),
                            ],
                        ],
                    }];
                expect(resolvedCalls).toContainEqual(expectedCall);
            });
            test('warns about dedicated subnets when render queue ALB and source subnets match', () => {
                // GIVEN
                createTarget();
                const dependent = new core_1.Construct(stack, 'Dependent');
                const registrationStatus = lib_1.SecretsManagementRegistrationStatus.REGISTERED;
                const role = lib_1.SecretsManagementRole.CLIENT;
                const vpcSubnets = {
                    subnetGroupName: RENDER_QUEUE_ALB_SUBNET_NAME,
                };
                // WHEN
                target.addSubnetIdentityRegistrationSetting({
                    dependent,
                    registrationStatus,
                    role,
                    vpc,
                    vpcSubnets,
                });
                expect(dependent.node.metadataEntry).toContainEqual(expect.objectContaining({
                    type: 'aws:cdk:warning',
                    data: `Deadline Secrets Management is enabled on the Repository and VPC subnets of the Render Queue match the subnets of ${dependent.node.path}. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components`,
                }));
            });
        });
        test('Repository with no admin credentials throws an error', () => {
            // GIVEN
            class RepositoryNoCredentials extends lib_1.Repository {
                constructor(scope, id, props) {
                    super(scope, id, props);
                    this.secretsManagementSettings = {
                        enabled: true,
                        credentials: undefined,
                    };
                }
            }
            repository = new RepositoryNoCredentials(dependencyStack, 'RepoNoCreds', {
                version,
                vpc,
            });
            // WHEN
            const when = createTarget;
            // THEN
            expect(when).toThrowError('Repository does not contain secrets management credentials');
        });
    });
    test('when Repository disables secrets management throws an exception', () => {
        // GIVEN
        repository = new lib_1.Repository(stack, 'Repository', {
            version,
            vpc,
            secretsManagementSettings: {
                enabled: false,
            },
        });
        // WHEN
        function when() {
            new secrets_management_1.SecretsManagementIdentityRegistration(stack, 'IdentityRegistrationSettings', {
                deploymentInstance,
                renderQueueSubnets: vpc.selectSubnets({
                    subnetGroupName: 'RenderQueueALB',
                }),
                repository,
                version,
            });
        }
        // THEN
        expect(when).toThrow();
    });
});
//# sourceMappingURL=data:application/json;base64,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