"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const __1 = require("../..");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const lib_1 = require("../lib");
const asset_constants_1 = require("./asset-constants");
describe('RenderQueue', () => {
    let app;
    let dependencyStack;
    let stack;
    let vpc;
    let rcsImage;
    let images;
    let repository;
    let version;
    let renderQueueVersion;
    let renderQueueCommon;
    // GIVEN
    beforeEach(() => {
        app = new core_1.App();
        dependencyStack = new core_1.Stack(app, 'DepStack');
        vpc = new aws_ec2_1.Vpc(dependencyStack, 'Vpc');
        version = new lib_1.VersionQuery(dependencyStack, 'Version');
        repository = new lib_1.Repository(dependencyStack, 'Repo', {
            version,
            vpc,
        });
        stack = new core_1.Stack(app, 'Stack');
        rcsImage = aws_ecs_1.ContainerImage.fromAsset(__dirname);
        images = {
            remoteConnectionServer: rcsImage,
        };
        renderQueueVersion = new lib_1.VersionQuery(stack, 'Version');
        renderQueueCommon = new lib_1.RenderQueue(stack, 'RenderQueueCommon', {
            images,
            repository,
            version: renderQueueVersion,
            vpc,
        });
    });
    afterEach(() => {
        jest.resetAllMocks();
    });
    test('creates cluster', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Cluster'));
    });
    test('creates service', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service'));
    });
    test('creates task definition', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::TaskDefinition'));
    });
    test('closed ingress by default', () => {
        // THEN
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            // The openListener=true option would create an ingress rule in the listener's SG.
            // make sure that we don't have that.
            SecurityGroupIngress: assert_1.arrayWith(assert_1.objectLike({})),
        }));
    });
    test('creates load balancer with default values', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 1, {
            LoadBalancerAttributes: [
                {
                    Key: 'deletion_protection.enabled',
                    Value: 'true',
                },
            ],
            Scheme: 'internal',
        }));
    });
    test('creates a log group with default prefix of "/renderfarm/"', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::LogRetention', {
            LogGroupName: '/renderfarm/RenderQueueCommon',
            RetentionInDays: 3,
        }));
    });
    test('configure the container log driver', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                assert_1.objectLike({
                    LogConfiguration: {
                        LogDriver: 'awslogs',
                        Options: {
                            'awslogs-group': {
                                'Fn::GetAtt': [
                                    'RenderQueueCommonLogGroupWrapperA0EF7057',
                                    'LogGroupName',
                                ],
                            },
                            'awslogs-stream-prefix': 'RCS',
                            'awslogs-region': { Ref: 'AWS::Region' },
                        },
                    },
                }),
            ],
        }));
    });
    test('child dependencies added', () => {
        // GIVEN
        const host = new aws_ec2_1.Instance(stack, 'Host', {
            vpc,
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        renderQueueCommon.addChildDependency(host);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
            DependsOn: assert_1.arrayWith('RenderQueueCommonLBPublicListener935F5635', 'RenderQueueCommonRCSTask2A4D5EA5', 'RenderQueueCommonAlbEc2ServicePatternService42BEFF4C', 'RenderQueueCommonWaitForStableServiceDB53E266'),
        }, assert_1.ResourcePart.CompleteDefinition));
    });
    describe('renderQueueSize.min', () => {
        describe('defaults to 1', () => {
            function assertSpecifiesMinSize(stackToAssert) {
                assert_1.expect(stackToAssert).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    MinSize: '1',
                }));
            }
            test('renderQueueSize unspecified', () => {
                // THEN
                assertSpecifiesMinSize(stack);
            });
            test('renderQueueSize.min unspecified', () => {
                // GIVEN
                const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    renderQueueSize: {},
                });
                // THEN
                assertSpecifiesMinSize(isolatedStack);
            });
        });
        // Asserts that at least one RCS container and ASG instance must be created.
        test('throws error when minimum size is 0', () => {
            // GIVEN
            const props = {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
                renderQueueSize: {
                    min: 0,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(stack, 'RenderQueue', props);
            })
                // THEN
                .toThrow('renderQueueSize.min capacity must be at least 1: got 0');
        });
        // Deadline before 10.1.10 requires that successive API requests are serviced by a single RCS.
        test('validates Deadline pre 10.1.10 has min value of at most 1', () => {
            // GIVEN
            const min = 2;
            const newStack = new core_1.Stack(app, 'NewStack');
            const versionOld = new lib_1.VersionQuery(newStack, 'VersionOld', { version: '10.1.9' });
            const props = {
                images,
                repository,
                version: versionOld,
                vpc,
                renderQueueSize: {
                    min,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(newStack, 'RenderQueue', props);
            })
                // THEN
                .toThrow(`renderQueueSize.min for Deadline version less than 10.1.10.0 cannot be greater than 1 - got ${min}`);
        });
        // Asserts that when the renderQueueSize.min prop is specified, the underlying ASG's min property is set accordingly.
        test.each([
            [1],
            [2],
            [10],
        ])('configures minimum number of ASG instances to %d', (min) => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            const props = {
                images,
                repository,
                version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                vpc,
                renderQueueSize: {
                    min,
                },
            };
            // WHEN
            new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                MinSize: min.toString(),
            }));
        });
    });
    describe('renderQueueSize.max', () => {
        describe('defaults to 1', () => {
            function assertSpecifiesMaxSize(stackToAssert) {
                assert_1.expect(stackToAssert).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    MaxSize: '1',
                }));
            }
            test('renderQueueSize unspecified', () => {
                // THEN
                assertSpecifiesMaxSize(stack);
            });
            test('renderQueueSize.max unspecified', () => {
                // GIVEN
                const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    renderQueueSize: {},
                });
                // THEN
                assertSpecifiesMaxSize(isolatedStack);
            });
        });
        // Deadline before 10.1.10 requires that successive API requests are serviced by a single RCS.
        test('validates Deadline pre 10.1.10 has max value of at most 1', () => {
            // GIVEN
            const max = 2;
            const newStack = new core_1.Stack(app, 'NewStack');
            const versionOld = new lib_1.VersionQuery(newStack, 'VersionOld', { version: '10.1.9' });
            const props = {
                images,
                repository,
                version: versionOld,
                vpc,
                renderQueueSize: {
                    max,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(newStack, 'RenderQueue', props);
            })
                // THEN
                .toThrow(`renderQueueSize.max for Deadline version less than 10.1.10.0 cannot be greater than 1 - got ${max}`);
        });
        // Asserts that when the renderQueueSize.max prop is specified, the underlying ASG's max property is set accordingly.
        test.each([
            [1],
            [2],
            [10],
        ])('configures maximum number of ASG instances to %d', (max) => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            const props = {
                images,
                repository,
                version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                vpc,
                renderQueueSize: {
                    max,
                },
            };
            // WHEN
            new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                MaxSize: max.toString(),
            }));
        });
    });
    describe('renderQueueSize.desired', () => {
        describe('defaults', () => {
            test('unset ASG desired', () => {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    DesiredCapacity: assert_1.ABSENT,
                }));
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                    DesiredCount: 1,
                }));
            });
        });
        test('validates Deadline pre 10.1.10 has desired value of at most 1', () => {
            // GIVEN
            const desired = 2;
            const newStack = new core_1.Stack(app, 'NewStack');
            const versionOld = new lib_1.VersionQuery(newStack, 'VersionOld', { version: '10.1.9' });
            const props = {
                images,
                repository,
                version: versionOld,
                vpc,
                renderQueueSize: {
                    desired,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(newStack, 'RenderQueue', props);
            })
                // THEN
                .toThrow(`renderQueueSize.desired for Deadline version less than 10.1.10.0 cannot be greater than 1 - got ${desired}`);
        });
        test.each([
            [1],
            [2],
            [10],
        ])('is specified to %d', (desired) => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            const props = {
                images,
                repository,
                version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                vpc,
                renderQueueSize: {
                    desired,
                },
            };
            // WHEN
            new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                DesiredCapacity: desired.toString(),
            }));
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                DesiredCount: desired,
            }));
        });
    });
    describe('trafficEncryption', () => {
        describe('defaults', () => {
            let isolatedStack;
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    trafficEncryption: {},
                };
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            // THEN
            test('to HTTPS internally between ALB and RCS', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
                    Protocol: 'HTTPS',
                    Port: 4433,
                }));
            });
            test('to HTTPS externally between clients and ALB', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Protocol: 'HTTPS',
                    Port: 4433,
                }));
            });
        });
        describe('when interalProtocol is HTTPS', () => {
            let isolatedStack;
            let renderQueue;
            let caCertPemLogicalId;
            let caCertPkcsLogicalId;
            let caCertPkcsPassphraseLogicalId;
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    trafficEncryption: {
                        internalProtocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS,
                    },
                };
                // WHEN
                renderQueue = new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
                caCertPemLogicalId = isolatedStack.getLogicalId(renderQueue.node.findChild('TlsCaCertPem').node.defaultChild);
                const caCertPkcs = renderQueue.node.findChild('TlsRcsCertBundle');
                const caCertPkcsPassphrase = caCertPkcs.node.findChild('Passphrase');
                caCertPkcsLogicalId = isolatedStack.getLogicalId(caCertPkcs.node.defaultChild);
                caCertPkcsPassphraseLogicalId = isolatedStack.getLogicalId(caCertPkcsPassphrase.node.defaultChild);
            });
            // THEN
            test('ALB connects with HTTPS to port 4433', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
                    Protocol: 'HTTPS',
                    Port: 4433,
                }));
            });
            test('creates RCS cert', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('Custom::RFDK_X509Generator', {
                    ServiceToken: {
                        'Fn::GetAtt': assert_1.arrayWith('Arn'),
                    },
                    DistinguishedName: { CN: 'renderfarm.local' },
                    Secret: {
                        NamePrefix: 'IsolatedStack/RenderQueue/TlsCaCertPem',
                    },
                }));
            });
            test('grants read access to secrets containing the certs and passphrase', () => {
                const taskDef = renderQueue.node.findChild('RCSTask');
                const taskRoleLogicalId = isolatedStack.getLogicalId(taskDef.taskRole.node.defaultChild);
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    caCertPemLogicalId,
                                    'Cert',
                                ],
                            },
                        }, {
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    caCertPkcsLogicalId,
                                    'Cert',
                                ],
                            },
                        }, {
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: { Ref: caCertPkcsPassphraseLogicalId },
                        }),
                        Version: '2012-10-17',
                    },
                    Roles: assert_1.arrayWith({ Ref: taskRoleLogicalId }),
                }));
            });
            test('configures environment variables for cert secret URIs', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                    ContainerDefinitions: assert_1.arrayWith(assert_1.deepObjectLike({
                        Environment: assert_1.arrayWith({
                            Name: 'RCS_TLS_CA_CERT_URI',
                            Value: {
                                'Fn::GetAtt': [
                                    caCertPemLogicalId,
                                    'Cert',
                                ],
                            },
                        }, {
                            Name: 'RCS_TLS_CERT_URI',
                            Value: {
                                'Fn::GetAtt': [
                                    caCertPkcsLogicalId,
                                    'Cert',
                                ],
                            },
                        }, {
                            Name: 'RCS_TLS_CERT_PASSPHRASE_URI',
                            Value: { Ref: caCertPkcsPassphraseLogicalId },
                        }),
                    })),
                }));
            });
        });
        describe('when internal protocol is HTTP', () => {
            let isolatedStack;
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const nonSmRepository = new lib_1.Repository(dependencyStack, 'NonSMRepository', {
                    vpc,
                    version,
                    secretsManagementSettings: { enabled: false },
                });
                const props = {
                    images,
                    repository: nonSmRepository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    trafficEncryption: {
                        internalProtocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
                        externalTLS: { enabled: false },
                    },
                };
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            // THEN
            test('no certs are created', () => {
                assert_1.expect(isolatedStack).notTo(assert_1.haveResource('Custom::RFDK_X509Generator'));
            });
            test('ALB connects with HTTP to port 8080', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
                    Protocol: 'HTTP',
                    Port: 8080,
                }));
            });
        });
        describe('externalProtocol is HTTPS', () => {
            let isolatedStack;
            const CERT_ARN = 'certarn';
            const CA_ARN = 'arn:aws:secretsmanager:123456789012:secret:ca/arn';
            const ZONE_NAME = 'renderfarm.local';
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                    vpc,
                    zoneName: ZONE_NAME,
                });
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    trafficEncryption: {
                        externalTLS: {
                            acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Certificate', CERT_ARN),
                            acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretPartialArn(stack, 'CA_Cert', CA_ARN),
                        },
                    },
                    hostname: {
                        hostname: 'renderqueue',
                        zone,
                    },
                };
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            test('sets the listener port to 4433', () => {
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Port: 4433,
                }));
            });
            test('sets the listener protocol to HTTPS', () => {
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Protocol: 'HTTPS',
                }));
            });
            test('configures the ALB listener to use the specified ACM certificate', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Protocol: 'HTTPS',
                    Certificates: assert_1.arrayWith({
                        CertificateArn: CERT_ARN,
                    }),
                }));
            });
            test('raises an error when a cert is specified without a hosted zone', () => {
                // GIVEN
                const props = {
                    images,
                    repository,
                    version: renderQueueVersion,
                    vpc,
                    trafficEncryption: {
                        externalTLS: {
                            acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Cert', 'certArn'),
                            acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretArn(stack, 'CA_Cert2', CA_ARN),
                        },
                    },
                };
                // WHEN
                expect(() => {
                    new lib_1.RenderQueue(stack, 'RenderQueue', props);
                })
                    // THEN
                    .toThrow(/The hostname for the render queue must be defined if supplying your own certificates./);
            });
            test('raises an error when a cert is specified without a hostname', () => {
                // GIVEN
                const zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZoneNoName', {
                    vpc,
                    zoneName: ZONE_NAME,
                });
                const props = {
                    images,
                    repository,
                    version: renderQueueVersion,
                    vpc,
                    trafficEncryption: {
                        externalTLS: {
                            acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Cert', 'certArn'),
                            acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretArn(stack, 'CA_Cert2', CA_ARN),
                        },
                    },
                    hostname: { zone },
                };
                // WHEN
                expect(() => {
                    new lib_1.RenderQueue(stack, 'RenderQueue', props);
                })
                    // THEN
                    .toThrow(/A hostname must be supplied if a certificate is supplied, with the common name of the certificate matching the hostname \+ domain name/);
            });
        });
        describe('externalProtocol is HTTPS importing cert', () => {
            describe('passing cases', () => {
                let isolatedStack;
                let zone;
                const ZONE_NAME = 'renderfarm.local';
                const HOSTNAME = 'server';
                beforeEach(() => {
                    // GIVEN
                    isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                    zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                        vpc,
                        zoneName: ZONE_NAME,
                    });
                    const caCert = new __1.X509CertificatePem(isolatedStack, 'CaCert', {
                        subject: {
                            cn: `ca.${ZONE_NAME}`,
                        },
                    });
                    const serverCert = new __1.X509CertificatePem(isolatedStack, 'ServerCert', {
                        subject: {
                            cn: `${HOSTNAME}.${ZONE_NAME}`,
                        },
                        signingCertificate: caCert,
                    });
                    const nonSmRepository = new lib_1.Repository(dependencyStack, 'NonSMRepository', {
                        vpc,
                        version,
                        secretsManagementSettings: { enabled: false },
                    });
                    const props = {
                        images,
                        repository: nonSmRepository,
                        version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                        vpc,
                        trafficEncryption: {
                            externalTLS: {
                                rfdkCertificate: serverCert,
                            },
                            internalProtocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
                        },
                        hostname: {
                            zone,
                            hostname: HOSTNAME,
                        },
                    };
                    // WHEN
                    new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
                });
                test('sets the listener port to 4433', () => {
                    // THEN
                    assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                        Port: 4433,
                    }));
                });
                test('sets the listener protocol to HTTPS', () => {
                    // THEN
                    assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                        Protocol: 'HTTPS',
                    }));
                });
                test('Imports Cert to ACM', () => {
                    // THEN
                    assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('Custom::RFDK_AcmImportedCertificate', {
                        X509CertificatePem: {
                            Cert: {
                                'Fn::GetAtt': [
                                    'ServerCert',
                                    'Cert',
                                ],
                            },
                            Key: {
                                'Fn::GetAtt': [
                                    'ServerCert',
                                    'Key',
                                ],
                            },
                            Passphrase: {
                                Ref: 'ServerCertPassphraseE4C3CB38',
                            },
                            CertChain: {
                                'Fn::GetAtt': [
                                    'ServerCert',
                                    'CertChain',
                                ],
                            },
                        },
                    }));
                });
            });
            describe('failure cases,', () => {
                test('Throws when missing cert chain', () => {
                    const ZONE_NAME = 'renderfarm.local';
                    const HOSTNAME = 'server';
                    // GIVEN
                    const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                    const zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                        vpc,
                        zoneName: ZONE_NAME,
                    });
                    const rootCert = new __1.X509CertificatePem(isolatedStack, 'RootCert', {
                        subject: {
                            cn: `ca.${ZONE_NAME}`,
                        },
                    });
                    const props = {
                        images,
                        repository,
                        version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                        vpc,
                        trafficEncryption: {
                            externalTLS: {
                                rfdkCertificate: rootCert,
                            },
                            internalProtocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
                        },
                        hostname: {
                            zone,
                            hostname: HOSTNAME,
                        },
                    };
                    // WHEN
                    expect(() => {
                        new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
                    })
                        // THEN
                        .toThrow(/Provided rfdkCertificate does not contain a certificate chain/);
                });
            });
        });
        test('Creates default RFDK cert if no cert given', () => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            const props = {
                images,
                repository,
                version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                vpc,
                trafficEncryption: {
                    externalTLS: {},
                },
            };
            const rq = new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            const rootCa = rq.node.findChild('RootCA').node.defaultChild;
            const rootCaGen = rootCa.node.defaultChild;
            const rfdkCert = rq.node.findChild('RenderQueuePemCert').node.defaultChild;
            const rfdkCertGen = rfdkCert.node.defaultChild;
            const acmCert = rq.node.findChild('AcmCert').node.defaultChild;
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('Custom::RFDK_X509Generator', {
                Passphrase: isolatedStack.resolve(rootCa.passphrase),
            }));
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('Custom::RFDK_X509Generator', {
                Passphrase: isolatedStack.resolve(rfdkCert.passphrase),
                SigningCertificate: {
                    Cert: isolatedStack.resolve(rootCaGen.getAtt('Cert')),
                    Key: isolatedStack.resolve(rootCaGen.getAtt('Key')),
                    Passphrase: isolatedStack.resolve(rootCa.passphrase),
                    CertChain: '',
                },
            }));
            assert_1.expect(isolatedStack).to(assert_1.countResources('Custom::RFDK_AcmImportedCertificate', 1));
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('Custom::RFDK_AcmImportedCertificate', {
                X509CertificatePem: {
                    Cert: isolatedStack.resolve(rfdkCertGen.getAtt('Cert')),
                    Key: isolatedStack.resolve(rfdkCertGen.getAtt('Key')),
                    Passphrase: isolatedStack.resolve(rfdkCert.passphrase),
                    CertChain: isolatedStack.resolve(rfdkCertGen.getAtt('CertChain')),
                },
            }));
            assert_1.expect(isolatedStack).to(assert_1.countResources('AWS::ElasticLoadBalancingV2::Listener', 1));
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                Certificates: [
                    {
                        CertificateArn: isolatedStack.resolve(acmCert.certificateArn),
                    },
                ],
            }));
        });
        test('Throws if given ACM cert and RFDK Cert', () => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            const ZONE_NAME = 'renderfarm.local';
            const CERT_ARN = 'certArn';
            const CA_ARN = 'arn:aws:secretsmanager:123456789012:secret:ca/arn';
            const zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                vpc,
                zoneName: ZONE_NAME,
            });
            const caCert = new __1.X509CertificatePem(isolatedStack, 'CaCert', {
                subject: {
                    cn: `ca.${ZONE_NAME}`,
                },
            });
            const serverCert = new __1.X509CertificatePem(isolatedStack, 'ServerCert', {
                subject: {
                    cn: `server.${ZONE_NAME}`,
                },
                signingCertificate: caCert,
            });
            const props = {
                images,
                repository,
                version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                vpc,
                trafficEncryption: {
                    externalTLS: {
                        acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(isolatedStack, 'Certificate', CERT_ARN),
                        acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretArn(isolatedStack, 'CA_Cert', CA_ARN),
                        rfdkCertificate: serverCert,
                    },
                },
                hostname: {
                    zone,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            })
                // THEN
                .toThrow(/Exactly one of externalTLS.acmCertificate and externalTLS.rfdkCertificate must be provided when using externalTLS/);
        });
        test('Throws if ACM Cert is given without a cert chain', () => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            const HOSTNAME = 'renderqueue';
            const ZONE_NAME = 'renderfarm.local';
            const CERT_ARN = 'certArn';
            const zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                vpc,
                zoneName: ZONE_NAME,
            });
            const props = {
                images,
                repository,
                version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                vpc,
                trafficEncryption: {
                    externalTLS: {
                        acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(isolatedStack, 'Certificate', CERT_ARN),
                    },
                },
                hostname: {
                    hostname: HOSTNAME,
                    zone,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            })
                // THEN
                .toThrow(/externalTLS.acmCertificateChain must be provided when using externalTLS.acmCertificate./);
        });
    });
    describe('Client Connection', () => {
        describe('externalProtocol is http', () => {
            let isolatedStack;
            let zone;
            const ZONE_NAME = 'renderfarm.local';
            let rq;
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                    vpc,
                    zoneName: ZONE_NAME,
                });
                const nonSmRepository = new lib_1.Repository(dependencyStack, 'NonSMRepository', {
                    vpc,
                    version,
                    secretsManagementSettings: { enabled: false },
                });
                const props = {
                    images,
                    repository: nonSmRepository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    hostname: {
                        zone,
                    },
                    trafficEncryption: { externalTLS: { enabled: false } },
                };
                // WHEN
                rq = new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            test('ECS can connect', () => {
                // WHEN
                const hosts = [new aws_ec2_1.Instance(isolatedStack, 'Host', {
                        vpc,
                        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                    })];
                const role = new aws_iam_1.Role(isolatedStack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
                const env = rq.configureClientECS({
                    hosts,
                    grantee: role,
                });
                // THEN
                expect(env).toHaveProperty('RENDER_QUEUE_URI');
                expect(env.RENDER_QUEUE_URI).toMatch(/http:\/\/.*:8080$/);
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 8080,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(hosts[0].connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
                    DependsOn: assert_1.arrayWith('RenderQueueLBPublicListenerBBF15D5F', 'RenderQueueRCSTaskA9AE70D3'),
                }, assert_1.ResourcePart.CompleteDefinition));
            });
            test('Linux Instance can connect', () => {
                // WHEN
                const host = new aws_ec2_1.Instance(isolatedStack, 'Host', {
                    vpc,
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                    machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                });
                rq.configureClientInstance({
                    host,
                });
                // THEN
                const userData = isolatedStack.resolve(host.userData.render());
                expect(userData).toStrictEqual({
                    'Fn::Join': [
                        '',
                        [
                            "#!/bin/bash\nmkdir -p $(dirname '/tmp/",
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\')\naws s3 cp \'s3://',
                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                            '/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' \'/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\'\n' +
                                'if [ -f \"/etc/profile.d/deadlineclient.sh\" ]; then\n' +
                                '  source \"/etc/profile.d/deadlineclient.sh\"\n' +
                                'fi\n' +
                                '"${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            `" --render-queue "http://renderqueue.${ZONE_NAME}:8080" \n` +
                                'rm -f "/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '"\n' +
                                'if service --status-all | grep -q "Deadline 10 Launcher"; then\n' +
                                '  service deadline10launcher restart\n' +
                                'fi',
                        ],
                    ],
                });
                // Make sure we execute the script with the correct args
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 8080,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(host.connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
                    DependsOn: assert_1.arrayWith('RenderQueueLBPublicListenerBBF15D5F', 'RenderQueueRCSTaskA9AE70D3'),
                }, assert_1.ResourcePart.CompleteDefinition));
            });
            test('Windows Instance can connect', () => {
                // WHEN
                const host = new aws_ec2_1.Instance(isolatedStack, 'Host', {
                    vpc,
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                    machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_CORE_BASE),
                });
                rq.configureClientInstance({
                    host,
                });
                // THEN
                const userData = isolatedStack.resolve(host.userData.render());
                expect(userData).toStrictEqual({
                    'Fn::Join': [
                        '',
                        [
                            '<powershell>mkdir (Split-Path -Path \'C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' ) -ea 0\n' +
                                'Read-S3Object -BucketName \'',
                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                            '\' -key \'',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' -file \'C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' -ErrorAction Stop\n' +
                                '$ErrorActionPreference = "Stop"\n' +
                                '$DEADLINE_PATH = (get-item env:"DEADLINE_PATH").Value\n' +
                                '& "$DEADLINE_PATH/deadlinecommand.exe" -executeScriptNoGui "C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            `" --render-queue "http://renderqueue.${ZONE_NAME}:8080"  2>&1\n` +
                                'Remove-Item -Path "C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '"\n' +
                                'If (Get-Service "deadline10launcherservice" -ErrorAction SilentlyContinue) {\n' +
                                '  Restart-Service "deadline10launcherservice"\n' +
                                '} Else {\n' +
                                '  & "$DEADLINE_PATH/deadlinelauncher.exe" -shutdownall 2>&1\n' +
                                '  & "$DEADLINE_PATH/deadlinelauncher.exe" -nogui 2>&1\n' +
                                '}</powershell>',
                        ],
                    ],
                });
                // Make sure we execute the script with the correct args
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 8080,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(host.connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
                    DependsOn: assert_1.arrayWith('RenderQueueLBPublicListenerBBF15D5F', 'RenderQueueRCSTaskA9AE70D3'),
                }, assert_1.ResourcePart.CompleteDefinition));
            });
        });
        describe('externalProtocol is https', () => {
            let isolatedStack;
            let zone;
            let rq;
            const HOSTNAME = 'renderqueue';
            const ZONE_NAME = 'renderfarm.local';
            const CERT_ARN = 'arn:a:b:c:dcertarn';
            const CA_ARN = 'arn:aws:secretsmanager:123456789012:secret:ca/arn';
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                    vpc,
                    zoneName: ZONE_NAME,
                });
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    hostname: {
                        hostname: HOSTNAME,
                        zone,
                    },
                    trafficEncryption: {
                        externalTLS: {
                            acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Certificate', CERT_ARN),
                            acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretArn(stack, 'CA_Cert', CA_ARN),
                        },
                    },
                };
                // WHEN
                rq = new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            test('ECS can connect', () => {
                // WHEN
                const hosts = [new aws_ec2_1.Instance(isolatedStack, 'Host', {
                        vpc,
                        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                    })];
                const role = new aws_iam_1.Role(isolatedStack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
                const env = rq.configureClientECS({
                    hosts,
                    grantee: role,
                });
                // THEN
                expect(env).toHaveProperty('RENDER_QUEUE_URI');
                expect(env.RENDER_QUEUE_URI).toMatch(/https:\/\/.*:4433$/);
                expect(env).toHaveProperty('RENDER_QUEUE_TLS_CA_CERT_URI', CA_ARN);
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 4433,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(hosts[0].connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
            });
            test('Linux Instance can connect', () => {
                // WHEN
                const host = new aws_ec2_1.Instance(isolatedStack, 'Host', {
                    vpc,
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                    machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                });
                rq.configureClientInstance({
                    host,
                });
                // THEN
                const userData = isolatedStack.resolve(host.userData.render());
                expect(userData).toStrictEqual({
                    'Fn::Join': [
                        '',
                        [
                            "#!/bin/bash\nmkdir -p $(dirname '/tmp/",
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\')\naws s3 cp \'s3://',
                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                            '/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' \'/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\'\n' +
                                'if [ -f \"/etc/profile.d/deadlineclient.sh\" ]; then\n' +
                                '  source \"/etc/profile.d/deadlineclient.sh\"\n' +
                                'fi\n' +
                                '"${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            `" --render-queue "https://renderqueue.${ZONE_NAME}:4433" --tls-ca "${CA_ARN}"\n` +
                                'rm -f "/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '"\n' +
                                'if service --status-all | grep -q "Deadline 10 Launcher"; then\n' +
                                '  service deadline10launcher restart\n' +
                                'fi',
                        ],
                    ],
                });
                // Make sure we execute the script with the correct args
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 4433,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(host.connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
            });
            test('Windows Instance can connect', () => {
                // WHEN
                const host = new aws_ec2_1.Instance(isolatedStack, 'Host', {
                    vpc,
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                    machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_CORE_BASE),
                });
                rq.configureClientInstance({
                    host,
                });
                // THEN
                const userData = isolatedStack.resolve(host.userData.render());
                expect(userData).toStrictEqual({
                    'Fn::Join': [
                        '',
                        [
                            '<powershell>mkdir (Split-Path -Path \'C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' ) -ea 0\n' +
                                'Read-S3Object -BucketName \'',
                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                            '\' -key \'',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' -file \'C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' -ErrorAction Stop\n' +
                                '$ErrorActionPreference = "Stop"\n' +
                                '$DEADLINE_PATH = (get-item env:"DEADLINE_PATH").Value\n' +
                                '& "$DEADLINE_PATH/deadlinecommand.exe" -executeScriptNoGui "C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            `" --render-queue "https://renderqueue.${ZONE_NAME}:4433" --tls-ca \"${CA_ARN}\" 2>&1\n` +
                                'Remove-Item -Path "C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '"\n' +
                                'If (Get-Service "deadline10launcherservice" -ErrorAction SilentlyContinue) {\n' +
                                '  Restart-Service "deadline10launcherservice"\n' +
                                '} Else {\n' +
                                '  & "$DEADLINE_PATH/deadlinelauncher.exe" -shutdownall 2>&1\n' +
                                '  & "$DEADLINE_PATH/deadlinelauncher.exe" -nogui 2>&1\n' +
                                '}</powershell>',
                        ],
                    ],
                });
                // Make sure we execute the script with the correct args
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 4433,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(host.connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
            });
        });
    });
    test('can specify subnets', () => {
        // GIVEN
        const subnets = [
            aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet1', {
                subnetId: 'SubnetID1',
                availabilityZone: 'us-west-2a',
            }),
            aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet2', {
                subnetId: 'SubnetID2',
                availabilityZone: 'us-west-2b',
            }),
        ];
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        const props = {
            images,
            repository,
            version: new lib_1.VersionQuery(isolatedStack, 'Version'),
            vpc,
            vpcSubnets: {
                subnets,
            },
            vpcSubnetsAlb: {
                subnets,
            },
        };
        // WHEN
        new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
        assert_1.expect(isolatedStack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
            VPCZoneIdentifier: assert_1.arrayWith('SubnetID1', 'SubnetID2'),
        }));
        assert_1.expect(isolatedStack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Subnets: [
                'SubnetID1',
                'SubnetID2',
            ],
        }));
    });
    test('can specify instance type', () => {
        // GIVEN
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        const props = {
            images,
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.C5, aws_ec2_1.InstanceSize.LARGE),
            repository,
            version: new lib_1.VersionQuery(isolatedStack, 'Version'),
            vpc,
        };
        // WHEN
        new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
        // THEN
        assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            InstanceType: 'c5.large',
        }));
    });
    test('no deletion protection', () => {
        // GIVEN
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        const props = {
            images,
            repository,
            version: new lib_1.VersionQuery(isolatedStack, 'Version'),
            vpc,
            deletionProtection: false,
        };
        // WHEN
        new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
        // THEN
        assert_1.expect(isolatedStack).to(assert_1.not(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: assert_1.arrayWith({
                Key: 'deletion_protection.enabled',
                Value: 'true',
            }),
            Scheme: assert_1.ABSENT,
            Type: assert_1.ABSENT,
        })));
    });
    test('drop invalid http header fields enabled', () => {
        // GIVEN
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        const props = {
            images,
            repository,
            version: new lib_1.VersionQuery(isolatedStack, 'Version'),
            vpc,
        };
        // WHEN
        new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
        // THEN
        assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: assert_1.arrayWith({
                Key: 'routing.http.drop_invalid_header_fields.enabled',
                Value: 'true',
            }),
        }));
    });
    describe('hostname', () => {
        // GIVEN
        const zoneName = 'mydomain.local';
        describe('not specified, with no TLS', () => {
            let isolatedStack;
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const nonSmRepository = new lib_1.Repository(dependencyStack, 'NonSMRepository', {
                    vpc,
                    version,
                    secretsManagementSettings: { enabled: false },
                });
                const props = {
                    images,
                    repository: nonSmRepository,
                    trafficEncryption: { externalTLS: { enabled: false } },
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                };
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            // THEN
            test('does not create a record set', () => {
                assert_1.expect(isolatedStack).notTo(assert_1.haveResource('AWS::Route53::RecordSet'));
            });
        });
        test('not specified, with TLS', () => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            const props = {
                images,
                repository,
                version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                vpc,
                trafficEncryption: {
                    externalTLS: {},
                },
            };
            const renderQueue = new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            assert_1.expect(isolatedStack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
                Name: 'renderqueue.aws-rfdk.com.',
                Type: 'A',
                AliasTarget: assert_1.objectLike({
                    HostedZoneId: isolatedStack.resolve(renderQueue.loadBalancer.loadBalancerCanonicalHostedZoneId),
                }),
            }));
        });
        describe('specified with zone but no hostname', () => {
            let zone;
            let isolatedStack;
            let renderQueue;
            beforeEach(() => {
                // GIVEN
                zone = new aws_route53_1.PrivateHostedZone(dependencyStack, 'Zone', {
                    vpc,
                    zoneName,
                });
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const props = {
                    images,
                    repository,
                    version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                    vpc,
                    hostname: {
                        zone,
                    },
                };
                // WHEN
                renderQueue = new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            // THEN
            test('creates a record set using default hostname', () => {
                const loadBalancerLogicalId = dependencyStack.getLogicalId(renderQueue.loadBalancer.node.defaultChild);
                assert_1.expect(isolatedStack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
                    Name: `renderqueue.${zoneName}.`,
                    Type: 'A',
                    AliasTarget: assert_1.objectLike({
                        HostedZoneId: {
                            'Fn::GetAtt': [
                                loadBalancerLogicalId,
                                'CanonicalHostedZoneID',
                            ],
                        },
                    }),
                }));
            });
        });
        test.each([
            [false],
            [true],
        ])('specified with TLS enabled == %s', (isTlsEnabled) => {
            // GIVEN
            const zone = new aws_route53_1.PrivateHostedZone(dependencyStack, 'Zone', {
                vpc,
                zoneName,
            });
            const hostname = 'testrq';
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            const nonSmRepository = new lib_1.Repository(dependencyStack, 'NonSMRepository', {
                vpc,
                version,
                secretsManagementSettings: { enabled: false },
            });
            const props = {
                images,
                repository: nonSmRepository,
                version: new lib_1.VersionQuery(isolatedStack, 'Version'),
                vpc,
                hostname: {
                    hostname,
                    zone,
                },
                trafficEncryption: {
                    externalTLS: { enabled: isTlsEnabled },
                },
            };
            // WHEN
            const renderQueue = new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            // THEN
            const loadBalancerLogicalId = dependencyStack.getLogicalId(renderQueue.loadBalancer.node.defaultChild);
            assert_1.expect(isolatedStack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
                Name: `${hostname}.${zoneName}.`,
                Type: 'A',
                AliasTarget: assert_1.objectLike({
                    HostedZoneId: {
                        'Fn::GetAtt': [
                            loadBalancerLogicalId,
                            'CanonicalHostedZoneID',
                        ],
                    },
                }),
            }));
        });
        test.each([
            ['rq.somedomain.local'],
            ['1startswithnumber'],
            ['-startswithhyphen'],
            ['endswith-'],
            ['contains$symbol'],
            ['thisstringisexactlysixtyfourcharacterswhichisonelargerthanthemax'],
        ])('.hostname validation - %s', (hostname) => {
            // GIVEN
            const zone = new aws_route53_1.PrivateHostedZone(dependencyStack, 'Zone', {
                zoneName: 'somedomain.local',
                vpc,
            });
            const props = {
                images,
                repository,
                version,
                vpc,
                hostname: {
                    hostname,
                    zone,
                },
            };
            // WHEN
            function when() {
                new lib_1.RenderQueue(stack, 'NewRenderQueue', props);
            }
            // THEN
            expect(when).toThrow(/Invalid RenderQueue hostname/);
        });
    });
    describe('Access Logs', () => {
        let isolatedStack;
        let isolatedVpc;
        let isolatedRepository;
        let isolatedVersion;
        let isolatedimages;
        let accessBucket;
        beforeEach(() => {
            // GIVEN
            isolatedStack = new core_1.Stack(app, 'IsolatedStack', {
                env: {
                    region: 'us-east-1',
                },
            });
            isolatedVpc = new aws_ec2_1.Vpc(isolatedStack, 'Vpc');
            isolatedVersion = new lib_1.VersionQuery(isolatedStack, 'Version');
            isolatedRepository = new lib_1.Repository(isolatedStack, 'Repo', {
                version: isolatedVersion,
                vpc: isolatedVpc,
            });
            isolatedimages = {
                remoteConnectionServer: rcsImage,
            };
            accessBucket = new aws_s3_1.Bucket(isolatedStack, 'AccessBucket');
        });
        test('enabling access logs sets attributes and policies', () => {
            // GIVEN
            const props = {
                images: isolatedimages,
                repository: isolatedRepository,
                version: isolatedVersion,
                vpc: isolatedVpc,
                accessLogs: {
                    destinationBucket: accessBucket,
                },
            };
            // WHEN
            new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
                LoadBalancerAttributes: assert_1.arrayWith({
                    Key: 'access_logs.s3.enabled',
                    Value: 'true',
                }, {
                    Key: 'access_logs.s3.bucket',
                    Value: {
                        Ref: 'AccessBucketE2803D76',
                    },
                }),
            }));
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
                Bucket: {
                    Ref: 'AccessBucketE2803D76',
                },
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: 's3:PutObject',
                        Condition: {
                            StringEquals: {
                                's3:x-amz-acl': 'bucket-owner-full-control',
                            },
                        },
                        Effect: 'Allow',
                        Principal: {
                            Service: 'delivery.logs.amazonaws.com',
                        },
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'AccessBucketE2803D76',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    }, {
                        Action: 's3:GetBucketAcl',
                        Effect: 'Allow',
                        Principal: {
                            Service: 'delivery.logs.amazonaws.com',
                        },
                        Resource: {
                            'Fn::GetAtt': [
                                'AccessBucketE2803D76',
                                'Arn',
                            ],
                        },
                    }, {
                        Action: [
                            's3:PutObject*',
                            's3:Abort*',
                        ],
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::127311923021:root',
                                    ],
                                ],
                            },
                        },
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'AccessBucketE2803D76',
                                            'Arn',
                                        ],
                                    },
                                    '/AWSLogs/',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    }),
                },
            }));
        });
        test('enabling access logs works with prefix', () => {
            // GIVEN
            const props = {
                images: isolatedimages,
                repository: isolatedRepository,
                version: isolatedVersion,
                vpc: isolatedVpc,
                accessLogs: {
                    destinationBucket: accessBucket,
                    prefix: 'PREFIX_STRING',
                },
            };
            // WHEN
            new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
                LoadBalancerAttributes: assert_1.arrayWith({
                    Key: 'access_logs.s3.enabled',
                    Value: 'true',
                }, {
                    Key: 'access_logs.s3.bucket',
                    Value: {
                        Ref: 'AccessBucketE2803D76',
                    },
                }, {
                    Key: 'access_logs.s3.prefix',
                    Value: 'PREFIX_STRING',
                }),
            }));
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
                Bucket: {
                    Ref: 'AccessBucketE2803D76',
                },
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: 's3:PutObject',
                        Condition: {
                            StringEquals: {
                                's3:x-amz-acl': 'bucket-owner-full-control',
                            },
                        },
                        Effect: 'Allow',
                        Principal: {
                            Service: 'delivery.logs.amazonaws.com',
                        },
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'AccessBucketE2803D76',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    }, {
                        Action: 's3:GetBucketAcl',
                        Effect: 'Allow',
                        Principal: {
                            Service: 'delivery.logs.amazonaws.com',
                        },
                        Resource: {
                            'Fn::GetAtt': [
                                'AccessBucketE2803D76',
                                'Arn',
                            ],
                        },
                    }, {
                        Action: [
                            's3:PutObject*',
                            's3:Abort*',
                        ],
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::127311923021:root',
                                    ],
                                ],
                            },
                        },
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'AccessBucketE2803D76',
                                            'Arn',
                                        ],
                                    },
                                    '/PREFIX_STRING/AWSLogs/',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    }),
                },
            }));
        });
    });
    describe('tagging', () => {
        tag_helpers_1.testConstructTags({
            constructName: 'RenderQueue',
            createConstruct: () => {
                return stack;
            },
            resourceTypeCounts: {
                'AWS::ECS::Cluster': 1,
                'AWS::EC2::SecurityGroup': 2,
                'AWS::IAM::Role': 10,
                'AWS::AutoScaling::AutoScalingGroup': 1,
                'AWS::Lambda::Function': 6,
                'AWS::SNS::Topic': 1,
                'AWS::ECS::TaskDefinition': 1,
                'AWS::DynamoDB::Table': 5,
                'AWS::SecretsManager::Secret': 4,
                'AWS::ElasticLoadBalancingV2::LoadBalancer': 1,
                'AWS::ElasticLoadBalancingV2::TargetGroup': 1,
                'AWS::ECS::Service': 1,
            },
        });
    });
    describe('SEP Policies', () => {
        test('with resource tracker', () => {
            // WHEN
            renderQueueCommon.addSEPPolicies();
            // THEN
            assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::IAM::Role', 1, {
                ManagedPolicyArns: assert_1.arrayWith({
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':iam::aws:policy/AWSThinkboxDeadlineSpotEventPluginAdminPolicy',
                        ],
                    ],
                }, {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':iam::aws:policy/AWSThinkboxDeadlineResourceTrackerAdminPolicy',
                        ],
                    ],
                }),
            }));
        });
        test('no resource tracker', () => {
            // WHEN
            renderQueueCommon.addSEPPolicies(false);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                ManagedPolicyArns: assert_1.arrayWith({
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':iam::aws:policy/AWSThinkboxDeadlineSpotEventPluginAdminPolicy',
                        ],
                    ],
                }),
            }));
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::IAM::Role', {
                ManagedPolicyArns: assert_1.arrayWith({
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':iam::aws:policy/AWSThinkboxDeadlineResourceTrackerAdminPolicy',
                        ],
                    ],
                }),
            }));
        });
    });
    test('creates WaitForStableService by default', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_WaitForStableService', {
            cluster: stack.resolve(renderQueueCommon.cluster.clusterArn),
            // eslint-disable-next-line dot-notation
            services: [stack.resolve(renderQueueCommon['pattern'].service.serviceArn)],
        }));
    });
    describe('Security Groups', () => {
        let backendSecurityGroup;
        let frontendSecurityGroup;
        beforeEach(() => {
            backendSecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'ASGSecurityGroup', { vpc });
            frontendSecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'LBSecurityGroup', { vpc });
        });
        test('adds security groups on construction', () => {
            // GIVEN
            const securityGroups = {
                backend: backendSecurityGroup,
                frontend: frontendSecurityGroup,
            };
            // WHEN
            new lib_1.RenderQueue(stack, 'RenderQueue', {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
                securityGroups,
            });
            // THEN
            assertSecurityGroupsWereAdded(securityGroups);
        });
        test('adds backend security groups post-construction', () => {
            // GIVEN
            const renderQueue = new lib_1.RenderQueue(stack, 'RenderQueue', {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
            });
            // WHEN
            renderQueue.addBackendSecurityGroups(backendSecurityGroup);
            // THEN
            assertSecurityGroupsWereAdded({
                backend: backendSecurityGroup,
            });
        });
        test('adds frontend security groups post-construction', () => {
            // GIVEN
            const renderQueue = new lib_1.RenderQueue(stack, 'RenderQueue', {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
            });
            // WHEN
            renderQueue.addFrontendSecurityGroups(frontendSecurityGroup);
            // THEN
            assertSecurityGroupsWereAdded({
                frontend: frontendSecurityGroup,
            });
        });
        test('security groups added post-construction are not attached to Connections object', () => {
            // GIVEN
            const renderQueue = new lib_1.RenderQueue(stack, 'RenderQueue', {
                images,
                repository,
                version: renderQueueVersion,
                vpc,
            });
            renderQueue.addBackendSecurityGroups(backendSecurityGroup);
            renderQueue.addFrontendSecurityGroups(frontendSecurityGroup);
            const peerSecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'PeerSecurityGroup', { vpc });
            // WHEN
            renderQueue.connections.allowFrom(peerSecurityGroup, aws_ec2_1.Port.tcp(22));
            // THEN
            // Existing LoadBalancer security groups shouldn't have the ingress rule added
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                FromPort: 22,
                ToPort: 22,
                GroupId: stack.resolve(frontendSecurityGroup.securityGroupId),
                SourceSecurityGroupId: stack.resolve(peerSecurityGroup.securityGroupId),
            }));
            // Existing AutoScalingGroup security groups shouldn't have the ingress rule added
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                FromPort: 22,
                ToPort: 22,
                GroupId: stack.resolve(backendSecurityGroup.securityGroupId),
                SourceSecurityGroupId: stack.resolve(peerSecurityGroup.securityGroupId),
            }));
        });
        function assertSecurityGroupsWereAdded(securityGroups) {
            if (securityGroups.backend !== undefined) {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
                    SecurityGroups: assert_1.arrayWith(stack.resolve(securityGroups.backend.securityGroupId)),
                }));
            }
            if (securityGroups.frontend !== undefined) {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
                    SecurityGroups: assert_1.arrayWith(stack.resolve(securityGroups.frontend.securityGroupId)),
                }));
            }
        }
    });
    test('validates VersionQuery is not in a different stack', () => {
        // GIVEN
        const newStack = new core_1.Stack(app, 'NewStack');
        // WHEN
        new lib_1.RenderQueue(newStack, 'RenderQueueNew', {
            images,
            repository,
            version,
            vpc,
        });
        // WHEN
        function synth() {
            assert_1.SynthUtils.synthesize(newStack);
        }
        // THEN
        expect(synth).toThrow('A VersionQuery can not be supplied from a different stack');
    });
    test('Does not enable filesystem cache by default', () => {
        assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            UserData: {
                'Fn::Base64': {
                    'Fn::Join': assert_1.arrayWith(assert_1.arrayWith(' >> /etc/ecs/ecs.config\nsudo iptables --insert FORWARD 1 --in-interface docker+ --destination 169.254.169.254/32 --jump DROP\nsudo service iptables save\necho ECS_AWSVPC_BLOCK_IMDS=true >> /etc/ecs/ecs.config\nyum install -yq awscli unzip\n# RenderQueue file caching enabled\nmkdir -p $(dirname \'/tmp/')),
                },
            },
        }));
    });
    test('Enables filesystem cache if required', () => {
        // GIVEN
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        // WHEN
        new lib_1.RenderQueue(isolatedStack, 'RenderQueue', {
            images,
            repository,
            version: new lib_1.VersionQuery(isolatedStack, 'Version'),
            vpc,
            enableLocalFileCaching: true,
        });
        // THEN
        // Note: If this test breaks/fails, then it is probable that the
        //  'Does not enable filesystem cache by default' test above will also require
        //  updating/fixing.
        assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            UserData: {
                'Fn::Base64': {
                    'Fn::Join': assert_1.arrayWith(assert_1.arrayWith(' >> /etc/ecs/ecs.config\nsudo iptables --insert FORWARD 1 --in-interface docker+ --destination 169.254.169.254/32 --jump DROP\nsudo service iptables save\necho ECS_AWSVPC_BLOCK_IMDS=true >> /etc/ecs/ecs.config\nyum install -yq awscli unzip\n# RenderQueue file caching enabled\nmkdir -p $(dirname \'/tmp/')),
                },
            },
        }));
    });
    test('runs as RCS user', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: assert_1.arrayWith(assert_1.objectLike({ User: '1000:1000' })),
        }));
    });
    describe('Secrets Management', () => {
        let rqSecretsManagementProps;
        beforeEach(() => {
            rqSecretsManagementProps = {
                vpc,
                images,
                repository,
                version: renderQueueVersion,
                trafficEncryption: {
                    internalProtocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS,
                    externalTLS: { enabled: true },
                },
            };
        });
        test('throws if internal protocol is not HTTPS', () => {
            // WHEN
            expect(() => new lib_1.RenderQueue(stack, 'SecretsManagementRenderQueue', {
                ...rqSecretsManagementProps,
                trafficEncryption: {
                    internalProtocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
                },
            }))
                // THEN
                .toThrowError(/The internal protocol on the Render Queue is not HTTPS./);
        });
        test('throws if external TLS is not enabled', () => {
            // WHEN
            expect(() => new lib_1.RenderQueue(stack, 'SecretsManagementRenderQueue', {
                ...rqSecretsManagementProps,
                trafficEncryption: {
                    externalTLS: { enabled: false },
                },
            }))
                // THEN
                .toThrowError(/External TLS on the Render Queue is not enabled./);
        });
        test('throws if repository does not have SM credentials', () => {
            // WHEN
            expect(() => new lib_1.RenderQueue(stack, 'SecretsManagementRenderQueue', {
                ...rqSecretsManagementProps,
                repository: {
                    ...repository,
                    secretsManagementSettings: {
                        ...repository.secretsManagementSettings,
                        credentials: undefined,
                    },
                },
            }))
                // THEN
                .toThrowError(/The Repository does not have Secrets Management credentials/);
        });
        test('throws if deadline version is too low', () => {
            // GIVEN
            const oldVersion = new lib_1.VersionQuery(new core_1.Stack(app, 'OldDeadlineVersionStack'), 'OldDeadlineVersion', { version: '10.0.0.0' });
            // WHEN
            expect(() => new lib_1.RenderQueue(stack, 'SecretsManagementRenderQueue', {
                ...rqSecretsManagementProps,
                version: oldVersion,
            }))
                // THEN
                /* eslint-disable-next-line dot-notation */
                .toThrowError(`The supplied Deadline version (${oldVersion.versionString}) does not support Deadline Secrets Management in RFDK. Either upgrade Deadline to the minimum required version (${lib_1.Version.MINIMUM_SECRETS_MANAGEMENT_VERSION.versionString}) or disable the feature in the Repository's construct properties.`);
        });
        test('grants read permissions to secrets management credentials', () => {
            // WHEN
            const rq = new lib_1.RenderQueue(stack, 'SecretsManagementRenderQueue', rqSecretsManagementProps);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: assert_1.objectLike({
                    Statement: assert_1.arrayWith({
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: stack.resolve(repository.secretsManagementSettings.credentials.node.defaultChild.ref),
                    }),
                }),
                Roles: [stack.resolve(rq.node.tryFindChild('RCSTask').taskRole.roleName)],
            }));
        });
        test('defines secrets management credentials environment variable', () => {
            // WHEN
            new lib_1.RenderQueue(stack, 'SecretsManagementRenderQueue', rqSecretsManagementProps);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: assert_1.arrayWith(assert_1.objectLike({
                    Environment: assert_1.arrayWith({
                        Name: 'RCS_SM_CREDENTIALS_URI',
                        Value: stack.resolve(repository.secretsManagementSettings.credentials.node.defaultChild.ref),
                    }),
                })),
            }));
        });
        test('creates and mounts docker volume for deadline key pairs', () => {
            // WHEN
            new lib_1.RenderQueue(stack, 'SecretsManagementRenderQueue', rqSecretsManagementProps);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: assert_1.arrayWith(assert_1.objectLike({
                    MountPoints: assert_1.arrayWith({
                        ContainerPath: '/home/ec2-user/.config/.mono/keypairs',
                        ReadOnly: false,
                        SourceVolume: 'deadline-user-keypairs',
                    }),
                })),
                Volumes: assert_1.arrayWith({
                    DockerVolumeConfiguration: {
                        Autoprovision: true,
                        Driver: 'local',
                        Scope: 'shared',
                    },
                    Name: 'deadline-user-keypairs',
                }),
            }));
        });
        describe('client calls .configureSecretsManagementAutoRegistration()', () => {
            let callParams;
            let clientInstance;
            let identityRegistrationSettings;
            let launchConfiguration;
            let rqVpcSubnets;
            const RQ_SUBNET_IDS = ['SubnetID1', 'SubnetID2'];
            beforeEach(() => {
                // GIVEN
                const subnets = [
                    aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet1', {
                        subnetId: RQ_SUBNET_IDS[0],
                        availabilityZone: 'us-west-2a',
                    }),
                    aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet2', {
                        subnetId: RQ_SUBNET_IDS[1],
                        availabilityZone: 'us-west-2b',
                    }),
                ];
                rqVpcSubnets = {
                    subnets,
                };
                const rq = new lib_1.RenderQueue(stack, 'SecretsManagementRenderQueue', {
                    ...rqSecretsManagementProps,
                    vpcSubnets: rqVpcSubnets,
                });
                clientInstance = new aws_ec2_1.Instance(stack, 'ClientInstance', {
                    instanceType: new aws_ec2_1.InstanceType('t3.micro'),
                    machineImage: new aws_ec2_1.AmazonLinuxImage(),
                    vpc,
                });
                callParams = {
                    dependent: clientInstance,
                    registrationStatus: lib_1.SecretsManagementRegistrationStatus.REGISTERED,
                    role: lib_1.SecretsManagementRole.CLIENT,
                    vpc,
                    vpcSubnets: { subnetType: aws_ec2_1.SubnetType.PRIVATE },
                };
                launchConfiguration = (
                // @ts-ignore
                rq.deploymentInstance
                    .node.findChild('ASG')
                    .node.findChild('LaunchConfig'));
                // @ts-ignore
                identityRegistrationSettings = rq.identityRegistrationSettings;
                jest.spyOn(identityRegistrationSettings, 'addSubnetIdentityRegistrationSetting');
                // WHEN
                rq.configureSecretsManagementAutoRegistration(callParams);
            });
            test('registration is delegated to SecretsManagementIdentityRegistration', () => {
                // THEN
                expect(identityRegistrationSettings.addSubnetIdentityRegistrationSetting).toHaveBeenCalledWith(callParams);
            });
            test('deployment instance is created using specified subnets', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    LaunchConfigurationName: stack.resolve(launchConfiguration.ref),
                    VPCZoneIdentifier: assert_1.arrayWith(...RQ_SUBNET_IDS),
                }));
            });
        });
    });
    test('.backendConnections is associated with ASG security group rules', () => {
        // GIVEN
        const instance = new aws_ec2_1.Instance(dependencyStack, 'BackendConnectionInstance', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.MICRO),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux(),
            vpc,
        });
        const portNumber = 5555;
        const port = aws_ec2_1.Port.tcp(portNumber);
        const asgSecurityGroup = renderQueueCommon.asg.connections.securityGroups[0];
        // WHEN
        renderQueueCommon.backendConnections.allowFrom(instance, port);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            Description: `from ${instance.connections.securityGroups[0].uniqueId}:${portNumber}`,
            GroupId: stack.resolve(asgSecurityGroup.securityGroupId),
            SourceSecurityGroupId: stack.resolve(instance.connections.securityGroups[0].securityGroupId),
            FromPort: portNumber,
            ToPort: portNumber,
        }));
    });
});
//# sourceMappingURL=data:application/json;base64,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