"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Version = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * This class is reposonsible to do basic operations on version format.
 *
 * @stability stable
 */
class Version {
    /**
     * @stability stable
     */
    constructor(components) {
        if (components.length != 4) {
            throw new Error('Invalid version format. Version should contain exactly 4 components.');
        }
        components.forEach((component) => {
            if (component < 0) {
                throw new RangeError('Invalid version format. None of the version components can be negative.');
            }
            if (!Number.isInteger(component)) {
                throw new RangeError('Invalid version format. None of the version components can contain decimal values.');
            }
        });
        this.components = components;
    }
    /**
     * This method parses the input string and returns the version object.
     *
     * @param version version string to parse.
     * @stability stable
     */
    static parse(version) {
        if (!Version.validateVersionFormat(version)) {
            throw new TypeError(`Invalid version format. Expected format 'a.b.c.d', found '${version}'`);
        }
        return new Version(version.split('.').map(x => parseInt(x)));
    }
    /**
     * This method validates the given string for a sequence '.' separated numbers.
     *
     * @param version the string to be validated.
     *
     * @returns true if the format is correct, else false.
     */
    static validateVersionFormat(version) {
        /**
         * Regex: ^\d+(?:\.\d+){3}$
         * Matches a sequence of '.' separated numbers with exactly 4 digits.
         * - ^ asserts position at start of a line.
         * - \d+ Matches one or more digits.
         * - (?:\.\d+) Matches a dot and the following one or more digits.
         * - {3} Matches previous pattern exactly 3 times.
         * - $ asserts position at the end of a line
         */
        if (version.match(/^\d+(?:\.\d+){3}$/)) {
            return true;
        }
        return false;
    }
    /**
     * The major version number.
     *
     * @stability stable
     * @inheritdoc true
     */
    get majorVersion() {
        return this.components[0];
    }
    /**
     * The minor version number.
     *
     * @stability stable
     * @inheritdoc true
     */
    get minorVersion() {
        return this.components[1];
    }
    /**
     * The release version number.
     *
     * @stability stable
     * @inheritdoc true
     */
    get releaseVersion() {
        return this.components[2];
    }
    /**
     * The patch version number.
     *
     * @stability stable
     * @inheritdoc true
     */
    get patchVersion() {
        return this.components[3];
    }
    /**
     * This method compares two version strings.
     *
     * @returns true if this version is greater than the provided version;
     * false if this version is less than or equal to the provided verison.
     * @stability stable
     */
    isGreaterThan(version) {
        return this.compare(version) > 0;
    }
    /**
     * This method compares two version strings.
     *
     * @returns true if this version is less than the provided version;
     * false if this version is greater than or equal to the provided verison.
     * @stability stable
     */
    isLessThan(version) {
        return this.compare(version) < 0;
    }
    /**
     * This method compares two version strings.
     *
     * @returns true if this version is equal to the provided version;
     * false otherwise.
     * @stability stable
     */
    isEqual(version) {
        return this.compare(version) == 0;
    }
    /**
     * The method returns the version components in dot separated string format.
     *
     * @stability stable
     */
    toString() {
        return this.components.join('.');
    }
    /**
     * A string representation of the version using the best available information at synthesis-time.
     *
     * This value is not guaranteed to be resolved, and is intended for output to CDK users.
     *
     * @stability stable
     * @inheritdoc true
     */
    get versionString() {
        return this.toString();
    }
    /**
     * This method compares 2 versions.
     *
     * @param version version to compare
     *
     * @returns negative value if this version is less than the provided version;
     * 0 if both the versions are equal;
     * positive value if this version is greater than the provided verison.
     */
    compare(version) {
        if (this.components.length != version.components.length) {
            throw new TypeError('Component count in both the versions should be same.');
        }
        for (let i = 0; i < version.components.length; i++) {
            const diff = this.components[i] - version.components[i];
            if (diff != 0) {
                return diff;
            }
        }
        return 0;
    }
}
exports.Version = Version;
_a = JSII_RTTI_SYMBOL_1;
Version[_a] = { fqn: "aws-rfdk.deadline.Version", version: "0.38.0" };
/**
 * This variable holds the value for minimum supported deadline version.
 *
 * @stability stable
 */
Version.MINIMUM_SUPPORTED_DEADLINE_VERSION = new Version([10, 1, 9, 2]);
/**
 * The minimum Deadline version required to enable Deadline Secrets Management.
 *
 * @stability stable
 */
Version.MINIMUM_SECRETS_MANAGEMENT_VERSION = new Version([10, 1, 19, 0]);
//# sourceMappingURL=data:application/json;base64,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