"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SpotEventPluginFleet = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const spot_event_plugin_fleet_ref_1 = require("./spot-event-plugin-fleet-ref");
const worker_configuration_1 = require("./worker-configuration");
/**
 * This construct reperesents a fleet from the Spot Fleet Request created by the Spot Event Plugin.
 *
 * This fleet is intended to be used as input for the {@link @aws-rfdk/deadline#ConfigureSpotEventPlugin} construct.
 *
 * The construct itself doesn't create the Spot Fleet Request, but deploys all the resources
 * required for the Spot Fleet Request and generates the Spot Fleet Configuration setting:
 * a one to one mapping between a Deadline Group and Spot Fleet Request Configurations.
 *
 * ![architecture diagram](/diagrams/deadline/SpotEventPluginFleet.svg)
 *
 * Resources Deployed
 * ------------------------
 * - An Instance Role, corresponding IAM Policy and an Instance Profile.
 * - A Fleet Role and corresponding IAM Policy.
 * - An Amazon CloudWatch log group that contains the Deadline Worker, Deadline Launcher, and instance-startup logs for the instances
 *    in the fleet.
 * - A security Group if security groups are not provided.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The data that is stored on your Worker's local EBS volume can include temporary working files from the applications
 *    that are rendering your jobs and tasks. That data can be sensitive or privileged, so we recommend that you encrypt
 *    the data volumes of these instances using either the provided option or by using an encrypted AMI as your source.
 * - The software on the AMI that is being used by this construct may pose a security risk. We recommend that you adopt a
 *    patching strategy to keep this software current with the latest security patches. Please see
 *    https://docs.aws.amazon.com/rfdk/latest/guide/patching-software.html for more information.
 *
 * @stability stable
 */
class SpotEventPluginFleet extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h;
        super(scope, id);
        this.defaultSubnets = !props.vpcSubnets;
        this.deadlineGroups = props.deadlineGroups.map(group => group.toLocaleLowerCase());
        this.deadlinePools = (_b = props.deadlinePools) === null || _b === void 0 ? void 0 : _b.map(pool => pool.toLocaleLowerCase());
        this.validateProps(props);
        this.securityGroups = (_c = props.securityGroups) !== null && _c !== void 0 ? _c : [new aws_ec2_1.SecurityGroup(this, 'SpotFleetSecurityGroup', { vpc: props.vpc })];
        this.connections = new aws_ec2_1.Connections({ securityGroups: this.securityGroups });
        this.connections.allowToDefaultPort(props.renderQueue.endpoint);
        this.fleetInstanceRole = (_d = props.fleetInstanceRole) !== null && _d !== void 0 ? _d : new aws_iam_1.Role(this, 'SpotFleetInstanceRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy'),
            ],
            description: `Spot Fleet instance role for ${id} in region ${core_1.Stack.of(scope).region}`,
        });
        this.instanceProfile = new aws_iam_1.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.fleetInstanceRole.roleName],
        });
        this.grantPrincipal = this.fleetInstanceRole;
        this.fleetRole = (_e = props.fleetRole) !== null && _e !== void 0 ? _e : new aws_iam_1.Role(this, 'SpotFleetRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('spotfleet.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole'),
            ],
            description: `Spot Fleet role for ${id} in region ${core_1.Stack.of(scope).region}`,
        });
        this.blockDevices = props.blockDevices;
        this.subnets = props.vpc.selectSubnets((_f = props.vpcSubnets) !== null && _f !== void 0 ? _f : { subnetType: aws_ec2_1.SubnetType.PRIVATE });
        this.instanceTypes = props.instanceTypes;
        this.allocationStrategy = (_g = props.allocationStrategy) !== null && _g !== void 0 ? _g : spot_event_plugin_fleet_ref_1.SpotFleetAllocationStrategy.LOWEST_PRICE;
        this.maxCapacity = props.maxCapacity;
        this.validUntil = props.validUntil;
        this.keyName = props.keyName;
        const imageConfig = props.workerMachineImage.getImage(this);
        this.osType = imageConfig.osType;
        this.userData = (_h = props.userData) !== null && _h !== void 0 ? _h : imageConfig.userData;
        this.imageId = imageConfig.imageId;
        const workerConfig = new worker_configuration_1.WorkerInstanceConfiguration(this, id, {
            worker: this,
            cloudWatchLogSettings: {
                logGroupPrefix: SpotEventPluginFleet.DEFAULT_LOG_GROUP_PREFIX,
                ...props.logGroupProps,
            },
            renderQueue: props.renderQueue,
            workerSettings: {
                groups: this.deadlineGroups,
                pools: this.deadlinePools,
                region: props.deadlineRegion,
            },
            userDataProvider: props.userDataProvider,
        });
        this.remoteControlPorts = aws_ec2_1.Port.tcpRange(workerConfig.listenerPort, workerConfig.listenerPort + SpotEventPluginFleet.MAX_WORKERS_PER_HOST);
        this.tags = new core_1.TagManager(core_1.TagType.KEY_VALUE, 'RFDK::SpotEventPluginFleet');
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Allow access to the Worker's remote command listener port (configured as a part of the WorkerConfiguration) for an IConnectable that is either in this stack, or in a stack that depends on this stack.
     *
     * If this stack depends on the other stack, use allowRemoteControlTo().
     * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/remote-control.html
     *
     * Common uses are:
     *
     *    Adding a SecurityGroup:
     *      `workerFleet.allowRemoteControlFrom(securityGroup)`
     *
     *    Adding a CIDR:
     *      `workerFleet.allowRemoteControlFrom(Peer.ipv4('10.0.0.0/24'))`
     *
     * @stability stable
     * @inheritdoc true
     */
    allowRemoteControlFrom(other) {
        this.connections.allowFrom(other.connections, this.remoteControlPorts, 'Worker remote command listening port');
    }
    /**
     * Allow access to the Worker's remote command listener port (configured as a part of the WorkerConfiguration) for an IConnectable that is either in this stack, or in a stack that this stack depends on.
     *
     * If the other stack depends on this stack, use allowRemoteControlFrom().
     * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/remote-control.html
     *
     * Common uses are:
     *
     *    Adding a SecurityGroup:
     *      `workerFleet.allowRemoteControlTo(securityGroup)`
     *
     *    Adding a CIDR:
     *      `workerFleet.allowRemoteControlTo(Peer.ipv4('10.0.0.0/24'))`
     *
     * @stability stable
     * @inheritdoc true
     */
    allowRemoteControlTo(other) {
        other.connections.allowTo(this.connections, this.remoteControlPorts, 'Worker remote command listening port');
    }
    validateProps(props) {
        this.validateFleetInstanceRole(props.fleetInstanceRole);
        this.validateInstanceTypes(props.instanceTypes);
        this.validateSubnets(props.vpc, props.vpcSubnets);
        this.validateGroups('deadlineGroups', this.deadlineGroups);
        this.validateRegion('deadlineRegion', props.deadlineRegion);
        this.validateBlockDevices(props.blockDevices);
    }
    validateFleetInstanceRole(fleetInstanceRole) {
        if (fleetInstanceRole) {
            if (core_1.Stack.of(fleetInstanceRole) !== core_1.Stack.of(this)) {
                throw new Error(`Fleet instance role should be created on the same stack as ${this.constructor.name} to avoid circular dependencies.`);
            }
        }
    }
    validateInstanceTypes(array) {
        if (array.length === 0) {
            throw new Error('At least one instance type is required for a Spot Fleet Request Configuration');
        }
    }
    validateSubnets(vpc, vpcSubnets) {
        const { subnets } = vpc.selectSubnets(vpcSubnets);
        if (subnets.length === 0) {
            core_1.Annotations.of(this).addError(`Did not find any subnets matching '${JSON.stringify(vpcSubnets)}', please use a different selection.`);
        }
    }
    validateGroups(property, array) {
        const regex = /^(?!none$)[a-z0-9-_]+$/g;
        if (array.length === 0) {
            throw new Error('At least one Deadline Group is required for a Spot Fleet Request Configuration');
        }
        array.forEach(value => {
            if (!regex.test(value)) {
                throw new Error(`Invalid value: ${value} for property '${property}'. Valid characters are a-z, 0-9, - and _. Also, group 'none' is reserved as the default group.`);
            }
        });
    }
    validateRegion(property, region) {
        const regex = /^(?!none$|all$|unrecognized$)[a-zA-Z0-9-_]+$/i;
        if (region && !regex.test(region)) {
            throw new Error(`Invalid value: ${region} for property '${property}'. Valid characters are A-Z, a-z, 0-9, - and _. ‘All’, ‘none’ and ‘unrecognized’ are reserved names that cannot be used.`);
        }
    }
    validateBlockDevices(blockDevices) {
        if (blockDevices === undefined) {
            core_1.Annotations.of(this).addWarning(`The spot-fleet ${this.node.id} is being created without being provided any block devices so the Source AMI's devices will be used. ` +
                'Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
        }
        else {
            blockDevices.forEach(device => {
                if (device.volume.ebsDevice === undefined) {
                    // Suppressed or Ephemeral Block Device
                    return;
                }
                const { iops, volumeType } = device.volume.ebsDevice;
                if (!iops) {
                    if (volumeType === aws_ec2_1.EbsDeviceVolumeType.IO1) {
                        throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                    }
                }
                else if (volumeType !== aws_ec2_1.EbsDeviceVolumeType.IO1) {
                    core_1.Annotations.of(this).addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
                }
                // encrypted is not exposed as part of ebsDeviceProps so we need to confirm it exists then access it via [].
                // eslint-disable-next-line dot-notation
                if (('encrypted' in device.volume.ebsDevice === false) || ('encrypted' in device.volume.ebsDevice && !device.volume.ebsDevice['encrypted'])) {
                    core_1.Annotations.of(this).addWarning(`The BlockDevice "${device.deviceName}" on the spot-fleet ${this.node.id} is not encrypted. ` +
                        'Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.');
                }
            });
        }
    }
}
exports.SpotEventPluginFleet = SpotEventPluginFleet;
_a = JSII_RTTI_SYMBOL_1;
SpotEventPluginFleet[_a] = { fqn: "aws-rfdk.deadline.SpotEventPluginFleet", version: "0.38.0" };
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
SpotEventPluginFleet.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
/**
 * This is the current maximum for number of workers that can be started on a single host. Currently the
 * only thing using this limit is the configuration of the listener ports. More than 8 workers can be started,
 * but only the first 8 will have their ports opened in the workers' security group.
 */
SpotEventPluginFleet.MAX_WORKERS_PER_HOST = 8;
//# sourceMappingURL=data:application/json;base64,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