"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretsManagementIdentityRegistration = void 0;
const path = require("path");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const _1 = require(".");
const core_2 = require("../../core");
/**
 * Construct that configures desired Deadline Secrets Management identity registration settings.
 *
 * Resources Deployed
 * ------------------------
 * - IAM policy statements are added to the IAM policy that is attached to the IAM role of the DeploymentInstance.
 *   These statements grant the DeploymentInstance the ability to fetch the Deadline Client installer, get the value of
 *   the AWS Secrets Manager secert containing the Deadline Secrets Management administrator credentials, get the value
 *   of the AWS Secrets Manager secert containing the Deadline Repository's database credentials,
 * - Security group ingress rule to allow the DeploymentInstance to connect to the Repository's database
 * - Security group ingress rule to allow the DeploymentInstance to connect to the Repository's file-system
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *   is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *   performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *   bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The instance deployed by this construct has read/write access to the Deadline Repository (database and
 *   file-system), the AWS Secrets Manager secrets containing credentials for the Database and the Deadline Secrets
 *   Management administrator. Access to the instance permits command and control of the render farm and should be
 *   restricted.
 */
class SecretsManagementIdentityRegistration extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.subnetRegistrations = new Map();
        if (!props.repository.secretsManagementSettings.enabled) {
            throw new Error('Secrets management is not enabled on repository');
        }
        /* istanbul ignore next */
        if (!props.repository.secretsManagementSettings.credentials) {
            throw new Error('Repository does not contain secrets management credentials');
        }
        this.adminCredentials = props.repository.secretsManagementSettings.credentials;
        this.deploymentInstance = props.deploymentInstance;
        this.renderQueueSubnets = props.renderQueueSubnets;
        // Download and install the Deadline Client
        this.installDeadlineClient(props);
        // Configure the Deadline Client to direct-connect to the repository
        props.repository.configureClientInstance({
            host: props.deploymentInstance,
            mountPoint: '/mnt/repository',
        });
        // Install python dependencies
        this.preparePythonEnvironment(props);
        const localScriptFile = this.preparePythonScript(props);
        this.runPythonScript(props, localScriptFile);
        props.deploymentInstance.addExecutionDependency(props.repository);
    }
    addSubnetIdentityRegistrationSetting(addSubnetProps) {
        if (addSubnetProps.role === _1.SecretsManagementRole.ADMINISTRATOR) {
            throw new Error('The Administrator role cannot be set using a Deadline identity registration setting');
        }
        const { vpc, vpcSubnets } = addSubnetProps;
        const selectedSubnets = vpc.selectSubnets(vpcSubnets);
        selectedSubnets.subnets.forEach(subnet => {
            if (this.renderQueueSubnets.subnets.some(rqSubnet => subnet == rqSubnet)) {
                core_1.Annotations.of(addSubnetProps.dependent).addWarning(`Deadline Secrets Management is enabled on the Repository and VPC subnets of the Render Queue match the subnets of ${addSubnetProps.dependent.node.path}. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components`);
            }
            const observedSubnet = this.subnetRegistrations.get(subnet.subnetId);
            if (observedSubnet) {
                if (observedSubnet.registrationStatus !== addSubnetProps.registrationStatus) {
                    throw new Error(`Subnet is already registered with registrationStatus "${observedSubnet.registrationStatus}" but another caller requested "${addSubnetProps.registrationStatus}"`);
                }
                else if (observedSubnet.role !== addSubnetProps.role) {
                    throw new Error(`Subnet is already registered with role "${observedSubnet.role}" but another caller requested "${addSubnetProps.role}"`);
                }
            }
            else {
                this.subnetRegistrations.set(subnet.subnetId, {
                    registrationStatus: addSubnetProps.registrationStatus,
                    role: addSubnetProps.role,
                });
            }
        });
        addSubnetProps.dependent.node.addDependency(this.deploymentInstance);
    }
    runPythonScript(props, localScriptFile) {
        // The command-line arguments to be passed to the script that configures the Deadline identity registration
        // settings
        const scriptArgs = core_1.Lazy.list({
            produce: () => {
                return [].concat([
                    // Region
                    '--region',
                    core_1.Stack.of(this).region,
                    // Admin credentials
                    '--credentials',
                    `"${this.adminCredentials.secretArn}"`,
                ], 
                // Subnets of the load balancer
                (props.renderQueueSubnets
                    .subnetIds
                    .map(subnetID => `--connection-subnet "${subnetID}"`)), 
                // Subnets of RFDK Deadline Client constructs
                (Array.from(this.subnetRegistrations.entries())
                    // Each setting becomes a comma (,) separated string of fields
                    //   <SUBNET_ID>,<ROLE>,<REGISTRATION_STATUS>
                    .map(subnetRegistrationEntry => {
                    const [subnetID, registrationSettingEffect] = subnetRegistrationEntry;
                    return [
                        subnetID,
                        registrationSettingEffect.role.toString(),
                        (registrationSettingEffect.registrationStatus).toString(),
                    ].join(',');
                })
                    // convert into argument key/value pair
                    .map(joinedSubnetArgValue => `--source-subnet "${joinedSubnetArgValue}"`)));
            },
        });
        // We can't use ScriptAsset.executeOn(...) because we need to run as "ec2-user".
        // This is because Repository.configureClientInstance(...) used above will store the credentials
        // in a per-user credential store that is only available to "ec2-user".
        props.deploymentInstance.userData.addCommands(`sudo --login -u ec2-user ${localScriptFile} ` + core_1.Fn.join(' ', scriptArgs));
    }
    preparePythonScript(props) {
        const script = new core_2.ScriptAsset(this, 'ConfigureIdentityRegistrationSettingScript', {
            path: path.join(__dirname, '..', 'scripts', 'python', 'configure_identity_registration_settings.py'),
        });
        // Grant access to ec2:DescribeSubnets. Subnet IPv4 CIDR ranges are not exposed through
        // CloudFormation attributes. Instead, we must query them using the EC2 API on the deployment instance
        props.deploymentInstance.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: ['ec2:DescribeSubnets'],
            // ec2:DescribeSubnets does not support resource level permissions. See
            // https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonec2.html
            resources: ['*'],
        }));
        this.adminCredentials.grantRead(props.deploymentInstance);
        script.grantRead(props.deploymentInstance);
        const localScriptFile = props.deploymentInstance.userData.addS3DownloadCommand({
            bucket: script.bucket,
            bucketKey: script.s3ObjectKey,
            localFile: '/home/ec2-user/configure_identity_registration_settings.py',
        });
        props.deploymentInstance.userData.addCommands(`chmod +x ${localScriptFile}`, `chown ec2-user:ec2-user ${localScriptFile}`);
        return localScriptFile;
    }
    installDeadlineClient(props) {
        props.version.linuxInstallers.client.s3Bucket.grantRead(props.deploymentInstance, props.version.linuxInstallers.client.objectKey);
        const clientInstallerPath = props.deploymentInstance.userData.addS3DownloadCommand({
            bucket: props.version.linuxInstallers.client.s3Bucket,
            bucketKey: props.version.linuxInstallers.client.objectKey,
        });
        props.deploymentInstance.userData.addCommands('set -x');
        props.deploymentInstance.userData.addCommands(`chmod +x "${clientInstallerPath}"`);
        props.deploymentInstance.userData.addCommands([
            // This is required b/c USER and HOME environment variables are not defined when running
            // user-data
            'sudo', '--login',
            // Run the Deadline Client installer
            `"${clientInstallerPath}"`,
            '--mode', 'unattended',
            '--connectiontype', 'Remote',
            '--proxyrootdir', '127.0.0.1:8080',
            '--noguimode', 'true',
            '--slavestartup', 'false',
            '--launcherdaemon', 'false',
            '--restartstalled', 'true',
            '--autoupdateoverride', 'False',
        ].join(' '));
    }
    preparePythonEnvironment(props) {
        // The script must run as ec2-user because Repository.configureClientInstance(...) used above will store the
        // credentials in a per-user credential store that is only available to "ec2-user".
        props.deploymentInstance.userData.addCommands('sudo -u ec2-user python3 -m pip install --user boto3');
    }
}
exports.SecretsManagementIdentityRegistration = SecretsManagementIdentityRegistration;
//# sourceMappingURL=data:application/json;base64,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