/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { LustreFileSystem } from '@aws-cdk/aws-fsx';
import { Asset } from '@aws-cdk/aws-s3-assets';
import { Construct, IConstruct } from '@aws-cdk/core';
import { IMountableLinuxFilesystem, IMountingInstance, LinuxMountPointProps } from './mountable-filesystem';
/**
 * Properties that are required to create a {@link MountableFsxLustre}.
 *
 * @stability stable
 */
export interface MountableFsxLustreProps {
    /**
     * The {@link https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-fsx.LustreFileSystem.html|FSx for Lustre} filesystem that will be mounted by the object.
     *
     * @stability stable
     */
    readonly filesystem: LustreFileSystem;
    /**
     * The fileset to mount.
     *
     * @default Mounts the root of the filesystem.
     * @see https://docs.aws.amazon.com/fsx/latest/LustreGuide/mounting-from-fileset.html
     * @stability stable
     */
    readonly fileset?: string;
    /**
     * Extra Lustre mount options that will be added to /etc/fstab for the file system. See: {@link http://manpages.ubuntu.com/manpages/precise/man8/mount.lustre.8.html}.
     *
     * The given values will be joined together into a single string by commas.
     * ex: ['soft', 'rsize=4096'] will become 'soft,rsize=4096'
     *
     * @default No extra options.
     * @stability stable
     */
    readonly extraMountOptions?: string[];
}
/**
 * This class encapsulates scripting that can be used to mount an Amazon FSx for Lustre File System onto an instance.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *    from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *    bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *    you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *    account to assist in post-incident analysis of compromised production environments.
 *
 * @stability stable
 */
export declare class MountableFsxLustre implements IMountableLinuxFilesystem {
    protected readonly scope: Construct;
    protected readonly props: MountableFsxLustreProps;
    /**
     * @stability stable
     */
    constructor(scope: Construct, props: MountableFsxLustreProps);
    /**
     * Mount the filesystem to the given instance at instance startup.
     *
     * This is accomplished by
     * adding scripting to the UserData of the instance to mount the filesystem on startup.
     * If required, the instance's security group is granted ingress to the filesystem's security
     * group on the required ports.
     *
     * @stability stable
     * @inheritdoc true
     */
    mountToLinuxInstance(target: IMountingInstance, mount: LinuxMountPointProps): void;
    /**
     * Returns whether the mounted file-system evaluates the UID/GID of the system user accessing the file-system.
     *
     * Some network file-systems provide features to fix a UID/GID for all access to the mounted file-system and ignore
     * the system user accessing the file. If this is the case, an implementing class must indicate this in the return
     * value.
     *
     * @stability stable
     * @inheritdoc true
     */
    usesUserPosixPermissions(): boolean;
    /**
     * Fetch the Asset singleton for the FSx for Lustre mounting scripts, or generate it if needed.
     *
     * @stability stable
     */
    protected mountAssetSingleton(scope: IConstruct): Asset;
}
