"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MongoDbInstance = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const _1 = require("./");
const runtime_info_1 = require("./runtime-info");
/**
 * This construct provides a {@link StaticPrivateIpServer} that is hosting MongoDB.
 *
 * The data for this MongoDB database
 * is stored in an Amazon Elastic Block Storage (EBS) Volume that is automatically attached to the instance when it is
 * launched, and is separate from the instance's root volume; it is recommended that you set up a backup schedule for
 * this volume.
 *
 * When this instance is first launched, or relaunched after an instance replacement, it will:
 * 1. Attach an EBS volume to /var/lib/mongo upon which the MongoDB data is stored;
 * 2. Automatically install the specified version of MongoDB, from the official Mongo Inc. sources;
 * 3. Create an admin user in that database if one has not yet been created -- the credentials for this user
 * can be provided by you, or randomly generated;
 * 4. Configure MongoDB to require authentication, and only allow encrypted connections over TLS.
 *
 * The instance's launch logs and MongoDB logs will be automatically stored in Amazon CloudWatch logs; the
 * default log group name is: /renderfarm/<this construct ID>
 *
 * Resources Deployed
 * ------------------------
 * - {@link StaticPrivateIpServer} that hosts MongoDB.
 * - An A-Record in the provided PrivateHostedZone to create a DNS entry for this server's static private IP.
 * - A Secret in AWS SecretsManager that contains the administrator credentials for MongoDB.
 * - An encrypted Amazon Elastic Block Store (EBS) Volume on which the MongoDB data is stored.
 * - Amazon CloudWatch log group that contains instance-launch and MongoDB application logs.
 *
 * Security Considerations
 * ------------------------
 * - The administrator credentials for MongoDB are stored in a Secret within AWS SecretsManager. You must strictly limit
 *    access to this secret to only entities that require it.
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The EBS Volume that is created by, or provided to, this construct is used to store the contents of your MongoDB data. To
 *    protect the sensitive data in your database, you should not grant access to this EBS Volume to any principal or instance
 *    other than the instance created by this construct. Furthermore, we recommend that you ensure that the volume that is
 *    used for this purpose is encrypted at rest.
 * - This construct uses this package's {@link StaticPrivateIpServer}, {@link MongoDbInstaller}, {@link CloudWatchAgent},
 *    {@link ExportingLogGroup}, and {@link MountableBlockVolume}. Security considerations that are outlined by the documentation
 *    for those constructs should also be taken into account.
 *
 * @stability stable
 */
class MongoDbInstance extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f;
        super(scope, id);
        this.version = props.mongoDb.version;
        // Select the subnet for this instance.
        const { subnets } = props.vpc.selectSubnets(props.vpcSubnets);
        if (subnets.length === 0) {
            throw new Error(`Did not find any subnets matching ${JSON.stringify(props.vpcSubnets)}. Please use a different selection.`);
        }
        const subnet = subnets[0];
        this.server = new _1.StaticPrivateIpServer(this, 'Server', {
            vpc: props.vpc,
            vpcSubnets: { subnets: [subnet] },
            instanceType: (_b = props.instanceType) !== null && _b !== void 0 ? _b : new aws_ec2_1.InstanceType('r5.large'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            blockDevices: [
                {
                    deviceName: '/dev/xvda',
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(MongoDbInstance.ROOT_DEVICE_SIZE.toGibibytes(), { encrypted: true }),
                },
            ],
            keyName: props.keyName,
            resourceSignalTimeout: core_1.Duration.minutes(5),
            role: props.role,
            securityGroup: props.securityGroup,
        });
        new aws_route53_1.ARecord(this, 'ARecord', {
            target: aws_route53_1.RecordTarget.fromIpAddresses(this.server.privateIpAddress),
            zone: props.mongoDb.dnsZone,
            recordName: props.mongoDb.hostname,
        });
        this.adminUser = (_c = props.mongoDb.adminUser) !== null && _c !== void 0 ? _c : new aws_secretsmanager_1.Secret(this, 'AdminUser', {
            description: `Admin credentials for the MongoDB database ${core_1.Names.uniqueId(this)}`,
            generateSecretString: {
                excludeCharacters: '"()$\'',
                excludePunctuation: true,
                includeSpace: false,
                passwordLength: 24,
                requireEachIncludedType: true,
                generateStringKey: 'password',
                secretStringTemplate: JSON.stringify({ username: 'admin' }),
            },
        });
        this.mongoDataVolume = (_e = (_d = props.mongoDb.mongoDataVolume) === null || _d === void 0 ? void 0 : _d.volume) !== null && _e !== void 0 ? _e : new aws_ec2_1.Volume(this, 'MongoDbData', {
            size: MongoDbInstance.DEFAULT_MONGO_DEVICE_SIZE,
            ...(_f = props.mongoDb.mongoDataVolume) === null || _f === void 0 ? void 0 : _f.volumeProps,
            availabilityZone: subnet.availabilityZone,
            encrypted: true,
        });
        const volumeMount = new _1.MountableBlockVolume(this, {
            blockVolume: this.mongoDataVolume,
            volumeFormat: _1.BlockVolumeFormat.XFS,
        });
        const mongoInstaller = new _1.MongoDbInstaller(this, {
            version: props.mongoDb.version,
            userSsplAcceptance: props.mongoDb.userSsplAcceptance,
        });
        // Set up the server's UserData.
        this.server.userData.addCommands('set -xefuo pipefail');
        this.server.userData.addSignalOnExitCommand(this.server.autoscalingGroup);
        this.configureCloudWatchLogStreams(this.server, id, props.logGroupProps); // MUST BE FIRST
        volumeMount.mountToLinuxInstance(this.server, {
            location: MongoDbInstance.MONGO_DEVICE_MOUNT_POINT,
        });
        mongoInstaller.installOnLinuxInstance(this.server);
        this.configureMongoDb(this.server, props.mongoDb);
        this.certificateChain = props.mongoDb.serverCertificate.certChain;
        this.connections = this.server.connections;
        this.grantPrincipal = this.server.grantPrincipal;
        this.port = 27017;
        this.role = this.server.role;
        this.userData = this.server.userData;
        this.fullHostname = `${props.mongoDb.hostname}.${props.mongoDb.dnsZone.zoneName}`;
        this.node.defaultChild = this.server;
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Adds security groups to the database.
     *
     * @stability stable
     * @inheritdoc true
     */
    addSecurityGroup(...securityGroups) {
        securityGroups === null || securityGroups === void 0 ? void 0 : securityGroups.forEach(securityGroup => this.server.autoscalingGroup.addSecurityGroup(securityGroup));
    }
    /**
     * Adds UserData commands to install & configure the CloudWatch Agent onto the instance.
     *
     * The commands configure the agent to stream the following logs to a new CloudWatch log group:
     *      - The cloud-init log
     *      - The MongoDB application log.
     *
     * @param host The instance/host to setup the CloudWatchAgent upon.
     * @param groupName Name to append to the log group prefix when forming the log group name.
     * @param logGroupProps Properties for the log group.
     * @stability stable
     */
    configureCloudWatchLogStreams(host, groupName, logGroupProps) {
        var _b;
        const prefix = (_b = logGroupProps === null || logGroupProps === void 0 ? void 0 : logGroupProps.logGroupPrefix) !== null && _b !== void 0 ? _b : MongoDbInstance.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = _1.LogGroupFactory.createOrFetch(this, 'MongoDbInstanceLogGroupWrapper', groupName, defaultedLogGroupProps);
        logGroup.grantWrite(host.grantPrincipal);
        const cloudWatchConfigurationBuilder = new _1.CloudWatchConfigBuilder(MongoDbInstance.CLOUDWATCH_LOG_FLUSH_INTERVAL);
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'MongoDB', '/var/log/mongodb/mongod.log');
        new _1.CloudWatchAgent(this, 'MongoDbInstanceLogsConfig', {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host,
        });
    }
    /**
     * Adds commands to the userData of the instance to install MongoDB, create an admin user if one does not exist, and to to start mongod running.
     *
     * @stability stable
     */
    configureMongoDb(instance, settings) {
        const scriptsAsset = new aws_s3_assets_1.Asset(this, 'MongoSetup', {
            path: path.join(__dirname, '..', 'scripts', 'mongodb', settings.version),
        });
        scriptsAsset.grantRead(instance.grantPrincipal);
        const scriptZipfile = instance.userData.addS3DownloadCommand({
            bucket: scriptsAsset.bucket,
            bucketKey: scriptsAsset.s3ObjectKey,
        });
        instance.userData.addCommands(
        // Ensure mongod is installed and stopped before we go any further
        'which mongod && test -f /etc/mongod.conf', 'sudo service mongod stop', 
        // We're going to make a temporary RAM filesystem for the mongo setup files.
        // This will let us write sensitive data to "disk" without worrying about it
        // being persisted in any physical disk, even temporarily.
        'MONGO_SETUP_DIR=$(mktemp -d)', 'mkdir -p "${MONGO_SETUP_DIR}"', 'sudo mount -t tmpfs -o size=50M tmpfs "${MONGO_SETUP_DIR}"', 'pushd "${MONGO_SETUP_DIR}"', `unzip ${scriptZipfile}`, 
        // Backup mongod.conf for now
        'cp /etc/mongod.conf .');
        const cert = settings.serverCertificate;
        instance.userData.addCommands(`bash serverCertFromSecrets.sh "${cert.cert.secretArn}" "${cert.certChain.secretArn}" "${cert.key.secretArn}" "${cert.passphrase.secretArn}"`);
        cert.cert.grantRead(instance.grantPrincipal);
        cert.certChain.grantRead(instance.grantPrincipal);
        cert.key.grantRead(instance.grantPrincipal);
        cert.passphrase.grantRead(instance.grantPrincipal);
        const certsDirectory = '/etc/mongod_certs';
        instance.userData.addCommands(
        // Move the certificates into place
        `sudo mkdir -p ${certsDirectory}`, `sudo mv ./ca.crt ./key.pem ${certsDirectory}`, 'sudo chown root.mongod -R /etc/mongod_certs/', // Something weird about shell interpretation. Can't use '*' on this or next line.
        'sudo chmod 640 -R /etc/mongod_certs/', 'sudo chmod 750 /etc/mongod_certs/', // Directory needs to be executable.
        // mongod user id might, potentially change on reboot. Make sure we own all mongo data
        `sudo chown mongod.mongod -R ${MongoDbInstance.MONGO_DEVICE_MOUNT_POINT}`, 
        // Configure mongod
        'bash ./setMongoLimits.sh', `bash ./setStoragePath.sh "${MongoDbInstance.MONGO_DEVICE_MOUNT_POINT}"`, 'bash ./setMongoNoAuth.sh', 'sudo service mongod start', `bash ./setAdminCredentials.sh "${this.adminUser.secretArn}"`);
        this.adminUser.grantRead(instance.grantPrincipal);
        instance.userData.addCommands(
        // Setup for live deployment, and start mongod
        'sudo service mongod stop', 'bash ./setLiveConfiguration.sh', 'sudo systemctl enable mongod', // Enable restart on reboot
        'sudo service mongod start', 'popd');
        instance.userData.addOnExitCommands(
        // Clean up the temporary RAM filesystem
        'test "${MONGO_SETUP_DIR} != "" && sudo umount "${MONGO_SETUP_DIR}');
    }
}
exports.MongoDbInstance = MongoDbInstance;
_a = JSII_RTTI_SYMBOL_1;
MongoDbInstance[_a] = { fqn: "aws-rfdk.MongoDbInstance", version: "0.38.0" };
// How often Cloudwatch logs will be flushed.
MongoDbInstance.CLOUDWATCH_LOG_FLUSH_INTERVAL = core_1.Duration.seconds(15);
// Default prefix for a LogGroup if one isn't provided in the props.
MongoDbInstance.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
// Size of the EBS volume for MongoDB data, if we create one.
MongoDbInstance.DEFAULT_MONGO_DEVICE_SIZE = core_1.Size.gibibytes(20);
// Mount point for the MongoDB data volume.
MongoDbInstance.MONGO_DEVICE_MOUNT_POINT = '/var/lib/mongo';
// Size of the root device volume on the instance.
MongoDbInstance.ROOT_DEVICE_SIZE = core_1.Size.gibibytes(10);
//# sourceMappingURL=data:application/json;base64,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