# copyright 2015-2022 LOGILAB S.A. (Paris, FRANCE), all rights reserved.
# contact http://www.logilab.fr -- mailto:contact@logilab.fr
#
# This program is free software: you can redistribute it and/or modify it under
# the terms of the GNU Lesser General Public License as published by the Free
# Software Foundation, either version 2.1 of the License, or (at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
# details.
#
# You should have received a copy of the GNU Lesser General Public License along
# with this program. If not, see <http://www.gnu.org/licenses/>.
"""SKOS dataimport utilities."""


from cubicweb import schema
from cubicweb.dataimport.importer import (ExtEntity, ExtEntitiesImporter, RelationMapping,
                                          cwuri2eid, use_extid_as_cwuri)


class SKOSRelationMapping(RelationMapping):

    def __getitem__(self, rtype):
        if rtype in ('exact_match', 'close_match'):
            rql = f'Any S,O WHERE S {rtype} O'
            return {tuple(x) for x in self.cnx.execute(rql)}
        return super().__getitem__(rtype)


def ext_dump_relations(cnx, extid2eid, extentity):
    """Wrapper around :func:`dump_relations` but receiving extentity and extdid2eid mapping as
    argument, to attempt to work around store's limitation on inlined relation.
    """
    eid = extid2eid[extentity.extid]
    etype = cnx.entity_type(eid)
    relations = []
    rschema = cnx.vreg.schema.rschema
    for subj, rtype, obj in dump_relations(cnx, eid, etype):
        if rschema(rtype).inlined:
            if subj is None:
                if obj is None:
                    obj = extentity.extid
                else:
                    extid2eid[obj] = obj
                extentity.values.setdefault(rtype, set()).add(obj)
            else:
                raise Exception(f"can't dump inlined object relation {rtype}")
        else:
            relations.append((subj, rtype, obj))
    return relations


def dump_relations(cnx, eid, etype):
    """Return a list of relation 3-uples `(subject_eid, relation, object_eid)` with None instead of
    `subject_eid` or `object_eid` depending on whether the entity type corresponding to `eid` is
    subject or object.
    """
    eschema = cnx.vreg.schema.eschema(etype)
    relations = []
    for rschema, _, role in eschema.relation_definitions():
        if rschema.rule:  # computed relation
            continue
        rtype = rschema.type
        if rtype in schema.VIRTUAL_RTYPES or rtype in schema.META_RTYPES:
            continue
        if role == 'subject':
            for object_eid, in cnx.execute('Any Y WHERE X %s Y, X eid %%(x)s' % rtype, {'x': eid}):
                if object_eid == eid:
                    object_eid = None
                relations.append((None, rtype, object_eid))
        else:
            for subject_eid, in cnx.execute('Any Y WHERE Y %s X, X eid %%(x)s' % rtype, {'x': eid}):
                if subject_eid == eid:
                    subject_eid = None
                relations.append((subject_eid, rtype, None))
    return relations


def store_skos_extentities(cnx, store, entities, import_log,
                           source=None, raise_on_error=False,
                           extid2eid=None, extid_as_cwuri=True):
    """Add SKOS external entities to the store. Don't commit/flush any data.

    Arguments:

    * `cnx`, RQL connection to the CubicWeb instance
    * `store`, store to use for the import
    * `entities`, iterable (usualy a generator) on external entities to import
    * `import_log`, import log instance to give to the store
    * `source`, optional source in which existing entities will be looked for
      (default to the system source)
    * `raise_on_error`, boolean flag telling if we should fail on error or simply log it (default
      to `False`)
    * `extid_as_cwuri`, boolean flag telling if we should use the external entities'extid as `cwuri`
      attribute of imported entities (default to `True`)
    """
    # only consider the system source for schemes and labels
    if source is None:
        source_eid = cnx.repo.system_source.eid
    else:
        source_eid = source.eid
    if extid2eid is None:
        extid2eid = cwuri2eid(cnx, ('ConceptScheme', 'Label'), source_eid=source_eid)
        # though concepts and external URIs may come from any source
        extid2eid.update(cwuri2eid(cnx, ('Concept', 'ExternalUri')))
    # plug function that turn previously known external uris by newly inserted concepts
    restore_relations = {}

    # pylint: disable=dangerous-default-value
    def externaluri_to_concept(extentity, cnx=cnx, extid2eid=extid2eid,
                               restore_relations=restore_relations):
        try:
            eid = extid2eid[extentity.extid]
        except KeyError:
            pass
        else:
            if extentity.etype == 'Concept' and cnx.entity_type(eid) == 'ExternalUri':
                # We have replaced the external uri by the new concept. As entities.extid column is
                # unique, we've to drop the external uri before inserting the concept, so we:
                #  1. record every relations from/to the external uri,
                #  2. remove it,
                #  3. insert the concept and
                #  4. reinsert relations using the concept instead
                #
                # 1. record relations from/to the external uri
                restore_relations[extentity.extid] = ext_dump_relations(cnx, extid2eid, extentity)
                # 2. remove the external uri entity
                cnx.execute('DELETE ExternalUri X WHERE X eid %(x)s', {'x': eid})
                # 3. drop its extid from the mapping to trigger insertion of the concept by the
                # importer
                del extid2eid[extentity.extid]
                # 4. will be done in SKOSExtEntitiesImporter
        return extentity

    entities = map(externaluri_to_concept, entities)
    # plug function to detect the concept scheme
    concept_schemes = []

    def record_scheme(extentity):
        if extentity.etype == 'ConceptScheme':
            concept_schemes.append(extentity.extid)
        return extentity

    entities = map(record_scheme, entities)
    etypes_order_hint = ('ConceptScheme', 'Concept', 'Label')
    existing_relations = SKOSRelationMapping(cnx, source)
    importer = SKOSExtEntitiesImporter(schema=cnx.vreg.schema, store=store,
                                       extid2eid=extid2eid, existing_relations=existing_relations,
                                       restore_relations=restore_relations,
                                       etypes_order_hint=etypes_order_hint, import_log=import_log,
                                       raise_on_error=raise_on_error)
    if extid_as_cwuri:
        set_cwuri = use_extid_as_cwuri(importer.extid2eid)
        entities = set_cwuri(entities)
    importer.import_entities(entities)
    stats = (importer.created, importer.updated)
    return stats, concept_schemes


class SKOSExtEntitiesImporter(ExtEntitiesImporter):
    """Override ExtEntitiesImporter to handle creation of additionnal relations to newly created
    concepts that replace a former external uri, and to create ExternalUri entities for URIs used in
    exact_match / close_match relations which have no known entity in the repository yet.
    """

    def __init__(self, *args, **kwargs):
        self.restore_relations = kwargs.pop('restore_relations')
        super().__init__(*args, **kwargs)

    def prepare_insert_entity(self, extentity):
        eid = super().prepare_insert_entity(extentity)
        # (4.) restore relations formerly from/to an equivalent external uri
        try:
            relations = self.restore_relations.pop(extentity.extid)
        except KeyError:
            return eid
        for subject_eid, rtype, object_eid in relations:
            if subject_eid is None:
                subject_eid = eid
            if object_eid is None:
                object_eid = eid
            self.store.prepare_insert_relation(subject_eid, rtype, object_eid)
        return eid

    def prepare_insert_deferred_relations(self, deferred):
        # create missing targets for exact_match and close_match relations
        for rtype in ('exact_match', 'close_match'):
            relations = deferred.get(rtype, ())
            for _, object_uri in relations:
                if object_uri not in self.extid2eid:
                    extentity = ExtEntity('ExternalUri', object_uri,
                                          values=dict(uri=str(object_uri),
                                                      cwuri=str(object_uri)))
                    self.prepare_insert_entity(extentity)
        return super().prepare_insert_deferred_relations(deferred)
