# -*- coding: utf-8 -*-

# Copyright (c) 2023 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a dialog to enter the parameters needed to connect to a WiFi
network.
"""

from PyQt6.QtCore import pyqtSlot
from PyQt6.QtWidgets import QDialog, QDialogButtonBox, QLineEdit

from eric7 import Preferences
from eric7.EricGui import EricPixmapCache

from .Ui_WifiConnectionDialog import Ui_WifiConnectionDialog


class WifiConnectionDialog(QDialog, Ui_WifiConnectionDialog):
    """
    Class implementing a dialog to enter the parameters needed to connect to a WiFi
    network.
    """

    def __init__(self, parent=None):
        """
        Constructor

        @param parent reference to the parent widget (defaults to None)
        @type QWidget (optional)
        """
        super().__init__(parent)
        self.setupUi(self)

        self.showPasswordButton.setIcon(EricPixmapCache.getIcon("showPassword"))

        self.buttonBox.button(QDialogButtonBox.StandardButton.Ok).setEnabled(False)

        # populate the field with data saved to the preferences
        self.ssidEdit.setText(Preferences.getMicroPython("WifiName"))
        self.passwordEdit.setText(Preferences.getMicroPython("WifiPassword"))

        msh = self.minimumSizeHint()
        self.resize(max(self.width(), msh.width()), msh.height())

    @pyqtSlot(str)
    def on_ssidEdit_textChanged(self, ssid):
        """
        Private slot handling a change of the SSID.

        @param ssid entered SSID
        @type str
        """
        self.buttonBox.button(QDialogButtonBox.StandardButton.Ok).setEnabled(bool(ssid))

    @pyqtSlot(bool)
    def on_showPasswordButton_clicked(self, checked):
        """
        Private slot to show or hide the password.

        @param checked state of the button
        @type bool
        """
        if checked:
            self.passwordEdit.setEchoMode(QLineEdit.EchoMode.Normal)
            self.showPasswordButton.setIcon(EricPixmapCache.getIcon("hidePassword"))
            self.showPasswordButton.setToolTip(self.tr("Press to hide the password"))
        else:
            self.passwordEdit.setEchoMode(QLineEdit.EchoMode.Password)
            self.showPasswordButton.setIcon(EricPixmapCache.getIcon("showPassword"))
            self.showPasswordButton.setToolTip(self.tr("Press to show the password"))

    @pyqtSlot()
    def accept(self):
        """
        Public slot accepting the dialog.
        """
        if self.rememberCheckBox.isChecked():
            Preferences.setMicroPython("WifiName", self.ssidEdit.text())
            Preferences.setMicroPython("WifiPassword", self.passwordEdit.text())

        super().accept()

    def getConnectionParameters(self):
        """
        Public method to get the entered connection parameters.

        @return tuple containing the SSID and the password
        @rtype tuple of (str, str)
        """
        return (
            self.ssidEdit.text(),
            self.passwordEdit.text(),
        )
