import os
import subprocess
from setuptools import setup
from setuptools import Extension

try:
    from Cython.Build import cythonize
    HAS_CYTHON = True
except ImportError:
    HAS_CYTHON = False

INCLUDE_DIRS = ['.', './Bmath', './math/cephes']

try:
    import numpy
    HAS_NUMPY = True
    INCLUDE_DIRS += [numpy.get_include()]
except ImportError:
    HAS_NUMPY = False

file_ext = '.pyx' if HAS_CYTHON else '.cpp'

def find_cpp(path, recursive=True, exclude=[]):
    ret = []
    if recursive:
        for root, subdirs, files in os.walk(path):
            found_exclusion = False
            for exclusion in exclude:
                if exclusion in root:
                    found_exclusion = True
            if found_exclusion:
                continue
            for file in files:
                if file[-4:] == ".cpp":
                    ret.append(os.path.join(root, file))
    else:
        for file in os.listdir(path):
            found_exclusion = False
            for exclusion in exclude:
                if exclusion in file:
                    found_exclusion = True
            if found_exclusion:
                continue
            if file[-4:] == ".cpp":
                ret.append(os.path.join(path, file))
    return ret

extensions = [Extension("pybsts", 
    ["Interfaces/Python/bsts/pybsts" + file_ext,
     "Interfaces/Python/bsts/model_manager.cpp",
     "Interfaces/Python/bsts/state_space_gaussian_model_manager.cpp",
     "Interfaces/Python/bsts/state_space_regression_model_manager.cpp",
     "Interfaces/Python/bsts/prior_specification.cpp",
     "Interfaces/Python/bsts/spike_slab_prior.cpp",
     "Interfaces/Python/bsts/create_state_model.cpp",
     "Interfaces/Python/bsts/list_io.cpp"] + \
     find_cpp("distributions/") + \
     find_cpp("Models/", exclude=["Bart", "Nnet", "test", "HMM", "IRT", "Mixtures"]) + \
     find_cpp(".", recursive=False) + \
     find_cpp("LinAlg/") + \
     find_cpp("math/") + \
     find_cpp("numopt/") + \
     find_cpp("Bmath/") + \
     find_cpp("Samplers/", exclude=["failed_experiments"]) + \
     find_cpp("stats/", exclude=["test"]) + \
     find_cpp("TargetFun/") + \
     find_cpp("cpputil/"),
    include_dirs=INCLUDE_DIRS,
    language="c++",
    libraries=[],
    library_dirs=[],
    extra_compile_args=['-std=c++11', '-DADD_'] + ["-I"+include_dir for include_dir in INCLUDE_DIRS],
    extra_link_args=[])]

if HAS_CYTHON:
    from Cython.Build import cythonize
    extensions = cythonize(extensions, compiler_directives={'language_level': 3})

setup(
    name="pybsts",
    author="Vitalii Ostrovskyi",
    author_email="vitos@vitos.org.ua",
    description="Python interface to Bayesian Structured Time Series",
    version='1.0.5',
    include_package_data=True,
    ext_modules=extensions,
    install_requires=['cython', 'numpy'],
    packages=['causal_impact'],
    package_dir={'causal_impact': 'Interfaces/Python/causal_impact'},
)
