// Copyright 2018 Google LLC. All Rights Reserved.
/*
  Copyright (C) 2005-2012 Steven L. Scott

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
*/

#ifndef BOOM_LINALG_EIGEN_HPP_
#define BOOM_LINALG_EIGEN_HPP_
#include <complex>
#include <vector>
#include "LinAlg/Matrix.hpp"
#include "LinAlg/SpdMatrix.hpp"
#include "LinAlg/Vector.hpp"

namespace BOOM {
  // This file contains tools to buld the eigen decomposition of
  // (a) A possibly rectangular matrix, and
  // (b) A symmetric, positive definite matrix.
  //
  // The package 'Eigen' is used under the covers to do the heavy lifting, but
  // these tools are not part of the Eigen package.
  //
  // TODO: rename this file to avoid confusion with the Eigen package.

  // Eigenstructure of a square, non-symmetric matrix.
  class EigenDecomposition {
   public:
    // Args:
    //   mat: The (square) matrix for which the eigendecomposition is desired.
    //   vectors: If 'true' then both eigenvalues and eigenvectors are computed.
    //     If 'false' then only eigenvalues are computed.
    EigenDecomposition(const Matrix &mat, bool vectors = true);

    // Complex conjugate eigenvalues occur consecutively.  The entry
    // with the positive imaginary part comes first.
    std::vector<std::complex<double>> eigenvalues() const {
      return eigenvalues_;
    }

    // The real and imaginary parts of all the eigenvalues.  If all
    // eigenvalues are real then the imaginary_values() will be a
    // vector of zeros (up to numerical accuracy).
    const Vector &real_eigenvalues() const { return real_eigenvalues_; }
    const Vector &imaginary_eigenvalues() const {
      return imaginary_eigenvalues_;
    }

    // Requests for eigenvectors will throw exceptions if eigenvectors
    // were not requested by the constructor.
    ConstVectorView real_eigenvector(int i) const;
    ConstVectorView imaginary_eigenvector(int i) const;
    std::vector<std::complex<double>> eigenvector(int i) const;

   private:
    std::vector<std::complex<double>> eigenvalues_;
    Vector real_eigenvalues_;
    Vector imaginary_eigenvalues_;

    // The real and imaginary parts of the (right) eigenvectors.  Each column is
    // (part of) an eigenvector.
    Matrix real_eigenvectors_;
    Matrix imaginary_eigenvectors_;
  };

  // Eigenvalues and vectors of an SpdMatrix.
  class SpdEigen {
   public:
    // Args:
    //   matrix:  The matrix whose eigendecomposition is desired.
    //   compute_vectors: If true then eigenvalues and eigenvectors are both
    //     computed.  If false only the eigenvalues are computed.
    SpdEigen(const SpdMatrix &matrix, bool compute_vectors);

    const Vector &eigenvalues() const { return eigenvalues_; }

    // This matrix is size zero if eigenvectors were not requested.
    const Matrix &eigenvectors() const { return right_vectors_; }

   private:
    Vector eigenvalues_;

    // The eigenvectors are the columns of right_vectors.  This matrix will be
    // of size zero if compute_vectors==false in the constructor.
    Matrix right_vectors_;
  };
}  // namespace BOOM

#endif  //  BOOM_LINALG_EIGEN_HPP_
