"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ScheduleExpression = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const events = require("aws-cdk-lib/aws-events");
const core_1 = require("aws-cdk-lib/core");
/**
 * ScheduleExpression for EventBridge Schedule
 *
 * You can choose from three schedule types when configuring your schedule: rate-based, cron-based, and one-time schedules.
 * Both rate-based and cron-based schedules are recurring schedules.
 *
 * @see https://docs.aws.amazon.com/scheduler/latest/UserGuide/schedule-types.html
 */
class ScheduleExpression {
    /**
     * Construct a one-time schedule from a date.
     *
     * @param date The date and time to use. The millisecond part will be ignored.
     * @param timeZone The time zone to use for interpreting the date. Default: - UTC
     */
    static at(date, timeZone) {
        try {
            const literal = date.toISOString().split('.')[0];
            return new LiteralScheduleExpression(`at(${literal})`, timeZone ?? core_1.TimeZone.ETC_UTC);
        }
        catch (e) {
            if (e instanceof RangeError) {
                throw new Error('Invalid date');
            }
            throw e;
        }
    }
    /**
     * Construct a schedule from a literal schedule expression
     * @param expression The expression to use. Must be in a format that EventBridge will recognize
     * @param timeZone The time zone to use for interpreting the expression. Default: - UTC
     */
    static expression(expression, timeZone) {
        return new LiteralScheduleExpression(expression, timeZone ?? core_1.TimeZone.ETC_UTC);
    }
    /**
     * Construct a recurring schedule from an interval and a time unit
     *
     * Rates may be defined with any unit of time, but when converted into minutes, the duration must be a positive whole number of minutes.
     */
    static rate(duration) {
        const schedule = events.Schedule.rate(duration);
        return new LiteralScheduleExpression(schedule.expressionString);
    }
    /**
     * Create a recurring schedule from a set of cron fields and time zone.
     */
    static cron(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_scheduler_alpha_CronOptionsWithTimezone(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.cron);
            }
            throw error;
        }
        const { timeZone, ...cronOptions } = options;
        const schedule = events.Schedule.cron(cronOptions);
        return new LiteralScheduleExpression(schedule.expressionString, timeZone);
    }
    constructor() { }
}
exports.ScheduleExpression = ScheduleExpression;
_a = JSII_RTTI_SYMBOL_1;
ScheduleExpression[_a] = { fqn: "@aws-cdk/aws-scheduler-alpha.ScheduleExpression", version: "2.96.1-alpha.0" };
const DEFAULT_TIMEZONE = core_1.TimeZone.ETC_UTC;
class LiteralScheduleExpression extends ScheduleExpression {
    constructor(expressionString, timeZone = DEFAULT_TIMEZONE) {
        super();
        this.expressionString = expressionString;
        this.timeZone = timeZone;
    }
}
//# sourceMappingURL=data:application/json;base64,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