import * as events from 'aws-cdk-lib/aws-events';
import { Duration, TimeZone } from 'aws-cdk-lib/core';
/**
 * ScheduleExpression for EventBridge Schedule
 *
 * You can choose from three schedule types when configuring your schedule: rate-based, cron-based, and one-time schedules.
 * Both rate-based and cron-based schedules are recurring schedules.
 *
 * @see https://docs.aws.amazon.com/scheduler/latest/UserGuide/schedule-types.html
 */
export declare abstract class ScheduleExpression {
    /**
     * Construct a one-time schedule from a date.
     *
     * @param date The date and time to use. The millisecond part will be ignored.
     * @param timeZone The time zone to use for interpreting the date. Default: - UTC
     */
    static at(date: Date, timeZone?: TimeZone): ScheduleExpression;
    /**
     * Construct a schedule from a literal schedule expression
     * @param expression The expression to use. Must be in a format that EventBridge will recognize
     * @param timeZone The time zone to use for interpreting the expression. Default: - UTC
     */
    static expression(expression: string, timeZone?: TimeZone): ScheduleExpression;
    /**
     * Construct a recurring schedule from an interval and a time unit
     *
     * Rates may be defined with any unit of time, but when converted into minutes, the duration must be a positive whole number of minutes.
     */
    static rate(duration: Duration): ScheduleExpression;
    /**
     * Create a recurring schedule from a set of cron fields and time zone.
     */
    static cron(options: CronOptionsWithTimezone): ScheduleExpression;
    /**
     * Retrieve the expression for this schedule
     */
    abstract readonly expressionString: string;
    /**
     * Retrieve the expression for this schedule
     */
    abstract readonly timeZone?: TimeZone;
    protected constructor();
}
/**
 * Options to configure a cron expression
 *
 * All fields are strings so you can use complex expressions. Absence of
 * a field implies '*' or '?', whichever one is appropriate.
 *
 * @see https://docs.aws.amazon.com/eventbridge/latest/userguide/scheduled-events.html#cron-expressions
 */
export interface CronOptionsWithTimezone extends events.CronOptions {
    /**
     * The timezone to run the schedule in
     *
     * @default - TimeZone.ETC_UTC
     */
    readonly timeZone?: TimeZone;
}
