"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Group = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const iam = require("aws-cdk-lib/aws-iam");
const aws_scheduler_1 = require("aws-cdk-lib/aws-scheduler");
const core_1 = require("aws-cdk-lib/core");
class GroupBase extends core_1.Resource {
    /**
     * Return the given named metric for this group schedules
     *
     * @default - sum over 5 minutes
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/Scheduler',
            metricName,
            dimensionsMap: { ScheduleGroup: this.groupName },
            statistic: 'sum',
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for the number of invocations that were throttled because it exceeds your service quotas.
     *
     * @see https://docs.aws.amazon.com/scheduler/latest/UserGuide/scheduler-quotas.html
     *
     * @default - sum over 5 minutes
     */
    metricThrottled(props) {
        return this.metric('InvocationThrottleCount', props);
    }
    /**
     * Metric for all invocation attempts.
     *
     * @default - sum over 5 minutes
     */
    metricAttempts(props) {
        return this.metric('InvocationAttemptCount', props);
    }
    /**
     * Emitted when the target returns an exception after EventBridge Scheduler calls the target API.
     *
     * @default - sum over 5 minutes
     */
    metricTargetErrors(props) {
        return this.metric('TargetErrorCount', props);
    }
    /**
     * Metric for invocation failures due to API throttling by the target.
     *
     * @default - sum over 5 minutes
     */
    metricTargetThrottled(props) {
        return this.metric('TargetErrorThrottledCount', props);
    }
    /**
     * Metric for dropped invocations when EventBridge Scheduler stops attempting to invoke the target after a schedule's retry policy has been exhausted.
     *
     * @default - sum over 5 minutes
     */
    metricDropped(props) {
        return this.metric('InvocationDroppedCount', props);
    }
    /**
     * Metric for invocations delivered to the DLQ
     *
     * @default - sum over 5 minutes
     */
    metricSentToDLQ(props) {
        return this.metric('InvocationsSentToDeadLetterCount', props);
    }
    /**
     * Metric for failed invocations that also failed to deliver to DLQ.
     *
     * @default - sum over 5 minutes
     */
    metricFailedToBeSentToDLQ(errorCode, props) {
        if (errorCode) {
            return this.metric(`InvocationsFailedToBeSentToDeadLetterCount_${errorCode}`, props);
        }
        return this.metric('InvocationsFailedToBeSentToDeadLetterCount', props);
    }
    /**
     * Metric for delivery of failed invocations to DLQ when the payload of the event sent to the DLQ exceeds the maximum size allowed by Amazon SQS.
     *
     * @default - sum over 5 minutes
     */
    metricSentToDLQTrunacted(props) {
        return this.metric('InvocationsSentToDeadLetterCount_Truncated_MessageSizeExceeded', props);
    }
    /**
     * Grant the indicated permissions on this group to the given principal
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.groupArn],
            scope: this,
        });
    }
    arnForScheduleInGroup(scheduleName) {
        return core_1.Arn.format({
            region: this.env.region,
            account: this.env.account,
            partition: core_1.Aws.PARTITION,
            service: 'scheduler',
            resource: 'schedule',
            resourceName: this.groupName + '/' + scheduleName,
        });
    }
    /**
     * Grant list and get schedule permissions for schedules in this group to the given principal
     */
    grantReadSchedules(identity) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: ['scheduler:GetSchedule', 'scheduler:ListSchedules'],
            resourceArns: [this.arnForScheduleInGroup('*')],
            scope: this,
        });
    }
    /**
     * Grant create and update schedule permissions for schedules in this group to the given principal
     */
    grantWriteSchedules(identity) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: ['scheduler:CreateSchedule', 'scheduler:UpdateSchedule'],
            resourceArns: [this.arnForScheduleInGroup('*')],
            scope: this,
        });
    }
    /**
     * Grant delete schedule permission for schedules in this group to the given principal
     */
    grantDeleteSchedules(identity) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: ['scheduler:DeleteSchedule'],
            resourceArns: [this.arnForScheduleInGroup('*')],
            scope: this,
        });
    }
}
class Group extends GroupBase {
    /**
     * Import an external group by ARN.
     *
     * @param scope construct scope
     * @param id construct id
     * @param groupArn the ARN of the group to import (e.g. `arn:aws:scheduler:region:account-id:schedule-group/group-name`)
     */
    static fromGroupArn(scope, id, groupArn) {
        const arnComponents = core_1.Stack.of(scope).splitArn(groupArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        const groupName = arnComponents.resourceName;
        class Import extends GroupBase {
            constructor() {
                super(...arguments);
                this.groupName = groupName;
                this.groupArn = groupArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import a default schedule group.
     *
     * @param scope construct scope
     * @param id construct id
     */
    static fromDefaultGroup(scope, id) {
        return Group.fromGroupName(scope, id, 'default');
    }
    /**
     * Import an existing group with a given name.
     *
     * @param scope construct scope
     * @param id construct id
     * @param groupName the name of the existing group to import
     */
    static fromGroupName(scope, id, groupName) {
        const groupArn = core_1.Stack.of(scope).formatArn({
            service: 'scheduler',
            resource: 'schedule-group',
            resourceName: groupName,
        });
        return Group.fromGroupArn(scope, id, groupArn);
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.groupName ?? core_1.PhysicalName.GENERATE_IF_NEEDED,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_scheduler_alpha_GroupProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Group);
            }
            throw error;
        }
        const group = new aws_scheduler_1.CfnScheduleGroup(this, 'Resource', {
            name: this.physicalName,
        });
        group.applyRemovalPolicy(props.removalPolicy);
        this.groupArn = this.getResourceArnAttribute(group.attrArn, {
            service: 'scheduler',
            resource: 'schedule-group',
            resourceName: this.physicalName,
        });
        this.groupName = this.physicalName;
    }
}
exports.Group = Group;
_a = JSII_RTTI_SYMBOL_1;
Group[_a] = { fqn: "@aws-cdk/aws-scheduler-alpha.Group", version: "2.96.1-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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