import * as cloudwatch from 'aws-cdk-lib/aws-cloudwatch';
import * as iam from 'aws-cdk-lib/aws-iam';
import { IResource, RemovalPolicy, Resource } from 'aws-cdk-lib/core';
import { Construct } from 'constructs';
export interface GroupProps {
    /**
     * The name of the schedule group.
     *
     * Up to 64 letters (uppercase and lowercase), numbers, hyphens, underscores and dots are allowed.
     *
     * @default - A unique name will be generated
     */
    readonly groupName?: string;
    /**
     * The removal policy for the group. If the group is removed also all schedules are removed.
     *
     * @default RemovalPolicy.RETAIN
     */
    readonly removalPolicy?: RemovalPolicy;
}
export interface IGroup extends IResource {
    /**
     * The name of the schedule group
     *
     * @attribute
     */
    readonly groupName: string;
    /**
     * The arn of the schedule group
     *
     * @attribute
     */
    readonly groupArn: string;
    /**
     * Return the given named metric for this group schedules
     *
     * @default - sum over 5 minutes
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of invocations that were throttled because it exceeds your service quotas.
     *
     * @see https://docs.aws.amazon.com/scheduler/latest/UserGuide/scheduler-quotas.html
     *
     * @default - sum over 5 minutes
     */
    metricThrottled(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for all invocation attempts.
     *
     * @default - sum over 5 minutes
     */
    metricAttempts(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Emitted when the target returns an exception after EventBridge Scheduler calls the target API.
     *
     * @default - sum over 5 minutes
     */
    metricTargetErrors(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for invocation failures due to API throttling by the target.
     *
     * @default - sum over 5 minutes
     */
    metricTargetThrottled(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for dropped invocations when EventBridge Scheduler stops attempting to invoke the target after a schedule's retry policy has been exhausted.
     *
     * @default - sum over 5 minutes
     */
    metricDropped(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for invocations delivered to the DLQ
     *
     * @default - sum over 5 minutes
     */
    metricSentToDLQ(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for failed invocations that also failed to deliver to DLQ.
     *
     * @default - sum over 5 minutes
     */
    metricFailedToBeSentToDLQ(errorCode?: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for delivery of failed invocations to DLQ when the payload of the event sent to the DLQ exceeds the maximum size allowed by Amazon SQS.
     *
     * @default - sum over 5 minutes
     */
    metricSentToDLQTrunacted(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Grant the indicated permissions on this group to the given principal
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Grant list and get schedule permissions for schedules in this group to the given principal
     */
    grantReadSchedules(identity: iam.IGrantable): iam.Grant;
    /**
     * Grant create and update schedule permissions for schedules in this group to the given principal
     */
    grantWriteSchedules(identity: iam.IGrantable): iam.Grant;
    /**
     * Grant delete schedule permission for schedules in this group to the given principal
     */
    grantDeleteSchedules(identity: iam.IGrantable): iam.Grant;
}
declare abstract class GroupBase extends Resource implements IGroup {
    /**
     * The name of the schedule group
     *
     * @attribute
     */
    abstract readonly groupName: string;
    /**
     * The arn of the schedule group
     *
     * @attribute
     */
    abstract readonly groupArn: string;
    /**
     * Return the given named metric for this group schedules
     *
     * @default - sum over 5 minutes
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of invocations that were throttled because it exceeds your service quotas.
     *
     * @see https://docs.aws.amazon.com/scheduler/latest/UserGuide/scheduler-quotas.html
     *
     * @default - sum over 5 minutes
     */
    metricThrottled(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for all invocation attempts.
     *
     * @default - sum over 5 minutes
     */
    metricAttempts(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Emitted when the target returns an exception after EventBridge Scheduler calls the target API.
     *
     * @default - sum over 5 minutes
     */
    metricTargetErrors(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for invocation failures due to API throttling by the target.
     *
     * @default - sum over 5 minutes
     */
    metricTargetThrottled(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for dropped invocations when EventBridge Scheduler stops attempting to invoke the target after a schedule's retry policy has been exhausted.
     *
     * @default - sum over 5 minutes
     */
    metricDropped(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for invocations delivered to the DLQ
     *
     * @default - sum over 5 minutes
     */
    metricSentToDLQ(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for failed invocations that also failed to deliver to DLQ.
     *
     * @default - sum over 5 minutes
     */
    metricFailedToBeSentToDLQ(errorCode?: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for delivery of failed invocations to DLQ when the payload of the event sent to the DLQ exceeds the maximum size allowed by Amazon SQS.
     *
     * @default - sum over 5 minutes
     */
    metricSentToDLQTrunacted(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Grant the indicated permissions on this group to the given principal
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    private arnForScheduleInGroup;
    /**
     * Grant list and get schedule permissions for schedules in this group to the given principal
     */
    grantReadSchedules(identity: iam.IGrantable): iam.Grant;
    /**
     * Grant create and update schedule permissions for schedules in this group to the given principal
     */
    grantWriteSchedules(identity: iam.IGrantable): iam.Grant;
    /**
     * Grant delete schedule permission for schedules in this group to the given principal
     */
    grantDeleteSchedules(identity: iam.IGrantable): iam.Grant;
}
export declare class Group extends GroupBase {
    /**
     * Import an external group by ARN.
     *
     * @param scope construct scope
     * @param id construct id
     * @param groupArn the ARN of the group to import (e.g. `arn:aws:scheduler:region:account-id:schedule-group/group-name`)
     */
    static fromGroupArn(scope: Construct, id: string, groupArn: string): IGroup;
    /**
     * Import a default schedule group.
     *
     * @param scope construct scope
     * @param id construct id
     */
    static fromDefaultGroup(scope: Construct, id: string): IGroup;
    /**
     * Import an existing group with a given name.
     *
     * @param scope construct scope
     * @param id construct id
     * @param groupName the name of the existing group to import
     */
    static fromGroupName(scope: Construct, id: string, groupName: string): IGroup;
    readonly groupName: string;
    readonly groupArn: string;
    constructor(scope: Construct, id: string, props: GroupProps);
}
export {};
