"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OrganizationOU = exports.OrganizationOUProvider = exports.processOrgObj = exports.OrgObjectTypes = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
/**
 * The supported OrgObject types.
 *
 * @stability stable
 */
var OrgObjectTypes;
(function (OrgObjectTypes) {
    OrgObjectTypes["OU"] = "ou";
    OrgObjectTypes["ACCOUNT"] = "account";
})(OrgObjectTypes = exports.OrgObjectTypes || (exports.OrgObjectTypes = {}));
;
;
/**
 * @function processOrgObj
 * Function to process an OrgObject and create the corresponding AWS resources
 *
 * @param {Construct} this The construct resources will be added to.
 * @param {custom_resources.Provider} provider The custom resource provider the custom resources will utilized to create the resources.
 * @param {OrgObject} obj The OrgObject to process.
 * @param {string | OrganizationOU} parent The parent of the OrgObject. This is either a string, like for the org root, or an OrganizationOU object from the same stack.
 */
function processOrgObj(provider, obj, parent) {
    var _c;
    if (obj.type === OrgObjectTypes.OU) {
        const parentStr = parent instanceof OrganizationOU ? parent.resource.ref : parent;
        let props = obj.properties;
        let id = (_c = obj.id) !== null && _c !== void 0 ? _c : obj.properties.name;
        const ou = new OrganizationOU(this, id, {
            provider,
            parent: parentStr,
            name: props.name,
            importOnDuplicate: props.importOnDuplicate,
            allowRecreateOnUpdate: props.allowRecreateOnUpdate,
        });
        obj.children.forEach(child => {
            processOrgObj.call(this, provider, child, ou);
        });
    }
}
exports.processOrgObj = processOrgObj;
/**
 * The provider for OU custom resources.
 *
 * This creates a lambda function that handles custom resource requests for creating/updating/deleting OUs.
 *
 * @stability stable
 */
class OrganizationOUProvider extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        let role;
        if (!props.role) {
            role = new aws_cdk_lib_1.aws_iam.Role(this, 'role', {
                assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            let policy = new aws_cdk_lib_1.aws_iam.ManagedPolicy(this, 'policy', {
                statements: [
                    new aws_cdk_lib_1.aws_iam.PolicyStatement({
                        actions: [
                            'organizations:UpdateOrganizationalUnit',
                            'organizations:DeleteOrganizationalUnit',
                            'organizations:ListOrganizationalUnitsForParent',
                            'organizations:CreateOrganizationalUnit',
                        ],
                        resources: ['*'],
                    }),
                ],
            });
            role.addManagedPolicy(policy);
            role.addManagedPolicy(aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'));
        }
        else {
            role = props.role;
        }
        const handlersPath = path.join(__dirname, '../handlers');
        const onEvent = new aws_cdk_lib_1.aws_lambda.Function(this, 'handler', {
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_9,
            code: aws_cdk_lib_1.aws_lambda.Code.fromAsset(handlersPath + '/ou'),
            handler: 'index.on_event',
            timeout: aws_cdk_lib_1.Duration.seconds(10),
            role,
        });
        this.provider = new aws_cdk_lib_1.custom_resources.Provider(this, 'provider', {
            onEventHandler: onEvent,
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_DAY,
        });
    }
}
exports.OrganizationOUProvider = OrganizationOUProvider;
_a = JSII_RTTI_SYMBOL_1;
OrganizationOUProvider[_a] = { fqn: "@renovosolutions/cdk-library-aws-organization.OrganizationOUProvider", version: "0.3.1" };
/**
 * The construct to create or update an Organization OU.
 *
 * This relies on the custom resource provider OrganizationOUProvider.
 *
 * @stability stable
 */
class OrganizationOU extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _c, _d;
        super(scope, id);
        const importOnDuplicate = (_c = props.importOnDuplicate) !== null && _c !== void 0 ? _c : false;
        const allowRecreateOnUpdate = (_d = props.allowRecreateOnUpdate) !== null && _d !== void 0 ? _d : false;
        const parentStr = props.parent instanceof OrganizationOU ? props.parent.resource.ref : props.parent;
        this.resource = new aws_cdk_lib_1.CustomResource(this, 'ou', {
            serviceToken: props.provider.serviceToken,
            properties: {
                Parent: parentStr,
                Name: props.name,
                ImportOnDuplicate: importOnDuplicate,
                AllowRecreateOnUpdate: allowRecreateOnUpdate,
            },
        });
        this.resource.node.addDependency(props.provider);
        if (props.parent instanceof OrganizationOU) {
            this.resource.node.addDependency(props.parent);
        }
        ;
    }
}
exports.OrganizationOU = OrganizationOU;
_b = JSII_RTTI_SYMBOL_1;
OrganizationOU[_b] = { fqn: "@renovosolutions/cdk-library-aws-organization.OrganizationOU", version: "0.3.1" };
//# sourceMappingURL=data:application/json;base64,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