# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['ptera']

package_data = \
{'': ['*']}

install_requires = \
['giving>=0.3.6,<0.4.0']

extras_require = \
{':python_version >= "3.6" and python_version < "3.7"': ['contextvars>=2.4,<3.0'],
 ':python_version >= "3.8" and python_version < "4.0"': ['codefind>=0.1.2,<0.2.0']}

setup_kwargs = {
    'name': 'ptera',
    'version': '0.5.1',
    'description': 'Call graph addressing library.',
    'long_description': '\n# Ptera\n\nPtera is a powerful way to instrument your code for logging, debugging and testing purposes. With a simple call to `ptera.probing()`, you can:\n\n* Obtain a stream of the values taken by any variable.\n* Probe multiple variables from multiple functions in multiple scopes.\n* Apply maps, filters, reductions, and much more to the streams.\n* Override the values of variables based on complex conditions.\n* Create external asserts or conditional breakpoints.\n* Et cetera :)\n\nThe main interface to ptera are the `global_probe` and `probing` functions. The only difference between them is that the first applies globally whereas the second is a context manager and applies only to the code inside a block:\n\n```python\nfrom ptera import global_probe, probing\n\ndef f(x):\n    y = x * x\n    return y + 1\n\nglobal_probe("f > y").print()\n\nf(9)  # prints {"y": 81}\n\nwith probing("f > y") as probe:\n    probe.print("y = {y}")\n\n    f(10)  # prints {"y": 100} and "y = 100"\n\nf(11)  # prints {"y": 121}\n```\n\n`print()` is only one of a myriad operators. Ptera\'s interface is inspired from functional reactive programming and is identical to the interface of [giving](https://github.com/breuleux/giving) (itself based on `rx`). [See here for a more complete list of operators.](https://giving.readthedocs.io/en/latest/ref-operators.html)\n\n\nNote: reduction operators such as `min` or `sum` are applied at program exit for `global_probe` or at the end of the `with` block with `probing`, so it is usually best to use `probing` for these.\n\n\n## Examples\n\nPtera is all about providing new ways to inspect what your programs are doing, so all examples will be based on this simple binary search function:\n\n```python\nfrom ptera import global_probe, probing\n\ndef f(arr, key):\n    lo = -1\n    hi = len(arr)\n    while lo < hi - 1:\n        mid = lo + (hi - lo) // 2\n        if (elem := arr[mid]) > key:\n            hi = mid\n        else:\n            lo = mid\n    return lo + 1\n\n##############################\n# THE PROBING CODE GOES HERE #\n##############################\n\nf(list(range(1, 350, 7)), 136)\n```\n\nTo get the output listed in the right column of the table below, the code in the left column should be inserted before the call to `f`, where the big comment is. Most of the methods on `global_probe` define the pipeline through which the probed values will be routed (the interface is inspired from functional reactive programming), so it is important to define them before the instrumented functions are called.\n\n<table>\n<tr>\n<th>Code</th>\n<th>Output</th>\n</tr>\n\n<!--\n####################\n####### ROW ########\n####################\n-->\n\n<tr>\n<td>\n\nThe `display` method provides a simple way to log values.\n\n```python\nglobal_probe("f > mid").display()\n```\n\n</td>\n<td>\n\n```\nmid: 24\nmid: 11\nmid: 17\nmid: 20\nmid: 18\nmid: 19\n```\n\n</td>\n</tr>\n<tr></tr>\n\n<!--\n####################\n####### ROW ########\n####################\n-->\n\n<tr>\n<td>\n\nThe `print` method lets you specify a format string.\n\n```python\nglobal_probe("f(mid) > elem").print("arr[{mid}] == {elem}")\n```\n\n</td>\n<td>\n\n```\narr[24] == 169\narr[11] == 78\narr[17] == 120\narr[20] == 141\narr[18] == 127\narr[19] == 134\n```\n\n</td>\n</tr>\n<tr></tr>\n\n<!--\n####################\n####### ROW ########\n####################\n-->\n\n<tr>\n<td>\n\nReductions are easy: extract the key and use `min`, `max`, etc.\n\n```python\nglobal_probe("f > lo")["lo"].max().print("max(lo) = {}")\nglobal_probe("f > hi")["hi"].min().print("min(hi) = {}")\n```\n\n</td>\n<td>\n\n```\nmax(lo) = 19\nmin(hi) = 20\n```\n\n</td>\n</tr>\n<tr></tr>\n\n<!--\n####################\n####### ROW ########\n####################\n-->\n\n<tr>\n<td>\n\nDefine assertions with `fail()` (for debugging, also try `.breakpoint()`!)\n\n```python\ndef unordered(xs):\n    return any(x > y for x, y in zip(xs[:-1], xs[1:]))\n\nprobe = global_probe("f > arr")["arr"]\nprobe.filter(unordered).fail("List is unordered: {}")\n\nf([1, 6, 30, 7], 18)\n```\n\n</td>\n<td>\n\n```\nTraceback (most recent call last):\n  ...\n  File "test.py", line 30, in <module>\n    f([1, 6, 30, 7], 18)\n  File "<string>", line 3, in f__ptera_redirect\n  File "test.py", line 3, in f\n    def f(arr, key):\ngiving.gvn.Failure: List is unordered: [1, 6, 30, 7]\n```\n\n</td>\n</tr>\n<tr></tr>\n\n<!--\n####################\n####### ROW ########\n####################\n-->\n\n<tr>\n<td>\n\nAccumulate into a list:\n\n```python\nresults = global_probe("f > mid")["mid"].accum()\nf(list(range(1, 350, 7)), 136)\nprint(results)\n```\n\nOR\n\n```python\nwith probing("f > mid")["mid"].values() as results:\n    f(list(range(1, 350, 7)), 136)\n\nprint(results)\n```\n\n</td>\n<td>\n\n```\n[24, 11, 17, 20, 18, 19]\n```\n\n</td>\n</tr>\n<tr></tr>\n\n</table>\n\n\n## probing\n\nUsage: `with ptera.probing(selector) as probe: ...`\n\nThe **selector** is a specification of which variables in which functions we want to stream through the probe. One of the variables must be the **focus** of the selector, meaning that the probe is triggered when *that* variable is set. The focus may be indicated either as `f(!x)` or `f > x` (the focus is `x` in both cases).\n\nThe **probe** is a wrapper around [rx.Observable](https://github.com/ReactiveX/RxPY) and supports a large number of [operators](https://giving.readthedocs.io/en/latest/ref-operators.html) such as `map`, `filter`, `min`, `average`, `throttle`, etc. (the interface is the same as in [giving](https://github.com/breuleux/giving)).\n\n\n### Example 1: intermediate variables\n\nPtera is capable of capturing any variable in a function, not just inputs and return values:\n\n```python\ndef fact(n):\n    curr = 1\n    for i in range(n):\n        curr = curr * (i + 1)\n    return curr\n\nwith probing("fact(i, !curr)").print():\n    fact(3)\n    # {\'curr\': 1}\n    # {\'curr\': 1, \'i\': 0}\n    # {\'curr\': 2, \'i\': 1}\n    # {\'curr\': 6, \'i\': 2}\n```\n\nThe "!" in the selector above means that the focus is `curr`. This means it is triggered when `curr` is set. This is why the first result does not have a value for `i`. You can use the selector `fact(!i, curr)` to focus on `i` instead:\n\n```python\nwith probing("fact(!i, curr)").print():\n    fact(3)\n    # {\'i\': 0, \'curr\': 1}\n    # {\'i\': 1, \'curr\': 1}\n    # {\'i\': 2, \'curr\': 2}\n```\n\nYou can see that the associations are different (curr is 2 when i is 2, whereas it was 6 with the other selector), but this is simply because they are now triggered when `i` is set.\n\n### Example 2: multiple scopes\n\nA selector may act on several nested scopes in a call graph. For example, the selector `f(x) >> g(y) >> h > z` would capture variables `x`, `y` and `z` from the scopes of three different functions, but only when `f` calls `g` and `g` calls `h` (either directly or indirectly). (Note: `f(x) > g(y) > h > z` is also legal and is supposed to represent direct calls, but it may behave in confusing ways depending on which functions are instrumented globally, so avoid it for the time being).\n\n```python\ndef f(x):\n    return g(x + 1) * g(-x - 1)\n\ndef g(x):\n    return x * 2\n\n# Use "as" to rename a variable if there is a name conflict\nwith probing("f(x) >> g > x as gx").print():\n    f(5)\n    # {\'gx\': 6, \'x\': 5}\n    # {\'gx\': -6, \'x\': 5}\n    g(10)\n    # Prints nothing\n```\n\n### Example 3: sibling calls\n\nSelectors can also specify variables on different paths in the call graph. For example:\n\n```python\ndef f(x):\n    v = g(x + 1) * h(-x - 1)\n    return v\n\ndef g(y):\n    return y * 2\n\ndef h(z):\n    return z * 3\n\nwith probing("f(x, g(y), h(!z))") as probe:\n    probe.print()\n    f(10)\n    # {\'z\': -11, \'x\': 10, \'y\': 11}\n```\n\nRemember to set the focus with `!`. It should ideally be on the last variable to be set.\n\nThere is currently no error if you don\'t set a focus, it will simply do nothing, so beware of that for the time being.\n\n### Example 4: tagging variables\n\nUsing annotations, variables can be given various tags, and probes can use these tags instead of variable names.\n\n```python\ndef fishy(x):\n    a: "@fish" = x + 1\n    b: "@fish & @trout" = x + 2\n    return a * b\n\nwith probing("fishy > $x:@trout").print():\n    fishy(10)\n    # {\'x\': 12}\n\nwith probing("fishy > $x:@fish").print():\n    fishy(10)\n    # {\'x\': 11}\n    # {\'x\': 12}\n```\n\nThe `$x` syntax means that we are not matching a variable called `x`, but instead matching any variable that has the right condition (in this case, the tags fish or trout) and offering it under the name `x`. You can pass `raw=True` to `probing` to get `Capture` objects instead of values. The `Capture` object gives access to the variable\'s actual name. For example:\n\n```python\nwith probing("fishy > $x:@fish", raw=True) as probe:\n    probe["x"].map(lambda x: {x.name: x.value}).print()\n    fishy(10)\n    # {\'a\': 11}\n    # {\'b\': 12}\n```\n\n\n### Example 5: overriding variables\n\nIt is also possible to override the value of a variable with the `override` (or `koverride`) methods:\n\n\n```python\ndef add_ct(x):\n    ct = 1\n    return x + ct\n\nwith probing("add_ct(x) > ct") as probe:\n    # The value of other variables can be used to compute the new value of ct\n    probe.override(lambda data: data["x"])\n\n    # You can also use koverride, which calls func(**data)\n    # probe.koverride(lambda x: x)\n\n    print(add_ct(3))   # sets ct = x = 3; prints 6\n    print(add_ct(10))  # sets ct = x = 20; prints 20\n```\n\n**Important:** override() only overrides the **focus variable**. As explained earlier, the focus variable is the one to the right of `>`, or the one prefixed with `!`. A Ptera selector is only triggered when the focus variable is set, so realistically it is the only one that it makes sense to override.\n\nThis is worth keeping in mind, because otherwise it\'s not always obvious what override is doing. For example:\n\n```python\nwith probing("add_ct(x) > ct") as probe:\n    # The focus is ct, so override will always set ct\n    # Therefore, this sets ct = 10 when x == 3:\n    probe.where(x=3).override(10)\n\n    print(add_ct(3))   # sets ct = 10; prints 13\n    print(add_ct(10))  # does not override anything; prints 11\n```\n\n\n## global_probe\n\n`global_probe` works more or less the same way as `probing`, but it is not a context manager: it just works globally from the moment of its creation. This means that streams created with `global_probe` only end when the program ends, so operators that wait for the full stream before triggering, such as `min()`, will run at program exit, which limits their usefulness.\n\n```python\nglobal_probe("fact() as result").print()\nfact(2)\n# {\'result\': 1}\n# {\'result\': 2}\nfact(3)\n# {\'result\': 1}\n# {\'result\': 2}\n# {\'result\': 6}\n```\n\n## Absolute probes\n\nHere is a notation to probe a function using an "absolute path" in the module system:\n\n```python\nglobal_probe("/xyz.submodule/Klass/method > x")\n\n# is essentially equivalent to:\n\nfrom xyz.submodule import Klass\nglobal_probe("Klass.method > x")\n```\n\nThe slashes represent a physical nesting rather than object attributes. For example, `/module.submodule/x/y` means:\n\n* Go in the file that defines `module.submodule`\n* Enter `def x` or `class x` (it will *not* work if `x` is imported from elsewhere)\n* Within that definition, enter `def y` or `class y`\n\nNote:\n\n* Unlike the normal notation, the absolute notation bypasses decorators: `/module/function` will probe the function inside the `def function(): ...` in `module.py`, so it will work even if the function was wrapped by a decorator (unless the decorator does not actually call the function).\n* Although the `/module/function/closure` notation can theoretically point to closures, **this does not work yet.** (It will, eventually.)\n* Use `/module.submodule/func`, *not* `/module/submodule/func`. The former roughly corresponds to `from module.submodule import func` and the latter to `from module import submodule; func = submodule.func`, which can be different in Python. It\'s a bit odd, but it works that way to properly address Python quirks.\n\n\n## Operators\n\nAll the [operators](https://giving.readthedocs.io/en/latest/ref-operators.html) defined in the `rx` and `giving` packages should be compatible with `global_probe` and `probing`. You can also define [custom operators](https://rxpy.readthedocs.io/en/latest/get_started.html#custom-operator).\n\n[Read this operator guide](https://giving.readthedocs.io/en/latest/guide.html#important-methods) for the most useful features (the `gv` variable in the examples has the same interface as probes).\n\n\n## Query language\n\n**Note:** this section contains some references to a different interface to `ptera` which is still valid, but not documented.\n\nHere is some code annotated with queries that will match various variables. The queries are not exhaustive, just examples.\n\n* The semicolon ";" is used to separate queries and it is not part of any query.\n* The hash character "#" *is* part of the query if there is no space after it, otherwise it starts a comment.\n\n```python\nfrom ptera import ptera, tag\n\nAnimal = tag.Animal\nThing = tag.Thing\n\n@tooled\ndef art(a, b):               # art > a ; art > b ; art(!a, b) ; art(a, !b)\n\n    a1: Animal = bee(a)      # a1 ; art > a1 ; art(!a1) ; art > $x\n                             # a1:Animal ; $x:Animal\n                             # art(!a1) > bee > d  # Focus on a1, also includes d\n                             # art > bee  # This refers to the bee function\n                             # * > a1 ; *(!a1)\n\n    a2: Thing = cap(b)       # a2 ; art > a2 ; art(!a2) ; art > $x\n                             # a2:Thing ; $x:Thing\n\n    return a1 + a2           # art > #value ; art(#value as art_result)\n                             # art() as art_result\n                             # art > $x\n\n\n@tooled\ndef bee(c):\n    c1 = c + 1               # bee > c1 ; art >> c1 ; art(a2) > bee > c1\n                             # bee > c1 as xyz\n\n    return c1                # bee > #value ; bee(c) as bee_value\n\n\n@tooled\ndef cap(d: Thing & int):     # cap > d ; $x:Thing ; cap > $x\n                             # art(bee(c)) > cap > d\n    return d * d\n```\n\n* The `!` operator marks the **focus** of the query. There will be one result for each time the focus is triggered, and when using `tweak` or `rewrite` the focus is what is being tweaked or rewritten.\n  * Other variables are supplemental information, available along with the focus in query results. They can also be used to compute a value for the focus *if* they are available by the time the focus is reached.\n  * The nesting operators `>` and `>>` automatically set the focus to the right hand side if the rhs is a single variable and the operator is not inside `(...)`.\n* The wildcard `*` stands in for any function.\n* The `>>` operator represents **deep nesting**. For example, `art >> c1` encompasses the pattern `art > bee > c1`.\n  * In general, `a >> z` encompasses `a > z`, `a > b > z`, `a > b > c > z`, `a > * > z`, and so on.\n* A function\'s return value corresponds to a special variable named `#value`.\n* `$x` will match any variable name. Getting the variable name for the capture is possible but requires the `map_full` method. For example:\n  * Query: `art > $x`\n  * Getting the names: `results.map_full(lambda x: x.name) == ["a1", "a2", "#value"]`\n  * Other fields accessible from `map_full` are `value`, `names` and `values`, the latter two being needed if multiple results are captured together.\n* Variable annotations are preserved and can be filtered on, using the `:` operator. However, *Ptera only recognizes tags* created using `ptera.Tag("XYZ")` or `ptera.tag.XYZ`. It will not filter over types.\n* `art(bee(c)) > cap > d` triggers on the variable `d` in calls to `cap`, but it will *also* include the value of `c` for all calls to `bee` inside `art`.\n  * If there are multiple calls to `bee`, all values of `c` will be pooled together, and it will be necessary to use `map_all` to retrieve the values (or `map_full`).\n',
    'author': 'Olivier Breuleux',
    'author_email': 'breuleux@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/breuleux/ptera',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
