import { IResource, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { AclCidr, AclTraffic } from './network-acl-types';
import { ISubnet, IVpc, SubnetSelection } from './vpc';
/**
 * A NetworkAcl.
 *
 * @stability stable
 */
export interface INetworkAcl extends IResource {
    /**
     * ID for the current Network ACL.
     *
     * @stability stable
     * @attribute true
     */
    readonly networkAclId: string;
    /**
     * Add a new entry to the ACL.
     *
     * @stability stable
     */
    addEntry(id: string, options: CommonNetworkAclEntryOptions): NetworkAclEntry;
}
/**
 * A NetworkAclBase that is not created in this template
 *
 *
 */
declare abstract class NetworkAclBase extends Resource implements INetworkAcl {
    abstract readonly networkAclId: string;
    /**
     * Add a new entry to the ACL.
     *
     * @stability stable
     */
    addEntry(id: string, options: CommonNetworkAclEntryOptions): NetworkAclEntry;
}
/**
 * Properties to create NetworkAcl.
 *
 * @stability stable
 */
export interface NetworkAclProps {
    /**
     * The name of the NetworkAcl.
     *
     * It is not recommended to use an explicit name.
     *
     * @default If you don't specify a networkAclName, AWS CloudFormation generates a
     * unique physical ID and uses that ID for the group name.
     * @stability stable
     */
    readonly networkAclName?: string;
    /**
     * The VPC in which to create the NetworkACL.
     *
     * @stability stable
     */
    readonly vpc: IVpc;
    /**
     * Subnets in the given VPC to associate the ACL with.
     *
     * More subnets can always be added later by calling
     * `associateWithSubnets()`.
     *
     * @default - No subnets associated
     * @stability stable
     */
    readonly subnetSelection?: SubnetSelection;
}
/**
 * Define a new custom network ACL.
 *
 * By default, will deny all inbound and outbound traffic unless entries are
 * added explicitly allowing it.
 *
 * @stability stable
 */
export declare class NetworkAcl extends NetworkAclBase {
    /**
     * Import an existing NetworkAcl into this app.
     *
     * @stability stable
     */
    static fromNetworkAclId(scope: Construct, id: string, networkAclId: string): INetworkAcl;
    /**
     * The ID of the NetworkACL.
     *
     * @stability stable
     * @attribute true
     */
    readonly networkAclId: string;
    /**
     * The VPC ID for this NetworkACL.
     *
     * @stability stable
     * @attribute true
     */
    readonly networkAclVpcId: string;
    private readonly networkAcl;
    private readonly vpc;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: NetworkAclProps);
    /**
     * Associate the ACL with a given set of subnets.
     *
     * @stability stable
     */
    associateWithSubnet(id: string, selection: SubnetSelection): void;
}
/**
 * What action to apply to traffic matching the ACL.
 *
 * @stability stable
 */
export declare enum Action {
    /**
     * Allow the traffic.
     *
     * @stability stable
     */
    ALLOW = "allow",
    /**
     * Deny the traffic.
     *
     * @stability stable
     */
    DENY = "deny"
}
/**
 * A NetworkAclEntry.
 *
 * @stability stable
 */
export interface INetworkAclEntry extends IResource {
    /**
     * The network ACL.
     *
     * @stability stable
     */
    readonly networkAcl: INetworkAcl;
}
/**
 * Base class for NetworkAclEntries
 *
 *
 */
declare abstract class NetworkAclEntryBase extends Resource implements INetworkAclEntry {
    abstract readonly networkAcl: INetworkAcl;
}
/**
 * Direction of traffic the AclEntry applies to.
 *
 * @stability stable
 */
export declare enum TrafficDirection {
    /**
     * Traffic leaving the subnet.
     *
     * @stability stable
     */
    EGRESS = 0,
    /**
     * Traffic entering the subnet.
     *
     * @stability stable
     */
    INGRESS = 1
}
/**
 * Basic NetworkACL entry props.
 *
 * @stability stable
 */
export interface CommonNetworkAclEntryOptions {
    /**
     * The name of the NetworkAclEntry.
     *
     * It is not recommended to use an explicit group name.
     *
     * @default If you don't specify a NetworkAclName, AWS CloudFormation generates a
     * unique physical ID and uses that ID for the group name.
     * @stability stable
     */
    readonly networkAclEntryName?: string;
    /**
     * The CIDR range to allow or deny.
     *
     * @stability stable
     */
    readonly cidr: AclCidr;
    /**
     * What kind of traffic this ACL rule applies to.
     *
     * @stability stable
     */
    readonly traffic: AclTraffic;
    /**
     * Traffic direction, with respect to the subnet, this rule applies to.
     *
     * @default TrafficDirection.INGRESS
     * @stability stable
     */
    readonly direction?: TrafficDirection;
    /**
     * Whether to allow or deny traffic that matches the rule; valid values are "allow" or "deny".
     *
     * Any traffic that is not explicitly allowed is automatically denied in a custom
     * ACL, all traffic is automatically allowed in a default ACL.
     *
     * @default ALLOW
     * @stability stable
     */
    readonly ruleAction?: Action;
    /**
     * Rule number to assign to the entry, such as 100.
     *
     * ACL entries are processed in ascending order by rule number.
     * Entries can't use the same rule number unless one is an egress rule and the other is an ingress rule.
     *
     * @stability stable
     */
    readonly ruleNumber: number;
}
/**
 * Properties to create NetworkAclEntry.
 *
 * @stability stable
 */
export interface NetworkAclEntryProps extends CommonNetworkAclEntryOptions {
    /**
     * The network ACL this entry applies to.
     *
     * @stability stable
     */
    readonly networkAcl: INetworkAcl;
}
/**
 * Define an entry in a Network ACL table.
 *
 * @stability stable
 */
export declare class NetworkAclEntry extends NetworkAclEntryBase {
    /**
     * The network ACL.
     *
     * @stability stable
     */
    readonly networkAcl: INetworkAcl;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: NetworkAclEntryProps);
}
/**
 * A SubnetNetworkAclAssociation.
 *
 * @stability stable
 */
export interface ISubnetNetworkAclAssociation extends IResource {
    /**
     * ID for the current SubnetNetworkAclAssociation.
     *
     * @stability stable
     * @attribute true
     */
    readonly subnetNetworkAclAssociationAssociationId: string;
}
/**
 * Properties to create a SubnetNetworkAclAssociation.
 *
 * @stability stable
 */
export interface SubnetNetworkAclAssociationProps {
    /**
     * The name of the SubnetNetworkAclAssociation.
     *
     * It is not recommended to use an explicit name.
     *
     * @default If you don't specify a SubnetNetworkAclAssociationName, AWS CloudFormation generates a
     * unique physical ID and uses that ID for the group name.
     * @stability stable
     */
    readonly subnetNetworkAclAssociationName?: string;
    /**
     * The Network ACL this association is defined for.
     *
     * @stability stable
     * @attribute true
     */
    readonly networkAcl: INetworkAcl;
    /**
     * ID of the Subnet.
     *
     * @stability stable
     * @attribute true
     */
    readonly subnet: ISubnet;
}
/**
 * Associate a network ACL with a subnet
 *
 *
 */
declare abstract class SubnetNetworkAclAssociationBase extends Resource implements ISubnetNetworkAclAssociation {
    abstract readonly subnetNetworkAclAssociationAssociationId: string;
}
/**
 * @stability stable
 */
export declare class SubnetNetworkAclAssociation extends SubnetNetworkAclAssociationBase {
    /**
     * @stability stable
     */
    static fromSubnetNetworkAclAssociationAssociationId(scope: Construct, id: string, subnetNetworkAclAssociationAssociationId: string): ISubnetNetworkAclAssociation;
    /**
     * ID for the current SubnetNetworkAclAssociation.
     *
     * @stability stable
     * @attribute true
     */
    readonly subnetNetworkAclAssociationAssociationId: string;
    /**
     * ID for the current Network ACL.
     *
     * @stability stable
     * @attribute true
     */
    readonly networkAcl: INetworkAcl;
    /**
     * ID of the Subnet.
     *
     * @stability stable
     * @attribute true
     */
    readonly subnet: ISubnet;
    private association;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: SubnetNetworkAclAssociationProps);
}
export {};
