"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClientVpnEndpoint = exports.ClientVpnUserBasedAuthentication = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const logs = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const client_vpn_authorization_rule_1 = require("./client-vpn-authorization-rule");
const client_vpn_route_1 = require("./client-vpn-route");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const network_util_1 = require("./network-util");
const security_group_1 = require("./security-group");
/**
 * User-based authentication for a client VPN endpoint.
 *
 * @stability stable
 */
class ClientVpnUserBasedAuthentication {
    /**
     * Active Directory authentication.
     *
     * @stability stable
     */
    static activeDirectory(directoryId) {
        return new ActiveDirectoryAuthentication(directoryId);
    }
    /**
     * Federated authentication.
     *
     * @stability stable
     */
    static federated(samlProvider, selfServiceSamlProvider) {
        return new FederatedAuthentication(samlProvider, selfServiceSamlProvider);
    }
}
exports.ClientVpnUserBasedAuthentication = ClientVpnUserBasedAuthentication;
_a = JSII_RTTI_SYMBOL_1;
ClientVpnUserBasedAuthentication[_a] = { fqn: "@aws-cdk/aws-ec2.ClientVpnUserBasedAuthentication", version: "1.138.2" };
/**
 * Active Directory authentication
 */
class ActiveDirectoryAuthentication extends ClientVpnUserBasedAuthentication {
    constructor(directoryId) {
        super();
        this.directoryId = directoryId;
    }
    render() {
        return {
            type: 'directory-service-authentication',
            activeDirectory: { directoryId: this.directoryId },
        };
    }
}
/**
 * Federated authentication
 */
class FederatedAuthentication extends ClientVpnUserBasedAuthentication {
    constructor(samlProvider, selfServiceSamlProvider) {
        super();
        this.samlProvider = samlProvider;
        this.selfServiceSamlProvider = selfServiceSamlProvider;
    }
    render() {
        var _c;
        return {
            type: 'federated-authentication',
            federatedAuthentication: {
                samlProviderArn: this.samlProvider.samlProviderArn,
                selfServiceSamlProviderArn: (_c = this.selfServiceSamlProvider) === null || _c === void 0 ? void 0 : _c.samlProviderArn,
            },
        };
    }
}
/**
 * A client VPN connnection.
 *
 * @stability stable
 */
class ClientVpnEndpoint extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h;
        super(scope, id);
        this._targetNetworksAssociated = new core_1.ConcreteDependable();
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_ClientVpnEndpointProps(props);
        if (!core_1.Token.isUnresolved(props.vpc.vpcCidrBlock)) {
            const clientCidr = new network_util_1.CidrBlock(props.cidr);
            const vpcCidr = new network_util_1.CidrBlock(props.vpc.vpcCidrBlock);
            if (vpcCidr.containsCidr(clientCidr)) {
                throw new Error('The client CIDR cannot overlap with the local CIDR of the VPC');
            }
        }
        if (props.dnsServers && props.dnsServers.length > 2) {
            throw new Error('A client VPN endpoint can have up to two DNS servers');
        }
        if (props.logging == false && (props.logGroup || props.logStream)) {
            throw new Error('Cannot specify `logGroup` or `logStream` when logging is disabled');
        }
        if (props.clientConnectionHandler
            && !core_1.Token.isUnresolved(props.clientConnectionHandler.functionName)
            && !props.clientConnectionHandler.functionName.startsWith('AWSClientVPN-')) {
            throw new Error('The name of the Lambda function must begin with the `AWSClientVPN-` prefix');
        }
        const logging = (_c = props.logging) !== null && _c !== void 0 ? _c : true;
        const logGroup = logging
            ? (_d = props.logGroup) !== null && _d !== void 0 ? _d : new logs.LogGroup(this, 'LogGroup') : undefined;
        const securityGroups = (_e = props.securityGroups) !== null && _e !== void 0 ? _e : [new security_group_1.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
            })];
        this.connections = new connections_1.Connections({ securityGroups });
        const endpoint = new ec2_generated_1.CfnClientVpnEndpoint(this, 'Resource', {
            authenticationOptions: renderAuthenticationOptions(props.clientCertificateArn, props.userBasedAuthentication),
            clientCidrBlock: props.cidr,
            clientConnectOptions: props.clientConnectionHandler
                ? {
                    enabled: true,
                    lambdaFunctionArn: props.clientConnectionHandler.functionArn,
                }
                : undefined,
            connectionLogOptions: {
                enabled: logging,
                cloudwatchLogGroup: logGroup === null || logGroup === void 0 ? void 0 : logGroup.logGroupName,
                cloudwatchLogStream: (_f = props.logStream) === null || _f === void 0 ? void 0 : _f.logStreamName,
            },
            description: props.description,
            dnsServers: props.dnsServers,
            securityGroupIds: securityGroups.map(s => s.securityGroupId),
            selfServicePortal: booleanToEnabledDisabled(props.selfServicePortal),
            serverCertificateArn: props.serverCertificateArn,
            splitTunnel: props.splitTunnel,
            transportProtocol: props.transportProtocol,
            vpcId: props.vpc.vpcId,
            vpnPort: props.port,
        });
        this.endpointId = endpoint.ref;
        if (props.userBasedAuthentication && ((_g = props.selfServicePortal) !== null && _g !== void 0 ? _g : true)) {
            // Output self-service portal URL
            new core_1.CfnOutput(this, 'SelfServicePortalUrl', {
                value: `https://self-service.clientvpn.amazonaws.com/endpoints/${this.endpointId}`,
            });
        }
        // Associate subnets
        const subnetIds = props.vpc.selectSubnets(props.vpcSubnets).subnetIds;
        if (core_1.Token.isUnresolved(subnetIds)) {
            throw new Error('Cannot associate subnets when VPC are imported from parameters or exports containing lists of subnet IDs.');
        }
        for (const [idx, subnetId] of Object.entries(subnetIds)) {
            this._targetNetworksAssociated.add(new ec2_generated_1.CfnClientVpnTargetNetworkAssociation(this, `Association${idx}`, {
                clientVpnEndpointId: this.endpointId,
                subnetId,
            }));
        }
        this.targetNetworksAssociated = this._targetNetworksAssociated;
        if ((_h = props.authorizeAllUsersToVpcCidr) !== null && _h !== void 0 ? _h : true) {
            this.addAuthorizationRule('AuthorizeAll', {
                cidr: props.vpc.vpcCidrBlock,
            });
        }
    }
    /**
     * Import an existing client VPN endpoint.
     *
     * @stability stable
     */
    static fromEndpointAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_ClientVpnEndpointAttributes(attrs);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.endpointId = attrs.endpointId;
                this.connections = new connections_1.Connections({ securityGroups: attrs.securityGroups });
                this.targetNetworksAssociated = new core_1.ConcreteDependable();
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds an authorization rule to this endpoint.
     *
     * @stability stable
     */
    addAuthorizationRule(id, props) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_ClientVpnAuthorizationRuleOptions(props);
        return new client_vpn_authorization_rule_1.ClientVpnAuthorizationRule(this, id, {
            ...props,
            clientVpnEndpoint: this,
        });
    }
    /**
     * Adds a route to this endpoint.
     *
     * @stability stable
     */
    addRoute(id, props) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_ClientVpnRouteOptions(props);
        return new client_vpn_route_1.ClientVpnRoute(this, id, {
            ...props,
            clientVpnEndpoint: this,
        });
    }
}
exports.ClientVpnEndpoint = ClientVpnEndpoint;
_b = JSII_RTTI_SYMBOL_1;
ClientVpnEndpoint[_b] = { fqn: "@aws-cdk/aws-ec2.ClientVpnEndpoint", version: "1.138.2" };
function renderAuthenticationOptions(clientCertificateArn, userBasedAuthentication) {
    const authenticationOptions = [];
    if (clientCertificateArn) {
        authenticationOptions.push({
            type: 'certificate-authentication',
            mutualAuthentication: {
                clientRootCertificateChainArn: clientCertificateArn,
            },
        });
    }
    if (userBasedAuthentication) {
        authenticationOptions.push(userBasedAuthentication.render());
    }
    if (authenticationOptions.length === 0) {
        throw new Error('A client VPN endpoint must use at least one authentication option');
    }
    return authenticationOptions;
}
function booleanToEnabledDisabled(val) {
    switch (val) {
        case undefined:
            return undefined;
        case true:
            return 'enabled';
        case false:
            return 'disabled';
    }
}
//# sourceMappingURL=data:application/json;base64,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