'''
# Amazon EC2 Construct Library

<!--BEGIN STABILITY BANNER-->---


![cfn-resources: Stable](https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge)

![cdk-constructs: Stable](https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge)

---
<!--END STABILITY BANNER-->

The `@aws-cdk/aws-ec2` package contains primitives for setting up networking and
instances.

```python
import aws_cdk.aws_ec2 as ec2
```

## VPC

Most projects need a Virtual Private Cloud to provide security by means of
network partitioning. This is achieved by creating an instance of
`Vpc`:

```python
vpc = ec2.Vpc(self, "VPC")
```

All default constructs require EC2 instances to be launched inside a VPC, so
you should generally start by defining a VPC whenever you need to launch
instances for your project.

### Subnet Types

A VPC consists of one or more subnets that instances can be placed into. CDK
distinguishes three different subnet types:

* **Public (`SubnetType.PUBLIC`)** - public subnets connect directly to the Internet using an
  Internet Gateway. If you want your instances to have a public IP address
  and be directly reachable from the Internet, you must place them in a
  public subnet.
* **Private with Internet Access (`SubnetType.PRIVATE_WITH_NAT`)** - instances in private subnets are not directly routable from the
  Internet, and connect out to the Internet via a NAT gateway. By default, a
  NAT gateway is created in every public subnet for maximum availability. Be
  aware that you will be charged for NAT gateways.
* **Isolated (`SubnetType.PRIVATE_ISOLATED`)** - isolated subnets do not route from or to the Internet, and
  as such do not require NAT gateways. They can only connect to or be
  connected to from other instances in the same VPC. A default VPC configuration
  will not include isolated subnets,

A default VPC configuration will create public and **private** subnets. However, if
`natGateways:0` **and** `subnetConfiguration` is undefined, default VPC configuration
will create public and **isolated** subnets. See [*Advanced Subnet Configuration*](#advanced-subnet-configuration)
below for information on how to change the default subnet configuration.

Constructs using the VPC will "launch instances" (or more accurately, create
Elastic Network Interfaces) into one or more of the subnets. They all accept
a property called `subnetSelection` (sometimes called `vpcSubnets`) to allow
you to select in what subnet to place the ENIs, usually defaulting to
*private* subnets if the property is omitted.

If you would like to save on the cost of NAT gateways, you can use
*isolated* subnets instead of *private* subnets (as described in Advanced
*Subnet Configuration*). If you need private instances to have
internet connectivity, another option is to reduce the number of NAT gateways
created by setting the `natGateways` property to a lower value (the default
is one NAT gateway per availability zone). Be aware that this may have
availability implications for your application.

[Read more about
subnets](https://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/VPC_Subnets.html).

### Control over availability zones

By default, a VPC will spread over at most 3 Availability Zones available to
it. To change the number of Availability Zones that the VPC will spread over,
specify the `maxAzs` property when defining it.

The number of Availability Zones that are available depends on the *region*
and *account* of the Stack containing the VPC. If the [region and account are
specified](https://docs.aws.amazon.com/cdk/latest/guide/environments.html) on
the Stack, the CLI will [look up the existing Availability
Zones](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-regions-availability-zones.html#using-regions-availability-zones-describe)
and get an accurate count. If region and account are not specified, the stack
could be deployed anywhere and it will have to make a safe choice, limiting
itself to 2 Availability Zones.

Therefore, to get the VPC to spread over 3 or more availability zones, you
must specify the environment where the stack will be deployed.

You can gain full control over the availability zones selection strategy by overriding the Stack's [`get availabilityZones()`](https://github.com/aws/aws-cdk/blob/master/packages/@aws-cdk/core/lib/stack.ts) method:

```text
// This example is only available in TypeScript

class MyStack extends Stack {

  constructor(scope: Construct, id: string, props?: StackProps) {
    super(scope, id, props);

    // ...
  }

  get availabilityZones(): string[] {
    return ['us-west-2a', 'us-west-2b'];
  }

}
```

Note that overriding the `get availabilityZones()` method will override the default behavior for all constructs defined within the Stack.

### Choosing subnets for resources

When creating resources that create Elastic Network Interfaces (such as
databases or instances), there is an option to choose which subnets to place
them in. For example, a VPC endpoint by default is placed into a subnet in
every availability zone, but you can override which subnets to use. The property
is typically called one of `subnets`, `vpcSubnets` or `subnetSelection`.

The example below will place the endpoint into two AZs (`us-east-1a` and `us-east-1c`),
in Isolated subnets:

```python
# vpc is of type Vpc


ec2.InterfaceVpcEndpoint(self, "VPC Endpoint",
    vpc=vpc,
    service=ec2.InterfaceVpcEndpointService("com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc", 443),
    subnets=ec2.SubnetSelection(
        subnet_type=ec2.SubnetType.ISOLATED,
        availability_zones=["us-east-1a", "us-east-1c"]
    )
)
```

You can also specify specific subnet objects for granular control:

```python
# vpc is of type Vpc
# subnet1 is of type Subnet
# subnet2 is of type Subnet


ec2.InterfaceVpcEndpoint(self, "VPC Endpoint",
    vpc=vpc,
    service=ec2.InterfaceVpcEndpointService("com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc", 443),
    subnets=ec2.SubnetSelection(
        subnets=[subnet1, subnet2]
    )
)
```

Which subnets are selected is evaluated as follows:

* `subnets`: if specific subnet objects are supplied, these are selected, and no other
  logic is used.
* `subnetType`/`subnetGroupName`: otherwise, a set of subnets is selected by
  supplying either type or name:

  * `subnetType` will select all subnets of the given type.
  * `subnetGroupName` should be used to distinguish between multiple groups of subnets of
    the same type (for example, you may want to separate your application instances and your
    RDS instances into two distinct groups of Isolated subnets).
  * If neither are given, the first available subnet group of a given type that
    exists in the VPC will be used, in this order: Private, then Isolated, then Public.
    In short: by default ENIs will preferentially be placed in subnets not connected to
    the Internet.
* `availabilityZones`/`onePerAz`: finally, some availability-zone based filtering may be done.
  This filtering by availability zones will only be possible if the VPC has been created or
  looked up in a non-environment agnostic stack (so account and region have been set and
  availability zones have been looked up).

  * `availabilityZones`: only the specific subnets from the selected subnet groups that are
    in the given availability zones will be returned.
  * `onePerAz`: per availability zone, a maximum of one subnet will be returned (Useful for resource
    types that do not allow creating two ENIs in the same availability zone).
* `subnetFilters`: additional filtering on subnets using any number of user-provided filters which
  extend `SubnetFilter`.  The following methods on the `SubnetFilter` class can be used to create
  a filter:

  * `byIds`: chooses subnets from a list of ids
  * `availabilityZones`: chooses subnets in the provided list of availability zones
  * `onePerAz`: chooses at most one subnet per availability zone
  * `containsIpAddresses`: chooses a subnet which contains *any* of the listed ip addresses
  * `byCidrMask`: chooses subnets that have the provided CIDR netmask

### Using NAT instances

By default, the `Vpc` construct will create NAT *gateways* for you, which
are managed by AWS. If you would prefer to use your own managed NAT
*instances* instead, specify a different value for the `natGatewayProvider`
property, as follows:

```python
# Configure the `natGatewayProvider` when defining a Vpc
nat_gateway_provider = ec2.NatProvider.instance(
    instance_type=ec2.InstanceType("t3.small")
)

vpc = ec2.Vpc(self, "MyVpc",
    nat_gateway_provider=nat_gateway_provider,

    # The 'natGateways' parameter now controls the number of NAT instances
    nat_gateways=2
)
```

The construct will automatically search for the most recent NAT gateway AMI.
If you prefer to use a custom AMI, use `machineImage: MachineImage.genericLinux({ ... })` and configure the right AMI ID for the
regions you want to deploy to.

By default, the NAT instances will route all traffic. To control what traffic
gets routed, pass `allowAllTraffic: false` and access the
`NatInstanceProvider.connections` member after having passed it to the VPC:

```python
# instance_type is of type InstanceType


provider = ec2.NatProvider.instance(
    instance_type=instance_type,
    allow_all_traffic=False
)
ec2.Vpc(self, "TheVPC",
    nat_gateway_provider=provider
)
provider.connections.allow_from(ec2.Peer.ipv4("1.2.3.4/8"), ec2.Port.tcp(80))
```

### Advanced Subnet Configuration

If the default VPC configuration (public and private subnets spanning the
size of the VPC) don't suffice for you, you can configure what subnets to
create by specifying the `subnetConfiguration` property. It allows you
to configure the number and size of all subnets. Specifying an advanced
subnet configuration could look like this:

```python
vpc = ec2.Vpc(self, "TheVPC",
    # 'cidr' configures the IP range and size of the entire VPC.
    # The IP space will be divided over the configured subnets.
    cidr="10.0.0.0/21",

    # 'maxAzs' configures the maximum number of availability zones to use
    max_azs=3,

    # 'subnetConfiguration' specifies the "subnet groups" to create.
    # Every subnet group will have a subnet for each AZ, so this
    # configuration will create `3 groups × 3 AZs = 9` subnets.
    subnet_configuration=[ec2.SubnetConfiguration(
        # 'subnetType' controls Internet access, as described above.
        subnet_type=ec2.SubnetType.PUBLIC,

        # 'name' is used to name this particular subnet group. You will have to
        # use the name for subnet selection if you have more than one subnet
        # group of the same type.
        name="Ingress",

        # 'cidrMask' specifies the IP addresses in the range of of individual
        # subnets in the group. Each of the subnets in this group will contain
        # `2^(32 address bits - 24 subnet bits) - 2 reserved addresses = 254`
        # usable IP addresses.
        #
        # If 'cidrMask' is left out the available address space is evenly
        # divided across the remaining subnet groups.
        cidr_mask=24
    ), ec2.SubnetConfiguration(
        cidr_mask=24,
        name="Application",
        subnet_type=ec2.SubnetType.PRIVATE_WITH_NAT
    ), ec2.SubnetConfiguration(
        cidr_mask=28,
        name="Database",
        subnet_type=ec2.SubnetType.PRIVATE_ISOLATED,

        # 'reserved' can be used to reserve IP address space. No resources will
        # be created for this subnet, but the IP range will be kept available for
        # future creation of this subnet, or even for future subdivision.
        reserved=True
    )
    ]
)
```

The example above is one possible configuration, but the user can use the
constructs above to implement many other network configurations.

The `Vpc` from the above configuration in a Region with three
availability zones will be the following:

Subnet Name       |Type      |IP Block      |AZ|Features
------------------|----------|--------------|--|--------
IngressSubnet1    |`PUBLIC`  |`10.0.0.0/24` |#1|NAT Gateway
IngressSubnet2    |`PUBLIC`  |`10.0.1.0/24` |#2|NAT Gateway
IngressSubnet3    |`PUBLIC`  |`10.0.2.0/24` |#3|NAT Gateway
ApplicationSubnet1|`PRIVATE` |`10.0.3.0/24` |#1|Route to NAT in IngressSubnet1
ApplicationSubnet2|`PRIVATE` |`10.0.4.0/24` |#2|Route to NAT in IngressSubnet2
ApplicationSubnet3|`PRIVATE` |`10.0.5.0/24` |#3|Route to NAT in IngressSubnet3
DatabaseSubnet1   |`ISOLATED`|`10.0.6.0/28` |#1|Only routes within the VPC
DatabaseSubnet2   |`ISOLATED`|`10.0.6.16/28`|#2|Only routes within the VPC
DatabaseSubnet3   |`ISOLATED`|`10.0.6.32/28`|#3|Only routes within the VPC

### Accessing the Internet Gateway

If you need access to the internet gateway, you can get its ID like so:

```python
# vpc is of type Vpc


igw_id = vpc.internet_gateway_id
```

For a VPC with only `ISOLATED` subnets, this value will be undefined.

This is only supported for VPCs created in the stack - currently you're
unable to get the ID for imported VPCs. To do that you'd have to specifically
look up the Internet Gateway by name, which would require knowing the name
beforehand.

This can be useful for configuring routing using a combination of gateways:
for more information see [Routing](#routing) below.

#### Routing

It's possible to add routes to any subnets using the `addRoute()` method. If for
example you want an isolated subnet to have a static route via the default
Internet Gateway created for the public subnet - perhaps for routing a VPN
connection - you can do so like this:

```python
vpc = ec2.Vpc(self, "VPC",
    subnet_configuration=[ec2.SubnetConfiguration(
        subnet_type=ec2.SubnetType.PUBLIC,
        name="Public"
    ), ec2.SubnetConfiguration(
        subnet_type=ec2.SubnetType.ISOLATED,
        name="Isolated"
    )]
)

(vpc.isolated_subnets[0]).add_route("StaticRoute",
    router_id=vpc.internet_gateway_id,
    router_type=ec2.RouterType.GATEWAY,
    destination_cidr_block="8.8.8.8/32"
)
```

*Note that we cast to `Subnet` here because the list of subnets only returns an
`ISubnet`.*

### Reserving subnet IP space

There are situations where the IP space for a subnet or number of subnets
will need to be reserved. This is useful in situations where subnets would
need to be added after the vpc is originally deployed, without causing IP
renumbering for existing subnets. The IP space for a subnet may be reserved
by setting the `reserved` subnetConfiguration property to true, as shown
below:

```python
vpc = ec2.Vpc(self, "TheVPC",
    nat_gateways=1,
    subnet_configuration=[ec2.SubnetConfiguration(
        cidr_mask=26,
        name="Public",
        subnet_type=ec2.SubnetType.PUBLIC
    ), ec2.SubnetConfiguration(
        cidr_mask=26,
        name="Application1",
        subnet_type=ec2.SubnetType.PRIVATE_WITH_NAT
    ), ec2.SubnetConfiguration(
        cidr_mask=26,
        name="Application2",
        subnet_type=ec2.SubnetType.PRIVATE_WITH_NAT,
        reserved=True
    ), ec2.SubnetConfiguration(
        cidr_mask=27,
        name="Database",
        subnet_type=ec2.SubnetType.ISOLATED
    )
    ]
)
```

In the example above, the subnet for Application2 is not actually provisioned
but its IP space is still reserved. If in the future this subnet needs to be
provisioned, then the `reserved: true` property should be removed. Reserving
parts of the IP space prevents the other subnets from getting renumbered.

### Sharing VPCs between stacks

If you are creating multiple `Stack`s inside the same CDK application, you
can reuse a VPC defined in one Stack in another by simply passing the VPC
instance around:

```python
#
# Stack1 creates the VPC
#
class Stack1(cdk.Stack):

    def __init__(self, scope, id, *, description=None, env=None, stackName=None, tags=None, synthesizer=None, terminationProtection=None, analyticsReporting=None):
        super().__init__(scope, id, description=description, env=env, stackName=stackName, tags=tags, synthesizer=synthesizer, terminationProtection=terminationProtection, analyticsReporting=analyticsReporting)

        self.vpc = ec2.Vpc(self, "VPC")

#
# Stack2 consumes the VPC
#
class Stack2(cdk.Stack):
    def __init__(self, scope, id, *, vpc, description=None, env=None, stackName=None, tags=None, synthesizer=None, terminationProtection=None, analyticsReporting=None):
        super().__init__(scope, id, vpc=vpc, description=description, env=env, stackName=stackName, tags=tags, synthesizer=synthesizer, terminationProtection=terminationProtection, analyticsReporting=analyticsReporting)

        # Pass the VPC to a construct that needs it
        ConstructThatTakesAVpc(self, "Construct",
            vpc=vpc
        )

stack1 = Stack1(app, "Stack1")
stack2 = Stack2(app, "Stack2",
    vpc=stack1.vpc
)
```

### Importing an existing VPC

If your VPC is created outside your CDK app, you can use `Vpc.fromLookup()`.
The CDK CLI will search for the specified VPC in the the stack's region and
account, and import the subnet configuration. Looking up can be done by VPC
ID, but more flexibly by searching for a specific tag on the VPC.

Subnet types will be determined from the `aws-cdk:subnet-type` tag on the
subnet if it exists, or the presence of a route to an Internet Gateway
otherwise. Subnet names will be determined from the `aws-cdk:subnet-name` tag
on the subnet if it exists, or will mirror the subnet type otherwise (i.e.
a public subnet will have the name `"Public"`).

The result of the `Vpc.fromLookup()` operation will be written to a file
called `cdk.context.json`. You must commit this file to source control so
that the lookup values are available in non-privileged environments such
as CI build steps, and to ensure your template builds are repeatable.

Here's how `Vpc.fromLookup()` can be used:

```python
vpc = ec2.Vpc.from_lookup(stack, "VPC",
    # This imports the default VPC but you can also
    # specify a 'vpcName' or 'tags'.
    is_default=True
)
```

`Vpc.fromLookup` is the recommended way to import VPCs. If for whatever
reason you do not want to use the context mechanism to look up a VPC at
synthesis time, you can also use `Vpc.fromVpcAttributes`. This has the
following limitations:

* Every subnet group in the VPC must have a subnet in each availability zone
  (for example, each AZ must have both a public and private subnet). Asymmetric
  VPCs are not supported.
* All VpcId, SubnetId, RouteTableId, ... parameters must either be known at
  synthesis time, or they must come from deploy-time list parameters whose
  deploy-time lengths are known at synthesis time.

Using `Vpc.fromVpcAttributes()` looks like this:

```python
vpc = ec2.Vpc.from_vpc_attributes(self, "VPC",
    vpc_id="vpc-1234",
    availability_zones=["us-east-1a", "us-east-1b"],

    # Either pass literals for all IDs
    public_subnet_ids=["s-12345", "s-67890"],

    # OR: import a list of known length
    private_subnet_ids=Fn.import_list_value("PrivateSubnetIds", 2),

    # OR: split an imported string to a list of known length
    isolated_subnet_ids=Fn.split(",", ssm.StringParameter.value_for_string_parameter(self, "MyParameter"), 2)
)
```

## Allowing Connections

In AWS, all network traffic in and out of **Elastic Network Interfaces** (ENIs)
is controlled by **Security Groups**. You can think of Security Groups as a
firewall with a set of rules. By default, Security Groups allow no incoming
(ingress) traffic and all outgoing (egress) traffic. You can add ingress rules
to them to allow incoming traffic streams. To exert fine-grained control over
egress traffic, set `allowAllOutbound: false` on the `SecurityGroup`, after
which you can add egress traffic rules.

You can manipulate Security Groups directly:

```python
my_security_group = ec2.SecurityGroup(self, "SecurityGroup",
    vpc=vpc,
    description="Allow ssh access to ec2 instances",
    allow_all_outbound=True
)
my_security_group.add_ingress_rule(ec2.Peer.any_ipv4(), ec2.Port.tcp(22), "allow ssh access from the world")
```

All constructs that create ENIs on your behalf (typically constructs that create
EC2 instances or other VPC-connected resources) will all have security groups
automatically assigned. Those constructs have an attribute called
**connections**, which is an object that makes it convenient to update the
security groups. If you want to allow connections between two constructs that
have security groups, you have to add an **Egress** rule to one Security Group,
and an **Ingress** rule to the other. The connections object will automatically
take care of this for you:

```python
# load_balancer is of type ApplicationLoadBalancer
# app_fleet is of type AutoScalingGroup
# db_fleet is of type AutoScalingGroup


# Allow connections from anywhere
load_balancer.connections.allow_from_any_ipv4(ec2.Port.tcp(443), "Allow inbound HTTPS")

# The same, but an explicit IP address
load_balancer.connections.allow_from(ec2.Peer.ipv4("1.2.3.4/32"), ec2.Port.tcp(443), "Allow inbound HTTPS")

# Allow connection between AutoScalingGroups
app_fleet.connections.allow_to(db_fleet, ec2.Port.tcp(443), "App can call database")
```

### Connection Peers

There are various classes that implement the connection peer part:

```python
# app_fleet is of type AutoScalingGroup
# db_fleet is of type AutoScalingGroup


# Simple connection peers
peer = ec2.Peer.ipv4("10.0.0.0/16")
peer = ec2.Peer.any_ipv4()
peer = ec2.Peer.ipv6("::0/0")
peer = ec2.Peer.any_ipv6()
peer = ec2.Peer.prefix_list("pl-12345")
app_fleet.connections.allow_to(peer, ec2.Port.tcp(443), "Allow outbound HTTPS")
```

Any object that has a security group can itself be used as a connection peer:

```python
# fleet1 is of type AutoScalingGroup
# fleet2 is of type AutoScalingGroup
# app_fleet is of type AutoScalingGroup


# These automatically create appropriate ingress and egress rules in both security groups
fleet1.connections.allow_to(fleet2, ec2.Port.tcp(80), "Allow between fleets")

app_fleet.connections.allow_from_any_ipv4(ec2.Port.tcp(80), "Allow from load balancer")
```

### Port Ranges

The connections that are allowed are specified by port ranges. A number of classes provide
the connection specifier:

```python
ec2.Port.tcp(80)
ec2.Port.tcp_range(60000, 65535)
ec2.Port.all_tcp()
ec2.Port.all_traffic()
```

> NOTE: This set is not complete yet; for example, there is no library support for ICMP at the moment.
> However, you can write your own classes to implement those.

### Default Ports

Some Constructs have default ports associated with them. For example, the
listener of a load balancer does (it's the public port), or instances of an
RDS database (it's the port the database is accepting connections on).

If the object you're calling the peering method on has a default port associated with it, you can call
`allowDefaultPortFrom()` and omit the port specifier. If the argument has an associated default port, call
`allowDefaultPortTo()`.

For example:

```python
# listener is of type ApplicationListener
# app_fleet is of type AutoScalingGroup
# rds_database is of type DatabaseCluster


# Port implicit in listener
listener.connections.allow_default_port_from_any_ipv4("Allow public")

# Port implicit in peer
app_fleet.connections.allow_default_port_to(rds_database, "Fleet can access database")
```

### Security group rules

By default, security group wills be added inline to the security group in the output cloud formation
template, if applicable.  This includes any static rules by ip address and port range.  This
optimization helps to minimize the size of the template.

In some environments this is not desirable, for example if your security group access is controlled
via tags. You can disable inline rules per security group or globally via the context key
`@aws-cdk/aws-ec2.securityGroupDisableInlineRules`.

```python
my_security_group_without_inline_rules = ec2.SecurityGroup(self, "SecurityGroup",
    vpc=vpc,
    description="Allow ssh access to ec2 instances",
    allow_all_outbound=True,
    disable_inline_rules=True
)
# This will add the rule as an external cloud formation construct
my_security_group_without_inline_rules.add_ingress_rule(ec2.Peer.any_ipv4(), ec2.Port.tcp(22), "allow ssh access from the world")
```

### Importing an existing security group

If you know the ID and the configuration of the security group to import, you can use `SecurityGroup.fromSecurityGroupId`:

```python
sg = ec2.SecurityGroup.from_security_group_id(self, "SecurityGroupImport", "sg-1234",
    allow_all_outbound=True
)
```

Alternatively, use lookup methods to import security groups if you do not know the ID or the configuration details. Method `SecurityGroup.fromLookupByName` looks up a security group if the secruity group ID is unknown.

```python
sg = ec2.SecurityGroup.from_lookup_by_name(self, "SecurityGroupLookup", "security-group-name", vpc)
```

If the security group ID is known and configuration details are unknown, use method `SecurityGroup.fromLookupById` instead. This method will lookup property `allowAllOutbound` from the current configuration of the security group.

```python
sg = ec2.SecurityGroup.from_lookup_by_id(self, "SecurityGroupLookup", "sg-1234")
```

The result of `SecurityGroup.fromLookupByName` and `SecurityGroup.fromLookupById` operations will be written to a file called `cdk.context.json`. You must commit this file to source control so that the lookup values are available in non-privileged environments such as CI build steps, and to ensure your template builds are repeatable.

## Machine Images (AMIs)

AMIs control the OS that gets launched when you start your EC2 instance. The EC2
library contains constructs to select the AMI you want to use.

Depending on the type of AMI, you select it a different way. Here are some
examples of things you might want to use:

```python
# Pick the right Amazon Linux edition. All arguments shown are optional
# and will default to these values when omitted.
amzn_linux = ec2.MachineImage.latest_amazon_linux(
    generation=ec2.AmazonLinuxGeneration.AMAZON_LINUX,
    edition=ec2.AmazonLinuxEdition.STANDARD,
    virtualization=ec2.AmazonLinuxVirt.HVM,
    storage=ec2.AmazonLinuxStorage.GENERAL_PURPOSE,
    cpu_type=ec2.AmazonLinuxCpuType.X86_64
)

# Pick a Windows edition to use
windows = ec2.MachineImage.latest_windows(ec2.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE)

# Read AMI id from SSM parameter store
ssm = ec2.MachineImage.from_sSMParameter("/my/ami", ec2.OperatingSystemType.LINUX)

# Look up the most recent image matching a set of AMI filters.
# In this case, look up the NAT instance AMI, by using a wildcard
# in the 'name' field:
nat_ami = ec2.MachineImage.lookup(
    name="amzn-ami-vpc-nat-*",
    owners=["amazon"]
)

# For other custom (Linux) images, instantiate a `GenericLinuxImage` with
# a map giving the AMI to in for each region:
linux = ec2.MachineImage.generic_linux({
    "us-east-1": "ami-97785bed",
    "eu-west-1": "ami-12345678"
})

# For other custom (Windows) images, instantiate a `GenericWindowsImage` with
# a map giving the AMI to in for each region:
generic_windows = ec2.MachineImage.generic_windows({
    "us-east-1": "ami-97785bed",
    "eu-west-1": "ami-12345678"
})
```

> NOTE: The AMIs selected by `MachineImage.lookup()` will be cached in
> `cdk.context.json`, so that your AutoScalingGroup instances aren't replaced while
> you are making unrelated changes to your CDK app.
>
> To query for the latest AMI again, remove the relevant cache entry from
> `cdk.context.json`, or use the `cdk context` command. For more information, see
> [Runtime Context](https://docs.aws.amazon.com/cdk/latest/guide/context.html) in the CDK
> developer guide.
>
> `MachineImage.genericLinux()`, `MachineImage.genericWindows()` will use `CfnMapping` in
> an agnostic stack.

## Special VPC configurations

### VPN connections to a VPC

Create your VPC with VPN connections by specifying the `vpnConnections` props (keys are construct `id`s):

```python
vpc = ec2.Vpc(self, "MyVpc",
    vpn_connections={
        "dynamic": ec2.VpnConnectionOptions( # Dynamic routing (BGP)
            ip="1.2.3.4"),
        "static": ec2.VpnConnectionOptions( # Static routing
            ip="4.5.6.7",
            static_routes=["192.168.10.0/24", "192.168.20.0/24"
            ])
    }
)
```

To create a VPC that can accept VPN connections, set `vpnGateway` to `true`:

```python
vpc = ec2.Vpc(self, "MyVpc",
    vpn_gateway=True
)
```

VPN connections can then be added:

```python
vpc.add_vpn_connection("Dynamic",
    ip="1.2.3.4"
)
```

By default, routes will be propagated on the route tables associated with the private subnets. If no
private subnets exists, isolated subnets are used. If no isolated subnets exists, public subnets are
used. Use the `Vpc` property `vpnRoutePropagation` to customize this behavior.

VPN connections expose [metrics (cloudwatch.Metric)](https://github.com/aws/aws-cdk/blob/master/packages/%40aws-cdk/aws-cloudwatch/README.md) across all tunnels in the account/region and per connection:

```python
# Across all tunnels in the account/region
all_data_out = ec2.VpnConnection.metric_all_tunnel_data_out()

# For a specific vpn connection
vpn_connection = vpc.add_vpn_connection("Dynamic",
    ip="1.2.3.4"
)
state = vpn_connection.metric_tunnel_state()
```

### VPC endpoints

A VPC endpoint enables you to privately connect your VPC to supported AWS services and VPC endpoint services powered by PrivateLink without requiring an internet gateway, NAT device, VPN connection, or AWS Direct Connect connection. Instances in your VPC do not require public IP addresses to communicate with resources in the service. Traffic between your VPC and the other service does not leave the Amazon network.

Endpoints are virtual devices. They are horizontally scaled, redundant, and highly available VPC components that allow communication between instances in your VPC and services without imposing availability risks or bandwidth constraints on your network traffic.

```python
# Add gateway endpoints when creating the VPC
vpc = ec2.Vpc(self, "MyVpc",
    gateway_endpoints={
        "S3": ec2.GatewayVpcEndpointOptions(
            service=ec2.GatewayVpcEndpointAwsService.S3
        )
    }
)

# Alternatively gateway endpoints can be added on the VPC
dynamo_db_endpoint = vpc.add_gateway_endpoint("DynamoDbEndpoint",
    service=ec2.GatewayVpcEndpointAwsService.DYNAMODB
)

# This allows to customize the endpoint policy
dynamo_db_endpoint.add_to_policy(
    iam.PolicyStatement( # Restrict to listing and describing tables
        principals=[iam.AnyPrincipal()],
        actions=["dynamodb:DescribeTable", "dynamodb:ListTables"],
        resources=["*"]))

# Add an interface endpoint
vpc.add_interface_endpoint("EcrDockerEndpoint",
    service=ec2.InterfaceVpcEndpointAwsService.ECR_DOCKER
)
```

By default, CDK will place a VPC endpoint in one subnet per AZ. If you wish to override the AZs CDK places the VPC endpoint in,
use the `subnets` parameter as follows:

```python
# vpc is of type Vpc


ec2.InterfaceVpcEndpoint(self, "VPC Endpoint",
    vpc=vpc,
    service=ec2.InterfaceVpcEndpointService("com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc", 443),
    # Choose which availability zones to place the VPC endpoint in, based on
    # available AZs
    subnets=ec2.SubnetSelection(
        availability_zones=["us-east-1a", "us-east-1c"]
    )
)
```

Per the [AWS documentation](https://aws.amazon.com/premiumsupport/knowledge-center/interface-endpoint-availability-zone/), not all
VPC endpoint services are available in all AZs. If you specify the parameter `lookupSupportedAzs`, CDK attempts to discover which
AZs an endpoint service is available in, and will ensure the VPC endpoint is not placed in a subnet that doesn't match those AZs.
These AZs will be stored in cdk.context.json.

```python
# vpc is of type Vpc


ec2.InterfaceVpcEndpoint(self, "VPC Endpoint",
    vpc=vpc,
    service=ec2.InterfaceVpcEndpointService("com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc", 443),
    # Choose which availability zones to place the VPC endpoint in, based on
    # available AZs
    lookup_supported_azs=True
)
```

Pre-defined AWS services are defined in the [InterfaceVpcEndpointAwsService](lib/vpc-endpoint.ts) class, and can be used to
create VPC endpoints without having to configure name, ports, etc. For example, a Keyspaces endpoint can be created for
use in your VPC:

```python
# vpc is of type Vpc


ec2.InterfaceVpcEndpoint(self, "VPC Endpoint",
    vpc=vpc,
    service=ec2.InterfaceVpcEndpointAwsService.KEYSPACES
)
```

#### Security groups for interface VPC endpoints

By default, interface VPC endpoints create a new security group and traffic is **not**
automatically allowed from the VPC CIDR.

Use the `connections` object to allow traffic to flow to the endpoint:

```python
# my_endpoint is of type InterfaceVpcEndpoint


my_endpoint.connections.allow_default_port_from_any_ipv4()
```

Alternatively, existing security groups can be used by specifying the `securityGroups` prop.

### VPC endpoint services

A VPC endpoint service enables you to expose a Network Load Balancer(s) as a provider service to consumers, who connect to your service over a VPC endpoint. You can restrict access to your service via allowed principals (anything that extends ArnPrincipal), and require that new connections be manually accepted.

```python
# network_load_balancer1 is of type NetworkLoadBalancer
# network_load_balancer2 is of type NetworkLoadBalancer


ec2.VpcEndpointService(self, "EndpointService",
    vpc_endpoint_service_load_balancers=[network_load_balancer1, network_load_balancer2],
    acceptance_required=True,
    allowed_principals=[iam.ArnPrincipal("arn:aws:iam::123456789012:root")]
)
```

Endpoint services support private DNS, which makes it easier for clients to connect to your service by automatically setting up DNS in their VPC.
You can enable private DNS on an endpoint service like so:

```python
from aws_cdk.aws_route53 import HostedZone, VpcEndpointServiceDomainName
# zone is of type HostedZone
# vpces is of type VpcEndpointService


VpcEndpointServiceDomainName(self, "EndpointDomain",
    endpoint_service=vpces,
    domain_name="my-stuff.aws-cdk.dev",
    public_hosted_zone=zone
)
```

Note: The domain name must be owned (registered through Route53) by the account the endpoint service is in, or delegated to the account.
The VpcEndpointServiceDomainName will handle the AWS side of domain verification, the process for which can be found
[here](https://docs.aws.amazon.com/vpc/latest/userguide/endpoint-services-dns-validation.html)

### Client VPN endpoint

AWS Client VPN is a managed client-based VPN service that enables you to securely access your AWS
resources and resources in your on-premises network. With Client VPN, you can access your resources
from any location using an OpenVPN-based VPN client.

Use the `addClientVpnEndpoint()` method to add a client VPN endpoint to a VPC:

```python
vpc.add_client_vpn_endpoint("Endpoint",
    cidr="10.100.0.0/16",
    server_certificate_arn="arn:aws:acm:us-east-1:123456789012:certificate/server-certificate-id",
    # Mutual authentication
    client_certificate_arn="arn:aws:acm:us-east-1:123456789012:certificate/client-certificate-id",
    # User-based authentication
    user_based_authentication=ec2.ClientVpnUserBasedAuthentication.federated(saml_provider)
)
```

The endpoint must use at least one [authentication method](https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/client-authentication.html):

* Mutual authentication with a client certificate
* User-based authentication (directory or federated)

If user-based authentication is used, the [self-service portal URL](https://docs.aws.amazon.com/vpn/latest/clientvpn-user/self-service-portal.html)
is made available via a CloudFormation output.

By default, a new security group is created and logging is enabled. Moreover, a rule to
authorize all users to the VPC CIDR is created.

To customize authorization rules, set the `authorizeAllUsersToVpcCidr` prop to `false`
and use `addAuthorizationRule()`:

```python
endpoint = vpc.add_client_vpn_endpoint("Endpoint",
    cidr="10.100.0.0/16",
    server_certificate_arn="arn:aws:acm:us-east-1:123456789012:certificate/server-certificate-id",
    user_based_authentication=ec2.ClientVpnUserBasedAuthentication.federated(saml_provider),
    authorize_all_users_to_vpc_cidr=False
)

endpoint.add_authorization_rule("Rule",
    cidr="10.0.10.0/32",
    group_id="group-id"
)
```

Use `addRoute()` to configure network routes:

```python
endpoint = vpc.add_client_vpn_endpoint("Endpoint",
    cidr="10.100.0.0/16",
    server_certificate_arn="arn:aws:acm:us-east-1:123456789012:certificate/server-certificate-id",
    user_based_authentication=ec2.ClientVpnUserBasedAuthentication.federated(saml_provider)
)

# Client-to-client access
endpoint.add_route("Route",
    cidr="10.100.0.0/16",
    target=ec2.ClientVpnRouteTarget.local()
)
```

Use the `connections` object of the endpoint to allow traffic to other security groups.

## Instances

You can use the `Instance` class to start up a single EC2 instance. For production setups, we recommend
you use an `AutoScalingGroup` from the `aws-autoscaling` module instead, as AutoScalingGroups will take
care of restarting your instance if it ever fails.

### Configuring Instances using CloudFormation Init (cfn-init)

CloudFormation Init allows you to configure your instances by writing files to them, installing software
packages, starting services and running arbitrary commands. By default, if any of the instance setup
commands throw an error, the deployment will fail and roll back to the previously known good state.
The following documentation also applies to `AutoScalingGroup`s.

For the full set of capabilities of this system, see the documentation for
[`AWS::CloudFormation::Init`](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-init.html).
Here is an example of applying some configuration to an instance:

```python
# vpc is of type Vpc
# instance_type is of type InstanceType
# machine_image is of type IMachineImage


ec2.Instance(self, "Instance",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=machine_image,

    # Showing the most complex setup, if you have simpler requirements
    # you can use `CloudFormationInit.fromElements()`.
    init=ec2.CloudFormationInit.from_config_sets(
        config_sets={
            # Applies the configs below in this order
            "default": ["yumPreinstall", "config"]
        },
        configs={
            "yum_preinstall": ec2.InitConfig([
                # Install an Amazon Linux package using yum
                ec2.InitPackage.yum("git")
            ]),
            "config": ec2.InitConfig([
                # Create a JSON file from tokens (can also create other files)
                ec2.InitFile.from_object("/etc/stack.json", {
                    "stack_id": Stack.of(self).stack_id,
                    "stack_name": Stack.of(self).stack_name,
                    "region": Stack.of(self).region
                }),

                # Create a group and user
                ec2.InitGroup.from_name("my-group"),
                ec2.InitUser.from_name("my-user"),

                # Install an RPM from the internet
                ec2.InitPackage.rpm("http://mirrors.ukfast.co.uk/sites/dl.fedoraproject.org/pub/epel/8/Everything/x86_64/Packages/r/rubygem-git-1.5.0-2.el8.noarch.rpm")
            ])
        }
    ),
    init_options=ec2.ApplyCloudFormationInitOptions(
        # Optional, which configsets to activate (['default'] by default)
        config_sets=["default"],

        # Optional, how long the installation is expected to take (5 minutes by default)
        timeout=Duration.minutes(30),

        # Optional, whether to include the --url argument when running cfn-init and cfn-signal commands (false by default)
        include_url=True,

        # Optional, whether to include the --role argument when running cfn-init and cfn-signal commands (false by default)
        include_role=True
    )
)
```

You can have services restarted after the init process has made changes to the system.
To do that, instantiate an `InitServiceRestartHandle` and pass it to the config elements
that need to trigger the restart and the service itself. For example, the following
config writes a config file for nginx, extracts an archive to the root directory, and then
restarts nginx so that it picks up the new config and files:

```python
# my_bucket is of type Bucket


handle = ec2.InitServiceRestartHandle()

ec2.CloudFormationInit.from_elements(
    ec2.InitFile.from_string("/etc/nginx/nginx.conf", "...", service_restart_handles=[handle]),
    ec2.InitSource.from_s3_object("/var/www/html", my_bucket, "html.zip", service_restart_handles=[handle]),
    ec2.InitService.enable("nginx",
        service_restart_handle=handle
    ))
```

### Bastion Hosts

A bastion host functions as an instance used to access servers and resources in a VPC without open up the complete VPC on a network level.
You can use bastion hosts using a standard SSH connection targeting port 22 on the host. As an alternative, you can connect the SSH connection
feature of AWS Systems Manager Session Manager, which does not need an opened security group. (https://aws.amazon.com/about-aws/whats-new/2019/07/session-manager-launches-tunneling-support-for-ssh-and-scp/)

A default bastion host for use via SSM can be configured like:

```python
host = ec2.BastionHostLinux(self, "BastionHost", vpc=vpc)
```

If you want to connect from the internet using SSH, you need to place the host into a public subnet. You can then configure allowed source hosts.

```python
host = ec2.BastionHostLinux(self, "BastionHost",
    vpc=vpc,
    subnet_selection=ec2.SubnetSelection(subnet_type=ec2.SubnetType.PUBLIC)
)
host.allow_ssh_access_from(ec2.Peer.ipv4("1.2.3.4/32"))
```

As there are no SSH public keys deployed on this machine, you need to use [EC2 Instance Connect](https://aws.amazon.com/de/blogs/compute/new-using-amazon-ec2-instance-connect-for-ssh-access-to-your-ec2-instances/)
with the command `aws ec2-instance-connect send-ssh-public-key` to provide your SSH public key.

EBS volume for the bastion host can be encrypted like:

```python
host = ec2.BastionHostLinux(self, "BastionHost",
    vpc=vpc,
    block_devices=[ec2.BlockDevice(
        device_name="EBSBastionHost",
        volume=ec2.BlockDeviceVolume.ebs(10,
            encrypted=True
        )
    )]
)
```

### Block Devices

To add EBS block device mappings, specify the `blockDevices` property. The following example sets the EBS-backed
root device (`/dev/sda1`) size to 50 GiB, and adds another EBS-backed device mapped to `/dev/sdm` that is 100 GiB in
size:

```python
# vpc is of type Vpc
# instance_type is of type InstanceType
# machine_image is of type IMachineImage


ec2.Instance(self, "Instance",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=machine_image,

    # ...

    block_devices=[ec2.BlockDevice(
        device_name="/dev/sda1",
        volume=ec2.BlockDeviceVolume.ebs(50)
    ), ec2.BlockDevice(
        device_name="/dev/sdm",
        volume=ec2.BlockDeviceVolume.ebs(100)
    )
    ]
)
```

### Volumes

Whereas a `BlockDeviceVolume` is an EBS volume that is created and destroyed as part of the creation and destruction of a specific instance. A `Volume` is for when you want an EBS volume separate from any particular instance. A `Volume` is an EBS block device that can be attached to, or detached from, any instance at any time. Some types of `Volume`s can also be attached to multiple instances at the same time to allow you to have shared storage between those instances.

A notable restriction is that a Volume can only be attached to instances in the same availability zone as the Volume itself.

The following demonstrates how to create a 500 GiB encrypted Volume in the `us-west-2a` availability zone, and give a role the ability to attach that Volume to a specific instance:

```python
# instance is of type Instance
# role is of type Role


volume = ec2.Volume(self, "Volume",
    availability_zone="us-west-2a",
    size=Size.gibibytes(500),
    encrypted=True
)

volume.grant_attach_volume(role, [instance])
```

#### Instances Attaching Volumes to Themselves

If you need to grant an instance the ability to attach/detach an EBS volume to/from itself, then using `grantAttachVolume` and `grantDetachVolume` as outlined above
will lead to an unresolvable circular reference between the instance role and the instance. In this case, use `grantAttachVolumeByResourceTag` and `grantDetachVolumeByResourceTag` as follows:

```python
# instance is of type Instance
# volume is of type Volume


attach_grant = volume.grant_attach_volume_by_resource_tag(instance.grant_principal, [instance])
detach_grant = volume.grant_detach_volume_by_resource_tag(instance.grant_principal, [instance])
```

#### Attaching Volumes

The Amazon EC2 documentation for
[Linux Instances](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/AmazonEBS.html) and
[Windows Instances](https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ebs-volumes.html) contains information on how
to attach and detach your Volumes to/from instances, and how to format them for use.

The following is a sample skeleton of EC2 UserData that can be used to attach a Volume to the Linux instance that it is running on:

```python
# instance is of type Instance
# volume is of type Volume


volume.grant_attach_volume_by_resource_tag(instance.grant_principal, [instance])
target_device = "/dev/xvdz"
instance.user_data.add_commands("TOKEN=$(curl -SsfX PUT \"http://169.254.169.254/latest/api/token\" -H \"X-aws-ec2-metadata-token-ttl-seconds: 21600\")", "INSTANCE_ID=$(curl -SsfH \"X-aws-ec2-metadata-token: $TOKEN\" http://169.254.169.254/latest/meta-data/instance-id)", f"aws --region {Stack.of(this).region} ec2 attach-volume --volume-id {volume.volumeId} --instance-id $INSTANCE_ID --device {targetDevice}", f"while ! test -e {targetDevice}; do sleep 1; done")
```

#### Tagging Volumes

You can configure [tag propagation on volume creation](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-propagatetagstovolumeoncreation).

```python
# vpc is of type Vpc
# instance_type is of type InstanceType
# machine_image is of type IMachineImage


ec2.Instance(self, "Instance",
    vpc=vpc,
    machine_image=machine_image,
    instance_type=instance_type,
    propagate_tags_to_volume_on_creation=True
)
```

### Configuring Instance Metadata Service (IMDS)

#### Toggling IMDSv1

You can configure [EC2 Instance Metadata Service](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-metadata.html) options to either
allow both IMDSv1 and IMDSv2 or enforce IMDSv2 when interacting with the IMDS.

To do this for a single `Instance`, you can use the `requireImdsv2` property.
The example below demonstrates IMDSv2 being required on a single `Instance`:

```python
# vpc is of type Vpc
# instance_type is of type InstanceType
# machine_image is of type IMachineImage


ec2.Instance(self, "Instance",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=machine_image,

    # ...

    require_imdsv2=True
)
```

You can also use the either the `InstanceRequireImdsv2Aspect` for EC2 instances or the `LaunchTemplateRequireImdsv2Aspect` for EC2 launch templates
to apply the operation to multiple instances or launch templates, respectively.

The following example demonstrates how to use the `InstanceRequireImdsv2Aspect` to require IMDSv2 for all EC2 instances in a stack:

```python
aspect = ec2.InstanceRequireImdsv2Aspect()
Aspects.of(self).add(aspect)
```

## VPC Flow Logs

VPC Flow Logs is a feature that enables you to capture information about the IP traffic going to and from network interfaces in your VPC. Flow log data can be published to Amazon CloudWatch Logs and Amazon S3. After you've created a flow log, you can retrieve and view its data in the chosen destination. ([https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html](https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html)).

By default a flow log will be created with CloudWatch Logs as the destination.

You can create a flow log like this:

```python
# vpc is of type Vpc


ec2.FlowLog(self, "FlowLog",
    resource_type=ec2.FlowLogResourceType.from_vpc(vpc)
)
```

Or you can add a Flow Log to a VPC by using the addFlowLog method like this:

```python
vpc = ec2.Vpc(self, "Vpc")

vpc.add_flow_log("FlowLog")
```

You can also add multiple flow logs with different destinations.

```python
vpc = ec2.Vpc(self, "Vpc")

vpc.add_flow_log("FlowLogS3",
    destination=ec2.FlowLogDestination.to_s3()
)

vpc.add_flow_log("FlowLogCloudWatch",
    traffic_type=ec2.FlowLogTrafficType.REJECT
)
```

By default the CDK will create the necessary resources for the destination. For the CloudWatch Logs destination
it will create a CloudWatch Logs Log Group as well as the IAM role with the necessary permissions to publish to
the log group. In the case of an S3 destination, it will create the S3 bucket.

If you want to customize any of the destination resources you can provide your own as part of the `destination`.

*CloudWatch Logs*

```python
# vpc is of type Vpc


log_group = logs.LogGroup(self, "MyCustomLogGroup")

role = iam.Role(self, "MyCustomRole",
    assumed_by=iam.ServicePrincipal("vpc-flow-logs.amazonaws.com")
)

ec2.FlowLog(self, "FlowLog",
    resource_type=ec2.FlowLogResourceType.from_vpc(vpc),
    destination=ec2.FlowLogDestination.to_cloud_watch_logs(log_group, role)
)
```

*S3*

```python
# vpc is of type Vpc


bucket = s3.Bucket(self, "MyCustomBucket")

ec2.FlowLog(self, "FlowLog",
    resource_type=ec2.FlowLogResourceType.from_vpc(vpc),
    destination=ec2.FlowLogDestination.to_s3(bucket)
)

ec2.FlowLog(self, "FlowLogWithKeyPrefix",
    resource_type=ec2.FlowLogResourceType.from_vpc(vpc),
    destination=ec2.FlowLogDestination.to_s3(bucket, "prefix/")
)
```

## User Data

User data enables you to run a script when your instances start up.  In order to configure these scripts you can add commands directly to the script
or you can use the UserData's convenience functions to aid in the creation of your script.

A user data could be configured to run a script found in an asset through the following:

```python
from aws_cdk.aws_s3_assets import Asset

# instance is of type Instance


asset = Asset(self, "Asset",
    path="./configure.sh"
)

local_path = instance.user_data.add_s3_download_command(
    bucket=asset.bucket,
    bucket_key=asset.s3_object_key,
    region="us-east-1"
)
instance.user_data.add_execute_file_command(
    file_path=local_path,
    arguments="--verbose -y"
)
asset.grant_read(instance.role)
```

### Multipart user data

In addition, to above the `MultipartUserData` can be used to change instance startup behavior. Multipart user data are composed
from separate parts forming archive. The most common parts are scripts executed during instance set-up. However, there are other
kinds, too.

The advantage of multipart archive is in flexibility when it's needed to add additional parts or to use specialized parts to
fine tune instance startup. Some services (like AWS Batch) supports only `MultipartUserData`.

The parts can be executed at different moment of instance start-up and can serve a different purposes. This is controlled by `contentType` property.
For common scripts, `text/x-shellscript; charset="utf-8"` can be used as content type.

In order to create archive the `MultipartUserData` has to be instantiated. Than, user can add parts to multipart archive using `addPart`. The `MultipartBody` contains methods supporting creation of body parts.

If the very custom part is required, it can be created using `MultipartUserData.fromRawBody`, in this case full control over content type,
transfer encoding, and body properties is given to the user.

Below is an example for creating multipart user data with single body part responsible for installing `awscli` and configuring maximum size
of storage used by Docker containers:

```python
boot_hook_conf = ec2.UserData.for_linux()
boot_hook_conf.add_commands("cloud-init-per once docker_options echo 'OPTIONS=\"${OPTIONS} --storage-opt dm.basesize=40G\"' >> /etc/sysconfig/docker")

setup_commands = ec2.UserData.for_linux()
setup_commands.add_commands("sudo yum install awscli && echo Packages installed らと > /var/tmp/setup")

multipart_user_data = ec2.MultipartUserData()
# The docker has to be configured at early stage, so content type is overridden to boothook
multipart_user_data.add_part(ec2.MultipartBody.from_user_data(boot_hook_conf, "text/cloud-boothook; charset=\"us-ascii\""))
# Execute the rest of setup
multipart_user_data.add_part(ec2.MultipartBody.from_user_data(setup_commands))

ec2.LaunchTemplate(self, "",
    user_data=multipart_user_data,
    block_devices=[]
)
```

For more information see
[Specifying Multiple User Data Blocks Using a MIME Multi Part Archive](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/bootstrap_container_instance.html#multi-part_user_data)

#### Using add*Command on MultipartUserData

To use the `add*Command` methods, that are inherited from the `UserData` interface, on `MultipartUserData` you must add a part
to the `MultipartUserData` and designate it as the reciever for these methods. This is accomplished by using the `addUserDataPart()`
method on `MultipartUserData` with the `makeDefault` argument set to `true`:

```python
multipart_user_data = ec2.MultipartUserData()
commands_user_data = ec2.UserData.for_linux()
multipart_user_data.add_user_data_part(commands_user_data, ec2.MultipartBody.SHELL_SCRIPT, True)

# Adding commands to the multipartUserData adds them to commandsUserData, and vice-versa.
multipart_user_data.add_commands("touch /root/multi.txt")
commands_user_data.add_commands("touch /root/userdata.txt")
```

When used on an EC2 instance, the above `multipartUserData` will create both `multi.txt` and `userdata.txt` in `/root`.

## Importing existing subnet

To import an existing Subnet, call `Subnet.fromSubnetAttributes()` or
`Subnet.fromSubnetId()`. Only if you supply the subnet's Availability Zone
and Route Table Ids when calling `Subnet.fromSubnetAttributes()` will you be
able to use the CDK features that use these values (such as selecting one
subnet per AZ).

Importing an existing subnet looks like this:

```python
# Supply all properties
subnet1 = ec2.Subnet.from_subnet_attributes(self, "SubnetFromAttributes",
    subnet_id="s-1234",
    availability_zone="pub-az-4465",
    route_table_id="rt-145"
)

# Supply only subnet id
subnet2 = ec2.Subnet.from_subnet_id(self, "SubnetFromId", "s-1234")
```

## Launch Templates

A Launch Template is a standardized template that contains the configuration information to launch an instance.
They can be used when launching instances on their own, through Amazon EC2 Auto Scaling, EC2 Fleet, and Spot Fleet.
Launch templates enable you to store launch parameters so that you do not have to specify them every time you launch
an instance. For information on Launch Templates please see the
[official documentation](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html).

The following demonstrates how to create a launch template with an Amazon Machine Image, and security group.

```python
# vpc is of type Vpc


template = ec2.LaunchTemplate(self, "LaunchTemplate",
    machine_image=ec2.MachineImage.latest_amazon_linux(),
    security_group=ec2.SecurityGroup(self, "LaunchTemplateSG",
        vpc=vpc
    )
)
```
'''
import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

from ._jsii import *

import aws_cdk.assets
import aws_cdk.aws_cloudwatch
import aws_cdk.aws_iam
import aws_cdk.aws_kms
import aws_cdk.aws_logs
import aws_cdk.aws_s3
import aws_cdk.aws_s3_assets
import aws_cdk.core
import constructs


class AclCidr(metaclass=jsii.JSIIAbstractClass, jsii_type="@aws-cdk/aws-ec2.AclCidr"):
    '''Either an IPv4 or an IPv6 CIDR.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ec2 as ec2
        
        acl_cidr = ec2.AclCidr.any_ipv4()
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="anyIpv4") # type: ignore[misc]
    @builtins.classmethod
    def any_ipv4(cls) -> "AclCidr":
        '''The CIDR containing all IPv4 addresses (i.e., 0.0.0.0/0).'''
        return typing.cast("AclCidr", jsii.sinvoke(cls, "anyIpv4", []))

    @jsii.member(jsii_name="anyIpv6") # type: ignore[misc]
    @builtins.classmethod
    def any_ipv6(cls) -> "AclCidr":
        '''The CIDR containing all IPv6 addresses (i.e., ::/0).'''
        return typing.cast("AclCidr", jsii.sinvoke(cls, "anyIpv6", []))

    @jsii.member(jsii_name="ipv4") # type: ignore[misc]
    @builtins.classmethod
    def ipv4(cls, ipv4_cidr: builtins.str) -> "AclCidr":
        '''An IP network range in CIDR notation (for example, 172.16.0.0/24).

        :param ipv4_cidr: -
        '''
        return typing.cast("AclCidr", jsii.sinvoke(cls, "ipv4", [ipv4_cidr]))

    @jsii.member(jsii_name="ipv6") # type: ignore[misc]
    @builtins.classmethod
    def ipv6(cls, ipv6_cidr: builtins.str) -> "AclCidr":
        '''An IPv6 network range in CIDR notation (for example, 2001:db8::/48).

        :param ipv6_cidr: -
        '''
        return typing.cast("AclCidr", jsii.sinvoke(cls, "ipv6", [ipv6_cidr]))

    @jsii.member(jsii_name="toCidrConfig") # type: ignore[misc]
    @abc.abstractmethod
    def to_cidr_config(self) -> "AclCidrConfig":
        ...


class _AclCidrProxy(AclCidr):
    @jsii.member(jsii_name="toCidrConfig")
    def to_cidr_config(self) -> "AclCidrConfig":
        return typing.cast("AclCidrConfig", jsii.invoke(self, "toCidrConfig", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, AclCidr).__jsii_proxy_class__ = lambda : _AclCidrProxy


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.AclCidrConfig",
    jsii_struct_bases=[],
    name_mapping={"cidr_block": "cidrBlock", "ipv6_cidr_block": "ipv6CidrBlock"},
)
class AclCidrConfig:
    def __init__(
        self,
        *,
        cidr_block: typing.Optional[builtins.str] = None,
        ipv6_cidr_block: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Acl Configuration for CIDR.

        :param cidr_block: Ipv4 CIDR.
        :param ipv6_cidr_block: Ipv6 CIDR.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            acl_cidr_config = ec2.AclCidrConfig(
                cidr_block="cidrBlock",
                ipv6_cidr_block="ipv6CidrBlock"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if cidr_block is not None:
            self._values["cidr_block"] = cidr_block
        if ipv6_cidr_block is not None:
            self._values["ipv6_cidr_block"] = ipv6_cidr_block

    @builtins.property
    def cidr_block(self) -> typing.Optional[builtins.str]:
        '''Ipv4 CIDR.'''
        result = self._values.get("cidr_block")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ipv6_cidr_block(self) -> typing.Optional[builtins.str]:
        '''Ipv6 CIDR.'''
        result = self._values.get("ipv6_cidr_block")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AclCidrConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.AclIcmp",
    jsii_struct_bases=[],
    name_mapping={"code": "code", "type": "type"},
)
class AclIcmp:
    def __init__(
        self,
        *,
        code: typing.Optional[jsii.Number] = None,
        type: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties to create Icmp.

        :param code: The Internet Control Message Protocol (ICMP) code. You can use -1 to specify all ICMP codes for the given ICMP type. Requirement is conditional: Required if you specify 1 (ICMP) for the protocol parameter.
        :param type: The Internet Control Message Protocol (ICMP) type. You can use -1 to specify all ICMP types. Conditional requirement: Required if you specify 1 (ICMP) for the CreateNetworkAclEntry protocol parameter.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            acl_icmp = ec2.AclIcmp(
                code=123,
                type=123
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if code is not None:
            self._values["code"] = code
        if type is not None:
            self._values["type"] = type

    @builtins.property
    def code(self) -> typing.Optional[jsii.Number]:
        '''The Internet Control Message Protocol (ICMP) code.

        You can use -1 to specify all ICMP
        codes for the given ICMP type. Requirement is conditional: Required if you
        specify 1 (ICMP) for the protocol parameter.
        '''
        result = self._values.get("code")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def type(self) -> typing.Optional[jsii.Number]:
        '''The Internet Control Message Protocol (ICMP) type.

        You can use -1 to specify all ICMP types.
        Conditional requirement: Required if you specify 1 (ICMP) for the CreateNetworkAclEntry protocol parameter.
        '''
        result = self._values.get("type")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AclIcmp(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.AclPortRange",
    jsii_struct_bases=[],
    name_mapping={"from_": "from", "to": "to"},
)
class AclPortRange:
    def __init__(
        self,
        *,
        from_: typing.Optional[jsii.Number] = None,
        to: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties to create PortRange.

        :param from_: The first port in the range. Required if you specify 6 (TCP) or 17 (UDP) for the protocol parameter.
        :param to: The last port in the range. Required if you specify 6 (TCP) or 17 (UDP) for the protocol parameter.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            acl_port_range = ec2.AclPortRange(
                from=123,
                to=123
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if from_ is not None:
            self._values["from_"] = from_
        if to is not None:
            self._values["to"] = to

    @builtins.property
    def from_(self) -> typing.Optional[jsii.Number]:
        '''The first port in the range.

        Required if you specify 6 (TCP) or 17 (UDP) for the protocol parameter.
        '''
        result = self._values.get("from_")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def to(self) -> typing.Optional[jsii.Number]:
        '''The last port in the range.

        Required if you specify 6 (TCP) or 17 (UDP) for the protocol parameter.
        '''
        result = self._values.get("to")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AclPortRange(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AclTraffic(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@aws-cdk/aws-ec2.AclTraffic",
):
    '''The traffic that is configured using a Network ACL entry.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ec2 as ec2
        
        acl_traffic = ec2.AclTraffic.all_traffic()
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="allTraffic") # type: ignore[misc]
    @builtins.classmethod
    def all_traffic(cls) -> "AclTraffic":
        '''Apply the ACL entry to all traffic.'''
        return typing.cast("AclTraffic", jsii.sinvoke(cls, "allTraffic", []))

    @jsii.member(jsii_name="icmp") # type: ignore[misc]
    @builtins.classmethod
    def icmp(
        cls,
        *,
        code: typing.Optional[jsii.Number] = None,
        type: typing.Optional[jsii.Number] = None,
    ) -> "AclTraffic":
        '''Apply the ACL entry to ICMP traffic of given type and code.

        :param code: The Internet Control Message Protocol (ICMP) code. You can use -1 to specify all ICMP codes for the given ICMP type. Requirement is conditional: Required if you specify 1 (ICMP) for the protocol parameter.
        :param type: The Internet Control Message Protocol (ICMP) type. You can use -1 to specify all ICMP types. Conditional requirement: Required if you specify 1 (ICMP) for the CreateNetworkAclEntry protocol parameter.
        '''
        props = AclIcmp(code=code, type=type)

        return typing.cast("AclTraffic", jsii.sinvoke(cls, "icmp", [props]))

    @jsii.member(jsii_name="icmpv6") # type: ignore[misc]
    @builtins.classmethod
    def icmpv6(
        cls,
        *,
        code: typing.Optional[jsii.Number] = None,
        type: typing.Optional[jsii.Number] = None,
    ) -> "AclTraffic":
        '''Apply the ACL entry to ICMPv6 traffic of given type and code.

        Requires an IPv6 CIDR block.

        :param code: The Internet Control Message Protocol (ICMP) code. You can use -1 to specify all ICMP codes for the given ICMP type. Requirement is conditional: Required if you specify 1 (ICMP) for the protocol parameter.
        :param type: The Internet Control Message Protocol (ICMP) type. You can use -1 to specify all ICMP types. Conditional requirement: Required if you specify 1 (ICMP) for the CreateNetworkAclEntry protocol parameter.
        '''
        props = AclIcmp(code=code, type=type)

        return typing.cast("AclTraffic", jsii.sinvoke(cls, "icmpv6", [props]))

    @jsii.member(jsii_name="tcpPort") # type: ignore[misc]
    @builtins.classmethod
    def tcp_port(cls, port: jsii.Number) -> "AclTraffic":
        '''Apply the ACL entry to TCP traffic on a given port.

        :param port: -
        '''
        return typing.cast("AclTraffic", jsii.sinvoke(cls, "tcpPort", [port]))

    @jsii.member(jsii_name="tcpPortRange") # type: ignore[misc]
    @builtins.classmethod
    def tcp_port_range(
        cls,
        start_port: jsii.Number,
        end_port: jsii.Number,
    ) -> "AclTraffic":
        '''Apply the ACL entry to TCP traffic on a given port range.

        :param start_port: -
        :param end_port: -
        '''
        return typing.cast("AclTraffic", jsii.sinvoke(cls, "tcpPortRange", [start_port, end_port]))

    @jsii.member(jsii_name="udpPort") # type: ignore[misc]
    @builtins.classmethod
    def udp_port(cls, port: jsii.Number) -> "AclTraffic":
        '''Apply the ACL entry to UDP traffic on a given port.

        :param port: -
        '''
        return typing.cast("AclTraffic", jsii.sinvoke(cls, "udpPort", [port]))

    @jsii.member(jsii_name="udpPortRange") # type: ignore[misc]
    @builtins.classmethod
    def udp_port_range(
        cls,
        start_port: jsii.Number,
        end_port: jsii.Number,
    ) -> "AclTraffic":
        '''Apply the ACL entry to UDP traffic on a given port range.

        :param start_port: -
        :param end_port: -
        '''
        return typing.cast("AclTraffic", jsii.sinvoke(cls, "udpPortRange", [start_port, end_port]))

    @jsii.member(jsii_name="toTrafficConfig") # type: ignore[misc]
    @abc.abstractmethod
    def to_traffic_config(self) -> "AclTrafficConfig":
        ...


class _AclTrafficProxy(AclTraffic):
    @jsii.member(jsii_name="toTrafficConfig")
    def to_traffic_config(self) -> "AclTrafficConfig":
        return typing.cast("AclTrafficConfig", jsii.invoke(self, "toTrafficConfig", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, AclTraffic).__jsii_proxy_class__ = lambda : _AclTrafficProxy


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.AclTrafficConfig",
    jsii_struct_bases=[],
    name_mapping={"protocol": "protocol", "icmp": "icmp", "port_range": "portRange"},
)
class AclTrafficConfig:
    def __init__(
        self,
        *,
        protocol: jsii.Number,
        icmp: typing.Optional[AclIcmp] = None,
        port_range: typing.Optional[AclPortRange] = None,
    ) -> None:
        '''Acl Configuration for traffic.

        :param protocol: The protocol number. A value of "-1" means all protocols. If you specify "-1" or a protocol number other than "6" (TCP), "17" (UDP), or "1" (ICMP), traffic on all ports is allowed, regardless of any ports or ICMP types or codes that you specify. If you specify protocol "58" (ICMPv6) and specify an IPv4 CIDR block, traffic for all ICMP types and codes allowed, regardless of any that you specify. If you specify protocol "58" (ICMPv6) and specify an IPv6 CIDR block, you must specify an ICMP type and code. Default: 17
        :param icmp: The Internet Control Message Protocol (ICMP) code and type. Default: - Required if specifying 1 (ICMP) for the protocol parameter.
        :param port_range: The range of port numbers for the UDP/TCP protocol. Default: - Required if specifying 6 (TCP) or 17 (UDP) for the protocol parameter

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            acl_traffic_config = ec2.AclTrafficConfig(
                protocol=123,
            
                # the properties below are optional
                icmp=ec2.AclIcmp(
                    code=123,
                    type=123
                ),
                port_range=ec2.AclPortRange(
                    from=123,
                    to=123
                )
            )
        '''
        if isinstance(icmp, dict):
            icmp = AclIcmp(**icmp)
        if isinstance(port_range, dict):
            port_range = AclPortRange(**port_range)
        self._values: typing.Dict[str, typing.Any] = {
            "protocol": protocol,
        }
        if icmp is not None:
            self._values["icmp"] = icmp
        if port_range is not None:
            self._values["port_range"] = port_range

    @builtins.property
    def protocol(self) -> jsii.Number:
        '''The protocol number.

        A value of "-1" means all protocols.

        If you specify "-1" or a protocol number other than "6" (TCP), "17" (UDP),
        or "1" (ICMP), traffic on all ports is allowed, regardless of any ports or
        ICMP types or codes that you specify.

        If you specify protocol "58" (ICMPv6) and specify an IPv4 CIDR
        block, traffic for all ICMP types and codes allowed, regardless of any that
        you specify. If you specify protocol "58" (ICMPv6) and specify an IPv6 CIDR
        block, you must specify an ICMP type and code.

        :default: 17
        '''
        result = self._values.get("protocol")
        assert result is not None, "Required property 'protocol' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def icmp(self) -> typing.Optional[AclIcmp]:
        '''The Internet Control Message Protocol (ICMP) code and type.

        :default: - Required if specifying 1 (ICMP) for the protocol parameter.
        '''
        result = self._values.get("icmp")
        return typing.cast(typing.Optional[AclIcmp], result)

    @builtins.property
    def port_range(self) -> typing.Optional[AclPortRange]:
        '''The range of port numbers for the UDP/TCP protocol.

        :default: - Required if specifying 6 (TCP) or 17 (UDP) for the protocol parameter
        '''
        result = self._values.get("port_range")
        return typing.cast(typing.Optional[AclPortRange], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AclTrafficConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@aws-cdk/aws-ec2.Action")
class Action(enum.Enum):
    '''What action to apply to traffic matching the ACL.'''

    ALLOW = "ALLOW"
    '''Allow the traffic.'''
    DENY = "DENY"
    '''Deny the traffic.'''


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.AddRouteOptions",
    jsii_struct_bases=[],
    name_mapping={
        "router_id": "routerId",
        "router_type": "routerType",
        "destination_cidr_block": "destinationCidrBlock",
        "destination_ipv6_cidr_block": "destinationIpv6CidrBlock",
        "enables_internet_connectivity": "enablesInternetConnectivity",
    },
)
class AddRouteOptions:
    def __init__(
        self,
        *,
        router_id: builtins.str,
        router_type: "RouterType",
        destination_cidr_block: typing.Optional[builtins.str] = None,
        destination_ipv6_cidr_block: typing.Optional[builtins.str] = None,
        enables_internet_connectivity: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Options for adding a new route to a subnet.

        :param router_id: The ID of the router. Can be an instance ID, gateway ID, etc, depending on the router type.
        :param router_type: What type of router to route this traffic to.
        :param destination_cidr_block: IPv4 range this route applies to. Default: '0.0.0.0/0'
        :param destination_ipv6_cidr_block: IPv6 range this route applies to. Default: - Uses IPv6
        :param enables_internet_connectivity: Whether this route will enable internet connectivity. If true, this route will be added before any AWS resources that depend on internet connectivity in the VPC will be created. Default: false

        :exampleMetadata: infused

        Example::

            vpc = ec2.Vpc(self, "VPC",
                subnet_configuration=[ec2.SubnetConfiguration(
                    subnet_type=ec2.SubnetType.PUBLIC,
                    name="Public"
                ), ec2.SubnetConfiguration(
                    subnet_type=ec2.SubnetType.ISOLATED,
                    name="Isolated"
                )]
            )
            
            (vpc.isolated_subnets[0]).add_route("StaticRoute",
                router_id=vpc.internet_gateway_id,
                router_type=ec2.RouterType.GATEWAY,
                destination_cidr_block="8.8.8.8/32"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "router_id": router_id,
            "router_type": router_type,
        }
        if destination_cidr_block is not None:
            self._values["destination_cidr_block"] = destination_cidr_block
        if destination_ipv6_cidr_block is not None:
            self._values["destination_ipv6_cidr_block"] = destination_ipv6_cidr_block
        if enables_internet_connectivity is not None:
            self._values["enables_internet_connectivity"] = enables_internet_connectivity

    @builtins.property
    def router_id(self) -> builtins.str:
        '''The ID of the router.

        Can be an instance ID, gateway ID, etc, depending on the router type.
        '''
        result = self._values.get("router_id")
        assert result is not None, "Required property 'router_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def router_type(self) -> "RouterType":
        '''What type of router to route this traffic to.'''
        result = self._values.get("router_type")
        assert result is not None, "Required property 'router_type' is missing"
        return typing.cast("RouterType", result)

    @builtins.property
    def destination_cidr_block(self) -> typing.Optional[builtins.str]:
        '''IPv4 range this route applies to.

        :default: '0.0.0.0/0'
        '''
        result = self._values.get("destination_cidr_block")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def destination_ipv6_cidr_block(self) -> typing.Optional[builtins.str]:
        '''IPv6 range this route applies to.

        :default: - Uses IPv6
        '''
        result = self._values.get("destination_ipv6_cidr_block")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def enables_internet_connectivity(self) -> typing.Optional[builtins.bool]:
        '''Whether this route will enable internet connectivity.

        If true, this route will be added before any AWS resources that depend
        on internet connectivity in the VPC will be created.

        :default: false
        '''
        result = self._values.get("enables_internet_connectivity")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AddRouteOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@aws-cdk/aws-ec2.AmazonLinuxCpuType")
class AmazonLinuxCpuType(enum.Enum):
    '''CPU type.

    :exampleMetadata: lit=test/example.images.lit.ts infused

    Example::

        # Pick the right Amazon Linux edition. All arguments shown are optional
        # and will default to these values when omitted.
        amzn_linux = ec2.MachineImage.latest_amazon_linux(
            generation=ec2.AmazonLinuxGeneration.AMAZON_LINUX,
            edition=ec2.AmazonLinuxEdition.STANDARD,
            virtualization=ec2.AmazonLinuxVirt.HVM,
            storage=ec2.AmazonLinuxStorage.GENERAL_PURPOSE,
            cpu_type=ec2.AmazonLinuxCpuType.X86_64
        )
        
        # Pick a Windows edition to use
        windows = ec2.MachineImage.latest_windows(ec2.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE)
        
        # Read AMI id from SSM parameter store
        ssm = ec2.MachineImage.from_sSMParameter("/my/ami", ec2.OperatingSystemType.LINUX)
        
        # Look up the most recent image matching a set of AMI filters.
        # In this case, look up the NAT instance AMI, by using a wildcard
        # in the 'name' field:
        nat_ami = ec2.MachineImage.lookup(
            name="amzn-ami-vpc-nat-*",
            owners=["amazon"]
        )
        
        # For other custom (Linux) images, instantiate a `GenericLinuxImage` with
        # a map giving the AMI to in for each region:
        linux = ec2.MachineImage.generic_linux({
            "us-east-1": "ami-97785bed",
            "eu-west-1": "ami-12345678"
        })
        
        # For other custom (Windows) images, instantiate a `GenericWindowsImage` with
        # a map giving the AMI to in for each region:
        generic_windows = ec2.MachineImage.generic_windows({
            "us-east-1": "ami-97785bed",
            "eu-west-1": "ami-12345678"
        })
    '''

    ARM_64 = "ARM_64"
    '''arm64 CPU type.'''
    X86_64 = "X86_64"
    '''x86_64 CPU type.'''


@jsii.enum(jsii_type="@aws-cdk/aws-ec2.AmazonLinuxEdition")
class AmazonLinuxEdition(enum.Enum):
    '''Amazon Linux edition.

    :exampleMetadata: lit=test/example.images.lit.ts infused

    Example::

        # Pick the right Amazon Linux edition. All arguments shown are optional
        # and will default to these values when omitted.
        amzn_linux = ec2.MachineImage.latest_amazon_linux(
            generation=ec2.AmazonLinuxGeneration.AMAZON_LINUX,
            edition=ec2.AmazonLinuxEdition.STANDARD,
            virtualization=ec2.AmazonLinuxVirt.HVM,
            storage=ec2.AmazonLinuxStorage.GENERAL_PURPOSE,
            cpu_type=ec2.AmazonLinuxCpuType.X86_64
        )
        
        # Pick a Windows edition to use
        windows = ec2.MachineImage.latest_windows(ec2.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE)
        
        # Read AMI id from SSM parameter store
        ssm = ec2.MachineImage.from_sSMParameter("/my/ami", ec2.OperatingSystemType.LINUX)
        
        # Look up the most recent image matching a set of AMI filters.
        # In this case, look up the NAT instance AMI, by using a wildcard
        # in the 'name' field:
        nat_ami = ec2.MachineImage.lookup(
            name="amzn-ami-vpc-nat-*",
            owners=["amazon"]
        )
        
        # For other custom (Linux) images, instantiate a `GenericLinuxImage` with
        # a map giving the AMI to in for each region:
        linux = ec2.MachineImage.generic_linux({
            "us-east-1": "ami-97785bed",
            "eu-west-1": "ami-12345678"
        })
        
        # For other custom (Windows) images, instantiate a `GenericWindowsImage` with
        # a map giving the AMI to in for each region:
        generic_windows = ec2.MachineImage.generic_windows({
            "us-east-1": "ami-97785bed",
            "eu-west-1": "ami-12345678"
        })
    '''

    STANDARD = "STANDARD"
    '''Standard edition.'''
    MINIMAL = "MINIMAL"
    '''Minimal edition.'''


@jsii.enum(jsii_type="@aws-cdk/aws-ec2.AmazonLinuxGeneration")
class AmazonLinuxGeneration(enum.Enum):
    '''What generation of Amazon Linux to use.

    :exampleMetadata: infused

    Example::

        # vpc is of type Vpc
        
        my_compute_env = batch.ComputeEnvironment(self, "ComputeEnv",
            compute_resources=batch.ComputeResources(
                image=ecs.EcsOptimizedAmi(
                    generation=ec2.AmazonLinuxGeneration.AMAZON_LINUX_2
                ),
                vpc=vpc
            )
        )
    '''

    AMAZON_LINUX = "AMAZON_LINUX"
    '''Amazon Linux.'''
    AMAZON_LINUX_2 = "AMAZON_LINUX_2"
    '''Amazon Linux 2.'''


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.AmazonLinuxImageProps",
    jsii_struct_bases=[],
    name_mapping={
        "cached_in_context": "cachedInContext",
        "cpu_type": "cpuType",
        "edition": "edition",
        "generation": "generation",
        "storage": "storage",
        "user_data": "userData",
        "virtualization": "virtualization",
    },
)
class AmazonLinuxImageProps:
    def __init__(
        self,
        *,
        cached_in_context: typing.Optional[builtins.bool] = None,
        cpu_type: typing.Optional[AmazonLinuxCpuType] = None,
        edition: typing.Optional[AmazonLinuxEdition] = None,
        generation: typing.Optional[AmazonLinuxGeneration] = None,
        storage: typing.Optional["AmazonLinuxStorage"] = None,
        user_data: typing.Optional["UserData"] = None,
        virtualization: typing.Optional["AmazonLinuxVirt"] = None,
    ) -> None:
        '''Amazon Linux image properties.

        :param cached_in_context: Whether the AMI ID is cached to be stable between deployments. By default, the newest image is used on each deployment. This will cause instances to be replaced whenever a new version is released, and may cause downtime if there aren't enough running instances in the AutoScalingGroup to reschedule the tasks on. If set to true, the AMI ID will be cached in ``cdk.context.json`` and the same value will be used on future runs. Your instances will not be replaced but your AMI version will grow old over time. To refresh the AMI lookup, you will have to evict the value from the cache using the ``cdk context`` command. See https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information. Can not be set to ``true`` in environment-agnostic stacks. Default: false
        :param cpu_type: CPU Type. Default: X86_64
        :param edition: What edition of Amazon Linux to use. Default: Standard
        :param generation: What generation of Amazon Linux to use. Default: AmazonLinux
        :param storage: What storage backed image to use. Default: GeneralPurpose
        :param user_data: Initial user data. Default: - Empty UserData for Linux machines
        :param virtualization: Virtualization type. Default: HVM

        :exampleMetadata: lit=test/example.images.lit.ts infused

        Example::

            # Pick the right Amazon Linux edition. All arguments shown are optional
            # and will default to these values when omitted.
            amzn_linux = ec2.MachineImage.latest_amazon_linux(
                generation=ec2.AmazonLinuxGeneration.AMAZON_LINUX,
                edition=ec2.AmazonLinuxEdition.STANDARD,
                virtualization=ec2.AmazonLinuxVirt.HVM,
                storage=ec2.AmazonLinuxStorage.GENERAL_PURPOSE,
                cpu_type=ec2.AmazonLinuxCpuType.X86_64
            )
            
            # Pick a Windows edition to use
            windows = ec2.MachineImage.latest_windows(ec2.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE)
            
            # Read AMI id from SSM parameter store
            ssm = ec2.MachineImage.from_sSMParameter("/my/ami", ec2.OperatingSystemType.LINUX)
            
            # Look up the most recent image matching a set of AMI filters.
            # In this case, look up the NAT instance AMI, by using a wildcard
            # in the 'name' field:
            nat_ami = ec2.MachineImage.lookup(
                name="amzn-ami-vpc-nat-*",
                owners=["amazon"]
            )
            
            # For other custom (Linux) images, instantiate a `GenericLinuxImage` with
            # a map giving the AMI to in for each region:
            linux = ec2.MachineImage.generic_linux({
                "us-east-1": "ami-97785bed",
                "eu-west-1": "ami-12345678"
            })
            
            # For other custom (Windows) images, instantiate a `GenericWindowsImage` with
            # a map giving the AMI to in for each region:
            generic_windows = ec2.MachineImage.generic_windows({
                "us-east-1": "ami-97785bed",
                "eu-west-1": "ami-12345678"
            })
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if cached_in_context is not None:
            self._values["cached_in_context"] = cached_in_context
        if cpu_type is not None:
            self._values["cpu_type"] = cpu_type
        if edition is not None:
            self._values["edition"] = edition
        if generation is not None:
            self._values["generation"] = generation
        if storage is not None:
            self._values["storage"] = storage
        if user_data is not None:
            self._values["user_data"] = user_data
        if virtualization is not None:
            self._values["virtualization"] = virtualization

    @builtins.property
    def cached_in_context(self) -> typing.Optional[builtins.bool]:
        '''Whether the AMI ID is cached to be stable between deployments.

        By default, the newest image is used on each deployment. This will cause
        instances to be replaced whenever a new version is released, and may cause
        downtime if there aren't enough running instances in the AutoScalingGroup
        to reschedule the tasks on.

        If set to true, the AMI ID will be cached in ``cdk.context.json`` and the
        same value will be used on future runs. Your instances will not be replaced
        but your AMI version will grow old over time. To refresh the AMI lookup,
        you will have to evict the value from the cache using the ``cdk context``
        command. See https://docs.aws.amazon.com/cdk/latest/guide/context.html for
        more information.

        Can not be set to ``true`` in environment-agnostic stacks.

        :default: false
        '''
        result = self._values.get("cached_in_context")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def cpu_type(self) -> typing.Optional[AmazonLinuxCpuType]:
        '''CPU Type.

        :default: X86_64
        '''
        result = self._values.get("cpu_type")
        return typing.cast(typing.Optional[AmazonLinuxCpuType], result)

    @builtins.property
    def edition(self) -> typing.Optional[AmazonLinuxEdition]:
        '''What edition of Amazon Linux to use.

        :default: Standard
        '''
        result = self._values.get("edition")
        return typing.cast(typing.Optional[AmazonLinuxEdition], result)

    @builtins.property
    def generation(self) -> typing.Optional[AmazonLinuxGeneration]:
        '''What generation of Amazon Linux to use.

        :default: AmazonLinux
        '''
        result = self._values.get("generation")
        return typing.cast(typing.Optional[AmazonLinuxGeneration], result)

    @builtins.property
    def storage(self) -> typing.Optional["AmazonLinuxStorage"]:
        '''What storage backed image to use.

        :default: GeneralPurpose
        '''
        result = self._values.get("storage")
        return typing.cast(typing.Optional["AmazonLinuxStorage"], result)

    @builtins.property
    def user_data(self) -> typing.Optional["UserData"]:
        '''Initial user data.

        :default: - Empty UserData for Linux machines
        '''
        result = self._values.get("user_data")
        return typing.cast(typing.Optional["UserData"], result)

    @builtins.property
    def virtualization(self) -> typing.Optional["AmazonLinuxVirt"]:
        '''Virtualization type.

        :default: HVM
        '''
        result = self._values.get("virtualization")
        return typing.cast(typing.Optional["AmazonLinuxVirt"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AmazonLinuxImageProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@aws-cdk/aws-ec2.AmazonLinuxStorage")
class AmazonLinuxStorage(enum.Enum):
    '''
    :exampleMetadata: lit=test/example.images.lit.ts infused

    Example::

        # Pick the right Amazon Linux edition. All arguments shown are optional
        # and will default to these values when omitted.
        amzn_linux = ec2.MachineImage.latest_amazon_linux(
            generation=ec2.AmazonLinuxGeneration.AMAZON_LINUX,
            edition=ec2.AmazonLinuxEdition.STANDARD,
            virtualization=ec2.AmazonLinuxVirt.HVM,
            storage=ec2.AmazonLinuxStorage.GENERAL_PURPOSE,
            cpu_type=ec2.AmazonLinuxCpuType.X86_64
        )
        
        # Pick a Windows edition to use
        windows = ec2.MachineImage.latest_windows(ec2.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE)
        
        # Read AMI id from SSM parameter store
        ssm = ec2.MachineImage.from_sSMParameter("/my/ami", ec2.OperatingSystemType.LINUX)
        
        # Look up the most recent image matching a set of AMI filters.
        # In this case, look up the NAT instance AMI, by using a wildcard
        # in the 'name' field:
        nat_ami = ec2.MachineImage.lookup(
            name="amzn-ami-vpc-nat-*",
            owners=["amazon"]
        )
        
        # For other custom (Linux) images, instantiate a `GenericLinuxImage` with
        # a map giving the AMI to in for each region:
        linux = ec2.MachineImage.generic_linux({
            "us-east-1": "ami-97785bed",
            "eu-west-1": "ami-12345678"
        })
        
        # For other custom (Windows) images, instantiate a `GenericWindowsImage` with
        # a map giving the AMI to in for each region:
        generic_windows = ec2.MachineImage.generic_windows({
            "us-east-1": "ami-97785bed",
            "eu-west-1": "ami-12345678"
        })
    '''

    EBS = "EBS"
    '''EBS-backed storage.'''
    GENERAL_PURPOSE = "GENERAL_PURPOSE"
    '''General Purpose-based storage (recommended).'''


@jsii.enum(jsii_type="@aws-cdk/aws-ec2.AmazonLinuxVirt")
class AmazonLinuxVirt(enum.Enum):
    '''Virtualization type for Amazon Linux.

    :exampleMetadata: lit=test/example.images.lit.ts infused

    Example::

        # Pick the right Amazon Linux edition. All arguments shown are optional
        # and will default to these values when omitted.
        amzn_linux = ec2.MachineImage.latest_amazon_linux(
            generation=ec2.AmazonLinuxGeneration.AMAZON_LINUX,
            edition=ec2.AmazonLinuxEdition.STANDARD,
            virtualization=ec2.AmazonLinuxVirt.HVM,
            storage=ec2.AmazonLinuxStorage.GENERAL_PURPOSE,
            cpu_type=ec2.AmazonLinuxCpuType.X86_64
        )
        
        # Pick a Windows edition to use
        windows = ec2.MachineImage.latest_windows(ec2.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE)
        
        # Read AMI id from SSM parameter store
        ssm = ec2.MachineImage.from_sSMParameter("/my/ami", ec2.OperatingSystemType.LINUX)
        
        # Look up the most recent image matching a set of AMI filters.
        # In this case, look up the NAT instance AMI, by using a wildcard
        # in the 'name' field:
        nat_ami = ec2.MachineImage.lookup(
            name="amzn-ami-vpc-nat-*",
            owners=["amazon"]
        )
        
        # For other custom (Linux) images, instantiate a `GenericLinuxImage` with
        # a map giving the AMI to in for each region:
        linux = ec2.MachineImage.generic_linux({
            "us-east-1": "ami-97785bed",
            "eu-west-1": "ami-12345678"
        })
        
        # For other custom (Windows) images, instantiate a `GenericWindowsImage` with
        # a map giving the AMI to in for each region:
        generic_windows = ec2.MachineImage.generic_windows({
            "us-east-1": "ami-97785bed",
            "eu-west-1": "ami-12345678"
        })
    '''

    HVM = "HVM"
    '''HVM virtualization (recommended).'''
    PV = "PV"
    '''PV virtualization.'''


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.ApplyCloudFormationInitOptions",
    jsii_struct_bases=[],
    name_mapping={
        "config_sets": "configSets",
        "embed_fingerprint": "embedFingerprint",
        "ignore_failures": "ignoreFailures",
        "include_role": "includeRole",
        "include_url": "includeUrl",
        "print_log": "printLog",
        "timeout": "timeout",
    },
)
class ApplyCloudFormationInitOptions:
    def __init__(
        self,
        *,
        config_sets: typing.Optional[typing.Sequence[builtins.str]] = None,
        embed_fingerprint: typing.Optional[builtins.bool] = None,
        ignore_failures: typing.Optional[builtins.bool] = None,
        include_role: typing.Optional[builtins.bool] = None,
        include_url: typing.Optional[builtins.bool] = None,
        print_log: typing.Optional[builtins.bool] = None,
        timeout: typing.Optional[aws_cdk.core.Duration] = None,
    ) -> None:
        '''Options for applying CloudFormation init to an instance or instance group.

        :param config_sets: ConfigSet to activate. Default: ['default']
        :param embed_fingerprint: Force instance replacement by embedding a config fingerprint. If ``true`` (the default), a hash of the config will be embedded into the UserData, so that if the config changes, the UserData changes. - If the EC2 instance is instance-store backed or ``userDataCausesReplacement`` is set, this will cause the instance to be replaced and the new configuration to be applied. - If the instance is EBS-backed and ``userDataCausesReplacement`` is not set, the change of UserData will make the instance restart but not be replaced, and the configuration will not be applied automatically. If ``false``, no hash will be embedded, and if the CloudFormation Init config changes nothing will happen to the running instance. If a config update introduces errors, you will not notice until after the CloudFormation deployment successfully finishes and the next instance fails to launch. Default: true
        :param ignore_failures: Don't fail the instance creation when cfn-init fails. You can use this to prevent CloudFormation from rolling back when instances fail to start up, to help in debugging. Default: false
        :param include_role: Include --role argument when running cfn-init and cfn-signal commands. This will be the IAM instance profile attached to the EC2 instance Default: false
        :param include_url: Include --url argument when running cfn-init and cfn-signal commands. This will be the cloudformation endpoint in the deployed region e.g. https://cloudformation.us-east-1.amazonaws.com Default: false
        :param print_log: Print the results of running cfn-init to the Instance System Log. By default, the output of running cfn-init is written to a log file on the instance. Set this to ``true`` to print it to the System Log (visible from the EC2 Console), ``false`` to not print it. (Be aware that the system log is refreshed at certain points in time of the instance life cycle, and successful execution may not always show up). Default: true
        :param timeout: Timeout waiting for the configuration to be applied. Default: Duration.minutes(5)

        :exampleMetadata: infused

        Example::

            # vpc is of type Vpc
            # instance_type is of type InstanceType
            # machine_image is of type IMachineImage
            
            
            ec2.Instance(self, "Instance",
                vpc=vpc,
                instance_type=instance_type,
                machine_image=machine_image,
            
                # Showing the most complex setup, if you have simpler requirements
                # you can use `CloudFormationInit.fromElements()`.
                init=ec2.CloudFormationInit.from_config_sets(
                    config_sets={
                        # Applies the configs below in this order
                        "default": ["yumPreinstall", "config"]
                    },
                    configs={
                        "yum_preinstall": ec2.InitConfig([
                            # Install an Amazon Linux package using yum
                            ec2.InitPackage.yum("git")
                        ]),
                        "config": ec2.InitConfig([
                            # Create a JSON file from tokens (can also create other files)
                            ec2.InitFile.from_object("/etc/stack.json", {
                                "stack_id": Stack.of(self).stack_id,
                                "stack_name": Stack.of(self).stack_name,
                                "region": Stack.of(self).region
                            }),
            
                            # Create a group and user
                            ec2.InitGroup.from_name("my-group"),
                            ec2.InitUser.from_name("my-user"),
            
                            # Install an RPM from the internet
                            ec2.InitPackage.rpm("http://mirrors.ukfast.co.uk/sites/dl.fedoraproject.org/pub/epel/8/Everything/x86_64/Packages/r/rubygem-git-1.5.0-2.el8.noarch.rpm")
                        ])
                    }
                ),
                init_options=ec2.ApplyCloudFormationInitOptions(
                    # Optional, which configsets to activate (['default'] by default)
                    config_sets=["default"],
            
                    # Optional, how long the installation is expected to take (5 minutes by default)
                    timeout=Duration.minutes(30),
            
                    # Optional, whether to include the --url argument when running cfn-init and cfn-signal commands (false by default)
                    include_url=True,
            
                    # Optional, whether to include the --role argument when running cfn-init and cfn-signal commands (false by default)
                    include_role=True
                )
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if config_sets is not None:
            self._values["config_sets"] = config_sets
        if embed_fingerprint is not None:
            self._values["embed_fingerprint"] = embed_fingerprint
        if ignore_failures is not None:
            self._values["ignore_failures"] = ignore_failures
        if include_role is not None:
            self._values["include_role"] = include_role
        if include_url is not None:
            self._values["include_url"] = include_url
        if print_log is not None:
            self._values["print_log"] = print_log
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def config_sets(self) -> typing.Optional[typing.List[builtins.str]]:
        '''ConfigSet to activate.

        :default: ['default']
        '''
        result = self._values.get("config_sets")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def embed_fingerprint(self) -> typing.Optional[builtins.bool]:
        '''Force instance replacement by embedding a config fingerprint.

        If ``true`` (the default), a hash of the config will be embedded into the
        UserData, so that if the config changes, the UserData changes.

        - If the EC2 instance is instance-store backed or
          ``userDataCausesReplacement`` is set, this will cause the instance to be
          replaced and the new configuration to be applied.
        - If the instance is EBS-backed and ``userDataCausesReplacement`` is not
          set, the change of UserData will make the instance restart but not be
          replaced, and the configuration will not be applied automatically.

        If ``false``, no hash will be embedded, and if the CloudFormation Init
        config changes nothing will happen to the running instance. If a
        config update introduces errors, you will not notice until after the
        CloudFormation deployment successfully finishes and the next instance
        fails to launch.

        :default: true
        '''
        result = self._values.get("embed_fingerprint")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def ignore_failures(self) -> typing.Optional[builtins.bool]:
        '''Don't fail the instance creation when cfn-init fails.

        You can use this to prevent CloudFormation from rolling back when
        instances fail to start up, to help in debugging.

        :default: false
        '''
        result = self._values.get("ignore_failures")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def include_role(self) -> typing.Optional[builtins.bool]:
        '''Include --role argument when running cfn-init and cfn-signal commands.

        This will be the IAM instance profile attached to the EC2 instance

        :default: false
        '''
        result = self._values.get("include_role")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def include_url(self) -> typing.Optional[builtins.bool]:
        '''Include --url argument when running cfn-init and cfn-signal commands.

        This will be the cloudformation endpoint in the deployed region
        e.g. https://cloudformation.us-east-1.amazonaws.com

        :default: false
        '''
        result = self._values.get("include_url")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def print_log(self) -> typing.Optional[builtins.bool]:
        '''Print the results of running cfn-init to the Instance System Log.

        By default, the output of running cfn-init is written to a log file
        on the instance. Set this to ``true`` to print it to the System Log
        (visible from the EC2 Console), ``false`` to not print it.

        (Be aware that the system log is refreshed at certain points in
        time of the instance life cycle, and successful execution may
        not always show up).

        :default: true
        '''
        result = self._values.get("print_log")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def timeout(self) -> typing.Optional[aws_cdk.core.Duration]:
        '''Timeout waiting for the configuration to be applied.

        :default: Duration.minutes(5)
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[aws_cdk.core.Duration], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ApplyCloudFormationInitOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.AttachInitOptions",
    jsii_struct_bases=[],
    name_mapping={
        "instance_role": "instanceRole",
        "platform": "platform",
        "user_data": "userData",
        "config_sets": "configSets",
        "embed_fingerprint": "embedFingerprint",
        "ignore_failures": "ignoreFailures",
        "include_role": "includeRole",
        "include_url": "includeUrl",
        "print_log": "printLog",
        "signal_resource": "signalResource",
    },
)
class AttachInitOptions:
    def __init__(
        self,
        *,
        instance_role: aws_cdk.aws_iam.IRole,
        platform: "OperatingSystemType",
        user_data: "UserData",
        config_sets: typing.Optional[typing.Sequence[builtins.str]] = None,
        embed_fingerprint: typing.Optional[builtins.bool] = None,
        ignore_failures: typing.Optional[builtins.bool] = None,
        include_role: typing.Optional[builtins.bool] = None,
        include_url: typing.Optional[builtins.bool] = None,
        print_log: typing.Optional[builtins.bool] = None,
        signal_resource: typing.Optional[aws_cdk.core.CfnResource] = None,
    ) -> None:
        '''Options for attaching a CloudFormationInit to a resource.

        :param instance_role: Instance role of the consuming instance or fleet.
        :param platform: OS Platform the init config will be used for.
        :param user_data: UserData to add commands to.
        :param config_sets: ConfigSet to activate. Default: ['default']
        :param embed_fingerprint: Whether to embed a hash into the userData. If ``true`` (the default), a hash of the config will be embedded into the UserData, so that if the config changes, the UserData changes and the instance will be replaced. If ``false``, no such hash will be embedded, and if the CloudFormation Init config changes nothing will happen to the running instance. Default: true
        :param ignore_failures: Don't fail the instance creation when cfn-init fails. You can use this to prevent CloudFormation from rolling back when instances fail to start up, to help in debugging. Default: false
        :param include_role: Include --role argument when running cfn-init and cfn-signal commands. This will be the IAM instance profile attached to the EC2 instance Default: false
        :param include_url: Include --url argument when running cfn-init and cfn-signal commands. This will be the cloudformation endpoint in the deployed region e.g. https://cloudformation.us-east-1.amazonaws.com Default: false
        :param print_log: Print the results of running cfn-init to the Instance System Log. By default, the output of running cfn-init is written to a log file on the instance. Set this to ``true`` to print it to the System Log (visible from the EC2 Console), ``false`` to not print it. (Be aware that the system log is refreshed at certain points in time of the instance life cycle, and successful execution may not always show up). Default: true
        :param signal_resource: When provided, signals this resource instead of the attached resource. You can use this to support signaling LaunchTemplate while attaching AutoScalingGroup Default: - if this property is undefined cfn-signal signals the attached resource

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            import aws_cdk.aws_iam as iam
            import aws_cdk.core as cdk
            
            # cfn_resource is of type CfnResource
            # role is of type Role
            # user_data is of type UserData
            
            attach_init_options = ec2.AttachInitOptions(
                instance_role=role,
                platform=ec2.OperatingSystemType.LINUX,
                user_data=user_data,
            
                # the properties below are optional
                config_sets=["configSets"],
                embed_fingerprint=False,
                ignore_failures=False,
                include_role=False,
                include_url=False,
                print_log=False,
                signal_resource=cfn_resource
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "instance_role": instance_role,
            "platform": platform,
            "user_data": user_data,
        }
        if config_sets is not None:
            self._values["config_sets"] = config_sets
        if embed_fingerprint is not None:
            self._values["embed_fingerprint"] = embed_fingerprint
        if ignore_failures is not None:
            self._values["ignore_failures"] = ignore_failures
        if include_role is not None:
            self._values["include_role"] = include_role
        if include_url is not None:
            self._values["include_url"] = include_url
        if print_log is not None:
            self._values["print_log"] = print_log
        if signal_resource is not None:
            self._values["signal_resource"] = signal_resource

    @builtins.property
    def instance_role(self) -> aws_cdk.aws_iam.IRole:
        '''Instance role of the consuming instance or fleet.'''
        result = self._values.get("instance_role")
        assert result is not None, "Required property 'instance_role' is missing"
        return typing.cast(aws_cdk.aws_iam.IRole, result)

    @builtins.property
    def platform(self) -> "OperatingSystemType":
        '''OS Platform the init config will be used for.'''
        result = self._values.get("platform")
        assert result is not None, "Required property 'platform' is missing"
        return typing.cast("OperatingSystemType", result)

    @builtins.property
    def user_data(self) -> "UserData":
        '''UserData to add commands to.'''
        result = self._values.get("user_data")
        assert result is not None, "Required property 'user_data' is missing"
        return typing.cast("UserData", result)

    @builtins.property
    def config_sets(self) -> typing.Optional[typing.List[builtins.str]]:
        '''ConfigSet to activate.

        :default: ['default']
        '''
        result = self._values.get("config_sets")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def embed_fingerprint(self) -> typing.Optional[builtins.bool]:
        '''Whether to embed a hash into the userData.

        If ``true`` (the default), a hash of the config will be embedded into the
        UserData, so that if the config changes, the UserData changes and
        the instance will be replaced.

        If ``false``, no such hash will be embedded, and if the CloudFormation Init
        config changes nothing will happen to the running instance.

        :default: true
        '''
        result = self._values.get("embed_fingerprint")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def ignore_failures(self) -> typing.Optional[builtins.bool]:
        '''Don't fail the instance creation when cfn-init fails.

        You can use this to prevent CloudFormation from rolling back when
        instances fail to start up, to help in debugging.

        :default: false
        '''
        result = self._values.get("ignore_failures")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def include_role(self) -> typing.Optional[builtins.bool]:
        '''Include --role argument when running cfn-init and cfn-signal commands.

        This will be the IAM instance profile attached to the EC2 instance

        :default: false
        '''
        result = self._values.get("include_role")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def include_url(self) -> typing.Optional[builtins.bool]:
        '''Include --url argument when running cfn-init and cfn-signal commands.

        This will be the cloudformation endpoint in the deployed region
        e.g. https://cloudformation.us-east-1.amazonaws.com

        :default: false
        '''
        result = self._values.get("include_url")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def print_log(self) -> typing.Optional[builtins.bool]:
        '''Print the results of running cfn-init to the Instance System Log.

        By default, the output of running cfn-init is written to a log file
        on the instance. Set this to ``true`` to print it to the System Log
        (visible from the EC2 Console), ``false`` to not print it.

        (Be aware that the system log is refreshed at certain points in
        time of the instance life cycle, and successful execution may
        not always show up).

        :default: true
        '''
        result = self._values.get("print_log")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def signal_resource(self) -> typing.Optional[aws_cdk.core.CfnResource]:
        '''When provided, signals this resource instead of the attached resource.

        You can use this to support signaling LaunchTemplate while attaching AutoScalingGroup

        :default: - if this property is undefined cfn-signal signals the attached resource
        '''
        result = self._values.get("signal_resource")
        return typing.cast(typing.Optional[aws_cdk.core.CfnResource], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AttachInitOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.BastionHostLinuxProps",
    jsii_struct_bases=[],
    name_mapping={
        "vpc": "vpc",
        "availability_zone": "availabilityZone",
        "block_devices": "blockDevices",
        "init": "init",
        "init_options": "initOptions",
        "instance_name": "instanceName",
        "instance_type": "instanceType",
        "machine_image": "machineImage",
        "security_group": "securityGroup",
        "subnet_selection": "subnetSelection",
    },
)
class BastionHostLinuxProps:
    def __init__(
        self,
        *,
        vpc: "IVpc",
        availability_zone: typing.Optional[builtins.str] = None,
        block_devices: typing.Optional[typing.Sequence["BlockDevice"]] = None,
        init: typing.Optional["CloudFormationInit"] = None,
        init_options: typing.Optional[ApplyCloudFormationInitOptions] = None,
        instance_name: typing.Optional[builtins.str] = None,
        instance_type: typing.Optional["InstanceType"] = None,
        machine_image: typing.Optional["IMachineImage"] = None,
        security_group: typing.Optional["ISecurityGroup"] = None,
        subnet_selection: typing.Optional["SubnetSelection"] = None,
    ) -> None:
        '''Properties of the bastion host.

        :param vpc: VPC to launch the instance in.
        :param availability_zone: In which AZ to place the instance within the VPC. Default: - Random zone.
        :param block_devices: Specifies how block devices are exposed to the instance. You can specify virtual devices and EBS volumes. Each instance that is launched has an associated root device volume, either an Amazon EBS volume or an instance store volume. You can use block device mappings to specify additional EBS volumes or instance store volumes to attach to an instance when it is launched. Default: - Uses the block device mapping of the AMI
        :param init: Apply the given CloudFormation Init configuration to the instance at startup. Default: - no CloudFormation init
        :param init_options: Use the given options for applying CloudFormation Init. Describes the configsets to use and the timeout to wait Default: - default options
        :param instance_name: The name of the instance. Default: 'BastionHost'
        :param instance_type: Type of instance to launch. Default: 't3.nano'
        :param machine_image: The machine image to use, assumed to have SSM Agent preinstalled. Default: - An Amazon Linux 2 image which is kept up-to-date automatically (the instance may be replaced on every deployment) and already has SSM Agent installed.
        :param security_group: Security Group to assign to this instance. Default: - create new security group with no inbound and all outbound traffic allowed
        :param subnet_selection: Select the subnets to run the bastion host in. Set this to PUBLIC if you need to connect to this instance via the internet and cannot use SSM. You have to allow port 22 manually by using the connections field Default: - private subnets of the supplied VPC

        :exampleMetadata: fixture=with-vpc infused

        Example::

            host = ec2.BastionHostLinux(self, "BastionHost",
                vpc=vpc,
                block_devices=[ec2.BlockDevice(
                    device_name="EBSBastionHost",
                    volume=ec2.BlockDeviceVolume.ebs(10,
                        encrypted=True
                    )
                )]
            )
        '''
        if isinstance(init_options, dict):
            init_options = ApplyCloudFormationInitOptions(**init_options)
        if isinstance(subnet_selection, dict):
            subnet_selection = SubnetSelection(**subnet_selection)
        self._values: typing.Dict[str, typing.Any] = {
            "vpc": vpc,
        }
        if availability_zone is not None:
            self._values["availability_zone"] = availability_zone
        if block_devices is not None:
            self._values["block_devices"] = block_devices
        if init is not None:
            self._values["init"] = init
        if init_options is not None:
            self._values["init_options"] = init_options
        if instance_name is not None:
            self._values["instance_name"] = instance_name
        if instance_type is not None:
            self._values["instance_type"] = instance_type
        if machine_image is not None:
            self._values["machine_image"] = machine_image
        if security_group is not None:
            self._values["security_group"] = security_group
        if subnet_selection is not None:
            self._values["subnet_selection"] = subnet_selection

    @builtins.property
    def vpc(self) -> "IVpc":
        '''VPC to launch the instance in.'''
        result = self._values.get("vpc")
        assert result is not None, "Required property 'vpc' is missing"
        return typing.cast("IVpc", result)

    @builtins.property
    def availability_zone(self) -> typing.Optional[builtins.str]:
        '''In which AZ to place the instance within the VPC.

        :default: - Random zone.
        '''
        result = self._values.get("availability_zone")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def block_devices(self) -> typing.Optional[typing.List["BlockDevice"]]:
        '''Specifies how block devices are exposed to the instance. You can specify virtual devices and EBS volumes.

        Each instance that is launched has an associated root device volume,
        either an Amazon EBS volume or an instance store volume.
        You can use block device mappings to specify additional EBS volumes or
        instance store volumes to attach to an instance when it is launched.

        :default: - Uses the block device mapping of the AMI

        :see: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html
        '''
        result = self._values.get("block_devices")
        return typing.cast(typing.Optional[typing.List["BlockDevice"]], result)

    @builtins.property
    def init(self) -> typing.Optional["CloudFormationInit"]:
        '''Apply the given CloudFormation Init configuration to the instance at startup.

        :default: - no CloudFormation init
        '''
        result = self._values.get("init")
        return typing.cast(typing.Optional["CloudFormationInit"], result)

    @builtins.property
    def init_options(self) -> typing.Optional[ApplyCloudFormationInitOptions]:
        '''Use the given options for applying CloudFormation Init.

        Describes the configsets to use and the timeout to wait

        :default: - default options
        '''
        result = self._values.get("init_options")
        return typing.cast(typing.Optional[ApplyCloudFormationInitOptions], result)

    @builtins.property
    def instance_name(self) -> typing.Optional[builtins.str]:
        '''The name of the instance.

        :default: 'BastionHost'
        '''
        result = self._values.get("instance_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def instance_type(self) -> typing.Optional["InstanceType"]:
        '''Type of instance to launch.

        :default: 't3.nano'
        '''
        result = self._values.get("instance_type")
        return typing.cast(typing.Optional["InstanceType"], result)

    @builtins.property
    def machine_image(self) -> typing.Optional["IMachineImage"]:
        '''The machine image to use, assumed to have SSM Agent preinstalled.

        :default:

        - An Amazon Linux 2 image which is kept up-to-date automatically (the instance
        may be replaced on every deployment) and already has SSM Agent installed.
        '''
        result = self._values.get("machine_image")
        return typing.cast(typing.Optional["IMachineImage"], result)

    @builtins.property
    def security_group(self) -> typing.Optional["ISecurityGroup"]:
        '''Security Group to assign to this instance.

        :default: - create new security group with no inbound and all outbound traffic allowed
        '''
        result = self._values.get("security_group")
        return typing.cast(typing.Optional["ISecurityGroup"], result)

    @builtins.property
    def subnet_selection(self) -> typing.Optional["SubnetSelection"]:
        '''Select the subnets to run the bastion host in.

        Set this to PUBLIC if you need to connect to this instance via the internet and cannot use SSM.
        You have to allow port 22 manually by using the connections field

        :default: - private subnets of the supplied VPC
        '''
        result = self._values.get("subnet_selection")
        return typing.cast(typing.Optional["SubnetSelection"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BastionHostLinuxProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.BlockDevice",
    jsii_struct_bases=[],
    name_mapping={
        "device_name": "deviceName",
        "volume": "volume",
        "mapping_enabled": "mappingEnabled",
    },
)
class BlockDevice:
    def __init__(
        self,
        *,
        device_name: builtins.str,
        volume: "BlockDeviceVolume",
        mapping_enabled: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Block device.

        :param device_name: The device name exposed to the EC2 instance. For example, a value like ``/dev/sdh``, ``xvdh``.
        :param volume: Defines the block device volume, to be either an Amazon EBS volume or an ephemeral instance store volume. For example, a value like ``BlockDeviceVolume.ebs(15)``, ``BlockDeviceVolume.ephemeral(0)``.
        :param mapping_enabled: If false, the device mapping will be suppressed. If set to false for the root device, the instance might fail the Amazon EC2 health check. Amazon EC2 Auto Scaling launches a replacement instance if the instance fails the health check. Default: true - device mapping is left untouched

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            # block_device_volume is of type BlockDeviceVolume
            
            block_device = ec2.BlockDevice(
                device_name="deviceName",
                volume=block_device_volume,
            
                # the properties below are optional
                mapping_enabled=False
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "device_name": device_name,
            "volume": volume,
        }
        if mapping_enabled is not None:
            self._values["mapping_enabled"] = mapping_enabled

    @builtins.property
    def device_name(self) -> builtins.str:
        '''The device name exposed to the EC2 instance.

        For example, a value like ``/dev/sdh``, ``xvdh``.

        :see: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/device_naming.html
        '''
        result = self._values.get("device_name")
        assert result is not None, "Required property 'device_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def volume(self) -> "BlockDeviceVolume":
        '''Defines the block device volume, to be either an Amazon EBS volume or an ephemeral instance store volume.

        For example, a value like ``BlockDeviceVolume.ebs(15)``, ``BlockDeviceVolume.ephemeral(0)``.
        '''
        result = self._values.get("volume")
        assert result is not None, "Required property 'volume' is missing"
        return typing.cast("BlockDeviceVolume", result)

    @builtins.property
    def mapping_enabled(self) -> typing.Optional[builtins.bool]:
        '''If false, the device mapping will be suppressed.

        If set to false for the root device, the instance might fail the Amazon EC2 health check.
        Amazon EC2 Auto Scaling launches a replacement instance if the instance fails the health check.

        :default: true - device mapping is left untouched
        '''
        result = self._values.get("mapping_enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BlockDevice(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class BlockDeviceVolume(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.BlockDeviceVolume",
):
    '''Describes a block device mapping for an EC2 instance or Auto Scaling group.

    :exampleMetadata: fixture=with-vpc infused

    Example::

        host = ec2.BastionHostLinux(self, "BastionHost",
            vpc=vpc,
            block_devices=[ec2.BlockDevice(
                device_name="EBSBastionHost",
                volume=ec2.BlockDeviceVolume.ebs(10,
                    encrypted=True
                )
            )]
        )
    '''

    def __init__(
        self,
        ebs_device: typing.Optional["EbsDeviceProps"] = None,
        virtual_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param ebs_device: EBS device info.
        :param virtual_name: Virtual device name.
        '''
        jsii.create(self.__class__, self, [ebs_device, virtual_name])

    @jsii.member(jsii_name="ebs") # type: ignore[misc]
    @builtins.classmethod
    def ebs(
        cls,
        volume_size: jsii.Number,
        *,
        encrypted: typing.Optional[builtins.bool] = None,
        delete_on_termination: typing.Optional[builtins.bool] = None,
        iops: typing.Optional[jsii.Number] = None,
        volume_type: typing.Optional["EbsDeviceVolumeType"] = None,
    ) -> "BlockDeviceVolume":
        '''Creates a new Elastic Block Storage device.

        :param volume_size: The volume size, in Gibibytes (GiB).
        :param encrypted: Specifies whether the EBS volume is encrypted. Encrypted EBS volumes can only be attached to instances that support Amazon EBS encryption Default: false
        :param delete_on_termination: Indicates whether to delete the volume when the instance is terminated. Default: - true for Amazon EC2 Auto Scaling, false otherwise (e.g. EBS)
        :param iops: The number of I/O operations per second (IOPS) to provision for the volume. Must only be set for {@link volumeType}: {@link EbsDeviceVolumeType.IO1} The maximum ratio of IOPS to volume size (in GiB) is 50:1, so for 5,000 provisioned IOPS, you need at least 100 GiB storage on the volume. Default: - none, required for {@link EbsDeviceVolumeType.IO1}
        :param volume_type: The EBS volume type. Default: {@link EbsDeviceVolumeType.GP2}
        '''
        options = EbsDeviceOptions(
            encrypted=encrypted,
            delete_on_termination=delete_on_termination,
            iops=iops,
            volume_type=volume_type,
        )

        return typing.cast("BlockDeviceVolume", jsii.sinvoke(cls, "ebs", [volume_size, options]))

    @jsii.member(jsii_name="ebsFromSnapshot") # type: ignore[misc]
    @builtins.classmethod
    def ebs_from_snapshot(
        cls,
        snapshot_id: builtins.str,
        *,
        volume_size: typing.Optional[jsii.Number] = None,
        delete_on_termination: typing.Optional[builtins.bool] = None,
        iops: typing.Optional[jsii.Number] = None,
        volume_type: typing.Optional["EbsDeviceVolumeType"] = None,
    ) -> "BlockDeviceVolume":
        '''Creates a new Elastic Block Storage device from an existing snapshot.

        :param snapshot_id: The snapshot ID of the volume to use.
        :param volume_size: The volume size, in Gibibytes (GiB). If you specify volumeSize, it must be equal or greater than the size of the snapshot. Default: - The snapshot size
        :param delete_on_termination: Indicates whether to delete the volume when the instance is terminated. Default: - true for Amazon EC2 Auto Scaling, false otherwise (e.g. EBS)
        :param iops: The number of I/O operations per second (IOPS) to provision for the volume. Must only be set for {@link volumeType}: {@link EbsDeviceVolumeType.IO1} The maximum ratio of IOPS to volume size (in GiB) is 50:1, so for 5,000 provisioned IOPS, you need at least 100 GiB storage on the volume. Default: - none, required for {@link EbsDeviceVolumeType.IO1}
        :param volume_type: The EBS volume type. Default: {@link EbsDeviceVolumeType.GP2}
        '''
        options = EbsDeviceSnapshotOptions(
            volume_size=volume_size,
            delete_on_termination=delete_on_termination,
            iops=iops,
            volume_type=volume_type,
        )

        return typing.cast("BlockDeviceVolume", jsii.sinvoke(cls, "ebsFromSnapshot", [snapshot_id, options]))

    @jsii.member(jsii_name="ephemeral") # type: ignore[misc]
    @builtins.classmethod
    def ephemeral(cls, volume_index: jsii.Number) -> "BlockDeviceVolume":
        '''Creates a virtual, ephemeral device.

        The name will be in the form ephemeral{volumeIndex}.

        :param volume_index: the volume index. Must be equal or greater than 0
        '''
        return typing.cast("BlockDeviceVolume", jsii.sinvoke(cls, "ephemeral", [volume_index]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ebsDevice")
    def ebs_device(self) -> typing.Optional["EbsDeviceProps"]:
        '''EBS device info.'''
        return typing.cast(typing.Optional["EbsDeviceProps"], jsii.get(self, "ebsDevice"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="virtualName")
    def virtual_name(self) -> typing.Optional[builtins.str]:
        '''Virtual device name.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "virtualName"))


@jsii.implements(aws_cdk.core.IInspectable)
class CfnCapacityReservation(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnCapacityReservation",
):
    '''A CloudFormation ``AWS::EC2::CapacityReservation``.

    Creates a new Capacity Reservation with the specified attributes. For more information, see `Capacity Reservations <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-capacity-reservations.html>`_ in the *Amazon EC2 User Guide* .

    :cloudformationResource: AWS::EC2::CapacityReservation
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ec2 as ec2
        
        cfn_capacity_reservation = ec2.CfnCapacityReservation(self, "MyCfnCapacityReservation",
            availability_zone="availabilityZone",
            instance_count=123,
            instance_platform="instancePlatform",
            instance_type="instanceType",
        
            # the properties below are optional
            ebs_optimized=False,
            end_date="endDate",
            end_date_type="endDateType",
            ephemeral_storage=False,
            instance_match_criteria="instanceMatchCriteria",
            out_post_arn="outPostArn",
            placement_group_arn="placementGroupArn",
            tag_specifications=[ec2.CfnCapacityReservation.TagSpecificationProperty(
                resource_type="resourceType",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )],
            tenancy="tenancy"
        )
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        availability_zone: builtins.str,
        instance_count: jsii.Number,
        instance_platform: builtins.str,
        instance_type: builtins.str,
        ebs_optimized: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        end_date: typing.Optional[builtins.str] = None,
        end_date_type: typing.Optional[builtins.str] = None,
        ephemeral_storage: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        instance_match_criteria: typing.Optional[builtins.str] = None,
        out_post_arn: typing.Optional[builtins.str] = None,
        placement_group_arn: typing.Optional[builtins.str] = None,
        tag_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnCapacityReservation.TagSpecificationProperty"]]]] = None,
        tenancy: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::CapacityReservation``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param availability_zone: The Availability Zone in which to create the Capacity Reservation.
        :param instance_count: The number of instances for which to reserve capacity. Valid range: 1 - 1000
        :param instance_platform: The type of operating system for which to reserve capacity.
        :param instance_type: The instance type for which to reserve capacity. For more information, see `Instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html>`_ in the *Amazon EC2 User Guide* .
        :param ebs_optimized: Indicates whether the Capacity Reservation supports EBS-optimized instances. This optimization provides dedicated throughput to Amazon EBS and an optimized configuration stack to provide optimal I/O performance. This optimization isn't available with all instance types. Additional usage charges apply when using an EBS- optimized instance.
        :param end_date: The date and time at which the Capacity Reservation expires. When a Capacity Reservation expires, the reserved capacity is released and you can no longer launch instances into it. The Capacity Reservation's state changes to ``expired`` when it reaches its end date and time. You must provide an ``EndDate`` value if ``EndDateType`` is ``limited`` . Omit ``EndDate`` if ``EndDateType`` is ``unlimited`` . If the ``EndDateType`` is ``limited`` , the Capacity Reservation is cancelled within an hour from the specified time. For example, if you specify 5/31/2019, 13:30:55, the Capacity Reservation is guaranteed to end between 13:30:55 and 14:30:55 on 5/31/2019.
        :param end_date_type: Indicates the way in which the Capacity Reservation ends. A Capacity Reservation can have one of the following end types: - ``unlimited`` - The Capacity Reservation remains active until you explicitly cancel it. Do not provide an ``EndDate`` if the ``EndDateType`` is ``unlimited`` . - ``limited`` - The Capacity Reservation expires automatically at a specified date and time. You must provide an ``EndDate`` value if the ``EndDateType`` value is ``limited`` .
        :param ephemeral_storage: Indicates whether the Capacity Reservation supports instances with temporary, block-level storage.
        :param instance_match_criteria: Indicates the type of instance launches that the Capacity Reservation accepts. The options include:. - ``open`` - The Capacity Reservation automatically matches all instances that have matching attributes (instance type, platform, and Availability Zone). Instances that have matching attributes run in the Capacity Reservation automatically without specifying any additional parameters. - ``targeted`` - The Capacity Reservation only accepts instances that have matching attributes (instance type, platform, and Availability Zone), and explicitly target the Capacity Reservation. This ensures that only permitted instances can use the reserved capacity. Default: ``open``
        :param out_post_arn: The Amazon Resource Name (ARN) of the Outpost on which the Capacity Reservation was created.
        :param placement_group_arn: ``AWS::EC2::CapacityReservation.PlacementGroupArn``.
        :param tag_specifications: The tags to apply to the Capacity Reservation during launch.
        :param tenancy: Indicates the tenancy of the Capacity Reservation. A Capacity Reservation can have one of the following tenancy settings:. - ``default`` - The Capacity Reservation is created on hardware that is shared with other AWS accounts . - ``dedicated`` - The Capacity Reservation is created on single-tenant hardware that is dedicated to a single AWS account .
        '''
        props = CfnCapacityReservationProps(
            availability_zone=availability_zone,
            instance_count=instance_count,
            instance_platform=instance_platform,
            instance_type=instance_type,
            ebs_optimized=ebs_optimized,
            end_date=end_date,
            end_date_type=end_date_type,
            ephemeral_storage=ephemeral_storage,
            instance_match_criteria=instance_match_criteria,
            out_post_arn=out_post_arn,
            placement_group_arn=placement_group_arn,
            tag_specifications=tag_specifications,
            tenancy=tenancy,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrAvailabilityZone")
    def attr_availability_zone(self) -> builtins.str:
        '''Returns the Availability Zone in which the capacity is reserved.

        For example: ``us-east-1a`` .

        :cloudformationAttribute: AvailabilityZone
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrAvailabilityZone"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrAvailableInstanceCount")
    def attr_available_instance_count(self) -> jsii.Number:
        '''Returns the remaining capacity, which indicates the number of instances that can be launched in the Capacity Reservation.

        For example: ``9`` .

        :cloudformationAttribute: AvailableInstanceCount
        '''
        return typing.cast(jsii.Number, jsii.get(self, "attrAvailableInstanceCount"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrInstanceType")
    def attr_instance_type(self) -> builtins.str:
        '''Returns the type of instance for which the capacity is reserved.

        For example: ``m4.large`` .

        :cloudformationAttribute: InstanceType
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrInstanceType"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrTenancy")
    def attr_tenancy(self) -> builtins.str:
        '''Returns the tenancy of the Capacity Reservation.

        For example: ``dedicated`` .

        :cloudformationAttribute: Tenancy
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrTenancy"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrTotalInstanceCount")
    def attr_total_instance_count(self) -> jsii.Number:
        '''Returns the total number of instances for which the Capacity Reservation reserves capacity.

        For example: ``15`` .

        :cloudformationAttribute: TotalInstanceCount
        '''
        return typing.cast(jsii.Number, jsii.get(self, "attrTotalInstanceCount"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="availabilityZone")
    def availability_zone(self) -> builtins.str:
        '''The Availability Zone in which to create the Capacity Reservation.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-availabilityzone
        '''
        return typing.cast(builtins.str, jsii.get(self, "availabilityZone"))

    @availability_zone.setter
    def availability_zone(self, value: builtins.str) -> None:
        jsii.set(self, "availabilityZone", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceCount")
    def instance_count(self) -> jsii.Number:
        '''The number of instances for which to reserve capacity.

        Valid range: 1 - 1000

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-instancecount
        '''
        return typing.cast(jsii.Number, jsii.get(self, "instanceCount"))

    @instance_count.setter
    def instance_count(self, value: jsii.Number) -> None:
        jsii.set(self, "instanceCount", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instancePlatform")
    def instance_platform(self) -> builtins.str:
        '''The type of operating system for which to reserve capacity.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-instanceplatform
        '''
        return typing.cast(builtins.str, jsii.get(self, "instancePlatform"))

    @instance_platform.setter
    def instance_platform(self, value: builtins.str) -> None:
        jsii.set(self, "instancePlatform", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceType")
    def instance_type(self) -> builtins.str:
        '''The instance type for which to reserve capacity.

        For more information, see `Instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html>`_ in the *Amazon EC2 User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-instancetype
        '''
        return typing.cast(builtins.str, jsii.get(self, "instanceType"))

    @instance_type.setter
    def instance_type(self, value: builtins.str) -> None:
        jsii.set(self, "instanceType", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ebsOptimized")
    def ebs_optimized(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''Indicates whether the Capacity Reservation supports EBS-optimized instances.

        This optimization provides dedicated throughput to Amazon EBS and an optimized configuration stack to provide optimal I/O performance. This optimization isn't available with all instance types. Additional usage charges apply when using an EBS- optimized instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-ebsoptimized
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "ebsOptimized"))

    @ebs_optimized.setter
    def ebs_optimized(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "ebsOptimized", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="endDate")
    def end_date(self) -> typing.Optional[builtins.str]:
        '''The date and time at which the Capacity Reservation expires.

        When a Capacity Reservation expires, the reserved capacity is released and you can no longer launch instances into it. The Capacity Reservation's state changes to ``expired`` when it reaches its end date and time.

        You must provide an ``EndDate`` value if ``EndDateType`` is ``limited`` . Omit ``EndDate`` if ``EndDateType`` is ``unlimited`` .

        If the ``EndDateType`` is ``limited`` , the Capacity Reservation is cancelled within an hour from the specified time. For example, if you specify 5/31/2019, 13:30:55, the Capacity Reservation is guaranteed to end between 13:30:55 and 14:30:55 on 5/31/2019.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-enddate
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "endDate"))

    @end_date.setter
    def end_date(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "endDate", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="endDateType")
    def end_date_type(self) -> typing.Optional[builtins.str]:
        '''Indicates the way in which the Capacity Reservation ends.

        A Capacity Reservation can have one of the following end types:

        - ``unlimited`` - The Capacity Reservation remains active until you explicitly cancel it. Do not provide an ``EndDate`` if the ``EndDateType`` is ``unlimited`` .
        - ``limited`` - The Capacity Reservation expires automatically at a specified date and time. You must provide an ``EndDate`` value if the ``EndDateType`` value is ``limited`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-enddatetype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "endDateType"))

    @end_date_type.setter
    def end_date_type(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "endDateType", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ephemeralStorage")
    def ephemeral_storage(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''Indicates whether the Capacity Reservation supports instances with temporary, block-level storage.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-ephemeralstorage
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "ephemeralStorage"))

    @ephemeral_storage.setter
    def ephemeral_storage(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "ephemeralStorage", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceMatchCriteria")
    def instance_match_criteria(self) -> typing.Optional[builtins.str]:
        '''Indicates the type of instance launches that the Capacity Reservation accepts. The options include:.

        - ``open`` - The Capacity Reservation automatically matches all instances that have matching attributes (instance type, platform, and Availability Zone). Instances that have matching attributes run in the Capacity Reservation automatically without specifying any additional parameters.
        - ``targeted`` - The Capacity Reservation only accepts instances that have matching attributes (instance type, platform, and Availability Zone), and explicitly target the Capacity Reservation. This ensures that only permitted instances can use the reserved capacity.

        Default: ``open``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-instancematchcriteria
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "instanceMatchCriteria"))

    @instance_match_criteria.setter
    def instance_match_criteria(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "instanceMatchCriteria", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="outPostArn")
    def out_post_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the Outpost on which the Capacity Reservation was created.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-outpostarn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "outPostArn"))

    @out_post_arn.setter
    def out_post_arn(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "outPostArn", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="placementGroupArn")
    def placement_group_arn(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::CapacityReservation.PlacementGroupArn``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-placementgrouparn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "placementGroupArn"))

    @placement_group_arn.setter
    def placement_group_arn(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "placementGroupArn", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tagSpecifications")
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnCapacityReservation.TagSpecificationProperty"]]]]:
        '''The tags to apply to the Capacity Reservation during launch.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-tagspecifications
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnCapacityReservation.TagSpecificationProperty"]]]], jsii.get(self, "tagSpecifications"))

    @tag_specifications.setter
    def tag_specifications(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnCapacityReservation.TagSpecificationProperty"]]]],
    ) -> None:
        jsii.set(self, "tagSpecifications", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tenancy")
    def tenancy(self) -> typing.Optional[builtins.str]:
        '''Indicates the tenancy of the Capacity Reservation. A Capacity Reservation can have one of the following tenancy settings:.

        - ``default`` - The Capacity Reservation is created on hardware that is shared with other AWS accounts .
        - ``dedicated`` - The Capacity Reservation is created on single-tenant hardware that is dedicated to a single AWS account .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-tenancy
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "tenancy"))

    @tenancy.setter
    def tenancy(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "tenancy", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnCapacityReservation.TagSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"resource_type": "resourceType", "tags": "tags"},
    )
    class TagSpecificationProperty:
        def __init__(
            self,
            *,
            resource_type: typing.Optional[builtins.str] = None,
            tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
        ) -> None:
            '''An array of key-value pairs to apply to this resource.

            For more information, see `Tag <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-resource-tags.html>`_ .

            :param resource_type: The type of resource to tag. Specify ``capacity-reservation`` .
            :param tags: The tags to apply to the resource.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservation-tagspecification.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                tag_specification_property = ec2.CfnCapacityReservation.TagSpecificationProperty(
                    resource_type="resourceType",
                    tags=[CfnTag(
                        key="key",
                        value="value"
                    )]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if resource_type is not None:
                self._values["resource_type"] = resource_type
            if tags is not None:
                self._values["tags"] = tags

        @builtins.property
        def resource_type(self) -> typing.Optional[builtins.str]:
            '''The type of resource to tag.

            Specify ``capacity-reservation`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservation-tagspecification.html#cfn-ec2-capacityreservation-tagspecification-resourcetype
            '''
            result = self._values.get("resource_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
            '''The tags to apply to the resource.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservation-tagspecification.html#cfn-ec2-capacityreservation-tagspecification-tags
            '''
            result = self._values.get("tags")
            return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TagSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnCapacityReservationFleet(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnCapacityReservationFleet",
):
    '''A CloudFormation ``AWS::EC2::CapacityReservationFleet``.

    Creates a new Capacity Reservation Fleet with the specified attributes. For more information, see `Capacity Reservation Fleets <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/cr-fleets.html>`_ in the *Amazon EC2 User Guide* .

    :cloudformationResource: AWS::EC2::CapacityReservationFleet
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ec2 as ec2
        
        cfn_capacity_reservation_fleet = ec2.CfnCapacityReservationFleet(self, "MyCfnCapacityReservationFleet",
            allocation_strategy="allocationStrategy",
            end_date="endDate",
            instance_match_criteria="instanceMatchCriteria",
            instance_type_specifications=[ec2.CfnCapacityReservationFleet.InstanceTypeSpecificationProperty(
                availability_zone="availabilityZone",
                availability_zone_id="availabilityZoneId",
                ebs_optimized=False,
                instance_platform="instancePlatform",
                instance_type="instanceType",
                priority=123,
                weight=123
            )],
            no_remove_end_date=False,
            remove_end_date=False,
            tag_specifications=[ec2.CfnCapacityReservationFleet.TagSpecificationProperty(
                resource_type="resourceType",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )],
            tenancy="tenancy",
            total_target_capacity=123
        )
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        allocation_strategy: typing.Optional[builtins.str] = None,
        end_date: typing.Optional[builtins.str] = None,
        instance_match_criteria: typing.Optional[builtins.str] = None,
        instance_type_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnCapacityReservationFleet.InstanceTypeSpecificationProperty"]]]] = None,
        no_remove_end_date: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        remove_end_date: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        tag_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnCapacityReservationFleet.TagSpecificationProperty"]]]] = None,
        tenancy: typing.Optional[builtins.str] = None,
        total_target_capacity: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::CapacityReservationFleet``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param allocation_strategy: The strategy used by the Capacity Reservation Fleet to determine which of the specified instance types to use. Currently, only the ``prioritized`` allocation strategy is supported. For more information, see `Allocation strategy <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#allocation-strategy>`_ in the Amazon EC2 User Guide. Valid values: ``prioritized``
        :param end_date: The date and time at which the Capacity Reservation Fleet expires. When the Capacity Reservation Fleet expires, its state changes to ``expired`` and all of the Capacity Reservations in the Fleet expire. The Capacity Reservation Fleet expires within an hour after the specified time. For example, if you specify ``5/31/2019`` , ``13:30:55`` , the Capacity Reservation Fleet is guaranteed to expire between ``13:30:55`` and ``14:30:55`` on ``5/31/2019`` .
        :param instance_match_criteria: Indicates the type of instance launches that the Capacity Reservation Fleet accepts. All Capacity Reservations in the Fleet inherit this instance matching criteria. Currently, Capacity Reservation Fleets support ``open`` instance matching criteria only. This means that instances that have matching attributes (instance type, platform, and Availability Zone) run in the Capacity Reservations automatically. Instances do not need to explicitly target a Capacity Reservation Fleet to use its reserved capacity.
        :param instance_type_specifications: Information about the instance types for which to reserve the capacity.
        :param no_remove_end_date: ``AWS::EC2::CapacityReservationFleet.NoRemoveEndDate``.
        :param remove_end_date: ``AWS::EC2::CapacityReservationFleet.RemoveEndDate``.
        :param tag_specifications: The tags to assign to the Capacity Reservation Fleet. The tags are automatically assigned to the Capacity Reservations in the Fleet.
        :param tenancy: Indicates the tenancy of the Capacity Reservation Fleet. All Capacity Reservations in the Fleet inherit this tenancy. The Capacity Reservation Fleet can have one of the following tenancy settings: - ``default`` - The Capacity Reservation Fleet is created on hardware that is shared with other AWS accounts . - ``dedicated`` - The Capacity Reservations are created on single-tenant hardware that is dedicated to a single AWS account .
        :param total_target_capacity: The total number of capacity units to be reserved by the Capacity Reservation Fleet. This value, together with the instance type weights that you assign to each instance type used by the Fleet determine the number of instances for which the Fleet reserves capacity. Both values are based on units that make sense for your workload. For more information, see `Total target capacity <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#target-capacity>`_ in the Amazon EC2 User Guide.
        '''
        props = CfnCapacityReservationFleetProps(
            allocation_strategy=allocation_strategy,
            end_date=end_date,
            instance_match_criteria=instance_match_criteria,
            instance_type_specifications=instance_type_specifications,
            no_remove_end_date=no_remove_end_date,
            remove_end_date=remove_end_date,
            tag_specifications=tag_specifications,
            tenancy=tenancy,
            total_target_capacity=total_target_capacity,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrCapacityReservationFleetId")
    def attr_capacity_reservation_fleet_id(self) -> builtins.str:
        '''The ID of the Capacity Reservation Fleet.

        :cloudformationAttribute: CapacityReservationFleetId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCapacityReservationFleetId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="allocationStrategy")
    def allocation_strategy(self) -> typing.Optional[builtins.str]:
        '''The strategy used by the Capacity Reservation Fleet to determine which of the specified instance types to use.

        Currently, only the ``prioritized`` allocation strategy is supported. For more information, see `Allocation strategy <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#allocation-strategy>`_ in the Amazon EC2 User Guide.

        Valid values: ``prioritized``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-allocationstrategy
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "allocationStrategy"))

    @allocation_strategy.setter
    def allocation_strategy(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "allocationStrategy", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="endDate")
    def end_date(self) -> typing.Optional[builtins.str]:
        '''The date and time at which the Capacity Reservation Fleet expires.

        When the Capacity Reservation Fleet expires, its state changes to ``expired`` and all of the Capacity Reservations in the Fleet expire.

        The Capacity Reservation Fleet expires within an hour after the specified time. For example, if you specify ``5/31/2019`` , ``13:30:55`` , the Capacity Reservation Fleet is guaranteed to expire between ``13:30:55`` and ``14:30:55`` on ``5/31/2019`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-enddate
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "endDate"))

    @end_date.setter
    def end_date(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "endDate", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceMatchCriteria")
    def instance_match_criteria(self) -> typing.Optional[builtins.str]:
        '''Indicates the type of instance launches that the Capacity Reservation Fleet accepts.

        All Capacity Reservations in the Fleet inherit this instance matching criteria.

        Currently, Capacity Reservation Fleets support ``open`` instance matching criteria only. This means that instances that have matching attributes (instance type, platform, and Availability Zone) run in the Capacity Reservations automatically. Instances do not need to explicitly target a Capacity Reservation Fleet to use its reserved capacity.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-instancematchcriteria
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "instanceMatchCriteria"))

    @instance_match_criteria.setter
    def instance_match_criteria(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "instanceMatchCriteria", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceTypeSpecifications")
    def instance_type_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnCapacityReservationFleet.InstanceTypeSpecificationProperty"]]]]:
        '''Information about the instance types for which to reserve the capacity.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-instancetypespecifications
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnCapacityReservationFleet.InstanceTypeSpecificationProperty"]]]], jsii.get(self, "instanceTypeSpecifications"))

    @instance_type_specifications.setter
    def instance_type_specifications(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnCapacityReservationFleet.InstanceTypeSpecificationProperty"]]]],
    ) -> None:
        jsii.set(self, "instanceTypeSpecifications", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="noRemoveEndDate")
    def no_remove_end_date(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::CapacityReservationFleet.NoRemoveEndDate``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-noremoveenddate
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "noRemoveEndDate"))

    @no_remove_end_date.setter
    def no_remove_end_date(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "noRemoveEndDate", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="removeEndDate")
    def remove_end_date(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::CapacityReservationFleet.RemoveEndDate``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-removeenddate
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "removeEndDate"))

    @remove_end_date.setter
    def remove_end_date(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "removeEndDate", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tagSpecifications")
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnCapacityReservationFleet.TagSpecificationProperty"]]]]:
        '''The tags to assign to the Capacity Reservation Fleet.

        The tags are automatically assigned to the Capacity Reservations in the Fleet.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-tagspecifications
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnCapacityReservationFleet.TagSpecificationProperty"]]]], jsii.get(self, "tagSpecifications"))

    @tag_specifications.setter
    def tag_specifications(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnCapacityReservationFleet.TagSpecificationProperty"]]]],
    ) -> None:
        jsii.set(self, "tagSpecifications", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tenancy")
    def tenancy(self) -> typing.Optional[builtins.str]:
        '''Indicates the tenancy of the Capacity Reservation Fleet.

        All Capacity Reservations in the Fleet inherit this tenancy. The Capacity Reservation Fleet can have one of the following tenancy settings:

        - ``default`` - The Capacity Reservation Fleet is created on hardware that is shared with other AWS accounts .
        - ``dedicated`` - The Capacity Reservations are created on single-tenant hardware that is dedicated to a single AWS account .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-tenancy
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "tenancy"))

    @tenancy.setter
    def tenancy(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "tenancy", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="totalTargetCapacity")
    def total_target_capacity(self) -> typing.Optional[jsii.Number]:
        '''The total number of capacity units to be reserved by the Capacity Reservation Fleet.

        This value, together with the instance type weights that you assign to each instance type used by the Fleet determine the number of instances for which the Fleet reserves capacity. Both values are based on units that make sense for your workload. For more information, see `Total target capacity <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#target-capacity>`_ in the Amazon EC2 User Guide.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-totaltargetcapacity
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "totalTargetCapacity"))

    @total_target_capacity.setter
    def total_target_capacity(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "totalTargetCapacity", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnCapacityReservationFleet.InstanceTypeSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "availability_zone": "availabilityZone",
            "availability_zone_id": "availabilityZoneId",
            "ebs_optimized": "ebsOptimized",
            "instance_platform": "instancePlatform",
            "instance_type": "instanceType",
            "priority": "priority",
            "weight": "weight",
        },
    )
    class InstanceTypeSpecificationProperty:
        def __init__(
            self,
            *,
            availability_zone: typing.Optional[builtins.str] = None,
            availability_zone_id: typing.Optional[builtins.str] = None,
            ebs_optimized: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            instance_platform: typing.Optional[builtins.str] = None,
            instance_type: typing.Optional[builtins.str] = None,
            priority: typing.Optional[jsii.Number] = None,
            weight: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''Specifies information about an instance type to use in a Capacity Reservation Fleet.

            ``InstanceTypeSpecification`` is a property of the `AWS::EC2::CapacityReservationFleet <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html>`_ resource.

            :param availability_zone: The Availability Zone in which the Capacity Reservation Fleet reserves the capacity. A Capacity Reservation Fleet can't span Availability Zones. All instance type specifications that you specify for the Fleet must use the same Availability Zone.
            :param availability_zone_id: The ID of the Availability Zone in which the Capacity Reservation Fleet reserves the capacity. A Capacity Reservation Fleet can't span Availability Zones. All instance type specifications that you specify for the Fleet must use the same Availability Zone.
            :param ebs_optimized: Indicates whether the Capacity Reservation Fleet supports EBS-optimized instances types. This optimization provides dedicated throughput to Amazon EBS and an optimized configuration stack to provide optimal I/O performance. This optimization isn't available with all instance types. Additional usage charges apply when using EBS-optimized instance types.
            :param instance_platform: The type of operating system for which the Capacity Reservation Fleet reserves capacity.
            :param instance_type: The instance type for which the Capacity Reservation Fleet reserves capacity.
            :param priority: The priority to assign to the instance type. This value is used to determine which of the instance types specified for the Fleet should be prioritized for use. A lower value indicates a high priority. For more information, see `Instance type priority <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#instance-priority>`_ in the Amazon EC2 User Guide.
            :param weight: The number of capacity units provided by the specified instance type. This value, together with the total target capacity that you specify for the Fleet determine the number of instances for which the Fleet reserves capacity. Both values are based on units that make sense for your workload. For more information, see `Total target capacity <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#target-capacity>`_ in the Amazon EC2 User Guide. Valid Range: Minimum value of ``0.001`` . Maximum value of ``99.999`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservationfleet-instancetypespecification.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                instance_type_specification_property = ec2.CfnCapacityReservationFleet.InstanceTypeSpecificationProperty(
                    availability_zone="availabilityZone",
                    availability_zone_id="availabilityZoneId",
                    ebs_optimized=False,
                    instance_platform="instancePlatform",
                    instance_type="instanceType",
                    priority=123,
                    weight=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if availability_zone is not None:
                self._values["availability_zone"] = availability_zone
            if availability_zone_id is not None:
                self._values["availability_zone_id"] = availability_zone_id
            if ebs_optimized is not None:
                self._values["ebs_optimized"] = ebs_optimized
            if instance_platform is not None:
                self._values["instance_platform"] = instance_platform
            if instance_type is not None:
                self._values["instance_type"] = instance_type
            if priority is not None:
                self._values["priority"] = priority
            if weight is not None:
                self._values["weight"] = weight

        @builtins.property
        def availability_zone(self) -> typing.Optional[builtins.str]:
            '''The Availability Zone in which the Capacity Reservation Fleet reserves the capacity.

            A Capacity Reservation Fleet can't span Availability Zones. All instance type specifications that you specify for the Fleet must use the same Availability Zone.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservationfleet-instancetypespecification.html#cfn-ec2-capacityreservationfleet-instancetypespecification-availabilityzone
            '''
            result = self._values.get("availability_zone")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def availability_zone_id(self) -> typing.Optional[builtins.str]:
            '''The ID of the Availability Zone in which the Capacity Reservation Fleet reserves the capacity.

            A Capacity Reservation Fleet can't span Availability Zones. All instance type specifications that you specify for the Fleet must use the same Availability Zone.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservationfleet-instancetypespecification.html#cfn-ec2-capacityreservationfleet-instancetypespecification-availabilityzoneid
            '''
            result = self._values.get("availability_zone_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ebs_optimized(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''Indicates whether the Capacity Reservation Fleet supports EBS-optimized instances types.

            This optimization provides dedicated throughput to Amazon EBS and an optimized configuration stack to provide optimal I/O performance. This optimization isn't available with all instance types. Additional usage charges apply when using EBS-optimized instance types.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservationfleet-instancetypespecification.html#cfn-ec2-capacityreservationfleet-instancetypespecification-ebsoptimized
            '''
            result = self._values.get("ebs_optimized")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def instance_platform(self) -> typing.Optional[builtins.str]:
            '''The type of operating system for which the Capacity Reservation Fleet reserves capacity.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservationfleet-instancetypespecification.html#cfn-ec2-capacityreservationfleet-instancetypespecification-instanceplatform
            '''
            result = self._values.get("instance_platform")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def instance_type(self) -> typing.Optional[builtins.str]:
            '''The instance type for which the Capacity Reservation Fleet reserves capacity.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservationfleet-instancetypespecification.html#cfn-ec2-capacityreservationfleet-instancetypespecification-instancetype
            '''
            result = self._values.get("instance_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def priority(self) -> typing.Optional[jsii.Number]:
            '''The priority to assign to the instance type.

            This value is used to determine which of the instance types specified for the Fleet should be prioritized for use. A lower value indicates a high priority. For more information, see `Instance type priority <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#instance-priority>`_ in the Amazon EC2 User Guide.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservationfleet-instancetypespecification.html#cfn-ec2-capacityreservationfleet-instancetypespecification-priority
            '''
            result = self._values.get("priority")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def weight(self) -> typing.Optional[jsii.Number]:
            '''The number of capacity units provided by the specified instance type.

            This value, together with the total target capacity that you specify for the Fleet determine the number of instances for which the Fleet reserves capacity. Both values are based on units that make sense for your workload. For more information, see `Total target capacity <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#target-capacity>`_ in the Amazon EC2 User Guide.

            Valid Range: Minimum value of ``0.001`` . Maximum value of ``99.999`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservationfleet-instancetypespecification.html#cfn-ec2-capacityreservationfleet-instancetypespecification-weight
            '''
            result = self._values.get("weight")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstanceTypeSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnCapacityReservationFleet.TagSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"resource_type": "resourceType", "tags": "tags"},
    )
    class TagSpecificationProperty:
        def __init__(
            self,
            *,
            resource_type: typing.Optional[builtins.str] = None,
            tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
        ) -> None:
            '''The tags to apply to a resource when the resource is being created.

            :param resource_type: The type of resource to tag on creation. Specify ``capacity-reservation-fleet`` . To tag a resource after it has been created, see `CreateTags <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html>`_ .
            :param tags: The tags to apply to the resource.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservationfleet-tagspecification.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                tag_specification_property = ec2.CfnCapacityReservationFleet.TagSpecificationProperty(
                    resource_type="resourceType",
                    tags=[CfnTag(
                        key="key",
                        value="value"
                    )]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if resource_type is not None:
                self._values["resource_type"] = resource_type
            if tags is not None:
                self._values["tags"] = tags

        @builtins.property
        def resource_type(self) -> typing.Optional[builtins.str]:
            '''The type of resource to tag on creation. Specify ``capacity-reservation-fleet`` .

            To tag a resource after it has been created, see `CreateTags <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservationfleet-tagspecification.html#cfn-ec2-capacityreservationfleet-tagspecification-resourcetype
            '''
            result = self._values.get("resource_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
            '''The tags to apply to the resource.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservationfleet-tagspecification.html#cfn-ec2-capacityreservationfleet-tagspecification-tags
            '''
            result = self._values.get("tags")
            return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TagSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnCapacityReservationFleetProps",
    jsii_struct_bases=[],
    name_mapping={
        "allocation_strategy": "allocationStrategy",
        "end_date": "endDate",
        "instance_match_criteria": "instanceMatchCriteria",
        "instance_type_specifications": "instanceTypeSpecifications",
        "no_remove_end_date": "noRemoveEndDate",
        "remove_end_date": "removeEndDate",
        "tag_specifications": "tagSpecifications",
        "tenancy": "tenancy",
        "total_target_capacity": "totalTargetCapacity",
    },
)
class CfnCapacityReservationFleetProps:
    def __init__(
        self,
        *,
        allocation_strategy: typing.Optional[builtins.str] = None,
        end_date: typing.Optional[builtins.str] = None,
        instance_match_criteria: typing.Optional[builtins.str] = None,
        instance_type_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnCapacityReservationFleet.InstanceTypeSpecificationProperty]]]] = None,
        no_remove_end_date: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        remove_end_date: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        tag_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnCapacityReservationFleet.TagSpecificationProperty]]]] = None,
        tenancy: typing.Optional[builtins.str] = None,
        total_target_capacity: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties for defining a ``CfnCapacityReservationFleet``.

        :param allocation_strategy: The strategy used by the Capacity Reservation Fleet to determine which of the specified instance types to use. Currently, only the ``prioritized`` allocation strategy is supported. For more information, see `Allocation strategy <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#allocation-strategy>`_ in the Amazon EC2 User Guide. Valid values: ``prioritized``
        :param end_date: The date and time at which the Capacity Reservation Fleet expires. When the Capacity Reservation Fleet expires, its state changes to ``expired`` and all of the Capacity Reservations in the Fleet expire. The Capacity Reservation Fleet expires within an hour after the specified time. For example, if you specify ``5/31/2019`` , ``13:30:55`` , the Capacity Reservation Fleet is guaranteed to expire between ``13:30:55`` and ``14:30:55`` on ``5/31/2019`` .
        :param instance_match_criteria: Indicates the type of instance launches that the Capacity Reservation Fleet accepts. All Capacity Reservations in the Fleet inherit this instance matching criteria. Currently, Capacity Reservation Fleets support ``open`` instance matching criteria only. This means that instances that have matching attributes (instance type, platform, and Availability Zone) run in the Capacity Reservations automatically. Instances do not need to explicitly target a Capacity Reservation Fleet to use its reserved capacity.
        :param instance_type_specifications: Information about the instance types for which to reserve the capacity.
        :param no_remove_end_date: ``AWS::EC2::CapacityReservationFleet.NoRemoveEndDate``.
        :param remove_end_date: ``AWS::EC2::CapacityReservationFleet.RemoveEndDate``.
        :param tag_specifications: The tags to assign to the Capacity Reservation Fleet. The tags are automatically assigned to the Capacity Reservations in the Fleet.
        :param tenancy: Indicates the tenancy of the Capacity Reservation Fleet. All Capacity Reservations in the Fleet inherit this tenancy. The Capacity Reservation Fleet can have one of the following tenancy settings: - ``default`` - The Capacity Reservation Fleet is created on hardware that is shared with other AWS accounts . - ``dedicated`` - The Capacity Reservations are created on single-tenant hardware that is dedicated to a single AWS account .
        :param total_target_capacity: The total number of capacity units to be reserved by the Capacity Reservation Fleet. This value, together with the instance type weights that you assign to each instance type used by the Fleet determine the number of instances for which the Fleet reserves capacity. Both values are based on units that make sense for your workload. For more information, see `Total target capacity <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#target-capacity>`_ in the Amazon EC2 User Guide.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            cfn_capacity_reservation_fleet_props = ec2.CfnCapacityReservationFleetProps(
                allocation_strategy="allocationStrategy",
                end_date="endDate",
                instance_match_criteria="instanceMatchCriteria",
                instance_type_specifications=[ec2.CfnCapacityReservationFleet.InstanceTypeSpecificationProperty(
                    availability_zone="availabilityZone",
                    availability_zone_id="availabilityZoneId",
                    ebs_optimized=False,
                    instance_platform="instancePlatform",
                    instance_type="instanceType",
                    priority=123,
                    weight=123
                )],
                no_remove_end_date=False,
                remove_end_date=False,
                tag_specifications=[ec2.CfnCapacityReservationFleet.TagSpecificationProperty(
                    resource_type="resourceType",
                    tags=[CfnTag(
                        key="key",
                        value="value"
                    )]
                )],
                tenancy="tenancy",
                total_target_capacity=123
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if allocation_strategy is not None:
            self._values["allocation_strategy"] = allocation_strategy
        if end_date is not None:
            self._values["end_date"] = end_date
        if instance_match_criteria is not None:
            self._values["instance_match_criteria"] = instance_match_criteria
        if instance_type_specifications is not None:
            self._values["instance_type_specifications"] = instance_type_specifications
        if no_remove_end_date is not None:
            self._values["no_remove_end_date"] = no_remove_end_date
        if remove_end_date is not None:
            self._values["remove_end_date"] = remove_end_date
        if tag_specifications is not None:
            self._values["tag_specifications"] = tag_specifications
        if tenancy is not None:
            self._values["tenancy"] = tenancy
        if total_target_capacity is not None:
            self._values["total_target_capacity"] = total_target_capacity

    @builtins.property
    def allocation_strategy(self) -> typing.Optional[builtins.str]:
        '''The strategy used by the Capacity Reservation Fleet to determine which of the specified instance types to use.

        Currently, only the ``prioritized`` allocation strategy is supported. For more information, see `Allocation strategy <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#allocation-strategy>`_ in the Amazon EC2 User Guide.

        Valid values: ``prioritized``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-allocationstrategy
        '''
        result = self._values.get("allocation_strategy")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def end_date(self) -> typing.Optional[builtins.str]:
        '''The date and time at which the Capacity Reservation Fleet expires.

        When the Capacity Reservation Fleet expires, its state changes to ``expired`` and all of the Capacity Reservations in the Fleet expire.

        The Capacity Reservation Fleet expires within an hour after the specified time. For example, if you specify ``5/31/2019`` , ``13:30:55`` , the Capacity Reservation Fleet is guaranteed to expire between ``13:30:55`` and ``14:30:55`` on ``5/31/2019`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-enddate
        '''
        result = self._values.get("end_date")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def instance_match_criteria(self) -> typing.Optional[builtins.str]:
        '''Indicates the type of instance launches that the Capacity Reservation Fleet accepts.

        All Capacity Reservations in the Fleet inherit this instance matching criteria.

        Currently, Capacity Reservation Fleets support ``open`` instance matching criteria only. This means that instances that have matching attributes (instance type, platform, and Availability Zone) run in the Capacity Reservations automatically. Instances do not need to explicitly target a Capacity Reservation Fleet to use its reserved capacity.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-instancematchcriteria
        '''
        result = self._values.get("instance_match_criteria")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def instance_type_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnCapacityReservationFleet.InstanceTypeSpecificationProperty]]]]:
        '''Information about the instance types for which to reserve the capacity.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-instancetypespecifications
        '''
        result = self._values.get("instance_type_specifications")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnCapacityReservationFleet.InstanceTypeSpecificationProperty]]]], result)

    @builtins.property
    def no_remove_end_date(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::CapacityReservationFleet.NoRemoveEndDate``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-noremoveenddate
        '''
        result = self._values.get("no_remove_end_date")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def remove_end_date(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::CapacityReservationFleet.RemoveEndDate``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-removeenddate
        '''
        result = self._values.get("remove_end_date")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnCapacityReservationFleet.TagSpecificationProperty]]]]:
        '''The tags to assign to the Capacity Reservation Fleet.

        The tags are automatically assigned to the Capacity Reservations in the Fleet.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-tagspecifications
        '''
        result = self._values.get("tag_specifications")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnCapacityReservationFleet.TagSpecificationProperty]]]], result)

    @builtins.property
    def tenancy(self) -> typing.Optional[builtins.str]:
        '''Indicates the tenancy of the Capacity Reservation Fleet.

        All Capacity Reservations in the Fleet inherit this tenancy. The Capacity Reservation Fleet can have one of the following tenancy settings:

        - ``default`` - The Capacity Reservation Fleet is created on hardware that is shared with other AWS accounts .
        - ``dedicated`` - The Capacity Reservations are created on single-tenant hardware that is dedicated to a single AWS account .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-tenancy
        '''
        result = self._values.get("tenancy")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def total_target_capacity(self) -> typing.Optional[jsii.Number]:
        '''The total number of capacity units to be reserved by the Capacity Reservation Fleet.

        This value, together with the instance type weights that you assign to each instance type used by the Fleet determine the number of instances for which the Fleet reserves capacity. Both values are based on units that make sense for your workload. For more information, see `Total target capacity <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#target-capacity>`_ in the Amazon EC2 User Guide.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-totaltargetcapacity
        '''
        result = self._values.get("total_target_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCapacityReservationFleetProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnCapacityReservationProps",
    jsii_struct_bases=[],
    name_mapping={
        "availability_zone": "availabilityZone",
        "instance_count": "instanceCount",
        "instance_platform": "instancePlatform",
        "instance_type": "instanceType",
        "ebs_optimized": "ebsOptimized",
        "end_date": "endDate",
        "end_date_type": "endDateType",
        "ephemeral_storage": "ephemeralStorage",
        "instance_match_criteria": "instanceMatchCriteria",
        "out_post_arn": "outPostArn",
        "placement_group_arn": "placementGroupArn",
        "tag_specifications": "tagSpecifications",
        "tenancy": "tenancy",
    },
)
class CfnCapacityReservationProps:
    def __init__(
        self,
        *,
        availability_zone: builtins.str,
        instance_count: jsii.Number,
        instance_platform: builtins.str,
        instance_type: builtins.str,
        ebs_optimized: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        end_date: typing.Optional[builtins.str] = None,
        end_date_type: typing.Optional[builtins.str] = None,
        ephemeral_storage: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        instance_match_criteria: typing.Optional[builtins.str] = None,
        out_post_arn: typing.Optional[builtins.str] = None,
        placement_group_arn: typing.Optional[builtins.str] = None,
        tag_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnCapacityReservation.TagSpecificationProperty]]]] = None,
        tenancy: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnCapacityReservation``.

        :param availability_zone: The Availability Zone in which to create the Capacity Reservation.
        :param instance_count: The number of instances for which to reserve capacity. Valid range: 1 - 1000
        :param instance_platform: The type of operating system for which to reserve capacity.
        :param instance_type: The instance type for which to reserve capacity. For more information, see `Instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html>`_ in the *Amazon EC2 User Guide* .
        :param ebs_optimized: Indicates whether the Capacity Reservation supports EBS-optimized instances. This optimization provides dedicated throughput to Amazon EBS and an optimized configuration stack to provide optimal I/O performance. This optimization isn't available with all instance types. Additional usage charges apply when using an EBS- optimized instance.
        :param end_date: The date and time at which the Capacity Reservation expires. When a Capacity Reservation expires, the reserved capacity is released and you can no longer launch instances into it. The Capacity Reservation's state changes to ``expired`` when it reaches its end date and time. You must provide an ``EndDate`` value if ``EndDateType`` is ``limited`` . Omit ``EndDate`` if ``EndDateType`` is ``unlimited`` . If the ``EndDateType`` is ``limited`` , the Capacity Reservation is cancelled within an hour from the specified time. For example, if you specify 5/31/2019, 13:30:55, the Capacity Reservation is guaranteed to end between 13:30:55 and 14:30:55 on 5/31/2019.
        :param end_date_type: Indicates the way in which the Capacity Reservation ends. A Capacity Reservation can have one of the following end types: - ``unlimited`` - The Capacity Reservation remains active until you explicitly cancel it. Do not provide an ``EndDate`` if the ``EndDateType`` is ``unlimited`` . - ``limited`` - The Capacity Reservation expires automatically at a specified date and time. You must provide an ``EndDate`` value if the ``EndDateType`` value is ``limited`` .
        :param ephemeral_storage: Indicates whether the Capacity Reservation supports instances with temporary, block-level storage.
        :param instance_match_criteria: Indicates the type of instance launches that the Capacity Reservation accepts. The options include:. - ``open`` - The Capacity Reservation automatically matches all instances that have matching attributes (instance type, platform, and Availability Zone). Instances that have matching attributes run in the Capacity Reservation automatically without specifying any additional parameters. - ``targeted`` - The Capacity Reservation only accepts instances that have matching attributes (instance type, platform, and Availability Zone), and explicitly target the Capacity Reservation. This ensures that only permitted instances can use the reserved capacity. Default: ``open``
        :param out_post_arn: The Amazon Resource Name (ARN) of the Outpost on which the Capacity Reservation was created.
        :param placement_group_arn: ``AWS::EC2::CapacityReservation.PlacementGroupArn``.
        :param tag_specifications: The tags to apply to the Capacity Reservation during launch.
        :param tenancy: Indicates the tenancy of the Capacity Reservation. A Capacity Reservation can have one of the following tenancy settings:. - ``default`` - The Capacity Reservation is created on hardware that is shared with other AWS accounts . - ``dedicated`` - The Capacity Reservation is created on single-tenant hardware that is dedicated to a single AWS account .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            cfn_capacity_reservation_props = ec2.CfnCapacityReservationProps(
                availability_zone="availabilityZone",
                instance_count=123,
                instance_platform="instancePlatform",
                instance_type="instanceType",
            
                # the properties below are optional
                ebs_optimized=False,
                end_date="endDate",
                end_date_type="endDateType",
                ephemeral_storage=False,
                instance_match_criteria="instanceMatchCriteria",
                out_post_arn="outPostArn",
                placement_group_arn="placementGroupArn",
                tag_specifications=[ec2.CfnCapacityReservation.TagSpecificationProperty(
                    resource_type="resourceType",
                    tags=[CfnTag(
                        key="key",
                        value="value"
                    )]
                )],
                tenancy="tenancy"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "availability_zone": availability_zone,
            "instance_count": instance_count,
            "instance_platform": instance_platform,
            "instance_type": instance_type,
        }
        if ebs_optimized is not None:
            self._values["ebs_optimized"] = ebs_optimized
        if end_date is not None:
            self._values["end_date"] = end_date
        if end_date_type is not None:
            self._values["end_date_type"] = end_date_type
        if ephemeral_storage is not None:
            self._values["ephemeral_storage"] = ephemeral_storage
        if instance_match_criteria is not None:
            self._values["instance_match_criteria"] = instance_match_criteria
        if out_post_arn is not None:
            self._values["out_post_arn"] = out_post_arn
        if placement_group_arn is not None:
            self._values["placement_group_arn"] = placement_group_arn
        if tag_specifications is not None:
            self._values["tag_specifications"] = tag_specifications
        if tenancy is not None:
            self._values["tenancy"] = tenancy

    @builtins.property
    def availability_zone(self) -> builtins.str:
        '''The Availability Zone in which to create the Capacity Reservation.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-availabilityzone
        '''
        result = self._values.get("availability_zone")
        assert result is not None, "Required property 'availability_zone' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def instance_count(self) -> jsii.Number:
        '''The number of instances for which to reserve capacity.

        Valid range: 1 - 1000

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-instancecount
        '''
        result = self._values.get("instance_count")
        assert result is not None, "Required property 'instance_count' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def instance_platform(self) -> builtins.str:
        '''The type of operating system for which to reserve capacity.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-instanceplatform
        '''
        result = self._values.get("instance_platform")
        assert result is not None, "Required property 'instance_platform' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def instance_type(self) -> builtins.str:
        '''The instance type for which to reserve capacity.

        For more information, see `Instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html>`_ in the *Amazon EC2 User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-instancetype
        '''
        result = self._values.get("instance_type")
        assert result is not None, "Required property 'instance_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def ebs_optimized(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''Indicates whether the Capacity Reservation supports EBS-optimized instances.

        This optimization provides dedicated throughput to Amazon EBS and an optimized configuration stack to provide optimal I/O performance. This optimization isn't available with all instance types. Additional usage charges apply when using an EBS- optimized instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-ebsoptimized
        '''
        result = self._values.get("ebs_optimized")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def end_date(self) -> typing.Optional[builtins.str]:
        '''The date and time at which the Capacity Reservation expires.

        When a Capacity Reservation expires, the reserved capacity is released and you can no longer launch instances into it. The Capacity Reservation's state changes to ``expired`` when it reaches its end date and time.

        You must provide an ``EndDate`` value if ``EndDateType`` is ``limited`` . Omit ``EndDate`` if ``EndDateType`` is ``unlimited`` .

        If the ``EndDateType`` is ``limited`` , the Capacity Reservation is cancelled within an hour from the specified time. For example, if you specify 5/31/2019, 13:30:55, the Capacity Reservation is guaranteed to end between 13:30:55 and 14:30:55 on 5/31/2019.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-enddate
        '''
        result = self._values.get("end_date")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def end_date_type(self) -> typing.Optional[builtins.str]:
        '''Indicates the way in which the Capacity Reservation ends.

        A Capacity Reservation can have one of the following end types:

        - ``unlimited`` - The Capacity Reservation remains active until you explicitly cancel it. Do not provide an ``EndDate`` if the ``EndDateType`` is ``unlimited`` .
        - ``limited`` - The Capacity Reservation expires automatically at a specified date and time. You must provide an ``EndDate`` value if the ``EndDateType`` value is ``limited`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-enddatetype
        '''
        result = self._values.get("end_date_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ephemeral_storage(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''Indicates whether the Capacity Reservation supports instances with temporary, block-level storage.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-ephemeralstorage
        '''
        result = self._values.get("ephemeral_storage")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def instance_match_criteria(self) -> typing.Optional[builtins.str]:
        '''Indicates the type of instance launches that the Capacity Reservation accepts. The options include:.

        - ``open`` - The Capacity Reservation automatically matches all instances that have matching attributes (instance type, platform, and Availability Zone). Instances that have matching attributes run in the Capacity Reservation automatically without specifying any additional parameters.
        - ``targeted`` - The Capacity Reservation only accepts instances that have matching attributes (instance type, platform, and Availability Zone), and explicitly target the Capacity Reservation. This ensures that only permitted instances can use the reserved capacity.

        Default: ``open``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-instancematchcriteria
        '''
        result = self._values.get("instance_match_criteria")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def out_post_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the Outpost on which the Capacity Reservation was created.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-outpostarn
        '''
        result = self._values.get("out_post_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def placement_group_arn(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::CapacityReservation.PlacementGroupArn``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-placementgrouparn
        '''
        result = self._values.get("placement_group_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnCapacityReservation.TagSpecificationProperty]]]]:
        '''The tags to apply to the Capacity Reservation during launch.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-tagspecifications
        '''
        result = self._values.get("tag_specifications")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnCapacityReservation.TagSpecificationProperty]]]], result)

    @builtins.property
    def tenancy(self) -> typing.Optional[builtins.str]:
        '''Indicates the tenancy of the Capacity Reservation. A Capacity Reservation can have one of the following tenancy settings:.

        - ``default`` - The Capacity Reservation is created on hardware that is shared with other AWS accounts .
        - ``dedicated`` - The Capacity Reservation is created on single-tenant hardware that is dedicated to a single AWS account .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-tenancy
        '''
        result = self._values.get("tenancy")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCapacityReservationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnCarrierGateway(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnCarrierGateway",
):
    '''A CloudFormation ``AWS::EC2::CarrierGateway``.

    Creates a carrier gateway. For more information about carrier gateways, see `Carrier gateways <https://docs.aws.amazon.com/wavelength/latest/developerguide/how-wavelengths-work.html#wavelength-carrier-gateway>`_ in the *AWS Wavelength Developer Guide* .

    :cloudformationResource: AWS::EC2::CarrierGateway
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-carriergateway.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ec2 as ec2
        
        cfn_carrier_gateway = ec2.CfnCarrierGateway(self, "MyCfnCarrierGateway",
            vpc_id="vpcId",
        
            # the properties below are optional
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        vpc_id: builtins.str,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::CarrierGateway``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param vpc_id: The ID of the VPC associated with the carrier gateway.
        :param tags: The tags assigned to the carrier gateway.
        '''
        props = CfnCarrierGatewayProps(vpc_id=vpc_id, tags=tags)

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrCarrierGatewayId")
    def attr_carrier_gateway_id(self) -> builtins.str:
        '''The ID of the carrier gateway.

        :cloudformationAttribute: CarrierGatewayId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCarrierGatewayId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrOwnerId")
    def attr_owner_id(self) -> builtins.str:
        '''The AWS account ID of the owner of the carrier gateway.

        :cloudformationAttribute: OwnerId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrOwnerId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrState")
    def attr_state(self) -> builtins.str:
        '''The state of the carrier gateway.

        :cloudformationAttribute: State
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrState"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''The tags assigned to the carrier gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-carriergateway.html#cfn-ec2-carriergateway-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="vpcId")
    def vpc_id(self) -> builtins.str:
        '''The ID of the VPC associated with the carrier gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-carriergateway.html#cfn-ec2-carriergateway-vpcid
        '''
        return typing.cast(builtins.str, jsii.get(self, "vpcId"))

    @vpc_id.setter
    def vpc_id(self, value: builtins.str) -> None:
        jsii.set(self, "vpcId", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnCarrierGatewayProps",
    jsii_struct_bases=[],
    name_mapping={"vpc_id": "vpcId", "tags": "tags"},
)
class CfnCarrierGatewayProps:
    def __init__(
        self,
        *,
        vpc_id: builtins.str,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Properties for defining a ``CfnCarrierGateway``.

        :param vpc_id: The ID of the VPC associated with the carrier gateway.
        :param tags: The tags assigned to the carrier gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-carriergateway.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            cfn_carrier_gateway_props = ec2.CfnCarrierGatewayProps(
                vpc_id="vpcId",
            
                # the properties below are optional
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "vpc_id": vpc_id,
        }
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def vpc_id(self) -> builtins.str:
        '''The ID of the VPC associated with the carrier gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-carriergateway.html#cfn-ec2-carriergateway-vpcid
        '''
        result = self._values.get("vpc_id")
        assert result is not None, "Required property 'vpc_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''The tags assigned to the carrier gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-carriergateway.html#cfn-ec2-carriergateway-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCarrierGatewayProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnClientVpnAuthorizationRule(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnClientVpnAuthorizationRule",
):
    '''A CloudFormation ``AWS::EC2::ClientVpnAuthorizationRule``.

    Specifies an ingress authorization rule to add to a Client VPN endpoint. Ingress authorization rules act as firewall rules that grant access to networks. You must configure ingress authorization rules to enable clients to access resources in AWS or on-premises networks.

    :cloudformationResource: AWS::EC2::ClientVpnAuthorizationRule
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ec2 as ec2
        
        cfn_client_vpn_authorization_rule = ec2.CfnClientVpnAuthorizationRule(self, "MyCfnClientVpnAuthorizationRule",
            client_vpn_endpoint_id="clientVpnEndpointId",
            target_network_cidr="targetNetworkCidr",
        
            # the properties below are optional
            access_group_id="accessGroupId",
            authorize_all_groups=False,
            description="description"
        )
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        client_vpn_endpoint_id: builtins.str,
        target_network_cidr: builtins.str,
        access_group_id: typing.Optional[builtins.str] = None,
        authorize_all_groups: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::ClientVpnAuthorizationRule``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param client_vpn_endpoint_id: The ID of the Client VPN endpoint.
        :param target_network_cidr: The IPv4 address range, in CIDR notation, of the network for which access is being authorized.
        :param access_group_id: The ID of the group to grant access to, for example, the Active Directory group or identity provider (IdP) group. Required if ``AuthorizeAllGroups`` is ``false`` or not specified.
        :param authorize_all_groups: Indicates whether to grant access to all clients. Specify ``true`` to grant all clients who successfully establish a VPN connection access to the network. Must be set to ``true`` if ``AccessGroupId`` is not specified.
        :param description: A brief description of the authorization rule.
        '''
        props = CfnClientVpnAuthorizationRuleProps(
            client_vpn_endpoint_id=client_vpn_endpoint_id,
            target_network_cidr=target_network_cidr,
            access_group_id=access_group_id,
            authorize_all_groups=authorize_all_groups,
            description=description,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="clientVpnEndpointId")
    def client_vpn_endpoint_id(self) -> builtins.str:
        '''The ID of the Client VPN endpoint.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-clientvpnendpointid
        '''
        return typing.cast(builtins.str, jsii.get(self, "clientVpnEndpointId"))

    @client_vpn_endpoint_id.setter
    def client_vpn_endpoint_id(self, value: builtins.str) -> None:
        jsii.set(self, "clientVpnEndpointId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="targetNetworkCidr")
    def target_network_cidr(self) -> builtins.str:
        '''The IPv4 address range, in CIDR notation, of the network for which access is being authorized.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-targetnetworkcidr
        '''
        return typing.cast(builtins.str, jsii.get(self, "targetNetworkCidr"))

    @target_network_cidr.setter
    def target_network_cidr(self, value: builtins.str) -> None:
        jsii.set(self, "targetNetworkCidr", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="accessGroupId")
    def access_group_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the group to grant access to, for example, the Active Directory group or identity provider (IdP) group.

        Required if ``AuthorizeAllGroups`` is ``false`` or not specified.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-accessgroupid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "accessGroupId"))

    @access_group_id.setter
    def access_group_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "accessGroupId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="authorizeAllGroups")
    def authorize_all_groups(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''Indicates whether to grant access to all clients.

        Specify ``true`` to grant all clients who successfully establish a VPN connection access to the network. Must be set to ``true`` if ``AccessGroupId`` is not specified.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-authorizeallgroups
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "authorizeAllGroups"))

    @authorize_all_groups.setter
    def authorize_all_groups(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "authorizeAllGroups", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A brief description of the authorization rule.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "description", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnClientVpnAuthorizationRuleProps",
    jsii_struct_bases=[],
    name_mapping={
        "client_vpn_endpoint_id": "clientVpnEndpointId",
        "target_network_cidr": "targetNetworkCidr",
        "access_group_id": "accessGroupId",
        "authorize_all_groups": "authorizeAllGroups",
        "description": "description",
    },
)
class CfnClientVpnAuthorizationRuleProps:
    def __init__(
        self,
        *,
        client_vpn_endpoint_id: builtins.str,
        target_network_cidr: builtins.str,
        access_group_id: typing.Optional[builtins.str] = None,
        authorize_all_groups: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnClientVpnAuthorizationRule``.

        :param client_vpn_endpoint_id: The ID of the Client VPN endpoint.
        :param target_network_cidr: The IPv4 address range, in CIDR notation, of the network for which access is being authorized.
        :param access_group_id: The ID of the group to grant access to, for example, the Active Directory group or identity provider (IdP) group. Required if ``AuthorizeAllGroups`` is ``false`` or not specified.
        :param authorize_all_groups: Indicates whether to grant access to all clients. Specify ``true`` to grant all clients who successfully establish a VPN connection access to the network. Must be set to ``true`` if ``AccessGroupId`` is not specified.
        :param description: A brief description of the authorization rule.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            cfn_client_vpn_authorization_rule_props = ec2.CfnClientVpnAuthorizationRuleProps(
                client_vpn_endpoint_id="clientVpnEndpointId",
                target_network_cidr="targetNetworkCidr",
            
                # the properties below are optional
                access_group_id="accessGroupId",
                authorize_all_groups=False,
                description="description"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "client_vpn_endpoint_id": client_vpn_endpoint_id,
            "target_network_cidr": target_network_cidr,
        }
        if access_group_id is not None:
            self._values["access_group_id"] = access_group_id
        if authorize_all_groups is not None:
            self._values["authorize_all_groups"] = authorize_all_groups
        if description is not None:
            self._values["description"] = description

    @builtins.property
    def client_vpn_endpoint_id(self) -> builtins.str:
        '''The ID of the Client VPN endpoint.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-clientvpnendpointid
        '''
        result = self._values.get("client_vpn_endpoint_id")
        assert result is not None, "Required property 'client_vpn_endpoint_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def target_network_cidr(self) -> builtins.str:
        '''The IPv4 address range, in CIDR notation, of the network for which access is being authorized.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-targetnetworkcidr
        '''
        result = self._values.get("target_network_cidr")
        assert result is not None, "Required property 'target_network_cidr' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def access_group_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the group to grant access to, for example, the Active Directory group or identity provider (IdP) group.

        Required if ``AuthorizeAllGroups`` is ``false`` or not specified.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-accessgroupid
        '''
        result = self._values.get("access_group_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def authorize_all_groups(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''Indicates whether to grant access to all clients.

        Specify ``true`` to grant all clients who successfully establish a VPN connection access to the network. Must be set to ``true`` if ``AccessGroupId`` is not specified.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-authorizeallgroups
        '''
        result = self._values.get("authorize_all_groups")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A brief description of the authorization rule.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnClientVpnAuthorizationRuleProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnClientVpnEndpoint(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnClientVpnEndpoint",
):
    '''A CloudFormation ``AWS::EC2::ClientVpnEndpoint``.

    Specifies a Client VPN endpoint. A Client VPN endpoint is the resource you create and configure to enable and manage client VPN sessions. It is the destination endpoint at which all client VPN sessions are terminated.

    :cloudformationResource: AWS::EC2::ClientVpnEndpoint
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ec2 as ec2
        
        cfn_client_vpn_endpoint = ec2.CfnClientVpnEndpoint(self, "MyCfnClientVpnEndpoint",
            authentication_options=[ec2.CfnClientVpnEndpoint.ClientAuthenticationRequestProperty(
                type="type",
        
                # the properties below are optional
                active_directory=ec2.CfnClientVpnEndpoint.DirectoryServiceAuthenticationRequestProperty(
                    directory_id="directoryId"
                ),
                federated_authentication=ec2.CfnClientVpnEndpoint.FederatedAuthenticationRequestProperty(
                    saml_provider_arn="samlProviderArn",
        
                    # the properties below are optional
                    self_service_saml_provider_arn="selfServiceSamlProviderArn"
                ),
                mutual_authentication=ec2.CfnClientVpnEndpoint.CertificateAuthenticationRequestProperty(
                    client_root_certificate_chain_arn="clientRootCertificateChainArn"
                )
            )],
            client_cidr_block="clientCidrBlock",
            connection_log_options=ec2.CfnClientVpnEndpoint.ConnectionLogOptionsProperty(
                enabled=False,
        
                # the properties below are optional
                cloudwatch_log_group="cloudwatchLogGroup",
                cloudwatch_log_stream="cloudwatchLogStream"
            ),
            server_certificate_arn="serverCertificateArn",
        
            # the properties below are optional
            client_connect_options=ec2.CfnClientVpnEndpoint.ClientConnectOptionsProperty(
                enabled=False,
        
                # the properties below are optional
                lambda_function_arn="lambdaFunctionArn"
            ),
            description="description",
            dns_servers=["dnsServers"],
            security_group_ids=["securityGroupIds"],
            self_service_portal="selfServicePortal",
            split_tunnel=False,
            tag_specifications=[ec2.CfnClientVpnEndpoint.TagSpecificationProperty(
                resource_type="resourceType",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )],
            transport_protocol="transportProtocol",
            vpc_id="vpcId",
            vpn_port=123
        )
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        authentication_options: typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union["CfnClientVpnEndpoint.ClientAuthenticationRequestProperty", aws_cdk.core.IResolvable]]],
        client_cidr_block: builtins.str,
        connection_log_options: typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.ConnectionLogOptionsProperty"],
        server_certificate_arn: builtins.str,
        client_connect_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.ClientConnectOptionsProperty"]] = None,
        description: typing.Optional[builtins.str] = None,
        dns_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
        self_service_portal: typing.Optional[builtins.str] = None,
        split_tunnel: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        tag_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.TagSpecificationProperty"]]]] = None,
        transport_protocol: typing.Optional[builtins.str] = None,
        vpc_id: typing.Optional[builtins.str] = None,
        vpn_port: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::ClientVpnEndpoint``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param authentication_options: Information about the authentication method to be used to authenticate clients.
        :param client_cidr_block: The IPv4 address range, in CIDR notation, from which to assign client IP addresses. The address range cannot overlap with the local CIDR of the VPC in which the associated subnet is located, or the routes that you add manually. The address range cannot be changed after the Client VPN endpoint has been created. The CIDR block should be /22 or greater.
        :param connection_log_options: Information about the client connection logging options. If you enable client connection logging, data about client connections is sent to a Cloudwatch Logs log stream. The following information is logged: - Client connection requests - Client connection results (successful and unsuccessful) - Reasons for unsuccessful client connection requests - Client connection termination time
        :param server_certificate_arn: The ARN of the server certificate. For more information, see the `AWS Certificate Manager User Guide <https://docs.aws.amazon.com/acm/latest/userguide/>`_ .
        :param client_connect_options: The options for managing connection authorization for new client connections.
        :param description: A brief description of the Client VPN endpoint.
        :param dns_servers: Information about the DNS servers to be used for DNS resolution. A Client VPN endpoint can have up to two DNS servers. If no DNS server is specified, the DNS address configured on the device is used for the DNS server.
        :param security_group_ids: The IDs of one or more security groups to apply to the target network. You must also specify the ID of the VPC that contains the security groups.
        :param self_service_portal: Specify whether to enable the self-service portal for the Client VPN endpoint. Default Value: ``enabled``
        :param split_tunnel: Indicates whether split-tunnel is enabled on the AWS Client VPN endpoint. By default, split-tunnel on a VPN endpoint is disabled. For information about split-tunnel VPN endpoints, see `Split-tunnel AWS Client VPN endpoint <https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/split-tunnel-vpn.html>`_ in the *AWS Client VPN Administrator Guide* .
        :param tag_specifications: The tags to apply to the Client VPN endpoint during creation.
        :param transport_protocol: The transport protocol to be used by the VPN session. Default value: ``udp``
        :param vpc_id: The ID of the VPC to associate with the Client VPN endpoint. If no security group IDs are specified in the request, the default security group for the VPC is applied.
        :param vpn_port: The port number to assign to the Client VPN endpoint for TCP and UDP traffic. Valid Values: ``443`` | ``1194`` Default Value: ``443``
        '''
        props = CfnClientVpnEndpointProps(
            authentication_options=authentication_options,
            client_cidr_block=client_cidr_block,
            connection_log_options=connection_log_options,
            server_certificate_arn=server_certificate_arn,
            client_connect_options=client_connect_options,
            description=description,
            dns_servers=dns_servers,
            security_group_ids=security_group_ids,
            self_service_portal=self_service_portal,
            split_tunnel=split_tunnel,
            tag_specifications=tag_specifications,
            transport_protocol=transport_protocol,
            vpc_id=vpc_id,
            vpn_port=vpn_port,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="authenticationOptions")
    def authentication_options(
        self,
    ) -> typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union["CfnClientVpnEndpoint.ClientAuthenticationRequestProperty", aws_cdk.core.IResolvable]]]:
        '''Information about the authentication method to be used to authenticate clients.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-authenticationoptions
        '''
        return typing.cast(typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union["CfnClientVpnEndpoint.ClientAuthenticationRequestProperty", aws_cdk.core.IResolvable]]], jsii.get(self, "authenticationOptions"))

    @authentication_options.setter
    def authentication_options(
        self,
        value: typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union["CfnClientVpnEndpoint.ClientAuthenticationRequestProperty", aws_cdk.core.IResolvable]]],
    ) -> None:
        jsii.set(self, "authenticationOptions", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="clientCidrBlock")
    def client_cidr_block(self) -> builtins.str:
        '''The IPv4 address range, in CIDR notation, from which to assign client IP addresses.

        The address range cannot overlap with the local CIDR of the VPC in which the associated subnet is located, or the routes that you add manually. The address range cannot be changed after the Client VPN endpoint has been created. The CIDR block should be /22 or greater.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-clientcidrblock
        '''
        return typing.cast(builtins.str, jsii.get(self, "clientCidrBlock"))

    @client_cidr_block.setter
    def client_cidr_block(self, value: builtins.str) -> None:
        jsii.set(self, "clientCidrBlock", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="connectionLogOptions")
    def connection_log_options(
        self,
    ) -> typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.ConnectionLogOptionsProperty"]:
        '''Information about the client connection logging options.

        If you enable client connection logging, data about client connections is sent to a Cloudwatch Logs log stream. The following information is logged:

        - Client connection requests
        - Client connection results (successful and unsuccessful)
        - Reasons for unsuccessful client connection requests
        - Client connection termination time

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-connectionlogoptions
        '''
        return typing.cast(typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.ConnectionLogOptionsProperty"], jsii.get(self, "connectionLogOptions"))

    @connection_log_options.setter
    def connection_log_options(
        self,
        value: typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.ConnectionLogOptionsProperty"],
    ) -> None:
        jsii.set(self, "connectionLogOptions", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="serverCertificateArn")
    def server_certificate_arn(self) -> builtins.str:
        '''The ARN of the server certificate.

        For more information, see the `AWS Certificate Manager User Guide <https://docs.aws.amazon.com/acm/latest/userguide/>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-servercertificatearn
        '''
        return typing.cast(builtins.str, jsii.get(self, "serverCertificateArn"))

    @server_certificate_arn.setter
    def server_certificate_arn(self, value: builtins.str) -> None:
        jsii.set(self, "serverCertificateArn", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="clientConnectOptions")
    def client_connect_options(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.ClientConnectOptionsProperty"]]:
        '''The options for managing connection authorization for new client connections.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-clientconnectoptions
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.ClientConnectOptionsProperty"]], jsii.get(self, "clientConnectOptions"))

    @client_connect_options.setter
    def client_connect_options(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.ClientConnectOptionsProperty"]],
    ) -> None:
        jsii.set(self, "clientConnectOptions", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A brief description of the Client VPN endpoint.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "description", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="dnsServers")
    def dns_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Information about the DNS servers to be used for DNS resolution.

        A Client VPN endpoint can have up to two DNS servers. If no DNS server is specified, the DNS address configured on the device is used for the DNS server.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-dnsservers
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "dnsServers"))

    @dns_servers.setter
    def dns_servers(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        jsii.set(self, "dnsServers", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="securityGroupIds")
    def security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The IDs of one or more security groups to apply to the target network.

        You must also specify the ID of the VPC that contains the security groups.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-securitygroupids
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "securityGroupIds"))

    @security_group_ids.setter
    def security_group_ids(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        jsii.set(self, "securityGroupIds", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="selfServicePortal")
    def self_service_portal(self) -> typing.Optional[builtins.str]:
        '''Specify whether to enable the self-service portal for the Client VPN endpoint.

        Default Value: ``enabled``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-selfserviceportal
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "selfServicePortal"))

    @self_service_portal.setter
    def self_service_portal(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "selfServicePortal", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="splitTunnel")
    def split_tunnel(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''Indicates whether split-tunnel is enabled on the AWS Client VPN endpoint.

        By default, split-tunnel on a VPN endpoint is disabled.

        For information about split-tunnel VPN endpoints, see `Split-tunnel AWS Client VPN endpoint <https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/split-tunnel-vpn.html>`_ in the *AWS Client VPN Administrator Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-splittunnel
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "splitTunnel"))

    @split_tunnel.setter
    def split_tunnel(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "splitTunnel", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tagSpecifications")
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.TagSpecificationProperty"]]]]:
        '''The tags to apply to the Client VPN endpoint during creation.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-tagspecifications
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.TagSpecificationProperty"]]]], jsii.get(self, "tagSpecifications"))

    @tag_specifications.setter
    def tag_specifications(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.TagSpecificationProperty"]]]],
    ) -> None:
        jsii.set(self, "tagSpecifications", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="transportProtocol")
    def transport_protocol(self) -> typing.Optional[builtins.str]:
        '''The transport protocol to be used by the VPN session.

        Default value: ``udp``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-transportprotocol
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "transportProtocol"))

    @transport_protocol.setter
    def transport_protocol(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "transportProtocol", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="vpcId")
    def vpc_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the VPC to associate with the Client VPN endpoint.

        If no security group IDs are specified in the request, the default security group for the VPC is applied.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-vpcid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "vpcId"))

    @vpc_id.setter
    def vpc_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "vpcId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="vpnPort")
    def vpn_port(self) -> typing.Optional[jsii.Number]:
        '''The port number to assign to the Client VPN endpoint for TCP and UDP traffic.

        Valid Values: ``443`` | ``1194``

        Default Value: ``443``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-vpnport
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "vpnPort"))

    @vpn_port.setter
    def vpn_port(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "vpnPort", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnClientVpnEndpoint.CertificateAuthenticationRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "client_root_certificate_chain_arn": "clientRootCertificateChainArn",
        },
    )
    class CertificateAuthenticationRequestProperty:
        def __init__(self, *, client_root_certificate_chain_arn: builtins.str) -> None:
            '''Information about the client certificate to be used for authentication.

            :param client_root_certificate_chain_arn: The ARN of the client certificate. The certificate must be signed by a certificate authority (CA) and it must be provisioned in AWS Certificate Manager (ACM).

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-certificateauthenticationrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                certificate_authentication_request_property = ec2.CfnClientVpnEndpoint.CertificateAuthenticationRequestProperty(
                    client_root_certificate_chain_arn="clientRootCertificateChainArn"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "client_root_certificate_chain_arn": client_root_certificate_chain_arn,
            }

        @builtins.property
        def client_root_certificate_chain_arn(self) -> builtins.str:
            '''The ARN of the client certificate.

            The certificate must be signed by a certificate authority (CA) and it must be provisioned in AWS Certificate Manager (ACM).

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-certificateauthenticationrequest.html#cfn-ec2-clientvpnendpoint-certificateauthenticationrequest-clientrootcertificatechainarn
            '''
            result = self._values.get("client_root_certificate_chain_arn")
            assert result is not None, "Required property 'client_root_certificate_chain_arn' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CertificateAuthenticationRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnClientVpnEndpoint.ClientAuthenticationRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "type": "type",
            "active_directory": "activeDirectory",
            "federated_authentication": "federatedAuthentication",
            "mutual_authentication": "mutualAuthentication",
        },
    )
    class ClientAuthenticationRequestProperty:
        def __init__(
            self,
            *,
            type: builtins.str,
            active_directory: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.DirectoryServiceAuthenticationRequestProperty"]] = None,
            federated_authentication: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.FederatedAuthenticationRequestProperty"]] = None,
            mutual_authentication: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.CertificateAuthenticationRequestProperty"]] = None,
        ) -> None:
            '''Describes the authentication method to be used by a Client VPN endpoint.

            For more information, see `Authentication <https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/authentication-authrization.html#client-authentication>`_ in the *AWS Client VPN Administrator Guide* .

            :param type: The type of client authentication to be used.
            :param active_directory: Information about the Active Directory to be used, if applicable. You must provide this information if *Type* is ``directory-service-authentication`` .
            :param federated_authentication: Information about the IAM SAML identity provider, if applicable.
            :param mutual_authentication: Information about the authentication certificates to be used, if applicable. You must provide this information if *Type* is ``certificate-authentication`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientauthenticationrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                client_authentication_request_property = ec2.CfnClientVpnEndpoint.ClientAuthenticationRequestProperty(
                    type="type",
                
                    # the properties below are optional
                    active_directory=ec2.CfnClientVpnEndpoint.DirectoryServiceAuthenticationRequestProperty(
                        directory_id="directoryId"
                    ),
                    federated_authentication=ec2.CfnClientVpnEndpoint.FederatedAuthenticationRequestProperty(
                        saml_provider_arn="samlProviderArn",
                
                        # the properties below are optional
                        self_service_saml_provider_arn="selfServiceSamlProviderArn"
                    ),
                    mutual_authentication=ec2.CfnClientVpnEndpoint.CertificateAuthenticationRequestProperty(
                        client_root_certificate_chain_arn="clientRootCertificateChainArn"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "type": type,
            }
            if active_directory is not None:
                self._values["active_directory"] = active_directory
            if federated_authentication is not None:
                self._values["federated_authentication"] = federated_authentication
            if mutual_authentication is not None:
                self._values["mutual_authentication"] = mutual_authentication

        @builtins.property
        def type(self) -> builtins.str:
            '''The type of client authentication to be used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientauthenticationrequest.html#cfn-ec2-clientvpnendpoint-clientauthenticationrequest-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def active_directory(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.DirectoryServiceAuthenticationRequestProperty"]]:
            '''Information about the Active Directory to be used, if applicable.

            You must provide this information if *Type* is ``directory-service-authentication`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientauthenticationrequest.html#cfn-ec2-clientvpnendpoint-clientauthenticationrequest-activedirectory
            '''
            result = self._values.get("active_directory")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.DirectoryServiceAuthenticationRequestProperty"]], result)

        @builtins.property
        def federated_authentication(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.FederatedAuthenticationRequestProperty"]]:
            '''Information about the IAM SAML identity provider, if applicable.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientauthenticationrequest.html#cfn-ec2-clientvpnendpoint-clientauthenticationrequest-federatedauthentication
            '''
            result = self._values.get("federated_authentication")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.FederatedAuthenticationRequestProperty"]], result)

        @builtins.property
        def mutual_authentication(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.CertificateAuthenticationRequestProperty"]]:
            '''Information about the authentication certificates to be used, if applicable.

            You must provide this information if *Type* is ``certificate-authentication`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientauthenticationrequest.html#cfn-ec2-clientvpnendpoint-clientauthenticationrequest-mutualauthentication
            '''
            result = self._values.get("mutual_authentication")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.CertificateAuthenticationRequestProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClientAuthenticationRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnClientVpnEndpoint.ClientConnectOptionsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "enabled": "enabled",
            "lambda_function_arn": "lambdaFunctionArn",
        },
    )
    class ClientConnectOptionsProperty:
        def __init__(
            self,
            *,
            enabled: typing.Union[builtins.bool, aws_cdk.core.IResolvable],
            lambda_function_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Indicates whether client connect options are enabled.

            The default is ``false`` (not enabled).

            :param enabled: Indicates whether client connect options are enabled. The default is ``false`` (not enabled).
            :param lambda_function_arn: The Amazon Resource Name (ARN) of the AWS Lambda function used for connection authorization.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientconnectoptions.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                client_connect_options_property = ec2.CfnClientVpnEndpoint.ClientConnectOptionsProperty(
                    enabled=False,
                
                    # the properties below are optional
                    lambda_function_arn="lambdaFunctionArn"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "enabled": enabled,
            }
            if lambda_function_arn is not None:
                self._values["lambda_function_arn"] = lambda_function_arn

        @builtins.property
        def enabled(self) -> typing.Union[builtins.bool, aws_cdk.core.IResolvable]:
            '''Indicates whether client connect options are enabled.

            The default is ``false`` (not enabled).

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientconnectoptions.html#cfn-ec2-clientvpnendpoint-clientconnectoptions-enabled
            '''
            result = self._values.get("enabled")
            assert result is not None, "Required property 'enabled' is missing"
            return typing.cast(typing.Union[builtins.bool, aws_cdk.core.IResolvable], result)

        @builtins.property
        def lambda_function_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the AWS Lambda function used for connection authorization.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientconnectoptions.html#cfn-ec2-clientvpnendpoint-clientconnectoptions-lambdafunctionarn
            '''
            result = self._values.get("lambda_function_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClientConnectOptionsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnClientVpnEndpoint.ConnectionLogOptionsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "enabled": "enabled",
            "cloudwatch_log_group": "cloudwatchLogGroup",
            "cloudwatch_log_stream": "cloudwatchLogStream",
        },
    )
    class ConnectionLogOptionsProperty:
        def __init__(
            self,
            *,
            enabled: typing.Union[builtins.bool, aws_cdk.core.IResolvable],
            cloudwatch_log_group: typing.Optional[builtins.str] = None,
            cloudwatch_log_stream: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Describes the client connection logging options for the Client VPN endpoint.

            :param enabled: Indicates whether connection logging is enabled.
            :param cloudwatch_log_group: The name of the CloudWatch Logs log group. Required if connection logging is enabled.
            :param cloudwatch_log_stream: The name of the CloudWatch Logs log stream to which the connection data is published.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-connectionlogoptions.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                connection_log_options_property = ec2.CfnClientVpnEndpoint.ConnectionLogOptionsProperty(
                    enabled=False,
                
                    # the properties below are optional
                    cloudwatch_log_group="cloudwatchLogGroup",
                    cloudwatch_log_stream="cloudwatchLogStream"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "enabled": enabled,
            }
            if cloudwatch_log_group is not None:
                self._values["cloudwatch_log_group"] = cloudwatch_log_group
            if cloudwatch_log_stream is not None:
                self._values["cloudwatch_log_stream"] = cloudwatch_log_stream

        @builtins.property
        def enabled(self) -> typing.Union[builtins.bool, aws_cdk.core.IResolvable]:
            '''Indicates whether connection logging is enabled.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-connectionlogoptions.html#cfn-ec2-clientvpnendpoint-connectionlogoptions-enabled
            '''
            result = self._values.get("enabled")
            assert result is not None, "Required property 'enabled' is missing"
            return typing.cast(typing.Union[builtins.bool, aws_cdk.core.IResolvable], result)

        @builtins.property
        def cloudwatch_log_group(self) -> typing.Optional[builtins.str]:
            '''The name of the CloudWatch Logs log group.

            Required if connection logging is enabled.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-connectionlogoptions.html#cfn-ec2-clientvpnendpoint-connectionlogoptions-cloudwatchloggroup
            '''
            result = self._values.get("cloudwatch_log_group")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def cloudwatch_log_stream(self) -> typing.Optional[builtins.str]:
            '''The name of the CloudWatch Logs log stream to which the connection data is published.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-connectionlogoptions.html#cfn-ec2-clientvpnendpoint-connectionlogoptions-cloudwatchlogstream
            '''
            result = self._values.get("cloudwatch_log_stream")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ConnectionLogOptionsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnClientVpnEndpoint.DirectoryServiceAuthenticationRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"directory_id": "directoryId"},
    )
    class DirectoryServiceAuthenticationRequestProperty:
        def __init__(self, *, directory_id: builtins.str) -> None:
            '''Describes the Active Directory to be used for client authentication.

            :param directory_id: The ID of the Active Directory to be used for authentication.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-directoryserviceauthenticationrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                directory_service_authentication_request_property = ec2.CfnClientVpnEndpoint.DirectoryServiceAuthenticationRequestProperty(
                    directory_id="directoryId"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "directory_id": directory_id,
            }

        @builtins.property
        def directory_id(self) -> builtins.str:
            '''The ID of the Active Directory to be used for authentication.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-directoryserviceauthenticationrequest.html#cfn-ec2-clientvpnendpoint-directoryserviceauthenticationrequest-directoryid
            '''
            result = self._values.get("directory_id")
            assert result is not None, "Required property 'directory_id' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DirectoryServiceAuthenticationRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnClientVpnEndpoint.FederatedAuthenticationRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "saml_provider_arn": "samlProviderArn",
            "self_service_saml_provider_arn": "selfServiceSamlProviderArn",
        },
    )
    class FederatedAuthenticationRequestProperty:
        def __init__(
            self,
            *,
            saml_provider_arn: builtins.str,
            self_service_saml_provider_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The IAM SAML identity provider used for federated authentication.

            :param saml_provider_arn: The Amazon Resource Name (ARN) of the IAM SAML identity provider.
            :param self_service_saml_provider_arn: The Amazon Resource Name (ARN) of the IAM SAML identity provider for the self-service portal.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-federatedauthenticationrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                federated_authentication_request_property = ec2.CfnClientVpnEndpoint.FederatedAuthenticationRequestProperty(
                    saml_provider_arn="samlProviderArn",
                
                    # the properties below are optional
                    self_service_saml_provider_arn="selfServiceSamlProviderArn"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "saml_provider_arn": saml_provider_arn,
            }
            if self_service_saml_provider_arn is not None:
                self._values["self_service_saml_provider_arn"] = self_service_saml_provider_arn

        @builtins.property
        def saml_provider_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the IAM SAML identity provider.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-federatedauthenticationrequest.html#cfn-ec2-clientvpnendpoint-federatedauthenticationrequest-samlproviderarn
            '''
            result = self._values.get("saml_provider_arn")
            assert result is not None, "Required property 'saml_provider_arn' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def self_service_saml_provider_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the IAM SAML identity provider for the self-service portal.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-federatedauthenticationrequest.html#cfn-ec2-clientvpnendpoint-federatedauthenticationrequest-selfservicesamlproviderarn
            '''
            result = self._values.get("self_service_saml_provider_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FederatedAuthenticationRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnClientVpnEndpoint.TagSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"resource_type": "resourceType", "tags": "tags"},
    )
    class TagSpecificationProperty:
        def __init__(
            self,
            *,
            resource_type: builtins.str,
            tags: typing.Sequence[aws_cdk.core.CfnTag],
        ) -> None:
            '''The tags to apply to a resource when the resource is being created.

            :param resource_type: The type of resource to tag.
            :param tags: The tags to apply to the resource.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-tagspecification.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                tag_specification_property = ec2.CfnClientVpnEndpoint.TagSpecificationProperty(
                    resource_type="resourceType",
                    tags=[CfnTag(
                        key="key",
                        value="value"
                    )]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "resource_type": resource_type,
                "tags": tags,
            }

        @builtins.property
        def resource_type(self) -> builtins.str:
            '''The type of resource to tag.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-tagspecification.html#cfn-ec2-clientvpnendpoint-tagspecification-resourcetype
            '''
            result = self._values.get("resource_type")
            assert result is not None, "Required property 'resource_type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def tags(self) -> typing.List[aws_cdk.core.CfnTag]:
            '''The tags to apply to the resource.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-tagspecification.html#cfn-ec2-clientvpnendpoint-tagspecification-tags
            '''
            result = self._values.get("tags")
            assert result is not None, "Required property 'tags' is missing"
            return typing.cast(typing.List[aws_cdk.core.CfnTag], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TagSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnClientVpnEndpointProps",
    jsii_struct_bases=[],
    name_mapping={
        "authentication_options": "authenticationOptions",
        "client_cidr_block": "clientCidrBlock",
        "connection_log_options": "connectionLogOptions",
        "server_certificate_arn": "serverCertificateArn",
        "client_connect_options": "clientConnectOptions",
        "description": "description",
        "dns_servers": "dnsServers",
        "security_group_ids": "securityGroupIds",
        "self_service_portal": "selfServicePortal",
        "split_tunnel": "splitTunnel",
        "tag_specifications": "tagSpecifications",
        "transport_protocol": "transportProtocol",
        "vpc_id": "vpcId",
        "vpn_port": "vpnPort",
    },
)
class CfnClientVpnEndpointProps:
    def __init__(
        self,
        *,
        authentication_options: typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[CfnClientVpnEndpoint.ClientAuthenticationRequestProperty, aws_cdk.core.IResolvable]]],
        client_cidr_block: builtins.str,
        connection_log_options: typing.Union[aws_cdk.core.IResolvable, CfnClientVpnEndpoint.ConnectionLogOptionsProperty],
        server_certificate_arn: builtins.str,
        client_connect_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnClientVpnEndpoint.ClientConnectOptionsProperty]] = None,
        description: typing.Optional[builtins.str] = None,
        dns_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
        self_service_portal: typing.Optional[builtins.str] = None,
        split_tunnel: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        tag_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnClientVpnEndpoint.TagSpecificationProperty]]]] = None,
        transport_protocol: typing.Optional[builtins.str] = None,
        vpc_id: typing.Optional[builtins.str] = None,
        vpn_port: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties for defining a ``CfnClientVpnEndpoint``.

        :param authentication_options: Information about the authentication method to be used to authenticate clients.
        :param client_cidr_block: The IPv4 address range, in CIDR notation, from which to assign client IP addresses. The address range cannot overlap with the local CIDR of the VPC in which the associated subnet is located, or the routes that you add manually. The address range cannot be changed after the Client VPN endpoint has been created. The CIDR block should be /22 or greater.
        :param connection_log_options: Information about the client connection logging options. If you enable client connection logging, data about client connections is sent to a Cloudwatch Logs log stream. The following information is logged: - Client connection requests - Client connection results (successful and unsuccessful) - Reasons for unsuccessful client connection requests - Client connection termination time
        :param server_certificate_arn: The ARN of the server certificate. For more information, see the `AWS Certificate Manager User Guide <https://docs.aws.amazon.com/acm/latest/userguide/>`_ .
        :param client_connect_options: The options for managing connection authorization for new client connections.
        :param description: A brief description of the Client VPN endpoint.
        :param dns_servers: Information about the DNS servers to be used for DNS resolution. A Client VPN endpoint can have up to two DNS servers. If no DNS server is specified, the DNS address configured on the device is used for the DNS server.
        :param security_group_ids: The IDs of one or more security groups to apply to the target network. You must also specify the ID of the VPC that contains the security groups.
        :param self_service_portal: Specify whether to enable the self-service portal for the Client VPN endpoint. Default Value: ``enabled``
        :param split_tunnel: Indicates whether split-tunnel is enabled on the AWS Client VPN endpoint. By default, split-tunnel on a VPN endpoint is disabled. For information about split-tunnel VPN endpoints, see `Split-tunnel AWS Client VPN endpoint <https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/split-tunnel-vpn.html>`_ in the *AWS Client VPN Administrator Guide* .
        :param tag_specifications: The tags to apply to the Client VPN endpoint during creation.
        :param transport_protocol: The transport protocol to be used by the VPN session. Default value: ``udp``
        :param vpc_id: The ID of the VPC to associate with the Client VPN endpoint. If no security group IDs are specified in the request, the default security group for the VPC is applied.
        :param vpn_port: The port number to assign to the Client VPN endpoint for TCP and UDP traffic. Valid Values: ``443`` | ``1194`` Default Value: ``443``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            cfn_client_vpn_endpoint_props = ec2.CfnClientVpnEndpointProps(
                authentication_options=[ec2.CfnClientVpnEndpoint.ClientAuthenticationRequestProperty(
                    type="type",
            
                    # the properties below are optional
                    active_directory=ec2.CfnClientVpnEndpoint.DirectoryServiceAuthenticationRequestProperty(
                        directory_id="directoryId"
                    ),
                    federated_authentication=ec2.CfnClientVpnEndpoint.FederatedAuthenticationRequestProperty(
                        saml_provider_arn="samlProviderArn",
            
                        # the properties below are optional
                        self_service_saml_provider_arn="selfServiceSamlProviderArn"
                    ),
                    mutual_authentication=ec2.CfnClientVpnEndpoint.CertificateAuthenticationRequestProperty(
                        client_root_certificate_chain_arn="clientRootCertificateChainArn"
                    )
                )],
                client_cidr_block="clientCidrBlock",
                connection_log_options=ec2.CfnClientVpnEndpoint.ConnectionLogOptionsProperty(
                    enabled=False,
            
                    # the properties below are optional
                    cloudwatch_log_group="cloudwatchLogGroup",
                    cloudwatch_log_stream="cloudwatchLogStream"
                ),
                server_certificate_arn="serverCertificateArn",
            
                # the properties below are optional
                client_connect_options=ec2.CfnClientVpnEndpoint.ClientConnectOptionsProperty(
                    enabled=False,
            
                    # the properties below are optional
                    lambda_function_arn="lambdaFunctionArn"
                ),
                description="description",
                dns_servers=["dnsServers"],
                security_group_ids=["securityGroupIds"],
                self_service_portal="selfServicePortal",
                split_tunnel=False,
                tag_specifications=[ec2.CfnClientVpnEndpoint.TagSpecificationProperty(
                    resource_type="resourceType",
                    tags=[CfnTag(
                        key="key",
                        value="value"
                    )]
                )],
                transport_protocol="transportProtocol",
                vpc_id="vpcId",
                vpn_port=123
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "authentication_options": authentication_options,
            "client_cidr_block": client_cidr_block,
            "connection_log_options": connection_log_options,
            "server_certificate_arn": server_certificate_arn,
        }
        if client_connect_options is not None:
            self._values["client_connect_options"] = client_connect_options
        if description is not None:
            self._values["description"] = description
        if dns_servers is not None:
            self._values["dns_servers"] = dns_servers
        if security_group_ids is not None:
            self._values["security_group_ids"] = security_group_ids
        if self_service_portal is not None:
            self._values["self_service_portal"] = self_service_portal
        if split_tunnel is not None:
            self._values["split_tunnel"] = split_tunnel
        if tag_specifications is not None:
            self._values["tag_specifications"] = tag_specifications
        if transport_protocol is not None:
            self._values["transport_protocol"] = transport_protocol
        if vpc_id is not None:
            self._values["vpc_id"] = vpc_id
        if vpn_port is not None:
            self._values["vpn_port"] = vpn_port

    @builtins.property
    def authentication_options(
        self,
    ) -> typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[CfnClientVpnEndpoint.ClientAuthenticationRequestProperty, aws_cdk.core.IResolvable]]]:
        '''Information about the authentication method to be used to authenticate clients.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-authenticationoptions
        '''
        result = self._values.get("authentication_options")
        assert result is not None, "Required property 'authentication_options' is missing"
        return typing.cast(typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[CfnClientVpnEndpoint.ClientAuthenticationRequestProperty, aws_cdk.core.IResolvable]]], result)

    @builtins.property
    def client_cidr_block(self) -> builtins.str:
        '''The IPv4 address range, in CIDR notation, from which to assign client IP addresses.

        The address range cannot overlap with the local CIDR of the VPC in which the associated subnet is located, or the routes that you add manually. The address range cannot be changed after the Client VPN endpoint has been created. The CIDR block should be /22 or greater.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-clientcidrblock
        '''
        result = self._values.get("client_cidr_block")
        assert result is not None, "Required property 'client_cidr_block' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def connection_log_options(
        self,
    ) -> typing.Union[aws_cdk.core.IResolvable, CfnClientVpnEndpoint.ConnectionLogOptionsProperty]:
        '''Information about the client connection logging options.

        If you enable client connection logging, data about client connections is sent to a Cloudwatch Logs log stream. The following information is logged:

        - Client connection requests
        - Client connection results (successful and unsuccessful)
        - Reasons for unsuccessful client connection requests
        - Client connection termination time

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-connectionlogoptions
        '''
        result = self._values.get("connection_log_options")
        assert result is not None, "Required property 'connection_log_options' is missing"
        return typing.cast(typing.Union[aws_cdk.core.IResolvable, CfnClientVpnEndpoint.ConnectionLogOptionsProperty], result)

    @builtins.property
    def server_certificate_arn(self) -> builtins.str:
        '''The ARN of the server certificate.

        For more information, see the `AWS Certificate Manager User Guide <https://docs.aws.amazon.com/acm/latest/userguide/>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-servercertificatearn
        '''
        result = self._values.get("server_certificate_arn")
        assert result is not None, "Required property 'server_certificate_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def client_connect_options(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnClientVpnEndpoint.ClientConnectOptionsProperty]]:
        '''The options for managing connection authorization for new client connections.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-clientconnectoptions
        '''
        result = self._values.get("client_connect_options")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnClientVpnEndpoint.ClientConnectOptionsProperty]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A brief description of the Client VPN endpoint.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def dns_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Information about the DNS servers to be used for DNS resolution.

        A Client VPN endpoint can have up to two DNS servers. If no DNS server is specified, the DNS address configured on the device is used for the DNS server.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-dnsservers
        '''
        result = self._values.get("dns_servers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The IDs of one or more security groups to apply to the target network.

        You must also specify the ID of the VPC that contains the security groups.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-securitygroupids
        '''
        result = self._values.get("security_group_ids")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def self_service_portal(self) -> typing.Optional[builtins.str]:
        '''Specify whether to enable the self-service portal for the Client VPN endpoint.

        Default Value: ``enabled``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-selfserviceportal
        '''
        result = self._values.get("self_service_portal")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def split_tunnel(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''Indicates whether split-tunnel is enabled on the AWS Client VPN endpoint.

        By default, split-tunnel on a VPN endpoint is disabled.

        For information about split-tunnel VPN endpoints, see `Split-tunnel AWS Client VPN endpoint <https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/split-tunnel-vpn.html>`_ in the *AWS Client VPN Administrator Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-splittunnel
        '''
        result = self._values.get("split_tunnel")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnClientVpnEndpoint.TagSpecificationProperty]]]]:
        '''The tags to apply to the Client VPN endpoint during creation.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-tagspecifications
        '''
        result = self._values.get("tag_specifications")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnClientVpnEndpoint.TagSpecificationProperty]]]], result)

    @builtins.property
    def transport_protocol(self) -> typing.Optional[builtins.str]:
        '''The transport protocol to be used by the VPN session.

        Default value: ``udp``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-transportprotocol
        '''
        result = self._values.get("transport_protocol")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def vpc_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the VPC to associate with the Client VPN endpoint.

        If no security group IDs are specified in the request, the default security group for the VPC is applied.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-vpcid
        '''
        result = self._values.get("vpc_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def vpn_port(self) -> typing.Optional[jsii.Number]:
        '''The port number to assign to the Client VPN endpoint for TCP and UDP traffic.

        Valid Values: ``443`` | ``1194``

        Default Value: ``443``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-vpnport
        '''
        result = self._values.get("vpn_port")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnClientVpnEndpointProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnClientVpnRoute(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnClientVpnRoute",
):
    '''A CloudFormation ``AWS::EC2::ClientVpnRoute``.

    Specifies a network route to add to a Client VPN endpoint. Each Client VPN endpoint has a route table that describes the available destination network routes. Each route in the route table specifies the path for traffic to specific resources or networks.

    A target network association must be created before you can specify a route. If you're setting up all the components of a Client VPN endpoint at the same time, you must use the `DependsOn Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html>`_ to declare a dependency on the ``AWS::EC2::ClientVpnTargetNetworkAssociation`` resource.

    :cloudformationResource: AWS::EC2::ClientVpnRoute
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ec2 as ec2
        
        cfn_client_vpn_route = ec2.CfnClientVpnRoute(self, "MyCfnClientVpnRoute",
            client_vpn_endpoint_id="clientVpnEndpointId",
            destination_cidr_block="destinationCidrBlock",
            target_vpc_subnet_id="targetVpcSubnetId",
        
            # the properties below are optional
            description="description"
        )
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        client_vpn_endpoint_id: builtins.str,
        destination_cidr_block: builtins.str,
        target_vpc_subnet_id: builtins.str,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::ClientVpnRoute``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param client_vpn_endpoint_id: The ID of the Client VPN endpoint to which to add the route.
        :param destination_cidr_block: The IPv4 address range, in CIDR notation, of the route destination. For example:. - To add a route for Internet access, enter ``0.0.0.0/0`` - To add a route for a peered VPC, enter the peered VPC's IPv4 CIDR range - To add a route for an on-premises network, enter the AWS Site-to-Site VPN connection's IPv4 CIDR range - To add a route for the local network, enter the client CIDR range
        :param target_vpc_subnet_id: The ID of the subnet through which you want to route traffic. The specified subnet must be an existing target network of the Client VPN endpoint. Alternatively, if you're adding a route for the local network, specify ``local`` .
        :param description: A brief description of the route.
        '''
        props = CfnClientVpnRouteProps(
            client_vpn_endpoint_id=client_vpn_endpoint_id,
            destination_cidr_block=destination_cidr_block,
            target_vpc_subnet_id=target_vpc_subnet_id,
            description=description,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="clientVpnEndpointId")
    def client_vpn_endpoint_id(self) -> builtins.str:
        '''The ID of the Client VPN endpoint to which to add the route.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html#cfn-ec2-clientvpnroute-clientvpnendpointid
        '''
        return typing.cast(builtins.str, jsii.get(self, "clientVpnEndpointId"))

    @client_vpn_endpoint_id.setter
    def client_vpn_endpoint_id(self, value: builtins.str) -> None:
        jsii.set(self, "clientVpnEndpointId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="destinationCidrBlock")
    def destination_cidr_block(self) -> builtins.str:
        '''The IPv4 address range, in CIDR notation, of the route destination. For example:.

        - To add a route for Internet access, enter ``0.0.0.0/0``
        - To add a route for a peered VPC, enter the peered VPC's IPv4 CIDR range
        - To add a route for an on-premises network, enter the AWS Site-to-Site VPN connection's IPv4 CIDR range
        - To add a route for the local network, enter the client CIDR range

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html#cfn-ec2-clientvpnroute-destinationcidrblock
        '''
        return typing.cast(builtins.str, jsii.get(self, "destinationCidrBlock"))

    @destination_cidr_block.setter
    def destination_cidr_block(self, value: builtins.str) -> None:
        jsii.set(self, "destinationCidrBlock", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="targetVpcSubnetId")
    def target_vpc_subnet_id(self) -> builtins.str:
        '''The ID of the subnet through which you want to route traffic.

        The specified subnet must be an existing target network of the Client VPN endpoint.

        Alternatively, if you're adding a route for the local network, specify ``local`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html#cfn-ec2-clientvpnroute-targetvpcsubnetid
        '''
        return typing.cast(builtins.str, jsii.get(self, "targetVpcSubnetId"))

    @target_vpc_subnet_id.setter
    def target_vpc_subnet_id(self, value: builtins.str) -> None:
        jsii.set(self, "targetVpcSubnetId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A brief description of the route.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html#cfn-ec2-clientvpnroute-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "description", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnClientVpnRouteProps",
    jsii_struct_bases=[],
    name_mapping={
        "client_vpn_endpoint_id": "clientVpnEndpointId",
        "destination_cidr_block": "destinationCidrBlock",
        "target_vpc_subnet_id": "targetVpcSubnetId",
        "description": "description",
    },
)
class CfnClientVpnRouteProps:
    def __init__(
        self,
        *,
        client_vpn_endpoint_id: builtins.str,
        destination_cidr_block: builtins.str,
        target_vpc_subnet_id: builtins.str,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnClientVpnRoute``.

        :param client_vpn_endpoint_id: The ID of the Client VPN endpoint to which to add the route.
        :param destination_cidr_block: The IPv4 address range, in CIDR notation, of the route destination. For example:. - To add a route for Internet access, enter ``0.0.0.0/0`` - To add a route for a peered VPC, enter the peered VPC's IPv4 CIDR range - To add a route for an on-premises network, enter the AWS Site-to-Site VPN connection's IPv4 CIDR range - To add a route for the local network, enter the client CIDR range
        :param target_vpc_subnet_id: The ID of the subnet through which you want to route traffic. The specified subnet must be an existing target network of the Client VPN endpoint. Alternatively, if you're adding a route for the local network, specify ``local`` .
        :param description: A brief description of the route.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            cfn_client_vpn_route_props = ec2.CfnClientVpnRouteProps(
                client_vpn_endpoint_id="clientVpnEndpointId",
                destination_cidr_block="destinationCidrBlock",
                target_vpc_subnet_id="targetVpcSubnetId",
            
                # the properties below are optional
                description="description"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "client_vpn_endpoint_id": client_vpn_endpoint_id,
            "destination_cidr_block": destination_cidr_block,
            "target_vpc_subnet_id": target_vpc_subnet_id,
        }
        if description is not None:
            self._values["description"] = description

    @builtins.property
    def client_vpn_endpoint_id(self) -> builtins.str:
        '''The ID of the Client VPN endpoint to which to add the route.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html#cfn-ec2-clientvpnroute-clientvpnendpointid
        '''
        result = self._values.get("client_vpn_endpoint_id")
        assert result is not None, "Required property 'client_vpn_endpoint_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def destination_cidr_block(self) -> builtins.str:
        '''The IPv4 address range, in CIDR notation, of the route destination. For example:.

        - To add a route for Internet access, enter ``0.0.0.0/0``
        - To add a route for a peered VPC, enter the peered VPC's IPv4 CIDR range
        - To add a route for an on-premises network, enter the AWS Site-to-Site VPN connection's IPv4 CIDR range
        - To add a route for the local network, enter the client CIDR range

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html#cfn-ec2-clientvpnroute-destinationcidrblock
        '''
        result = self._values.get("destination_cidr_block")
        assert result is not None, "Required property 'destination_cidr_block' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def target_vpc_subnet_id(self) -> builtins.str:
        '''The ID of the subnet through which you want to route traffic.

        The specified subnet must be an existing target network of the Client VPN endpoint.

        Alternatively, if you're adding a route for the local network, specify ``local`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html#cfn-ec2-clientvpnroute-targetvpcsubnetid
        '''
        result = self._values.get("target_vpc_subnet_id")
        assert result is not None, "Required property 'target_vpc_subnet_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A brief description of the route.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html#cfn-ec2-clientvpnroute-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnClientVpnRouteProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnClientVpnTargetNetworkAssociation(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnClientVpnTargetNetworkAssociation",
):
    '''A CloudFormation ``AWS::EC2::ClientVpnTargetNetworkAssociation``.

    Specifies a target network to associate with a Client VPN endpoint. A target network is a subnet in a VPC. You can associate multiple subnets from the same VPC with a Client VPN endpoint. You can associate only one subnet in each Availability Zone. We recommend that you associate at least two subnets to provide Availability Zone redundancy.

    :cloudformationResource: AWS::EC2::ClientVpnTargetNetworkAssociation
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpntargetnetworkassociation.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ec2 as ec2
        
        cfn_client_vpn_target_network_association = ec2.CfnClientVpnTargetNetworkAssociation(self, "MyCfnClientVpnTargetNetworkAssociation",
            client_vpn_endpoint_id="clientVpnEndpointId",
            subnet_id="subnetId"
        )
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        client_vpn_endpoint_id: builtins.str,
        subnet_id: builtins.str,
    ) -> None:
        '''Create a new ``AWS::EC2::ClientVpnTargetNetworkAssociation``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param client_vpn_endpoint_id: The ID of the Client VPN endpoint.
        :param subnet_id: The ID of the subnet to associate with the Client VPN endpoint.
        '''
        props = CfnClientVpnTargetNetworkAssociationProps(
            client_vpn_endpoint_id=client_vpn_endpoint_id, subnet_id=subnet_id
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="clientVpnEndpointId")
    def client_vpn_endpoint_id(self) -> builtins.str:
        '''The ID of the Client VPN endpoint.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpntargetnetworkassociation.html#cfn-ec2-clientvpntargetnetworkassociation-clientvpnendpointid
        '''
        return typing.cast(builtins.str, jsii.get(self, "clientVpnEndpointId"))

    @client_vpn_endpoint_id.setter
    def client_vpn_endpoint_id(self, value: builtins.str) -> None:
        jsii.set(self, "clientVpnEndpointId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="subnetId")
    def subnet_id(self) -> builtins.str:
        '''The ID of the subnet to associate with the Client VPN endpoint.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpntargetnetworkassociation.html#cfn-ec2-clientvpntargetnetworkassociation-subnetid
        '''
        return typing.cast(builtins.str, jsii.get(self, "subnetId"))

    @subnet_id.setter
    def subnet_id(self, value: builtins.str) -> None:
        jsii.set(self, "subnetId", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnClientVpnTargetNetworkAssociationProps",
    jsii_struct_bases=[],
    name_mapping={
        "client_vpn_endpoint_id": "clientVpnEndpointId",
        "subnet_id": "subnetId",
    },
)
class CfnClientVpnTargetNetworkAssociationProps:
    def __init__(
        self,
        *,
        client_vpn_endpoint_id: builtins.str,
        subnet_id: builtins.str,
    ) -> None:
        '''Properties for defining a ``CfnClientVpnTargetNetworkAssociation``.

        :param client_vpn_endpoint_id: The ID of the Client VPN endpoint.
        :param subnet_id: The ID of the subnet to associate with the Client VPN endpoint.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpntargetnetworkassociation.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            cfn_client_vpn_target_network_association_props = ec2.CfnClientVpnTargetNetworkAssociationProps(
                client_vpn_endpoint_id="clientVpnEndpointId",
                subnet_id="subnetId"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "client_vpn_endpoint_id": client_vpn_endpoint_id,
            "subnet_id": subnet_id,
        }

    @builtins.property
    def client_vpn_endpoint_id(self) -> builtins.str:
        '''The ID of the Client VPN endpoint.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpntargetnetworkassociation.html#cfn-ec2-clientvpntargetnetworkassociation-clientvpnendpointid
        '''
        result = self._values.get("client_vpn_endpoint_id")
        assert result is not None, "Required property 'client_vpn_endpoint_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def subnet_id(self) -> builtins.str:
        '''The ID of the subnet to associate with the Client VPN endpoint.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpntargetnetworkassociation.html#cfn-ec2-clientvpntargetnetworkassociation-subnetid
        '''
        result = self._values.get("subnet_id")
        assert result is not None, "Required property 'subnet_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnClientVpnTargetNetworkAssociationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnCustomerGateway(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnCustomerGateway",
):
    '''A CloudFormation ``AWS::EC2::CustomerGateway``.

    Specifies a customer gateway.

    :cloudformationResource: AWS::EC2::CustomerGateway
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customer-gateway.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ec2 as ec2
        
        cfn_customer_gateway = ec2.CfnCustomerGateway(self, "MyCfnCustomerGateway",
            bgp_asn=123,
            ip_address="ipAddress",
            type="type",
        
            # the properties below are optional
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        bgp_asn: jsii.Number,
        ip_address: builtins.str,
        type: builtins.str,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::CustomerGateway``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param bgp_asn: For devices that support BGP, the customer gateway's BGP ASN. Default: 65000
        :param ip_address: The Internet-routable IP address for the customer gateway's outside interface. The address must be static.
        :param type: The type of VPN connection that this customer gateway supports ( ``ipsec.1`` ).
        :param tags: One or more tags for the customer gateway.
        '''
        props = CfnCustomerGatewayProps(
            bgp_asn=bgp_asn, ip_address=ip_address, type=type, tags=tags
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''One or more tags for the customer gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customer-gateway.html#cfn-ec2-customergateway-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="bgpAsn")
    def bgp_asn(self) -> jsii.Number:
        '''For devices that support BGP, the customer gateway's BGP ASN.

        Default: 65000

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customer-gateway.html#cfn-ec2-customergateway-bgpasn
        '''
        return typing.cast(jsii.Number, jsii.get(self, "bgpAsn"))

    @bgp_asn.setter
    def bgp_asn(self, value: jsii.Number) -> None:
        jsii.set(self, "bgpAsn", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ipAddress")
    def ip_address(self) -> builtins.str:
        '''The Internet-routable IP address for the customer gateway's outside interface.

        The address must be static.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customer-gateway.html#cfn-ec2-customergateway-ipaddress
        '''
        return typing.cast(builtins.str, jsii.get(self, "ipAddress"))

    @ip_address.setter
    def ip_address(self, value: builtins.str) -> None:
        jsii.set(self, "ipAddress", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="type")
    def type(self) -> builtins.str:
        '''The type of VPN connection that this customer gateway supports ( ``ipsec.1`` ).

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customer-gateway.html#cfn-ec2-customergateway-type
        '''
        return typing.cast(builtins.str, jsii.get(self, "type"))

    @type.setter
    def type(self, value: builtins.str) -> None:
        jsii.set(self, "type", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnCustomerGatewayProps",
    jsii_struct_bases=[],
    name_mapping={
        "bgp_asn": "bgpAsn",
        "ip_address": "ipAddress",
        "type": "type",
        "tags": "tags",
    },
)
class CfnCustomerGatewayProps:
    def __init__(
        self,
        *,
        bgp_asn: jsii.Number,
        ip_address: builtins.str,
        type: builtins.str,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Properties for defining a ``CfnCustomerGateway``.

        :param bgp_asn: For devices that support BGP, the customer gateway's BGP ASN. Default: 65000
        :param ip_address: The Internet-routable IP address for the customer gateway's outside interface. The address must be static.
        :param type: The type of VPN connection that this customer gateway supports ( ``ipsec.1`` ).
        :param tags: One or more tags for the customer gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customer-gateway.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            cfn_customer_gateway_props = ec2.CfnCustomerGatewayProps(
                bgp_asn=123,
                ip_address="ipAddress",
                type="type",
            
                # the properties below are optional
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "bgp_asn": bgp_asn,
            "ip_address": ip_address,
            "type": type,
        }
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def bgp_asn(self) -> jsii.Number:
        '''For devices that support BGP, the customer gateway's BGP ASN.

        Default: 65000

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customer-gateway.html#cfn-ec2-customergateway-bgpasn
        '''
        result = self._values.get("bgp_asn")
        assert result is not None, "Required property 'bgp_asn' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def ip_address(self) -> builtins.str:
        '''The Internet-routable IP address for the customer gateway's outside interface.

        The address must be static.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customer-gateway.html#cfn-ec2-customergateway-ipaddress
        '''
        result = self._values.get("ip_address")
        assert result is not None, "Required property 'ip_address' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def type(self) -> builtins.str:
        '''The type of VPN connection that this customer gateway supports ( ``ipsec.1`` ).

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customer-gateway.html#cfn-ec2-customergateway-type
        '''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''One or more tags for the customer gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customer-gateway.html#cfn-ec2-customergateway-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCustomerGatewayProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnDHCPOptions(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnDHCPOptions",
):
    '''A CloudFormation ``AWS::EC2::DHCPOptions``.

    Specifies a set of DHCP options for your VPC.

    You must specify at least one of the following properties: ``DomainNameServers`` , ``NetbiosNameServers`` , ``NtpServers`` . If you specify ``NetbiosNameServers`` , you must specify ``NetbiosNodeType`` .

    :cloudformationResource: AWS::EC2::DHCPOptions
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcpoptions.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ec2 as ec2
        
        cfn_dHCPOptions = ec2.CfnDHCPOptions(self, "MyCfnDHCPOptions",
            domain_name="domainName",
            domain_name_servers=["domainNameServers"],
            netbios_name_servers=["netbiosNameServers"],
            netbios_node_type=123,
            ntp_servers=["ntpServers"],
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        domain_name: typing.Optional[builtins.str] = None,
        domain_name_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        netbios_name_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        netbios_node_type: typing.Optional[jsii.Number] = None,
        ntp_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::DHCPOptions``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param domain_name: This value is used to complete unqualified DNS hostnames. If you're using AmazonProvidedDNS in ``us-east-1`` , specify ``ec2.internal`` . If you're using AmazonProvidedDNS in another Region, specify *region* . ``compute.internal`` (for example, ``ap-northeast-1.compute.internal`` ). Otherwise, specify a domain name (for example, *MyCompany.com* ).
        :param domain_name_servers: The IPv4 addresses of up to four domain name servers, or AmazonProvidedDNS. The default DHCP option set specifies ``AmazonProvidedDNS`` . If specifying more than one domain name server, specify the IP addresses in a single parameter, separated by commas. To have your instance to receive a custom DNS hostname as specified in ``DomainName`` , you must set this to a custom DNS server.
        :param netbios_name_servers: The IPv4 addresses of up to four NetBIOS name servers.
        :param netbios_node_type: The NetBIOS node type (1, 2, 4, or 8). We recommend that you specify 2 (broadcast and multicast are not currently supported).
        :param ntp_servers: The IPv4 addresses of up to four Network Time Protocol (NTP) servers.
        :param tags: Any tags assigned to the DHCP options set.
        '''
        props = CfnDHCPOptionsProps(
            domain_name=domain_name,
            domain_name_servers=domain_name_servers,
            netbios_name_servers=netbios_name_servers,
            netbios_node_type=netbios_node_type,
            ntp_servers=ntp_servers,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrDhcpOptionsId")
    def attr_dhcp_options_id(self) -> builtins.str:
        '''The ID of the DHCP options set.

        :cloudformationAttribute: DhcpOptionsId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDhcpOptionsId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''Any tags assigned to the DHCP options set.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcpoptions.html#cfn-ec2-dhcpoptions-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="domainName")
    def domain_name(self) -> typing.Optional[builtins.str]:
        '''This value is used to complete unqualified DNS hostnames.

        If you're using AmazonProvidedDNS in ``us-east-1`` , specify ``ec2.internal`` . If you're using AmazonProvidedDNS in another Region, specify *region* . ``compute.internal`` (for example, ``ap-northeast-1.compute.internal`` ). Otherwise, specify a domain name (for example, *MyCompany.com* ).

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcpoptions.html#cfn-ec2-dhcpoptions-domainname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "domainName"))

    @domain_name.setter
    def domain_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "domainName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="domainNameServers")
    def domain_name_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The IPv4 addresses of up to four domain name servers, or AmazonProvidedDNS.

        The default DHCP option set specifies ``AmazonProvidedDNS`` . If specifying more than one domain name server, specify the IP addresses in a single parameter, separated by commas. To have your instance to receive a custom DNS hostname as specified in ``DomainName`` , you must set this to a custom DNS server.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcpoptions.html#cfn-ec2-dhcpoptions-domainnameservers
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "domainNameServers"))

    @domain_name_servers.setter
    def domain_name_servers(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        jsii.set(self, "domainNameServers", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="netbiosNameServers")
    def netbios_name_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The IPv4 addresses of up to four NetBIOS name servers.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcpoptions.html#cfn-ec2-dhcpoptions-netbiosnameservers
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "netbiosNameServers"))

    @netbios_name_servers.setter
    def netbios_name_servers(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        jsii.set(self, "netbiosNameServers", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="netbiosNodeType")
    def netbios_node_type(self) -> typing.Optional[jsii.Number]:
        '''The NetBIOS node type (1, 2, 4, or 8).

        We recommend that you specify 2 (broadcast and multicast are not currently supported).

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcpoptions.html#cfn-ec2-dhcpoptions-netbiosnodetype
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "netbiosNodeType"))

    @netbios_node_type.setter
    def netbios_node_type(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "netbiosNodeType", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ntpServers")
    def ntp_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The IPv4 addresses of up to four Network Time Protocol (NTP) servers.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcpoptions.html#cfn-ec2-dhcpoptions-ntpservers
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "ntpServers"))

    @ntp_servers.setter
    def ntp_servers(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        jsii.set(self, "ntpServers", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnDHCPOptionsProps",
    jsii_struct_bases=[],
    name_mapping={
        "domain_name": "domainName",
        "domain_name_servers": "domainNameServers",
        "netbios_name_servers": "netbiosNameServers",
        "netbios_node_type": "netbiosNodeType",
        "ntp_servers": "ntpServers",
        "tags": "tags",
    },
)
class CfnDHCPOptionsProps:
    def __init__(
        self,
        *,
        domain_name: typing.Optional[builtins.str] = None,
        domain_name_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        netbios_name_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        netbios_node_type: typing.Optional[jsii.Number] = None,
        ntp_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDHCPOptions``.

        :param domain_name: This value is used to complete unqualified DNS hostnames. If you're using AmazonProvidedDNS in ``us-east-1`` , specify ``ec2.internal`` . If you're using AmazonProvidedDNS in another Region, specify *region* . ``compute.internal`` (for example, ``ap-northeast-1.compute.internal`` ). Otherwise, specify a domain name (for example, *MyCompany.com* ).
        :param domain_name_servers: The IPv4 addresses of up to four domain name servers, or AmazonProvidedDNS. The default DHCP option set specifies ``AmazonProvidedDNS`` . If specifying more than one domain name server, specify the IP addresses in a single parameter, separated by commas. To have your instance to receive a custom DNS hostname as specified in ``DomainName`` , you must set this to a custom DNS server.
        :param netbios_name_servers: The IPv4 addresses of up to four NetBIOS name servers.
        :param netbios_node_type: The NetBIOS node type (1, 2, 4, or 8). We recommend that you specify 2 (broadcast and multicast are not currently supported).
        :param ntp_servers: The IPv4 addresses of up to four Network Time Protocol (NTP) servers.
        :param tags: Any tags assigned to the DHCP options set.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcpoptions.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            cfn_dHCPOptions_props = ec2.CfnDHCPOptionsProps(
                domain_name="domainName",
                domain_name_servers=["domainNameServers"],
                netbios_name_servers=["netbiosNameServers"],
                netbios_node_type=123,
                ntp_servers=["ntpServers"],
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if domain_name is not None:
            self._values["domain_name"] = domain_name
        if domain_name_servers is not None:
            self._values["domain_name_servers"] = domain_name_servers
        if netbios_name_servers is not None:
            self._values["netbios_name_servers"] = netbios_name_servers
        if netbios_node_type is not None:
            self._values["netbios_node_type"] = netbios_node_type
        if ntp_servers is not None:
            self._values["ntp_servers"] = ntp_servers
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def domain_name(self) -> typing.Optional[builtins.str]:
        '''This value is used to complete unqualified DNS hostnames.

        If you're using AmazonProvidedDNS in ``us-east-1`` , specify ``ec2.internal`` . If you're using AmazonProvidedDNS in another Region, specify *region* . ``compute.internal`` (for example, ``ap-northeast-1.compute.internal`` ). Otherwise, specify a domain name (for example, *MyCompany.com* ).

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcpoptions.html#cfn-ec2-dhcpoptions-domainname
        '''
        result = self._values.get("domain_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def domain_name_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The IPv4 addresses of up to four domain name servers, or AmazonProvidedDNS.

        The default DHCP option set specifies ``AmazonProvidedDNS`` . If specifying more than one domain name server, specify the IP addresses in a single parameter, separated by commas. To have your instance to receive a custom DNS hostname as specified in ``DomainName`` , you must set this to a custom DNS server.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcpoptions.html#cfn-ec2-dhcpoptions-domainnameservers
        '''
        result = self._values.get("domain_name_servers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def netbios_name_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The IPv4 addresses of up to four NetBIOS name servers.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcpoptions.html#cfn-ec2-dhcpoptions-netbiosnameservers
        '''
        result = self._values.get("netbios_name_servers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def netbios_node_type(self) -> typing.Optional[jsii.Number]:
        '''The NetBIOS node type (1, 2, 4, or 8).

        We recommend that you specify 2 (broadcast and multicast are not currently supported).

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcpoptions.html#cfn-ec2-dhcpoptions-netbiosnodetype
        '''
        result = self._values.get("netbios_node_type")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def ntp_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The IPv4 addresses of up to four Network Time Protocol (NTP) servers.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcpoptions.html#cfn-ec2-dhcpoptions-ntpservers
        '''
        result = self._values.get("ntp_servers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''Any tags assigned to the DHCP options set.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcpoptions.html#cfn-ec2-dhcpoptions-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDHCPOptionsProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnEC2Fleet(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet",
):
    '''A CloudFormation ``AWS::EC2::EC2Fleet``.

    Specifies the configuration information to launch a fleet--or group--of instances. An EC2 Fleet can launch multiple instance types across multiple Availability Zones, using the On-Demand Instance, Reserved Instance, and Spot Instance purchasing models together. Using EC2 Fleet, you can define separate On-Demand and Spot capacity targets, specify the instance types that work best for your applications, and specify how Amazon EC2 should distribute your fleet capacity within each purchasing model. For more information, see `Launching an EC2 Fleet <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .

    :cloudformationResource: AWS::EC2::EC2Fleet
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ec2 as ec2
        
        cfn_eC2_fleet = ec2.CfnEC2Fleet(self, "MyCfnEC2Fleet",
            launch_template_configs=[ec2.CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty(
                launch_template_specification=ec2.CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty(
                    launch_template_id="launchTemplateId",
                    launch_template_name="launchTemplateName",
                    version="version"
                ),
                overrides=[ec2.CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty(
                    availability_zone="availabilityZone",
                    instance_requirements=ec2.CfnEC2Fleet.InstanceRequirementsRequestProperty(
                        accelerator_count=ec2.CfnEC2Fleet.AcceleratorCountRequestProperty(
                            max=123,
                            min=123
                        ),
                        accelerator_manufacturers=["acceleratorManufacturers"],
                        accelerator_names=["acceleratorNames"],
                        accelerator_total_memory_mi_b=ec2.CfnEC2Fleet.AcceleratorTotalMemoryMiBRequestProperty(
                            max=123,
                            min=123
                        ),
                        accelerator_types=["acceleratorTypes"],
                        bare_metal="bareMetal",
                        baseline_ebs_bandwidth_mbps=ec2.CfnEC2Fleet.BaselineEbsBandwidthMbpsRequestProperty(
                            max=123,
                            min=123
                        ),
                        burstable_performance="burstablePerformance",
                        cpu_manufacturers=["cpuManufacturers"],
                        excluded_instance_types=["excludedInstanceTypes"],
                        instance_generations=["instanceGenerations"],
                        local_storage="localStorage",
                        local_storage_types=["localStorageTypes"],
                        memory_gi_bPer_vCpu=ec2.CfnEC2Fleet.MemoryGiBPerVCpuRequestProperty(
                            max=123,
                            min=123
                        ),
                        memory_mi_b=ec2.CfnEC2Fleet.MemoryMiBRequestProperty(
                            max=123,
                            min=123
                        ),
                        network_interface_count=ec2.CfnEC2Fleet.NetworkInterfaceCountRequestProperty(
                            max=123,
                            min=123
                        ),
                        on_demand_max_price_percentage_over_lowest_price=123,
                        require_hibernate_support=False,
                        spot_max_price_percentage_over_lowest_price=123,
                        total_local_storage_gb=ec2.CfnEC2Fleet.TotalLocalStorageGBRequestProperty(
                            max=123,
                            min=123
                        ),
                        v_cpu_count=ec2.CfnEC2Fleet.VCpuCountRangeRequestProperty(
                            max=123,
                            min=123
                        )
                    ),
                    instance_type="instanceType",
                    max_price="maxPrice",
                    placement=ec2.CfnEC2Fleet.PlacementProperty(
                        affinity="affinity",
                        availability_zone="availabilityZone",
                        group_name="groupName",
                        host_id="hostId",
                        host_resource_group_arn="hostResourceGroupArn",
                        partition_number=123,
                        spread_domain="spreadDomain",
                        tenancy="tenancy"
                    ),
                    priority=123,
                    subnet_id="subnetId",
                    weighted_capacity=123
                )]
            )],
            target_capacity_specification=ec2.CfnEC2Fleet.TargetCapacitySpecificationRequestProperty(
                total_target_capacity=123,
        
                # the properties below are optional
                default_target_capacity_type="defaultTargetCapacityType",
                on_demand_target_capacity=123,
                spot_target_capacity=123,
                target_capacity_unit_type="targetCapacityUnitType"
            ),
        
            # the properties below are optional
            context="context",
            excess_capacity_termination_policy="excessCapacityTerminationPolicy",
            on_demand_options=ec2.CfnEC2Fleet.OnDemandOptionsRequestProperty(
                allocation_strategy="allocationStrategy",
                capacity_reservation_options=ec2.CfnEC2Fleet.CapacityReservationOptionsRequestProperty(
                    usage_strategy="usageStrategy"
                ),
                max_total_price="maxTotalPrice",
                min_target_capacity=123,
                single_availability_zone=False,
                single_instance_type=False
            ),
            replace_unhealthy_instances=False,
            spot_options=ec2.CfnEC2Fleet.SpotOptionsRequestProperty(
                allocation_strategy="allocationStrategy",
                instance_interruption_behavior="instanceInterruptionBehavior",
                instance_pools_to_use_count=123,
                maintenance_strategies=ec2.CfnEC2Fleet.MaintenanceStrategiesProperty(
                    capacity_rebalance=ec2.CfnEC2Fleet.CapacityRebalanceProperty(
                        replacement_strategy="replacementStrategy",
                        termination_delay=123
                    )
                ),
                max_total_price="maxTotalPrice",
                min_target_capacity=123,
                single_availability_zone=False,
                single_instance_type=False
            ),
            tag_specifications=[ec2.CfnEC2Fleet.TagSpecificationProperty(
                resource_type="resourceType",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )],
            terminate_instances_with_expiration=False,
            type="type",
            valid_from="validFrom",
            valid_until="validUntil"
        )
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        launch_template_configs: typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty"]]],
        target_capacity_specification: typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.TargetCapacitySpecificationRequestProperty"],
        context: typing.Optional[builtins.str] = None,
        excess_capacity_termination_policy: typing.Optional[builtins.str] = None,
        on_demand_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.OnDemandOptionsRequestProperty"]] = None,
        replace_unhealthy_instances: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        spot_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.SpotOptionsRequestProperty"]] = None,
        tag_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.TagSpecificationProperty"]]]] = None,
        terminate_instances_with_expiration: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        type: typing.Optional[builtins.str] = None,
        valid_from: typing.Optional[builtins.str] = None,
        valid_until: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::EC2Fleet``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param launch_template_configs: The configuration for the EC2 Fleet.
        :param target_capacity_specification: The number of units to request.
        :param context: Reserved.
        :param excess_capacity_termination_policy: Indicates whether running instances should be terminated if the total target capacity of the EC2 Fleet is decreased below the current size of the EC2 Fleet.
        :param on_demand_options: Describes the configuration of On-Demand Instances in an EC2 Fleet.
        :param replace_unhealthy_instances: Indicates whether EC2 Fleet should replace unhealthy Spot Instances. Supported only for fleets of type ``maintain`` . For more information, see `EC2 Fleet health checks <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/manage-ec2-fleet.html#ec2-fleet-health-checks>`_ in the *Amazon EC2 User Guide* .
        :param spot_options: Describes the configuration of Spot Instances in an EC2 Fleet.
        :param tag_specifications: The key-value pair for tagging the EC2 Fleet request on creation. For more information, see `Tagging your resources <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Using_Tags.html#tag-resources>`_ . If the fleet type is ``instant`` , specify a resource type of ``fleet`` to tag the fleet or ``instance`` to tag the instances at launch. If the fleet type is ``maintain`` or ``request`` , specify a resource type of ``fleet`` to tag the fleet. You cannot specify a resource type of ``instance`` . To tag instances at launch, specify the tags in a `launch template <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html#create-launch-template>`_ .
        :param terminate_instances_with_expiration: Indicates whether running instances should be terminated when the EC2 Fleet expires.
        :param type: The fleet type. The default value is ``maintain`` . - ``maintain`` - The EC2 Fleet places an asynchronous request for your desired capacity, and continues to maintain your desired Spot capacity by replenishing interrupted Spot Instances. - ``request`` - The EC2 Fleet places an asynchronous one-time request for your desired capacity, but does submit Spot requests in alternative capacity pools if Spot capacity is unavailable, and does not maintain Spot capacity if Spot Instances are interrupted. - ``instant`` - The EC2 Fleet places a synchronous one-time request for your desired capacity, and returns errors for any instances that could not be launched. For more information, see `EC2 Fleet request types <https://docs.aws.amazon.com/https://docs.aws.amazon.com/ec2-fleet-request-type.html>`_ in the *Amazon EC2 User Guide* .
        :param valid_from: The start date and time of the request, in UTC format (for example, *YYYY* - *MM* - *DD* T *HH* : *MM* : *SS* Z). The default is to start fulfilling the request immediately.
        :param valid_until: The end date and time of the request, in UTC format (for example, *YYYY* - *MM* - *DD* T *HH* : *MM* : *SS* Z). At this point, no new EC2 Fleet requests are placed or able to fulfill the request. If no value is specified, the request remains until you cancel it.
        '''
        props = CfnEC2FleetProps(
            launch_template_configs=launch_template_configs,
            target_capacity_specification=target_capacity_specification,
            context=context,
            excess_capacity_termination_policy=excess_capacity_termination_policy,
            on_demand_options=on_demand_options,
            replace_unhealthy_instances=replace_unhealthy_instances,
            spot_options=spot_options,
            tag_specifications=tag_specifications,
            terminate_instances_with_expiration=terminate_instances_with_expiration,
            type=type,
            valid_from=valid_from,
            valid_until=valid_until,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrFleetId")
    def attr_fleet_id(self) -> builtins.str:
        '''The ID of the EC2 Fleet.

        :cloudformationAttribute: FleetId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrFleetId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="launchTemplateConfigs")
    def launch_template_configs(
        self,
    ) -> typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty"]]]:
        '''The configuration for the EC2 Fleet.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-launchtemplateconfigs
        '''
        return typing.cast(typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty"]]], jsii.get(self, "launchTemplateConfigs"))

    @launch_template_configs.setter
    def launch_template_configs(
        self,
        value: typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty"]]],
    ) -> None:
        jsii.set(self, "launchTemplateConfigs", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="targetCapacitySpecification")
    def target_capacity_specification(
        self,
    ) -> typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.TargetCapacitySpecificationRequestProperty"]:
        '''The number of units to request.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-targetcapacityspecification
        '''
        return typing.cast(typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.TargetCapacitySpecificationRequestProperty"], jsii.get(self, "targetCapacitySpecification"))

    @target_capacity_specification.setter
    def target_capacity_specification(
        self,
        value: typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.TargetCapacitySpecificationRequestProperty"],
    ) -> None:
        jsii.set(self, "targetCapacitySpecification", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="context")
    def context(self) -> typing.Optional[builtins.str]:
        '''Reserved.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-context
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "context"))

    @context.setter
    def context(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "context", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="excessCapacityTerminationPolicy")
    def excess_capacity_termination_policy(self) -> typing.Optional[builtins.str]:
        '''Indicates whether running instances should be terminated if the total target capacity of the EC2 Fleet is decreased below the current size of the EC2 Fleet.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-excesscapacityterminationpolicy
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "excessCapacityTerminationPolicy"))

    @excess_capacity_termination_policy.setter
    def excess_capacity_termination_policy(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        jsii.set(self, "excessCapacityTerminationPolicy", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="onDemandOptions")
    def on_demand_options(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.OnDemandOptionsRequestProperty"]]:
        '''Describes the configuration of On-Demand Instances in an EC2 Fleet.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-ondemandoptions
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.OnDemandOptionsRequestProperty"]], jsii.get(self, "onDemandOptions"))

    @on_demand_options.setter
    def on_demand_options(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.OnDemandOptionsRequestProperty"]],
    ) -> None:
        jsii.set(self, "onDemandOptions", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="replaceUnhealthyInstances")
    def replace_unhealthy_instances(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''Indicates whether EC2 Fleet should replace unhealthy Spot Instances.

        Supported only for fleets of type ``maintain`` . For more information, see `EC2 Fleet health checks <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/manage-ec2-fleet.html#ec2-fleet-health-checks>`_ in the *Amazon EC2 User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-replaceunhealthyinstances
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "replaceUnhealthyInstances"))

    @replace_unhealthy_instances.setter
    def replace_unhealthy_instances(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "replaceUnhealthyInstances", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="spotOptions")
    def spot_options(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.SpotOptionsRequestProperty"]]:
        '''Describes the configuration of Spot Instances in an EC2 Fleet.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-spotoptions
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.SpotOptionsRequestProperty"]], jsii.get(self, "spotOptions"))

    @spot_options.setter
    def spot_options(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.SpotOptionsRequestProperty"]],
    ) -> None:
        jsii.set(self, "spotOptions", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tagSpecifications")
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.TagSpecificationProperty"]]]]:
        '''The key-value pair for tagging the EC2 Fleet request on creation. For more information, see `Tagging your resources <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Using_Tags.html#tag-resources>`_ .

        If the fleet type is ``instant`` , specify a resource type of ``fleet`` to tag the fleet or ``instance`` to tag the instances at launch.

        If the fleet type is ``maintain`` or ``request`` , specify a resource type of ``fleet`` to tag the fleet. You cannot specify a resource type of ``instance`` . To tag instances at launch, specify the tags in a `launch template <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html#create-launch-template>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-tagspecifications
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.TagSpecificationProperty"]]]], jsii.get(self, "tagSpecifications"))

    @tag_specifications.setter
    def tag_specifications(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.TagSpecificationProperty"]]]],
    ) -> None:
        jsii.set(self, "tagSpecifications", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="terminateInstancesWithExpiration")
    def terminate_instances_with_expiration(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''Indicates whether running instances should be terminated when the EC2 Fleet expires.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-terminateinstanceswithexpiration
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "terminateInstancesWithExpiration"))

    @terminate_instances_with_expiration.setter
    def terminate_instances_with_expiration(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "terminateInstancesWithExpiration", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="type")
    def type(self) -> typing.Optional[builtins.str]:
        '''The fleet type. The default value is ``maintain`` .

        - ``maintain`` - The EC2 Fleet places an asynchronous request for your desired capacity, and continues to maintain your desired Spot capacity by replenishing interrupted Spot Instances.
        - ``request`` - The EC2 Fleet places an asynchronous one-time request for your desired capacity, but does submit Spot requests in alternative capacity pools if Spot capacity is unavailable, and does not maintain Spot capacity if Spot Instances are interrupted.
        - ``instant`` - The EC2 Fleet places a synchronous one-time request for your desired capacity, and returns errors for any instances that could not be launched.

        For more information, see `EC2 Fleet request types <https://docs.aws.amazon.com/https://docs.aws.amazon.com/ec2-fleet-request-type.html>`_ in the *Amazon EC2 User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-type
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "type"))

    @type.setter
    def type(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "type", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="validFrom")
    def valid_from(self) -> typing.Optional[builtins.str]:
        '''The start date and time of the request, in UTC format (for example, *YYYY* - *MM* - *DD* T *HH* : *MM* : *SS* Z).

        The default is to start fulfilling the request immediately.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-validfrom
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "validFrom"))

    @valid_from.setter
    def valid_from(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "validFrom", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="validUntil")
    def valid_until(self) -> typing.Optional[builtins.str]:
        '''The end date and time of the request, in UTC format (for example, *YYYY* - *MM* - *DD* T *HH* : *MM* : *SS* Z).

        At this point, no new EC2 Fleet requests are placed or able to fulfill the request. If no value is specified, the request remains until you cancel it.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-validuntil
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "validUntil"))

    @valid_until.setter
    def valid_until(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "validUntil", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.AcceleratorCountRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class AcceleratorCountRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The minimum and maximum number of accelerators (GPUs, FPGAs, or AWS Inferentia chips) on an instance.

            To exclude accelerator-enabled instance types, set ``Max`` to ``0`` .

            :param max: The maximum number of accelerators. To specify no maximum limit, omit this parameter. To exclude accelerator-enabled instance types, set ``Max`` to ``0`` .
            :param min: The minimum number of accelerators. To specify no minimum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-acceleratorcountrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                accelerator_count_request_property = ec2.CfnEC2Fleet.AcceleratorCountRequestProperty(
                    max=123,
                    min=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The maximum number of accelerators.

            To specify no maximum limit, omit this parameter. To exclude accelerator-enabled instance types, set ``Max`` to ``0`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-acceleratorcountrequest.html#cfn-ec2-ec2fleet-acceleratorcountrequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The minimum number of accelerators.

            To specify no minimum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-acceleratorcountrequest.html#cfn-ec2-ec2fleet-acceleratorcountrequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AcceleratorCountRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.AcceleratorTotalMemoryMiBRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class AcceleratorTotalMemoryMiBRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The minimum and maximum amount of total accelerator memory, in MiB.

            :param max: The maximum amount of accelerator memory, in MiB. To specify no maximum limit, omit this parameter.
            :param min: The minimum amount of accelerator memory, in MiB. To specify no minimum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-acceleratortotalmemorymibrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                accelerator_total_memory_mi_bRequest_property = ec2.CfnEC2Fleet.AcceleratorTotalMemoryMiBRequestProperty(
                    max=123,
                    min=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The maximum amount of accelerator memory, in MiB.

            To specify no maximum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-acceleratortotalmemorymibrequest.html#cfn-ec2-ec2fleet-acceleratortotalmemorymibrequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The minimum amount of accelerator memory, in MiB.

            To specify no minimum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-acceleratortotalmemorymibrequest.html#cfn-ec2-ec2fleet-acceleratortotalmemorymibrequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AcceleratorTotalMemoryMiBRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.BaselineEbsBandwidthMbpsRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class BaselineEbsBandwidthMbpsRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The minimum and maximum baseline bandwidth to Amazon EBS, in Mbps.

            For more information, see `Amazon EBS–optimized instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-optimized.html>`_ in the *Amazon EC2 User Guide* .

            :param max: The maximum baseline bandwidth, in Mbps. To specify no maximum limit, omit this parameter.
            :param min: The minimum baseline bandwidth, in Mbps. To specify no minimum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-baselineebsbandwidthmbpsrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                baseline_ebs_bandwidth_mbps_request_property = ec2.CfnEC2Fleet.BaselineEbsBandwidthMbpsRequestProperty(
                    max=123,
                    min=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The maximum baseline bandwidth, in Mbps.

            To specify no maximum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-baselineebsbandwidthmbpsrequest.html#cfn-ec2-ec2fleet-baselineebsbandwidthmbpsrequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The minimum baseline bandwidth, in Mbps.

            To specify no minimum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-baselineebsbandwidthmbpsrequest.html#cfn-ec2-ec2fleet-baselineebsbandwidthmbpsrequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BaselineEbsBandwidthMbpsRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.CapacityRebalanceProperty",
        jsii_struct_bases=[],
        name_mapping={
            "replacement_strategy": "replacementStrategy",
            "termination_delay": "terminationDelay",
        },
    )
    class CapacityRebalanceProperty:
        def __init__(
            self,
            *,
            replacement_strategy: typing.Optional[builtins.str] = None,
            termination_delay: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The Spot Instance replacement strategy to use when Amazon EC2 emits a rebalance notification signal that your Spot Instance is at an elevated risk of being interrupted.

            For more information, see `Capacity rebalancing <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet-capacity-rebalance.html>`_ in the *Amazon EC2 User Guide* .

            :param replacement_strategy: The replacement strategy to use. Only available for fleets of type ``maintain`` . ``launch`` - EC2 Fleet launches a replacement Spot Instance when a rebalance notification is emitted for an existing Spot Instance in the fleet. EC2 Fleet does not terminate the instances that receive a rebalance notification. You can terminate the old instances, or you can leave them running. You are charged for all instances while they are running. ``launch-before-terminate`` - EC2 Fleet launches a replacement Spot Instance when a rebalance notification is emitted for an existing Spot Instance in the fleet, and then, after a delay that you specify (in ``TerminationDelay`` ), terminates the instances that received a rebalance notification.
            :param termination_delay: The amount of time (in seconds) that Amazon EC2 waits before terminating the old Spot Instance after launching a new replacement Spot Instance. Required when ``ReplacementStrategy`` is set to ``launch-before-terminate`` . Not valid when ``ReplacementStrategy`` is set to ``launch`` . Valid values: Minimum value of ``120`` seconds. Maximum value of ``7200`` seconds.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-capacityrebalance.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                capacity_rebalance_property = ec2.CfnEC2Fleet.CapacityRebalanceProperty(
                    replacement_strategy="replacementStrategy",
                    termination_delay=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if replacement_strategy is not None:
                self._values["replacement_strategy"] = replacement_strategy
            if termination_delay is not None:
                self._values["termination_delay"] = termination_delay

        @builtins.property
        def replacement_strategy(self) -> typing.Optional[builtins.str]:
            '''The replacement strategy to use. Only available for fleets of type ``maintain`` .

            ``launch`` - EC2 Fleet launches a replacement Spot Instance when a rebalance notification is emitted for an existing Spot Instance in the fleet. EC2 Fleet does not terminate the instances that receive a rebalance notification. You can terminate the old instances, or you can leave them running. You are charged for all instances while they are running.

            ``launch-before-terminate`` - EC2 Fleet launches a replacement Spot Instance when a rebalance notification is emitted for an existing Spot Instance in the fleet, and then, after a delay that you specify (in ``TerminationDelay`` ), terminates the instances that received a rebalance notification.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-capacityrebalance.html#cfn-ec2-ec2fleet-capacityrebalance-replacementstrategy
            '''
            result = self._values.get("replacement_strategy")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def termination_delay(self) -> typing.Optional[jsii.Number]:
            '''The amount of time (in seconds) that Amazon EC2 waits before terminating the old Spot Instance after launching a new replacement Spot Instance.

            Required when ``ReplacementStrategy`` is set to ``launch-before-terminate`` .

            Not valid when ``ReplacementStrategy`` is set to ``launch`` .

            Valid values: Minimum value of ``120`` seconds. Maximum value of ``7200`` seconds.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-capacityrebalance.html#cfn-ec2-ec2fleet-capacityrebalance-terminationdelay
            '''
            result = self._values.get("termination_delay")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CapacityRebalanceProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.CapacityReservationOptionsRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"usage_strategy": "usageStrategy"},
    )
    class CapacityReservationOptionsRequestProperty:
        def __init__(
            self,
            *,
            usage_strategy: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Describes the strategy for using unused Capacity Reservations for fulfilling On-Demand capacity.

            .. epigraph::

               This strategy can only be used if the EC2 Fleet is of type ``instant`` .

            For more information about Capacity Reservations, see `On-Demand Capacity Reservations <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-capacity-reservations.html>`_ in the *Amazon EC2 User Guide* . For examples of using Capacity Reservations in an EC2 Fleet, see `EC2 Fleet example configurations <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet-examples.html>`_ in the *Amazon EC2 User Guide* .

            :param usage_strategy: Indicates whether to use unused Capacity Reservations for fulfilling On-Demand capacity. If you specify ``use-capacity-reservations-first`` , the fleet uses unused Capacity Reservations to fulfill On-Demand capacity up to the target On-Demand capacity. If multiple instance pools have unused Capacity Reservations, the On-Demand allocation strategy ( ``lowest-price`` or ``prioritized`` ) is applied. If the number of unused Capacity Reservations is less than the On-Demand target capacity, the remaining On-Demand target capacity is launched according to the On-Demand allocation strategy ( ``lowest-price`` or ``prioritized`` ). If you do not specify a value, the fleet fulfils the On-Demand capacity according to the chosen On-Demand allocation strategy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-capacityreservationoptionsrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                capacity_reservation_options_request_property = ec2.CfnEC2Fleet.CapacityReservationOptionsRequestProperty(
                    usage_strategy="usageStrategy"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if usage_strategy is not None:
                self._values["usage_strategy"] = usage_strategy

        @builtins.property
        def usage_strategy(self) -> typing.Optional[builtins.str]:
            '''Indicates whether to use unused Capacity Reservations for fulfilling On-Demand capacity.

            If you specify ``use-capacity-reservations-first`` , the fleet uses unused Capacity Reservations to fulfill On-Demand capacity up to the target On-Demand capacity. If multiple instance pools have unused Capacity Reservations, the On-Demand allocation strategy ( ``lowest-price`` or ``prioritized`` ) is applied. If the number of unused Capacity Reservations is less than the On-Demand target capacity, the remaining On-Demand target capacity is launched according to the On-Demand allocation strategy ( ``lowest-price`` or ``prioritized`` ).

            If you do not specify a value, the fleet fulfils the On-Demand capacity according to the chosen On-Demand allocation strategy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-capacityreservationoptionsrequest.html#cfn-ec2-ec2fleet-capacityreservationoptionsrequest-usagestrategy
            '''
            result = self._values.get("usage_strategy")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CapacityReservationOptionsRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "launch_template_specification": "launchTemplateSpecification",
            "overrides": "overrides",
        },
    )
    class FleetLaunchTemplateConfigRequestProperty:
        def __init__(
            self,
            *,
            launch_template_specification: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty"]] = None,
            overrides: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty"]]]] = None,
        ) -> None:
            '''Specifies a launch template and overrides for an EC2 Fleet.

            ``FleetLaunchTemplateConfigRequest`` is a property of the `AWS::EC2::EC2Fleet <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html>`_ resource.

            :param launch_template_specification: The launch template to use. You must specify either the launch template ID or launch template name in the request.
            :param overrides: Any parameters that you specify override the same parameters in the launch template. For fleets of type ``request`` and ``maintain`` , a maximum of 300 items is allowed across all launch templates.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateconfigrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                fleet_launch_template_config_request_property = ec2.CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty(
                    launch_template_specification=ec2.CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty(
                        launch_template_id="launchTemplateId",
                        launch_template_name="launchTemplateName",
                        version="version"
                    ),
                    overrides=[ec2.CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty(
                        availability_zone="availabilityZone",
                        instance_requirements=ec2.CfnEC2Fleet.InstanceRequirementsRequestProperty(
                            accelerator_count=ec2.CfnEC2Fleet.AcceleratorCountRequestProperty(
                                max=123,
                                min=123
                            ),
                            accelerator_manufacturers=["acceleratorManufacturers"],
                            accelerator_names=["acceleratorNames"],
                            accelerator_total_memory_mi_b=ec2.CfnEC2Fleet.AcceleratorTotalMemoryMiBRequestProperty(
                                max=123,
                                min=123
                            ),
                            accelerator_types=["acceleratorTypes"],
                            bare_metal="bareMetal",
                            baseline_ebs_bandwidth_mbps=ec2.CfnEC2Fleet.BaselineEbsBandwidthMbpsRequestProperty(
                                max=123,
                                min=123
                            ),
                            burstable_performance="burstablePerformance",
                            cpu_manufacturers=["cpuManufacturers"],
                            excluded_instance_types=["excludedInstanceTypes"],
                            instance_generations=["instanceGenerations"],
                            local_storage="localStorage",
                            local_storage_types=["localStorageTypes"],
                            memory_gi_bPer_vCpu=ec2.CfnEC2Fleet.MemoryGiBPerVCpuRequestProperty(
                                max=123,
                                min=123
                            ),
                            memory_mi_b=ec2.CfnEC2Fleet.MemoryMiBRequestProperty(
                                max=123,
                                min=123
                            ),
                            network_interface_count=ec2.CfnEC2Fleet.NetworkInterfaceCountRequestProperty(
                                max=123,
                                min=123
                            ),
                            on_demand_max_price_percentage_over_lowest_price=123,
                            require_hibernate_support=False,
                            spot_max_price_percentage_over_lowest_price=123,
                            total_local_storage_gb=ec2.CfnEC2Fleet.TotalLocalStorageGBRequestProperty(
                                max=123,
                                min=123
                            ),
                            v_cpu_count=ec2.CfnEC2Fleet.VCpuCountRangeRequestProperty(
                                max=123,
                                min=123
                            )
                        ),
                        instance_type="instanceType",
                        max_price="maxPrice",
                        placement=ec2.CfnEC2Fleet.PlacementProperty(
                            affinity="affinity",
                            availability_zone="availabilityZone",
                            group_name="groupName",
                            host_id="hostId",
                            host_resource_group_arn="hostResourceGroupArn",
                            partition_number=123,
                            spread_domain="spreadDomain",
                            tenancy="tenancy"
                        ),
                        priority=123,
                        subnet_id="subnetId",
                        weighted_capacity=123
                    )]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if launch_template_specification is not None:
                self._values["launch_template_specification"] = launch_template_specification
            if overrides is not None:
                self._values["overrides"] = overrides

        @builtins.property
        def launch_template_specification(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty"]]:
            '''The launch template to use.

            You must specify either the launch template ID or launch template name in the request.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateconfigrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateconfigrequest-launchtemplatespecification
            '''
            result = self._values.get("launch_template_specification")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty"]], result)

        @builtins.property
        def overrides(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty"]]]]:
            '''Any parameters that you specify override the same parameters in the launch template.

            For fleets of type ``request`` and ``maintain`` , a maximum of 300 items is allowed across all launch templates.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateconfigrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateconfigrequest-overrides
            '''
            result = self._values.get("overrides")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FleetLaunchTemplateConfigRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "availability_zone": "availabilityZone",
            "instance_requirements": "instanceRequirements",
            "instance_type": "instanceType",
            "max_price": "maxPrice",
            "placement": "placement",
            "priority": "priority",
            "subnet_id": "subnetId",
            "weighted_capacity": "weightedCapacity",
        },
    )
    class FleetLaunchTemplateOverridesRequestProperty:
        def __init__(
            self,
            *,
            availability_zone: typing.Optional[builtins.str] = None,
            instance_requirements: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.InstanceRequirementsRequestProperty"]] = None,
            instance_type: typing.Optional[builtins.str] = None,
            max_price: typing.Optional[builtins.str] = None,
            placement: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.PlacementProperty"]] = None,
            priority: typing.Optional[jsii.Number] = None,
            subnet_id: typing.Optional[builtins.str] = None,
            weighted_capacity: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''Specifies overrides for a launch template for an EC2 Fleet.

            ``FleetLaunchTemplateOverridesRequest`` is a property of the `FleetLaunchTemplateConfigRequest <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateconfigrequest.html>`_ property type.

            :param availability_zone: The Availability Zone in which to launch the instances.
            :param instance_requirements: The attributes for the instance types. When you specify instance attributes, Amazon EC2 will identify instance types with those attributes. .. epigraph:: If you specify ``InstanceRequirements`` , you can't specify ``InstanceTypes`` .
            :param instance_type: The instance type. .. epigraph:: If you specify ``InstanceTypes`` , you can't specify ``InstanceRequirements`` .
            :param max_price: The maximum price per unit hour that you are willing to pay for a Spot Instance.
            :param placement: The location where the instance launched, if applicable.
            :param priority: The priority for the launch template override. The highest priority is launched first. If the On-Demand ``AllocationStrategy`` is set to ``prioritized`` , EC2 Fleet uses priority to determine which launch template override to use first in fulfilling On-Demand capacity. If the Spot ``AllocationStrategy`` is set to ``capacity-optimized-prioritized`` , EC2 Fleet uses priority on a best-effort basis to determine which launch template override to use in fulfilling Spot capacity, but optimizes for capacity first. Valid values are whole numbers starting at ``0`` . The lower the number, the higher the priority. If no number is set, the launch template override has the lowest priority. You can set the same priority for different launch template overrides.
            :param subnet_id: The IDs of the subnets in which to launch the instances. Separate multiple subnet IDs using commas (for example, ``subnet-1234abcdeexample1, subnet-0987cdef6example2`` ). A request of type ``instant`` can have only one subnet ID.
            :param weighted_capacity: The number of units provided by the specified instance type.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                fleet_launch_template_overrides_request_property = ec2.CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty(
                    availability_zone="availabilityZone",
                    instance_requirements=ec2.CfnEC2Fleet.InstanceRequirementsRequestProperty(
                        accelerator_count=ec2.CfnEC2Fleet.AcceleratorCountRequestProperty(
                            max=123,
                            min=123
                        ),
                        accelerator_manufacturers=["acceleratorManufacturers"],
                        accelerator_names=["acceleratorNames"],
                        accelerator_total_memory_mi_b=ec2.CfnEC2Fleet.AcceleratorTotalMemoryMiBRequestProperty(
                            max=123,
                            min=123
                        ),
                        accelerator_types=["acceleratorTypes"],
                        bare_metal="bareMetal",
                        baseline_ebs_bandwidth_mbps=ec2.CfnEC2Fleet.BaselineEbsBandwidthMbpsRequestProperty(
                            max=123,
                            min=123
                        ),
                        burstable_performance="burstablePerformance",
                        cpu_manufacturers=["cpuManufacturers"],
                        excluded_instance_types=["excludedInstanceTypes"],
                        instance_generations=["instanceGenerations"],
                        local_storage="localStorage",
                        local_storage_types=["localStorageTypes"],
                        memory_gi_bPer_vCpu=ec2.CfnEC2Fleet.MemoryGiBPerVCpuRequestProperty(
                            max=123,
                            min=123
                        ),
                        memory_mi_b=ec2.CfnEC2Fleet.MemoryMiBRequestProperty(
                            max=123,
                            min=123
                        ),
                        network_interface_count=ec2.CfnEC2Fleet.NetworkInterfaceCountRequestProperty(
                            max=123,
                            min=123
                        ),
                        on_demand_max_price_percentage_over_lowest_price=123,
                        require_hibernate_support=False,
                        spot_max_price_percentage_over_lowest_price=123,
                        total_local_storage_gb=ec2.CfnEC2Fleet.TotalLocalStorageGBRequestProperty(
                            max=123,
                            min=123
                        ),
                        v_cpu_count=ec2.CfnEC2Fleet.VCpuCountRangeRequestProperty(
                            max=123,
                            min=123
                        )
                    ),
                    instance_type="instanceType",
                    max_price="maxPrice",
                    placement=ec2.CfnEC2Fleet.PlacementProperty(
                        affinity="affinity",
                        availability_zone="availabilityZone",
                        group_name="groupName",
                        host_id="hostId",
                        host_resource_group_arn="hostResourceGroupArn",
                        partition_number=123,
                        spread_domain="spreadDomain",
                        tenancy="tenancy"
                    ),
                    priority=123,
                    subnet_id="subnetId",
                    weighted_capacity=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if availability_zone is not None:
                self._values["availability_zone"] = availability_zone
            if instance_requirements is not None:
                self._values["instance_requirements"] = instance_requirements
            if instance_type is not None:
                self._values["instance_type"] = instance_type
            if max_price is not None:
                self._values["max_price"] = max_price
            if placement is not None:
                self._values["placement"] = placement
            if priority is not None:
                self._values["priority"] = priority
            if subnet_id is not None:
                self._values["subnet_id"] = subnet_id
            if weighted_capacity is not None:
                self._values["weighted_capacity"] = weighted_capacity

        @builtins.property
        def availability_zone(self) -> typing.Optional[builtins.str]:
            '''The Availability Zone in which to launch the instances.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest-availabilityzone
            '''
            result = self._values.get("availability_zone")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def instance_requirements(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.InstanceRequirementsRequestProperty"]]:
            '''The attributes for the instance types.

            When you specify instance attributes, Amazon EC2 will identify instance types with those attributes.
            .. epigraph::

               If you specify ``InstanceRequirements`` , you can't specify ``InstanceTypes`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest-instancerequirements
            '''
            result = self._values.get("instance_requirements")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.InstanceRequirementsRequestProperty"]], result)

        @builtins.property
        def instance_type(self) -> typing.Optional[builtins.str]:
            '''The instance type.

            .. epigraph::

               If you specify ``InstanceTypes`` , you can't specify ``InstanceRequirements`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest-instancetype
            '''
            result = self._values.get("instance_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def max_price(self) -> typing.Optional[builtins.str]:
            '''The maximum price per unit hour that you are willing to pay for a Spot Instance.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest-maxprice
            '''
            result = self._values.get("max_price")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def placement(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.PlacementProperty"]]:
            '''The location where the instance launched, if applicable.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest-placement
            '''
            result = self._values.get("placement")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.PlacementProperty"]], result)

        @builtins.property
        def priority(self) -> typing.Optional[jsii.Number]:
            '''The priority for the launch template override. The highest priority is launched first.

            If the On-Demand ``AllocationStrategy`` is set to ``prioritized`` , EC2 Fleet uses priority to determine which launch template override to use first in fulfilling On-Demand capacity.

            If the Spot ``AllocationStrategy`` is set to ``capacity-optimized-prioritized`` , EC2 Fleet uses priority on a best-effort basis to determine which launch template override to use in fulfilling Spot capacity, but optimizes for capacity first.

            Valid values are whole numbers starting at ``0`` . The lower the number, the higher the priority. If no number is set, the launch template override has the lowest priority. You can set the same priority for different launch template overrides.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest-priority
            '''
            result = self._values.get("priority")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def subnet_id(self) -> typing.Optional[builtins.str]:
            '''The IDs of the subnets in which to launch the instances.

            Separate multiple subnet IDs using commas (for example, ``subnet-1234abcdeexample1, subnet-0987cdef6example2`` ). A request of type ``instant`` can have only one subnet ID.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest-subnetid
            '''
            result = self._values.get("subnet_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def weighted_capacity(self) -> typing.Optional[jsii.Number]:
            '''The number of units provided by the specified instance type.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest-weightedcapacity
            '''
            result = self._values.get("weighted_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FleetLaunchTemplateOverridesRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "launch_template_id": "launchTemplateId",
            "launch_template_name": "launchTemplateName",
            "version": "version",
        },
    )
    class FleetLaunchTemplateSpecificationRequestProperty:
        def __init__(
            self,
            *,
            launch_template_id: typing.Optional[builtins.str] = None,
            launch_template_name: typing.Optional[builtins.str] = None,
            version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Specifies the launch template to use for an EC2 Fleet.

            You must specify either the launch template ID or launch template name in the request.

            ``FleetLaunchTemplateSpecificationRequest`` is a property of the `FleetLaunchTemplateConfigRequest <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateconfigrequest.html>`_ property type.

            :param launch_template_id: The ID of the launch template. If you specify the template ID, you can't specify the template name.
            :param launch_template_name: The name of the launch template. If you specify the template name, you can't specify the template ID.
            :param version: The launch template version number, ``$Latest`` , or ``$Default`` . You must specify a value, otherwise the request fails. If the value is ``$Latest`` , Amazon EC2 uses the latest version of the launch template. If the value is ``$Default`` , Amazon EC2 uses the default version of the launch template.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplatespecificationrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                fleet_launch_template_specification_request_property = ec2.CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty(
                    launch_template_id="launchTemplateId",
                    launch_template_name="launchTemplateName",
                    version="version"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if launch_template_id is not None:
                self._values["launch_template_id"] = launch_template_id
            if launch_template_name is not None:
                self._values["launch_template_name"] = launch_template_name
            if version is not None:
                self._values["version"] = version

        @builtins.property
        def launch_template_id(self) -> typing.Optional[builtins.str]:
            '''The ID of the launch template.

            If you specify the template ID, you can't specify the template name.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplatespecificationrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplatespecificationrequest-launchtemplateid
            '''
            result = self._values.get("launch_template_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def launch_template_name(self) -> typing.Optional[builtins.str]:
            '''The name of the launch template.

            If you specify the template name, you can't specify the template ID.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplatespecificationrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplatespecificationrequest-launchtemplatename
            '''
            result = self._values.get("launch_template_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def version(self) -> typing.Optional[builtins.str]:
            '''The launch template version number, ``$Latest`` , or ``$Default`` . You must specify a value, otherwise the request fails.

            If the value is ``$Latest`` , Amazon EC2 uses the latest version of the launch template.

            If the value is ``$Default`` , Amazon EC2 uses the default version of the launch template.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplatespecificationrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplatespecificationrequest-version
            '''
            result = self._values.get("version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FleetLaunchTemplateSpecificationRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.InstanceRequirementsRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "accelerator_count": "acceleratorCount",
            "accelerator_manufacturers": "acceleratorManufacturers",
            "accelerator_names": "acceleratorNames",
            "accelerator_total_memory_mib": "acceleratorTotalMemoryMiB",
            "accelerator_types": "acceleratorTypes",
            "bare_metal": "bareMetal",
            "baseline_ebs_bandwidth_mbps": "baselineEbsBandwidthMbps",
            "burstable_performance": "burstablePerformance",
            "cpu_manufacturers": "cpuManufacturers",
            "excluded_instance_types": "excludedInstanceTypes",
            "instance_generations": "instanceGenerations",
            "local_storage": "localStorage",
            "local_storage_types": "localStorageTypes",
            "memory_gib_per_v_cpu": "memoryGiBPerVCpu",
            "memory_mib": "memoryMiB",
            "network_interface_count": "networkInterfaceCount",
            "on_demand_max_price_percentage_over_lowest_price": "onDemandMaxPricePercentageOverLowestPrice",
            "require_hibernate_support": "requireHibernateSupport",
            "spot_max_price_percentage_over_lowest_price": "spotMaxPricePercentageOverLowestPrice",
            "total_local_storage_gb": "totalLocalStorageGb",
            "v_cpu_count": "vCpuCount",
        },
    )
    class InstanceRequirementsRequestProperty:
        def __init__(
            self,
            *,
            accelerator_count: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.AcceleratorCountRequestProperty"]] = None,
            accelerator_manufacturers: typing.Optional[typing.Sequence[builtins.str]] = None,
            accelerator_names: typing.Optional[typing.Sequence[builtins.str]] = None,
            accelerator_total_memory_mib: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.AcceleratorTotalMemoryMiBRequestProperty"]] = None,
            accelerator_types: typing.Optional[typing.Sequence[builtins.str]] = None,
            bare_metal: typing.Optional[builtins.str] = None,
            baseline_ebs_bandwidth_mbps: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.BaselineEbsBandwidthMbpsRequestProperty"]] = None,
            burstable_performance: typing.Optional[builtins.str] = None,
            cpu_manufacturers: typing.Optional[typing.Sequence[builtins.str]] = None,
            excluded_instance_types: typing.Optional[typing.Sequence[builtins.str]] = None,
            instance_generations: typing.Optional[typing.Sequence[builtins.str]] = None,
            local_storage: typing.Optional[builtins.str] = None,
            local_storage_types: typing.Optional[typing.Sequence[builtins.str]] = None,
            memory_gib_per_v_cpu: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.MemoryGiBPerVCpuRequestProperty"]] = None,
            memory_mib: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.MemoryMiBRequestProperty"]] = None,
            network_interface_count: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.NetworkInterfaceCountRequestProperty"]] = None,
            on_demand_max_price_percentage_over_lowest_price: typing.Optional[jsii.Number] = None,
            require_hibernate_support: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            spot_max_price_percentage_over_lowest_price: typing.Optional[jsii.Number] = None,
            total_local_storage_gb: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.TotalLocalStorageGBRequestProperty"]] = None,
            v_cpu_count: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.VCpuCountRangeRequestProperty"]] = None,
        ) -> None:
            '''The attributes for the instance types.

            When you specify instance attributes, Amazon EC2 will identify instance types with these attributes.

            When you specify multiple parameters, you get instance types that satisfy all of the specified parameters. If you specify multiple values for a parameter, you get instance types that satisfy any of the specified values.
            .. epigraph::

               You must specify ``VCpuCount`` and ``MemoryMiB`` . All other parameters are optional. Any unspecified optional parameter is set to its default.

            For more information, see `Attribute-based instance type selection for EC2 Fleet <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet-attribute-based-instance-type-selection.html>`_ , `Attribute-based instance type selection for Spot Fleet <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/spot-fleet-attribute-based-instance-type-selection.html>`_ , and `Spot placement score <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/spot-placement-score.html>`_ in the *Amazon EC2 User Guide* .

            :param accelerator_count: The minimum and maximum number of accelerators (GPUs, FPGAs, or AWS Inferentia chips) on an instance. To exclude accelerator-enabled instance types, set ``Max`` to ``0`` . Default: No minimum or maximum limits
            :param accelerator_manufacturers: Indicates whether instance types must have accelerators by specific manufacturers. - For instance types with NVIDIA devices, specify ``nvidia`` . - For instance types with AMD devices, specify ``amd`` . - For instance types with AWS devices, specify ``amazon-web-services`` . - For instance types with Xilinx devices, specify ``xilinx`` . Default: Any manufacturer
            :param accelerator_names: The accelerators that must be on the instance type. - For instance types with NVIDIA A100 GPUs, specify ``a100`` . - For instance types with NVIDIA V100 GPUs, specify ``v100`` . - For instance types with NVIDIA K80 GPUs, specify ``k80`` . - For instance types with NVIDIA T4 GPUs, specify ``t4`` . - For instance types with NVIDIA M60 GPUs, specify ``m60`` . - For instance types with AMD Radeon Pro V520 GPUs, specify ``radeon-pro-v520`` . - For instance types with Xilinx VU9P FPGAs, specify ``vu9p`` . Default: Any accelerator
            :param accelerator_total_memory_mib: The minimum and maximum amount of total accelerator memory, in MiB. Default: No minimum or maximum limits
            :param accelerator_types: The accelerator types that must be on the instance type. - To include instance types with GPU hardware, specify ``gpu`` . - To include instance types with FPGA hardware, specify ``fpga`` . - To include instance types with inference hardware, specify ``inference`` . Default: Any accelerator type
            :param bare_metal: Indicates whether bare metal instance types must be included, excluded, or required. - To include bare metal instance types, specify ``included`` . - To require only bare metal instance types, specify ``required`` . - To exclude bare metal instance types, specify ``excluded`` . Default: ``excluded``
            :param baseline_ebs_bandwidth_mbps: The minimum and maximum baseline bandwidth to Amazon EBS, in Mbps. For more information, see `Amazon EBS–optimized instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-optimized.html>`_ in the *Amazon EC2 User Guide* . Default: No minimum or maximum limits
            :param burstable_performance: Indicates whether burstable performance T instance types are included, excluded, or required. For more information, see `Burstable performance instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances.html>`_ . - To include burstable performance instance types, specify ``included`` . - To require only burstable performance instance types, specify ``required`` . - To exclude burstable performance instance types, specify ``excluded`` . Default: ``excluded``
            :param cpu_manufacturers: The CPU manufacturers to include. - For instance types with Intel CPUs, specify ``intel`` . - For instance types with AMD CPUs, specify ``amd`` . - For instance types with AWS CPUs, specify ``amazon-web-services`` . .. epigraph:: Don't confuse the CPU manufacturer with the CPU architecture. Instances will be launched with a compatible CPU architecture based on the Amazon Machine Image (AMI) that you specify in your launch template. Default: Any manufacturer
            :param excluded_instance_types: The instance types to exclude. You can use strings with one or more wild cards, represented by an asterisk ( ``*`` ), to exclude an instance family, type, size, or generation. The following are examples: ``m5.8xlarge`` , ``c5*.*`` , ``m5a.*`` , ``r*`` , ``*3*`` . For example, if you specify ``c5*`` ,Amazon EC2 will exclude the entire C5 instance family, which includes all C5a and C5n instance types. If you specify ``m5a.*`` , Amazon EC2 will exclude all the M5a instance types, but not the M5n instance types. Default: No excluded instance types
            :param instance_generations: Indicates whether current or previous generation instance types are included. The current generation instance types are recommended for use. Current generation instance types are typically the latest two to three generations in each instance family. For more information, see `Instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html>`_ in the *Amazon EC2 User Guide* . For current generation instance types, specify ``current`` . For previous generation instance types, specify ``previous`` . Default: Current and previous generation instance types
            :param local_storage: Indicates whether instance types with instance store volumes are included, excluded, or required. For more information, `Amazon EC2 instance store <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html>`_ in the *Amazon EC2 User Guide* . - To include instance types with instance store volumes, specify ``included`` . - To require only instance types with instance store volumes, specify ``required`` . - To exclude instance types with instance store volumes, specify ``excluded`` . Default: ``included``
            :param local_storage_types: The type of local storage that is required. - For instance types with hard disk drive (HDD) storage, specify ``hdd`` . - For instance types with solid state drive (SDD) storage, specify ``sdd`` . Default: ``hdd`` and ``sdd``
            :param memory_gib_per_v_cpu: The minimum and maximum amount of memory per vCPU, in GiB. Default: No minimum or maximum limits
            :param memory_mib: The minimum and maximum amount of memory, in MiB.
            :param network_interface_count: The minimum and maximum number of network interfaces. Default: No minimum or maximum limits
            :param on_demand_max_price_percentage_over_lowest_price: The price protection threshold for On-Demand Instances. This is the maximum you’ll pay for an On-Demand Instance, expressed as a percentage above the cheapest M, C, or R instance type with your specified attributes. When Amazon EC2 selects instance types with your attributes, it excludes instance types priced above your threshold. The parameter accepts an integer, which Amazon EC2 interprets as a percentage. To turn off price protection, specify a high value, such as ``999999`` . This parameter is not supported for `GetSpotPlacementScores <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetSpotPlacementScores.html>`_ and `GetInstanceTypesFromInstanceRequirements <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetInstanceTypesFromInstanceRequirements.html>`_ . Default: ``20``
            :param require_hibernate_support: Indicates whether instance types must support hibernation for On-Demand Instances. This parameter is not supported for `GetSpotPlacementScores <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetSpotPlacementScores.html>`_ . Default: ``false``
            :param spot_max_price_percentage_over_lowest_price: The price protection threshold for Spot Instance. This is the maximum you’ll pay for an Spot Instance, expressed as a percentage above the cheapest M, C, or R instance type with your specified attributes. When Amazon EC2 selects instance types with your attributes, it excludes instance types priced above your threshold. The parameter accepts an integer, which Amazon EC2 interprets as a percentage. To turn off price protection, specify a high value, such as ``999999`` . This parameter is not supported for `GetSpotPlacementScores <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetSpotPlacementScores.html>`_ and `GetInstanceTypesFromInstanceRequirements <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetInstanceTypesFromInstanceRequirements.html>`_ . Default: ``100``
            :param total_local_storage_gb: The minimum and maximum amount of total local storage, in GB. Default: No minimum or maximum limits
            :param v_cpu_count: The minimum and maximum number of vCPUs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                instance_requirements_request_property = ec2.CfnEC2Fleet.InstanceRequirementsRequestProperty(
                    accelerator_count=ec2.CfnEC2Fleet.AcceleratorCountRequestProperty(
                        max=123,
                        min=123
                    ),
                    accelerator_manufacturers=["acceleratorManufacturers"],
                    accelerator_names=["acceleratorNames"],
                    accelerator_total_memory_mi_b=ec2.CfnEC2Fleet.AcceleratorTotalMemoryMiBRequestProperty(
                        max=123,
                        min=123
                    ),
                    accelerator_types=["acceleratorTypes"],
                    bare_metal="bareMetal",
                    baseline_ebs_bandwidth_mbps=ec2.CfnEC2Fleet.BaselineEbsBandwidthMbpsRequestProperty(
                        max=123,
                        min=123
                    ),
                    burstable_performance="burstablePerformance",
                    cpu_manufacturers=["cpuManufacturers"],
                    excluded_instance_types=["excludedInstanceTypes"],
                    instance_generations=["instanceGenerations"],
                    local_storage="localStorage",
                    local_storage_types=["localStorageTypes"],
                    memory_gi_bPer_vCpu=ec2.CfnEC2Fleet.MemoryGiBPerVCpuRequestProperty(
                        max=123,
                        min=123
                    ),
                    memory_mi_b=ec2.CfnEC2Fleet.MemoryMiBRequestProperty(
                        max=123,
                        min=123
                    ),
                    network_interface_count=ec2.CfnEC2Fleet.NetworkInterfaceCountRequestProperty(
                        max=123,
                        min=123
                    ),
                    on_demand_max_price_percentage_over_lowest_price=123,
                    require_hibernate_support=False,
                    spot_max_price_percentage_over_lowest_price=123,
                    total_local_storage_gb=ec2.CfnEC2Fleet.TotalLocalStorageGBRequestProperty(
                        max=123,
                        min=123
                    ),
                    v_cpu_count=ec2.CfnEC2Fleet.VCpuCountRangeRequestProperty(
                        max=123,
                        min=123
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if accelerator_count is not None:
                self._values["accelerator_count"] = accelerator_count
            if accelerator_manufacturers is not None:
                self._values["accelerator_manufacturers"] = accelerator_manufacturers
            if accelerator_names is not None:
                self._values["accelerator_names"] = accelerator_names
            if accelerator_total_memory_mib is not None:
                self._values["accelerator_total_memory_mib"] = accelerator_total_memory_mib
            if accelerator_types is not None:
                self._values["accelerator_types"] = accelerator_types
            if bare_metal is not None:
                self._values["bare_metal"] = bare_metal
            if baseline_ebs_bandwidth_mbps is not None:
                self._values["baseline_ebs_bandwidth_mbps"] = baseline_ebs_bandwidth_mbps
            if burstable_performance is not None:
                self._values["burstable_performance"] = burstable_performance
            if cpu_manufacturers is not None:
                self._values["cpu_manufacturers"] = cpu_manufacturers
            if excluded_instance_types is not None:
                self._values["excluded_instance_types"] = excluded_instance_types
            if instance_generations is not None:
                self._values["instance_generations"] = instance_generations
            if local_storage is not None:
                self._values["local_storage"] = local_storage
            if local_storage_types is not None:
                self._values["local_storage_types"] = local_storage_types
            if memory_gib_per_v_cpu is not None:
                self._values["memory_gib_per_v_cpu"] = memory_gib_per_v_cpu
            if memory_mib is not None:
                self._values["memory_mib"] = memory_mib
            if network_interface_count is not None:
                self._values["network_interface_count"] = network_interface_count
            if on_demand_max_price_percentage_over_lowest_price is not None:
                self._values["on_demand_max_price_percentage_over_lowest_price"] = on_demand_max_price_percentage_over_lowest_price
            if require_hibernate_support is not None:
                self._values["require_hibernate_support"] = require_hibernate_support
            if spot_max_price_percentage_over_lowest_price is not None:
                self._values["spot_max_price_percentage_over_lowest_price"] = spot_max_price_percentage_over_lowest_price
            if total_local_storage_gb is not None:
                self._values["total_local_storage_gb"] = total_local_storage_gb
            if v_cpu_count is not None:
                self._values["v_cpu_count"] = v_cpu_count

        @builtins.property
        def accelerator_count(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.AcceleratorCountRequestProperty"]]:
            '''The minimum and maximum number of accelerators (GPUs, FPGAs, or AWS Inferentia chips) on an instance.

            To exclude accelerator-enabled instance types, set ``Max`` to ``0`` .

            Default: No minimum or maximum limits

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-acceleratorcount
            '''
            result = self._values.get("accelerator_count")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.AcceleratorCountRequestProperty"]], result)

        @builtins.property
        def accelerator_manufacturers(
            self,
        ) -> typing.Optional[typing.List[builtins.str]]:
            '''Indicates whether instance types must have accelerators by specific manufacturers.

            - For instance types with NVIDIA devices, specify ``nvidia`` .
            - For instance types with AMD devices, specify ``amd`` .
            - For instance types with AWS devices, specify ``amazon-web-services`` .
            - For instance types with Xilinx devices, specify ``xilinx`` .

            Default: Any manufacturer

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-acceleratormanufacturers
            '''
            result = self._values.get("accelerator_manufacturers")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def accelerator_names(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The accelerators that must be on the instance type.

            - For instance types with NVIDIA A100 GPUs, specify ``a100`` .
            - For instance types with NVIDIA V100 GPUs, specify ``v100`` .
            - For instance types with NVIDIA K80 GPUs, specify ``k80`` .
            - For instance types with NVIDIA T4 GPUs, specify ``t4`` .
            - For instance types with NVIDIA M60 GPUs, specify ``m60`` .
            - For instance types with AMD Radeon Pro V520 GPUs, specify ``radeon-pro-v520`` .
            - For instance types with Xilinx VU9P FPGAs, specify ``vu9p`` .

            Default: Any accelerator

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-acceleratornames
            '''
            result = self._values.get("accelerator_names")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def accelerator_total_memory_mib(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.AcceleratorTotalMemoryMiBRequestProperty"]]:
            '''The minimum and maximum amount of total accelerator memory, in MiB.

            Default: No minimum or maximum limits

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-acceleratortotalmemorymib
            '''
            result = self._values.get("accelerator_total_memory_mib")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.AcceleratorTotalMemoryMiBRequestProperty"]], result)

        @builtins.property
        def accelerator_types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The accelerator types that must be on the instance type.

            - To include instance types with GPU hardware, specify ``gpu`` .
            - To include instance types with FPGA hardware, specify ``fpga`` .
            - To include instance types with inference hardware, specify ``inference`` .

            Default: Any accelerator type

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-acceleratortypes
            '''
            result = self._values.get("accelerator_types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def bare_metal(self) -> typing.Optional[builtins.str]:
            '''Indicates whether bare metal instance types must be included, excluded, or required.

            - To include bare metal instance types, specify ``included`` .
            - To require only bare metal instance types, specify ``required`` .
            - To exclude bare metal instance types, specify ``excluded`` .

            Default: ``excluded``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-baremetal
            '''
            result = self._values.get("bare_metal")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def baseline_ebs_bandwidth_mbps(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.BaselineEbsBandwidthMbpsRequestProperty"]]:
            '''The minimum and maximum baseline bandwidth to Amazon EBS, in Mbps.

            For more information, see `Amazon EBS–optimized instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-optimized.html>`_ in the *Amazon EC2 User Guide* .

            Default: No minimum or maximum limits

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-baselineebsbandwidthmbps
            '''
            result = self._values.get("baseline_ebs_bandwidth_mbps")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.BaselineEbsBandwidthMbpsRequestProperty"]], result)

        @builtins.property
        def burstable_performance(self) -> typing.Optional[builtins.str]:
            '''Indicates whether burstable performance T instance types are included, excluded, or required.

            For more information, see `Burstable performance instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances.html>`_ .

            - To include burstable performance instance types, specify ``included`` .
            - To require only burstable performance instance types, specify ``required`` .
            - To exclude burstable performance instance types, specify ``excluded`` .

            Default: ``excluded``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-burstableperformance
            '''
            result = self._values.get("burstable_performance")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def cpu_manufacturers(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The CPU manufacturers to include.

            - For instance types with Intel CPUs, specify ``intel`` .
            - For instance types with AMD CPUs, specify ``amd`` .
            - For instance types with AWS CPUs, specify ``amazon-web-services`` .

            .. epigraph::

               Don't confuse the CPU manufacturer with the CPU architecture. Instances will be launched with a compatible CPU architecture based on the Amazon Machine Image (AMI) that you specify in your launch template.

            Default: Any manufacturer

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-cpumanufacturers
            '''
            result = self._values.get("cpu_manufacturers")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def excluded_instance_types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The instance types to exclude.

            You can use strings with one or more wild cards, represented by an asterisk ( ``*`` ), to exclude an instance family, type, size, or generation. The following are examples: ``m5.8xlarge`` , ``c5*.*`` , ``m5a.*`` , ``r*`` , ``*3*`` .

            For example, if you specify ``c5*`` ,Amazon EC2 will exclude the entire C5 instance family, which includes all C5a and C5n instance types. If you specify ``m5a.*`` , Amazon EC2 will exclude all the M5a instance types, but not the M5n instance types.

            Default: No excluded instance types

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-excludedinstancetypes
            '''
            result = self._values.get("excluded_instance_types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def instance_generations(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Indicates whether current or previous generation instance types are included.

            The current generation instance types are recommended for use. Current generation instance types are typically the latest two to three generations in each instance family. For more information, see `Instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html>`_ in the *Amazon EC2 User Guide* .

            For current generation instance types, specify ``current`` .

            For previous generation instance types, specify ``previous`` .

            Default: Current and previous generation instance types

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-instancegenerations
            '''
            result = self._values.get("instance_generations")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def local_storage(self) -> typing.Optional[builtins.str]:
            '''Indicates whether instance types with instance store volumes are included, excluded, or required.

            For more information, `Amazon EC2 instance store <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html>`_ in the *Amazon EC2 User Guide* .

            - To include instance types with instance store volumes, specify ``included`` .
            - To require only instance types with instance store volumes, specify ``required`` .
            - To exclude instance types with instance store volumes, specify ``excluded`` .

            Default: ``included``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-localstorage
            '''
            result = self._values.get("local_storage")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def local_storage_types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The type of local storage that is required.

            - For instance types with hard disk drive (HDD) storage, specify ``hdd`` .
            - For instance types with solid state drive (SDD) storage, specify ``sdd`` .

            Default: ``hdd`` and ``sdd``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-localstoragetypes
            '''
            result = self._values.get("local_storage_types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def memory_gib_per_v_cpu(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.MemoryGiBPerVCpuRequestProperty"]]:
            '''The minimum and maximum amount of memory per vCPU, in GiB.

            Default: No minimum or maximum limits

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-memorygibpervcpu
            '''
            result = self._values.get("memory_gib_per_v_cpu")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.MemoryGiBPerVCpuRequestProperty"]], result)

        @builtins.property
        def memory_mib(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.MemoryMiBRequestProperty"]]:
            '''The minimum and maximum amount of memory, in MiB.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-memorymib
            '''
            result = self._values.get("memory_mib")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.MemoryMiBRequestProperty"]], result)

        @builtins.property
        def network_interface_count(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.NetworkInterfaceCountRequestProperty"]]:
            '''The minimum and maximum number of network interfaces.

            Default: No minimum or maximum limits

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-networkinterfacecount
            '''
            result = self._values.get("network_interface_count")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.NetworkInterfaceCountRequestProperty"]], result)

        @builtins.property
        def on_demand_max_price_percentage_over_lowest_price(
            self,
        ) -> typing.Optional[jsii.Number]:
            '''The price protection threshold for On-Demand Instances.

            This is the maximum you’ll pay for an On-Demand Instance, expressed as a percentage above the cheapest M, C, or R instance type with your specified attributes. When Amazon EC2 selects instance types with your attributes, it excludes instance types priced above your threshold.

            The parameter accepts an integer, which Amazon EC2 interprets as a percentage.

            To turn off price protection, specify a high value, such as ``999999`` .

            This parameter is not supported for `GetSpotPlacementScores <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetSpotPlacementScores.html>`_ and `GetInstanceTypesFromInstanceRequirements <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetInstanceTypesFromInstanceRequirements.html>`_ .

            Default: ``20``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-ondemandmaxpricepercentageoverlowestprice
            '''
            result = self._values.get("on_demand_max_price_percentage_over_lowest_price")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def require_hibernate_support(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''Indicates whether instance types must support hibernation for On-Demand Instances.

            This parameter is not supported for `GetSpotPlacementScores <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetSpotPlacementScores.html>`_ .

            Default: ``false``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-requirehibernatesupport
            '''
            result = self._values.get("require_hibernate_support")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def spot_max_price_percentage_over_lowest_price(
            self,
        ) -> typing.Optional[jsii.Number]:
            '''The price protection threshold for Spot Instance.

            This is the maximum you’ll pay for an Spot Instance, expressed as a percentage above the cheapest M, C, or R instance type with your specified attributes. When Amazon EC2 selects instance types with your attributes, it excludes instance types priced above your threshold.

            The parameter accepts an integer, which Amazon EC2 interprets as a percentage.

            To turn off price protection, specify a high value, such as ``999999`` .

            This parameter is not supported for `GetSpotPlacementScores <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetSpotPlacementScores.html>`_ and `GetInstanceTypesFromInstanceRequirements <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetInstanceTypesFromInstanceRequirements.html>`_ .

            Default: ``100``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-spotmaxpricepercentageoverlowestprice
            '''
            result = self._values.get("spot_max_price_percentage_over_lowest_price")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def total_local_storage_gb(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.TotalLocalStorageGBRequestProperty"]]:
            '''The minimum and maximum amount of total local storage, in GB.

            Default: No minimum or maximum limits

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-totallocalstoragegb
            '''
            result = self._values.get("total_local_storage_gb")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.TotalLocalStorageGBRequestProperty"]], result)

        @builtins.property
        def v_cpu_count(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.VCpuCountRangeRequestProperty"]]:
            '''The minimum and maximum number of vCPUs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-vcpucount
            '''
            result = self._values.get("v_cpu_count")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.VCpuCountRangeRequestProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstanceRequirementsRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.MaintenanceStrategiesProperty",
        jsii_struct_bases=[],
        name_mapping={"capacity_rebalance": "capacityRebalance"},
    )
    class MaintenanceStrategiesProperty:
        def __init__(
            self,
            *,
            capacity_rebalance: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.CapacityRebalanceProperty"]] = None,
        ) -> None:
            '''The strategies for managing your Spot Instances that are at an elevated risk of being interrupted.

            :param capacity_rebalance: The strategy to use when Amazon EC2 emits a signal that your Spot Instance is at an elevated risk of being interrupted.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-maintenancestrategies.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                maintenance_strategies_property = ec2.CfnEC2Fleet.MaintenanceStrategiesProperty(
                    capacity_rebalance=ec2.CfnEC2Fleet.CapacityRebalanceProperty(
                        replacement_strategy="replacementStrategy",
                        termination_delay=123
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if capacity_rebalance is not None:
                self._values["capacity_rebalance"] = capacity_rebalance

        @builtins.property
        def capacity_rebalance(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.CapacityRebalanceProperty"]]:
            '''The strategy to use when Amazon EC2 emits a signal that your Spot Instance is at an elevated risk of being interrupted.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-maintenancestrategies.html#cfn-ec2-ec2fleet-maintenancestrategies-capacityrebalance
            '''
            result = self._values.get("capacity_rebalance")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.CapacityRebalanceProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MaintenanceStrategiesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.MemoryGiBPerVCpuRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class MemoryGiBPerVCpuRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The minimum and maximum amount of memory per vCPU, in GiB.

            :param max: The maximum amount of memory per vCPU, in GiB. To specify no maximum limit, omit this parameter.
            :param min: The minimum amount of memory per vCPU, in GiB. To specify no minimum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-memorygibpervcpurequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                memory_gi_bPer_vCpu_request_property = ec2.CfnEC2Fleet.MemoryGiBPerVCpuRequestProperty(
                    max=123,
                    min=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The maximum amount of memory per vCPU, in GiB.

            To specify no maximum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-memorygibpervcpurequest.html#cfn-ec2-ec2fleet-memorygibpervcpurequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The minimum amount of memory per vCPU, in GiB.

            To specify no minimum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-memorygibpervcpurequest.html#cfn-ec2-ec2fleet-memorygibpervcpurequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MemoryGiBPerVCpuRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.MemoryMiBRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class MemoryMiBRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The minimum and maximum amount of memory, in MiB.

            :param max: ``CfnEC2Fleet.MemoryMiBRequestProperty.Max``.
            :param min: ``CfnEC2Fleet.MemoryMiBRequestProperty.Min``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-memorymibrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                memory_mi_bRequest_property = ec2.CfnEC2Fleet.MemoryMiBRequestProperty(
                    max=123,
                    min=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''``CfnEC2Fleet.MemoryMiBRequestProperty.Max``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-memorymibrequest.html#cfn-ec2-ec2fleet-memorymibrequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''``CfnEC2Fleet.MemoryMiBRequestProperty.Min``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-memorymibrequest.html#cfn-ec2-ec2fleet-memorymibrequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MemoryMiBRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.NetworkInterfaceCountRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class NetworkInterfaceCountRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The minimum and maximum number of network interfaces.

            :param max: The maximum number of network interfaces. To specify no maximum limit, omit this parameter.
            :param min: The minimum number of network interfaces. To specify no minimum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-networkinterfacecountrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                network_interface_count_request_property = ec2.CfnEC2Fleet.NetworkInterfaceCountRequestProperty(
                    max=123,
                    min=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The maximum number of network interfaces.

            To specify no maximum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-networkinterfacecountrequest.html#cfn-ec2-ec2fleet-networkinterfacecountrequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The minimum number of network interfaces.

            To specify no minimum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-networkinterfacecountrequest.html#cfn-ec2-ec2fleet-networkinterfacecountrequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "NetworkInterfaceCountRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.OnDemandOptionsRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "allocation_strategy": "allocationStrategy",
            "capacity_reservation_options": "capacityReservationOptions",
            "max_total_price": "maxTotalPrice",
            "min_target_capacity": "minTargetCapacity",
            "single_availability_zone": "singleAvailabilityZone",
            "single_instance_type": "singleInstanceType",
        },
    )
    class OnDemandOptionsRequestProperty:
        def __init__(
            self,
            *,
            allocation_strategy: typing.Optional[builtins.str] = None,
            capacity_reservation_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.CapacityReservationOptionsRequestProperty"]] = None,
            max_total_price: typing.Optional[builtins.str] = None,
            min_target_capacity: typing.Optional[jsii.Number] = None,
            single_availability_zone: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            single_instance_type: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        ) -> None:
            '''Specifies the allocation strategy of On-Demand Instances in an EC2 Fleet.

            ``OnDemandOptionsRequest`` is a property of the `AWS::EC2::EC2Fleet <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html>`_ resource.

            :param allocation_strategy: The strategy that determines the order of the launch template overrides to use in fulfilling On-Demand capacity. ``lowest-price`` - EC2 Fleet uses price to determine the order, launching the lowest price first. ``prioritized`` - EC2 Fleet uses the priority that you assigned to each launch template override, launching the highest priority first. Default: ``lowest-price``
            :param capacity_reservation_options: The strategy for using unused Capacity Reservations for fulfilling On-Demand capacity. Supported only for fleets of type ``instant`` .
            :param max_total_price: The maximum amount per hour for On-Demand Instances that you're willing to pay.
            :param min_target_capacity: The minimum target capacity for On-Demand Instances in the fleet. If the minimum target capacity is not reached, the fleet launches no instances. Supported only for fleets of type ``instant`` . At least one of the following must be specified: ``SingleAvailabilityZone`` | ``SingleInstanceType``
            :param single_availability_zone: Indicates that the fleet launches all On-Demand Instances into a single Availability Zone. Supported only for fleets of type ``instant`` .
            :param single_instance_type: Indicates that the fleet uses a single instance type to launch all On-Demand Instances in the fleet. Supported only for fleets of type ``instant`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-ondemandoptionsrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                on_demand_options_request_property = ec2.CfnEC2Fleet.OnDemandOptionsRequestProperty(
                    allocation_strategy="allocationStrategy",
                    capacity_reservation_options=ec2.CfnEC2Fleet.CapacityReservationOptionsRequestProperty(
                        usage_strategy="usageStrategy"
                    ),
                    max_total_price="maxTotalPrice",
                    min_target_capacity=123,
                    single_availability_zone=False,
                    single_instance_type=False
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if allocation_strategy is not None:
                self._values["allocation_strategy"] = allocation_strategy
            if capacity_reservation_options is not None:
                self._values["capacity_reservation_options"] = capacity_reservation_options
            if max_total_price is not None:
                self._values["max_total_price"] = max_total_price
            if min_target_capacity is not None:
                self._values["min_target_capacity"] = min_target_capacity
            if single_availability_zone is not None:
                self._values["single_availability_zone"] = single_availability_zone
            if single_instance_type is not None:
                self._values["single_instance_type"] = single_instance_type

        @builtins.property
        def allocation_strategy(self) -> typing.Optional[builtins.str]:
            '''The strategy that determines the order of the launch template overrides to use in fulfilling On-Demand capacity.

            ``lowest-price`` - EC2 Fleet uses price to determine the order, launching the lowest price first.

            ``prioritized`` - EC2 Fleet uses the priority that you assigned to each launch template override, launching the highest priority first.

            Default: ``lowest-price``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-ondemandoptionsrequest.html#cfn-ec2-ec2fleet-ondemandoptionsrequest-allocationstrategy
            '''
            result = self._values.get("allocation_strategy")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def capacity_reservation_options(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.CapacityReservationOptionsRequestProperty"]]:
            '''The strategy for using unused Capacity Reservations for fulfilling On-Demand capacity.

            Supported only for fleets of type ``instant`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-ondemandoptionsrequest.html#cfn-ec2-ec2fleet-ondemandoptionsrequest-capacityreservationoptions
            '''
            result = self._values.get("capacity_reservation_options")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.CapacityReservationOptionsRequestProperty"]], result)

        @builtins.property
        def max_total_price(self) -> typing.Optional[builtins.str]:
            '''The maximum amount per hour for On-Demand Instances that you're willing to pay.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-ondemandoptionsrequest.html#cfn-ec2-ec2fleet-ondemandoptionsrequest-maxtotalprice
            '''
            result = self._values.get("max_total_price")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def min_target_capacity(self) -> typing.Optional[jsii.Number]:
            '''The minimum target capacity for On-Demand Instances in the fleet.

            If the minimum target capacity is not reached, the fleet launches no instances.

            Supported only for fleets of type ``instant`` .

            At least one of the following must be specified: ``SingleAvailabilityZone`` | ``SingleInstanceType``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-ondemandoptionsrequest.html#cfn-ec2-ec2fleet-ondemandoptionsrequest-mintargetcapacity
            '''
            result = self._values.get("min_target_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def single_availability_zone(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''Indicates that the fleet launches all On-Demand Instances into a single Availability Zone.

            Supported only for fleets of type ``instant`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-ondemandoptionsrequest.html#cfn-ec2-ec2fleet-ondemandoptionsrequest-singleavailabilityzone
            '''
            result = self._values.get("single_availability_zone")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def single_instance_type(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''Indicates that the fleet uses a single instance type to launch all On-Demand Instances in the fleet.

            Supported only for fleets of type ``instant`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-ondemandoptionsrequest.html#cfn-ec2-ec2fleet-ondemandoptionsrequest-singleinstancetype
            '''
            result = self._values.get("single_instance_type")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OnDemandOptionsRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.PlacementProperty",
        jsii_struct_bases=[],
        name_mapping={
            "affinity": "affinity",
            "availability_zone": "availabilityZone",
            "group_name": "groupName",
            "host_id": "hostId",
            "host_resource_group_arn": "hostResourceGroupArn",
            "partition_number": "partitionNumber",
            "spread_domain": "spreadDomain",
            "tenancy": "tenancy",
        },
    )
    class PlacementProperty:
        def __init__(
            self,
            *,
            affinity: typing.Optional[builtins.str] = None,
            availability_zone: typing.Optional[builtins.str] = None,
            group_name: typing.Optional[builtins.str] = None,
            host_id: typing.Optional[builtins.str] = None,
            host_resource_group_arn: typing.Optional[builtins.str] = None,
            partition_number: typing.Optional[jsii.Number] = None,
            spread_domain: typing.Optional[builtins.str] = None,
            tenancy: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Describes the placement of an instance.

            :param affinity: The affinity setting for the instance on the Dedicated Host. This parameter is not supported for the `ImportInstance <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportInstance.html>`_ command. This parameter is not supported by `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ .
            :param availability_zone: The Availability Zone of the instance. If not specified, an Availability Zone will be automatically chosen for you based on the load balancing criteria for the Region. This parameter is not supported by `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ .
            :param group_name: The name of the placement group the instance is in.
            :param host_id: The ID of the Dedicated Host on which the instance resides. This parameter is not supported for the `ImportInstance <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportInstance.html>`_ command. This parameter is not supported by `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ .
            :param host_resource_group_arn: The ARN of the host resource group in which to launch the instances. If you specify a host resource group ARN, omit the *Tenancy* parameter or set it to ``host`` . This parameter is not supported by `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ .
            :param partition_number: The number of the partition that the instance is in. Valid only if the placement group strategy is set to ``partition`` . This parameter is not supported by `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ .
            :param spread_domain: Reserved for future use. This parameter is not supported by `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ .
            :param tenancy: The tenancy of the instance (if the instance is running in a VPC). An instance with a tenancy of ``dedicated`` runs on single-tenant hardware. The ``host`` tenancy is not supported for the `ImportInstance <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportInstance.html>`_ command. This parameter is not supported by `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ . T3 instances that use the ``unlimited`` CPU credit option do not support ``host`` tenancy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                placement_property = ec2.CfnEC2Fleet.PlacementProperty(
                    affinity="affinity",
                    availability_zone="availabilityZone",
                    group_name="groupName",
                    host_id="hostId",
                    host_resource_group_arn="hostResourceGroupArn",
                    partition_number=123,
                    spread_domain="spreadDomain",
                    tenancy="tenancy"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if affinity is not None:
                self._values["affinity"] = affinity
            if availability_zone is not None:
                self._values["availability_zone"] = availability_zone
            if group_name is not None:
                self._values["group_name"] = group_name
            if host_id is not None:
                self._values["host_id"] = host_id
            if host_resource_group_arn is not None:
                self._values["host_resource_group_arn"] = host_resource_group_arn
            if partition_number is not None:
                self._values["partition_number"] = partition_number
            if spread_domain is not None:
                self._values["spread_domain"] = spread_domain
            if tenancy is not None:
                self._values["tenancy"] = tenancy

        @builtins.property
        def affinity(self) -> typing.Optional[builtins.str]:
            '''The affinity setting for the instance on the Dedicated Host. This parameter is not supported for the `ImportInstance <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportInstance.html>`_ command.

            This parameter is not supported by `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html#cfn-ec2-ec2fleet-placement-affinity
            '''
            result = self._values.get("affinity")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def availability_zone(self) -> typing.Optional[builtins.str]:
            '''The Availability Zone of the instance.

            If not specified, an Availability Zone will be automatically chosen for you based on the load balancing criteria for the Region.

            This parameter is not supported by `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html#cfn-ec2-ec2fleet-placement-availabilityzone
            '''
            result = self._values.get("availability_zone")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def group_name(self) -> typing.Optional[builtins.str]:
            '''The name of the placement group the instance is in.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html#cfn-ec2-ec2fleet-placement-groupname
            '''
            result = self._values.get("group_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def host_id(self) -> typing.Optional[builtins.str]:
            '''The ID of the Dedicated Host on which the instance resides.

            This parameter is not supported for the `ImportInstance <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportInstance.html>`_ command.

            This parameter is not supported by `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html#cfn-ec2-ec2fleet-placement-hostid
            '''
            result = self._values.get("host_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def host_resource_group_arn(self) -> typing.Optional[builtins.str]:
            '''The ARN of the host resource group in which to launch the instances.

            If you specify a host resource group ARN, omit the *Tenancy* parameter or set it to ``host`` .

            This parameter is not supported by `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html#cfn-ec2-ec2fleet-placement-hostresourcegrouparn
            '''
            result = self._values.get("host_resource_group_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def partition_number(self) -> typing.Optional[jsii.Number]:
            '''The number of the partition that the instance is in.

            Valid only if the placement group strategy is set to ``partition`` .

            This parameter is not supported by `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html#cfn-ec2-ec2fleet-placement-partitionnumber
            '''
            result = self._values.get("partition_number")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def spread_domain(self) -> typing.Optional[builtins.str]:
            '''Reserved for future use.

            This parameter is not supported by `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html#cfn-ec2-ec2fleet-placement-spreaddomain
            '''
            result = self._values.get("spread_domain")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def tenancy(self) -> typing.Optional[builtins.str]:
            '''The tenancy of the instance (if the instance is running in a VPC).

            An instance with a tenancy of ``dedicated`` runs on single-tenant hardware. The ``host`` tenancy is not supported for the `ImportInstance <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportInstance.html>`_ command.

            This parameter is not supported by `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ .

            T3 instances that use the ``unlimited`` CPU credit option do not support ``host`` tenancy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html#cfn-ec2-ec2fleet-placement-tenancy
            '''
            result = self._values.get("tenancy")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PlacementProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.SpotOptionsRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "allocation_strategy": "allocationStrategy",
            "instance_interruption_behavior": "instanceInterruptionBehavior",
            "instance_pools_to_use_count": "instancePoolsToUseCount",
            "maintenance_strategies": "maintenanceStrategies",
            "max_total_price": "maxTotalPrice",
            "min_target_capacity": "minTargetCapacity",
            "single_availability_zone": "singleAvailabilityZone",
            "single_instance_type": "singleInstanceType",
        },
    )
    class SpotOptionsRequestProperty:
        def __init__(
            self,
            *,
            allocation_strategy: typing.Optional[builtins.str] = None,
            instance_interruption_behavior: typing.Optional[builtins.str] = None,
            instance_pools_to_use_count: typing.Optional[jsii.Number] = None,
            maintenance_strategies: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.MaintenanceStrategiesProperty"]] = None,
            max_total_price: typing.Optional[builtins.str] = None,
            min_target_capacity: typing.Optional[jsii.Number] = None,
            single_availability_zone: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            single_instance_type: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        ) -> None:
            '''Specifies the configuration of Spot Instances for an EC2 Fleet.

            ``SpotOptionsRequest`` is a property of the `AWS::EC2::EC2Fleet <https://docs.aws.amazon.com//AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html>`_ resource.

            :param allocation_strategy: Indicates how to allocate the target Spot Instance capacity across the Spot Instance pools specified by the EC2 Fleet. If the allocation strategy is ``lowestPrice`` , EC2 Fleet launches instances from the Spot Instance pools with the lowest price. This is the default allocation strategy. If the allocation strategy is ``diversified`` , EC2 Fleet launches instances from all the Spot Instance pools that you specify. If the allocation strategy is ``capacityOptimized`` , EC2 Fleet launches instances from Spot Instance pools that are optimally chosen based on the available Spot Instance capacity. *Allowed Values* : ``lowestPrice`` | ``diversified`` | ``capacityOptimized`` | ``capacityOptimizedPrioritized``
            :param instance_interruption_behavior: The behavior when a Spot Instance is interrupted. Default: ``terminate``
            :param instance_pools_to_use_count: The number of Spot pools across which to allocate your target Spot capacity. Supported only when Spot ``AllocationStrategy`` is set to ``lowest-price`` . EC2 Fleet selects the cheapest Spot pools and evenly allocates your target Spot capacity across the number of Spot pools that you specify. Note that EC2 Fleet attempts to draw Spot Instances from the number of pools that you specify on a best effort basis. If a pool runs out of Spot capacity before fulfilling your target capacity, EC2 Fleet will continue to fulfill your request by drawing from the next cheapest pool. To ensure that your target capacity is met, you might receive Spot Instances from more than the number of pools that you specified. Similarly, if most of the pools have no Spot capacity, you might receive your full target capacity from fewer than the number of pools that you specified.
            :param maintenance_strategies: The strategies for managing your Spot Instances that are at an elevated risk of being interrupted.
            :param max_total_price: The maximum amount per hour for Spot Instances that you're willing to pay.
            :param min_target_capacity: The minimum target capacity for Spot Instances in the fleet. If the minimum target capacity is not reached, the fleet launches no instances. Supported only for fleets of type ``instant`` . At least one of the following must be specified: ``SingleAvailabilityZone`` | ``SingleInstanceType``
            :param single_availability_zone: Indicates that the fleet launches all Spot Instances into a single Availability Zone. Supported only for fleets of type ``instant`` .
            :param single_instance_type: Indicates that the fleet uses a single instance type to launch all Spot Instances in the fleet. Supported only for fleets of type ``instant`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                spot_options_request_property = ec2.CfnEC2Fleet.SpotOptionsRequestProperty(
                    allocation_strategy="allocationStrategy",
                    instance_interruption_behavior="instanceInterruptionBehavior",
                    instance_pools_to_use_count=123,
                    maintenance_strategies=ec2.CfnEC2Fleet.MaintenanceStrategiesProperty(
                        capacity_rebalance=ec2.CfnEC2Fleet.CapacityRebalanceProperty(
                            replacement_strategy="replacementStrategy",
                            termination_delay=123
                        )
                    ),
                    max_total_price="maxTotalPrice",
                    min_target_capacity=123,
                    single_availability_zone=False,
                    single_instance_type=False
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if allocation_strategy is not None:
                self._values["allocation_strategy"] = allocation_strategy
            if instance_interruption_behavior is not None:
                self._values["instance_interruption_behavior"] = instance_interruption_behavior
            if instance_pools_to_use_count is not None:
                self._values["instance_pools_to_use_count"] = instance_pools_to_use_count
            if maintenance_strategies is not None:
                self._values["maintenance_strategies"] = maintenance_strategies
            if max_total_price is not None:
                self._values["max_total_price"] = max_total_price
            if min_target_capacity is not None:
                self._values["min_target_capacity"] = min_target_capacity
            if single_availability_zone is not None:
                self._values["single_availability_zone"] = single_availability_zone
            if single_instance_type is not None:
                self._values["single_instance_type"] = single_instance_type

        @builtins.property
        def allocation_strategy(self) -> typing.Optional[builtins.str]:
            '''Indicates how to allocate the target Spot Instance capacity across the Spot Instance pools specified by the EC2 Fleet.

            If the allocation strategy is ``lowestPrice`` , EC2 Fleet launches instances from the Spot Instance pools with the lowest price. This is the default allocation strategy.

            If the allocation strategy is ``diversified`` , EC2 Fleet launches instances from all the Spot Instance pools that you specify.

            If the allocation strategy is ``capacityOptimized`` , EC2 Fleet launches instances from Spot Instance pools that are optimally chosen based on the available Spot Instance capacity.

            *Allowed Values* : ``lowestPrice`` | ``diversified`` | ``capacityOptimized`` | ``capacityOptimizedPrioritized``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html#cfn-ec2-ec2fleet-spotoptionsrequest-allocationstrategy
            '''
            result = self._values.get("allocation_strategy")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def instance_interruption_behavior(self) -> typing.Optional[builtins.str]:
            '''The behavior when a Spot Instance is interrupted.

            Default: ``terminate``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html#cfn-ec2-ec2fleet-spotoptionsrequest-instanceinterruptionbehavior
            '''
            result = self._values.get("instance_interruption_behavior")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def instance_pools_to_use_count(self) -> typing.Optional[jsii.Number]:
            '''The number of Spot pools across which to allocate your target Spot capacity.

            Supported only when Spot ``AllocationStrategy`` is set to ``lowest-price`` . EC2 Fleet selects the cheapest Spot pools and evenly allocates your target Spot capacity across the number of Spot pools that you specify.

            Note that EC2 Fleet attempts to draw Spot Instances from the number of pools that you specify on a best effort basis. If a pool runs out of Spot capacity before fulfilling your target capacity, EC2 Fleet will continue to fulfill your request by drawing from the next cheapest pool. To ensure that your target capacity is met, you might receive Spot Instances from more than the number of pools that you specified. Similarly, if most of the pools have no Spot capacity, you might receive your full target capacity from fewer than the number of pools that you specified.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html#cfn-ec2-ec2fleet-spotoptionsrequest-instancepoolstousecount
            '''
            result = self._values.get("instance_pools_to_use_count")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def maintenance_strategies(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.MaintenanceStrategiesProperty"]]:
            '''The strategies for managing your Spot Instances that are at an elevated risk of being interrupted.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html#cfn-ec2-ec2fleet-spotoptionsrequest-maintenancestrategies
            '''
            result = self._values.get("maintenance_strategies")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.MaintenanceStrategiesProperty"]], result)

        @builtins.property
        def max_total_price(self) -> typing.Optional[builtins.str]:
            '''The maximum amount per hour for Spot Instances that you're willing to pay.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html#cfn-ec2-ec2fleet-spotoptionsrequest-maxtotalprice
            '''
            result = self._values.get("max_total_price")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def min_target_capacity(self) -> typing.Optional[jsii.Number]:
            '''The minimum target capacity for Spot Instances in the fleet.

            If the minimum target capacity is not reached, the fleet launches no instances.

            Supported only for fleets of type ``instant`` .

            At least one of the following must be specified: ``SingleAvailabilityZone`` | ``SingleInstanceType``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html#cfn-ec2-ec2fleet-spotoptionsrequest-mintargetcapacity
            '''
            result = self._values.get("min_target_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def single_availability_zone(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''Indicates that the fleet launches all Spot Instances into a single Availability Zone.

            Supported only for fleets of type ``instant`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html#cfn-ec2-ec2fleet-spotoptionsrequest-singleavailabilityzone
            '''
            result = self._values.get("single_availability_zone")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def single_instance_type(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''Indicates that the fleet uses a single instance type to launch all Spot Instances in the fleet.

            Supported only for fleets of type ``instant`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html#cfn-ec2-ec2fleet-spotoptionsrequest-singleinstancetype
            '''
            result = self._values.get("single_instance_type")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SpotOptionsRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.TagSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"resource_type": "resourceType", "tags": "tags"},
    )
    class TagSpecificationProperty:
        def __init__(
            self,
            *,
            resource_type: typing.Optional[builtins.str] = None,
            tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
        ) -> None:
            '''Specifies the tags to apply to a resource when the resource is being created for an EC2 Fleet.

            ``TagSpecification`` is a property of the `AWS::EC2::EC2Fleet <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html>`_ resource.

            :param resource_type: The type of resource to tag. ``ResourceType`` must be ``fleet`` .
            :param tags: The tags to apply to the resource.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-tagspecification.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                tag_specification_property = ec2.CfnEC2Fleet.TagSpecificationProperty(
                    resource_type="resourceType",
                    tags=[CfnTag(
                        key="key",
                        value="value"
                    )]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if resource_type is not None:
                self._values["resource_type"] = resource_type
            if tags is not None:
                self._values["tags"] = tags

        @builtins.property
        def resource_type(self) -> typing.Optional[builtins.str]:
            '''The type of resource to tag.

            ``ResourceType`` must be ``fleet`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-tagspecification.html#cfn-ec2-ec2fleet-tagspecification-resourcetype
            '''
            result = self._values.get("resource_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
            '''The tags to apply to the resource.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-tagspecification.html#cfn-ec2-ec2fleet-tagspecification-tags
            '''
            result = self._values.get("tags")
            return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TagSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.TargetCapacitySpecificationRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "total_target_capacity": "totalTargetCapacity",
            "default_target_capacity_type": "defaultTargetCapacityType",
            "on_demand_target_capacity": "onDemandTargetCapacity",
            "spot_target_capacity": "spotTargetCapacity",
            "target_capacity_unit_type": "targetCapacityUnitType",
        },
    )
    class TargetCapacitySpecificationRequestProperty:
        def __init__(
            self,
            *,
            total_target_capacity: jsii.Number,
            default_target_capacity_type: typing.Optional[builtins.str] = None,
            on_demand_target_capacity: typing.Optional[jsii.Number] = None,
            spot_target_capacity: typing.Optional[jsii.Number] = None,
            target_capacity_unit_type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Specifies the number of units to request for an EC2 Fleet.

            You can choose to set the target capacity in terms of instances or a performance characteristic that is important to your application workload, such as vCPUs, memory, or I/O. If the request type is ``maintain`` , you can specify a target capacity of ``0`` and add capacity later.

            ``TargetCapacitySpecificationRequest`` is a property of the `AWS::EC2::EC2Fleet <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html>`_ resource.

            :param total_target_capacity: The number of units to request, filled using ``DefaultTargetCapacityType`` .
            :param default_target_capacity_type: The default ``TotalTargetCapacity`` , which is either ``Spot`` or ``On-Demand`` .
            :param on_demand_target_capacity: The number of On-Demand units to request.
            :param spot_target_capacity: The number of Spot units to request.
            :param target_capacity_unit_type: The unit for the target capacity. Default: ``units`` (translates to number of instances)

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-targetcapacityspecificationrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                target_capacity_specification_request_property = ec2.CfnEC2Fleet.TargetCapacitySpecificationRequestProperty(
                    total_target_capacity=123,
                
                    # the properties below are optional
                    default_target_capacity_type="defaultTargetCapacityType",
                    on_demand_target_capacity=123,
                    spot_target_capacity=123,
                    target_capacity_unit_type="targetCapacityUnitType"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "total_target_capacity": total_target_capacity,
            }
            if default_target_capacity_type is not None:
                self._values["default_target_capacity_type"] = default_target_capacity_type
            if on_demand_target_capacity is not None:
                self._values["on_demand_target_capacity"] = on_demand_target_capacity
            if spot_target_capacity is not None:
                self._values["spot_target_capacity"] = spot_target_capacity
            if target_capacity_unit_type is not None:
                self._values["target_capacity_unit_type"] = target_capacity_unit_type

        @builtins.property
        def total_target_capacity(self) -> jsii.Number:
            '''The number of units to request, filled using ``DefaultTargetCapacityType`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-targetcapacityspecificationrequest.html#cfn-ec2-ec2fleet-targetcapacityspecificationrequest-totaltargetcapacity
            '''
            result = self._values.get("total_target_capacity")
            assert result is not None, "Required property 'total_target_capacity' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def default_target_capacity_type(self) -> typing.Optional[builtins.str]:
            '''The default ``TotalTargetCapacity`` , which is either ``Spot`` or ``On-Demand`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-targetcapacityspecificationrequest.html#cfn-ec2-ec2fleet-targetcapacityspecificationrequest-defaulttargetcapacitytype
            '''
            result = self._values.get("default_target_capacity_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def on_demand_target_capacity(self) -> typing.Optional[jsii.Number]:
            '''The number of On-Demand units to request.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-targetcapacityspecificationrequest.html#cfn-ec2-ec2fleet-targetcapacityspecificationrequest-ondemandtargetcapacity
            '''
            result = self._values.get("on_demand_target_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def spot_target_capacity(self) -> typing.Optional[jsii.Number]:
            '''The number of Spot units to request.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-targetcapacityspecificationrequest.html#cfn-ec2-ec2fleet-targetcapacityspecificationrequest-spottargetcapacity
            '''
            result = self._values.get("spot_target_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def target_capacity_unit_type(self) -> typing.Optional[builtins.str]:
            '''The unit for the target capacity.

            Default: ``units`` (translates to number of instances)

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-targetcapacityspecificationrequest.html#cfn-ec2-ec2fleet-targetcapacityspecificationrequest-targetcapacityunittype
            '''
            result = self._values.get("target_capacity_unit_type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TargetCapacitySpecificationRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.TotalLocalStorageGBRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class TotalLocalStorageGBRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The minimum and maximum amount of total local storage, in GB.

            :param max: The maximum amount of total local storage, in GB. To specify no maximum limit, omit this parameter.
            :param min: The minimum amount of total local storage, in GB. To specify no minimum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-totallocalstoragegbrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                total_local_storage_gBRequest_property = ec2.CfnEC2Fleet.TotalLocalStorageGBRequestProperty(
                    max=123,
                    min=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The maximum amount of total local storage, in GB.

            To specify no maximum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-totallocalstoragegbrequest.html#cfn-ec2-ec2fleet-totallocalstoragegbrequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The minimum amount of total local storage, in GB.

            To specify no minimum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-totallocalstoragegbrequest.html#cfn-ec2-ec2fleet-totallocalstoragegbrequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TotalLocalStorageGBRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.VCpuCountRangeRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class VCpuCountRangeRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The minimum and maximum number of vCPUs.

            :param max: The maximum number of vCPUs. To specify no maximum limit, omit this parameter.
            :param min: The minimum number of vCPUs. To specify no minimum limit, specify ``0`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-vcpucountrangerequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                v_cpu_count_range_request_property = ec2.CfnEC2Fleet.VCpuCountRangeRequestProperty(
                    max=123,
                    min=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The maximum number of vCPUs.

            To specify no maximum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-vcpucountrangerequest.html#cfn-ec2-ec2fleet-vcpucountrangerequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The minimum number of vCPUs.

            To specify no minimum limit, specify ``0`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-vcpucountrangerequest.html#cfn-ec2-ec2fleet-vcpucountrangerequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VCpuCountRangeRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnEC2FleetProps",
    jsii_struct_bases=[],
    name_mapping={
        "launch_template_configs": "launchTemplateConfigs",
        "target_capacity_specification": "targetCapacitySpecification",
        "context": "context",
        "excess_capacity_termination_policy": "excessCapacityTerminationPolicy",
        "on_demand_options": "onDemandOptions",
        "replace_unhealthy_instances": "replaceUnhealthyInstances",
        "spot_options": "spotOptions",
        "tag_specifications": "tagSpecifications",
        "terminate_instances_with_expiration": "terminateInstancesWithExpiration",
        "type": "type",
        "valid_from": "validFrom",
        "valid_until": "validUntil",
    },
)
class CfnEC2FleetProps:
    def __init__(
        self,
        *,
        launch_template_configs: typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty]]],
        target_capacity_specification: typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.TargetCapacitySpecificationRequestProperty],
        context: typing.Optional[builtins.str] = None,
        excess_capacity_termination_policy: typing.Optional[builtins.str] = None,
        on_demand_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.OnDemandOptionsRequestProperty]] = None,
        replace_unhealthy_instances: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        spot_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.SpotOptionsRequestProperty]] = None,
        tag_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.TagSpecificationProperty]]]] = None,
        terminate_instances_with_expiration: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        type: typing.Optional[builtins.str] = None,
        valid_from: typing.Optional[builtins.str] = None,
        valid_until: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnEC2Fleet``.

        :param launch_template_configs: The configuration for the EC2 Fleet.
        :param target_capacity_specification: The number of units to request.
        :param context: Reserved.
        :param excess_capacity_termination_policy: Indicates whether running instances should be terminated if the total target capacity of the EC2 Fleet is decreased below the current size of the EC2 Fleet.
        :param on_demand_options: Describes the configuration of On-Demand Instances in an EC2 Fleet.
        :param replace_unhealthy_instances: Indicates whether EC2 Fleet should replace unhealthy Spot Instances. Supported only for fleets of type ``maintain`` . For more information, see `EC2 Fleet health checks <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/manage-ec2-fleet.html#ec2-fleet-health-checks>`_ in the *Amazon EC2 User Guide* .
        :param spot_options: Describes the configuration of Spot Instances in an EC2 Fleet.
        :param tag_specifications: The key-value pair for tagging the EC2 Fleet request on creation. For more information, see `Tagging your resources <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Using_Tags.html#tag-resources>`_ . If the fleet type is ``instant`` , specify a resource type of ``fleet`` to tag the fleet or ``instance`` to tag the instances at launch. If the fleet type is ``maintain`` or ``request`` , specify a resource type of ``fleet`` to tag the fleet. You cannot specify a resource type of ``instance`` . To tag instances at launch, specify the tags in a `launch template <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html#create-launch-template>`_ .
        :param terminate_instances_with_expiration: Indicates whether running instances should be terminated when the EC2 Fleet expires.
        :param type: The fleet type. The default value is ``maintain`` . - ``maintain`` - The EC2 Fleet places an asynchronous request for your desired capacity, and continues to maintain your desired Spot capacity by replenishing interrupted Spot Instances. - ``request`` - The EC2 Fleet places an asynchronous one-time request for your desired capacity, but does submit Spot requests in alternative capacity pools if Spot capacity is unavailable, and does not maintain Spot capacity if Spot Instances are interrupted. - ``instant`` - The EC2 Fleet places a synchronous one-time request for your desired capacity, and returns errors for any instances that could not be launched. For more information, see `EC2 Fleet request types <https://docs.aws.amazon.com/https://docs.aws.amazon.com/ec2-fleet-request-type.html>`_ in the *Amazon EC2 User Guide* .
        :param valid_from: The start date and time of the request, in UTC format (for example, *YYYY* - *MM* - *DD* T *HH* : *MM* : *SS* Z). The default is to start fulfilling the request immediately.
        :param valid_until: The end date and time of the request, in UTC format (for example, *YYYY* - *MM* - *DD* T *HH* : *MM* : *SS* Z). At this point, no new EC2 Fleet requests are placed or able to fulfill the request. If no value is specified, the request remains until you cancel it.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            cfn_eC2_fleet_props = ec2.CfnEC2FleetProps(
                launch_template_configs=[ec2.CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty(
                    launch_template_specification=ec2.CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty(
                        launch_template_id="launchTemplateId",
                        launch_template_name="launchTemplateName",
                        version="version"
                    ),
                    overrides=[ec2.CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty(
                        availability_zone="availabilityZone",
                        instance_requirements=ec2.CfnEC2Fleet.InstanceRequirementsRequestProperty(
                            accelerator_count=ec2.CfnEC2Fleet.AcceleratorCountRequestProperty(
                                max=123,
                                min=123
                            ),
                            accelerator_manufacturers=["acceleratorManufacturers"],
                            accelerator_names=["acceleratorNames"],
                            accelerator_total_memory_mi_b=ec2.CfnEC2Fleet.AcceleratorTotalMemoryMiBRequestProperty(
                                max=123,
                                min=123
                            ),
                            accelerator_types=["acceleratorTypes"],
                            bare_metal="bareMetal",
                            baseline_ebs_bandwidth_mbps=ec2.CfnEC2Fleet.BaselineEbsBandwidthMbpsRequestProperty(
                                max=123,
                                min=123
                            ),
                            burstable_performance="burstablePerformance",
                            cpu_manufacturers=["cpuManufacturers"],
                            excluded_instance_types=["excludedInstanceTypes"],
                            instance_generations=["instanceGenerations"],
                            local_storage="localStorage",
                            local_storage_types=["localStorageTypes"],
                            memory_gi_bPer_vCpu=ec2.CfnEC2Fleet.MemoryGiBPerVCpuRequestProperty(
                                max=123,
                                min=123
                            ),
                            memory_mi_b=ec2.CfnEC2Fleet.MemoryMiBRequestProperty(
                                max=123,
                                min=123
                            ),
                            network_interface_count=ec2.CfnEC2Fleet.NetworkInterfaceCountRequestProperty(
                                max=123,
                                min=123
                            ),
                            on_demand_max_price_percentage_over_lowest_price=123,
                            require_hibernate_support=False,
                            spot_max_price_percentage_over_lowest_price=123,
                            total_local_storage_gb=ec2.CfnEC2Fleet.TotalLocalStorageGBRequestProperty(
                                max=123,
                                min=123
                            ),
                            v_cpu_count=ec2.CfnEC2Fleet.VCpuCountRangeRequestProperty(
                                max=123,
                                min=123
                            )
                        ),
                        instance_type="instanceType",
                        max_price="maxPrice",
                        placement=ec2.CfnEC2Fleet.PlacementProperty(
                            affinity="affinity",
                            availability_zone="availabilityZone",
                            group_name="groupName",
                            host_id="hostId",
                            host_resource_group_arn="hostResourceGroupArn",
                            partition_number=123,
                            spread_domain="spreadDomain",
                            tenancy="tenancy"
                        ),
                        priority=123,
                        subnet_id="subnetId",
                        weighted_capacity=123
                    )]
                )],
                target_capacity_specification=ec2.CfnEC2Fleet.TargetCapacitySpecificationRequestProperty(
                    total_target_capacity=123,
            
                    # the properties below are optional
                    default_target_capacity_type="defaultTargetCapacityType",
                    on_demand_target_capacity=123,
                    spot_target_capacity=123,
                    target_capacity_unit_type="targetCapacityUnitType"
                ),
            
                # the properties below are optional
                context="context",
                excess_capacity_termination_policy="excessCapacityTerminationPolicy",
                on_demand_options=ec2.CfnEC2Fleet.OnDemandOptionsRequestProperty(
                    allocation_strategy="allocationStrategy",
                    capacity_reservation_options=ec2.CfnEC2Fleet.CapacityReservationOptionsRequestProperty(
                        usage_strategy="usageStrategy"
                    ),
                    max_total_price="maxTotalPrice",
                    min_target_capacity=123,
                    single_availability_zone=False,
                    single_instance_type=False
                ),
                replace_unhealthy_instances=False,
                spot_options=ec2.CfnEC2Fleet.SpotOptionsRequestProperty(
                    allocation_strategy="allocationStrategy",
                    instance_interruption_behavior="instanceInterruptionBehavior",
                    instance_pools_to_use_count=123,
                    maintenance_strategies=ec2.CfnEC2Fleet.MaintenanceStrategiesProperty(
                        capacity_rebalance=ec2.CfnEC2Fleet.CapacityRebalanceProperty(
                            replacement_strategy="replacementStrategy",
                            termination_delay=123
                        )
                    ),
                    max_total_price="maxTotalPrice",
                    min_target_capacity=123,
                    single_availability_zone=False,
                    single_instance_type=False
                ),
                tag_specifications=[ec2.CfnEC2Fleet.TagSpecificationProperty(
                    resource_type="resourceType",
                    tags=[CfnTag(
                        key="key",
                        value="value"
                    )]
                )],
                terminate_instances_with_expiration=False,
                type="type",
                valid_from="validFrom",
                valid_until="validUntil"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "launch_template_configs": launch_template_configs,
            "target_capacity_specification": target_capacity_specification,
        }
        if context is not None:
            self._values["context"] = context
        if excess_capacity_termination_policy is not None:
            self._values["excess_capacity_termination_policy"] = excess_capacity_termination_policy
        if on_demand_options is not None:
            self._values["on_demand_options"] = on_demand_options
        if replace_unhealthy_instances is not None:
            self._values["replace_unhealthy_instances"] = replace_unhealthy_instances
        if spot_options is not None:
            self._values["spot_options"] = spot_options
        if tag_specifications is not None:
            self._values["tag_specifications"] = tag_specifications
        if terminate_instances_with_expiration is not None:
            self._values["terminate_instances_with_expiration"] = terminate_instances_with_expiration
        if type is not None:
            self._values["type"] = type
        if valid_from is not None:
            self._values["valid_from"] = valid_from
        if valid_until is not None:
            self._values["valid_until"] = valid_until

    @builtins.property
    def launch_template_configs(
        self,
    ) -> typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty]]]:
        '''The configuration for the EC2 Fleet.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-launchtemplateconfigs
        '''
        result = self._values.get("launch_template_configs")
        assert result is not None, "Required property 'launch_template_configs' is missing"
        return typing.cast(typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty]]], result)

    @builtins.property
    def target_capacity_specification(
        self,
    ) -> typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.TargetCapacitySpecificationRequestProperty]:
        '''The number of units to request.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-targetcapacityspecification
        '''
        result = self._values.get("target_capacity_specification")
        assert result is not None, "Required property 'target_capacity_specification' is missing"
        return typing.cast(typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.TargetCapacitySpecificationRequestProperty], result)

    @builtins.property
    def context(self) -> typing.Optional[builtins.str]:
        '''Reserved.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-context
        '''
        result = self._values.get("context")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def excess_capacity_termination_policy(self) -> typing.Optional[builtins.str]:
        '''Indicates whether running instances should be terminated if the total target capacity of the EC2 Fleet is decreased below the current size of the EC2 Fleet.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-excesscapacityterminationpolicy
        '''
        result = self._values.get("excess_capacity_termination_policy")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def on_demand_options(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.OnDemandOptionsRequestProperty]]:
        '''Describes the configuration of On-Demand Instances in an EC2 Fleet.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-ondemandoptions
        '''
        result = self._values.get("on_demand_options")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.OnDemandOptionsRequestProperty]], result)

    @builtins.property
    def replace_unhealthy_instances(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''Indicates whether EC2 Fleet should replace unhealthy Spot Instances.

        Supported only for fleets of type ``maintain`` . For more information, see `EC2 Fleet health checks <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/manage-ec2-fleet.html#ec2-fleet-health-checks>`_ in the *Amazon EC2 User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-replaceunhealthyinstances
        '''
        result = self._values.get("replace_unhealthy_instances")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def spot_options(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.SpotOptionsRequestProperty]]:
        '''Describes the configuration of Spot Instances in an EC2 Fleet.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-spotoptions
        '''
        result = self._values.get("spot_options")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.SpotOptionsRequestProperty]], result)

    @builtins.property
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.TagSpecificationProperty]]]]:
        '''The key-value pair for tagging the EC2 Fleet request on creation. For more information, see `Tagging your resources <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Using_Tags.html#tag-resources>`_ .

        If the fleet type is ``instant`` , specify a resource type of ``fleet`` to tag the fleet or ``instance`` to tag the instances at launch.

        If the fleet type is ``maintain`` or ``request`` , specify a resource type of ``fleet`` to tag the fleet. You cannot specify a resource type of ``instance`` . To tag instances at launch, specify the tags in a `launch template <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html#create-launch-template>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-tagspecifications
        '''
        result = self._values.get("tag_specifications")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.TagSpecificationProperty]]]], result)

    @builtins.property
    def terminate_instances_with_expiration(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''Indicates whether running instances should be terminated when the EC2 Fleet expires.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-terminateinstanceswithexpiration
        '''
        result = self._values.get("terminate_instances_with_expiration")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def type(self) -> typing.Optional[builtins.str]:
        '''The fleet type. The default value is ``maintain`` .

        - ``maintain`` - The EC2 Fleet places an asynchronous request for your desired capacity, and continues to maintain your desired Spot capacity by replenishing interrupted Spot Instances.
        - ``request`` - The EC2 Fleet places an asynchronous one-time request for your desired capacity, but does submit Spot requests in alternative capacity pools if Spot capacity is unavailable, and does not maintain Spot capacity if Spot Instances are interrupted.
        - ``instant`` - The EC2 Fleet places a synchronous one-time request for your desired capacity, and returns errors for any instances that could not be launched.

        For more information, see `EC2 Fleet request types <https://docs.aws.amazon.com/https://docs.aws.amazon.com/ec2-fleet-request-type.html>`_ in the *Amazon EC2 User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-type
        '''
        result = self._values.get("type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def valid_from(self) -> typing.Optional[builtins.str]:
        '''The start date and time of the request, in UTC format (for example, *YYYY* - *MM* - *DD* T *HH* : *MM* : *SS* Z).

        The default is to start fulfilling the request immediately.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-validfrom
        '''
        result = self._values.get("valid_from")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def valid_until(self) -> typing.Optional[builtins.str]:
        '''The end date and time of the request, in UTC format (for example, *YYYY* - *MM* - *DD* T *HH* : *MM* : *SS* Z).

        At this point, no new EC2 Fleet requests are placed or able to fulfill the request. If no value is specified, the request remains until you cancel it.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-validuntil
        '''
        result = self._values.get("valid_until")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnEC2FleetProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnEIP(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnEIP",
):
    '''A CloudFormation ``AWS::EC2::EIP``.

    Specifies an Elastic IP (EIP) address and can, optionally, associate it with an Amazon EC2 instance.

    You can allocate an Elastic IP address from an address pool owned by AWS or from an address pool created from a public IPv4 address range that you have brought to AWS for use with your AWS resources using bring your own IP addresses (BYOIP). For more information, see `Bring Your Own IP Addresses (BYOIP) <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-byoip.html>`_ in the *Amazon EC2 User Guide* .

    [EC2-VPC] If you release an Elastic IP address, you might be able to recover it. You cannot recover an Elastic IP address that you released after it is allocated to another AWS account. You cannot recover an Elastic IP address for EC2-Classic. To attempt to recover an Elastic IP address that you released, specify it in this operation.

    An Elastic IP address is for use either in the EC2-Classic platform or in a VPC. By default, you can allocate 5 Elastic IP addresses for EC2-Classic per Region and 5 Elastic IP addresses for EC2-VPC per Region.

    For more information, see `Elastic IP Addresses <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/elastic-ip-addresses-eip.html>`_ in the *Amazon EC2 User Guide* .

    :cloudformationResource: AWS::EC2::EIP
    :exampleMetadata: infused
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip.html

    Example::

        # instance is of type Instance
        
        # my_zone is of type HostedZone
        
        
        elastic_ip = ec2.CfnEIP(self, "EIP",
            domain="vpc",
            instance_id=instance.instance_id
        )
        route53.ARecord(self, "ARecord",
            zone=my_zone,
            target=route53.RecordTarget.from_ip_addresses(elastic_ip.ref)
        )
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        domain: typing.Optional[builtins.str] = None,
        instance_id: typing.Optional[builtins.str] = None,
        public_ipv4_pool: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::EIP``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param domain: Indicates whether the Elastic IP address is for use with instances in a VPC or instance in EC2-Classic. Default: If the Region supports EC2-Classic, the default is ``standard`` . Otherwise, the default is ``vpc`` . Use when allocating an address for use with a VPC if the Region supports EC2-Classic. If you define an Elastic IP address and associate it with a VPC that is defined in the same template, you must declare a dependency on the VPC-gateway attachment by using the `DependsOn Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html>`_ on this resource.
        :param instance_id: The ID of the instance. .. epigraph:: Updates to the ``InstanceId`` property may require *some interruptions* . Updates on an EIP reassociates the address on its associated resource.
        :param public_ipv4_pool: The ID of an address pool that you own. Use this parameter to let Amazon EC2 select an address from the address pool. .. epigraph:: Updates to the ``PublicIpv4Pool`` property may require *some interruptions* . Updates on an EIP reassociates the address on its associated resource.
        :param tags: Any tags assigned to the Elastic IP address. .. epigraph:: Updates to the ``Tags`` property may require *some interruptions* . Updates on an EIP reassociates the address on its associated resource.
        '''
        props = CfnEIPProps(
            domain=domain,
            instance_id=instance_id,
            public_ipv4_pool=public_ipv4_pool,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrAllocationId")
    def attr_allocation_id(self) -> builtins.str:
        '''The ID that AWS assigns to represent the allocation of the address for use with Amazon VPC.

        This is returned only for VPC elastic IP addresses. For example, ``eipalloc-5723d13e`` .

        :cloudformationAttribute: AllocationId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrAllocationId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''Any tags assigned to the Elastic IP address.

        .. epigraph::

           Updates to the ``Tags`` property may require *some interruptions* . Updates on an EIP reassociates the address on its associated resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip.html#cfn-ec2-eip-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="domain")
    def domain(self) -> typing.Optional[builtins.str]:
        '''Indicates whether the Elastic IP address is for use with instances in a VPC or instance in EC2-Classic.

        Default: If the Region supports EC2-Classic, the default is ``standard`` . Otherwise, the default is ``vpc`` .

        Use when allocating an address for use with a VPC if the Region supports EC2-Classic.

        If you define an Elastic IP address and associate it with a VPC that is defined in the same template, you must declare a dependency on the VPC-gateway attachment by using the `DependsOn Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html>`_ on this resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip.html#cfn-ec2-eip-domain
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "domain"))

    @domain.setter
    def domain(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "domain", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceId")
    def instance_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the instance.

        .. epigraph::

           Updates to the ``InstanceId`` property may require *some interruptions* . Updates on an EIP reassociates the address on its associated resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip.html#cfn-ec2-eip-instanceid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "instanceId"))

    @instance_id.setter
    def instance_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "instanceId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="publicIpv4Pool")
    def public_ipv4_pool(self) -> typing.Optional[builtins.str]:
        '''The ID of an address pool that you own.

        Use this parameter to let Amazon EC2 select an address from the address pool.
        .. epigraph::

           Updates to the ``PublicIpv4Pool`` property may require *some interruptions* . Updates on an EIP reassociates the address on its associated resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip.html#cfn-ec2-eip-publicipv4pool
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "publicIpv4Pool"))

    @public_ipv4_pool.setter
    def public_ipv4_pool(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "publicIpv4Pool", value)


@jsii.implements(aws_cdk.core.IInspectable)
class CfnEIPAssociation(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnEIPAssociation",
):
    '''A CloudFormation ``AWS::EC2::EIPAssociation``.

    Associates an Elastic IP address with an instance or a network interface. Before you can use an Elastic IP address, you must allocate it to your account.

    An Elastic IP address is for use in either the EC2-Classic platform or in a VPC. For more information, see `Elastic IP Addresses <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/elastic-ip-addresses-eip.html>`_ in the *Amazon EC2 User Guide* .

    [EC2-Classic, VPC in an EC2-VPC-only account] If the Elastic IP address is already associated with a different instance, it is disassociated from that instance and associated with the specified instance. If you associate an Elastic IP address with an instance that has an existing Elastic IP address, the existing address is disassociated from the instance, but remains allocated to your account.

    [VPC in an EC2-Classic account] If you don't specify a private IP address, the Elastic IP address is associated with the primary IP address. If the Elastic IP address is already associated with a different instance or a network interface, you get an error unless you allow reassociation. You cannot associate an Elastic IP address with an instance or network interface that has an existing Elastic IP address.

    :cloudformationResource: AWS::EC2::EIPAssociation
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip-association.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ec2 as ec2
        
        cfn_eIPAssociation = ec2.CfnEIPAssociation(self, "MyCfnEIPAssociation",
            allocation_id="allocationId",
            eip="eip",
            instance_id="instanceId",
            network_interface_id="networkInterfaceId",
            private_ip_address="privateIpAddress"
        )
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        allocation_id: typing.Optional[builtins.str] = None,
        eip: typing.Optional[builtins.str] = None,
        instance_id: typing.Optional[builtins.str] = None,
        network_interface_id: typing.Optional[builtins.str] = None,
        private_ip_address: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::EIPAssociation``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param allocation_id: [EC2-VPC] The allocation ID. This is required for EC2-VPC.
        :param eip: [EC2-Classic] The Elastic IP address to associate with the instance. This is required for EC2-Classic.
        :param instance_id: The ID of the instance. The instance must have exactly one attached network interface. For EC2-VPC, you can specify either the instance ID or the network interface ID, but not both. For EC2-Classic, you must specify an instance ID and the instance must be in the running state.
        :param network_interface_id: [EC2-VPC] The ID of the network interface. If the instance has more than one network interface, you must specify a network interface ID. For EC2-VPC, you can specify either the instance ID or the network interface ID, but not both.
        :param private_ip_address: [EC2-VPC] The primary or secondary private IP address to associate with the Elastic IP address. If no private IP address is specified, the Elastic IP address is associated with the primary private IP address.
        '''
        props = CfnEIPAssociationProps(
            allocation_id=allocation_id,
            eip=eip,
            instance_id=instance_id,
            network_interface_id=network_interface_id,
            private_ip_address=private_ip_address,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="allocationId")
    def allocation_id(self) -> typing.Optional[builtins.str]:
        '''[EC2-VPC] The allocation ID.

        This is required for EC2-VPC.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip-association.html#cfn-ec2-eipassociation-allocationid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "allocationId"))

    @allocation_id.setter
    def allocation_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "allocationId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="eip")
    def eip(self) -> typing.Optional[builtins.str]:
        '''[EC2-Classic] The Elastic IP address to associate with the instance.

        This is required for EC2-Classic.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip-association.html#cfn-ec2-eipassociation-eip
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "eip"))

    @eip.setter
    def eip(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "eip", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceId")
    def instance_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the instance.

        The instance must have exactly one attached network interface. For EC2-VPC, you can specify either the instance ID or the network interface ID, but not both. For EC2-Classic, you must specify an instance ID and the instance must be in the running state.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip-association.html#cfn-ec2-eipassociation-instanceid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "instanceId"))

    @instance_id.setter
    def instance_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "instanceId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="networkInterfaceId")
    def network_interface_id(self) -> typing.Optional[builtins.str]:
        '''[EC2-VPC] The ID of the network interface.

        If the instance has more than one network interface, you must specify a network interface ID.

        For EC2-VPC, you can specify either the instance ID or the network interface ID, but not both.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip-association.html#cfn-ec2-eipassociation-networkinterfaceid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "networkInterfaceId"))

    @network_interface_id.setter
    def network_interface_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "networkInterfaceId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="privateIpAddress")
    def private_ip_address(self) -> typing.Optional[builtins.str]:
        '''[EC2-VPC] The primary or secondary private IP address to associate with the Elastic IP address.

        If no private IP address is specified, the Elastic IP address is associated with the primary private IP address.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip-association.html#cfn-ec2-eipassociation-PrivateIpAddress
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "privateIpAddress"))

    @private_ip_address.setter
    def private_ip_address(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "privateIpAddress", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnEIPAssociationProps",
    jsii_struct_bases=[],
    name_mapping={
        "allocation_id": "allocationId",
        "eip": "eip",
        "instance_id": "instanceId",
        "network_interface_id": "networkInterfaceId",
        "private_ip_address": "privateIpAddress",
    },
)
class CfnEIPAssociationProps:
    def __init__(
        self,
        *,
        allocation_id: typing.Optional[builtins.str] = None,
        eip: typing.Optional[builtins.str] = None,
        instance_id: typing.Optional[builtins.str] = None,
        network_interface_id: typing.Optional[builtins.str] = None,
        private_ip_address: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnEIPAssociation``.

        :param allocation_id: [EC2-VPC] The allocation ID. This is required for EC2-VPC.
        :param eip: [EC2-Classic] The Elastic IP address to associate with the instance. This is required for EC2-Classic.
        :param instance_id: The ID of the instance. The instance must have exactly one attached network interface. For EC2-VPC, you can specify either the instance ID or the network interface ID, but not both. For EC2-Classic, you must specify an instance ID and the instance must be in the running state.
        :param network_interface_id: [EC2-VPC] The ID of the network interface. If the instance has more than one network interface, you must specify a network interface ID. For EC2-VPC, you can specify either the instance ID or the network interface ID, but not both.
        :param private_ip_address: [EC2-VPC] The primary or secondary private IP address to associate with the Elastic IP address. If no private IP address is specified, the Elastic IP address is associated with the primary private IP address.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip-association.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            cfn_eIPAssociation_props = ec2.CfnEIPAssociationProps(
                allocation_id="allocationId",
                eip="eip",
                instance_id="instanceId",
                network_interface_id="networkInterfaceId",
                private_ip_address="privateIpAddress"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if allocation_id is not None:
            self._values["allocation_id"] = allocation_id
        if eip is not None:
            self._values["eip"] = eip
        if instance_id is not None:
            self._values["instance_id"] = instance_id
        if network_interface_id is not None:
            self._values["network_interface_id"] = network_interface_id
        if private_ip_address is not None:
            self._values["private_ip_address"] = private_ip_address

    @builtins.property
    def allocation_id(self) -> typing.Optional[builtins.str]:
        '''[EC2-VPC] The allocation ID.

        This is required for EC2-VPC.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip-association.html#cfn-ec2-eipassociation-allocationid
        '''
        result = self._values.get("allocation_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def eip(self) -> typing.Optional[builtins.str]:
        '''[EC2-Classic] The Elastic IP address to associate with the instance.

        This is required for EC2-Classic.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip-association.html#cfn-ec2-eipassociation-eip
        '''
        result = self._values.get("eip")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def instance_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the instance.

        The instance must have exactly one attached network interface. For EC2-VPC, you can specify either the instance ID or the network interface ID, but not both. For EC2-Classic, you must specify an instance ID and the instance must be in the running state.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip-association.html#cfn-ec2-eipassociation-instanceid
        '''
        result = self._values.get("instance_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def network_interface_id(self) -> typing.Optional[builtins.str]:
        '''[EC2-VPC] The ID of the network interface.

        If the instance has more than one network interface, you must specify a network interface ID.

        For EC2-VPC, you can specify either the instance ID or the network interface ID, but not both.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip-association.html#cfn-ec2-eipassociation-networkinterfaceid
        '''
        result = self._values.get("network_interface_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def private_ip_address(self) -> typing.Optional[builtins.str]:
        '''[EC2-VPC] The primary or secondary private IP address to associate with the Elastic IP address.

        If no private IP address is specified, the Elastic IP address is associated with the primary private IP address.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip-association.html#cfn-ec2-eipassociation-PrivateIpAddress
        '''
        result = self._values.get("private_ip_address")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnEIPAssociationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnEIPProps",
    jsii_struct_bases=[],
    name_mapping={
        "domain": "domain",
        "instance_id": "instanceId",
        "public_ipv4_pool": "publicIpv4Pool",
        "tags": "tags",
    },
)
class CfnEIPProps:
    def __init__(
        self,
        *,
        domain: typing.Optional[builtins.str] = None,
        instance_id: typing.Optional[builtins.str] = None,
        public_ipv4_pool: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Properties for defining a ``CfnEIP``.

        :param domain: Indicates whether the Elastic IP address is for use with instances in a VPC or instance in EC2-Classic. Default: If the Region supports EC2-Classic, the default is ``standard`` . Otherwise, the default is ``vpc`` . Use when allocating an address for use with a VPC if the Region supports EC2-Classic. If you define an Elastic IP address and associate it with a VPC that is defined in the same template, you must declare a dependency on the VPC-gateway attachment by using the `DependsOn Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html>`_ on this resource.
        :param instance_id: The ID of the instance. .. epigraph:: Updates to the ``InstanceId`` property may require *some interruptions* . Updates on an EIP reassociates the address on its associated resource.
        :param public_ipv4_pool: The ID of an address pool that you own. Use this parameter to let Amazon EC2 select an address from the address pool. .. epigraph:: Updates to the ``PublicIpv4Pool`` property may require *some interruptions* . Updates on an EIP reassociates the address on its associated resource.
        :param tags: Any tags assigned to the Elastic IP address. .. epigraph:: Updates to the ``Tags`` property may require *some interruptions* . Updates on an EIP reassociates the address on its associated resource.

        :exampleMetadata: infused
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip.html

        Example::

            # instance is of type Instance
            
            # my_zone is of type HostedZone
            
            
            elastic_ip = ec2.CfnEIP(self, "EIP",
                domain="vpc",
                instance_id=instance.instance_id
            )
            route53.ARecord(self, "ARecord",
                zone=my_zone,
                target=route53.RecordTarget.from_ip_addresses(elastic_ip.ref)
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if domain is not None:
            self._values["domain"] = domain
        if instance_id is not None:
            self._values["instance_id"] = instance_id
        if public_ipv4_pool is not None:
            self._values["public_ipv4_pool"] = public_ipv4_pool
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def domain(self) -> typing.Optional[builtins.str]:
        '''Indicates whether the Elastic IP address is for use with instances in a VPC or instance in EC2-Classic.

        Default: If the Region supports EC2-Classic, the default is ``standard`` . Otherwise, the default is ``vpc`` .

        Use when allocating an address for use with a VPC if the Region supports EC2-Classic.

        If you define an Elastic IP address and associate it with a VPC that is defined in the same template, you must declare a dependency on the VPC-gateway attachment by using the `DependsOn Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html>`_ on this resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip.html#cfn-ec2-eip-domain
        '''
        result = self._values.get("domain")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def instance_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the instance.

        .. epigraph::

           Updates to the ``InstanceId`` property may require *some interruptions* . Updates on an EIP reassociates the address on its associated resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip.html#cfn-ec2-eip-instanceid
        '''
        result = self._values.get("instance_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def public_ipv4_pool(self) -> typing.Optional[builtins.str]:
        '''The ID of an address pool that you own.

        Use this parameter to let Amazon EC2 select an address from the address pool.
        .. epigraph::

           Updates to the ``PublicIpv4Pool`` property may require *some interruptions* . Updates on an EIP reassociates the address on its associated resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip.html#cfn-ec2-eip-publicipv4pool
        '''
        result = self._values.get("public_ipv4_pool")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''Any tags assigned to the Elastic IP address.

        .. epigraph::

           Updates to the ``Tags`` property may require *some interruptions* . Updates on an EIP reassociates the address on its associated resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip.html#cfn-ec2-eip-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnEIPProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnEgressOnlyInternetGateway(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnEgressOnlyInternetGateway",
):
    '''A CloudFormation ``AWS::EC2::EgressOnlyInternetGateway``.

    [IPv6 only] Specifies an egress-only internet gateway for your VPC. An egress-only internet gateway is used to enable outbound communication over IPv6 from instances in your VPC to the internet, and prevents hosts outside of your VPC from initiating an IPv6 connection with your instance.

    :cloudformationResource: AWS::EC2::EgressOnlyInternetGateway
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-egressonlyinternetgateway.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ec2 as ec2
        
        cfn_egress_only_internet_gateway = ec2.CfnEgressOnlyInternetGateway(self, "MyCfnEgressOnlyInternetGateway",
            vpc_id="vpcId"
        )
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        vpc_id: builtins.str,
    ) -> None:
        '''Create a new ``AWS::EC2::EgressOnlyInternetGateway``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param vpc_id: The ID of the VPC for which to create the egress-only internet gateway.
        '''
        props = CfnEgressOnlyInternetGatewayProps(vpc_id=vpc_id)

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The ID of the egress-only internet gateway.

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="vpcId")
    def vpc_id(self) -> builtins.str:
        '''The ID of the VPC for which to create the egress-only internet gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-egressonlyinternetgateway.html#cfn-ec2-egressonlyinternetgateway-vpcid
        '''
        return typing.cast(builtins.str, jsii.get(self, "vpcId"))

    @vpc_id.setter
    def vpc_id(self, value: builtins.str) -> None:
        jsii.set(self, "vpcId", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnEgressOnlyInternetGatewayProps",
    jsii_struct_bases=[],
    name_mapping={"vpc_id": "vpcId"},
)
class CfnEgressOnlyInternetGatewayProps:
    def __init__(self, *, vpc_id: builtins.str) -> None:
        '''Properties for defining a ``CfnEgressOnlyInternetGateway``.

        :param vpc_id: The ID of the VPC for which to create the egress-only internet gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-egressonlyinternetgateway.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            cfn_egress_only_internet_gateway_props = ec2.CfnEgressOnlyInternetGatewayProps(
                vpc_id="vpcId"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "vpc_id": vpc_id,
        }

    @builtins.property
    def vpc_id(self) -> builtins.str:
        '''The ID of the VPC for which to create the egress-only internet gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-egressonlyinternetgateway.html#cfn-ec2-egressonlyinternetgateway-vpcid
        '''
        result = self._values.get("vpc_id")
        assert result is not None, "Required property 'vpc_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnEgressOnlyInternetGatewayProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnEnclaveCertificateIamRoleAssociation(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnEnclaveCertificateIamRoleAssociation",
):
    '''A CloudFormation ``AWS::EC2::EnclaveCertificateIamRoleAssociation``.

    Associates an AWS Identity and Access Management (IAM) role with an AWS Certificate Manager (ACM) certificate. This enables the certificate to be used by the ACM for Nitro Enclaves application inside an enclave. For more information, see `AWS Certificate Manager for Nitro Enclaves <https://docs.aws.amazon.com/enclaves/latest/user/nitro-enclave-refapp.html>`_ in the *AWS Nitro Enclaves User Guide* .

    When the IAM role is associated with the ACM certificate, the certificate, certificate chain, and encrypted private key are placed in an Amazon S3 bucket that only the associated IAM role can access. The private key of the certificate is encrypted with an AWS managed key that has an attached attestation-based key policy.

    To enable the IAM role to access the Amazon S3 object, you must grant it permission to call ``s3:GetObject`` on the Amazon S3 bucket returned by the command. To enable the IAM role to access the KMS key, you must grant it permission to call ``kms:Decrypt`` on the KMS key returned by the command. For more information, see `Grant the role permission to access the certificate and encryption key <https://docs.aws.amazon.com/enclaves/latest/user/nitro-enclave-refapp.html#add-policy>`_ in the *AWS Nitro Enclaves User Guide* .

    :cloudformationResource: AWS::EC2::EnclaveCertificateIamRoleAssociation
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-enclavecertificateiamroleassociation.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ec2 as ec2
        
        cfn_enclave_certificate_iam_role_association = ec2.CfnEnclaveCertificateIamRoleAssociation(self, "MyCfnEnclaveCertificateIamRoleAssociation",
            certificate_arn="certificateArn",
            role_arn="roleArn"
        )
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        certificate_arn: builtins.str,
        role_arn: builtins.str,
    ) -> None:
        '''Create a new ``AWS::EC2::EnclaveCertificateIamRoleAssociation``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param certificate_arn: The ARN of the ACM certificate with which to associate the IAM role.
        :param role_arn: The ARN of the IAM role to associate with the ACM certificate. You can associate up to 16 IAM roles with an ACM certificate.
        '''
        props = CfnEnclaveCertificateIamRoleAssociationProps(
            certificate_arn=certificate_arn, role_arn=role_arn
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrCertificateS3BucketName")
    def attr_certificate_s3_bucket_name(self) -> builtins.str:
        '''The name of the Amazon S3 bucket to which the certificate was uploaded.

        :cloudformationAttribute: CertificateS3BucketName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCertificateS3BucketName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrCertificateS3ObjectKey")
    def attr_certificate_s3_object_key(self) -> builtins.str:
        '''The Amazon S3 object key where the certificate, certificate chain, and encrypted private key bundle are stored.

        The object key is formatted as follows: ``role_arn`` / ``certificate_arn`` .

        :cloudformationAttribute: CertificateS3ObjectKey
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCertificateS3ObjectKey"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrEncryptionKmsKeyId")
    def attr_encryption_kms_key_id(self) -> builtins.str:
        '''The ID of the AWS KMS key used to encrypt the private key of the certificate.

        :cloudformationAttribute: EncryptionKmsKeyId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrEncryptionKmsKeyId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="certificateArn")
    def certificate_arn(self) -> builtins.str:
        '''The ARN of the ACM certificate with which to associate the IAM role.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-enclavecertificateiamroleassociation.html#cfn-ec2-enclavecertificateiamroleassociation-certificatearn
        '''
        return typing.cast(builtins.str, jsii.get(self, "certificateArn"))

    @certificate_arn.setter
    def certificate_arn(self, value: builtins.str) -> None:
        jsii.set(self, "certificateArn", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="roleArn")
    def role_arn(self) -> builtins.str:
        '''The ARN of the IAM role to associate with the ACM certificate.

        You can associate up to 16 IAM roles with an ACM certificate.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-enclavecertificateiamroleassociation.html#cfn-ec2-enclavecertificateiamroleassociation-rolearn
        '''
        return typing.cast(builtins.str, jsii.get(self, "roleArn"))

    @role_arn.setter
    def role_arn(self, value: builtins.str) -> None:
        jsii.set(self, "roleArn", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnEnclaveCertificateIamRoleAssociationProps",
    jsii_struct_bases=[],
    name_mapping={"certificate_arn": "certificateArn", "role_arn": "roleArn"},
)
class CfnEnclaveCertificateIamRoleAssociationProps:
    def __init__(
        self,
        *,
        certificate_arn: builtins.str,
        role_arn: builtins.str,
    ) -> None:
        '''Properties for defining a ``CfnEnclaveCertificateIamRoleAssociation``.

        :param certificate_arn: The ARN of the ACM certificate with which to associate the IAM role.
        :param role_arn: The ARN of the IAM role to associate with the ACM certificate. You can associate up to 16 IAM roles with an ACM certificate.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-enclavecertificateiamroleassociation.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            cfn_enclave_certificate_iam_role_association_props = ec2.CfnEnclaveCertificateIamRoleAssociationProps(
                certificate_arn="certificateArn",
                role_arn="roleArn"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "certificate_arn": certificate_arn,
            "role_arn": role_arn,
        }

    @builtins.property
    def certificate_arn(self) -> builtins.str:
        '''The ARN of the ACM certificate with which to associate the IAM role.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-enclavecertificateiamroleassociation.html#cfn-ec2-enclavecertificateiamroleassociation-certificatearn
        '''
        result = self._values.get("certificate_arn")
        assert result is not None, "Required property 'certificate_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def role_arn(self) -> builtins.str:
        '''The ARN of the IAM role to associate with the ACM certificate.

        You can associate up to 16 IAM roles with an ACM certificate.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-enclavecertificateiamroleassociation.html#cfn-ec2-enclavecertificateiamroleassociation-rolearn
        '''
        result = self._values.get("role_arn")
        assert result is not None, "Required property 'role_arn' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnEnclaveCertificateIamRoleAssociationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnFlowLog(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnFlowLog",
):
    '''A CloudFormation ``AWS::EC2::FlowLog``.

    Specifies a VPC flow log that captures IP traffic for a specified network interface, subnet, or VPC. To view the log data, use Amazon CloudWatch Logs (CloudWatch Logs) to help troubleshoot connection issues. For example, you can use a flow log to investigate why certain traffic isn't reaching an instance, which can help you diagnose overly restrictive security group rules. For more information, see `VPC Flow Logs <https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html>`_ in the *Amazon VPC User Guide* .

    :cloudformationResource: AWS::EC2::FlowLog
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ec2 as ec2
        
        cfn_flow_log = ec2.CfnFlowLog(self, "MyCfnFlowLog",
            resource_id="resourceId",
            resource_type="resourceType",
            traffic_type="trafficType",
        
            # the properties below are optional
            deliver_logs_permission_arn="deliverLogsPermissionArn",
            log_destination="logDestination",
            log_destination_type="logDestinationType",
            log_format="logFormat",
            log_group_name="logGroupName",
            max_aggregation_interval=123,
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        resource_id: builtins.str,
        resource_type: builtins.str,
        traffic_type: builtins.str,
        deliver_logs_permission_arn: typing.Optional[builtins.str] = None,
        log_destination: typing.Optional[builtins.str] = None,
        log_destination_type: typing.Optional[builtins.str] = None,
        log_format: typing.Optional[builtins.str] = None,
        log_group_name: typing.Optional[builtins.str] = None,
        max_aggregation_interval: typing.Optional[jsii.Number] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::FlowLog``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param resource_id: The ID of the subnet, network interface, or VPC for which you want to create a flow log.
        :param resource_type: The type of resource for which to create the flow log. For example, if you specified a VPC ID for the ``ResourceId`` property, specify ``VPC`` for this property.
        :param traffic_type: The type of traffic to log. You can log traffic that the resource accepts or rejects, or all traffic.
        :param deliver_logs_permission_arn: The ARN for the IAM role that permits Amazon EC2 to publish flow logs to a CloudWatch Logs log group in your account. If you specify ``LogDestinationType`` as ``s3`` , do not specify ``DeliverLogsPermissionArn`` or ``LogGroupName`` .
        :param log_destination: The destination to which the flow log data is to be published. Flow log data can be published to a CloudWatch Logs log group or an Amazon S3 bucket. The value specified for this parameter depends on the value specified for ``LogDestinationType`` . If ``LogDestinationType`` is not specified or ``cloud-watch-logs`` , specify the Amazon Resource Name (ARN) of the CloudWatch Logs log group. For example, to publish to a log group called ``my-logs`` , specify ``arn:aws:logs:us-east-1:123456789012:log-group:my-logs`` . Alternatively, use ``LogGroupName`` instead. If LogDestinationType is ``s3`` , specify the ARN of the Amazon S3 bucket. You can also specify a subfolder in the bucket. To specify a subfolder in the bucket, use the following ARN format: ``bucket_ARN/subfolder_name/`` . For example, to specify a subfolder named ``my-logs`` in a bucket named ``my-bucket`` , use the following ARN: ``arn:aws:s3:::my-bucket/my-logs/`` . You cannot use ``AWSLogs`` as a subfolder name. This is a reserved term.
        :param log_destination_type: The type of destination to which the flow log data is to be published. Flow log data can be published to CloudWatch Logs or Amazon S3. To publish flow log data to CloudWatch Logs, specify ``cloud-watch-logs`` . To publish flow log data to Amazon S3, specify ``s3`` . If you specify ``LogDestinationType`` as ``s3`` , do not specify ``DeliverLogsPermissionArn`` or ``LogGroupName`` . Default: ``cloud-watch-logs``
        :param log_format: The fields to include in the flow log record, in the order in which they should appear. For a list of available fields, see `Flow Log Records <https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html#flow-log-records>`_ . If you omit this parameter, the flow log is created using the default format. If you specify this parameter, you must specify at least one field. Specify the fields using the ``${field-id}`` format, separated by spaces.
        :param log_group_name: The name of a new or existing CloudWatch Logs log group where Amazon EC2 publishes your flow logs. If you specify ``LogDestinationType`` as ``s3`` , do not specify ``DeliverLogsPermissionArn`` or ``LogGroupName`` .
        :param max_aggregation_interval: The maximum interval of time during which a flow of packets is captured and aggregated into a flow log record. You can specify 60 seconds (1 minute) or 600 seconds (10 minutes). When a network interface is attached to a `Nitro-based instance <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances>`_ , the aggregation interval is always 60 seconds or less, regardless of the value that you specify. Default: 600
        :param tags: The tags for the flow log.
        '''
        props = CfnFlowLogProps(
            resource_id=resource_id,
            resource_type=resource_type,
            traffic_type=traffic_type,
            deliver_logs_permission_arn=deliver_logs_permission_arn,
            log_destination=log_destination,
            log_destination_type=log_destination_type,
            log_format=log_format,
            log_group_name=log_group_name,
            max_aggregation_interval=max_aggregation_interval,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The ID of the flow log.

        For example, ``fl-123456abc123abc1`` .

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''The tags for the flow log.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="resourceId")
    def resource_id(self) -> builtins.str:
        '''The ID of the subnet, network interface, or VPC for which you want to create a flow log.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-resourceid
        '''
        return typing.cast(builtins.str, jsii.get(self, "resourceId"))

    @resource_id.setter
    def resource_id(self, value: builtins.str) -> None:
        jsii.set(self, "resourceId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="resourceType")
    def resource_type(self) -> builtins.str:
        '''The type of resource for which to create the flow log.

        For example, if you specified a VPC ID for the ``ResourceId`` property, specify ``VPC`` for this property.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-resourcetype
        '''
        return typing.cast(builtins.str, jsii.get(self, "resourceType"))

    @resource_type.setter
    def resource_type(self, value: builtins.str) -> None:
        jsii.set(self, "resourceType", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="trafficType")
    def traffic_type(self) -> builtins.str:
        '''The type of traffic to log.

        You can log traffic that the resource accepts or rejects, or all traffic.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-traffictype
        '''
        return typing.cast(builtins.str, jsii.get(self, "trafficType"))

    @traffic_type.setter
    def traffic_type(self, value: builtins.str) -> None:
        jsii.set(self, "trafficType", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="deliverLogsPermissionArn")
    def deliver_logs_permission_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN for the IAM role that permits Amazon EC2 to publish flow logs to a CloudWatch Logs log group in your account.

        If you specify ``LogDestinationType`` as ``s3`` , do not specify ``DeliverLogsPermissionArn`` or ``LogGroupName`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-deliverlogspermissionarn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "deliverLogsPermissionArn"))

    @deliver_logs_permission_arn.setter
    def deliver_logs_permission_arn(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "deliverLogsPermissionArn", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="logDestination")
    def log_destination(self) -> typing.Optional[builtins.str]:
        '''The destination to which the flow log data is to be published.

        Flow log data can be published to a CloudWatch Logs log group or an Amazon S3 bucket. The value specified for this parameter depends on the value specified for ``LogDestinationType`` .

        If ``LogDestinationType`` is not specified or ``cloud-watch-logs`` , specify the Amazon Resource Name (ARN) of the CloudWatch Logs log group. For example, to publish to a log group called ``my-logs`` , specify ``arn:aws:logs:us-east-1:123456789012:log-group:my-logs`` . Alternatively, use ``LogGroupName`` instead.

        If LogDestinationType is ``s3`` , specify the ARN of the Amazon S3 bucket. You can also specify a subfolder in the bucket. To specify a subfolder in the bucket, use the following ARN format: ``bucket_ARN/subfolder_name/`` . For example, to specify a subfolder named ``my-logs`` in a bucket named ``my-bucket`` , use the following ARN: ``arn:aws:s3:::my-bucket/my-logs/`` . You cannot use ``AWSLogs`` as a subfolder name. This is a reserved term.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-logdestination
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "logDestination"))

    @log_destination.setter
    def log_destination(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "logDestination", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="logDestinationType")
    def log_destination_type(self) -> typing.Optional[builtins.str]:
        '''The type of destination to which the flow log data is to be published.

        Flow log data can be published to CloudWatch Logs or Amazon S3. To publish flow log data to CloudWatch Logs, specify ``cloud-watch-logs`` . To publish flow log data to Amazon S3, specify ``s3`` .

        If you specify ``LogDestinationType`` as ``s3`` , do not specify ``DeliverLogsPermissionArn`` or ``LogGroupName`` .

        Default: ``cloud-watch-logs``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-logdestinationtype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "logDestinationType"))

    @log_destination_type.setter
    def log_destination_type(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "logDestinationType", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="logFormat")
    def log_format(self) -> typing.Optional[builtins.str]:
        '''The fields to include in the flow log record, in the order in which they should appear.

        For a list of available fields, see `Flow Log Records <https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html#flow-log-records>`_ . If you omit this parameter, the flow log is created using the default format. If you specify this parameter, you must specify at least one field.

        Specify the fields using the ``${field-id}`` format, separated by spaces.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-logformat
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "logFormat"))

    @log_format.setter
    def log_format(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "logFormat", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="logGroupName")
    def log_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of a new or existing CloudWatch Logs log group where Amazon EC2 publishes your flow logs.

        If you specify ``LogDestinationType`` as ``s3`` , do not specify ``DeliverLogsPermissionArn`` or ``LogGroupName`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-loggroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "logGroupName"))

    @log_group_name.setter
    def log_group_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "logGroupName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="maxAggregationInterval")
    def max_aggregation_interval(self) -> typing.Optional[jsii.Number]:
        '''The maximum interval of time during which a flow of packets is captured and aggregated into a flow log record.

        You can specify 60 seconds (1 minute) or 600 seconds (10 minutes).

        When a network interface is attached to a `Nitro-based instance <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances>`_ , the aggregation interval is always 60 seconds or less, regardless of the value that you specify.

        Default: 600

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-maxaggregationinterval
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "maxAggregationInterval"))

    @max_aggregation_interval.setter
    def max_aggregation_interval(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "maxAggregationInterval", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnFlowLogProps",
    jsii_struct_bases=[],
    name_mapping={
        "resource_id": "resourceId",
        "resource_type": "resourceType",
        "traffic_type": "trafficType",
        "deliver_logs_permission_arn": "deliverLogsPermissionArn",
        "log_destination": "logDestination",
        "log_destination_type": "logDestinationType",
        "log_format": "logFormat",
        "log_group_name": "logGroupName",
        "max_aggregation_interval": "maxAggregationInterval",
        "tags": "tags",
    },
)
class CfnFlowLogProps:
    def __init__(
        self,
        *,
        resource_id: builtins.str,
        resource_type: builtins.str,
        traffic_type: builtins.str,
        deliver_logs_permission_arn: typing.Optional[builtins.str] = None,
        log_destination: typing.Optional[builtins.str] = None,
        log_destination_type: typing.Optional[builtins.str] = None,
        log_format: typing.Optional[builtins.str] = None,
        log_group_name: typing.Optional[builtins.str] = None,
        max_aggregation_interval: typing.Optional[jsii.Number] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Properties for defining a ``CfnFlowLog``.

        :param resource_id: The ID of the subnet, network interface, or VPC for which you want to create a flow log.
        :param resource_type: The type of resource for which to create the flow log. For example, if you specified a VPC ID for the ``ResourceId`` property, specify ``VPC`` for this property.
        :param traffic_type: The type of traffic to log. You can log traffic that the resource accepts or rejects, or all traffic.
        :param deliver_logs_permission_arn: The ARN for the IAM role that permits Amazon EC2 to publish flow logs to a CloudWatch Logs log group in your account. If you specify ``LogDestinationType`` as ``s3`` , do not specify ``DeliverLogsPermissionArn`` or ``LogGroupName`` .
        :param log_destination: The destination to which the flow log data is to be published. Flow log data can be published to a CloudWatch Logs log group or an Amazon S3 bucket. The value specified for this parameter depends on the value specified for ``LogDestinationType`` . If ``LogDestinationType`` is not specified or ``cloud-watch-logs`` , specify the Amazon Resource Name (ARN) of the CloudWatch Logs log group. For example, to publish to a log group called ``my-logs`` , specify ``arn:aws:logs:us-east-1:123456789012:log-group:my-logs`` . Alternatively, use ``LogGroupName`` instead. If LogDestinationType is ``s3`` , specify the ARN of the Amazon S3 bucket. You can also specify a subfolder in the bucket. To specify a subfolder in the bucket, use the following ARN format: ``bucket_ARN/subfolder_name/`` . For example, to specify a subfolder named ``my-logs`` in a bucket named ``my-bucket`` , use the following ARN: ``arn:aws:s3:::my-bucket/my-logs/`` . You cannot use ``AWSLogs`` as a subfolder name. This is a reserved term.
        :param log_destination_type: The type of destination to which the flow log data is to be published. Flow log data can be published to CloudWatch Logs or Amazon S3. To publish flow log data to CloudWatch Logs, specify ``cloud-watch-logs`` . To publish flow log data to Amazon S3, specify ``s3`` . If you specify ``LogDestinationType`` as ``s3`` , do not specify ``DeliverLogsPermissionArn`` or ``LogGroupName`` . Default: ``cloud-watch-logs``
        :param log_format: The fields to include in the flow log record, in the order in which they should appear. For a list of available fields, see `Flow Log Records <https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html#flow-log-records>`_ . If you omit this parameter, the flow log is created using the default format. If you specify this parameter, you must specify at least one field. Specify the fields using the ``${field-id}`` format, separated by spaces.
        :param log_group_name: The name of a new or existing CloudWatch Logs log group where Amazon EC2 publishes your flow logs. If you specify ``LogDestinationType`` as ``s3`` , do not specify ``DeliverLogsPermissionArn`` or ``LogGroupName`` .
        :param max_aggregation_interval: The maximum interval of time during which a flow of packets is captured and aggregated into a flow log record. You can specify 60 seconds (1 minute) or 600 seconds (10 minutes). When a network interface is attached to a `Nitro-based instance <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances>`_ , the aggregation interval is always 60 seconds or less, regardless of the value that you specify. Default: 600
        :param tags: The tags for the flow log.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            cfn_flow_log_props = ec2.CfnFlowLogProps(
                resource_id="resourceId",
                resource_type="resourceType",
                traffic_type="trafficType",
            
                # the properties below are optional
                deliver_logs_permission_arn="deliverLogsPermissionArn",
                log_destination="logDestination",
                log_destination_type="logDestinationType",
                log_format="logFormat",
                log_group_name="logGroupName",
                max_aggregation_interval=123,
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "resource_id": resource_id,
            "resource_type": resource_type,
            "traffic_type": traffic_type,
        }
        if deliver_logs_permission_arn is not None:
            self._values["deliver_logs_permission_arn"] = deliver_logs_permission_arn
        if log_destination is not None:
            self._values["log_destination"] = log_destination
        if log_destination_type is not None:
            self._values["log_destination_type"] = log_destination_type
        if log_format is not None:
            self._values["log_format"] = log_format
        if log_group_name is not None:
            self._values["log_group_name"] = log_group_name
        if max_aggregation_interval is not None:
            self._values["max_aggregation_interval"] = max_aggregation_interval
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def resource_id(self) -> builtins.str:
        '''The ID of the subnet, network interface, or VPC for which you want to create a flow log.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-resourceid
        '''
        result = self._values.get("resource_id")
        assert result is not None, "Required property 'resource_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def resource_type(self) -> builtins.str:
        '''The type of resource for which to create the flow log.

        For example, if you specified a VPC ID for the ``ResourceId`` property, specify ``VPC`` for this property.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-resourcetype
        '''
        result = self._values.get("resource_type")
        assert result is not None, "Required property 'resource_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def traffic_type(self) -> builtins.str:
        '''The type of traffic to log.

        You can log traffic that the resource accepts or rejects, or all traffic.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-traffictype
        '''
        result = self._values.get("traffic_type")
        assert result is not None, "Required property 'traffic_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def deliver_logs_permission_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN for the IAM role that permits Amazon EC2 to publish flow logs to a CloudWatch Logs log group in your account.

        If you specify ``LogDestinationType`` as ``s3`` , do not specify ``DeliverLogsPermissionArn`` or ``LogGroupName`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-deliverlogspermissionarn
        '''
        result = self._values.get("deliver_logs_permission_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def log_destination(self) -> typing.Optional[builtins.str]:
        '''The destination to which the flow log data is to be published.

        Flow log data can be published to a CloudWatch Logs log group or an Amazon S3 bucket. The value specified for this parameter depends on the value specified for ``LogDestinationType`` .

        If ``LogDestinationType`` is not specified or ``cloud-watch-logs`` , specify the Amazon Resource Name (ARN) of the CloudWatch Logs log group. For example, to publish to a log group called ``my-logs`` , specify ``arn:aws:logs:us-east-1:123456789012:log-group:my-logs`` . Alternatively, use ``LogGroupName`` instead.

        If LogDestinationType is ``s3`` , specify the ARN of the Amazon S3 bucket. You can also specify a subfolder in the bucket. To specify a subfolder in the bucket, use the following ARN format: ``bucket_ARN/subfolder_name/`` . For example, to specify a subfolder named ``my-logs`` in a bucket named ``my-bucket`` , use the following ARN: ``arn:aws:s3:::my-bucket/my-logs/`` . You cannot use ``AWSLogs`` as a subfolder name. This is a reserved term.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-logdestination
        '''
        result = self._values.get("log_destination")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def log_destination_type(self) -> typing.Optional[builtins.str]:
        '''The type of destination to which the flow log data is to be published.

        Flow log data can be published to CloudWatch Logs or Amazon S3. To publish flow log data to CloudWatch Logs, specify ``cloud-watch-logs`` . To publish flow log data to Amazon S3, specify ``s3`` .

        If you specify ``LogDestinationType`` as ``s3`` , do not specify ``DeliverLogsPermissionArn`` or ``LogGroupName`` .

        Default: ``cloud-watch-logs``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-logdestinationtype
        '''
        result = self._values.get("log_destination_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def log_format(self) -> typing.Optional[builtins.str]:
        '''The fields to include in the flow log record, in the order in which they should appear.

        For a list of available fields, see `Flow Log Records <https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html#flow-log-records>`_ . If you omit this parameter, the flow log is created using the default format. If you specify this parameter, you must specify at least one field.

        Specify the fields using the ``${field-id}`` format, separated by spaces.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-logformat
        '''
        result = self._values.get("log_format")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def log_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of a new or existing CloudWatch Logs log group where Amazon EC2 publishes your flow logs.

        If you specify ``LogDestinationType`` as ``s3`` , do not specify ``DeliverLogsPermissionArn`` or ``LogGroupName`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-loggroupname
        '''
        result = self._values.get("log_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_aggregation_interval(self) -> typing.Optional[jsii.Number]:
        '''The maximum interval of time during which a flow of packets is captured and aggregated into a flow log record.

        You can specify 60 seconds (1 minute) or 600 seconds (10 minutes).

        When a network interface is attached to a `Nitro-based instance <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances>`_ , the aggregation interval is always 60 seconds or less, regardless of the value that you specify.

        Default: 600

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-maxaggregationinterval
        '''
        result = self._values.get("max_aggregation_interval")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''The tags for the flow log.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnFlowLogProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnGatewayRouteTableAssociation(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnGatewayRouteTableAssociation",
):
    '''A CloudFormation ``AWS::EC2::GatewayRouteTableAssociation``.

    Associates a virtual private gateway or internet gateway with a route table. The gateway and route table must be in the same VPC. This association causes the incoming traffic to the gateway to be routed according to the routes in the route table.

    :cloudformationResource: AWS::EC2::GatewayRouteTableAssociation
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-gatewayroutetableassociation.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ec2 as ec2
        
        cfn_gateway_route_table_association = ec2.CfnGatewayRouteTableAssociation(self, "MyCfnGatewayRouteTableAssociation",
            gateway_id="gatewayId",
            route_table_id="routeTableId"
        )
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        gateway_id: builtins.str,
        route_table_id: builtins.str,
    ) -> None:
        '''Create a new ``AWS::EC2::GatewayRouteTableAssociation``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param gateway_id: The ID of the gateway.
        :param route_table_id: The ID of the route table.
        '''
        props = CfnGatewayRouteTableAssociationProps(
            gateway_id=gateway_id, route_table_id=route_table_id
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrAssociationId")
    def attr_association_id(self) -> builtins.str:
        '''The ID of the route table association.

        :cloudformationAttribute: AssociationId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrAssociationId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="gatewayId")
    def gateway_id(self) -> builtins.str:
        '''The ID of the gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-gatewayroutetableassociation.html#cfn-ec2-gatewayroutetableassociation-gatewayid
        '''
        return typing.cast(builtins.str, jsii.get(self, "gatewayId"))

    @gateway_id.setter
    def gateway_id(self, value: builtins.str) -> None:
        jsii.set(self, "gatewayId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="routeTableId")
    def route_table_id(self) -> builtins.str:
        '''The ID of the route table.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-gatewayroutetableassociation.html#cfn-ec2-gatewayroutetableassociation-routetableid
        '''
        return typing.cast(builtins.str, jsii.get(self, "routeTableId"))

    @route_table_id.setter
    def route_table_id(self, value: builtins.str) -> None:
        jsii.set(self, "routeTableId", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnGatewayRouteTableAssociationProps",
    jsii_struct_bases=[],
    name_mapping={"gateway_id": "gatewayId", "route_table_id": "routeTableId"},
)
class CfnGatewayRouteTableAssociationProps:
    def __init__(
        self,
        *,
        gateway_id: builtins.str,
        route_table_id: builtins.str,
    ) -> None:
        '''Properties for defining a ``CfnGatewayRouteTableAssociation``.

        :param gateway_id: The ID of the gateway.
        :param route_table_id: The ID of the route table.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-gatewayroutetableassociation.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            cfn_gateway_route_table_association_props = ec2.CfnGatewayRouteTableAssociationProps(
                gateway_id="gatewayId",
                route_table_id="routeTableId"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "gateway_id": gateway_id,
            "route_table_id": route_table_id,
        }

    @builtins.property
    def gateway_id(self) -> builtins.str:
        '''The ID of the gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-gatewayroutetableassociation.html#cfn-ec2-gatewayroutetableassociation-gatewayid
        '''
        result = self._values.get("gateway_id")
        assert result is not None, "Required property 'gateway_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def route_table_id(self) -> builtins.str:
        '''The ID of the route table.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-gatewayroutetableassociation.html#cfn-ec2-gatewayroutetableassociation-routetableid
        '''
        result = self._values.get("route_table_id")
        assert result is not None, "Required property 'route_table_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnGatewayRouteTableAssociationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnHost(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnHost",
):
    '''A CloudFormation ``AWS::EC2::Host``.

    Allocates a fully dedicated physical server for launching EC2 instances. Because the host is fully dedicated for your use, it can help you address compliance requirements and reduce costs by allowing you to use your existing server-bound software licenses. For more information, see `Dedicated Hosts <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/dedicated-hosts-overview.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .

    :cloudformationResource: AWS::EC2::Host
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ec2 as ec2
        
        cfn_host = ec2.CfnHost(self, "MyCfnHost",
            availability_zone="availabilityZone",
            instance_type="instanceType",
        
            # the properties below are optional
            auto_placement="autoPlacement",
            host_recovery="hostRecovery"
        )
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        availability_zone: builtins.str,
        instance_type: builtins.str,
        auto_placement: typing.Optional[builtins.str] = None,
        host_recovery: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::Host``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param availability_zone: The Availability Zone in which to allocate the Dedicated Host.
        :param instance_type: Specifies the instance type to be supported by the Dedicated Hosts. If you specify an instance type, the Dedicated Hosts support instances of the specified instance type only.
        :param auto_placement: Indicates whether the host accepts any untargeted instance launches that match its instance type configuration, or if it only accepts Host tenancy instance launches that specify its unique host ID. For more information, see `Understanding auto-placement and affinity <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/how-dedicated-hosts-work.html#dedicated-hosts-understanding>`_ in the *Amazon EC2 User Guide* . Default: ``on``
        :param host_recovery: Indicates whether to enable or disable host recovery for the Dedicated Host. Host recovery is disabled by default. For more information, see `Host recovery <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/dedicated-hosts-recovery.html>`_ in the *Amazon EC2 User Guide* . Default: ``off``
        '''
        props = CfnHostProps(
            availability_zone=availability_zone,
            instance_type=instance_type,
            auto_placement=auto_placement,
            host_recovery=host_recovery,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="availabilityZone")
    def availability_zone(self) -> builtins.str:
        '''The Availability Zone in which to allocate the Dedicated Host.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html#cfn-ec2-host-availabilityzone
        '''
        return typing.cast(builtins.str, jsii.get(self, "availabilityZone"))

    @availability_zone.setter
    def availability_zone(self, value: builtins.str) -> None:
        jsii.set(self, "availabilityZone", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceType")
    def instance_type(self) -> builtins.str:
        '''Specifies the instance type to be supported by the Dedicated Hosts.

        If you specify an instance type, the Dedicated Hosts support instances of the specified instance type only.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html#cfn-ec2-host-instancetype
        '''
        return typing.cast(builtins.str, jsii.get(self, "instanceType"))

    @instance_type.setter
    def instance_type(self, value: builtins.str) -> None:
        jsii.set(self, "instanceType", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="autoPlacement")
    def auto_placement(self) -> typing.Optional[builtins.str]:
        '''Indicates whether the host accepts any untargeted instance launches that match its instance type configuration, or if it only accepts Host tenancy instance launches that specify its unique host ID.

        For more information, see `Understanding auto-placement and affinity <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/how-dedicated-hosts-work.html#dedicated-hosts-understanding>`_ in the *Amazon EC2 User Guide* .

        Default: ``on``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html#cfn-ec2-host-autoplacement
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "autoPlacement"))

    @auto_placement.setter
    def auto_placement(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "autoPlacement", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="hostRecovery")
    def host_recovery(self) -> typing.Optional[builtins.str]:
        '''Indicates whether to enable or disable host recovery for the Dedicated Host.

        Host recovery is disabled by default. For more information, see `Host recovery <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/dedicated-hosts-recovery.html>`_ in the *Amazon EC2 User Guide* .

        Default: ``off``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html#cfn-ec2-host-hostrecovery
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "hostRecovery"))

    @host_recovery.setter
    def host_recovery(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "hostRecovery", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnHostProps",
    jsii_struct_bases=[],
    name_mapping={
        "availability_zone": "availabilityZone",
        "instance_type": "instanceType",
        "auto_placement": "autoPlacement",
        "host_recovery": "hostRecovery",
    },
)
class CfnHostProps:
    def __init__(
        self,
        *,
        availability_zone: builtins.str,
        instance_type: builtins.str,
        auto_placement: typing.Optional[builtins.str] = None,
        host_recovery: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnHost``.

        :param availability_zone: The Availability Zone in which to allocate the Dedicated Host.
        :param instance_type: Specifies the instance type to be supported by the Dedicated Hosts. If you specify an instance type, the Dedicated Hosts support instances of the specified instance type only.
        :param auto_placement: Indicates whether the host accepts any untargeted instance launches that match its instance type configuration, or if it only accepts Host tenancy instance launches that specify its unique host ID. For more information, see `Understanding auto-placement and affinity <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/how-dedicated-hosts-work.html#dedicated-hosts-understanding>`_ in the *Amazon EC2 User Guide* . Default: ``on``
        :param host_recovery: Indicates whether to enable or disable host recovery for the Dedicated Host. Host recovery is disabled by default. For more information, see `Host recovery <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/dedicated-hosts-recovery.html>`_ in the *Amazon EC2 User Guide* . Default: ``off``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            cfn_host_props = ec2.CfnHostProps(
                availability_zone="availabilityZone",
                instance_type="instanceType",
            
                # the properties below are optional
                auto_placement="autoPlacement",
                host_recovery="hostRecovery"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "availability_zone": availability_zone,
            "instance_type": instance_type,
        }
        if auto_placement is not None:
            self._values["auto_placement"] = auto_placement
        if host_recovery is not None:
            self._values["host_recovery"] = host_recovery

    @builtins.property
    def availability_zone(self) -> builtins.str:
        '''The Availability Zone in which to allocate the Dedicated Host.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html#cfn-ec2-host-availabilityzone
        '''
        result = self._values.get("availability_zone")
        assert result is not None, "Required property 'availability_zone' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def instance_type(self) -> builtins.str:
        '''Specifies the instance type to be supported by the Dedicated Hosts.

        If you specify an instance type, the Dedicated Hosts support instances of the specified instance type only.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html#cfn-ec2-host-instancetype
        '''
        result = self._values.get("instance_type")
        assert result is not None, "Required property 'instance_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def auto_placement(self) -> typing.Optional[builtins.str]:
        '''Indicates whether the host accepts any untargeted instance launches that match its instance type configuration, or if it only accepts Host tenancy instance launches that specify its unique host ID.

        For more information, see `Understanding auto-placement and affinity <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/how-dedicated-hosts-work.html#dedicated-hosts-understanding>`_ in the *Amazon EC2 User Guide* .

        Default: ``on``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html#cfn-ec2-host-autoplacement
        '''
        result = self._values.get("auto_placement")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def host_recovery(self) -> typing.Optional[builtins.str]:
        '''Indicates whether to enable or disable host recovery for the Dedicated Host.

        Host recovery is disabled by default. For more information, see `Host recovery <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/dedicated-hosts-recovery.html>`_ in the *Amazon EC2 User Guide* .

        Default: ``off``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html#cfn-ec2-host-hostrecovery
        '''
        result = self._values.get("host_recovery")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnHostProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnIPAM(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnIPAM",
):
    '''A CloudFormation ``AWS::EC2::IPAM``.

    IPAM is a VPC feature that you can use to automate your IP address management workflows including assigning, tracking, troubleshooting, and auditing IP addresses across AWS Regions and accounts throughout your AWS Organization. For more information, see `What is IPAM? <https://docs.aws.amazon.com//vpc/latest/ipam/what-is-it-ipam.html>`_ in the *Amazon VPC IPAM User Guide* .

    :cloudformationResource: AWS::EC2::IPAM
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipam.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ec2 as ec2
        
        cfn_iPAM = ec2.CfnIPAM(self, "MyCfnIPAM",
            description="description",
            operating_regions=[ec2.CfnIPAM.IpamOperatingRegionProperty(
                region_name="regionName"
            )],
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        description: typing.Optional[builtins.str] = None,
        operating_regions: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnIPAM.IpamOperatingRegionProperty"]]]] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::IPAM``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param description: The description for the IPAM.
        :param operating_regions: The operating Regions for an IPAM. Operating Regions are AWS Regions where the IPAM is allowed to manage IP address CIDRs. IPAM only discovers and monitors resources in the AWS Regions you select as operating Regions. For more information about operating Regions, see `Create an IPAM <https://docs.aws.amazon.com//vpc/latest/ipam/create-ipam.html>`_ in the *Amazon VPC IPAM User Guide* .
        :param tags: The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the tag value as the filter value. For example, to find all resources that have a tag with the key ``Owner`` and the value ``TeamA`` , specify ``tag:Owner`` for the filter name and ``TeamA`` for the filter value.
        '''
        props = CfnIPAMProps(
            description=description, operating_regions=operating_regions, tags=tags
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The ARN of the IPAM.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrIpamId")
    def attr_ipam_id(self) -> builtins.str:
        '''The ID of the IPAM.

        :cloudformationAttribute: IpamId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrIpamId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrPrivateDefaultScopeId")
    def attr_private_default_scope_id(self) -> builtins.str:
        '''The ID of the IPAM's default private scope.

        :cloudformationAttribute: PrivateDefaultScopeId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrPrivateDefaultScopeId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrPublicDefaultScopeId")
    def attr_public_default_scope_id(self) -> builtins.str:
        '''The ID of the IPAM's default public scope.

        :cloudformationAttribute: PublicDefaultScopeId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrPublicDefaultScopeId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrScopeCount")
    def attr_scope_count(self) -> jsii.Number:
        '''The number of scopes in the IPAM.

        The scope quota is 5.

        :cloudformationAttribute: ScopeCount
        '''
        return typing.cast(jsii.Number, jsii.get(self, "attrScopeCount"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''The key/value combination of a tag assigned to the resource.

        Use the tag key in the filter name and the tag value as the filter value. For example, to find all resources that have a tag with the key ``Owner`` and the value ``TeamA`` , specify ``tag:Owner`` for the filter name and ``TeamA`` for the filter value.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipam.html#cfn-ec2-ipam-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''The description for the IPAM.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipam.html#cfn-ec2-ipam-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "description", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="operatingRegions")
    def operating_regions(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnIPAM.IpamOperatingRegionProperty"]]]]:
        '''The operating Regions for an IPAM.

        Operating Regions are AWS Regions where the IPAM is allowed to manage IP address CIDRs. IPAM only discovers and monitors resources in the AWS Regions you select as operating Regions.

        For more information about operating Regions, see `Create an IPAM <https://docs.aws.amazon.com//vpc/latest/ipam/create-ipam.html>`_ in the *Amazon VPC IPAM User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipam.html#cfn-ec2-ipam-operatingregions
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnIPAM.IpamOperatingRegionProperty"]]]], jsii.get(self, "operatingRegions"))

    @operating_regions.setter
    def operating_regions(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnIPAM.IpamOperatingRegionProperty"]]]],
    ) -> None:
        jsii.set(self, "operatingRegions", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnIPAM.IpamOperatingRegionProperty",
        jsii_struct_bases=[],
        name_mapping={"region_name": "regionName"},
    )
    class IpamOperatingRegionProperty:
        def __init__(self, *, region_name: builtins.str) -> None:
            '''The operating Regions for an IPAM.

            Operating Regions are AWS Regions where the IPAM is allowed to manage IP address CIDRs. IPAM only discovers and monitors resources in the AWS Regions you select as operating Regions.

            For more information about operating Regions, see `Create an IPAM <https://docs.aws.amazon.com//vpc/latest/ipam/create-ipam.html>`_ in the *Amazon VPC IPAM User Guide* .

            :param region_name: The name of the operating Region.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ipam-ipamoperatingregion.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                ipam_operating_region_property = ec2.CfnIPAM.IpamOperatingRegionProperty(
                    region_name="regionName"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "region_name": region_name,
            }

        @builtins.property
        def region_name(self) -> builtins.str:
            '''The name of the operating Region.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ipam-ipamoperatingregion.html#cfn-ec2-ipam-ipamoperatingregion-regionname
            '''
            result = self._values.get("region_name")
            assert result is not None, "Required property 'region_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "IpamOperatingRegionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnIPAMAllocation(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnIPAMAllocation",
):
    '''A CloudFormation ``AWS::EC2::IPAMAllocation``.

    In IPAM, an allocation is a CIDR assignment from an IPAM pool to another resource or IPAM pool.

    :cloudformationResource: AWS::EC2::IPAMAllocation
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamallocation.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ec2 as ec2
        
        cfn_iPAMAllocation = ec2.CfnIPAMAllocation(self, "MyCfnIPAMAllocation",
            ipam_pool_id="ipamPoolId",
        
            # the properties below are optional
            cidr="cidr",
            description="description",
            netmask_length=123
        )
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        ipam_pool_id: builtins.str,
        cidr: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        netmask_length: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::IPAMAllocation``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param ipam_pool_id: The ID of the IPAM pool from which you would like to allocate a CIDR.
        :param cidr: The CIDR you would like to allocate from the IPAM pool. Note the following:. - If there is no DefaultNetmaskLength allocation rule set on the pool, you must specify either the NetmaskLength or the CIDR. - If the DefaultNetmaskLength allocation rule is set on the pool, you can specify either the NetmaskLength or the CIDR and the DefaultNetmaskLength allocation rule will be ignored. Possible values: Any available IPv4 or IPv6 CIDR.
        :param description: A description for the allocation.
        :param netmask_length: The netmask length of the CIDR you would like to allocate from the IPAM pool. Note the following:. - If there is no DefaultNetmaskLength allocation rule set on the pool, you must specify either the NetmaskLength or the CIDR. - If the DefaultNetmaskLength allocation rule is set on the pool, you can specify either the NetmaskLength or the CIDR and the DefaultNetmaskLength allocation rule will be ignored. Possible netmask lengths for IPv4 addresses are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.
        '''
        props = CfnIPAMAllocationProps(
            ipam_pool_id=ipam_pool_id,
            cidr=cidr,
            description=description,
            netmask_length=netmask_length,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrIpamPoolAllocationId")
    def attr_ipam_pool_allocation_id(self) -> builtins.str:
        '''The ID of an allocation.

        :cloudformationAttribute: IpamPoolAllocationId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrIpamPoolAllocationId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ipamPoolId")
    def ipam_pool_id(self) -> builtins.str:
        '''The ID of the IPAM pool from which you would like to allocate a CIDR.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamallocation.html#cfn-ec2-ipamallocation-ipampoolid
        '''
        return typing.cast(builtins.str, jsii.get(self, "ipamPoolId"))

    @ipam_pool_id.setter
    def ipam_pool_id(self, value: builtins.str) -> None:
        jsii.set(self, "ipamPoolId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cidr")
    def cidr(self) -> typing.Optional[builtins.str]:
        '''The CIDR you would like to allocate from the IPAM pool. Note the following:.

        - If there is no DefaultNetmaskLength allocation rule set on the pool, you must specify either the NetmaskLength or the CIDR.
        - If the DefaultNetmaskLength allocation rule is set on the pool, you can specify either the NetmaskLength or the CIDR and the DefaultNetmaskLength allocation rule will be ignored.

        Possible values: Any available IPv4 or IPv6 CIDR.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamallocation.html#cfn-ec2-ipamallocation-cidr
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "cidr"))

    @cidr.setter
    def cidr(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "cidr", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A description for the allocation.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamallocation.html#cfn-ec2-ipamallocation-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "description", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="netmaskLength")
    def netmask_length(self) -> typing.Optional[jsii.Number]:
        '''The netmask length of the CIDR you would like to allocate from the IPAM pool. Note the following:.

        - If there is no DefaultNetmaskLength allocation rule set on the pool, you must specify either the NetmaskLength or the CIDR.
        - If the DefaultNetmaskLength allocation rule is set on the pool, you can specify either the NetmaskLength or the CIDR and the DefaultNetmaskLength allocation rule will be ignored.

        Possible netmask lengths for IPv4 addresses are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamallocation.html#cfn-ec2-ipamallocation-netmasklength
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "netmaskLength"))

    @netmask_length.setter
    def netmask_length(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "netmaskLength", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnIPAMAllocationProps",
    jsii_struct_bases=[],
    name_mapping={
        "ipam_pool_id": "ipamPoolId",
        "cidr": "cidr",
        "description": "description",
        "netmask_length": "netmaskLength",
    },
)
class CfnIPAMAllocationProps:
    def __init__(
        self,
        *,
        ipam_pool_id: builtins.str,
        cidr: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        netmask_length: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties for defining a ``CfnIPAMAllocation``.

        :param ipam_pool_id: The ID of the IPAM pool from which you would like to allocate a CIDR.
        :param cidr: The CIDR you would like to allocate from the IPAM pool. Note the following:. - If there is no DefaultNetmaskLength allocation rule set on the pool, you must specify either the NetmaskLength or the CIDR. - If the DefaultNetmaskLength allocation rule is set on the pool, you can specify either the NetmaskLength or the CIDR and the DefaultNetmaskLength allocation rule will be ignored. Possible values: Any available IPv4 or IPv6 CIDR.
        :param description: A description for the allocation.
        :param netmask_length: The netmask length of the CIDR you would like to allocate from the IPAM pool. Note the following:. - If there is no DefaultNetmaskLength allocation rule set on the pool, you must specify either the NetmaskLength or the CIDR. - If the DefaultNetmaskLength allocation rule is set on the pool, you can specify either the NetmaskLength or the CIDR and the DefaultNetmaskLength allocation rule will be ignored. Possible netmask lengths for IPv4 addresses are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamallocation.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            cfn_iPAMAllocation_props = ec2.CfnIPAMAllocationProps(
                ipam_pool_id="ipamPoolId",
            
                # the properties below are optional
                cidr="cidr",
                description="description",
                netmask_length=123
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "ipam_pool_id": ipam_pool_id,
        }
        if cidr is not None:
            self._values["cidr"] = cidr
        if description is not None:
            self._values["description"] = description
        if netmask_length is not None:
            self._values["netmask_length"] = netmask_length

    @builtins.property
    def ipam_pool_id(self) -> builtins.str:
        '''The ID of the IPAM pool from which you would like to allocate a CIDR.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamallocation.html#cfn-ec2-ipamallocation-ipampoolid
        '''
        result = self._values.get("ipam_pool_id")
        assert result is not None, "Required property 'ipam_pool_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def cidr(self) -> typing.Optional[builtins.str]:
        '''The CIDR you would like to allocate from the IPAM pool. Note the following:.

        - If there is no DefaultNetmaskLength allocation rule set on the pool, you must specify either the NetmaskLength or the CIDR.
        - If the DefaultNetmaskLength allocation rule is set on the pool, you can specify either the NetmaskLength or the CIDR and the DefaultNetmaskLength allocation rule will be ignored.

        Possible values: Any available IPv4 or IPv6 CIDR.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamallocation.html#cfn-ec2-ipamallocation-cidr
        '''
        result = self._values.get("cidr")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description for the allocation.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamallocation.html#cfn-ec2-ipamallocation-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def netmask_length(self) -> typing.Optional[jsii.Number]:
        '''The netmask length of the CIDR you would like to allocate from the IPAM pool. Note the following:.

        - If there is no DefaultNetmaskLength allocation rule set on the pool, you must specify either the NetmaskLength or the CIDR.
        - If the DefaultNetmaskLength allocation rule is set on the pool, you can specify either the NetmaskLength or the CIDR and the DefaultNetmaskLength allocation rule will be ignored.

        Possible netmask lengths for IPv4 addresses are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamallocation.html#cfn-ec2-ipamallocation-netmasklength
        '''
        result = self._values.get("netmask_length")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnIPAMAllocationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnIPAMPool(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnIPAMPool",
):
    '''A CloudFormation ``AWS::EC2::IPAMPool``.

    In IPAM, a pool is a collection of contiguous IP addresses CIDRs. Pools enable you to organize your IP addresses according to your routing and security needs. For example, if you have separate routing and security needs for development and production applications, you can create a pool for each.

    :cloudformationResource: AWS::EC2::IPAMPool
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ec2 as ec2
        
        cfn_iPAMPool = ec2.CfnIPAMPool(self, "MyCfnIPAMPool",
            address_family="addressFamily",
            ipam_scope_id="ipamScopeId",
        
            # the properties below are optional
            allocation_default_netmask_length=123,
            allocation_max_netmask_length=123,
            allocation_min_netmask_length=123,
            allocation_resource_tags=[CfnTag(
                key="key",
                value="value"
            )],
            auto_import=False,
            description="description",
            locale="locale",
            provisioned_cidrs=[ec2.CfnIPAMPool.ProvisionedCidrProperty(
                cidr="cidr"
            )],
            publicly_advertisable=False,
            source_ipam_pool_id="sourceIpamPoolId",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        address_family: builtins.str,
        ipam_scope_id: builtins.str,
        allocation_default_netmask_length: typing.Optional[jsii.Number] = None,
        allocation_max_netmask_length: typing.Optional[jsii.Number] = None,
        allocation_min_netmask_length: typing.Optional[jsii.Number] = None,
        allocation_resource_tags: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, aws_cdk.core.CfnTag]]]] = None,
        auto_import: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        description: typing.Optional[builtins.str] = None,
        locale: typing.Optional[builtins.str] = None,
        provisioned_cidrs: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnIPAMPool.ProvisionedCidrProperty"]]]] = None,
        publicly_advertisable: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        source_ipam_pool_id: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::IPAMPool``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param address_family: The address family of the pool.
        :param ipam_scope_id: The ID of the scope in which you would like to create the IPAM pool.
        :param allocation_default_netmask_length: The default netmask length for allocations added to this pool. If, for example, the CIDR assigned to this pool is 10.0.0.0/8 and you enter 16 here, new allocations will default to 10.0.0.0/16.
        :param allocation_max_netmask_length: The maximum netmask length possible for CIDR allocations in this IPAM pool to be compliant. The maximum netmask length must be greater than the minimum netmask length. Possible netmask lengths for IPv4 addresses are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.
        :param allocation_min_netmask_length: The minimum netmask length required for CIDR allocations in this IPAM pool to be compliant. The minimum netmask length must be less than the maximum netmask length. Possible netmask lengths for IPv4 addresses are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.
        :param allocation_resource_tags: Tags that are required for resources that use CIDRs from this IPAM pool. Resources that do not have these tags will not be allowed to allocate space from the pool. If the resources have their tags changed after they have allocated space or if the allocation tagging requirements are changed on the pool, the resource may be marked as noncompliant.
        :param auto_import: If selected, IPAM will continuously look for resources within the CIDR range of this pool and automatically import them as allocations into your IPAM. The CIDRs that will be allocated for these resources must not already be allocated to other resources in order for the import to succeed. IPAM will import a CIDR regardless of its compliance with the pool's allocation rules, so a resource might be imported and subsequently marked as noncompliant. If IPAM discovers multiple CIDRs that overlap, IPAM will import the largest CIDR only. If IPAM discovers multiple CIDRs with matching CIDRs, IPAM will randomly import one of them only. A locale must be set on the pool for this feature to work.
        :param description: The description of the IPAM pool.
        :param locale: The locale of the IPAM pool. In IPAM, the locale is the AWS Region where you want to make an IPAM pool available for allocations. Only resources in the same Region as the locale of the pool can get IP address allocations from the pool. You can only allocate a CIDR for a VPC, for example, from an IPAM pool that shares a locale with the VPC’s Region. Note that once you choose a Locale for a pool, you cannot modify it. If you choose an AWS Region for locale that has not been configured as an operating Region for the IPAM, you'll get an error.
        :param provisioned_cidrs: Information about the CIDRs provisioned to an IPAM pool.
        :param publicly_advertisable: Determines if a pool is publicly advertisable. This option is not available for pools with AddressFamily set to ``ipv4`` .
        :param source_ipam_pool_id: The ID of the source IPAM pool. You can use this option to create an IPAM pool within an existing source pool.
        :param tags: The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the tag value as the filter value. For example, to find all resources that have a tag with the key ``Owner`` and the value ``TeamA`` , specify ``tag:Owner`` for the filter name and ``TeamA`` for the filter value.
        '''
        props = CfnIPAMPoolProps(
            address_family=address_family,
            ipam_scope_id=ipam_scope_id,
            allocation_default_netmask_length=allocation_default_netmask_length,
            allocation_max_netmask_length=allocation_max_netmask_length,
            allocation_min_netmask_length=allocation_min_netmask_length,
            allocation_resource_tags=allocation_resource_tags,
            auto_import=auto_import,
            description=description,
            locale=locale,
            provisioned_cidrs=provisioned_cidrs,
            publicly_advertisable=publicly_advertisable,
            source_ipam_pool_id=source_ipam_pool_id,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The ARN of the IPAM pool.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrIpamArn")
    def attr_ipam_arn(self) -> builtins.str:
        '''The ARN of the IPAM.

        :cloudformationAttribute: IpamArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrIpamArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrIpamPoolId")
    def attr_ipam_pool_id(self) -> builtins.str:
        '''The ID of the IPAM pool.

        :cloudformationAttribute: IpamPoolId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrIpamPoolId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrIpamScopeArn")
    def attr_ipam_scope_arn(self) -> builtins.str:
        '''The ARN of the scope of the IPAM pool.

        :cloudformationAttribute: IpamScopeArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrIpamScopeArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrIpamScopeType")
    def attr_ipam_scope_type(self) -> builtins.str:
        '''The scope of the IPAM.

        :cloudformationAttribute: IpamScopeType
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrIpamScopeType"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrPoolDepth")
    def attr_pool_depth(self) -> jsii.Number:
        '''The depth of pools in your IPAM pool.

        The pool depth quota is 10.

        :cloudformationAttribute: PoolDepth
        '''
        return typing.cast(jsii.Number, jsii.get(self, "attrPoolDepth"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrState")
    def attr_state(self) -> builtins.str:
        '''The state of the IPAM pool.

        :cloudformationAttribute: State
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrState"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrStateMessage")
    def attr_state_message(self) -> builtins.str:
        '''A message related to the failed creation of an IPAM pool.

        :cloudformationAttribute: StateMessage
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrStateMessage"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''The key/value combination of a tag assigned to the resource.

        Use the tag key in the filter name and the tag value as the filter value. For example, to find all resources that have a tag with the key ``Owner`` and the value ``TeamA`` , specify ``tag:Owner`` for the filter name and ``TeamA`` for the filter value.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="addressFamily")
    def address_family(self) -> builtins.str:
        '''The address family of the pool.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-addressfamily
        '''
        return typing.cast(builtins.str, jsii.get(self, "addressFamily"))

    @address_family.setter
    def address_family(self, value: builtins.str) -> None:
        jsii.set(self, "addressFamily", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ipamScopeId")
    def ipam_scope_id(self) -> builtins.str:
        '''The ID of the scope in which you would like to create the IPAM pool.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-ipamscopeid
        '''
        return typing.cast(builtins.str, jsii.get(self, "ipamScopeId"))

    @ipam_scope_id.setter
    def ipam_scope_id(self, value: builtins.str) -> None:
        jsii.set(self, "ipamScopeId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="allocationDefaultNetmaskLength")
    def allocation_default_netmask_length(self) -> typing.Optional[jsii.Number]:
        '''The default netmask length for allocations added to this pool.

        If, for example, the CIDR assigned to this pool is 10.0.0.0/8 and you enter 16 here, new allocations will default to 10.0.0.0/16.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-allocationdefaultnetmasklength
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "allocationDefaultNetmaskLength"))

    @allocation_default_netmask_length.setter
    def allocation_default_netmask_length(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        jsii.set(self, "allocationDefaultNetmaskLength", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="allocationMaxNetmaskLength")
    def allocation_max_netmask_length(self) -> typing.Optional[jsii.Number]:
        '''The maximum netmask length possible for CIDR allocations in this IPAM pool to be compliant.

        The maximum netmask length must be greater than the minimum netmask length. Possible netmask lengths for IPv4 addresses are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-allocationmaxnetmasklength
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "allocationMaxNetmaskLength"))

    @allocation_max_netmask_length.setter
    def allocation_max_netmask_length(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        jsii.set(self, "allocationMaxNetmaskLength", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="allocationMinNetmaskLength")
    def allocation_min_netmask_length(self) -> typing.Optional[jsii.Number]:
        '''The minimum netmask length required for CIDR allocations in this IPAM pool to be compliant.

        The minimum netmask length must be less than the maximum netmask length. Possible netmask lengths for IPv4 addresses are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-allocationminnetmasklength
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "allocationMinNetmaskLength"))

    @allocation_min_netmask_length.setter
    def allocation_min_netmask_length(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        jsii.set(self, "allocationMinNetmaskLength", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="allocationResourceTags")
    def allocation_resource_tags(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, aws_cdk.core.CfnTag]]]]:
        '''Tags that are required for resources that use CIDRs from this IPAM pool.

        Resources that do not have these tags will not be allowed to allocate space from the pool. If the resources have their tags changed after they have allocated space or if the allocation tagging requirements are changed on the pool, the resource may be marked as noncompliant.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-allocationresourcetags
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, aws_cdk.core.CfnTag]]]], jsii.get(self, "allocationResourceTags"))

    @allocation_resource_tags.setter
    def allocation_resource_tags(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, aws_cdk.core.CfnTag]]]],
    ) -> None:
        jsii.set(self, "allocationResourceTags", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="autoImport")
    def auto_import(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''If selected, IPAM will continuously look for resources within the CIDR range of this pool and automatically import them as allocations into your IPAM.

        The CIDRs that will be allocated for these resources must not already be allocated to other resources in order for the import to succeed. IPAM will import a CIDR regardless of its compliance with the pool's allocation rules, so a resource might be imported and subsequently marked as noncompliant. If IPAM discovers multiple CIDRs that overlap, IPAM will import the largest CIDR only. If IPAM discovers multiple CIDRs with matching CIDRs, IPAM will randomly import one of them only.

        A locale must be set on the pool for this feature to work.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-autoimport
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "autoImport"))

    @auto_import.setter
    def auto_import(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "autoImport", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the IPAM pool.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "description", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="locale")
    def locale(self) -> typing.Optional[builtins.str]:
        '''The locale of the IPAM pool.

        In IPAM, the locale is the AWS Region where you want to make an IPAM pool available for allocations. Only resources in the same Region as the locale of the pool can get IP address allocations from the pool. You can only allocate a CIDR for a VPC, for example, from an IPAM pool that shares a locale with the VPC’s Region. Note that once you choose a Locale for a pool, you cannot modify it. If you choose an AWS Region for locale that has not been configured as an operating Region for the IPAM, you'll get an error.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-locale
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "locale"))

    @locale.setter
    def locale(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "locale", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="provisionedCidrs")
    def provisioned_cidrs(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnIPAMPool.ProvisionedCidrProperty"]]]]:
        '''Information about the CIDRs provisioned to an IPAM pool.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-provisionedcidrs
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnIPAMPool.ProvisionedCidrProperty"]]]], jsii.get(self, "provisionedCidrs"))

    @provisioned_cidrs.setter
    def provisioned_cidrs(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnIPAMPool.ProvisionedCidrProperty"]]]],
    ) -> None:
        jsii.set(self, "provisionedCidrs", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="publiclyAdvertisable")
    def publicly_advertisable(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''Determines if a pool is publicly advertisable.

        This option is not available for pools with AddressFamily set to ``ipv4`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-publiclyadvertisable
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "publiclyAdvertisable"))

    @publicly_advertisable.setter
    def publicly_advertisable(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "publiclyAdvertisable", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sourceIpamPoolId")
    def source_ipam_pool_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the source IPAM pool.

        You can use this option to create an IPAM pool within an existing source pool.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-sourceipampoolid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceIpamPoolId"))

    @source_ipam_pool_id.setter
    def source_ipam_pool_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "sourceIpamPoolId", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnIPAMPool.ProvisionedCidrProperty",
        jsii_struct_bases=[],
        name_mapping={"cidr": "cidr"},
    )
    class ProvisionedCidrProperty:
        def __init__(self, *, cidr: builtins.str) -> None:
            '''The CIDR provisioned to the IPAM pool.

            A CIDR is a representation of an IP address and its associated network mask (or netmask) and refers to a range of IP addresses. An IPv4 CIDR example is ``10.24.34.0/23`` . An IPv6 CIDR example is ``2001:DB8::/32`` .

            :param cidr: The CIDR provisioned to the IPAM pool. A CIDR is a representation of an IP address and its associated network mask (or netmask) and refers to a range of IP addresses. An IPv4 CIDR example is ``10.24.34.0/23`` . An IPv6 CIDR example is ``2001:DB8::/32`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ipampool-provisionedcidr.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                provisioned_cidr_property = ec2.CfnIPAMPool.ProvisionedCidrProperty(
                    cidr="cidr"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "cidr": cidr,
            }

        @builtins.property
        def cidr(self) -> builtins.str:
            '''The CIDR provisioned to the IPAM pool.

            A CIDR is a representation of an IP address and its associated network mask (or netmask) and refers to a range of IP addresses. An IPv4 CIDR example is ``10.24.34.0/23`` . An IPv6 CIDR example is ``2001:DB8::/32`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ipampool-provisionedcidr.html#cfn-ec2-ipampool-provisionedcidr-cidr
            '''
            result = self._values.get("cidr")
            assert result is not None, "Required property 'cidr' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ProvisionedCidrProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnIPAMPoolProps",
    jsii_struct_bases=[],
    name_mapping={
        "address_family": "addressFamily",
        "ipam_scope_id": "ipamScopeId",
        "allocation_default_netmask_length": "allocationDefaultNetmaskLength",
        "allocation_max_netmask_length": "allocationMaxNetmaskLength",
        "allocation_min_netmask_length": "allocationMinNetmaskLength",
        "allocation_resource_tags": "allocationResourceTags",
        "auto_import": "autoImport",
        "description": "description",
        "locale": "locale",
        "provisioned_cidrs": "provisionedCidrs",
        "publicly_advertisable": "publiclyAdvertisable",
        "source_ipam_pool_id": "sourceIpamPoolId",
        "tags": "tags",
    },
)
class CfnIPAMPoolProps:
    def __init__(
        self,
        *,
        address_family: builtins.str,
        ipam_scope_id: builtins.str,
        allocation_default_netmask_length: typing.Optional[jsii.Number] = None,
        allocation_max_netmask_length: typing.Optional[jsii.Number] = None,
        allocation_min_netmask_length: typing.Optional[jsii.Number] = None,
        allocation_resource_tags: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, aws_cdk.core.CfnTag]]]] = None,
        auto_import: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        description: typing.Optional[builtins.str] = None,
        locale: typing.Optional[builtins.str] = None,
        provisioned_cidrs: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnIPAMPool.ProvisionedCidrProperty]]]] = None,
        publicly_advertisable: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        source_ipam_pool_id: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Properties for defining a ``CfnIPAMPool``.

        :param address_family: The address family of the pool.
        :param ipam_scope_id: The ID of the scope in which you would like to create the IPAM pool.
        :param allocation_default_netmask_length: The default netmask length for allocations added to this pool. If, for example, the CIDR assigned to this pool is 10.0.0.0/8 and you enter 16 here, new allocations will default to 10.0.0.0/16.
        :param allocation_max_netmask_length: The maximum netmask length possible for CIDR allocations in this IPAM pool to be compliant. The maximum netmask length must be greater than the minimum netmask length. Possible netmask lengths for IPv4 addresses are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.
        :param allocation_min_netmask_length: The minimum netmask length required for CIDR allocations in this IPAM pool to be compliant. The minimum netmask length must be less than the maximum netmask length. Possible netmask lengths for IPv4 addresses are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.
        :param allocation_resource_tags: Tags that are required for resources that use CIDRs from this IPAM pool. Resources that do not have these tags will not be allowed to allocate space from the pool. If the resources have their tags changed after they have allocated space or if the allocation tagging requirements are changed on the pool, the resource may be marked as noncompliant.
        :param auto_import: If selected, IPAM will continuously look for resources within the CIDR range of this pool and automatically import them as allocations into your IPAM. The CIDRs that will be allocated for these resources must not already be allocated to other resources in order for the import to succeed. IPAM will import a CIDR regardless of its compliance with the pool's allocation rules, so a resource might be imported and subsequently marked as noncompliant. If IPAM discovers multiple CIDRs that overlap, IPAM will import the largest CIDR only. If IPAM discovers multiple CIDRs with matching CIDRs, IPAM will randomly import one of them only. A locale must be set on the pool for this feature to work.
        :param description: The description of the IPAM pool.
        :param locale: The locale of the IPAM pool. In IPAM, the locale is the AWS Region where you want to make an IPAM pool available for allocations. Only resources in the same Region as the locale of the pool can get IP address allocations from the pool. You can only allocate a CIDR for a VPC, for example, from an IPAM pool that shares a locale with the VPC’s Region. Note that once you choose a Locale for a pool, you cannot modify it. If you choose an AWS Region for locale that has not been configured as an operating Region for the IPAM, you'll get an error.
        :param provisioned_cidrs: Information about the CIDRs provisioned to an IPAM pool.
        :param publicly_advertisable: Determines if a pool is publicly advertisable. This option is not available for pools with AddressFamily set to ``ipv4`` .
        :param source_ipam_pool_id: The ID of the source IPAM pool. You can use this option to create an IPAM pool within an existing source pool.
        :param tags: The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the tag value as the filter value. For example, to find all resources that have a tag with the key ``Owner`` and the value ``TeamA`` , specify ``tag:Owner`` for the filter name and ``TeamA`` for the filter value.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            cfn_iPAMPool_props = ec2.CfnIPAMPoolProps(
                address_family="addressFamily",
                ipam_scope_id="ipamScopeId",
            
                # the properties below are optional
                allocation_default_netmask_length=123,
                allocation_max_netmask_length=123,
                allocation_min_netmask_length=123,
                allocation_resource_tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                auto_import=False,
                description="description",
                locale="locale",
                provisioned_cidrs=[ec2.CfnIPAMPool.ProvisionedCidrProperty(
                    cidr="cidr"
                )],
                publicly_advertisable=False,
                source_ipam_pool_id="sourceIpamPoolId",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "address_family": address_family,
            "ipam_scope_id": ipam_scope_id,
        }
        if allocation_default_netmask_length is not None:
            self._values["allocation_default_netmask_length"] = allocation_default_netmask_length
        if allocation_max_netmask_length is not None:
            self._values["allocation_max_netmask_length"] = allocation_max_netmask_length
        if allocation_min_netmask_length is not None:
            self._values["allocation_min_netmask_length"] = allocation_min_netmask_length
        if allocation_resource_tags is not None:
            self._values["allocation_resource_tags"] = allocation_resource_tags
        if auto_import is not None:
            self._values["auto_import"] = auto_import
        if description is not None:
            self._values["description"] = description
        if locale is not None:
            self._values["locale"] = locale
        if provisioned_cidrs is not None:
            self._values["provisioned_cidrs"] = provisioned_cidrs
        if publicly_advertisable is not None:
            self._values["publicly_advertisable"] = publicly_advertisable
        if source_ipam_pool_id is not None:
            self._values["source_ipam_pool_id"] = source_ipam_pool_id
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def address_family(self) -> builtins.str:
        '''The address family of the pool.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-addressfamily
        '''
        result = self._values.get("address_family")
        assert result is not None, "Required property 'address_family' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def ipam_scope_id(self) -> builtins.str:
        '''The ID of the scope in which you would like to create the IPAM pool.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-ipamscopeid
        '''
        result = self._values.get("ipam_scope_id")
        assert result is not None, "Required property 'ipam_scope_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def allocation_default_netmask_length(self) -> typing.Optional[jsii.Number]:
        '''The default netmask length for allocations added to this pool.

        If, for example, the CIDR assigned to this pool is 10.0.0.0/8 and you enter 16 here, new allocations will default to 10.0.0.0/16.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-allocationdefaultnetmasklength
        '''
        result = self._values.get("allocation_default_netmask_length")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def allocation_max_netmask_length(self) -> typing.Optional[jsii.Number]:
        '''The maximum netmask length possible for CIDR allocations in this IPAM pool to be compliant.

        The maximum netmask length must be greater than the minimum netmask length. Possible netmask lengths for IPv4 addresses are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-allocationmaxnetmasklength
        '''
        result = self._values.get("allocation_max_netmask_length")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def allocation_min_netmask_length(self) -> typing.Optional[jsii.Number]:
        '''The minimum netmask length required for CIDR allocations in this IPAM pool to be compliant.

        The minimum netmask length must be less than the maximum netmask length. Possible netmask lengths for IPv4 addresses are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-allocationminnetmasklength
        '''
        result = self._values.get("allocation_min_netmask_length")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def allocation_resource_tags(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, aws_cdk.core.CfnTag]]]]:
        '''Tags that are required for resources that use CIDRs from this IPAM pool.

        Resources that do not have these tags will not be allowed to allocate space from the pool. If the resources have their tags changed after they have allocated space or if the allocation tagging requirements are changed on the pool, the resource may be marked as noncompliant.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-allocationresourcetags
        '''
        result = self._values.get("allocation_resource_tags")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, aws_cdk.core.CfnTag]]]], result)

    @builtins.property
    def auto_import(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''If selected, IPAM will continuously look for resources within the CIDR range of this pool and automatically import them as allocations into your IPAM.

        The CIDRs that will be allocated for these resources must not already be allocated to other resources in order for the import to succeed. IPAM will import a CIDR regardless of its compliance with the pool's allocation rules, so a resource might be imported and subsequently marked as noncompliant. If IPAM discovers multiple CIDRs that overlap, IPAM will import the largest CIDR only. If IPAM discovers multiple CIDRs with matching CIDRs, IPAM will randomly import one of them only.

        A locale must be set on the pool for this feature to work.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-autoimport
        '''
        result = self._values.get("auto_import")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the IPAM pool.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def locale(self) -> typing.Optional[builtins.str]:
        '''The locale of the IPAM pool.

        In IPAM, the locale is the AWS Region where you want to make an IPAM pool available for allocations. Only resources in the same Region as the locale of the pool can get IP address allocations from the pool. You can only allocate a CIDR for a VPC, for example, from an IPAM pool that shares a locale with the VPC’s Region. Note that once you choose a Locale for a pool, you cannot modify it. If you choose an AWS Region for locale that has not been configured as an operating Region for the IPAM, you'll get an error.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-locale
        '''
        result = self._values.get("locale")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def provisioned_cidrs(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnIPAMPool.ProvisionedCidrProperty]]]]:
        '''Information about the CIDRs provisioned to an IPAM pool.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-provisionedcidrs
        '''
        result = self._values.get("provisioned_cidrs")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnIPAMPool.ProvisionedCidrProperty]]]], result)

    @builtins.property
    def publicly_advertisable(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''Determines if a pool is publicly advertisable.

        This option is not available for pools with AddressFamily set to ``ipv4`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-publiclyadvertisable
        '''
        result = self._values.get("publicly_advertisable")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def source_ipam_pool_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the source IPAM pool.

        You can use this option to create an IPAM pool within an existing source pool.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-sourceipampoolid
        '''
        result = self._values.get("source_ipam_pool_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''The key/value combination of a tag assigned to the resource.

        Use the tag key in the filter name and the tag value as the filter value. For example, to find all resources that have a tag with the key ``Owner`` and the value ``TeamA`` , specify ``tag:Owner`` for the filter name and ``TeamA`` for the filter value.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html#cfn-ec2-ipampool-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnIPAMPoolProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnIPAMProps",
    jsii_struct_bases=[],
    name_mapping={
        "description": "description",
        "operating_regions": "operatingRegions",
        "tags": "tags",
    },
)
class CfnIPAMProps:
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        operating_regions: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnIPAM.IpamOperatingRegionProperty]]]] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Properties for defining a ``CfnIPAM``.

        :param description: The description for the IPAM.
        :param operating_regions: The operating Regions for an IPAM. Operating Regions are AWS Regions where the IPAM is allowed to manage IP address CIDRs. IPAM only discovers and monitors resources in the AWS Regions you select as operating Regions. For more information about operating Regions, see `Create an IPAM <https://docs.aws.amazon.com//vpc/latest/ipam/create-ipam.html>`_ in the *Amazon VPC IPAM User Guide* .
        :param tags: The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the tag value as the filter value. For example, to find all resources that have a tag with the key ``Owner`` and the value ``TeamA`` , specify ``tag:Owner`` for the filter name and ``TeamA`` for the filter value.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipam.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            cfn_iPAMProps = ec2.CfnIPAMProps(
                description="description",
                operating_regions=[ec2.CfnIPAM.IpamOperatingRegionProperty(
                    region_name="regionName"
                )],
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if description is not None:
            self._values["description"] = description
        if operating_regions is not None:
            self._values["operating_regions"] = operating_regions
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''The description for the IPAM.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipam.html#cfn-ec2-ipam-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def operating_regions(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnIPAM.IpamOperatingRegionProperty]]]]:
        '''The operating Regions for an IPAM.

        Operating Regions are AWS Regions where the IPAM is allowed to manage IP address CIDRs. IPAM only discovers and monitors resources in the AWS Regions you select as operating Regions.

        For more information about operating Regions, see `Create an IPAM <https://docs.aws.amazon.com//vpc/latest/ipam/create-ipam.html>`_ in the *Amazon VPC IPAM User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipam.html#cfn-ec2-ipam-operatingregions
        '''
        result = self._values.get("operating_regions")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnIPAM.IpamOperatingRegionProperty]]]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''The key/value combination of a tag assigned to the resource.

        Use the tag key in the filter name and the tag value as the filter value. For example, to find all resources that have a tag with the key ``Owner`` and the value ``TeamA`` , specify ``tag:Owner`` for the filter name and ``TeamA`` for the filter value.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipam.html#cfn-ec2-ipam-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnIPAMProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnIPAMScope(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnIPAMScope",
):
    '''A CloudFormation ``AWS::EC2::IPAMScope``.

    In IPAM, a scope is the highest-level container within IPAM. An IPAM contains two default scopes. Each scope represents the IP space for a single network. The private scope is intended for all private IP address space. The public scope is intended for all public IP address space. Scopes enable you to reuse IP addresses across multiple unconnected networks without causing IP address overlap or conflict.

    For more information, see `How IPAM works <https://docs.aws.amazon.com//vpc/latest/ipam/how-it-works-ipam.html>`_ in the *Amazon VPC IPAM User Guide*

    :cloudformationResource: AWS::EC2::IPAMScope
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamscope.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ec2 as ec2
        
        cfn_iPAMScope = ec2.CfnIPAMScope(self, "MyCfnIPAMScope",
            ipam_id="ipamId",
        
            # the properties below are optional
            description="description",
            ipam_scope_type="ipamScopeType",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        ipam_id: builtins.str,
        description: typing.Optional[builtins.str] = None,
        ipam_scope_type: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::IPAMScope``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param ipam_id: The ID of the IPAM for which you're creating this scope.
        :param description: The description of the scope.
        :param ipam_scope_type: The type of the scope.
        :param tags: The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the tag value as the filter value. For example, to find all resources that have a tag with the key ``Owner`` and the value ``TeamA`` , specify ``tag:Owner`` for the filter name and ``TeamA`` for the filter value.
        '''
        props = CfnIPAMScopeProps(
            ipam_id=ipam_id,
            description=description,
            ipam_scope_type=ipam_scope_type,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The ARN of the scope.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrIpamArn")
    def attr_ipam_arn(self) -> builtins.str:
        '''The ARN of an IPAM.

        :cloudformationAttribute: IpamArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrIpamArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrIpamScopeId")
    def attr_ipam_scope_id(self) -> builtins.str:
        '''The ID of an IPAM scope.

        :cloudformationAttribute: IpamScopeId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrIpamScopeId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrIsDefault")
    def attr_is_default(self) -> aws_cdk.core.IResolvable:
        '''Defines if the scope is the default scope or not.

        :cloudformationAttribute: IsDefault
        '''
        return typing.cast(aws_cdk.core.IResolvable, jsii.get(self, "attrIsDefault"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrPoolCount")
    def attr_pool_count(self) -> jsii.Number:
        '''The number of pools in a scope.

        :cloudformationAttribute: PoolCount
        '''
        return typing.cast(jsii.Number, jsii.get(self, "attrPoolCount"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''The key/value combination of a tag assigned to the resource.

        Use the tag key in the filter name and the tag value as the filter value. For example, to find all resources that have a tag with the key ``Owner`` and the value ``TeamA`` , specify ``tag:Owner`` for the filter name and ``TeamA`` for the filter value.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamscope.html#cfn-ec2-ipamscope-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ipamId")
    def ipam_id(self) -> builtins.str:
        '''The ID of the IPAM for which you're creating this scope.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamscope.html#cfn-ec2-ipamscope-ipamid
        '''
        return typing.cast(builtins.str, jsii.get(self, "ipamId"))

    @ipam_id.setter
    def ipam_id(self, value: builtins.str) -> None:
        jsii.set(self, "ipamId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the scope.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamscope.html#cfn-ec2-ipamscope-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "description", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ipamScopeType")
    def ipam_scope_type(self) -> typing.Optional[builtins.str]:
        '''The type of the scope.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamscope.html#cfn-ec2-ipamscope-ipamscopetype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "ipamScopeType"))

    @ipam_scope_type.setter
    def ipam_scope_type(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "ipamScopeType", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnIPAMScopeProps",
    jsii_struct_bases=[],
    name_mapping={
        "ipam_id": "ipamId",
        "description": "description",
        "ipam_scope_type": "ipamScopeType",
        "tags": "tags",
    },
)
class CfnIPAMScopeProps:
    def __init__(
        self,
        *,
        ipam_id: builtins.str,
        description: typing.Optional[builtins.str] = None,
        ipam_scope_type: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Properties for defining a ``CfnIPAMScope``.

        :param ipam_id: The ID of the IPAM for which you're creating this scope.
        :param description: The description of the scope.
        :param ipam_scope_type: The type of the scope.
        :param tags: The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the tag value as the filter value. For example, to find all resources that have a tag with the key ``Owner`` and the value ``TeamA`` , specify ``tag:Owner`` for the filter name and ``TeamA`` for the filter value.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamscope.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            cfn_iPAMScope_props = ec2.CfnIPAMScopeProps(
                ipam_id="ipamId",
            
                # the properties below are optional
                description="description",
                ipam_scope_type="ipamScopeType",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "ipam_id": ipam_id,
        }
        if description is not None:
            self._values["description"] = description
        if ipam_scope_type is not None:
            self._values["ipam_scope_type"] = ipam_scope_type
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def ipam_id(self) -> builtins.str:
        '''The ID of the IPAM for which you're creating this scope.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamscope.html#cfn-ec2-ipamscope-ipamid
        '''
        result = self._values.get("ipam_id")
        assert result is not None, "Required property 'ipam_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the scope.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamscope.html#cfn-ec2-ipamscope-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ipam_scope_type(self) -> typing.Optional[builtins.str]:
        '''The type of the scope.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamscope.html#cfn-ec2-ipamscope-ipamscopetype
        '''
        result = self._values.get("ipam_scope_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''The key/value combination of a tag assigned to the resource.

        Use the tag key in the filter name and the tag value as the filter value. For example, to find all resources that have a tag with the key ``Owner`` and the value ``TeamA`` , specify ``tag:Owner`` for the filter name and ``TeamA`` for the filter value.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamscope.html#cfn-ec2-ipamscope-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnIPAMScopeProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnInstance(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnInstance",
):
    '''A CloudFormation ``AWS::EC2::Instance``.

    Specifies an EC2 instance.

    If an Elastic IP address is attached to your instance, AWS CloudFormation reattaches the Elastic IP address after it updates the instance. For more information about updating stacks, see `AWS CloudFormation Stacks Updates <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks.html>`_ .

    :cloudformationResource: AWS::EC2::Instance
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ec2 as ec2
        
        cfn_instance = ec2.CfnInstance(self, "MyCfnInstance",
            additional_info="additionalInfo",
            affinity="affinity",
            availability_zone="availabilityZone",
            block_device_mappings=[ec2.CfnInstance.BlockDeviceMappingProperty(
                device_name="deviceName",
        
                # the properties below are optional
                ebs=ec2.CfnInstance.EbsProperty(
                    delete_on_termination=False,
                    encrypted=False,
                    iops=123,
                    kms_key_id="kmsKeyId",
                    snapshot_id="snapshotId",
                    volume_size=123,
                    volume_type="volumeType"
                ),
                no_device=ec2.CfnInstance.NoDeviceProperty(),
                virtual_name="virtualName"
            )],
            cpu_options=ec2.CfnInstance.CpuOptionsProperty(
                core_count=123,
                threads_per_core=123
            ),
            credit_specification=ec2.CfnInstance.CreditSpecificationProperty(
                cpu_credits="cpuCredits"
            ),
            disable_api_termination=False,
            ebs_optimized=False,
            elastic_gpu_specifications=[ec2.CfnInstance.ElasticGpuSpecificationProperty(
                type="type"
            )],
            elastic_inference_accelerators=[ec2.CfnInstance.ElasticInferenceAcceleratorProperty(
                type="type",
        
                # the properties below are optional
                count=123
            )],
            enclave_options=ec2.CfnInstance.EnclaveOptionsProperty(
                enabled=False
            ),
            hibernation_options=ec2.CfnInstance.HibernationOptionsProperty(
                configured=False
            ),
            host_id="hostId",
            host_resource_group_arn="hostResourceGroupArn",
            iam_instance_profile="iamInstanceProfile",
            image_id="imageId",
            instance_initiated_shutdown_behavior="instanceInitiatedShutdownBehavior",
            instance_type="instanceType",
            ipv6_address_count=123,
            ipv6_addresses=[ec2.CfnInstance.InstanceIpv6AddressProperty(
                ipv6_address="ipv6Address"
            )],
            kernel_id="kernelId",
            key_name="keyName",
            launch_template=ec2.CfnInstance.LaunchTemplateSpecificationProperty(
                version="version",
        
                # the properties below are optional
                launch_template_id="launchTemplateId",
                launch_template_name="launchTemplateName"
            ),
            license_specifications=[ec2.CfnInstance.LicenseSpecificationProperty(
                license_configuration_arn="licenseConfigurationArn"
            )],
            monitoring=False,
            network_interfaces=[ec2.CfnInstance.NetworkInterfaceProperty(
                device_index="deviceIndex",
        
                # the properties below are optional
                associate_public_ip_address=False,
                delete_on_termination=False,
                description="description",
                group_set=["groupSet"],
                ipv6_address_count=123,
                ipv6_addresses=[ec2.CfnInstance.InstanceIpv6AddressProperty(
                    ipv6_address="ipv6Address"
                )],
                network_interface_id="networkInterfaceId",
                private_ip_address="privateIpAddress",
                private_ip_addresses=[ec2.CfnInstance.PrivateIpAddressSpecificationProperty(
                    primary=False,
                    private_ip_address="privateIpAddress"
                )],
                secondary_private_ip_address_count=123,
                subnet_id="subnetId"
            )],
            placement_group_name="placementGroupName",
            private_ip_address="privateIpAddress",
            propagate_tags_to_volume_on_creation=False,
            ramdisk_id="ramdiskId",
            security_group_ids=["securityGroupIds"],
            security_groups=["securityGroups"],
            source_dest_check=False,
            ssm_associations=[ec2.CfnInstance.SsmAssociationProperty(
                document_name="documentName",
        
                # the properties below are optional
                association_parameters=[ec2.CfnInstance.AssociationParameterProperty(
                    key="key",
                    value=["value"]
                )]
            )],
            subnet_id="subnetId",
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            tenancy="tenancy",
            user_data="userData",
            volumes=[ec2.CfnInstance.VolumeProperty(
                device="device",
                volume_id="volumeId"
            )]
        )
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        additional_info: typing.Optional[builtins.str] = None,
        affinity: typing.Optional[builtins.str] = None,
        availability_zone: typing.Optional[builtins.str] = None,
        block_device_mappings: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.BlockDeviceMappingProperty"]]]] = None,
        cpu_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.CpuOptionsProperty"]] = None,
        credit_specification: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.CreditSpecificationProperty"]] = None,
        disable_api_termination: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        ebs_optimized: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        elastic_gpu_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.ElasticGpuSpecificationProperty"]]]] = None,
        elastic_inference_accelerators: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.ElasticInferenceAcceleratorProperty"]]]] = None,
        enclave_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.EnclaveOptionsProperty"]] = None,
        hibernation_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.HibernationOptionsProperty"]] = None,
        host_id: typing.Optional[builtins.str] = None,
        host_resource_group_arn: typing.Optional[builtins.str] = None,
        iam_instance_profile: typing.Optional[builtins.str] = None,
        image_id: typing.Optional[builtins.str] = None,
        instance_initiated_shutdown_behavior: typing.Optional[builtins.str] = None,
        instance_type: typing.Optional[builtins.str] = None,
        ipv6_address_count: typing.Optional[jsii.Number] = None,
        ipv6_addresses: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.InstanceIpv6AddressProperty"]]]] = None,
        kernel_id: typing.Optional[builtins.str] = None,
        key_name: typing.Optional[builtins.str] = None,
        launch_template: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.LaunchTemplateSpecificationProperty"]] = None,
        license_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.LicenseSpecificationProperty"]]]] = None,
        monitoring: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        network_interfaces: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.NetworkInterfaceProperty"]]]] = None,
        placement_group_name: typing.Optional[builtins.str] = None,
        private_ip_address: typing.Optional[builtins.str] = None,
        propagate_tags_to_volume_on_creation: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        ramdisk_id: typing.Optional[builtins.str] = None,
        security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
        security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
        source_dest_check: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        ssm_associations: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.SsmAssociationProperty"]]]] = None,
        subnet_id: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
        tenancy: typing.Optional[builtins.str] = None,
        user_data: typing.Optional[builtins.str] = None,
        volumes: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.VolumeProperty"]]]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::Instance``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param additional_info: This property is reserved for internal use. If you use it, the stack fails with this error: ``Bad property set: [Testing this property] (Service: AmazonEC2; Status Code: 400; Error Code: InvalidParameterCombination; Request ID: 0XXXXXX-49c7-4b40-8bcc-76885dcXXXXX)`` .
        :param affinity: Indicates whether the instance is associated with a dedicated host. If you want the instance to always restart on the same host on which it was launched, specify ``host`` . If you want the instance to restart on any available host, but try to launch onto the last host it ran on (on a best-effort basis), specify ``default`` .
        :param availability_zone: The Availability Zone of the instance. If not specified, an Availability Zone will be automatically chosen for you based on the load balancing criteria for the Region. This parameter is not supported by `DescribeImageAttribute <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImageAttribute.html>`_ .
        :param block_device_mappings: The block device mapping entries that defines the block devices to attach to the instance at launch. By default, the block devices specified in the block device mapping for the AMI are used. You can override the AMI block device mapping using the instance block device mapping. For the root volume, you can override only the volume size, volume type, volume encryption settings, and the ``DeleteOnTermination`` setting. .. epigraph:: After the instance is running, you can modify only the ``DeleteOnTermination`` parameter for the attached volumes without interrupting the instance. Modifying any other parameter results in instance `replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ .
        :param cpu_options: The CPU options for the instance. For more information, see `Optimize CPU options <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-optimize-cpu.html>`_ in the *Amazon Elastic Compute Cloud User Guide* .
        :param credit_specification: The credit option for CPU usage of the burstable performance instance. Valid values are ``standard`` and ``unlimited`` . To change this attribute after launch, use `ModifyInstanceCreditSpecification <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceCreditSpecification.html>`_ . For more information, see `Burstable performance instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances.html>`_ in the *Amazon EC2 User Guide* . Default: ``standard`` (T2 instances) or ``unlimited`` (T3/T3a instances) For T3 instances with ``host`` tenancy, only ``standard`` is supported.
        :param disable_api_termination: If you set this parameter to ``true`` , you can't terminate the instance using the Amazon EC2 console, CLI, or API; otherwise, you can. To change this attribute after launch, use `ModifyInstanceAttribute <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceAttribute.html>`_ . Alternatively, if you set ``InstanceInitiatedShutdownBehavior`` to ``terminate`` , you can terminate the instance by running the shutdown command from the instance. Default: ``false``
        :param ebs_optimized: Indicates whether the instance is optimized for Amazon EBS I/O. This optimization provides dedicated throughput to Amazon EBS and an optimized configuration stack to provide optimal Amazon EBS I/O performance. This optimization isn't available with all instance types. Additional usage charges apply when using an EBS-optimized instance. Default: ``false``
        :param elastic_gpu_specifications: An elastic GPU to associate with the instance. An Elastic GPU is a GPU resource that you can attach to your Windows instance to accelerate the graphics performance of your applications. For more information, see `Amazon EC2 Elastic GPUs <https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/elastic-graphics.html>`_ in the *Amazon EC2 User Guide* .
        :param elastic_inference_accelerators: An elastic inference accelerator to associate with the instance. Elastic inference accelerators are a resource you can attach to your Amazon EC2 instances to accelerate your Deep Learning (DL) inference workloads. You cannot specify accelerators from different generations in the same request.
        :param enclave_options: Indicates whether the instance is enabled for AWS Nitro Enclaves.
        :param hibernation_options: Indicates whether an instance is enabled for hibernation. For more information, see `Hibernate your instance <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Hibernate.html>`_ in the *Amazon EC2 User Guide* . You can't enable hibernation and AWS Nitro Enclaves on the same instance.
        :param host_id: If you specify host for the ``Affinity`` property, the ID of a dedicated host that the instance is associated with. If you don't specify an ID, Amazon EC2 launches the instance onto any available, compatible dedicated host in your account. This type of launch is called an untargeted launch. Note that for untargeted launches, you must have a compatible, dedicated host available to successfully launch instances.
        :param host_resource_group_arn: The ARN of the host resource group in which to launch the instances. If you specify a host resource group ARN, omit the *Tenancy* parameter or set it to ``host`` .
        :param iam_instance_profile: The name of an IAM instance profile. To create a new IAM instance profile, use the `AWS::IAM::InstanceProfile <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-iam-instanceprofile.html>`_ resource.
        :param image_id: The ID of the AMI. An AMI ID is required to launch an instance and must be specified here or in a launch template.
        :param instance_initiated_shutdown_behavior: Indicates whether an instance stops or terminates when you initiate shutdown from the instance (using the operating system command for system shutdown). Default: ``stop``
        :param instance_type: The instance type. For more information, see `Instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html>`_ in the *Amazon EC2 User Guide* . Default: ``m1.small``
        :param ipv6_address_count: [EC2-VPC] The number of IPv6 addresses to associate with the primary network interface. Amazon EC2 chooses the IPv6 addresses from the range of your subnet. You cannot specify this option and the option to assign specific IPv6 addresses in the same request. You can specify this option if you've specified a minimum number of instances to launch. You cannot specify this option and the network interfaces option in the same request.
        :param ipv6_addresses: [EC2-VPC] The IPv6 addresses from the range of the subnet to associate with the primary network interface. You cannot specify this option and the option to assign a number of IPv6 addresses in the same request. You cannot specify this option if you've specified a minimum number of instances to launch. You cannot specify this option and the network interfaces option in the same request.
        :param kernel_id: The ID of the kernel. .. epigraph:: We recommend that you use PV-GRUB instead of kernels and RAM disks. For more information, see `PV-GRUB <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/UserProvidedkernels.html>`_ in the *Amazon EC2 User Guide* .
        :param key_name: The name of the key pair. You can create a key pair using `CreateKeyPair <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateKeyPair.html>`_ or `ImportKeyPair <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportKeyPair.html>`_ . .. epigraph:: If you do not specify a key pair, you can't connect to the instance unless you choose an AMI that is configured to allow users another way to log in.
        :param launch_template: The launch template to use to launch the instances. Any parameters that you specify in the AWS CloudFormation template override the same parameters in the launch template. You can specify either the name or ID of a launch template, but not both.
        :param license_specifications: The license configurations.
        :param monitoring: Specifies whether detailed monitoring is enabled for the instance.
        :param network_interfaces: The network interfaces to associate with the instance. .. epigraph:: If you use this property to point to a network interface, you must terminate the original interface before attaching a new one to allow the update of the instance to succeed. If this resource has a public IP address and is also in a VPC that is defined in the same template, you must use the `DependsOn Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html>`_ to declare a dependency on the VPC-gateway attachment.
        :param placement_group_name: The name of an existing placement group that you want to launch the instance into (cluster | partition | spread).
        :param private_ip_address: [EC2-VPC] The primary IPv4 address. You must specify a value from the IPv4 address range of the subnet. Only one private IP address can be designated as primary. You can't specify this option if you've specified the option to designate a private IP address as the primary IP address in a network interface specification. You cannot specify this option if you're launching more than one instance in the request. You cannot specify this option and the network interfaces option in the same request. If you make an update to an instance that requires replacement, you must assign a new private IP address. During a replacement, AWS CloudFormation creates a new instance but doesn't delete the old instance until the stack has successfully updated. If the stack update fails, AWS CloudFormation uses the old instance to roll back the stack to the previous working state. The old and new instances cannot have the same private IP address.
        :param propagate_tags_to_volume_on_creation: Whether to propagate the EC2 instance tags to the EBS volumes.
        :param ramdisk_id: The ID of the RAM disk to select. Some kernels require additional drivers at launch. Check the kernel requirements for information about whether you need to specify a RAM disk. To find kernel requirements, go to the AWS Resource Center and search for the kernel ID. .. epigraph:: We recommend that you use PV-GRUB instead of kernels and RAM disks. For more information, see `PV-GRUB <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/UserProvidedkernels.html>`_ in the *Amazon EC2 User Guide* .
        :param security_group_ids: The IDs of the security groups. You can create a security group using `CreateSecurityGroup <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSecurityGroup.html>`_ . If you specify a network interface, you must specify any security groups as part of the network interface.
        :param security_groups: [EC2-Classic, default VPC] The names of the security groups. For a nondefault VPC, you must use security group IDs instead. You cannot specify this option and the network interfaces option in the same request. The list can contain both the name of existing Amazon EC2 security groups or references to AWS::EC2::SecurityGroup resources created in the template. Default: Amazon EC2 uses the default security group.
        :param source_dest_check: Enable or disable source/destination checks, which ensure that the instance is either the source or the destination of any traffic that it receives. If the value is ``true`` , source/destination checks are enabled; otherwise, they are disabled. The default value is ``true`` . You must disable source/destination checks if the instance runs services such as network address translation, routing, or firewalls.
        :param ssm_associations: The SSM `document <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html>`_ and parameter values in AWS Systems Manager to associate with this instance. To use this property, you must specify an IAM instance profile role for the instance. For more information, see `Create an Instance Profile for Systems Manager <https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-configuring-access-role.html>`_ in the *AWS Systems Manager User Guide* . .. epigraph:: You can currently associate only one document with an instance.
        :param subnet_id: [EC2-VPC] The ID of the subnet to launch the instance into. If you specify a network interface, you must specify any subnets as part of the network interface.
        :param tags: The tags to add to the instance. These tags are not applied to the EBS volumes, such as the root volume.
        :param tenancy: The tenancy of the instance (if the instance is running in a VPC). An instance with a tenancy of ``dedicated`` runs on single-tenant hardware.
        :param user_data: The user data to make available to the instance. For more information, see `Run commands on your Linux instance at launch <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/user-data.html>`_ and `Run commands on your Windows instance at launch <https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ec2-windows-user-data.html>`_ . If you are using a command line tool, base64-encoding is performed for you, and you can load the text from a file. Otherwise, you must provide base64-encoded text. User data is limited to 16 KB.
        :param volumes: The volumes to attach to the instance.
        '''
        props = CfnInstanceProps(
            additional_info=additional_info,
            affinity=affinity,
            availability_zone=availability_zone,
            block_device_mappings=block_device_mappings,
            cpu_options=cpu_options,
            credit_specification=credit_specification,
            disable_api_termination=disable_api_termination,
            ebs_optimized=ebs_optimized,
            elastic_gpu_specifications=elastic_gpu_specifications,
            elastic_inference_accelerators=elastic_inference_accelerators,
            enclave_options=enclave_options,
            hibernation_options=hibernation_options,
            host_id=host_id,
            host_resource_group_arn=host_resource_group_arn,
            iam_instance_profile=iam_instance_profile,
            image_id=image_id,
            instance_initiated_shutdown_behavior=instance_initiated_shutdown_behavior,
            instance_type=instance_type,
            ipv6_address_count=ipv6_address_count,
            ipv6_addresses=ipv6_addresses,
            kernel_id=kernel_id,
            key_name=key_name,
            launch_template=launch_template,
            license_specifications=license_specifications,
            monitoring=monitoring,
            network_interfaces=network_interfaces,
            placement_group_name=placement_group_name,
            private_ip_address=private_ip_address,
            propagate_tags_to_volume_on_creation=propagate_tags_to_volume_on_creation,
            ramdisk_id=ramdisk_id,
            security_group_ids=security_group_ids,
            security_groups=security_groups,
            source_dest_check=source_dest_check,
            ssm_associations=ssm_associations,
            subnet_id=subnet_id,
            tags=tags,
            tenancy=tenancy,
            user_data=user_data,
            volumes=volumes,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrAvailabilityZone")
    def attr_availability_zone(self) -> builtins.str:
        '''The Availability Zone where the specified instance is launched. For example: ``us-east-1b`` .

        You can retrieve a list of all Availability Zones for a Region by using the `Fn::GetAZs <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-getavailabilityzones.html>`_ intrinsic function.

        :cloudformationAttribute: AvailabilityZone
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrAvailabilityZone"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrPrivateDnsName")
    def attr_private_dns_name(self) -> builtins.str:
        '''
        :cloudformationAttribute: PrivateDnsName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrPrivateDnsName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrPrivateIp")
    def attr_private_ip(self) -> builtins.str:
        '''The private IP address of the specified instance.

        For example: ``10.24.34.0`` .

        :cloudformationAttribute: PrivateIp
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrPrivateIp"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrPublicDnsName")
    def attr_public_dns_name(self) -> builtins.str:
        '''The public DNS name of the specified instance.

        For example: ``ec2-107-20-50-45.compute-1.amazonaws.com`` .

        :cloudformationAttribute: PublicDnsName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrPublicDnsName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrPublicIp")
    def attr_public_ip(self) -> builtins.str:
        '''The public IP address of the specified instance.

        For example: ``192.0.2.0`` .

        :cloudformationAttribute: PublicIp
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrPublicIp"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''The tags to add to the instance.

        These tags are not applied to the EBS volumes, such as the root volume.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="additionalInfo")
    def additional_info(self) -> typing.Optional[builtins.str]:
        '''This property is reserved for internal use.

        If you use it, the stack fails with this error: ``Bad property set: [Testing this property] (Service: AmazonEC2; Status Code: 400; Error Code: InvalidParameterCombination; Request ID: 0XXXXXX-49c7-4b40-8bcc-76885dcXXXXX)`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-additionalinfo
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "additionalInfo"))

    @additional_info.setter
    def additional_info(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "additionalInfo", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="affinity")
    def affinity(self) -> typing.Optional[builtins.str]:
        '''Indicates whether the instance is associated with a dedicated host.

        If you want the instance to always restart on the same host on which it was launched, specify ``host`` . If you want the instance to restart on any available host, but try to launch onto the last host it ran on (on a best-effort basis), specify ``default`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-affinity
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "affinity"))

    @affinity.setter
    def affinity(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "affinity", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="availabilityZone")
    def availability_zone(self) -> typing.Optional[builtins.str]:
        '''The Availability Zone of the instance.

        If not specified, an Availability Zone will be automatically chosen for you based on the load balancing criteria for the Region.

        This parameter is not supported by `DescribeImageAttribute <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImageAttribute.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-availabilityzone
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "availabilityZone"))

    @availability_zone.setter
    def availability_zone(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "availabilityZone", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="blockDeviceMappings")
    def block_device_mappings(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.BlockDeviceMappingProperty"]]]]:
        '''The block device mapping entries that defines the block devices to attach to the instance at launch.

        By default, the block devices specified in the block device mapping for the AMI are used. You can override the AMI block device mapping using the instance block device mapping. For the root volume, you can override only the volume size, volume type, volume encryption settings, and the ``DeleteOnTermination`` setting.
        .. epigraph::

           After the instance is running, you can modify only the ``DeleteOnTermination`` parameter for the attached volumes without interrupting the instance. Modifying any other parameter results in instance `replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-blockdevicemappings
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.BlockDeviceMappingProperty"]]]], jsii.get(self, "blockDeviceMappings"))

    @block_device_mappings.setter
    def block_device_mappings(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.BlockDeviceMappingProperty"]]]],
    ) -> None:
        jsii.set(self, "blockDeviceMappings", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cpuOptions")
    def cpu_options(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.CpuOptionsProperty"]]:
        '''The CPU options for the instance.

        For more information, see `Optimize CPU options <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-optimize-cpu.html>`_ in the *Amazon Elastic Compute Cloud User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-cpuoptions
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.CpuOptionsProperty"]], jsii.get(self, "cpuOptions"))

    @cpu_options.setter
    def cpu_options(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.CpuOptionsProperty"]],
    ) -> None:
        jsii.set(self, "cpuOptions", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="creditSpecification")
    def credit_specification(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.CreditSpecificationProperty"]]:
        '''The credit option for CPU usage of the burstable performance instance.

        Valid values are ``standard`` and ``unlimited`` . To change this attribute after launch, use `ModifyInstanceCreditSpecification <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceCreditSpecification.html>`_ . For more information, see `Burstable performance instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances.html>`_ in the *Amazon EC2 User Guide* .

        Default: ``standard`` (T2 instances) or ``unlimited`` (T3/T3a instances)

        For T3 instances with ``host`` tenancy, only ``standard`` is supported.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-creditspecification
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.CreditSpecificationProperty"]], jsii.get(self, "creditSpecification"))

    @credit_specification.setter
    def credit_specification(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.CreditSpecificationProperty"]],
    ) -> None:
        jsii.set(self, "creditSpecification", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="disableApiTermination")
    def disable_api_termination(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''If you set this parameter to ``true`` , you can't terminate the instance using the Amazon EC2 console, CLI, or API;

        otherwise, you can. To change this attribute after launch, use `ModifyInstanceAttribute <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceAttribute.html>`_ . Alternatively, if you set ``InstanceInitiatedShutdownBehavior`` to ``terminate`` , you can terminate the instance by running the shutdown command from the instance.

        Default: ``false``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-disableapitermination
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "disableApiTermination"))

    @disable_api_termination.setter
    def disable_api_termination(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "disableApiTermination", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ebsOptimized")
    def ebs_optimized(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''Indicates whether the instance is optimized for Amazon EBS I/O.

        This optimization provides dedicated throughput to Amazon EBS and an optimized configuration stack to provide optimal Amazon EBS I/O performance. This optimization isn't available with all instance types. Additional usage charges apply when using an EBS-optimized instance.

        Default: ``false``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-ebsoptimized
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "ebsOptimized"))

    @ebs_optimized.setter
    def ebs_optimized(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "ebsOptimized", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="elasticGpuSpecifications")
    def elastic_gpu_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.ElasticGpuSpecificationProperty"]]]]:
        '''An elastic GPU to associate with the instance.

        An Elastic GPU is a GPU resource that you can attach to your Windows instance to accelerate the graphics performance of your applications. For more information, see `Amazon EC2 Elastic GPUs <https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/elastic-graphics.html>`_ in the *Amazon EC2 User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-elasticgpuspecifications
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.ElasticGpuSpecificationProperty"]]]], jsii.get(self, "elasticGpuSpecifications"))

    @elastic_gpu_specifications.setter
    def elastic_gpu_specifications(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.ElasticGpuSpecificationProperty"]]]],
    ) -> None:
        jsii.set(self, "elasticGpuSpecifications", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="elasticInferenceAccelerators")
    def elastic_inference_accelerators(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.ElasticInferenceAcceleratorProperty"]]]]:
        '''An elastic inference accelerator to associate with the instance.

        Elastic inference accelerators are a resource you can attach to your Amazon EC2 instances to accelerate your Deep Learning (DL) inference workloads.

        You cannot specify accelerators from different generations in the same request.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-elasticinferenceaccelerators
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.ElasticInferenceAcceleratorProperty"]]]], jsii.get(self, "elasticInferenceAccelerators"))

    @elastic_inference_accelerators.setter
    def elastic_inference_accelerators(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.ElasticInferenceAcceleratorProperty"]]]],
    ) -> None:
        jsii.set(self, "elasticInferenceAccelerators", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="enclaveOptions")
    def enclave_options(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.EnclaveOptionsProperty"]]:
        '''Indicates whether the instance is enabled for AWS Nitro Enclaves.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-enclaveoptions
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.EnclaveOptionsProperty"]], jsii.get(self, "enclaveOptions"))

    @enclave_options.setter
    def enclave_options(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.EnclaveOptionsProperty"]],
    ) -> None:
        jsii.set(self, "enclaveOptions", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="hibernationOptions")
    def hibernation_options(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.HibernationOptionsProperty"]]:
        '''Indicates whether an instance is enabled for hibernation.

        For more information, see `Hibernate your instance <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Hibernate.html>`_ in the *Amazon EC2 User Guide* .

        You can't enable hibernation and AWS Nitro Enclaves on the same instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-hibernationoptions
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.HibernationOptionsProperty"]], jsii.get(self, "hibernationOptions"))

    @hibernation_options.setter
    def hibernation_options(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.HibernationOptionsProperty"]],
    ) -> None:
        jsii.set(self, "hibernationOptions", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="hostId")
    def host_id(self) -> typing.Optional[builtins.str]:
        '''If you specify host for the ``Affinity`` property, the ID of a dedicated host that the instance is associated with.

        If you don't specify an ID, Amazon EC2 launches the instance onto any available, compatible dedicated host in your account. This type of launch is called an untargeted launch. Note that for untargeted launches, you must have a compatible, dedicated host available to successfully launch instances.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-hostid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "hostId"))

    @host_id.setter
    def host_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "hostId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="hostResourceGroupArn")
    def host_resource_group_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN of the host resource group in which to launch the instances.

        If you specify a host resource group ARN, omit the *Tenancy* parameter or set it to ``host`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-hostresourcegrouparn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "hostResourceGroupArn"))

    @host_resource_group_arn.setter
    def host_resource_group_arn(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "hostResourceGroupArn", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="iamInstanceProfile")
    def iam_instance_profile(self) -> typing.Optional[builtins.str]:
        '''The name of an IAM instance profile.

        To create a new IAM instance profile, use the `AWS::IAM::InstanceProfile <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-iam-instanceprofile.html>`_ resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-iaminstanceprofile
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "iamInstanceProfile"))

    @iam_instance_profile.setter
    def iam_instance_profile(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "iamInstanceProfile", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="imageId")
    def image_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the AMI.

        An AMI ID is required to launch an instance and must be specified here or in a launch template.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-imageid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "imageId"))

    @image_id.setter
    def image_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "imageId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceInitiatedShutdownBehavior")
    def instance_initiated_shutdown_behavior(self) -> typing.Optional[builtins.str]:
        '''Indicates whether an instance stops or terminates when you initiate shutdown from the instance (using the operating system command for system shutdown).

        Default: ``stop``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-instanceinitiatedshutdownbehavior
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "instanceInitiatedShutdownBehavior"))

    @instance_initiated_shutdown_behavior.setter
    def instance_initiated_shutdown_behavior(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        jsii.set(self, "instanceInitiatedShutdownBehavior", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceType")
    def instance_type(self) -> typing.Optional[builtins.str]:
        '''The instance type. For more information, see `Instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html>`_ in the *Amazon EC2 User Guide* .

        Default: ``m1.small``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-instancetype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "instanceType"))

    @instance_type.setter
    def instance_type(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "instanceType", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ipv6AddressCount")
    def ipv6_address_count(self) -> typing.Optional[jsii.Number]:
        '''[EC2-VPC] The number of IPv6 addresses to associate with the primary network interface.

        Amazon EC2 chooses the IPv6 addresses from the range of your subnet. You cannot specify this option and the option to assign specific IPv6 addresses in the same request. You can specify this option if you've specified a minimum number of instances to launch.

        You cannot specify this option and the network interfaces option in the same request.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-ipv6addresscount
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "ipv6AddressCount"))

    @ipv6_address_count.setter
    def ipv6_address_count(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "ipv6AddressCount", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ipv6Addresses")
    def ipv6_addresses(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.InstanceIpv6AddressProperty"]]]]:
        '''[EC2-VPC] The IPv6 addresses from the range of the subnet to associate with the primary network interface.

        You cannot specify this option and the option to assign a number of IPv6 addresses in the same request. You cannot specify this option if you've specified a minimum number of instances to launch.

        You cannot specify this option and the network interfaces option in the same request.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-ipv6addresses
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.InstanceIpv6AddressProperty"]]]], jsii.get(self, "ipv6Addresses"))

    @ipv6_addresses.setter
    def ipv6_addresses(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.InstanceIpv6AddressProperty"]]]],
    ) -> None:
        jsii.set(self, "ipv6Addresses", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="kernelId")
    def kernel_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the kernel.

        .. epigraph::

           We recommend that you use PV-GRUB instead of kernels and RAM disks. For more information, see `PV-GRUB <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/UserProvidedkernels.html>`_ in the *Amazon EC2 User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-kernelid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "kernelId"))

    @kernel_id.setter
    def kernel_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "kernelId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="keyName")
    def key_name(self) -> typing.Optional[builtins.str]:
        '''The name of the key pair. You can create a key pair using `CreateKeyPair <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateKeyPair.html>`_ or `ImportKeyPair <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportKeyPair.html>`_ .

        .. epigraph::

           If you do not specify a key pair, you can't connect to the instance unless you choose an AMI that is configured to allow users another way to log in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-keyname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "keyName"))

    @key_name.setter
    def key_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "keyName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="launchTemplate")
    def launch_template(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.LaunchTemplateSpecificationProperty"]]:
        '''The launch template to use to launch the instances.

        Any parameters that you specify in the AWS CloudFormation template override the same parameters in the launch template. You can specify either the name or ID of a launch template, but not both.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-launchtemplate
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.LaunchTemplateSpecificationProperty"]], jsii.get(self, "launchTemplate"))

    @launch_template.setter
    def launch_template(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.LaunchTemplateSpecificationProperty"]],
    ) -> None:
        jsii.set(self, "launchTemplate", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="licenseSpecifications")
    def license_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.LicenseSpecificationProperty"]]]]:
        '''The license configurations.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-licensespecifications
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.LicenseSpecificationProperty"]]]], jsii.get(self, "licenseSpecifications"))

    @license_specifications.setter
    def license_specifications(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.LicenseSpecificationProperty"]]]],
    ) -> None:
        jsii.set(self, "licenseSpecifications", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="monitoring")
    def monitoring(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''Specifies whether detailed monitoring is enabled for the instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-monitoring
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "monitoring"))

    @monitoring.setter
    def monitoring(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "monitoring", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="networkInterfaces")
    def network_interfaces(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.NetworkInterfaceProperty"]]]]:
        '''The network interfaces to associate with the instance.

        .. epigraph::

           If you use this property to point to a network interface, you must terminate the original interface before attaching a new one to allow the update of the instance to succeed.

           If this resource has a public IP address and is also in a VPC that is defined in the same template, you must use the `DependsOn Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html>`_ to declare a dependency on the VPC-gateway attachment.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-networkinterfaces
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.NetworkInterfaceProperty"]]]], jsii.get(self, "networkInterfaces"))

    @network_interfaces.setter
    def network_interfaces(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.NetworkInterfaceProperty"]]]],
    ) -> None:
        jsii.set(self, "networkInterfaces", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="placementGroupName")
    def placement_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of an existing placement group that you want to launch the instance into (cluster | partition | spread).

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-placementgroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "placementGroupName"))

    @placement_group_name.setter
    def placement_group_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "placementGroupName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="privateIpAddress")
    def private_ip_address(self) -> typing.Optional[builtins.str]:
        '''[EC2-VPC] The primary IPv4 address. You must specify a value from the IPv4 address range of the subnet.

        Only one private IP address can be designated as primary. You can't specify this option if you've specified the option to designate a private IP address as the primary IP address in a network interface specification. You cannot specify this option if you're launching more than one instance in the request.

        You cannot specify this option and the network interfaces option in the same request.

        If you make an update to an instance that requires replacement, you must assign a new private IP address. During a replacement, AWS CloudFormation creates a new instance but doesn't delete the old instance until the stack has successfully updated. If the stack update fails, AWS CloudFormation uses the old instance to roll back the stack to the previous working state. The old and new instances cannot have the same private IP address.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-privateipaddress
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "privateIpAddress"))

    @private_ip_address.setter
    def private_ip_address(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "privateIpAddress", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="propagateTagsToVolumeOnCreation")
    def propagate_tags_to_volume_on_creation(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''Whether to propagate the EC2 instance tags to the EBS volumes.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-propagatetagstovolumeoncreation
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "propagateTagsToVolumeOnCreation"))

    @propagate_tags_to_volume_on_creation.setter
    def propagate_tags_to_volume_on_creation(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "propagateTagsToVolumeOnCreation", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ramdiskId")
    def ramdisk_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the RAM disk to select.

        Some kernels require additional drivers at launch. Check the kernel requirements for information about whether you need to specify a RAM disk. To find kernel requirements, go to the AWS Resource Center and search for the kernel ID.
        .. epigraph::

           We recommend that you use PV-GRUB instead of kernels and RAM disks. For more information, see `PV-GRUB <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/UserProvidedkernels.html>`_ in the *Amazon EC2 User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-ramdiskid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "ramdiskId"))

    @ramdisk_id.setter
    def ramdisk_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "ramdiskId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="securityGroupIds")
    def security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The IDs of the security groups. You can create a security group using `CreateSecurityGroup <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSecurityGroup.html>`_ .

        If you specify a network interface, you must specify any security groups as part of the network interface.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-securitygroupids
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "securityGroupIds"))

    @security_group_ids.setter
    def security_group_ids(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        jsii.set(self, "securityGroupIds", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="securityGroups")
    def security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
        '''[EC2-Classic, default VPC] The names of the security groups.

        For a nondefault VPC, you must use security group IDs instead.

        You cannot specify this option and the network interfaces option in the same request. The list can contain both the name of existing Amazon EC2 security groups or references to AWS::EC2::SecurityGroup resources created in the template.

        Default: Amazon EC2 uses the default security group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-securitygroups
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "securityGroups"))

    @security_groups.setter
    def security_groups(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        jsii.set(self, "securityGroups", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sourceDestCheck")
    def source_dest_check(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''Enable or disable source/destination checks, which ensure that the instance is either the source or the destination of any traffic that it receives.

        If the value is ``true`` , source/destination checks are enabled; otherwise, they are disabled. The default value is ``true`` . You must disable source/destination checks if the instance runs services such as network address translation, routing, or firewalls.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-sourcedestcheck
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "sourceDestCheck"))

    @source_dest_check.setter
    def source_dest_check(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "sourceDestCheck", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ssmAssociations")
    def ssm_associations(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.SsmAssociationProperty"]]]]:
        '''The SSM `document <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html>`_ and parameter values in AWS Systems Manager to associate with this instance. To use this property, you must specify an IAM instance profile role for the instance. For more information, see `Create an Instance Profile for Systems Manager <https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-configuring-access-role.html>`_ in the *AWS Systems Manager User Guide* .

        .. epigraph::

           You can currently associate only one document with an instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-ssmassociations
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.SsmAssociationProperty"]]]], jsii.get(self, "ssmAssociations"))

    @ssm_associations.setter
    def ssm_associations(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.SsmAssociationProperty"]]]],
    ) -> None:
        jsii.set(self, "ssmAssociations", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="subnetId")
    def subnet_id(self) -> typing.Optional[builtins.str]:
        '''[EC2-VPC] The ID of the subnet to launch the instance into.

        If you specify a network interface, you must specify any subnets as part of the network interface.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-subnetid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "subnetId"))

    @subnet_id.setter
    def subnet_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "subnetId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tenancy")
    def tenancy(self) -> typing.Optional[builtins.str]:
        '''The tenancy of the instance (if the instance is running in a VPC).

        An instance with a tenancy of ``dedicated`` runs on single-tenant hardware.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-tenancy
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "tenancy"))

    @tenancy.setter
    def tenancy(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "tenancy", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="userData")
    def user_data(self) -> typing.Optional[builtins.str]:
        '''The user data to make available to the instance.

        For more information, see `Run commands on your Linux instance at launch <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/user-data.html>`_ and `Run commands on your Windows instance at launch <https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ec2-windows-user-data.html>`_ . If you are using a command line tool, base64-encoding is performed for you, and you can load the text from a file. Otherwise, you must provide base64-encoded text. User data is limited to 16 KB.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-userdata
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "userData"))

    @user_data.setter
    def user_data(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "userData", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="volumes")
    def volumes(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.VolumeProperty"]]]]:
        '''The volumes to attach to the instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-volumes
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.VolumeProperty"]]]], jsii.get(self, "volumes"))

    @volumes.setter
    def volumes(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.VolumeProperty"]]]],
    ) -> None:
        jsii.set(self, "volumes", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.AssociationParameterProperty",
        jsii_struct_bases=[],
        name_mapping={"key": "key", "value": "value"},
    )
    class AssociationParameterProperty:
        def __init__(
            self,
            *,
            key: builtins.str,
            value: typing.Sequence[builtins.str],
        ) -> None:
            '''Specifies input parameter values for an SSM document in AWS Systems Manager .

            ``AssociationParameter`` is a property of the `Amazon EC2 Instance SsmAssociation <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-ssmassociations.html>`_ property.

            :param key: The name of an input parameter that is in the associated SSM document.
            :param value: The value of an input parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-ssmassociations-associationparameters.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                association_parameter_property = ec2.CfnInstance.AssociationParameterProperty(
                    key="key",
                    value=["value"]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "key": key,
                "value": value,
            }

        @builtins.property
        def key(self) -> builtins.str:
            '''The name of an input parameter that is in the associated SSM document.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-ssmassociations-associationparameters.html#cfn-ec2-instance-ssmassociations-associationparameters-key
            '''
            result = self._values.get("key")
            assert result is not None, "Required property 'key' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> typing.List[builtins.str]:
            '''The value of an input parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-ssmassociations-associationparameters.html#cfn-ec2-instance-ssmassociations-associationparameters-value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(typing.List[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AssociationParameterProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.BlockDeviceMappingProperty",
        jsii_struct_bases=[],
        name_mapping={
            "device_name": "deviceName",
            "ebs": "ebs",
            "no_device": "noDevice",
            "virtual_name": "virtualName",
        },
    )
    class BlockDeviceMappingProperty:
        def __init__(
            self,
            *,
            device_name: builtins.str,
            ebs: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.EbsProperty"]] = None,
            no_device: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.NoDeviceProperty"]] = None,
            virtual_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Specifies a block device mapping for an instance.

            You must specify exactly one of the following properties: ``VirtualName`` , ``Ebs`` , or ``NoDevice`` .

            ``BlockDeviceMapping`` is a property of the `AWS::EC2::Instance <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html>`_ resource.
            .. epigraph::

               After the instance is running, you can modify only the ``DeleteOnTermination`` parameter for the attached volumes without interrupting the instance. Modifying any other parameter results in instance `replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ .

            :param device_name: The device name (for example, ``/dev/sdh`` or ``xvdh`` ). .. epigraph:: After the instance is running, this parameter is used to specify the device name of the block device mapping to update.
            :param ebs: Parameters used to automatically set up EBS volumes when the instance is launched. .. epigraph:: After the instance is running, you can modify only the ``DeleteOnTermination`` parameter for the attached volumes without interrupting the instance. Modifying any other parameter results in instance `replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-no-interrupt>`_ .
            :param no_device: To omit the device from the block device mapping, specify an empty string. .. epigraph:: After the instance is running, modifying this parameter results in instance `replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ .
            :param virtual_name: The virtual device name ( ``ephemeral`` N). The name must be in the form ``ephemeral`` *X* where *X* is a number starting from zero (0). For example, an instance type with 2 available instance store volumes can specify mappings for ``ephemeral0`` and ``ephemeral1`` . The number of available instance store volumes depends on the instance type. After you connect to the instance, you must mount the volume. NVMe instance store volumes are automatically enumerated and assigned a device name. Including them in your block device mapping has no effect. *Constraints* : For M3 instances, you must specify instance store volumes in the block device mapping for the instance. When you launch an M3 instance, we ignore any instance store volumes specified in the block device mapping for the AMI. .. epigraph:: After the instance is running, modifying this parameter results in instance `replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-mapping.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                block_device_mapping_property = ec2.CfnInstance.BlockDeviceMappingProperty(
                    device_name="deviceName",
                
                    # the properties below are optional
                    ebs=ec2.CfnInstance.EbsProperty(
                        delete_on_termination=False,
                        encrypted=False,
                        iops=123,
                        kms_key_id="kmsKeyId",
                        snapshot_id="snapshotId",
                        volume_size=123,
                        volume_type="volumeType"
                    ),
                    no_device=ec2.CfnInstance.NoDeviceProperty(),
                    virtual_name="virtualName"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "device_name": device_name,
            }
            if ebs is not None:
                self._values["ebs"] = ebs
            if no_device is not None:
                self._values["no_device"] = no_device
            if virtual_name is not None:
                self._values["virtual_name"] = virtual_name

        @builtins.property
        def device_name(self) -> builtins.str:
            '''The device name (for example, ``/dev/sdh`` or ``xvdh`` ).

            .. epigraph::

               After the instance is running, this parameter is used to specify the device name of the block device mapping to update.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-mapping.html#cfn-ec2-blockdev-mapping-devicename
            '''
            result = self._values.get("device_name")
            assert result is not None, "Required property 'device_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def ebs(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.EbsProperty"]]:
            '''Parameters used to automatically set up EBS volumes when the instance is launched.

            .. epigraph::

               After the instance is running, you can modify only the ``DeleteOnTermination`` parameter for the attached volumes without interrupting the instance. Modifying any other parameter results in instance `replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-no-interrupt>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-mapping.html#cfn-ec2-blockdev-mapping-ebs
            '''
            result = self._values.get("ebs")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.EbsProperty"]], result)

        @builtins.property
        def no_device(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.NoDeviceProperty"]]:
            '''To omit the device from the block device mapping, specify an empty string.

            .. epigraph::

               After the instance is running, modifying this parameter results in instance `replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-mapping.html#cfn-ec2-blockdev-mapping-nodevice
            '''
            result = self._values.get("no_device")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.NoDeviceProperty"]], result)

        @builtins.property
        def virtual_name(self) -> typing.Optional[builtins.str]:
            '''The virtual device name ( ``ephemeral`` N).

            The name must be in the form ``ephemeral`` *X* where *X* is a number starting from zero (0). For example, an instance type with 2 available instance store volumes can specify mappings for ``ephemeral0`` and ``ephemeral1`` . The number of available instance store volumes depends on the instance type. After you connect to the instance, you must mount the volume.

            NVMe instance store volumes are automatically enumerated and assigned a device name. Including them in your block device mapping has no effect.

            *Constraints* : For M3 instances, you must specify instance store volumes in the block device mapping for the instance. When you launch an M3 instance, we ignore any instance store volumes specified in the block device mapping for the AMI.
            .. epigraph::

               After the instance is running, modifying this parameter results in instance `replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-mapping.html#cfn-ec2-blockdev-mapping-virtualname
            '''
            result = self._values.get("virtual_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BlockDeviceMappingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.CpuOptionsProperty",
        jsii_struct_bases=[],
        name_mapping={"core_count": "coreCount", "threads_per_core": "threadsPerCore"},
    )
    class CpuOptionsProperty:
        def __init__(
            self,
            *,
            core_count: typing.Optional[jsii.Number] = None,
            threads_per_core: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''Specifies the CPU options for the instance.

            When you specify CPU options, you must specify both the number of CPU cores and threads per core.

            For more information, see `Optimize CPU options <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-optimize-cpu.html>`_ in the *Amazon Elastic Compute Cloud User Guide* .

            :param core_count: The number of CPU cores for the instance.
            :param threads_per_core: The number of threads per CPU core.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-cpuoptions.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                cpu_options_property = ec2.CfnInstance.CpuOptionsProperty(
                    core_count=123,
                    threads_per_core=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if core_count is not None:
                self._values["core_count"] = core_count
            if threads_per_core is not None:
                self._values["threads_per_core"] = threads_per_core

        @builtins.property
        def core_count(self) -> typing.Optional[jsii.Number]:
            '''The number of CPU cores for the instance.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-cpuoptions.html#cfn-ec2-instance-cpuoptions-corecount
            '''
            result = self._values.get("core_count")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def threads_per_core(self) -> typing.Optional[jsii.Number]:
            '''The number of threads per CPU core.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-cpuoptions.html#cfn-ec2-instance-cpuoptions-threadspercore
            '''
            result = self._values.get("threads_per_core")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CpuOptionsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.CreditSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"cpu_credits": "cpuCredits"},
    )
    class CreditSpecificationProperty:
        def __init__(
            self,
            *,
            cpu_credits: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Specifies the credit option for CPU usage of a T2, T3, or T3a instance.

            ``CreditSpecification`` is a property of the `AWS::EC2::Instance <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html>`_ resource.

            :param cpu_credits: The credit option for CPU usage of the instance. Valid values are ``standard`` and ``unlimited`` . ``T3`` instances launch as ``unlimited`` by default. ``T2`` instances launch as ``standard`` by default.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-creditspecification.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                credit_specification_property = ec2.CfnInstance.CreditSpecificationProperty(
                    cpu_credits="cpuCredits"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if cpu_credits is not None:
                self._values["cpu_credits"] = cpu_credits

        @builtins.property
        def cpu_credits(self) -> typing.Optional[builtins.str]:
            '''The credit option for CPU usage of the instance.

            Valid values are ``standard`` and ``unlimited`` . ``T3`` instances launch as ``unlimited`` by default. ``T2`` instances launch as ``standard`` by default.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-creditspecification.html#cfn-ec2-instance-creditspecification-cpucredits
            '''
            result = self._values.get("cpu_credits")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CreditSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.EbsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "delete_on_termination": "deleteOnTermination",
            "encrypted": "encrypted",
            "iops": "iops",
            "kms_key_id": "kmsKeyId",
            "snapshot_id": "snapshotId",
            "volume_size": "volumeSize",
            "volume_type": "volumeType",
        },
    )
    class EbsProperty:
        def __init__(
            self,
            *,
            delete_on_termination: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            encrypted: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            iops: typing.Optional[jsii.Number] = None,
            kms_key_id: typing.Optional[builtins.str] = None,
            snapshot_id: typing.Optional[builtins.str] = None,
            volume_size: typing.Optional[jsii.Number] = None,
            volume_type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Specifies a block device for an EBS volume.

            ``Ebs`` is a property of the `Amazon EC2 BlockDeviceMapping <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-mapping.html>`_ property.
            .. epigraph::

               After the instance is running, you can modify only the ``DeleteOnTermination`` parameters for the attached volumes without interrupting the instance. Modifying any other parameter results in instance `replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ .

            :param delete_on_termination: Indicates whether the EBS volume is deleted on instance termination. For more information, see `Preserving Amazon EBS volumes on instance termination <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/terminating-instances.html#preserving-volumes-on-termination>`_ in the *Amazon EC2 User Guide* .
            :param encrypted: Indicates whether the volume should be encrypted. The effect of setting the encryption state to ``true`` depends on the volume origin (new or from a snapshot), starting encryption state, ownership, and whether encryption by default is enabled. For more information, see `Encryption by default <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#encryption-by-default>`_ in the *Amazon Elastic Compute Cloud User Guide* . Encrypted Amazon EBS volumes must be attached to instances that support Amazon EBS encryption. For more information, see `Supported instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#EBSEncryption_supported_instances>`_ . .. epigraph:: After the instance is running, modifying this parameter results in instance `replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ .
            :param iops: The number of I/O operations per second (IOPS). For ``gp3`` , ``io1`` , and ``io2`` volumes, this represents the number of IOPS that are provisioned for the volume. For ``gp2`` volumes, this represents the baseline performance of the volume and the rate at which the volume accumulates I/O credits for bursting. The following are the supported values for each volume type: - ``gp3`` : 3,000-16,000 IOPS - ``io1`` : 100-64,000 IOPS - ``io2`` : 100-64,000 IOPS For ``io1`` and ``io2`` volumes, we guarantee 64,000 IOPS only for `Instances built on the Nitro System <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances>`_ . Other instance families guarantee performance up to 32,000 IOPS. This parameter is required for ``io1`` and ``io2`` volumes. The default for ``gp3`` volumes is 3,000 IOPS. This parameter is not supported for ``gp2`` , ``st1`` , ``sc1`` , or ``standard`` volumes. .. epigraph:: After the instance is running, modifying this parameter results in instance `replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ .
            :param kms_key_id: The identifier of the AWS KMS key to use for Amazon EBS encryption. If ``KmsKeyId`` is specified, the encrypted state must be ``true`` . If the encrypted state is ``true`` but you do not specify ``KmsKeyId`` , your KMS key for EBS is used. You can specify the KMS key using any of the following: - Key ID. For example, 1234abcd-12ab-34cd-56ef-1234567890ab. - Key alias. For example, alias/ExampleAlias. - Key ARN. For example, arn:aws:kms:us-east-1:012345678910:1234abcd-12ab-34cd-56ef-1234567890ab. - Alias ARN. For example, arn:aws:kms:us-east-1:012345678910:alias/ExampleAlias. .. epigraph:: After the instance is running, modifying this parameter results in instance `replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ .
            :param snapshot_id: The ID of the snapshot. If you specify both ``SnapshotId`` and ``VolumeSize`` , ``VolumeSize`` must be equal or greater than the size of the snapshot. .. epigraph:: After the instance is running, modifying this parameter results in instance `replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ .
            :param volume_size: The size of the volume, in GiBs. You must specify either a snapshot ID or a volume size. If you specify a snapshot, the default is the snapshot size. You can specify a volume size that is equal to or larger than the snapshot size. The following are the supported volumes sizes for each volume type: - ``gp2`` and ``gp3`` :1-16,384 - ``io1`` and ``io2`` : 4-16,384 - ``st1`` and ``sc1`` : 125-16,384 - ``standard`` : 1-1,024 .. epigraph:: After the instance is running, modifying this parameter results in instance `replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ .
            :param volume_type: The volume type. For more information, see `Amazon EBS volume types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html>`_ in the *Amazon EC2 User Guide* . If the volume type is ``io1`` or ``io2`` , you must specify the IOPS that the volume supports. .. epigraph:: After the instance is running, modifying this parameter results in instance `replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-template.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                ebs_property = ec2.CfnInstance.EbsProperty(
                    delete_on_termination=False,
                    encrypted=False,
                    iops=123,
                    kms_key_id="kmsKeyId",
                    snapshot_id="snapshotId",
                    volume_size=123,
                    volume_type="volumeType"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if delete_on_termination is not None:
                self._values["delete_on_termination"] = delete_on_termination
            if encrypted is not None:
                self._values["encrypted"] = encrypted
            if iops is not None:
                self._values["iops"] = iops
            if kms_key_id is not None:
                self._values["kms_key_id"] = kms_key_id
            if snapshot_id is not None:
                self._values["snapshot_id"] = snapshot_id
            if volume_size is not None:
                self._values["volume_size"] = volume_size
            if volume_type is not None:
                self._values["volume_type"] = volume_type

        @builtins.property
        def delete_on_termination(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''Indicates whether the EBS volume is deleted on instance termination.

            For more information, see `Preserving Amazon EBS volumes on instance termination <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/terminating-instances.html#preserving-volumes-on-termination>`_ in the *Amazon EC2 User Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-template.html#cfn-ec2-blockdev-template-deleteontermination
            '''
            result = self._values.get("delete_on_termination")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def encrypted(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''Indicates whether the volume should be encrypted.

            The effect of setting the encryption state to ``true`` depends on the volume origin (new or from a snapshot), starting encryption state, ownership, and whether encryption by default is enabled. For more information, see `Encryption by default <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#encryption-by-default>`_ in the *Amazon Elastic Compute Cloud User Guide* .

            Encrypted Amazon EBS volumes must be attached to instances that support Amazon EBS encryption. For more information, see `Supported instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#EBSEncryption_supported_instances>`_ .
            .. epigraph::

               After the instance is running, modifying this parameter results in instance `replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-template.html#cfn-ec2-blockdev-template-encrypted
            '''
            result = self._values.get("encrypted")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def iops(self) -> typing.Optional[jsii.Number]:
            '''The number of I/O operations per second (IOPS).

            For ``gp3`` , ``io1`` , and ``io2`` volumes, this represents the number of IOPS that are provisioned for the volume. For ``gp2`` volumes, this represents the baseline performance of the volume and the rate at which the volume accumulates I/O credits for bursting.

            The following are the supported values for each volume type:

            - ``gp3`` : 3,000-16,000 IOPS
            - ``io1`` : 100-64,000 IOPS
            - ``io2`` : 100-64,000 IOPS

            For ``io1`` and ``io2`` volumes, we guarantee 64,000 IOPS only for `Instances built on the Nitro System <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances>`_ . Other instance families guarantee performance up to 32,000 IOPS.

            This parameter is required for ``io1`` and ``io2`` volumes. The default for ``gp3`` volumes is 3,000 IOPS. This parameter is not supported for ``gp2`` , ``st1`` , ``sc1`` , or ``standard`` volumes.
            .. epigraph::

               After the instance is running, modifying this parameter results in instance `replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-template.html#cfn-ec2-blockdev-template-iops
            '''
            result = self._values.get("iops")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def kms_key_id(self) -> typing.Optional[builtins.str]:
            '''The identifier of the AWS KMS key to use for Amazon EBS encryption.

            If ``KmsKeyId`` is specified, the encrypted state must be ``true`` . If the encrypted state is ``true`` but you do not specify ``KmsKeyId`` , your KMS key for EBS is used.

            You can specify the KMS key using any of the following:

            - Key ID. For example, 1234abcd-12ab-34cd-56ef-1234567890ab.
            - Key alias. For example, alias/ExampleAlias.
            - Key ARN. For example, arn:aws:kms:us-east-1:012345678910:1234abcd-12ab-34cd-56ef-1234567890ab.
            - Alias ARN. For example, arn:aws:kms:us-east-1:012345678910:alias/ExampleAlias.

            .. epigraph::

               After the instance is running, modifying this parameter results in instance `replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-template.html#cfn-ec2-instance-ebs-kmskeyid
            '''
            result = self._values.get("kms_key_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def snapshot_id(self) -> typing.Optional[builtins.str]:
            '''The ID of the snapshot.

            If you specify both ``SnapshotId`` and ``VolumeSize`` , ``VolumeSize`` must be equal or greater than the size of the snapshot.
            .. epigraph::

               After the instance is running, modifying this parameter results in instance `replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-template.html#cfn-ec2-blockdev-template-snapshotid
            '''
            result = self._values.get("snapshot_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def volume_size(self) -> typing.Optional[jsii.Number]:
            '''The size of the volume, in GiBs.

            You must specify either a snapshot ID or a volume size. If you specify a snapshot, the default is the snapshot size. You can specify a volume size that is equal to or larger than the snapshot size.

            The following are the supported volumes sizes for each volume type:

            - ``gp2`` and ``gp3`` :1-16,384
            - ``io1`` and ``io2`` : 4-16,384
            - ``st1`` and ``sc1`` : 125-16,384
            - ``standard`` : 1-1,024

            .. epigraph::

               After the instance is running, modifying this parameter results in instance `replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-template.html#cfn-ec2-blockdev-template-volumesize
            '''
            result = self._values.get("volume_size")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def volume_type(self) -> typing.Optional[builtins.str]:
            '''The volume type.

            For more information, see `Amazon EBS volume types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html>`_ in the *Amazon EC2 User Guide* . If the volume type is ``io1`` or ``io2`` , you must specify the IOPS that the volume supports.
            .. epigraph::

               After the instance is running, modifying this parameter results in instance `replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-template.html#cfn-ec2-blockdev-template-volumetype
            '''
            result = self._values.get("volume_type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EbsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.ElasticGpuSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type"},
    )
    class ElasticGpuSpecificationProperty:
        def __init__(self, *, type: builtins.str) -> None:
            '''Specifies the type of Elastic GPU.

            An Elastic GPU is a GPU resource that you can attach to your Amazon EC2 instance to accelerate the graphics performance of your applications. For more information, see `Amazon EC2 Elastic GPUs <https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/elastic-graphics.html>`_ in the *Amazon EC2 User Guide for Windows Instances* .

            ``ElasticGpuSpecification`` is a property of the `AWS::EC2::Instance <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html>`_ resource.

            :param type: The type of Elastic Graphics accelerator. For more information about the values to specify for ``Type`` , see `Elastic Graphics Basics <https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/elastic-graphics.html#elastic-graphics-basics>`_ , specifically the Elastic Graphics accelerator column, in the *Amazon Elastic Compute Cloud User Guide for Windows Instances* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-elasticgpuspecification.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                elastic_gpu_specification_property = ec2.CfnInstance.ElasticGpuSpecificationProperty(
                    type="type"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "type": type,
            }

        @builtins.property
        def type(self) -> builtins.str:
            '''The type of Elastic Graphics accelerator.

            For more information about the values to specify for ``Type`` , see `Elastic Graphics Basics <https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/elastic-graphics.html#elastic-graphics-basics>`_ , specifically the Elastic Graphics accelerator column, in the *Amazon Elastic Compute Cloud User Guide for Windows Instances* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-elasticgpuspecification.html#cfn-ec2-instance-elasticgpuspecification-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ElasticGpuSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.ElasticInferenceAcceleratorProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type", "count": "count"},
    )
    class ElasticInferenceAcceleratorProperty:
        def __init__(
            self,
            *,
            type: builtins.str,
            count: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''Specifies the Elastic Inference Accelerator for the instance.

            ``ElasticInferenceAccelerator`` is a property of the `AWS::EC2::Instance <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html>`_ resource.

            :param type: The type of elastic inference accelerator. The possible values are ``eia1.medium`` , ``eia1.large`` , ``eia1.xlarge`` , ``eia2.medium`` , ``eia2.large`` , and ``eia2.xlarge`` .
            :param count: The number of elastic inference accelerators to attach to the instance.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-elasticinferenceaccelerator.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                elastic_inference_accelerator_property = ec2.CfnInstance.ElasticInferenceAcceleratorProperty(
                    type="type",
                
                    # the properties below are optional
                    count=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "type": type,
            }
            if count is not None:
                self._values["count"] = count

        @builtins.property
        def type(self) -> builtins.str:
            '''The type of elastic inference accelerator.

            The possible values are ``eia1.medium`` , ``eia1.large`` , ``eia1.xlarge`` , ``eia2.medium`` , ``eia2.large`` , and ``eia2.xlarge`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-elasticinferenceaccelerator.html#cfn-ec2-instance-elasticinferenceaccelerator-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def count(self) -> typing.Optional[jsii.Number]:
            '''The number of elastic inference accelerators to attach to the instance.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-elasticinferenceaccelerator.html#cfn-ec2-instance-elasticinferenceaccelerator-count
            '''
            result = self._values.get("count")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ElasticInferenceAcceleratorProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.EnclaveOptionsProperty",
        jsii_struct_bases=[],
        name_mapping={"enabled": "enabled"},
    )
    class EnclaveOptionsProperty:
        def __init__(
            self,
            *,
            enabled: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        ) -> None:
            '''Indicates whether the instance is enabled for AWS Nitro Enclaves.

            :param enabled: If this parameter is set to ``true`` , the instance is enabled for AWS Nitro Enclaves; otherwise, it is not enabled for AWS Nitro Enclaves.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-enclaveoptions.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                enclave_options_property = ec2.CfnInstance.EnclaveOptionsProperty(
                    enabled=False
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if enabled is not None:
                self._values["enabled"] = enabled

        @builtins.property
        def enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''If this parameter is set to ``true`` , the instance is enabled for AWS Nitro Enclaves;

            otherwise, it is not enabled for AWS Nitro Enclaves.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-enclaveoptions.html#cfn-ec2-instance-enclaveoptions-enabled
            '''
            result = self._values.get("enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EnclaveOptionsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.HibernationOptionsProperty",
        jsii_struct_bases=[],
        name_mapping={"configured": "configured"},
    )
    class HibernationOptionsProperty:
        def __init__(
            self,
            *,
            configured: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        ) -> None:
            '''Specifies the hibernation options for the instance.

            ``HibernationOptions`` is a property of the `AWS::EC2::Instance <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html>`_ resource.

            :param configured: If you set this parameter to ``true`` , your instance is enabled for hibernation. Default: ``false``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-hibernationoptions.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                hibernation_options_property = ec2.CfnInstance.HibernationOptionsProperty(
                    configured=False
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if configured is not None:
                self._values["configured"] = configured

        @builtins.property
        def configured(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''If you set this parameter to ``true`` , your instance is enabled for hibernation.

            Default: ``false``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-hibernationoptions.html#cfn-ec2-instance-hibernationoptions-configured
            '''
            result = self._values.get("configured")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HibernationOptionsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.InstanceIpv6AddressProperty",
        jsii_struct_bases=[],
        name_mapping={"ipv6_address": "ipv6Address"},
    )
    class InstanceIpv6AddressProperty:
        def __init__(self, *, ipv6_address: builtins.str) -> None:
            '''Specifies the IPv6 address for the instance.

            ``InstanceIpv6Address`` is a property of the `AWS::EC2::Instance <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html>`_ resource.

            :param ipv6_address: The IPv6 address.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-instanceipv6address.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                instance_ipv6_address_property = ec2.CfnInstance.InstanceIpv6AddressProperty(
                    ipv6_address="ipv6Address"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "ipv6_address": ipv6_address,
            }

        @builtins.property
        def ipv6_address(self) -> builtins.str:
            '''The IPv6 address.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-instanceipv6address.html#cfn-ec2-instance-instanceipv6address-ipv6address
            '''
            result = self._values.get("ipv6_address")
            assert result is not None, "Required property 'ipv6_address' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstanceIpv6AddressProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.LaunchTemplateSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "version": "version",
            "launch_template_id": "launchTemplateId",
            "launch_template_name": "launchTemplateName",
        },
    )
    class LaunchTemplateSpecificationProperty:
        def __init__(
            self,
            *,
            version: builtins.str,
            launch_template_id: typing.Optional[builtins.str] = None,
            launch_template_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Specifies a launch template. You must specify either the launch template ID or launch template name.

            ``LaunchTemplateSpecification`` is a property of the `AWS::EC2::Instance <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html>`_ resource.

            :param version: The version number of the launch template. AWS CloudFormation does not support specifying ``$Latest`` , or ``$Default`` for the template version number.
            :param launch_template_id: The ID of the launch template.
            :param launch_template_name: The name of the launch template.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-launchtemplatespecification.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                launch_template_specification_property = ec2.CfnInstance.LaunchTemplateSpecificationProperty(
                    version="version",
                
                    # the properties below are optional
                    launch_template_id="launchTemplateId",
                    launch_template_name="launchTemplateName"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "version": version,
            }
            if launch_template_id is not None:
                self._values["launch_template_id"] = launch_template_id
            if launch_template_name is not None:
                self._values["launch_template_name"] = launch_template_name

        @builtins.property
        def version(self) -> builtins.str:
            '''The version number of the launch template.

            AWS CloudFormation does not support specifying ``$Latest`` , or ``$Default`` for the template version number.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-launchtemplatespecification.html#cfn-ec2-instance-launchtemplatespecification-version
            '''
            result = self._values.get("version")
            assert result is not None, "Required property 'version' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def launch_template_id(self) -> typing.Optional[builtins.str]:
            '''The ID of the launch template.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-launchtemplatespecification.html#cfn-ec2-instance-launchtemplatespecification-launchtemplateid
            '''
            result = self._values.get("launch_template_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def launch_template_name(self) -> typing.Optional[builtins.str]:
            '''The name of the launch template.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-launchtemplatespecification.html#cfn-ec2-instance-launchtemplatespecification-launchtemplatename
            '''
            result = self._values.get("launch_template_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LaunchTemplateSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.LicenseSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"license_configuration_arn": "licenseConfigurationArn"},
    )
    class LicenseSpecificationProperty:
        def __init__(self, *, license_configuration_arn: builtins.str) -> None:
            '''Specifies the license configuration to use.

            ``LicenseSpecification`` is a property of the `AWS::EC2::Instance <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html>`_ resource.

            :param license_configuration_arn: The Amazon Resource Name (ARN) of the license configuration.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-licensespecification.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                license_specification_property = ec2.CfnInstance.LicenseSpecificationProperty(
                    license_configuration_arn="licenseConfigurationArn"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "license_configuration_arn": license_configuration_arn,
            }

        @builtins.property
        def license_configuration_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the license configuration.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-licensespecification.html#cfn-ec2-instance-licensespecification-licenseconfigurationarn
            '''
            result = self._values.get("license_configuration_arn")
            assert result is not None, "Required property 'license_configuration_arn' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LicenseSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.NetworkInterfaceProperty",
        jsii_struct_bases=[],
        name_mapping={
            "device_index": "deviceIndex",
            "associate_public_ip_address": "associatePublicIpAddress",
            "delete_on_termination": "deleteOnTermination",
            "description": "description",
            "group_set": "groupSet",
            "ipv6_address_count": "ipv6AddressCount",
            "ipv6_addresses": "ipv6Addresses",
            "network_interface_id": "networkInterfaceId",
            "private_ip_address": "privateIpAddress",
            "private_ip_addresses": "privateIpAddresses",
            "secondary_private_ip_address_count": "secondaryPrivateIpAddressCount",
            "subnet_id": "subnetId",
        },
    )
    class NetworkInterfaceProperty:
        def __init__(
            self,
            *,
            device_index: builtins.str,
            associate_public_ip_address: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            delete_on_termination: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            description: typing.Optional[builtins.str] = None,
            group_set: typing.Optional[typing.Sequence[builtins.str]] = None,
            ipv6_address_count: typing.Optional[jsii.Number] = None,
            ipv6_addresses: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.InstanceIpv6AddressProperty"]]]] = None,
            network_interface_id: typing.Optional[builtins.str] = None,
            private_ip_address: typing.Optional[builtins.str] = None,
            private_ip_addresses: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.PrivateIpAddressSpecificationProperty"]]]] = None,
            secondary_private_ip_address_count: typing.Optional[jsii.Number] = None,
            subnet_id: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Specifies a network interface that is to be attached to an instance.

            You can create a network interface when launching an instance. For an example, see the `AWS::EC2::Instance examples <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#aws-properties-ec2-instance--examples--Automatically_assign_a_public_IP_address>`_ .

            Alternatively, you can attach an existing network interface when launching an instance. For an example, see the `AWS::EC2:NetworkInterface examples <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html#aws-resource-ec2-network-interface--examples--Basic_network_interface>`_ .

            :param device_index: The position of the network interface in the attachment order. A primary network interface has a device index of 0. If you create a network interface when launching an instance, you must specify the device index.
            :param associate_public_ip_address: Indicates whether to assign a public IPv4 address to an instance. Applies only if creating a network interface when launching an instance. The network interface must be the primary network interface. If launching into a default subnet, the default value is ``true`` .
            :param delete_on_termination: Indicates whether the network interface is deleted when the instance is terminated. Applies only if creating a network interface when launching an instance.
            :param description: The description of the network interface. Applies only if creating a network interface when launching an instance.
            :param group_set: The IDs of the security groups for the network interface. Applies only if creating a network interface when launching an instance.
            :param ipv6_address_count: A number of IPv6 addresses to assign to the network interface. Amazon EC2 chooses the IPv6 addresses from the range of the subnet. You cannot specify this option and the option to assign specific IPv6 addresses in the same request. You can specify this option if you've specified a minimum number of instances to launch.
            :param ipv6_addresses: One or more IPv6 addresses to assign to the network interface. You cannot specify this option and the option to assign a number of IPv6 addresses in the same request. You cannot specify this option if you've specified a minimum number of instances to launch.
            :param network_interface_id: The ID of the network interface, when attaching an existing network interface.
            :param private_ip_address: The private IPv4 address of the network interface. Applies only if creating a network interface when launching an instance.
            :param private_ip_addresses: One or more private IPv4 addresses to assign to the network interface. Only one private IPv4 address can be designated as primary.
            :param secondary_private_ip_address_count: The number of secondary private IPv4 addresses. You can't specify this option and specify more than one private IP address using the private IP addresses option.
            :param subnet_id: The ID of the subnet associated with the network interface. Applies only if creating a network interface when launching an instance.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                network_interface_property = ec2.CfnInstance.NetworkInterfaceProperty(
                    device_index="deviceIndex",
                
                    # the properties below are optional
                    associate_public_ip_address=False,
                    delete_on_termination=False,
                    description="description",
                    group_set=["groupSet"],
                    ipv6_address_count=123,
                    ipv6_addresses=[ec2.CfnInstance.InstanceIpv6AddressProperty(
                        ipv6_address="ipv6Address"
                    )],
                    network_interface_id="networkInterfaceId",
                    private_ip_address="privateIpAddress",
                    private_ip_addresses=[ec2.CfnInstance.PrivateIpAddressSpecificationProperty(
                        primary=False,
                        private_ip_address="privateIpAddress"
                    )],
                    secondary_private_ip_address_count=123,
                    subnet_id="subnetId"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "device_index": device_index,
            }
            if associate_public_ip_address is not None:
                self._values["associate_public_ip_address"] = associate_public_ip_address
            if delete_on_termination is not None:
                self._values["delete_on_termination"] = delete_on_termination
            if description is not None:
                self._values["description"] = description
            if group_set is not None:
                self._values["group_set"] = group_set
            if ipv6_address_count is not None:
                self._values["ipv6_address_count"] = ipv6_address_count
            if ipv6_addresses is not None:
                self._values["ipv6_addresses"] = ipv6_addresses
            if network_interface_id is not None:
                self._values["network_interface_id"] = network_interface_id
            if private_ip_address is not None:
                self._values["private_ip_address"] = private_ip_address
            if private_ip_addresses is not None:
                self._values["private_ip_addresses"] = private_ip_addresses
            if secondary_private_ip_address_count is not None:
                self._values["secondary_private_ip_address_count"] = secondary_private_ip_address_count
            if subnet_id is not None:
                self._values["subnet_id"] = subnet_id

        @builtins.property
        def device_index(self) -> builtins.str:
            '''The position of the network interface in the attachment order.

            A primary network interface has a device index of 0.

            If you create a network interface when launching an instance, you must specify the device index.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html#aws-properties-ec2-network-iface-embedded-deviceindex
            '''
            result = self._values.get("device_index")
            assert result is not None, "Required property 'device_index' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def associate_public_ip_address(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''Indicates whether to assign a public IPv4 address to an instance.

            Applies only if creating a network interface when launching an instance. The network interface must be the primary network interface. If launching into a default subnet, the default value is ``true`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html#aws-properties-ec2-network-iface-embedded-associatepubip
            '''
            result = self._values.get("associate_public_ip_address")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def delete_on_termination(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''Indicates whether the network interface is deleted when the instance is terminated.

            Applies only if creating a network interface when launching an instance.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html#aws-properties-ec2-network-iface-embedded-delete
            '''
            result = self._values.get("delete_on_termination")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def description(self) -> typing.Optional[builtins.str]:
            '''The description of the network interface.

            Applies only if creating a network interface when launching an instance.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html#aws-properties-ec2-network-iface-embedded-description
            '''
            result = self._values.get("description")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def group_set(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The IDs of the security groups for the network interface.

            Applies only if creating a network interface when launching an instance.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html#aws-properties-ec2-network-iface-embedded-groupset
            '''
            result = self._values.get("group_set")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def ipv6_address_count(self) -> typing.Optional[jsii.Number]:
            '''A number of IPv6 addresses to assign to the network interface.

            Amazon EC2 chooses the IPv6 addresses from the range of the subnet. You cannot specify this option and the option to assign specific IPv6 addresses in the same request. You can specify this option if you've specified a minimum number of instances to launch.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html#cfn-ec2-instance-networkinterface-ipv6addresscount
            '''
            result = self._values.get("ipv6_address_count")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def ipv6_addresses(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.InstanceIpv6AddressProperty"]]]]:
            '''One or more IPv6 addresses to assign to the network interface.

            You cannot specify this option and the option to assign a number of IPv6 addresses in the same request. You cannot specify this option if you've specified a minimum number of instances to launch.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html#cfn-ec2-instance-networkinterface-ipv6addresses
            '''
            result = self._values.get("ipv6_addresses")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.InstanceIpv6AddressProperty"]]]], result)

        @builtins.property
        def network_interface_id(self) -> typing.Optional[builtins.str]:
            '''The ID of the network interface, when attaching an existing network interface.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html#aws-properties-ec2-network-iface-embedded-network-iface
            '''
            result = self._values.get("network_interface_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def private_ip_address(self) -> typing.Optional[builtins.str]:
            '''The private IPv4 address of the network interface.

            Applies only if creating a network interface when launching an instance.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html#aws-properties-ec2-network-iface-embedded-privateipaddress
            '''
            result = self._values.get("private_ip_address")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def private_ip_addresses(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.PrivateIpAddressSpecificationProperty"]]]]:
            '''One or more private IPv4 addresses to assign to the network interface.

            Only one private IPv4 address can be designated as primary.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html#aws-properties-ec2-network-iface-embedded-privateipaddresses
            '''
            result = self._values.get("private_ip_addresses")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.PrivateIpAddressSpecificationProperty"]]]], result)

        @builtins.property
        def secondary_private_ip_address_count(self) -> typing.Optional[jsii.Number]:
            '''The number of secondary private IPv4 addresses.

            You can't specify this option and specify more than one private IP address using the private IP addresses option.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html#aws-properties-ec2-network-iface-embedded-secondprivateip
            '''
            result = self._values.get("secondary_private_ip_address_count")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def subnet_id(self) -> typing.Optional[builtins.str]:
            '''The ID of the subnet associated with the network interface.

            Applies only if creating a network interface when launching an instance.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html#aws-properties-ec2-network-iface-embedded-subnetid
            '''
            result = self._values.get("subnet_id")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "NetworkInterfaceProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.NoDeviceProperty",
        jsii_struct_bases=[],
        name_mapping={},
    )
    class NoDeviceProperty:
        def __init__(self) -> None:
            '''Suppresses the specified device included in the block device mapping of the AMI.

            To suppress a device, specify an empty string.

            ``NoDevice`` is a property of the `Amazon EC2 BlockDeviceMapping <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-mapping.html>`_ property.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-nodevice.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                no_device_property = ec2.CfnInstance.NoDeviceProperty()
            '''
            self._values: typing.Dict[str, typing.Any] = {}

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "NoDeviceProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.PrivateIpAddressSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"primary": "primary", "private_ip_address": "privateIpAddress"},
    )
    class PrivateIpAddressSpecificationProperty:
        def __init__(
            self,
            *,
            primary: typing.Union[builtins.bool, aws_cdk.core.IResolvable],
            private_ip_address: builtins.str,
        ) -> None:
            '''Specifies a secondary private IPv4 address for a network interface.

            ``PrivateIpAddressSpecification`` is a property of the `AWS::EC2::NetworkInterface <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html>`_ resource.

            :param primary: Indicates whether the private IPv4 address is the primary private IPv4 address. Only one IPv4 address can be designated as primary.
            :param private_ip_address: The private IPv4 addresses.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-interface-privateipspec.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                private_ip_address_specification_property = ec2.CfnInstance.PrivateIpAddressSpecificationProperty(
                    primary=False,
                    private_ip_address="privateIpAddress"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "primary": primary,
                "private_ip_address": private_ip_address,
            }

        @builtins.property
        def primary(self) -> typing.Union[builtins.bool, aws_cdk.core.IResolvable]:
            '''Indicates whether the private IPv4 address is the primary private IPv4 address.

            Only one IPv4 address can be designated as primary.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-interface-privateipspec.html#cfn-ec2-networkinterface-privateipspecification-primary
            '''
            result = self._values.get("primary")
            assert result is not None, "Required property 'primary' is missing"
            return typing.cast(typing.Union[builtins.bool, aws_cdk.core.IResolvable], result)

        @builtins.property
        def private_ip_address(self) -> builtins.str:
            '''The private IPv4 addresses.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-interface-privateipspec.html#cfn-ec2-networkinterface-privateipspecification-privateipaddress
            '''
            result = self._values.get("private_ip_address")
            assert result is not None, "Required property 'private_ip_address' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PrivateIpAddressSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.SsmAssociationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "document_name": "documentName",
            "association_parameters": "associationParameters",
        },
    )
    class SsmAssociationProperty:
        def __init__(
            self,
            *,
            document_name: builtins.str,
            association_parameters: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.AssociationParameterProperty"]]]] = None,
        ) -> None:
            '''Specifies the SSM document and parameter values in AWS Systems Manager to associate with an instance.

            ``SsmAssociations`` is a property of the `AWS::EC2::Instance <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html>`_ resource.

            :param document_name: The name of an SSM document to associate with the instance.
            :param association_parameters: The input parameter values to use with the associated SSM document.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-ssmassociations.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                ssm_association_property = ec2.CfnInstance.SsmAssociationProperty(
                    document_name="documentName",
                
                    # the properties below are optional
                    association_parameters=[ec2.CfnInstance.AssociationParameterProperty(
                        key="key",
                        value=["value"]
                    )]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "document_name": document_name,
            }
            if association_parameters is not None:
                self._values["association_parameters"] = association_parameters

        @builtins.property
        def document_name(self) -> builtins.str:
            '''The name of an SSM document to associate with the instance.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-ssmassociations.html#cfn-ec2-instance-ssmassociations-documentname
            '''
            result = self._values.get("document_name")
            assert result is not None, "Required property 'document_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def association_parameters(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.AssociationParameterProperty"]]]]:
            '''The input parameter values to use with the associated SSM document.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-ssmassociations.html#cfn-ec2-instance-ssmassociations-associationparameters
            '''
            result = self._values.get("association_parameters")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.AssociationParameterProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SsmAssociationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.VolumeProperty",
        jsii_struct_bases=[],
        name_mapping={"device": "device", "volume_id": "volumeId"},
    )
    class VolumeProperty:
        def __init__(self, *, device: builtins.str, volume_id: builtins.str) -> None:
            '''Specifies a volume to attach to an instance.

            ``Volume`` is an embedded property of the `AWS::EC2::Instance <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html>`_ resource.

            :param device: The device name (for example, ``/dev/sdh`` or ``xvdh`` ).
            :param volume_id: The ID of the EBS volume. The volume and instance must be within the same Availability Zone.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-mount-point.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                volume_property = ec2.CfnInstance.VolumeProperty(
                    device="device",
                    volume_id="volumeId"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "device": device,
                "volume_id": volume_id,
            }

        @builtins.property
        def device(self) -> builtins.str:
            '''The device name (for example, ``/dev/sdh`` or ``xvdh`` ).

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-mount-point.html#cfn-ec2-mountpoint-device
            '''
            result = self._values.get("device")
            assert result is not None, "Required property 'device' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def volume_id(self) -> builtins.str:
            '''The ID of the EBS volume.

            The volume and instance must be within the same Availability Zone.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-mount-point.html#cfn-ec2-mountpoint-volumeid
            '''
            result = self._values.get("volume_id")
            assert result is not None, "Required property 'volume_id' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VolumeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnInstanceProps",
    jsii_struct_bases=[],
    name_mapping={
        "additional_info": "additionalInfo",
        "affinity": "affinity",
        "availability_zone": "availabilityZone",
        "block_device_mappings": "blockDeviceMappings",
        "cpu_options": "cpuOptions",
        "credit_specification": "creditSpecification",
        "disable_api_termination": "disableApiTermination",
        "ebs_optimized": "ebsOptimized",
        "elastic_gpu_specifications": "elasticGpuSpecifications",
        "elastic_inference_accelerators": "elasticInferenceAccelerators",
        "enclave_options": "enclaveOptions",
        "hibernation_options": "hibernationOptions",
        "host_id": "hostId",
        "host_resource_group_arn": "hostResourceGroupArn",
        "iam_instance_profile": "iamInstanceProfile",
        "image_id": "imageId",
        "instance_initiated_shutdown_behavior": "instanceInitiatedShutdownBehavior",
        "instance_type": "instanceType",
        "ipv6_address_count": "ipv6AddressCount",
        "ipv6_addresses": "ipv6Addresses",
        "kernel_id": "kernelId",
        "key_name": "keyName",
        "launch_template": "launchTemplate",
        "license_specifications": "licenseSpecifications",
        "monitoring": "monitoring",
        "network_interfaces": "networkInterfaces",
        "placement_group_name": "placementGroupName",
        "private_ip_address": "privateIpAddress",
        "propagate_tags_to_volume_on_creation": "propagateTagsToVolumeOnCreation",
        "ramdisk_id": "ramdiskId",
        "security_group_ids": "securityGroupIds",
        "security_groups": "securityGroups",
        "source_dest_check": "sourceDestCheck",
        "ssm_associations": "ssmAssociations",
        "subnet_id": "subnetId",
        "tags": "tags",
        "tenancy": "tenancy",
        "user_data": "userData",
        "volumes": "volumes",
    },
)
class CfnInstanceProps:
    def __init__(
        self,
        *,
        additional_info: typing.Optional[builtins.str] = None,
        affinity: typing.Optional[builtins.str] = None,
        availability_zone: typing.Optional[builtins.str] = None,
        block_device_mappings: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnInstance.BlockDeviceMappingProperty]]]] = None,
        cpu_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.CpuOptionsProperty]] = None,
        credit_specification: typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.CreditSpecificationProperty]] = None,
        disable_api_termination: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        ebs_optimized: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        elastic_gpu_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnInstance.ElasticGpuSpecificationProperty]]]] = None,
        elastic_inference_accelerators: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnInstance.ElasticInferenceAcceleratorProperty]]]] = None,
        enclave_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.EnclaveOptionsProperty]] = None,
        hibernation_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.HibernationOptionsProperty]] = None,
        host_id: typing.Optional[builtins.str] = None,
        host_resource_group_arn: typing.Optional[builtins.str] = None,
        iam_instance_profile: typing.Optional[builtins.str] = None,
        image_id: typing.Optional[builtins.str] = None,
        instance_initiated_shutdown_behavior: typing.Optional[builtins.str] = None,
        instance_type: typing.Optional[builtins.str] = None,
        ipv6_address_count: typing.Optional[jsii.Number] = None,
        ipv6_addresses: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnInstance.InstanceIpv6AddressProperty]]]] = None,
        kernel_id: typing.Optional[builtins.str] = None,
        key_name: typing.Optional[builtins.str] = None,
        launch_template: typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.LaunchTemplateSpecificationProperty]] = None,
        license_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnInstance.LicenseSpecificationProperty]]]] = None,
        monitoring: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        network_interfaces: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnInstance.NetworkInterfaceProperty]]]] = None,
        placement_group_name: typing.Optional[builtins.str] = None,
        private_ip_address: typing.Optional[builtins.str] = None,
        propagate_tags_to_volume_on_creation: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        ramdisk_id: typing.Optional[builtins.str] = None,
        security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
        security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
        source_dest_check: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        ssm_associations: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnInstance.SsmAssociationProperty]]]] = None,
        subnet_id: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
        tenancy: typing.Optional[builtins.str] = None,
        user_data: typing.Optional[builtins.str] = None,
        volumes: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnInstance.VolumeProperty]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnInstance``.

        :param additional_info: This property is reserved for internal use. If you use it, the stack fails with this error: ``Bad property set: [Testing this property] (Service: AmazonEC2; Status Code: 400; Error Code: InvalidParameterCombination; Request ID: 0XXXXXX-49c7-4b40-8bcc-76885dcXXXXX)`` .
        :param affinity: Indicates whether the instance is associated with a dedicated host. If you want the instance to always restart on the same host on which it was launched, specify ``host`` . If you want the instance to restart on any available host, but try to launch onto the last host it ran on (on a best-effort basis), specify ``default`` .
        :param availability_zone: The Availability Zone of the instance. If not specified, an Availability Zone will be automatically chosen for you based on the load balancing criteria for the Region. This parameter is not supported by `DescribeImageAttribute <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImageAttribute.html>`_ .
        :param block_device_mappings: The block device mapping entries that defines the block devices to attach to the instance at launch. By default, the block devices specified in the block device mapping for the AMI are used. You can override the AMI block device mapping using the instance block device mapping. For the root volume, you can override only the volume size, volume type, volume encryption settings, and the ``DeleteOnTermination`` setting. .. epigraph:: After the instance is running, you can modify only the ``DeleteOnTermination`` parameter for the attached volumes without interrupting the instance. Modifying any other parameter results in instance `replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ .
        :param cpu_options: The CPU options for the instance. For more information, see `Optimize CPU options <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-optimize-cpu.html>`_ in the *Amazon Elastic Compute Cloud User Guide* .
        :param credit_specification: The credit option for CPU usage of the burstable performance instance. Valid values are ``standard`` and ``unlimited`` . To change this attribute after launch, use `ModifyInstanceCreditSpecification <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceCreditSpecification.html>`_ . For more information, see `Burstable performance instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances.html>`_ in the *Amazon EC2 User Guide* . Default: ``standard`` (T2 instances) or ``unlimited`` (T3/T3a instances) For T3 instances with ``host`` tenancy, only ``standard`` is supported.
        :param disable_api_termination: If you set this parameter to ``true`` , you can't terminate the instance using the Amazon EC2 console, CLI, or API; otherwise, you can. To change this attribute after launch, use `ModifyInstanceAttribute <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceAttribute.html>`_ . Alternatively, if you set ``InstanceInitiatedShutdownBehavior`` to ``terminate`` , you can terminate the instance by running the shutdown command from the instance. Default: ``false``
        :param ebs_optimized: Indicates whether the instance is optimized for Amazon EBS I/O. This optimization provides dedicated throughput to Amazon EBS and an optimized configuration stack to provide optimal Amazon EBS I/O performance. This optimization isn't available with all instance types. Additional usage charges apply when using an EBS-optimized instance. Default: ``false``
        :param elastic_gpu_specifications: An elastic GPU to associate with the instance. An Elastic GPU is a GPU resource that you can attach to your Windows instance to accelerate the graphics performance of your applications. For more information, see `Amazon EC2 Elastic GPUs <https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/elastic-graphics.html>`_ in the *Amazon EC2 User Guide* .
        :param elastic_inference_accelerators: An elastic inference accelerator to associate with the instance. Elastic inference accelerators are a resource you can attach to your Amazon EC2 instances to accelerate your Deep Learning (DL) inference workloads. You cannot specify accelerators from different generations in the same request.
        :param enclave_options: Indicates whether the instance is enabled for AWS Nitro Enclaves.
        :param hibernation_options: Indicates whether an instance is enabled for hibernation. For more information, see `Hibernate your instance <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Hibernate.html>`_ in the *Amazon EC2 User Guide* . You can't enable hibernation and AWS Nitro Enclaves on the same instance.
        :param host_id: If you specify host for the ``Affinity`` property, the ID of a dedicated host that the instance is associated with. If you don't specify an ID, Amazon EC2 launches the instance onto any available, compatible dedicated host in your account. This type of launch is called an untargeted launch. Note that for untargeted launches, you must have a compatible, dedicated host available to successfully launch instances.
        :param host_resource_group_arn: The ARN of the host resource group in which to launch the instances. If you specify a host resource group ARN, omit the *Tenancy* parameter or set it to ``host`` .
        :param iam_instance_profile: The name of an IAM instance profile. To create a new IAM instance profile, use the `AWS::IAM::InstanceProfile <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-iam-instanceprofile.html>`_ resource.
        :param image_id: The ID of the AMI. An AMI ID is required to launch an instance and must be specified here or in a launch template.
        :param instance_initiated_shutdown_behavior: Indicates whether an instance stops or terminates when you initiate shutdown from the instance (using the operating system command for system shutdown). Default: ``stop``
        :param instance_type: The instance type. For more information, see `Instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html>`_ in the *Amazon EC2 User Guide* . Default: ``m1.small``
        :param ipv6_address_count: [EC2-VPC] The number of IPv6 addresses to associate with the primary network interface. Amazon EC2 chooses the IPv6 addresses from the range of your subnet. You cannot specify this option and the option to assign specific IPv6 addresses in the same request. You can specify this option if you've specified a minimum number of instances to launch. You cannot specify this option and the network interfaces option in the same request.
        :param ipv6_addresses: [EC2-VPC] The IPv6 addresses from the range of the subnet to associate with the primary network interface. You cannot specify this option and the option to assign a number of IPv6 addresses in the same request. You cannot specify this option if you've specified a minimum number of instances to launch. You cannot specify this option and the network interfaces option in the same request.
        :param kernel_id: The ID of the kernel. .. epigraph:: We recommend that you use PV-GRUB instead of kernels and RAM disks. For more information, see `PV-GRUB <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/UserProvidedkernels.html>`_ in the *Amazon EC2 User Guide* .
        :param key_name: The name of the key pair. You can create a key pair using `CreateKeyPair <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateKeyPair.html>`_ or `ImportKeyPair <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportKeyPair.html>`_ . .. epigraph:: If you do not specify a key pair, you can't connect to the instance unless you choose an AMI that is configured to allow users another way to log in.
        :param launch_template: The launch template to use to launch the instances. Any parameters that you specify in the AWS CloudFormation template override the same parameters in the launch template. You can specify either the name or ID of a launch template, but not both.
        :param license_specifications: The license configurations.
        :param monitoring: Specifies whether detailed monitoring is enabled for the instance.
        :param network_interfaces: The network interfaces to associate with the instance. .. epigraph:: If you use this property to point to a network interface, you must terminate the original interface before attaching a new one to allow the update of the instance to succeed. If this resource has a public IP address and is also in a VPC that is defined in the same template, you must use the `DependsOn Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html>`_ to declare a dependency on the VPC-gateway attachment.
        :param placement_group_name: The name of an existing placement group that you want to launch the instance into (cluster | partition | spread).
        :param private_ip_address: [EC2-VPC] The primary IPv4 address. You must specify a value from the IPv4 address range of the subnet. Only one private IP address can be designated as primary. You can't specify this option if you've specified the option to designate a private IP address as the primary IP address in a network interface specification. You cannot specify this option if you're launching more than one instance in the request. You cannot specify this option and the network interfaces option in the same request. If you make an update to an instance that requires replacement, you must assign a new private IP address. During a replacement, AWS CloudFormation creates a new instance but doesn't delete the old instance until the stack has successfully updated. If the stack update fails, AWS CloudFormation uses the old instance to roll back the stack to the previous working state. The old and new instances cannot have the same private IP address.
        :param propagate_tags_to_volume_on_creation: Whether to propagate the EC2 instance tags to the EBS volumes.
        :param ramdisk_id: The ID of the RAM disk to select. Some kernels require additional drivers at launch. Check the kernel requirements for information about whether you need to specify a RAM disk. To find kernel requirements, go to the AWS Resource Center and search for the kernel ID. .. epigraph:: We recommend that you use PV-GRUB instead of kernels and RAM disks. For more information, see `PV-GRUB <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/UserProvidedkernels.html>`_ in the *Amazon EC2 User Guide* .
        :param security_group_ids: The IDs of the security groups. You can create a security group using `CreateSecurityGroup <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSecurityGroup.html>`_ . If you specify a network interface, you must specify any security groups as part of the network interface.
        :param security_groups: [EC2-Classic, default VPC] The names of the security groups. For a nondefault VPC, you must use security group IDs instead. You cannot specify this option and the network interfaces option in the same request. The list can contain both the name of existing Amazon EC2 security groups or references to AWS::EC2::SecurityGroup resources created in the template. Default: Amazon EC2 uses the default security group.
        :param source_dest_check: Enable or disable source/destination checks, which ensure that the instance is either the source or the destination of any traffic that it receives. If the value is ``true`` , source/destination checks are enabled; otherwise, they are disabled. The default value is ``true`` . You must disable source/destination checks if the instance runs services such as network address translation, routing, or firewalls.
        :param ssm_associations: The SSM `document <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html>`_ and parameter values in AWS Systems Manager to associate with this instance. To use this property, you must specify an IAM instance profile role for the instance. For more information, see `Create an Instance Profile for Systems Manager <https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-configuring-access-role.html>`_ in the *AWS Systems Manager User Guide* . .. epigraph:: You can currently associate only one document with an instance.
        :param subnet_id: [EC2-VPC] The ID of the subnet to launch the instance into. If you specify a network interface, you must specify any subnets as part of the network interface.
        :param tags: The tags to add to the instance. These tags are not applied to the EBS volumes, such as the root volume.
        :param tenancy: The tenancy of the instance (if the instance is running in a VPC). An instance with a tenancy of ``dedicated`` runs on single-tenant hardware.
        :param user_data: The user data to make available to the instance. For more information, see `Run commands on your Linux instance at launch <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/user-data.html>`_ and `Run commands on your Windows instance at launch <https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ec2-windows-user-data.html>`_ . If you are using a command line tool, base64-encoding is performed for you, and you can load the text from a file. Otherwise, you must provide base64-encoded text. User data is limited to 16 KB.
        :param volumes: The volumes to attach to the instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            cfn_instance_props = ec2.CfnInstanceProps(
                additional_info="additionalInfo",
                affinity="affinity",
                availability_zone="availabilityZone",
                block_device_mappings=[ec2.CfnInstance.BlockDeviceMappingProperty(
                    device_name="deviceName",
            
                    # the properties below are optional
                    ebs=ec2.CfnInstance.EbsProperty(
                        delete_on_termination=False,
                        encrypted=False,
                        iops=123,
                        kms_key_id="kmsKeyId",
                        snapshot_id="snapshotId",
                        volume_size=123,
                        volume_type="volumeType"
                    ),
                    no_device=ec2.CfnInstance.NoDeviceProperty(),
                    virtual_name="virtualName"
                )],
                cpu_options=ec2.CfnInstance.CpuOptionsProperty(
                    core_count=123,
                    threads_per_core=123
                ),
                credit_specification=ec2.CfnInstance.CreditSpecificationProperty(
                    cpu_credits="cpuCredits"
                ),
                disable_api_termination=False,
                ebs_optimized=False,
                elastic_gpu_specifications=[ec2.CfnInstance.ElasticGpuSpecificationProperty(
                    type="type"
                )],
                elastic_inference_accelerators=[ec2.CfnInstance.ElasticInferenceAcceleratorProperty(
                    type="type",
            
                    # the properties below are optional
                    count=123
                )],
                enclave_options=ec2.CfnInstance.EnclaveOptionsProperty(
                    enabled=False
                ),
                hibernation_options=ec2.CfnInstance.HibernationOptionsProperty(
                    configured=False
                ),
                host_id="hostId",
                host_resource_group_arn="hostResourceGroupArn",
                iam_instance_profile="iamInstanceProfile",
                image_id="imageId",
                instance_initiated_shutdown_behavior="instanceInitiatedShutdownBehavior",
                instance_type="instanceType",
                ipv6_address_count=123,
                ipv6_addresses=[ec2.CfnInstance.InstanceIpv6AddressProperty(
                    ipv6_address="ipv6Address"
                )],
                kernel_id="kernelId",
                key_name="keyName",
                launch_template=ec2.CfnInstance.LaunchTemplateSpecificationProperty(
                    version="version",
            
                    # the properties below are optional
                    launch_template_id="launchTemplateId",
                    launch_template_name="launchTemplateName"
                ),
                license_specifications=[ec2.CfnInstance.LicenseSpecificationProperty(
                    license_configuration_arn="licenseConfigurationArn"
                )],
                monitoring=False,
                network_interfaces=[ec2.CfnInstance.NetworkInterfaceProperty(
                    device_index="deviceIndex",
            
                    # the properties below are optional
                    associate_public_ip_address=False,
                    delete_on_termination=False,
                    description="description",
                    group_set=["groupSet"],
                    ipv6_address_count=123,
                    ipv6_addresses=[ec2.CfnInstance.InstanceIpv6AddressProperty(
                        ipv6_address="ipv6Address"
                    )],
                    network_interface_id="networkInterfaceId",
                    private_ip_address="privateIpAddress",
                    private_ip_addresses=[ec2.CfnInstance.PrivateIpAddressSpecificationProperty(
                        primary=False,
                        private_ip_address="privateIpAddress"
                    )],
                    secondary_private_ip_address_count=123,
                    subnet_id="subnetId"
                )],
                placement_group_name="placementGroupName",
                private_ip_address="privateIpAddress",
                propagate_tags_to_volume_on_creation=False,
                ramdisk_id="ramdiskId",
                security_group_ids=["securityGroupIds"],
                security_groups=["securityGroups"],
                source_dest_check=False,
                ssm_associations=[ec2.CfnInstance.SsmAssociationProperty(
                    document_name="documentName",
            
                    # the properties below are optional
                    association_parameters=[ec2.CfnInstance.AssociationParameterProperty(
                        key="key",
                        value=["value"]
                    )]
                )],
                subnet_id="subnetId",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                tenancy="tenancy",
                user_data="userData",
                volumes=[ec2.CfnInstance.VolumeProperty(
                    device="device",
                    volume_id="volumeId"
                )]
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if additional_info is not None:
            self._values["additional_info"] = additional_info
        if affinity is not None:
            self._values["affinity"] = affinity
        if availability_zone is not None:
            self._values["availability_zone"] = availability_zone
        if block_device_mappings is not None:
            self._values["block_device_mappings"] = block_device_mappings
        if cpu_options is not None:
            self._values["cpu_options"] = cpu_options
        if credit_specification is not None:
            self._values["credit_specification"] = credit_specification
        if disable_api_termination is not None:
            self._values["disable_api_termination"] = disable_api_termination
        if ebs_optimized is not None:
            self._values["ebs_optimized"] = ebs_optimized
        if elastic_gpu_specifications is not None:
            self._values["elastic_gpu_specifications"] = elastic_gpu_specifications
        if elastic_inference_accelerators is not None:
            self._values["elastic_inference_accelerators"] = elastic_inference_accelerators
        if enclave_options is not None:
            self._values["enclave_options"] = enclave_options
        if hibernation_options is not None:
            self._values["hibernation_options"] = hibernation_options
        if host_id is not None:
            self._values["host_id"] = host_id
        if host_resource_group_arn is not None:
            self._values["host_resource_group_arn"] = host_resource_group_arn
        if iam_instance_profile is not None:
            self._values["iam_instance_profile"] = iam_instance_profile
        if image_id is not None:
            self._values["image_id"] = image_id
        if instance_initiated_shutdown_behavior is not None:
            self._values["instance_initiated_shutdown_behavior"] = instance_initiated_shutdown_behavior
        if instance_type is not None:
            self._values["instance_type"] = instance_type
        if ipv6_address_count is not None:
            self._values["ipv6_address_count"] = ipv6_address_count
        if ipv6_addresses is not None:
            self._values["ipv6_addresses"] = ipv6_addresses
        if kernel_id is not None:
            self._values["kernel_id"] = kernel_id
        if key_name is not None:
            self._values["key_name"] = key_name
        if launch_template is not None:
            self._values["launch_template"] = launch_template
        if license_specifications is not None:
            self._values["license_specifications"] = license_specifications
        if monitoring is not None:
            self._values["monitoring"] = monitoring
        if network_interfaces is not None:
            self._values["network_interfaces"] = network_interfaces
        if placement_group_name is not None:
            self._values["placement_group_name"] = placement_group_name
        if private_ip_address is not None:
            self._values["private_ip_address"] = private_ip_address
        if propagate_tags_to_volume_on_creation is not None:
            self._values["propagate_tags_to_volume_on_creation"] = propagate_tags_to_volume_on_creation
        if ramdisk_id is not None:
            self._values["ramdisk_id"] = ramdisk_id
        if security_group_ids is not None:
            self._values["security_group_ids"] = security_group_ids
        if security_groups is not None:
            self._values["security_groups"] = security_groups
        if source_dest_check is not None:
            self._values["source_dest_check"] = source_dest_check
        if ssm_associations is not None:
            self._values["ssm_associations"] = ssm_associations
        if subnet_id is not None:
            self._values["subnet_id"] = subnet_id
        if tags is not None:
            self._values["tags"] = tags
        if tenancy is not None:
            self._values["tenancy"] = tenancy
        if user_data is not None:
            self._values["user_data"] = user_data
        if volumes is not None:
            self._values["volumes"] = volumes

    @builtins.property
    def additional_info(self) -> typing.Optional[builtins.str]:
        '''This property is reserved for internal use.

        If you use it, the stack fails with this error: ``Bad property set: [Testing this property] (Service: AmazonEC2; Status Code: 400; Error Code: InvalidParameterCombination; Request ID: 0XXXXXX-49c7-4b40-8bcc-76885dcXXXXX)`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-additionalinfo
        '''
        result = self._values.get("additional_info")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def affinity(self) -> typing.Optional[builtins.str]:
        '''Indicates whether the instance is associated with a dedicated host.

        If you want the instance to always restart on the same host on which it was launched, specify ``host`` . If you want the instance to restart on any available host, but try to launch onto the last host it ran on (on a best-effort basis), specify ``default`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-affinity
        '''
        result = self._values.get("affinity")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def availability_zone(self) -> typing.Optional[builtins.str]:
        '''The Availability Zone of the instance.

        If not specified, an Availability Zone will be automatically chosen for you based on the load balancing criteria for the Region.

        This parameter is not supported by `DescribeImageAttribute <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImageAttribute.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-availabilityzone
        '''
        result = self._values.get("availability_zone")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def block_device_mappings(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.BlockDeviceMappingProperty]]]]:
        '''The block device mapping entries that defines the block devices to attach to the instance at launch.

        By default, the block devices specified in the block device mapping for the AMI are used. You can override the AMI block device mapping using the instance block device mapping. For the root volume, you can override only the volume size, volume type, volume encryption settings, and the ``DeleteOnTermination`` setting.
        .. epigraph::

           After the instance is running, you can modify only the ``DeleteOnTermination`` parameter for the attached volumes without interrupting the instance. Modifying any other parameter results in instance `replacement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-update-behaviors.html#update-replacement>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-blockdevicemappings
        '''
        result = self._values.get("block_device_mappings")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.BlockDeviceMappingProperty]]]], result)

    @builtins.property
    def cpu_options(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.CpuOptionsProperty]]:
        '''The CPU options for the instance.

        For more information, see `Optimize CPU options <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-optimize-cpu.html>`_ in the *Amazon Elastic Compute Cloud User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-cpuoptions
        '''
        result = self._values.get("cpu_options")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.CpuOptionsProperty]], result)

    @builtins.property
    def credit_specification(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.CreditSpecificationProperty]]:
        '''The credit option for CPU usage of the burstable performance instance.

        Valid values are ``standard`` and ``unlimited`` . To change this attribute after launch, use `ModifyInstanceCreditSpecification <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceCreditSpecification.html>`_ . For more information, see `Burstable performance instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances.html>`_ in the *Amazon EC2 User Guide* .

        Default: ``standard`` (T2 instances) or ``unlimited`` (T3/T3a instances)

        For T3 instances with ``host`` tenancy, only ``standard`` is supported.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-creditspecification
        '''
        result = self._values.get("credit_specification")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.CreditSpecificationProperty]], result)

    @builtins.property
    def disable_api_termination(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''If you set this parameter to ``true`` , you can't terminate the instance using the Amazon EC2 console, CLI, or API;

        otherwise, you can. To change this attribute after launch, use `ModifyInstanceAttribute <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceAttribute.html>`_ . Alternatively, if you set ``InstanceInitiatedShutdownBehavior`` to ``terminate`` , you can terminate the instance by running the shutdown command from the instance.

        Default: ``false``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-disableapitermination
        '''
        result = self._values.get("disable_api_termination")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def ebs_optimized(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''Indicates whether the instance is optimized for Amazon EBS I/O.

        This optimization provides dedicated throughput to Amazon EBS and an optimized configuration stack to provide optimal Amazon EBS I/O performance. This optimization isn't available with all instance types. Additional usage charges apply when using an EBS-optimized instance.

        Default: ``false``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-ebsoptimized
        '''
        result = self._values.get("ebs_optimized")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def elastic_gpu_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.ElasticGpuSpecificationProperty]]]]:
        '''An elastic GPU to associate with the instance.

        An Elastic GPU is a GPU resource that you can attach to your Windows instance to accelerate the graphics performance of your applications. For more information, see `Amazon EC2 Elastic GPUs <https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/elastic-graphics.html>`_ in the *Amazon EC2 User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-elasticgpuspecifications
        '''
        result = self._values.get("elastic_gpu_specifications")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.ElasticGpuSpecificationProperty]]]], result)

    @builtins.property
    def elastic_inference_accelerators(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.ElasticInferenceAcceleratorProperty]]]]:
        '''An elastic inference accelerator to associate with the instance.

        Elastic inference accelerators are a resource you can attach to your Amazon EC2 instances to accelerate your Deep Learning (DL) inference workloads.

        You cannot specify accelerators from different generations in the same request.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-elasticinferenceaccelerators
        '''
        result = self._values.get("elastic_inference_accelerators")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.ElasticInferenceAcceleratorProperty]]]], result)

    @builtins.property
    def enclave_options(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.EnclaveOptionsProperty]]:
        '''Indicates whether the instance is enabled for AWS Nitro Enclaves.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-enclaveoptions
        '''
        result = self._values.get("enclave_options")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.EnclaveOptionsProperty]], result)

    @builtins.property
    def hibernation_options(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.HibernationOptionsProperty]]:
        '''Indicates whether an instance is enabled for hibernation.

        For more information, see `Hibernate your instance <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Hibernate.html>`_ in the *Amazon EC2 User Guide* .

        You can't enable hibernation and AWS Nitro Enclaves on the same instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-hibernationoptions
        '''
        result = self._values.get("hibernation_options")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.HibernationOptionsProperty]], result)

    @builtins.property
    def host_id(self) -> typing.Optional[builtins.str]:
        '''If you specify host for the ``Affinity`` property, the ID of a dedicated host that the instance is associated with.

        If you don't specify an ID, Amazon EC2 launches the instance onto any available, compatible dedicated host in your account. This type of launch is called an untargeted launch. Note that for untargeted launches, you must have a compatible, dedicated host available to successfully launch instances.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-hostid
        '''
        result = self._values.get("host_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def host_resource_group_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN of the host resource group in which to launch the instances.

        If you specify a host resource group ARN, omit the *Tenancy* parameter or set it to ``host`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-hostresourcegrouparn
        '''
        result = self._values.get("host_resource_group_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def iam_instance_profile(self) -> typing.Optional[builtins.str]:
        '''The name of an IAM instance profile.

        To create a new IAM instance profile, use the `AWS::IAM::InstanceProfile <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-iam-instanceprofile.html>`_ resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-iaminstanceprofile
        '''
        result = self._values.get("iam_instance_profile")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def image_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the AMI.

        An AMI ID is required to launch an instance and must be specified here or in a launch template.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-imageid
        '''
        result = self._values.get("image_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def instance_initiated_shutdown_behavior(self) -> typing.Optional[builtins.str]:
        '''Indicates whether an instance stops or terminates when you initiate shutdown from the instance (using the operating system command for system shutdown).

        Default: ``stop``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-instanceinitiatedshutdownbehavior
        '''
        result = self._values.get("instance_initiated_shutdown_behavior")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def instance_type(self) -> typing.Optional[builtins.str]:
        '''The instance type. For more information, see `Instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html>`_ in the *Amazon EC2 User Guide* .

        Default: ``m1.small``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-instancetype
        '''
        result = self._values.get("instance_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ipv6_address_count(self) -> typing.Optional[jsii.Number]:
        '''[EC2-VPC] The number of IPv6 addresses to associate with the primary network interface.

        Amazon EC2 chooses the IPv6 addresses from the range of your subnet. You cannot specify this option and the option to assign specific IPv6 addresses in the same request. You can specify this option if you've specified a minimum number of instances to launch.

        You cannot specify this option and the network interfaces option in the same request.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-ipv6addresscount
        '''
        result = self._values.get("ipv6_address_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def ipv6_addresses(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.InstanceIpv6AddressProperty]]]]:
        '''[EC2-VPC] The IPv6 addresses from the range of the subnet to associate with the primary network interface.

        You cannot specify this option and the option to assign a number of IPv6 addresses in the same request. You cannot specify this option if you've specified a minimum number of instances to launch.

        You cannot specify this option and the network interfaces option in the same request.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-ipv6addresses
        '''
        result = self._values.get("ipv6_addresses")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.InstanceIpv6AddressProperty]]]], result)

    @builtins.property
    def kernel_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the kernel.

        .. epigraph::

           We recommend that you use PV-GRUB instead of kernels and RAM disks. For more information, see `PV-GRUB <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/UserProvidedkernels.html>`_ in the *Amazon EC2 User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-kernelid
        '''
        result = self._values.get("kernel_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def key_name(self) -> typing.Optional[builtins.str]:
        '''The name of the key pair. You can create a key pair using `CreateKeyPair <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateKeyPair.html>`_ or `ImportKeyPair <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportKeyPair.html>`_ .

        .. epigraph::

           If you do not specify a key pair, you can't connect to the instance unless you choose an AMI that is configured to allow users another way to log in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-keyname
        '''
        result = self._values.get("key_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def launch_template(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.LaunchTemplateSpecificationProperty]]:
        '''The launch template to use to launch the instances.

        Any parameters that you specify in the AWS CloudFormation template override the same parameters in the launch template. You can specify either the name or ID of a launch template, but not both.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-launchtemplate
        '''
        result = self._values.get("launch_template")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.LaunchTemplateSpecificationProperty]], result)

    @builtins.property
    def license_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.LicenseSpecificationProperty]]]]:
        '''The license configurations.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-licensespecifications
        '''
        result = self._values.get("license_specifications")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.LicenseSpecificationProperty]]]], result)

    @builtins.property
    def monitoring(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''Specifies whether detailed monitoring is enabled for the instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-monitoring
        '''
        result = self._values.get("monitoring")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def network_interfaces(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.NetworkInterfaceProperty]]]]:
        '''The network interfaces to associate with the instance.

        .. epigraph::

           If you use this property to point to a network interface, you must terminate the original interface before attaching a new one to allow the update of the instance to succeed.

           If this resource has a public IP address and is also in a VPC that is defined in the same template, you must use the `DependsOn Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html>`_ to declare a dependency on the VPC-gateway attachment.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-networkinterfaces
        '''
        result = self._values.get("network_interfaces")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.NetworkInterfaceProperty]]]], result)

    @builtins.property
    def placement_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of an existing placement group that you want to launch the instance into (cluster | partition | spread).

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-placementgroupname
        '''
        result = self._values.get("placement_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def private_ip_address(self) -> typing.Optional[builtins.str]:
        '''[EC2-VPC] The primary IPv4 address. You must specify a value from the IPv4 address range of the subnet.

        Only one private IP address can be designated as primary. You can't specify this option if you've specified the option to designate a private IP address as the primary IP address in a network interface specification. You cannot specify this option if you're launching more than one instance in the request.

        You cannot specify this option and the network interfaces option in the same request.

        If you make an update to an instance that requires replacement, you must assign a new private IP address. During a replacement, AWS CloudFormation creates a new instance but doesn't delete the old instance until the stack has successfully updated. If the stack update fails, AWS CloudFormation uses the old instance to roll back the stack to the previous working state. The old and new instances cannot have the same private IP address.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-privateipaddress
        '''
        result = self._values.get("private_ip_address")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def propagate_tags_to_volume_on_creation(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''Whether to propagate the EC2 instance tags to the EBS volumes.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-propagatetagstovolumeoncreation
        '''
        result = self._values.get("propagate_tags_to_volume_on_creation")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def ramdisk_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the RAM disk to select.

        Some kernels require additional drivers at launch. Check the kernel requirements for information about whether you need to specify a RAM disk. To find kernel requirements, go to the AWS Resource Center and search for the kernel ID.
        .. epigraph::

           We recommend that you use PV-GRUB instead of kernels and RAM disks. For more information, see `PV-GRUB <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/UserProvidedkernels.html>`_ in the *Amazon EC2 User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-ramdiskid
        '''
        result = self._values.get("ramdisk_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The IDs of the security groups. You can create a security group using `CreateSecurityGroup <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSecurityGroup.html>`_ .

        If you specify a network interface, you must specify any security groups as part of the network interface.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-securitygroupids
        '''
        result = self._values.get("security_group_ids")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
        '''[EC2-Classic, default VPC] The names of the security groups.

        For a nondefault VPC, you must use security group IDs instead.

        You cannot specify this option and the network interfaces option in the same request. The list can contain both the name of existing Amazon EC2 security groups or references to AWS::EC2::SecurityGroup resources created in the template.

        Default: Amazon EC2 uses the default security group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-securitygroups
        '''
        result = self._values.get("security_groups")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def source_dest_check(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''Enable or disable source/destination checks, which ensure that the instance is either the source or the destination of any traffic that it receives.

        If the value is ``true`` , source/destination checks are enabled; otherwise, they are disabled. The default value is ``true`` . You must disable source/destination checks if the instance runs services such as network address translation, routing, or firewalls.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-sourcedestcheck
        '''
        result = self._values.get("source_dest_check")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def ssm_associations(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.SsmAssociationProperty]]]]:
        '''The SSM `document <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html>`_ and parameter values in AWS Systems Manager to associate with this instance. To use this property, you must specify an IAM instance profile role for the instance. For more information, see `Create an Instance Profile for Systems Manager <https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-configuring-access-role.html>`_ in the *AWS Systems Manager User Guide* .

        .. epigraph::

           You can currently associate only one document with an instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-ssmassociations
        '''
        result = self._values.get("ssm_associations")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.SsmAssociationProperty]]]], result)

    @builtins.property
    def subnet_id(self) -> typing.Optional[builtins.str]:
        '''[EC2-VPC] The ID of the subnet to launch the instance into.

        If you specify a network interface, you must specify any subnets as part of the network interface.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-subnetid
        '''
        result = self._values.get("subnet_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''The tags to add to the instance.

        These tags are not applied to the EBS volumes, such as the root volume.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    @builtins.property
    def tenancy(self) -> typing.Optional[builtins.str]:
        '''The tenancy of the instance (if the instance is running in a VPC).

        An instance with a tenancy of ``dedicated`` runs on single-tenant hardware.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-tenancy
        '''
        result = self._values.get("tenancy")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def user_data(self) -> typing.Optional[builtins.str]:
        '''The user data to make available to the instance.

        For more information, see `Run commands on your Linux instance at launch <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/user-data.html>`_ and `Run commands on your Windows instance at launch <https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ec2-windows-user-data.html>`_ . If you are using a command line tool, base64-encoding is performed for you, and you can load the text from a file. Otherwise, you must provide base64-encoded text. User data is limited to 16 KB.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-userdata
        '''
        result = self._values.get("user_data")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def volumes(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.VolumeProperty]]]]:
        '''The volumes to attach to the instance.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-volumes
        '''
        result = self._values.get("volumes")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.VolumeProperty]]]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnInstanceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnInternetGateway(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnInternetGateway",
):
    '''A CloudFormation ``AWS::EC2::InternetGateway``.

    Allocates an internet gateway for use with a VPC. After creating the Internet gateway, you then attach it to a VPC.

    :cloudformationResource: AWS::EC2::InternetGateway
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-internetgateway.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_ec2 as ec2
        
        cfn_internet_gateway = ec2.CfnInternetGateway(self, "MyCfnInternetGateway",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::InternetGateway``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param tags: Any tags to assign to the internet gateway.
        '''
        props = CfnInternetGatewayProps(tags=tags)

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrInternetGatewayId")
    def attr_internet_gateway_id(self) -> builtins.str:
        '''The ID of the internet gateway.

        :cloudformationAttribute: InternetGatewayId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrInternetGatewayId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''Any tags to assign to the internet gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-internetgateway.html#cfn-ec2-internetgateway-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnInternetGatewayProps",
    jsii_struct_bases=[],
    name_mapping={"tags": "tags"},
)
class CfnInternetGatewayProps:
    def __init__(
        self,
        *,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Properties for defining a ``CfnInternetGateway``.

        :param tags: Any tags to assign to the internet gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-internetgateway.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_ec2 as ec2
            
            cfn_internet_gateway_props = ec2.CfnInternetGatewayProps(
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''Any tags to assign to the internet gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-internetgateway.html#cfn-ec2-internetgateway-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnInternetGatewayProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnLaunchTemplate(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate",
):
    '''A CloudFormation ``AWS::EC2::LaunchTemplate``.

    Specifies a launch template for an Amazon EC2 instance. A launch template contains the parameters to launch an instance. For more information, see `Launch an instance from a launch template <https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ec2-launch-templates.html>`_ in the *Amazon EC2 User Guide* .

    :cloudformationResource: AWS::EC2::LaunchTemplate
    :exampleMetadata: infused
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-launchtemplate.html

    Example::

        # cluster is of type Cluster
        
        
        user_data = """MIME-Version: 1.0
        Content-Type: multipart/mixed; boundary="==MYBOUNDARY=="
        
        --==MYBOUNDARY==
        Content-Type: text/x-shellscript; charset="us-ascii"
        
        #!/bin/bash
        echo "Running custom user data script"
        
        --==MYBOUNDARY==--\\
        """
        lt = ec2.CfnLaunchTemplate(self, "LaunchTemplate",
            launch_template_data=ec2.CfnLaunchTemplate.LaunchTemplateDataProperty(
                instance_type="t3.small",
                user_data=Fn.base64(user_data)
            )
        )
        
        cluster.add_nodegroup_capacity("extra-ng",
            launch_template_spec=eks.LaunchTemplateSpec(
                id=lt.ref,
                version=lt.attr_latest_version_number
            )
        )
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        launch_template_data: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.LaunchTemplateDataProperty"]] = None,
        launch_template_name: typing.Optional[builtins.str] = None,
        tag_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.LaunchTemplateTagSpecificationProperty"]]]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::LaunchTemplate``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param launch_template_data: The information for the launch template.
        :param launch_template_name: A name for the launch template.
        :param tag_specifications: The tags to apply to the launch template during creation.
        '''
        props = CfnLaunchTemplateProps(
            launch_template_data=launch_template_data,
            launch_template_name=launch_template_name,
            tag_specifications=tag_specifications,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrDefaultVersionNumber")
    def attr_default_version_number(self) -> builtins.str:
        '''The default version of the launch template, such as 2.

        The default version of a launch template cannot be specified in AWS CloudFormation . The default version can be set in the Amazon EC2 Console or by using the ``modify-launch-template`` AWS CLI command.

        :cloudformationAttribute: DefaultVersionNumber
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDefaultVersionNumber"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrLatestVersionNumber")
    def attr_latest_version_number(self) -> builtins.str:
        '''The latest version of the launch template, such as ``5`` .

        :cloudformationAttribute: LatestVersionNumber
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrLatestVersionNumber"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="launchTemplateData")
    def launch_template_data(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.LaunchTemplateDataProperty"]]:
        '''The information for the launch template.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-launchtemplate.html#cfn-ec2-launchtemplate-launchtemplatedata
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.LaunchTemplateDataProperty"]], jsii.get(self, "launchTemplateData"))

    @launch_template_data.setter
    def launch_template_data(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.LaunchTemplateDataProperty"]],
    ) -> None:
        jsii.set(self, "launchTemplateData", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="launchTemplateName")
    def launch_template_name(self) -> typing.Optional[builtins.str]:
        '''A name for the launch template.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-launchtemplate.html#cfn-ec2-launchtemplate-launchtemplatename
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "launchTemplateName"))

    @launch_template_name.setter
    def launch_template_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "launchTemplateName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tagSpecifications")
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.LaunchTemplateTagSpecificationProperty"]]]]:
        '''The tags to apply to the launch template during creation.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-launchtemplate.html#cfn-ec2-launchtemplate-tagspecifications
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.LaunchTemplateTagSpecificationProperty"]]]], jsii.get(self, "tagSpecifications"))

    @tag_specifications.setter
    def tag_specifications(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.LaunchTemplateTagSpecificationProperty"]]]],
    ) -> None:
        jsii.set(self, "tagSpecifications", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.AcceleratorCountProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class AcceleratorCountProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The minimum and maximum number of accelerators (GPUs, FPGAs, or AWS Inferentia chips) on an instance.

            :param max: The maximum number of accelerators. To specify no maximum limit, omit this parameter. To exclude accelerator-enabled instance types, set ``Max`` to ``0`` .
            :param min: The minimum number of accelerators. To specify no minimum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-acceleratorcount.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                accelerator_count_property = ec2.CfnLaunchTemplate.AcceleratorCountProperty(
                    max=123,
                    min=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The maximum number of accelerators.

            To specify no maximum limit, omit this parameter. To exclude accelerator-enabled instance types, set ``Max`` to ``0`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-acceleratorcount.html#cfn-ec2-launchtemplate-acceleratorcount-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The minimum number of accelerators.

            To specify no minimum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-acceleratorcount.html#cfn-ec2-launchtemplate-acceleratorcount-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AcceleratorCountProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.AcceleratorTotalMemoryMiBProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class AcceleratorTotalMemoryMiBProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The minimum and maximum amount of total accelerator memory, in MiB.

            :param max: The maximum amount of accelerator memory, in MiB. To specify no maximum limit, omit this parameter.
            :param min: The minimum amount of accelerator memory, in MiB. To specify no minimum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-acceleratortotalmemorymib.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                accelerator_total_memory_mi_bProperty = ec2.CfnLaunchTemplate.AcceleratorTotalMemoryMiBProperty(
                    max=123,
                    min=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The maximum amount of accelerator memory, in MiB.

            To specify no maximum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-acceleratortotalmemorymib.html#cfn-ec2-launchtemplate-acceleratortotalmemorymib-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The minimum amount of accelerator memory, in MiB.

            To specify no minimum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-acceleratortotalmemorymib.html#cfn-ec2-launchtemplate-acceleratortotalmemorymib-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AcceleratorTotalMemoryMiBProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.BaselineEbsBandwidthMbpsProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class BaselineEbsBandwidthMbpsProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The minimum and maximum baseline bandwidth to Amazon EBS, in Mbps.

            For more information, see `Amazon EBS–optimized instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-optimized.html>`_ in the *Amazon EC2 User Guide* .

            :param max: The maximum baseline bandwidth, in Mbps. To specify no maximum limit, omit this parameter.
            :param min: The minimum baseline bandwidth, in Mbps. To specify no minimum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-baselineebsbandwidthmbps.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                baseline_ebs_bandwidth_mbps_property = ec2.CfnLaunchTemplate.BaselineEbsBandwidthMbpsProperty(
                    max=123,
                    min=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The maximum baseline bandwidth, in Mbps.

            To specify no maximum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-baselineebsbandwidthmbps.html#cfn-ec2-launchtemplate-baselineebsbandwidthmbps-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The minimum baseline bandwidth, in Mbps.

            To specify no minimum limit, omit this parameter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-baselineebsbandwidthmbps.html#cfn-ec2-launchtemplate-baselineebsbandwidthmbps-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BaselineEbsBandwidthMbpsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.BlockDeviceMappingProperty",
        jsii_struct_bases=[],
        name_mapping={
            "device_name": "deviceName",
            "ebs": "ebs",
            "no_device": "noDevice",
            "virtual_name": "virtualName",
        },
    )
    class BlockDeviceMappingProperty:
        def __init__(
            self,
            *,
            device_name: typing.Optional[builtins.str] = None,
            ebs: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.EbsProperty"]] = None,
            no_device: typing.Optional[builtins.str] = None,
            virtual_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Information about a block device mapping for an Amazon EC2 launch template.

            ``BlockDeviceMapping`` is a property of `AWS::EC2::LaunchTemplate LaunchTemplateData <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html>`_ .

            :param device_name: The device name (for example, /dev/sdh or xvdh).
            :param ebs: Parameters used to automatically set up EBS volumes when the instance is launched.
            :param no_device: To omit the device from the block device mapping, specify an empty string.
            :param virtual_name: The virtual device name (ephemeralN). Instance store volumes are numbered starting from 0. An instance type with 2 available instance store volumes can specify mappings for ephemeral0 and ephemeral1. The number of available instance store volumes depends on the instance type. After you connect to the instance, you must mount the volume.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-blockdevicemapping.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                block_device_mapping_property = ec2.CfnLaunchTemplate.BlockDeviceMappingProperty(
                    device_name="deviceName",
                    ebs=ec2.CfnLaunchTemplate.EbsProperty(
                        delete_on_termination=False,
                        encrypted=False,
                        iops=123,
                        kms_key_id="kmsKeyId",
                        snapshot_id="snapshotId",
                        throughput=123,
                        volume_size=123,
                        volume_type="volumeType"
                    ),
                    no_device="noDevice",
                    virtual_name="virtualName"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if device_name is not None:
                self._values["device_name"] = device_name
            if ebs is not None:
                self._values["ebs"] = ebs
            if no_device is not None:
                self._values["no_device"] = no_device
            if virtual_name is not None:
                self._values["virtual_name"] = virtual_name

        @builtins.property
        def device_name(self) -> typing.Optional[builtins.str]:
            '''The device name (for example, /dev/sdh or xvdh).

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-blockdevicemapping.html#cfn-ec2-launchtemplate-blockdevicemapping-devicename
            '''
            result = self._values.get("device_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ebs(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.EbsProperty"]]:
            '''Parameters used to automatically set up EBS volumes when the instance is launched.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-blockdevicemapping.html#cfn-ec2-launchtemplate-blockdevicemapping-ebs
            '''
            result = self._values.get("ebs")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.EbsProperty"]], result)

        @builtins.property
        def no_device(self) -> typing.Optional[builtins.str]:
            '''To omit the device from the block device mapping, specify an empty string.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-blockdevicemapping.html#cfn-ec2-launchtemplate-blockdevicemapping-nodevice
            '''
            result = self._values.get("no_device")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def virtual_name(self) -> typing.Optional[builtins.str]:
            '''The virtual device name (ephemeralN).

            Instance store volumes are numbered starting from 0. An instance type with 2 available instance store volumes can specify mappings for ephemeral0 and ephemeral1. The number of available instance store volumes depends on the instance type. After you connect to the instance, you must mount the volume.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-blockdevicemapping.html#cfn-ec2-launchtemplate-blockdevicemapping-virtualname
            '''
            result = self._values.get("virtual_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BlockDeviceMappingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.CapacityReservationSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "capacity_reservation_preference": "capacityReservationPreference",
            "capacity_reservation_target": "capacityReservationTarget",
        },
    )
    class CapacityReservationSpecificationProperty:
        def __init__(
            self,
            *,
            capacity_reservation_preference: typing.Optional[builtins.str] = None,
            capacity_reservation_target: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.CapacityReservationTargetProperty"]] = None,
        ) -> None:
            '''Specifies an instance's Capacity Reservation targeting option. You can specify only one option at a time.

            ``CapacityReservationSpecification`` is a property of `AWS::EC2::LaunchTemplate LaunchTemplateData <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html>`_ .

            :param capacity_reservation_preference: Indicates the instance's Capacity Reservation preferences. Possible preferences include:. - ``open`` - The instance can run in any ``open`` Capacity Reservation that has matching attributes (instance type, platform, Availability Zone). - ``none`` - The instance avoids running in a Capacity Reservation even if one is available. The instance runs in On-Demand capacity.
            :param capacity_reservation_target: Information about the target Capacity Reservation or Capacity Reservation group.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-capacityreservationspecification.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                capacity_reservation_specification_property = ec2.CfnLaunchTemplate.CapacityReservationSpecificationProperty(
                    capacity_reservation_preference="capacityReservationPreference",
                    capacity_reservation_target=ec2.CfnLaunchTemplate.CapacityReservationTargetProperty(
                        capacity_reservation_id="capacityReservationId",
                        capacity_reservation_resource_group_arn="capacityReservationResourceGroupArn"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if capacity_reservation_preference is not None:
                self._values["capacity_reservation_preference"] = capacity_reservation_preference
            if capacity_reservation_target is not None:
                self._values["capacity_reservation_target"] = capacity_reservation_target

        @builtins.property
        def capacity_reservation_preference(self) -> typing.Optional[builtins.str]:
            '''Indicates the instance's Capacity Reservation preferences. Possible preferences include:.

            - ``open`` - The instance can run in any ``open`` Capacity Reservation that has matching attributes (instance type, platform, Availability Zone).
            - ``none`` - The instance avoids running in a Capacity Reservation even if one is available. The instance runs in On-Demand capacity.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-capacityreservationspecification.html#cfn-ec2-launchtemplate-launchtemplatedata-capacityreservationspecification-capacityreservationpreference
            '''
            result = self._values.get("capacity_reservation_preference")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def capacity_reservation_target(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.CapacityReservationTargetProperty"]]:
            '''Information about the target Capacity Reservation or Capacity Reservation group.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-capacityreservationspecification.html#cfn-ec2-launchtemplate-launchtemplatedata-capacityreservationspecification-capacityreservationtarget
            '''
            result = self._values.get("capacity_reservation_target")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.CapacityReservationTargetProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CapacityReservationSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.CapacityReservationTargetProperty",
        jsii_struct_bases=[],
        name_mapping={
            "capacity_reservation_id": "capacityReservationId",
            "capacity_reservation_resource_group_arn": "capacityReservationResourceGroupArn",
        },
    )
    class CapacityReservationTargetProperty:
        def __init__(
            self,
            *,
            capacity_reservation_id: typing.Optional[builtins.str] = None,
            capacity_reservation_resource_group_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Specifies a target Capacity Reservation.

            ``CapacityReservationTarget`` is a property of the `Amazon EC2 LaunchTemplate LaunchTemplateData <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html>`_ property type.

            :param capacity_reservation_id: The ID of the Capacity Reservation in which to run the instance.
            :param capacity_reservation_resource_group_arn: The ARN of the Capacity Reservation resource group in which to run the instance.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-capacityreservationtarget.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_ec2 as ec2
                
                capacity_reservation_target_property = ec2.CfnLaunchTemplate.CapacityReservationTargetProperty(
                    capacity_reservation_id="capacityReservationId",
                    capacity_reservation_resource_group_arn="capacityReservationResourceGroupArn"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if capacity_reservation_id is not None:
                self._values["capacity_reservation_id"] = capacity_reservation_id
            if capacity_reservation_resource_group_arn is not None:
                self._values["capacity_reservation_resource_group_arn"] = capacity_reservation_resource_group_arn

        @builtins.property
        def capacity_reservation_id(self) -> typing.Optional[builtins.str]:
            '''The ID of the Capacity Reservation in which to run the instance.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-capacityreservationtarget.html#cfn-ec2-launchtemplate-capacityreservationtarget-capacityreservationid
            '''
            result = self._values.get("capacity_reservation_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def capacity_reservation_resource_group_arn(
            self,
        ) -> typing.Optional[builtins.str]:
            '''The ARN of the Capacity Reservation resource group in which to run the instance.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-capacityreservationtarget.html#cfn-ec2-launchtemplate-capacityreservationtarget-capacityreservationresourcegrouparn
            '''
            result = self._values.get("capacity_reservation_resource_group_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs