# Topic guides and how-tos

These sections cover common operations and topics that are relevant to developing this theme.

## Make a release

This theme uses GitHub tags and releases to automatically push new releases to
PyPI. For information on this process, see [the release checklist](https://github.com/pydata/pydata-sphinx-theme/wiki/Release-checklist#release-instructions).

## Update JavaScript dependencies and their versions

The javascript dependencies for this package are defined in `package.json`, and broken down into a few categories like `dependencies` and `devDependencies`.

To update or add JS dependency, modify (or append to) the list of packages that are listed in each of these sections.
The next time you build the documentation (either with `nox` or with `stb`), these new dependencies will be installed and bundled with the theme.

## Using `nox`

Here are a few extra tips for using `nox`.

:::{seealso}
The [`nox` command line documentation](https://nox.thea.codes/en/stable/usage.html) has a lot of helpful tips for extra functionality you can enable with the CLI.
:::

### Re-install dependencies

To re-execute the installation commands, use this pattern:

```console
$ nox -s docs -- reinstall
```

Or to completely remove the environment generated by `nox` and start from scratch:

```console
$ rm -rf .nox/docs
```

### Use `nox` with your global environment

If you'd like to use `nox` with your **global** environment (the one from which you are calling `nox`), you can do so with:

```console
$ nox --force-venv-backend none

# alternatively:
$ nox --no-venv
```

Using `none` will re-use your current global environment.
See
[the nox documentation](https://nox.thea.codes/en/stable/usage.html#forcing-the-sessions-backend) for more details.

## Using `pre-commit`

Here are a few tips for using `pre-commit`:

### Skip the pre-commit checks

Run the following command:

```console
$ git commit --no-verify
```

### Run pre-commit on all files

By default, `pre-commit` will run its checks on files that have been modified in a commit.
To instead run it on all files, use this command:

```console
$ pre-commit run --all-files

# Alternatively
$ pre-commit run -a
```

## Web asset compiling and bundling

All of the CSS and JS assets stored in `src/pydata_sphinx_theme/assets` will be compiled and bundled with the theme when you build it locally.
These bundled assets will be placed in `src/pydata_sphinx_theme/theme/pydata_sphinx_theme/static`.

The configuration that defines what happens upon compilation is at `webpack.config.js`.

When assets are compiled, a `<hash>` is generated for each, and appended to the end of the asset's reference in the HTML templates of the theme.
This ensures the correct asset versions are served when viewers return to your
site after upgrading the theme.

Web fonts, and their supporting CSS, are copied into
`src/pydata_sphinx_theme/theme/pydata_sphinx_theme/static/vendor/<font name>/<font version>/`.

The links to these unique file names are captured as Jinja2 macros that are defined in HTML templates bundled with the theme.

## Accessibility checks

The accessibility checking tools can find a number of common HTML patterns which
assistive technology can't help users understand.

In addition to [Lighthouse](https://developers.google.com/web/tools/lighthouse)
in CI, the `pa11y` stack is installed as part of the development environment.

The key components are:

- [pa11y](https://github.com/pa11y/pa11y) which uses a headless browser to analyze
  an HTML page with a configurable set of rules based on publish standards
- [Pa11y-CI](https://github.com/pa11y/pa11y-ci) runs `pa11y` on multiple pages
- [pa11y-reporter-html](https://github.com/pa11y/pa11y-reporter-html) generates
  some nice HTML reports, suitable for review

:::{note}
Presently, the _default_ `pa11y` ruleset, `WCAG2AA` is used, a subset of
the [Web Content Accessibility Guidelines](https://www.w3.org/TR/WCAG21).
The [Quick Reference](https://www.w3.org/WAI/WCAG21/quickref) may provide
lighter reading.
:::

### Errors in CI/CD and what to do

We have a list of **known accessibility problems** in the file `docs/scripts/a11y-roadmap.txt`.
This contains a list of errors that we aim to fix in the future, and that **do not cause tests to fail**.

When a pa11y accessibility audit is run in our CI/CD, it checks for any errors that are _not_ on this list, and if it finds them it will cause the job to error.

When you see an error in your CI/CD job, look at the logs under the `Run accessibility audit` job.
You should see an output that looks like this:

```
JSON: /tmp/pa11y/pa11y-864/pa11y-ci-results.json
Roadmap: /home/runner/work/pydata-sphinx-theme/pydata-sphinx-theme/docs/a11y-roadmap.txt
not on roadmap:
  WCAG2AA.Principle2.Guideline2_4.2_4_1.G1,G123,G124.NoSuchID: 4
on roadmap:
  WCAG2AA.Principle1.Guideline1_3.1_3_1.H39.3.LayoutTable: 1
  WCAG2AA.Principle1.Guideline1_3.1_3_1.H43,H63: 1
  WCAG2AA.Principle1.Guideline1_3.1_3_1.H43.HeadersRequired: 1
  WCAG2AA.Principle1.Guideline1_4.1_4_3.G18.Fail: 1828
  WCAG2AA.Principle3.Guideline3_2.3_2_2.H32.2: 48
  WCAG2AA.Principle4.Guideline4_1.4_1_2.H91.A.EmptyNoId: 9
passed: false
total errors: 1892
```

The problems that caused an error are in the `not on roadmap` section.
Anything that is "not on the roadmap" is an error we have unexpectedly introduced in the PR.
These should be identified and fixed.

### Fix accessibility errors

We keep a list of **known accessibility issues** in the {download}`accessibility roadmap <../scripts/a11y-roadmap.txt>`.
These are issues which are currently flagged by the toolset, but that have not yet
been fixed.

To start working on one of the accessibility roadmap items, comment out one of the
lines in `docs/a11y-roadmap.txt`, and re-run the audit to establish a baseline.

Then, fix the issue in either the HTML templates, CSS, or python code, and re-run
the audit until it is fixed.

### Run an accessibility audit locally

To run the accessibility problem finder locally:

```console
$ nox -s compile  # Compile the theme assets
$ nox -s docs  # Build the documentation
$ python docs/scripts/a11y.py  # Run a helper script for an accessibility audit
```

The output of the last command includes:

- a short summary of the current state of the accessibility rules we are trying to maintain
- local paths to JSON and HTML reports which contain all of the issues found

## Change fonts

Three "styles" of the [FontAwesome 5 Free](https://fontawesome.com/icons?m=free)
icon font are used for {ref}`icon links <icon-links>` and admonitions, and is
the only `vendored` font. Further font choices are described in the {ref}`customizing`
section of the user guide, and require some knowledge of HTML and CSS.

The remaining vendored font selection is:

- managed as a dependency in `package.json`

  - allowing the version to be managed centrally

- copied directly into the site statics, including licenses

  - allowing the chosen font to be replaced (or removed entirely) with minimal
    templating changes: practically, changing the icon font is difficult at this
    point.

- partially preloaded

  - reducing flicker and re-layout artifacts of early icon renders

- mostly managed in `webpack.js`

  - allowing upgrades to be handled in a relatively sane, manageable way, to
    ensure the most recent icons

### Upgrade a font

If _only_ the version of the `existing` font must change, for example to enable
new icons, edit the appropriate font version in `package.json`.
Then re-compile the theme with:

```console
$ nox -s compile
```

### Change a font

If the above doesn't work, for example if file names for an existing font change,
or a new font variant altogether is being added, hand-editing of `webpack.config.js`
is required. The steps are roughly:

- install the new font, as above
- in `webpack.config.js`:
  - add the new font to `vendorVersions` and `vendorPaths`
  - add new `link` tags to the appropriate macro in `macroTemplate`
  - add the new font files (including the license) to `CopyPlugin`
  - remove references to the font being replaced/removed, if applicable
  - see the `font-awesome` sections of this configuration to see what the end-result configuration looks like.
- re-compile the theme's assets: `nox -s compile`

## Update our kitchen sink documents

The [kitchen sink reference](../demo/kitchen-sink/index.rst) is for demonstrating as much syntax and style for Sphinx builds as possible.
It is copied directly from the [`sphinx-themes.org` documentation](https://sphinx-themes.org/) so that we use standardized reference docs compared with other communities.
The source files for these pages are stored [in the `sphinx-themes.org` repository](https://github.com/sphinx-themes/sphinx-themes.org/raw/master/sample-docs/kitchen-sink/).

If you'd like to update our local files with any changes that have been made to the `sphinx-themes.org` files, simply copy/paste those changes into our local files and make a commit.

Here's a list of our pages and where they come from in `sphinx-themes.org`:

- [`index.rst`](../demo/kitchen-sink/index.rst) ([source](https://github.com/sphinx-themes/sphinx-themes.org/blob/master/sample-docs/kitchen-sink/index.rst))
- [`api.rst`](../demo/kitchen-sink/api.rst) ([source](https://github.com/sphinx-themes/sphinx-themes.org/blob/master/sample-docs/kitchen-sink/api.rst))
- [`lists-and-tables.rst`](../demo/kitchen-sink/lists-and-tables.rst) ([source](https://github.com/sphinx-themes/sphinx-themes.org/blob/master/sample-docs/kitchen-sink/lists-and-tables.rst))
- [`paragraph-markup.rst`](../demo/kitchen-sink/paragraph-markup.rst) ([source](https://github.com/sphinx-themes/sphinx-themes.org/blob/master/sample-docs/kitchen-sink/paragraph-markup.rst))

:::{note}
To demonstrate extra styles and syntax that is not in the Kitchen sink, use the [Theme Elements reference](../demo/theme-elements.md).
:::
