"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const sqs = require("aws-cdk-lib/aws-sqs");
const kms = require("aws-cdk-lib/aws-kms");
const s3 = require("aws-cdk-lib/aws-s3");
const assertions_1 = require("aws-cdk-lib/assertions");
const defaults = require("@aws-solutions-constructs/core");
// --------------------------------------------------------------
// Test the getter methods
// --------------------------------------------------------------
test('Test getter methods', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const filter = { prefix: 'the/place', suffix: '*.mp3' };
    const props = {
        deployDeadLetterQueue: true,
        maxReceiveCount: 0,
        queueProps: {},
        s3EventTypes: [s3.EventType.OBJECT_REMOVED],
        s3EventFilters: [filter]
    };
    const app = new lib_1.S3ToSqs(stack, 'test-s3-sqs', props);
    // Assertion 1
    expect(app.sqsQueue !== null);
    // Assertion 2
    expect(app.deadLetterQueue !== null);
    // Assertion 3
    expect(app.s3Bucket !== null);
});
// --------------------------------------------------------------
// Test deployment w/ existing queue
// --------------------------------------------------------------
test('Test deployment w/ existing queue', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const kmsKey = new kms.Key(stack, 'ExistingQueueEncryptionKey', {});
    const queue = new sqs.Queue(stack, 'existing-queue-obj', {
        queueName: 'existing-queue-obj',
        encryptionMasterKey: kmsKey
    });
    new lib_1.S3ToSqs(stack, 'test-s3-sqs', {
        existingQueueObj: queue
    });
    // Assertion 1
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("Custom::S3BucketNotifications", {
        NotificationConfiguration: {
            QueueConfigurations: [
                {
                    Events: ['s3:ObjectCreated:*'],
                    QueueArn: {
                        "Fn::GetAtt": [
                            "existingqueueobjF8AF0ED1",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
// --------------------------------------------------------------
// Test deployment w/ existing s3 Bucket
// --------------------------------------------------------------
test('Test deployment w/ existing Bucket', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const buildS3BucketResponse = defaults.buildS3Bucket(stack, {});
    new lib_1.S3ToSqs(stack, 'test-s3-sqs', {
        existingBucketObj: buildS3BucketResponse.bucket
    });
    // Assertion 1
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("Custom::S3BucketNotifications", {
        NotificationConfiguration: {
            QueueConfigurations: [
                {
                    Events: ['s3:ObjectCreated:*'],
                    QueueArn: {
                        "Fn::GetAtt": [
                            "tests3sqsqueue810CCE19",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
// --------------------------------------------------------------
// Pattern deployment w/ bucket block public access override
// --------------------------------------------------------------
test('Pattern deployment w/ bucket versioning turned off', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        bucketProps: {
            blockPublicAccess: {
                blockPublicAcls: false,
                blockPublicPolicy: true,
                ignorePublicAcls: false,
                restrictPublicBuckets: true
            }
        }
    };
    new lib_1.S3ToSqs(stack, 'test-s3-sqs', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::S3::Bucket", {
        PublicAccessBlockConfiguration: {
            BlockPublicAcls: false,
            BlockPublicPolicy: true,
            IgnorePublicAcls: false,
            RestrictPublicBuckets: true
        }
    });
});
// --------------------------------------------------------------
// Test deployment w/ specific s3 event types
// --------------------------------------------------------------
test('Test deployment w/ s3 event types and filters', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const filter = {
        prefix: 'the/place',
        suffix: '*.mp3'
    };
    const props = {
        deployDeadLetterQueue: true,
        maxReceiveCount: 0,
        queueProps: {},
        s3EventTypes: [s3.EventType.OBJECT_REMOVED],
        s3EventFilters: [filter]
    };
    new lib_1.S3ToSqs(stack, 'test-s3-sqs', props);
    // Assertion 1
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("Custom::S3BucketNotifications", {
        NotificationConfiguration: {
            QueueConfigurations: [
                {
                    Events: ['s3:ObjectRemoved:*'],
                    Filter: {
                        Key: {
                            FilterRules: [
                                {
                                    Name: 'suffix',
                                    Value: '*.mp3'
                                },
                                {
                                    Name: 'prefix',
                                    Value: 'the/place'
                                }
                            ]
                        }
                    },
                    QueueArn: {
                        "Fn::GetAtt": [
                            "tests3sqsqueue810CCE19",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
// --------------------------------------------------------------
// Test deployment w/ SSE encryption enabled using customer managed KMS CMK
// --------------------------------------------------------------
test('Test deployment w/ SSE encryption enabled using customer managed KMS CMK', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    new lib_1.S3ToSqs(stack, 'test-s3-sqs', {
        enableEncryptionWithCustomerManagedKey: true
    });
    // Assertion 1
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("Custom::S3BucketNotifications", 1);
    // Assertion 2
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "tests3sqsEncryptionKeyFD4D5946",
                "Arn"
            ]
        }
    });
    // Assertion 3
    template.hasResourceProperties('AWS::KMS::Key', {
        EnableKeyRotation: true
    });
});
// --------------------------------------------------------------
// Test bad call with existingBucket and bucketProps
// --------------------------------------------------------------
test("Test bad call with existingBucket and bucketProps", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const testBucket = new s3.Bucket(stack, 'test-bucket', {});
    const app = () => {
        // Helper declaration
        new lib_1.S3ToSqs(stack, "bad-s3-args", {
            existingBucketObj: testBucket,
            bucketProps: {
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY
            },
        });
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide bucketProps or existingBucketObj, but not both.\n');
});
// --------------------------------------------------------------
// s3 bucket with bucket, loggingBucket, and auto delete objects
// --------------------------------------------------------------
test('s3 bucket with bucket, loggingBucket, and auto delete objects', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.S3ToSqs(stack, 's3-sqs', {
        bucketProps: {
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        },
        loggingBucketProps: {
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::S3::Bucket", {
        AccessControl: "LogDeliveryWrite"
    });
    template.hasResourceProperties("Custom::S3AutoDeleteObjects", {
        ServiceToken: {
            "Fn::GetAtt": [
                "CustomS3AutoDeleteObjectsCustomResourceProviderHandler9D90184F",
                "Arn"
            ]
        },
        BucketName: {
            Ref: "s3sqsS3LoggingBucketD877FC52"
        }
    });
});
// --------------------------------------------------------------
// s3 bucket with one content bucket and no logging bucket
// --------------------------------------------------------------
test('s3 bucket with one content bucket and no logging bucket', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.S3ToSqs(stack, 's3-sqs', {
        bucketProps: {
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        },
        logS3AccessLogs: false
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::S3::Bucket", 1);
});
test('Queue is encrypted with imported CMK when set on encryptionKey prop', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const key = new kms.Key(stack, 'cmk');
    new lib_1.S3ToSqs(stack, 'test-s3-sqs', {
        encryptionKey: key
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "cmk01DE03DA",
                "Arn"
            ]
        }
    });
});
test('Queue is encrypted with provided encryptionKeyProps', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.S3ToSqs(stack, 'test-s3-sqs', {
        encryptionKeyProps: {
            alias: 'new-key-alias-from-props'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "tests3sqsEncryptionKeyFD4D5946",
                "Arn"
            ]
        }
    });
    template.hasResourceProperties('AWS::KMS::Alias', {
        AliasName: 'alias/new-key-alias-from-props',
        TargetKeyId: {
            "Fn::GetAtt": [
                "tests3sqsEncryptionKeyFD4D5946",
                "Arn"
            ]
        }
    });
});
test('Queue is encrypted with imported CMK when set on queueProps.encryptionMasterKey prop', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const key = new kms.Key(stack, 'cmk');
    new lib_1.S3ToSqs(stack, 'test-s3-sqs', {
        queueProps: {
            encryptionMasterKey: key
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "cmk01DE03DA",
                "Arn"
            ]
        }
    });
});
test('Queue is encrypted by default with Customer-managed KMS key when no other encryption properties are set', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.S3ToSqs(stack, 'test-s3-sqs', {});
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "tests3sqsEncryptionKeyFD4D5946",
                "Arn"
            ]
        }
    });
});
test('Queue is encrypted with SQS-managed KMS Key when enable encryption flag is false', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.S3ToSqs(stack, 'test-s3-sqs', {
        enableEncryptionWithCustomerManagedKey: false
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
});
//# sourceMappingURL=data:application/json;base64,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