# -*- coding: utf-8 -*-
# 
# Outsourced plotting class methods from respective parent classes
# 

# Builtin/3rd party package imports
import os
import numpy as np

# Local imports
from syncopy.shared.errors import SPYValueError, SPYError, SPYTypeError, SPYWarning
from syncopy.shared.parsers import scalar_parser
from syncopy.plotting.spy_plotting import (pltErrMsg, pltConfig, _prep_toilim_avg, 
                                           _setup_figure, _setup_colorbar, _prep_plots)

# Conditional matplotlib import
from syncopy import __plt__
if __plt__:
    import matplotlib.pyplot as plt

#: available panel settings for :func:`~syncopy.multipanelplot`
availablePanels = tuple(["channels", "trials", "tapers"])

__all__ = []


def singlepanelplot(self, trials="all", channels="all", tapers="all", 
                    toilim=None, foilim=None, avg_channels=True, avg_tapers=True,  
                    interp="spline36", cmap="plasma", vmin=None, vmax=None, 
                    title=None, grid=None, fig=None, **kwargs):
    """
    Plot contents of :class:`~syncopy.SpectralData` objects using single-panel figure(s)
    
    Please refer to :func:`syncopy.singlepanelplot` for detailed usage information. 
    
    Examples
    --------
    Show frequency range 30-80 Hz of channel `'ecog_mua2'` averaged across 
    trials 2, 4, and 6:
    
    >>> fig = spy.singlepanelplot(freqData, trials=[2, 4, 6], channels=["ecog_mua2"],
                                  foilim=[30, 80])
                                  
    Overlay channel `'ecog_mua3'` with same settings:
    
    >>> fig2 = spy.singlepanelplot(freqData, trials=[2, 4, 6], channels=['ecog_mua3'],
                                   foilim=[30, 80], fig=fig)
                                   
    Plot time-frequency contents of channel `'ecog_mua1'` present in both objects 
    `tfData1` and `tfData2` using the 'viridis' colormap, a plot grid, manually 
    defined lower and upper color value limits and no interpolation
    
    >>> fig1, fig2 = spy.singlepanelplot(tfData1, tfData2, channels=['ecog_mua1'],
                                         cmap="viridis", vmin=0.25, vmax=0.95, 
                                         interp=None, grid=True, overlay=False)
    
    Note that overlay plotting is **not** supported for time-frequency objects. 
 
    See also
    --------
    syncopy.singlepanelplot : visualize Syncopy data objects using single-panel plots
    """
    
    # Collect input arguments in dict `inputArgs` and process them
    inputArgs = locals()
    inputArgs.pop("self")
    (dimArrs, 
    dimCounts, 
    isTimeFrequency, 
    complexConversion, 
    pltDtype, 
    dataLbl) =  _prep_spectral_plots(self, "singlepanelplot", **inputArgs)
    (nTrials, nChan, nFreq, nTap) = dimCounts
    (trList, chArr, freqArr, tpArr) = dimArrs
    
    # If we're overlaying, ensure data and plot type match up    
    if hasattr(fig, "objCount"): 
        if isTimeFrequency:
            msg = "Overlay plotting not supported for time-frequency data"
            raise SPYError(msg)
        if not hasattr(fig, "spectralPlot"):
            lgl = "figure visualizing data from a Syncopy `SpectralData` object"
            act = "visualization of other Syncopy data"
            raise SPYValueError(legal=lgl, varname="fig", actual=act)
        if hasattr(fig, "multipanelplot"):
            lgl = "single-panel figure generated by `singleplot`"
            act = "multi-panel figure generated by `multipanelplot`"
            raise SPYValueError(legal=lgl, varname="fig", actual=act)

    # No time-frequency shenanigans: this is a simple power-spectrum (line-plot)
    if not isTimeFrequency:
        
        # Generic titles for figures
        overlayTitle = "Overlay of {} datasets"
    
        # Either create new figure or fetch existing
        if fig is None:
            fig, ax = _setup_figure(1, xLabel="Frequency (Hz)", yLabel=dataLbl, grid=grid)
            fig.spectralPlot = True
        else:
            ax, = fig.get_axes()        

        # Average across channels, tapers or both using local helper func
        nTime = 1
        if not avg_channels and not avg_tapers and nTap > 1:
            msg = "Either channels or trials need to be averaged for single-panel plot"
            SPYWarning(msg)
            return
        if avg_channels and not avg_tapers:
            panelTitle = "{} tapers averaged across {} channels and {} trials".format(nTap, nChan, nTrials)
            pltArr = _compute_pltArr(self, nFreq, nTap, nTime, complexConversion, pltDtype, 
                                     avg1="channel")
        if avg_tapers and not avg_channels:
            panelTitle = "{} channels averaged across {} tapers and {} trials".format(nChan, nTap, nTrials)
            pltArr = _compute_pltArr(self, nFreq, nChan, nTime, complexConversion, pltDtype, 
                                     avg1="taper")
        if avg_tapers and avg_channels:
            panelTitle = "Average of {} channels, {} tapers and {} trials".format(nChan, nTap, nTrials)
            pltArr = _compute_pltArr(self, nFreq, 1, nTime, complexConversion, pltDtype, 
                                     avg1="taper", avg2="channel")

        # Perform the actual plotting
        ax.plot(freqArr, np.log10(pltArr), label=os.path.basename(self.filename))
        ax.set_xlim([freqArr[0], freqArr[-1]])
                
        # Set plot title depending on dataset overlay
        if fig.objCount == 0:
            if title is None:
                title = panelTitle
            ax.set_title(title, size=pltConfig["singleTitleSize"])
        else:
            handles, labels = ax.get_legend_handles_labels()
            ax.legend(handles, labels)
            if title is None:
                title = overlayTitle.format(len(handles))
            ax.set_title(title, size=pltConfig["singleTitleSize"])
        
    else:
        
        # For a single-panel TF visualization, we need to average across both tapers + channels        
        if not avg_channels and (not avg_tapers and nTap > 1):
            msg = "Single-panel time-frequency visualization requires averaging " +\
                "across both tapers and channels"
            SPYWarning(msg)
            return
        
        # Compute (and verify) length of selected time intervals and assemble array for plotting
        panelTitle = "Average of {} channels, {} tapers and {} trials".format(nChan, nTap, nTrials)
        tLengths = _prep_toilim_avg(self)
        nTime = tLengths[0]
        pltArr = _compute_pltArr(self, nFreq, 1, nTime, complexConversion, pltDtype, 
                                 avg1="taper", avg2="channel")

        # Prepare figure
        fig, ax, cax = _setup_figure(1, xLabel="Time (s)", yLabel="Frequency (Hz)", 
                                     include_colorbar=True, grid=grid)
        fig.spectralPlot = True
        
        # Use `imshow` to render array as image
        time = self.time[trList[0]][self._selection.time[0]]
        ax.imshow(pltArr, origin="lower", interpolation=interp, 
                  cmap=cmap, vmin=vmin, vmax=vmax,
                  extent=(time[0], time[-1], freqArr[0], freqArr[-1]), aspect="auto")
        cbar = _setup_colorbar(fig, ax, cax, label=dataLbl.replace(" (dB)", ""))
        if title is None:
            title = panelTitle
        ax.set_title(title, size=pltConfig["singleTitleSize"])

    # Increment overlay-counter and draw figure
    fig.objCount += 1
    plt.draw()
    self._selection = None
    return fig


def multipanelplot(self, trials="all", channels="all", tapers="all", toilim=None, foilim=None,
                   avg_channels=False, avg_tapers=True, avg_trials=True, panels="channels",
                   interp="spline36", cmap="plasma", vmin=None, vmax=None,
                   title=None, grid=None, fig=None, **kwargs):
    """
    Plot contents of :class:`~syncopy.SpectralData` objects using multi-panel figure(s)
    
    Please refer to :func:`syncopy.multipanelplot` for detailed usage information. 
    
    Examples
    --------
    Use 16 panels to show frequency range 30-80 Hz of first 16 channels in `freqData` 
    averaged across trials 2, 4, and 6:
    
    >>> fig = spy.multipanelplot(freqData, trials=[2, 4, 6], channels=range(16),
                                 foilim=[30, 80], panels="channels")
                                  
    Same settings, but each panel represents a trial:
    
    >>> fig = spy.multipanelplot(freqData, trials=[2, 4, 6], channels=range(16),
                                 foilim=[30, 80], panels="trials", avg_trials=False, 
                                 avg_channels=True)
    
    Plot time-frequency contents of channels `'ecog_mua1'` and `'ecog_mua2'` of 
    `tfData` 
        
    >>> fig = spy.multipanelplot(tfData, channels=['ecog_mua1', 'ecog_mua2'])
    
    Note that multi-panel overlay plotting is **not** supported for 
    :class:`~syncopy.SpectralData` objects.
    
    See also
    --------
    syncopy.multipanelplot : visualize Syncopy data objects using multi-panel plots
    """

    # Collect input arguments in dict `inputArgs` and process them
    inputArgs = locals()
    inputArgs.pop("self")
    (dimArrs, 
    dimCounts, 
    isTimeFrequency, 
    complexConversion, 
    pltDtype, 
    dataLbl) =  _prep_spectral_plots(self, "multipanelplot", **inputArgs)
    (nTrials, nChan, nFreq, nTap) = dimCounts
    (trList, chArr, freqArr, tpArr) = dimArrs

    # No overlaying here...
    if hasattr(fig, "objCount"): 
        msg = "Overlays of multi-panel `SpectralData` plots not supported"
        raise SPYError(msg)
        
    # Ensure panel-specification makes sense and is compatible w/averaging selection        
    if not isinstance(panels, str):
        raise SPYTypeError(panels, varname="panels", expected="str")
    if panels not in availablePanels:
        lgl = "'" + "or '".join(opt + "' " for opt in availablePanels)
        raise SPYValueError(legal=lgl, varname="panels", actual=panels)
    if (panels == "channels" and avg_channels) or (panels == "trials" and avg_trials) \
        or (panels == "tapers" and avg_tapers):
        msg = "Cannot use `panels = {}` and average across {} at the same time. "
        SPYWarning(msg.format(panels, panels))
        return

    # Ensure the proper amount of averaging was specified
    avgFlags = [avg_channels, avg_trials, avg_tapers]
    if sum(avgFlags) == 0 and nTap * nTrials > 1:
        msg = "Need to average across at least one of tapers, channels or trials " +\
            "for visualization. "
        SPYWarning(msg)
        return
    if sum(avgFlags) == 3:
        msg = "Averaging across trials, channels and tapers results in " +\
            "single-panel plot. Please use `singlepanelplot` instead"
        SPYWarning(msg)
        return
    if isTimeFrequency:
        if sum(avgFlags) != 2:
            msg = "Multi-panel time-frequency visualization requires averaging across " +\
                "two out of three dimensions (tapers, channels trials)"
            SPYWarning(msg)
            return
        
    # Prepare figure (same for all cases)
    if panels == "channels":
        npanels = nChan
    elif panels == "trials":
        npanels = nTrials
    else:   # ``panels == "tapers"``
        npanels = nTap
    
    # Construct subplot panel layout or vet provided layout
    nrow = kwargs.get("nrow", None)
    ncol = kwargs.get("ncol", None)
    if not isTimeFrequency:
        fig, ax_arr = _setup_figure(npanels, nrow=nrow, ncol=ncol,
                                    xLabel="Frequency (Hz)",
                                    yLabel=dataLbl, grid=grid,
                                    include_colorbar=False, 
                                    sharex=True, sharey=True)
    else:
        fig, ax_arr, cax = _setup_figure(npanels, nrow=nrow, ncol=ncol,
                                         xLabel="Time (s)",
                                         yLabel="Frequency (Hz)", grid=grid,
                                         include_colorbar=True, 
                                         sharex=True, sharey=True)
    
    # Monkey-patch object-counter to newly created figure
    fig.spectralPlot = True

    # Start with the "simple" case: "regular" spectra, no time involved
    if not isTimeFrequency:

        # We're not dealing w/TF data here
        nTime = 1
        N = 1

        # For each panel stratification, set corresponding positional and 
        # keyword args for iteratively calling `_compute_pltArr`        
        if panels == "channels":
            
            panelVar = "channel"
            panelValues = chArr
            panelTitles = chArr
            
            if not avg_trials and avg_tapers:
                avgDim1 = "taper"
                avgDim2 = None
                innerVar = "trial"
                innerValues = trList
                majorTitle = "{} trials averaged across {} tapers".format(nTrials, nTap)
                showLegend = True
            elif avg_trials and not avg_tapers:
                avgDim1 = None
                avgDim2 = None
                innerVar = "taper"
                innerValues = tpArr
                majorTitle = "{} tapers averaged across {} trials".format(nTap, nTrials)
                showLegend = True
            else:   # `avg_trials` and `avg_tapers`
                avgDim1 = "taper"
                avgDim2 = None
                innerVar = "trial"
                innerValues = ["all"]
                majorTitle = " Average of {} tapers and {} trials".format(nTap, nTrials)
                showLegend = False
        
        elif panels == "trials":
            
            panelVar = "trial"
            panelValues = trList
            panelTitles = ["Trial #{}".format(trlno) for trlno in trList]
            
            if not avg_channels and avg_tapers:
                avgDim1 = "taper"
                avgDim2 = None
                innerVar = "channel"
                innerValues = chArr
                majorTitle = "{} channels averaged across {} tapers".format(nChan, nTap)
                showLegend = True
            elif avg_channels and not avg_tapers:
                avgDim1 = "channel"
                avgDim2 = None
                innerVar = "taper"
                innerValues = tpArr
                majorTitle = "{} tapers averaged across {} channels".format(nTap, nChan)
                showLegend = True
            else:   # `avg_channels` and `avg_tapers`
                avgDim1 = "taper"
                avgDim2 = "channel"
                innerVar = "trial"
                innerValues = ["all"]
                majorTitle = " Average of {} channels and {} tapers".format(nChan, nTap)
                showLegend = False
                    
        else:  # panels = "tapers"
            
            panelVar = "taper"
            panelValues = tpArr
            panelTitles = ["Taper #{}".format(tpno) for tpno in tpArr]
            
            if not avg_trials and avg_channels:
                avgDim1 = "channel"
                avgDim2 = None
                innerVar = "trial"
                innerValues = trList
                majorTitle = "{} trials averaged across {} channels".format(nTrials, nChan)
                showLegend = True
            elif avg_trials and not avg_channels:
                avgDim1 = None
                avgDim2 = None
                innerVar = "channel"
                innerValues = chArr
                majorTitle = "{} channels averaged across {} trials".format(nChan, nTrials)
                showLegend = True
            else:   # `avg_trials` and `avg_channels`
                avgDim1 = "channel"
                avgDim2 = None
                innerVar = "trial"
                innerValues = ["all"]
                majorTitle = " Average of {} channels and {} trials".format(nChan, nTrials)
                showLegend = False

        # Loop over panels, within each panel, loop over `innerValues` to (potentially)
        # plot multiple spectra per panel        
        kwargs = {"avg1": avgDim1, "avg2": avgDim2}
        for panelCount, panelVal in enumerate(panelValues):
            kwargs[panelVar] = panelVal
            for innerVal in innerValues:
                kwargs[innerVar] = innerVal
                pltArr = _compute_pltArr(self, nFreq, N, nTime, complexConversion, pltDtype, **kwargs)
                ax_arr[panelCount].plot(freqArr, np.log10(pltArr), 
                                        label=innerVar.capitalize() + " " + str(innerVal))
            ax_arr[panelCount].set_title(panelTitles[panelCount], size=pltConfig["multiTitleSize"])
        if showLegend:
            handles, labels = ax_arr[0].get_legend_handles_labels()
            ax_arr[0].legend(handles, labels)
        if title is None:
            fig.suptitle(majorTitle, size=pltConfig["singleTitleSize"])

    # Now, multi-panel time-frequency visualizations
    else:
        
        # Compute (and verify) length of selected time intervals
        tLengths = _prep_toilim_avg(self)
        nTime = tLengths[0]
        time = self.time[trList[0]][self._selection.time[0]]
        N = 1
        
        if panels == "channels":
            panelVar = "channel"
            panelValues = chArr
            panelTitles = chArr
            majorTitle = " Average of {} tapers and {} trials".format(nTap, nTrials)
            avgDim1 = "taper"
            avgDim2 = None
            
        elif panels == "trials":
            panelVar = "trial"
            panelValues = trList
            panelTitles = ["Trial #{}".format(trlno) for trlno in trList]
            majorTitle = " Average of {} channels and {} tapers".format(nChan, nTap)
            avgDim1 = "taper"
            avgDim2 = "channel"

        else:  # panels = "tapers"
            panelVar = "taper"
            panelValues = tpArr
            panelTitles = ["Taper #{}".format(tpno) for tpno in tpArr]
            majorTitle = " Average of {} channels and {} trials".format(nChan, nTrials)
            avgDim1 = "channel"
            avgDim2 = None

        # Loop over panels, within each panel, loop over `innerValues` to (potentially)
        # plot multiple spectra per panel        
        kwargs = {"avg1": avgDim1, "avg2": avgDim2}
        vmins = []
        vmaxs = []
        for panelCount, panelVal in enumerate(panelValues):
            kwargs[panelVar] = panelVal
            pltArr = _compute_pltArr(self, nFreq, N, nTime, complexConversion, pltDtype, **kwargs)
            vmins.append(pltArr.min())
            vmaxs.append(pltArr.max())
            ax_arr[panelCount].imshow(pltArr, origin="lower", interpolation=interp, cmap=cmap, 
                                      extent=(time[0], time[-1], freqArr[0], freqArr[-1]), 
                                      aspect="auto")
            ax_arr[panelCount].set_title(panelTitles[panelCount], size=pltConfig["multiTitleSize"])

        # Render colorbar
        if vmin is None:
            vmin = min(vmins)
        if vmax is None:
            vmax = max(vmaxs)
        cbar = _setup_colorbar(fig, ax_arr, cax, label=dataLbl.replace(" (dB)", ""), 
                               outline=False, vmin=vmin, vmax=vmax)
        if title is None:
            fig.suptitle(majorTitle, size=pltConfig["singleTitleSize"])

    # Increment overlay-counter and draw figure
    fig.objCount += 1
    plt.draw()
    self._selection = None
    return fig
    
def _compute_pltArr(self, nFreq, N, nTime, complexConversion, pltDtype,
                    avg1="channel", avg2=None, trial="all", channel="all", 
                    freq="all", taper="all"):
    """
    Local helper that extracts/averages data from :class:`~syncopy.SpectralData` object
    
    Parameters
    ----------
    self : :class:`~syncopy.SpectralData` object
        Syncopy :class:`~syncopy.SpectralData` object that is being processed by 
        the respective :meth:`.singlepanelplot` or :meth:`.multipanelplot` class methods
        defined in this module. 
    nFreq : int
        Number of frequencies of interest
    N : int
        Size of free dimension post averaging. Depending on `avg1` and `avg2` 
        can be either `nChan`, `nTap` or 1
    nTime : int
        Number of time-points of interest. If object does not contain time-frequency
        data, `nTime` has to be 1
    complexConversion : callable
        Automatically set by :meth:`~syncopy.plotting._plot_spectral._prep_spectral_plots` 
        to (potentially) convert complex Fourier coefficients to float. 
    pltDtype : str or :class:`numpy.dtype`
        Automatically set by :meth:`~syncopy.plotting._plot_spectral._prep_spectral_plots`: 
        numeric type of (potentially converted) complex Fourier coefficients. 
    avg1 : str or None
        First dimension for averaging. If `None`, no mean-value is computed. Otherwise, 
        `avg1` can be either `"channel"` or `"taper"`. 
    avg2 : str or None
        Second dimension for averaging. If `None`, no mean-value is computed. Otherwise, 
        `avg2` can be either `"channel"` or `"taper"`. 
    trial : str or list
        Either list of trial indices or `"all"`; set by 
        :meth:`~syncopy.plotting._plot_spectral._prep_spectral_plots`
    channel : str or :class:`numpy.ndarray`
        Either array of channel specifiers or `"all"`; set by 
        :meth:`~syncopy.plotting._plot_spectral._prep_spectral_plots`
    freq : str or :class:`numpy.ndarray`
        Either array of frequency specifiers or `"all"`; set by 
        :meth:`~syncopy.plotting._plot_spectral._prep_spectral_plots`
    taper : str or :class:`numpy.ndarray`
        Either array of taper specifiers or `"all"`; set by 
        :meth:`~syncopy.plotting._plot_spectral._prep_spectral_plots`

    Returns
    -------
    pltArr : 1D, 2D or 3D :class:`numpy.ndarray`
        Extracted/averaged data ready for plotting; if the :class:`~syncopy.SpectralData`
        input object contains time-frequency data, `pltArr` is a three-dimensional
        array of shape ``(nFreq, nTime, N)``, otherwise `pltArr` is two-dimensional
        with shape ``(nFreq, N)`` for ``N > 1``, or a one-dimensional ``(nFreq,)``
        array if ``N = 1``. 

    Notes
    -----
    This is an auxiliary method that is intended purely for internal use. Please
    refer to the user-exposed methods :func:`~syncopy.singlepanelplot` and/or
    :func:`~syncopy.multipanelplot` to actually generate plots of Syncopy data objects. 
    """
    
    # Prepare indexing list respecting potential non-default `dimord`s
    idx = [slice(None), slice(None), slice(None), slice(None)]
    timeIdx = self.dimord.index("time")
    chanIdx = self.dimord.index("channel")
    freqIdx = self.dimord.index("freq")
    taperIdx = self.dimord.index("taper")
    
    if trial == "all":
        trList = self._selection.trials
    else:
        trList = [trial]
    nTrls = len(trList)
    useFancy = self._selection._useFancy
    if channel == "all":
        idx[chanIdx] = self._selection.channel
    else:
        idx[chanIdx] = np.where(self.channel == channel)[0]
        useFancy = True
    if freq ==  "all":       
        idx[freqIdx] = self._selection.freq
    else:
        idx[freqIdx] = np.where(self.freq == freq)[0]
        useFancy = True
    if taper == "all":
        idx[taperIdx] = self._selection.taper
    else:
        idx[taperIdx] = [taper]
        useFancy = True
        
    if nTime == 1:
        pltArr = np.zeros((nFreq, N), dtype=pltDtype).squeeze()         # `squeeze` in case `N = 1`
    else:
        pltArr = np.zeros((nFreq, nTime, N), dtype=pltDtype).squeeze()  # `squeeze` for `singlepanelplot`

    for tk, trlno in enumerate(trList):
        trlArr = complexConversion(self._get_trial(trlno))
        idx[timeIdx] = self._selection.time[tk]
        if not useFancy:
            trlArr = trlArr[tuple(idx)]
        else:
            trlArr = trlArr[idx[0], ...][:, idx[1], ...][:, :, idx[2], :][..., idx[3]]
        if avg1:
            trlArr = trlArr.mean(axis=self.dimord.index(avg1), keepdims=True)
        if avg2:
            trlArr = trlArr.mean(axis=self.dimord.index(avg2), keepdims=True)
        pltArr += np.swapaxes(trlArr, freqIdx, 0).squeeze()
    return pltArr / len(trList)


def _prep_spectral_plots(self, name, **inputArgs):
    """
    Local helper that performs sanity checks and sets up data selection
    
    Parameters
    ----------
    self : :class:`~syncopy.SpectralData` object
        Syncopy :class:`~syncopy.SpectralData` object that is being processed by 
        the respective :meth:`.singlepanelplot` or :meth:`.multipanelplot` class methods
        defined in this module. 
    name : str
        Name of caller (i.e., "singlepanelplot" or "multipanelplot")
    inputArgs : dict
        Input arguments of caller (i.e., :meth:`.singlepanelplot` or :meth:`.multipanelplot`)
        collected in dictionary
        
    Returns
    -------
    dimArrs : tuple
        Four-element tuple containing (in this order): `trList`, list of (selected) 
        trials to visualize, `chArr`, 1D :class:`numpy.ndarray` of channel specifiers
        based on provided user selection, `freqArr`, 1D :class:`numpy.ndarray` of 
        frequency specifiers based on provided user selection, `tpArr`, 
        1D :class:`numpy.ndarray` of taper specifiers based on provided user selection. 
        Note that `"all"` and `None` selections are converted to arrays ready for
        indexing. 
    dimCounts : tuple
        Four-element tuple holding sizes of corresponding selection arrays comprised
        in `dimArrs`. Elements are (in this order): number of (selected) trials 
        `nTrials`, number of (selected) channels `nChan`, number of (selected) 
        frequencies `nFreq`, number of (selected) tapers `nTap`. 
    isTimeFrequency : bool
        If `True`, input object contains time-frequency data, `False` otherwise
    complexConversion : callable
        Lambda function that performs complex-to-float conversion of Fourier 
        coefficients (if necessary). 
    pltDtype : str or :class:`numpy.dtype`
        Numeric type of (potentially converted) complex Fourier coefficients. 
    dataLbl : str
        Caption for y-axis or colorbar (depending on value of `isTimeFrequency`). 
        
    Notes
    -----
    This is an auxiliary method that is intended purely for internal use. Please
    refer to the user-exposed methods :func:`~syncopy.singlepanelplot` and/or
    :func:`~syncopy.multipanelplot` to actually generate plots of Syncopy data objects. 
        
    See also
    --------
    :meth:`syncopy.plotting.spy_plotting._prep_plots` : General basic input parsing for all Syncopy plotting routines
    """
    
    # Basic sanity checks for all plotting routines w/any Syncopy object
    _prep_plots(self, name, **inputArgs)
    
    # Ensure our binary flags are actually binary
    if not isinstance(inputArgs["avg_channels"], bool):
        raise SPYTypeError(inputArgs["avg_channels"], varname="avg_channels", expected="bool")
    if not isinstance(inputArgs["avg_tapers"], bool):
        raise SPYTypeError(inputArgs["avg_tapers"], varname="avg_tapers", expected="bool")
    if not isinstance(inputArgs.get("avg_trials", True), bool):
        raise SPYTypeError(inputArgs["avg_trials"], varname="avg_trials", expected="bool")
    
    # Pass provided selections on to `Selector` class which performs error 
    # checking and generates required indexing arrays
    self._selection = {"trials": inputArgs["trials"], 
                       "channels": inputArgs["channels"], 
                       "tapers": inputArgs["tapers"],
                       "toilim": inputArgs["toilim"],
                       "foilim": inputArgs["foilim"]}
    
    # Ensure any optional keywords controlling plotting appearance make sense
    if inputArgs["title"] is not None:
        if not isinstance(inputArgs["title"], str):
            raise SPYTypeError(inputArgs["title"], varname="title", expected="str")
    if inputArgs["grid"] is not None:
        if not isinstance(inputArgs["grid"], bool):
            raise SPYTypeError(inputArgs["grid"], varname="grid", expected="bool")

    # Get trial/channel/taper count and collect quantities in tuple
    trList = self._selection.trials
    nTrials = len(trList)
    chArr = self.channel[self._selection.channel]
    nChan = chArr.size
    freqArr = self.freq[self._selection.freq]
    nFreq = freqArr.size
    tpArr = np.arange(self.taper.size)[self._selection.taper]
    nTap = tpArr.size
    dimCounts = (nTrials, nChan, nFreq, nTap)
    dimArrs = (trList, chArr, freqArr, tpArr)

    # Determine whether we're dealing w/tf data
    isTimeFrequency = False
    if any([t.size > 1 for t in self.time]):
        isTimeFrequency = True
        
    # Ensure provided min/max range for plotting TF data makes sense
    vminmax = False
    if inputArgs.get("vmin", None) is not None:
        try:
            scalar_parser(inputArgs["vmin"], varname="vmin")
        except Exception as exc:
            raise exc 
        vminmax = True
    if inputArgs.get("vmax", None) is not None:
        try:
            scalar_parser(inputArgs["vmax"], varname="vmax")
        except Exception as exc:
            raise exc 
        vminmax = True
    if inputArgs.get("vmin", None) and inputArgs.get("vmax", None):
        if inputArgs["vmin"] >= inputArgs["vmax"]:
            lgl = "minimal data range bound to be less than provided maximum "
            act = "vmax < vmin"
            raise SPYValueError(legal=lgl, varname="vmin/vamx", actual=act)
    if vminmax and not isTimeFrequency:
        msg = "`vmin` and `vmax` is only used for time-frequency visualizations"
        SPYWarning(msg)
        
    # Check for complex entries in data and set datatype for plotting arrays 
    # constructed below (always use floats w/same precision as data)
    if "complex" in self.data.dtype.name:
        msg = "Found complex Fourier coefficients - visualization will use absolute values."
        SPYWarning(msg)
        complexConversion = lambda x: np.absolute(x).real
        pltDtype = "f{}".format(self.data.dtype.itemsize)
        dataLbl = "Absolute Frequency (dB)"
    else:
        complexConversion = lambda x: x
        pltDtype = self.data.dtype
        dataLbl = "Power (dB)"
    
    return dimArrs, dimCounts, isTimeFrequency, complexConversion, pltDtype, dataLbl
