"""Option Dataclasses"""

import re
from dataclasses import dataclass
from typing import Optional, Union


@dataclass
class _DefaultField:
    """Field info in options classes"""

    name: str
    type: type
    default: object


@dataclass
class Options:
    """Base Class for differnt types of command line options"""

    def __init__(self, **kwargs):
        default = self.defaults()
        for option in kwargs:
            if option in default:
                setattr(self, option, kwargs[option])

    @staticmethod
    def convert_name(value: str) -> str:
        """Add flag marker and replace underscores with dashes in name"""
        return "--" + value.replace("_", "-")

    # pylint: disable=no-member
    @classmethod
    def defaults(cls) -> dict[_DefaultField]:
        """Get list of fields for an Options dataclass"""
        values = {}
        for field in cls.__dataclass_fields__.values():
            values[field.name] = _DefaultField(field.name, field.type, field.default)
        return values

    def parse(self) -> list[Optional[Union[str, int]]]:
        """Turn options into list for argv

        :return: options for the command line
        :rtype: list[Optional[Union[str, int]]]
        """
        args = []
        # pylint: disable=no-member
        for key, value in self.__dataclass_fields__.items():
            attr = getattr(self, key)
            if attr is not None and value.default != attr:
                flag = self.convert_name(key)
                if value.type is bool:
                    if attr is not value.default:
                        args.append(flag)
                elif value.type is list:
                    for val in attr:
                        args.extend([flag, val])
                else:
                    args.extend([flag, attr])
        return args


# pylint: disable=too-many-instance-attributes
@dataclass
class CommonOptions(Options):
    """Common Options for all Borg commands"""

    # work on log level CRITICAL
    critical: bool = False
    # work on log level ERROR
    error: bool = False
    # work on log level WARNING
    warning: bool = False
    # work on log level INFO
    info: bool = False
    # work on log level INFO
    verbose: bool = False
    # work on log level DEBUG
    debug: bool = False
    # show progress information
    progress: bool = False
    #: output one JSON object per log line instead of formatted text.
    log_json: bool = False
    # wait at most SECONDS for acquiring a repository/cache lock (default: 1).
    lock_wait: int = None
    # bypass locking mechanism
    bypass_lock: bool = False
    # show/log the borg version
    show_version: bool = False
    # show/log the return code (rc)
    show_rc: bool = False
    # set umask to M (local and remote, default: 0077)
    umask: str = None
    # use PATH as borg executable on the remote (default: “borg”)
    remote_path: str = None
    # set remote network upload rate limit in kiByte/s (default: 0=unlimited)
    remote_ratelimit: int = None
    # treat part files like normal files (e.g. to list/extract them)
    consider_part_files: bool = False
    # write execution profile in Borg format into FILE.
    # For local use a Python-compatible file can be generated by suffixing FILE with “.pyprof”
    debug_profile: str = None
    # Use this command to connect to the ‘borg serve’ process (default: ‘ssh’)
    rsh: str = None

    # pylint: disable=useless-super-delegation
    def __init__(self, **kwargs):
        super().__init__(**kwargs)

    def __post_init__(self):
        if self.umask and not re.match(r"^[0-9]{4}", self.umask):
            raise ValueError("umask must be in format 0000 permission code, eg: 0077")


@dataclass
class ExclusionOptions(Options):
    """Options for excluding various files from backup"""

    # exclude paths matching PATTERN
    exclude: list[str] = None
    # read exclude patterns from EXCLUDEFILE, one per line
    exclude_from: str = None
    # experimental: include/exclude paths matching PATTERN
    pattern: str = None
    # experimental: read include/exclude patterns from PATTERNFILE, one per line
    patterns_from: str = None

    # pylint: disable=useless-super-delegation
    def __init__(self, **kwargs):
        super().__init__(**kwargs)


@dataclass
class ExclusionInput(ExclusionOptions):
    """Exclusion Options when inputing data to the archive"""

    # exclude directories that contain a CACHEDIR.TAG file
    # (http://www.bford.info/cachedir/spec.html)
    exclude_caches: bool = False
    # exclude directories that are tagged by containing a filesystem
    # object with the given NAME
    exclude_if_present: list[str] = None
    # if tag objects are specified with --exclude-if-present, don’t omit the tag
    # objects themselves from the backup archive
    keep_exclude_tags: bool = False
    # alternate to keep_exclude_tags
    keep_tag_files: bool = False
    # exclude files flagged NODUMP
    exclude_nodump: bool = False

    # pylint: disable=useless-super-delegation
    def __init__(self, **kwargs):
        super().__init__(**kwargs)

    def __post_init__(self):
        if isinstance(self.exclude, str):
            self.exclude = [self.exclude]
        if isinstance(self.exclude_if_present, str):
            self.exclude_if_present = [self.exclude_if_present]


@dataclass
class ExclusionOutput(ExclusionOptions):
    """Exclusion Options when outputing data in the archive"""

    # Remove the specified number of leading path elements.
    # Paths with fewer elements will be silently skipped
    strip_componts: int = None

    # pylint: disable=useless-super-delegation
    def __init__(self, **kwargs):
        super().__init__(**kwargs)


@dataclass
class FilesystemOptions(Options):
    """Options for how to handle filesystem attributes"""

    # stay in the same file system and do not store mount points of other file systems.
    # This might behave different from your expectations, see the docs.
    one_file_system: bool = False
    # only store numeric user and group identifiers
    numeric_owner: bool = False
    # do not store atime into archive
    noatime: bool = False
    # do not store ctime into archive
    noctime: bool = False
    # do not store birthtime (creation date) into archive
    nobirthtime: bool = False
    # do not read and store bsdflags (e.g. NODUMP, IMMUTABLE) into archive
    nobsdflags: bool = False
    # do not read and store ACLs into archive
    noacls: bool = False
    # do not read and store xattrs into archive
    noxattrs: bool = False
    # ignore inode data in the file metadata cache used to detect unchanged files.
    ignore_inode: bool = False
    # operate files cache in MODE. default: ctime,size,inode
    files_cache: str = None
    # open and read block and char device files as well as FIFOs as if they were regular files.
    # Also follows symlinks pointing to these kinds of files.
    read_special: bool = False

    # pylint: disable=useless-super-delegation
    def __init__(self, **kwargs):
        super().__init__(**kwargs)


@dataclass
class ArchiveOptions(Options):
    """Options related to the archive"""

    # pylint: disable=useless-super-delegation
    def __init__(self, **kwargs):
        super().__init__(**kwargs)


@dataclass
class ArchiveInput(ArchiveOptions):
    """Archive Options when inputing data to the archive"""

    # add a comment text to the archive
    comment: str = None
    # manually specify the archive creation date/time (UTC, yyyy-mm-ddThh:mm:ss format).
    # Alternatively, give a reference file/directory.
    timestamp: str = None
    # write checkpoint every SECONDS seconds (Default: 1800)
    checkpoint_interval: int = None
    # specify the chunker parameters (CHUNK_MIN_EXP, CHUNK_MAX_EXP, HASH_MASK_BITS,
    # HASH_WINDOW_SIZE). default: 19,23,21,4095
    chunker_params: str = None
    # select compression algorithm, see the output of the “borg help
    # compression” command for details.
    compression: str = None

    # pylint: disable=useless-super-delegation
    def __init__(self, **kwargs):
        super().__init__(**kwargs)


@dataclass
class ArchivePattern(ArchiveOptions):
    """Archive Options when outputing data in the archive"""

    # only consider archive names starting with this prefix.
    prefix: str = None
    # only consider archive names matching the glob.
    # sh: rules apply, see “borg help patterns”. --prefix
    # and --glob-archives are mutually exclusive.
    glob_archives: str = None

    # pylint: disable=useless-super-delegation
    def __init__(self, **kwargs):
        super().__init__(**kwargs)


@dataclass
class ArchiveOutput(ArchivePattern):
    """Archive options when filtering output"""

    # Comma-separated list of sorting keys; valid keys
    # are: timestamp, name, id; default is: timestamp
    sort_by: str = None
    # consider first N archives after other filters were applied
    first: int = None
    # consider last N archives after other filters were applied
    last: int = None
    # pylint: disable=useless-super-delegation
    def __init__(self, **kwargs):
        super().__init__(**kwargs)
