#!/usr/bin/env python3
"""
Configuration Generator CLI for MCP Proxy Adapter
Generates configurations based on command line flags.

Author: Vasiliy Zdanovskiy
email: vasilyvz@gmail.com
"""
import argparse
import json
import sys
from pathlib import Path
from typing import Dict, Any, Optional

# Add the project root to the path to import config_builder
sys.path.insert(0, str(Path(__file__).parent.parent.parent))

from mcp_proxy_adapter.examples.config_builder import ConfigBuilder, ConfigFactory, Protocol, AuthMethod


def create_config_from_flags(
    protocol: str,
    token: bool = False,
    roles: bool = False,
    host: str = "127.0.0.1",
    port: int = 8000,
    cert_dir: str = "./certs",
    key_dir: str = "./keys",
    output_dir: str = "./configs"
) -> Dict[str, Any]:
    """
    Create configuration based on command line flags.
    
    Args:
        protocol: Protocol type (http, https, mtls)
        token: Enable token authentication
        roles: Enable role-based access control
        host: Server host
        port: Server port
        cert_dir: Certificate directory
        key_dir: Key directory
        output_dir: Output directory for configs
        
    Returns:
        Configuration dictionary
    """
    # Use the simplified config builder
    from config_builder import create_config_from_flags as simple_create_config
    
    return simple_create_config(protocol, token, roles, port)


def save_config(config: Dict[str, Any], filename: str, output_dir: str) -> Path:
    """Save configuration to file."""
    output_path = Path(output_dir)
    output_path.mkdir(parents=True, exist_ok=True)
    
    config_file = output_path / f"{filename}.json"
    with open(config_file, 'w', encoding='utf-8') as f:
        json.dump(config, f, indent=2, ensure_ascii=False)
    
    return config_file


def create_full_config_with_all_options(host: str = "127.0.0.1", port: int = 20000) -> Dict[str, Any]:
    """
    Create a full configuration with all options enabled but set to HTTP base.
    This allows testing all features by enabling different sections.
    
    Args:
        host: Server host
        port: Server port
        
    Returns:
        Full configuration dictionary with all options
    """
    builder = ConfigBuilder()
    
    # Set server configuration
    builder.set_server(host=host, port=port)
    
    # Set to HTTP base (no SSL by default)
    builder.set_protocol(Protocol.HTTP)
    
    # Enable all authentication options but keep them disabled by default
    api_keys = {
        "admin": "admin-secret-key",
        "user": "user-secret-key", 
        "readonly": "readonly-secret-key"
    }
    roles = {
        "admin": ["read", "write", "delete", "admin"],
        "user": ["read", "write"],
        "readonly": ["read"]
    }
    
    # Set up all authentication methods but keep security disabled
    builder.set_auth(AuthMethod.TOKEN_ROLES, api_keys=api_keys, roles=roles)
    
    # Disable security by default (will be enabled in tests)
    config = builder.build()
    config["security"]["enabled"] = False
    
    # Add protocol variants for easy switching
    config["protocol_variants"] = {
        "http": {"server": {"protocol": "http"}},
        "https": {"server": {"protocol": "https"}},
        "mtls": {"server": {"protocol": "mtls"}}
    }
    
    # Add authentication configurations for easy switching
    config["auth_variants"] = {
        "none": {"security": {"enabled": False}},
        "token": {
            "security": {
                "enabled": True,
                "tokens": api_keys,
                "roles": roles,
                "roles_file": None
            }
        },
        "token_roles": {
            "security": {
                "enabled": True,
                "tokens": api_keys,
                "roles": roles,
                "roles_file": "configs/roles.json"
            }
        }
    }
    
    return config


def generate_all_configs(output_dir: str = "./configs", host: str = "127.0.0.1") -> None:
    """Generate all standard configurations."""
    configs = [
        # HTTP configurations
        ("http", False, False, 20000),
        ("http", True, True, 20001),  # token=True always includes roles
        ("http", True, True, 20002),  # token_roles is same as token now
        
        # HTTPS configurations
        ("https", False, False, 20003),
        ("https", True, True, 20004),  # token=True always includes roles
        ("https", True, True, 20005),  # token_roles is same as token now
        
        # mTLS configurations
        ("mtls", False, False, 20006),
        ("mtls", True, True, 20007),  # token=True always includes roles
        ("mtls", True, True, 20008),  # token_roles is same as token now
    ]
    
    print("🔧 Generating MCP Proxy Adapter configurations...")
    print("=" * 60)
    
    generated_files = []
    
    for protocol, token, roles, port in configs:
        # Create configuration name
        name_parts = [protocol]
        if token:
            name_parts.append("token")
        if roles:
            name_parts.append("roles")
        
        config_name = "_".join(name_parts)
        
        # Generate configuration
        config = create_config_from_flags(
            protocol=protocol,
            token=token,
            roles=roles,
            host=host,
            port=port,
            output_dir=output_dir
        )
        
        # Save configuration
        config_file = save_config(config, config_name, output_dir)
        generated_files.append(config_file)
        
        print(f"✅ Created {config_name}.json (port {port})")
    
    # Create roles.json file if any role-based configs were generated
    roles_config = {
        "enabled": True,
        "default_policy": {
            "deny_by_default": False,
            "require_role_match": False,
            "case_sensitive": False,
            "allow_wildcard": False
        },
        "roles": {
            "admin": ["read", "write", "delete", "admin"],
            "user": ["read", "write"],
            "readonly": ["read"],
            "guest": ["read"],
            "proxy": ["read", "write"]
        },
        "permissions": {
            "read": ["GET"],
            "write": ["POST", "PUT", "PATCH"],
            "delete": ["DELETE"],
            "admin": ["*"]
        }
    }
    
    roles_file = Path(output_dir) / "roles.json"
    with open(roles_file, 'w', encoding='utf-8') as f:
        json.dump(roles_config, f, indent=2, ensure_ascii=False)
    print(f"✅ Created roles.json")
    
    print(f"\n🎉 Generated {len(generated_files)} configurations in {output_dir}/")
    print("\n📋 Generated configurations:")
    for config_file in generated_files:
        print(f"  - {config_file.name}")


def main():
    """Main CLI function."""
    parser = argparse.ArgumentParser(
        description="MCP Proxy Adapter Configuration Generator",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Generate all standard configurations
  python generate_config.py --all
  
  # Generate full config with all options (HTTP base)
  python generate_config.py --full-config
  
  # Generate specific configuration
  python generate_config.py --protocol https --token --roles --port 8080
  
  # Generate HTTP configuration with token auth
  python generate_config.py --protocol http --token
  
  # Generate mTLS configuration with roles
  python generate_config.py --protocol mtls --roles
        """
    )
    
    # Configuration options
    parser.add_argument("--protocol", choices=["http", "https", "mtls"], 
                       help="Protocol type (http, https, mtls)")
    parser.add_argument("--token", action="store_true", 
                       help="Enable token authentication")
    parser.add_argument("--roles", action="store_true", 
                       help="Enable role-based access control")
    parser.add_argument("--all", action="store_true", 
                       help="Generate all standard configurations")
    parser.add_argument("--full-config", action="store_true", 
                       help="Generate full config with all options (HTTP base)")
    
    # Server configuration
    parser.add_argument("--host", default="127.0.0.1", 
                       help="Server host (default: 127.0.0.1)")
    parser.add_argument("--port", type=int, default=8000, 
                       help="Server port (default: 8000)")
    
    # Paths
    parser.add_argument("--cert-dir", default="./certs", 
                       help="Certificate directory (default: ./certs)")
    parser.add_argument("--key-dir", default="./keys", 
                       help="Key directory (default: ./keys)")
    parser.add_argument("--output-dir", default="./configs", 
                       help="Output directory (default: ./configs)")
    
    # Output options
    parser.add_argument("--output", "-o", 
                       help="Output filename (without extension)")
    parser.add_argument("--stdout", action="store_true", 
                       help="Output to stdout instead of file")
    
    args = parser.parse_args()
    
    try:
        if args.all:
            # Generate all configurations
            generate_all_configs(
                output_dir=args.output_dir,
                host=args.host
            )
        elif args.full_config:
            # Generate full config with all options
            config = create_full_config_with_all_options(
                host=args.host,
                port=args.port
            )
            
            if args.stdout:
                # Output to stdout
                print(json.dumps(config, indent=2, ensure_ascii=False))
            else:
                # Save to file
                filename = args.output or "full_config"
                config_file = save_config(config, filename, args.output_dir)
                print(f"✅ Full configuration saved to: {config_file}")
        elif args.protocol:
            # Generate specific configuration
            config = create_config_from_flags(
                protocol=args.protocol,
                token=args.token,
                roles=args.roles,
                host=args.host,
                port=args.port,
                cert_dir=args.cert_dir,
                key_dir=args.key_dir,
                output_dir=args.output_dir
            )
            
            if args.stdout:
                # Output to stdout
                print(json.dumps(config, indent=2, ensure_ascii=False))
            else:
                # Save to file
                if args.output:
                    filename = args.output
                else:
                    # Generate filename from flags
                    name_parts = [args.protocol]
                    if args.token:
                        name_parts.append("token")
                    if args.roles:
                        name_parts.append("roles")
                    filename = "_".join(name_parts)
                
                config_file = save_config(config, filename, args.output_dir)
                print(f"✅ Configuration saved to: {config_file}")
        else:
            parser.print_help()
            return 1
        
        return 0
        
    except Exception as e:
        print(f"❌ Error: {e}", file=sys.stderr)
        return 1


if __name__ == "__main__":
    sys.exit(main())
