"""
Built-in Command Hooks
This module demonstrates hooks for built-in commands in the full application example.
Author: Vasiliy Zdanovskiy
email: vasilyvz@gmail.com
"""

import logging
from typing import Dict, Any, Optional
from datetime import datetime

logger = logging.getLogger(__name__)


class BuiltinCommandHooks:
    """Hooks for built-in commands."""

    @staticmethod
    def before_echo_command(params: Dict[str, Any]) -> Dict[str, Any]:
        """Hook executed before echo command."""
        logger.info(f"🔧 Built-in hook: before_echo_command with params: {params}")
        # Add timestamp to message
        if "message" in params:
            timestamp = datetime.now().isoformat()
            params["message"] = f"[{timestamp}] {params['message']}"
        return params

    @staticmethod
    def after_echo_command(result: Dict[str, Any]) -> Dict[str, Any]:
        """Hook executed after echo command."""
        logger.info(f"🔧 Built-in hook: after_echo_command with result: {result}")
        # Add hook metadata
        result["hook_metadata"] = {
            "hook_type": "builtin_after_echo",
            "timestamp": datetime.now().isoformat(),
            "processed": True,
        }
        return result

    @staticmethod
    def before_health_command(params: Dict[str, Any]) -> Dict[str, Any]:
        """Hook executed before health command."""
        logger.info(f"🔧 Built-in hook: before_health_command with params: {params}")
        # Add custom health check parameters
        params["include_detailed_info"] = True
        params["check_dependencies"] = True
        return params

    @staticmethod
    def after_health_command(result: Dict[str, Any]) -> Dict[str, Any]:
        """Hook executed after health command."""
        logger.info(f"🔧 Built-in hook: after_health_command with result: {result}")
        # Add custom health metrics
        if "status" in result and result["status"] == "healthy":
            result["custom_metrics"] = {
                "uptime": "24h",
                "memory_usage": "45%",
                "cpu_usage": "12%",
            }
        return result

    @staticmethod
    def before_config_command(params: Dict[str, Any]) -> Dict[str, Any]:
        """Hook executed before config command."""
        logger.info(f"🔧 Built-in hook: before_config_command with params: {params}")
        # Add configuration validation
        params["validate_config"] = True
        params["include_secrets"] = False
        return params

    @staticmethod
    def after_config_command(result: Dict[str, Any]) -> Dict[str, Any]:
        """Hook executed after config command."""
        logger.info(f"🔧 Built-in hook: after_config_command with result: {result}")
        # Add configuration metadata
        result["config_metadata"] = {
            "last_modified": datetime.now().isoformat(),
            "version": "1.0.0",
            "environment": "development",
        }
        return result
