#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""Tests for `pylivemaker` package."""

from click.testing import CliRunner

from livemaker import cli


def test_lmar(shared_datadir):
    """Test lmar."""
    runner = CliRunner()

    # simple archive
    result = runner.invoke(cli.lmar, ["l", str(shared_datadir / "test.dat")])
    assert result.exit_code == 0
    assert "hello.txt" in result.output

    # simple exe
    result = runner.invoke(cli.lmar, ["l", str(shared_datadir / "test.exe")])
    assert result.exit_code == 0
    assert "hello.txt" in result.output

    # actual LiveMaker save file (from LiveNovel tutorial game)
    result = runner.invoke(cli.lmar, ["l", str(shared_datadir / "save.dat")])
    assert result.exit_code == 0
    assert "STATUSSAVEDATA" in result.output


def test_lmlsb(shared_datadir):
    """Test lmlsb."""
    runner = CliRunner()

    result = runner.invoke(cli.lmlsb, ["probe", str(shared_datadir / "gamemain.lsb")])
    assert result.exit_code == 0
    assert "LiveMaker compiled LSB script file" in result.output


def test_validate(shared_datadir):
    """Test lmlsb validation.

    This verifies that we can read/write both LSB and LiveNovel scripts that
    match LiveMaker's binaries.

    The sample LSB's used in these tests were generated by the most recent version of
    LiveMaker3 (setup03171228f.exe) and LiveNovel ADV-1 (adv-type1-03171228.exe).
    To generate your own test LSB's, just follow the tutorial from the LiveNovel docs.

    """
    runner = CliRunner()

    # LM3 ゲームメイン.lsb (from LiveNovel tutorial game)
    result = runner.invoke(cli.lmlsb, ["validate", str(shared_datadir / "gamemain.lsb")])
    assert result.exit_code == 0

    # LM3 00000001.lsb (from LiveNovel tutorial game)
    result = runner.invoke(cli.lmlsb, ["validate", str(shared_datadir / "00000001.lsb")])
    assert result.exit_code == 0


def test_extractcsv(shared_datadir, tmp_path):
    """Test lmlsb extractcsv."""
    runner = CliRunner()
    csv_file = tmp_path / "test.csv"

    expected = """
ID,Label,Context,Original text,Translated text
pylm:text:00000001.lsb:8:0,00000003,,"Hello world!
Page break",
pylm:text:00000001.lsb:8:1,00000003,,Wait for click,
pylm:text:00000001.lsb:8:2,00000003,,After wait,
pylm:text:00000001.lsb:8:3,00000003,,Text speed fast,
pylm:text:00000001.lsb:8:4,00000003,,Text speed slow,
pylm:text:00000001.lsb:8:5,00000003,,Text speed normal,
""".strip()

    result = runner.invoke(cli.lmlsb, ["extractcsv", str(shared_datadir / "00000001.lsb"), str(csv_file)])
    assert result.exit_code == 0

    with open(csv_file, encoding="utf-8") as f:
        assert f.read().splitlines() == expected.splitlines()


def test_insertcsv(shared_datadir, tmp_path):
    """Test lmlsb extractcsv."""
    runner = CliRunner()
    csv_file = tmp_path / "test.csv"

    lines = """
ID,Label,Context,Original text,Translated text
pylm:text:00000001.lsb:8:0,00000003,,"Hello world!","Goodbye world!"
pylm:text:00000001.lsb:8:1,00000003,,Wait for click,
""".strip()

    with open(csv_file, "w", encoding="utf-8") as f:
        f.write(lines)

    result = runner.invoke(cli.lmlsb, ["insertcsv", str(shared_datadir / "00000001.lsb"), str(csv_file)])
    assert result.exit_code == 0
    assert "Translated 1 lines" in result.output
    assert "Ignored 1 untranslated lines" in result.output
