# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pytile']

package_data = \
{'': ['*']}

install_requires = \
['aiohttp>=3.7.4,<4.0.0', 'pylint>=2.5.2,<3.0.0']

setup_kwargs = {
    'name': 'pytile',
    'version': '5.2.1',
    'description': 'A simple Python API for Tile® Bluetooth trackers',
    'long_description': '# 📡 pytile: A simple Python API for Tile® Bluetooth trackers\n\n[![CI](https://github.com/bachya/pytile/workflows/CI/badge.svg)](https://github.com/bachya/pytile/actions)\n[![PyPi](https://img.shields.io/pypi/v/pytile.svg)](https://pypi.python.org/pypi/pytile)\n[![Version](https://img.shields.io/pypi/pyversions/pytile.svg)](https://pypi.python.org/pypi/pytile)\n[![License](https://img.shields.io/pypi/l/pytile.svg)](https://github.com/bachya/pytile/blob/master/LICENSE)\n[![Code Coverage](https://codecov.io/gh/bachya/pytile/branch/master/graph/badge.svg)](https://codecov.io/gh/bachya/pytile)\n[![Maintainability](https://api.codeclimate.com/v1/badges/71eb642c735e33adcdfc/maintainability)](https://codeclimate.com/github/bachya/pytile/maintainability)\n[![Say Thanks](https://img.shields.io/badge/SayThanks-!-1EAEDB.svg)](https://saythanks.io/to/bachya)\n\n`pytile` is a simple Python library for retrieving information on\n[Tile® Bluetooth trackers](https://www.thetileapp.com/en-us/) (including last\nlocation and more).\n\nThis library is built on an unpublished, unofficial Tile API; it may alter or\ncease operation at any point.\n\n- [Python Versions](#python-versions)\n- [Installation](#installation)\n- [Usage](#usage)\n- [Contributing](#contributing)\n\n# NOTE: Version 5.0.0\n\nVersion 5.0.0 is a complete re-architecture of `pytile` – as such, the API has changed.\nPlease read the documentation carefully!\n\n# Python Versions\n\n`pytile` is currently supported on:\n\n* Python 3.6\n* Python 3.7\n* Python 3.8\n* Python 3.9\n\n# Installation\n\n```python\npip install pytile\n```\n\n# Usage\n\n## Getting an API Object\n\n`pytile` usage starts with an [`aiohttp`](https://github.com/aio-libs/aiohttp) `ClientSession` –\nnote that this ClientSession is required to properly authenticate the library:\n\n```python\nimport asyncio\n\nfrom aiohttp import ClientSession\n\nfrom pytile import async_login\n\n\nasync def main() -> None:\n    """Run!"""\n    async with ClientSession() as session:\n        api = await async_login("<EMAIL>", "<PASSWORD>", session)\n\n\nasyncio.run(main())\n```\n\nIf for some reason you need to use a specific client UUID (to, say, ensure that the\nTile API sees you as a client it\'s seen before) or a specific locale, you can do\nso easily:\n\n```python\nimport asyncio\n\nfrom aiohttp import ClientSession\n\nfrom pytile import async_login\n\n\nasync def main() -> None:\n    """Run!"""\n    async with ClientSession() as session:\n        api = await async_login(\n            "<EMAIL>", "<PASSWORD>", session, client_uuid="MY_UUID", locale="en-GB"\n        )\n\n\nasyncio.run(main())\n```\n\n## Getting Tiles\n\n```python\nimport asyncio\n\nfrom aiohttp import ClientSession\n\nfrom pytile import async_login\n\n\nasync def main() -> None:\n    """Run!"""\n    async with ClientSession() as session:\n        api = await async_login("<EMAIL>", "<PASSWORD>", session)\n\n        tiles = await api.async_get_tiles()\n\n\nasyncio.run(main())\n```\n\nThe `async_get_tiles` coroutine returns a dict with Tile UUIDs as the keys and `Tile`\nobjects as the values.\n\n### The `Tile` Object\n\nThe Tile object comes with several properties:\n\n* `accuracy`: the location accuracy of the Tile\n* `altitude`: the altitude of the Tile\n* `archetype`: the internal reference string that describes the Tile\'s "family"\n* `dead`: whether the Tile is inactive\n* `firmware_version`: the Tile\'s firmware version\n* `hardware_version`: the Tile\'s hardware version\n* `kind`: the kind of Tile (e.g., `TILE`, `PHONE`)\n* `last_timestamp`: the timestamp at which the current attributes were received\n* `latitude`: the latitude of the Tile\n* `longitude`: the latitude of the Tile\n* `lost`: whether the Tile has been marked as "lost"\n* `lost_timestamp`: the timestamp at which the Tile was last marked as "lost"\n* `name`: the name of the Tile\n* `uuid`: the Tile UUID\n* `visible`: whether the Tile is visible in the mobile app\n\n```python\nimport asyncio\n\nfrom aiohttp import ClientSession\n\nfrom pytile import async_login\n\n\nasync def main() -> None:\n    """Run!"""\n    async with ClientSession() as session:\n        api = await async_login("<EMAIL>", "<PASSWORD>", session)\n\n        tiles = await api.async_get_tiles()\n\n        for tile_uuid, tile in tiles.items():\n            print(f"The Tile\'s name is {tile.name}")\n            # ...\n\n\nasyncio.run(main())\n```\n\nIn addition to these properties, the `Tile` object comes with an `async_update` coroutine\nwhich requests new data from the Tile cloud API for this Tile:\n\n```python\nimport asyncio\n\nfrom aiohttp import ClientSession\n\nfrom pytile import async_login\n\n\nasync def main() -> None:\n    """Run!"""\n    async with ClientSession() as session:\n        api = await async_login("<EMAIL>", "<PASSWORD>", session)\n\n        tiles = await api.async_get_tiles()\n\n        for tile_uuid, tile in tiles.items():\n            await tile.async_update()\n\n\nasyncio.run(main())\n```\n\n# Contributing\n\n1. [Check for open features/bugs](https://github.com/bachya/pytile/issues)\n  or [initiate a discussion on one](https://github.com/bachya/pytile/issues/new).\n2. [Fork the repository](https://github.com/bachya/pytile/fork).\n3. (_optional, but highly recommended_) Create a virtual environment: `python3 -m venv .venv`\n4. (_optional, but highly recommended_) Enter the virtual environment: `source ./.venv/bin/activate`\n5. Install the dev environment: `script/setup`\n6. Code your new feature or bug fix.\n7. Write tests that cover your new functionality.\n8. Run tests and ensure 100% code coverage: `script/test`\n9. Update `README.md` with any new documentation.\n10. Add yourself to `AUTHORS.md`.\n11. Submit a pull request!\n',
    'author': 'Aaron Bach',
    'author_email': 'bachya1208@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/bachya/pytile',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6.0,<4.0.0',
}


setup(**setup_kwargs)
