/*
  * This program is free software: you can redistribute it and/or modify
  * it under the terms of the GNU General Public License as published by
  * the Free Software Foundation, either version 3 of the License, or
  * any later version.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  * GNU General Public License for more details.
  *
  * You should have received a copy of the GNU General Public License
  * along with this program.  If not, see <http://www.gnu.org/licenses/>.
  *
  * Additional permission under GNU GPL version 3 section 7
  *
  * If you modify this Program, or any covered work, by linking or combining
  * it with OpenSSL (or a modified version of that library), containing parts
  * covered by the terms of OpenSSL License and SSLeay License, the licensors
  * of this Program grant you additional permission to convey the resulting work.
  *
  */
#include <iostream>
#include <xmmintrin.h>
#include <qrandomx/qrxminer.h>
#include <misc/bignum.h>
#include <pow/powhelper.h>
#include <qrandomx/threadedqrandomx.h>
#include "gtest/gtest.h"

#define MINEXPECTEDMXCSR 8064
#define MAXEXPECTEDMXCSR 8127

namespace {
  TEST(QRXMiner, PassesTarget) {
    QRXMiner qm;

    std::vector<uint8_t> target{
            0x3E, 0xE5, 0x3F, 0xE1, 0xAC, 0xF3, 0x55, 0x92,
            0x66, 0xD8, 0x43, 0x89, 0xCE, 0xDE, 0x99, 0x33,
            0xC6, 0x8F, 0xC5, 0x1E, 0xD0, 0xA6, 0xC7, 0x91,
            0xF8, 0xF9, 0xE8, 0x9D, 0xB6, 0x23, 0xF0, 0xF6
    };

    ASSERT_TRUE(PoWHelper::passesTarget(target, target));

    std::cout << std::endl;
    std::cout << printByteVector2(target) << std::endl;

    // Iterate changing a single byte
    for (int i = 0; i<32; i++) {
      std::vector<uint8_t> below_1 = target;
      below_1[31-i]--;

      std::cout << printByteVector2(below_1) << std::endl;
      ASSERT_TRUE(PoWHelper::passesTarget(below_1, target));

      std::vector<uint8_t> over_1 = target;
      over_1[31-i]++;
      std::cout << printByteVector2(over_1) << std::endl;
      ASSERT_FALSE(PoWHelper::passesTarget(over_1, target));
    }
    ASSERT_GE(_mm_getcsr(), MINEXPECTEDMXCSR);
    ASSERT_LE(_mm_getcsr(), MAXEXPECTEDMXCSR);
  }

  TEST(QRXMiner, Run1Thread)
  {
    QRXMiner qm;
    ThreadedQRandomX qrx;

    uint64_t main_height = 10;
    uint64_t seed_height = qrx.getSeedHeight(main_height);

    std::vector<uint8_t> seed_hash {
            0x2a, 0x1c, 0x4a, 0x94, 0x33, 0xf1, 0xde, 0x36,
            0xf8, 0xb9, 0x9c, 0x7c, 0x5a, 0xce, 0xb7, 0xbd,
            0x2e, 0xb3, 0x9e, 0x1e, 0xad, 0x64, 0x8e, 0xa5,
            0x82, 0x27, 0xd3, 0x99, 0xad, 0x84, 0xc7, 0x24
    };

    std::vector<uint8_t> input {
            0x00, 0x9d, 0x6b, 0x07, 0x38, 0x20, 0xb0, 0x0a,
            0x5b, 0x60, 0xc7, 0x64, 0x6f, 0x3d, 0x30, 0x46,
            0xa3, 0x49, 0x49, 0xcd, 0x93, 0x9b, 0x2b, 0x52,
            0xb9, 0x9f, 0x1b, 0xb2, 0xde, 0x1f, 0x4c, 0x92,
            0x12, 0x7a, 0xe9, 0xe4, 0xc3, 0xe3, 0x9e, 0xe0,
            0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x58,
            0x22, 0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x54, 0xdb, 0x8c, 0x8e, 0xa6, 0x6f, 0xc0, 0x5b,
            0x1d, 0x12, 0x24, 0x25, 0x5a, 0x24, 0x70, 0x8e,
            0x5d, 0x67, 0x33, 0xdc
    };

    int miners = 0;

    std::vector<uint8_t> target = {
            0x0F, 0xFF, 0xFF, 0xE1, 0xAC, 0xF3, 0x55, 0x92,
            0x66, 0xD8, 0x43, 0x89, 0xCE, 0xDE, 0x99, 0x33,
            0xC6, 0x8F, 0xC5, 0x1E, 0xD0, 0xA6, 0xC7, 0x91,
            0xF8, 0xF9, 0xE8, 0x9D, 0xB6, 0x23, 0xF0, 0x0F
    };

    qm.start(main_height, seed_height, seed_hash, input, 0, target);

    qm.waitForAnswer(60);

    ASSERT_TRUE(qm.solutionAvailable());

    EXPECT_EQ(72, qm.solutionNonce());
    std::cout << printByteVector(qm.solutionHash()) << std::endl;

    std::vector<uint8_t> expected_winner{
            0x00, 0x00, 0x00, 0x48, 0x38, 0x20, 0xb0, 0x0a,
            0x5b, 0x60, 0xc7, 0x64, 0x6f, 0x3d, 0x30, 0x46,
            0xa3, 0x49, 0x49, 0xcd, 0x93, 0x9b, 0x2b, 0x52,
            0xb9, 0x9f, 0x1b, 0xb2, 0xde, 0x1f, 0x4c, 0x92,
            0x12, 0x7a, 0xe9, 0xe4, 0xc3, 0xe3, 0x9e, 0xe0,
            0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x58,
            0x22, 0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x54, 0xdb, 0x8c, 0x8e, 0xa6, 0x6f, 0xc0, 0x5b,
            0x1d, 0x12, 0x24, 0x25, 0x5a, 0x24, 0x70, 0x8e,
            0x5d, 0x67, 0x33, 0xdc
    };

    std::vector<uint8_t> expected_hash{
            0x57, 0xe8, 0xb8, 0x49, 0xdf, 0x3b, 0xc6, 0x3f,
            0x79, 0x42, 0x71, 0x7d, 0x16, 0xdf, 0xc9, 0x17,
            0xb8, 0xc3, 0xb5, 0x62, 0x7c, 0xf8, 0x7d, 0xf9,
            0x96, 0xcb, 0x39, 0xf2, 0xe0, 0x30, 0x00, 0x0c
    };

    EXPECT_EQ(expected_winner, qm.solutionInput());
    std::cout << printByteVector2(qm.solutionHash()) << std::endl;

    EXPECT_EQ(expected_hash, qm.solutionHash());
    ASSERT_GE(_mm_getcsr(), MINEXPECTEDMXCSR);
    ASSERT_LE(_mm_getcsr(), MAXEXPECTEDMXCSR);
  }

  TEST(QRXMiner, RunThreads_KeepHashing)
  {
    QRXMiner qm;
    ThreadedQRandomX qrx;

    uint64_t main_height = 10;
    uint64_t seed_height = qrx.getSeedHeight(main_height);

    std::vector<uint8_t> seed_hash {
            0x2a, 0x1c, 0x4a, 0x94, 0x33, 0xf1, 0xde, 0x36,
            0xf8, 0xb9, 0x9c, 0x7c, 0x5a, 0xce, 0xb7, 0xbd,
            0x2e, 0xb3, 0x9e, 0x1e, 0xad, 0x64, 0x8e, 0xa5,
            0x82, 0x27, 0xd3, 0x99, 0xad, 0x84, 0xc7, 0x24
    };

    std::vector<uint8_t> input(80);

    std::vector<uint8_t> boundary = {
            0x00, 0x06, 0xFF, 0xE1, 0xAC, 0xF3, 0x55, 0x92,
            0x66, 0xD8, 0x43, 0x89, 0xCE, 0xDE, 0x99, 0x33,
            0xC6, 0x8F, 0xC5, 0x1E, 0xD0, 0xA6, 0xC7, 0x91,
            0xF8, 0xF9, 0xE8, 0x9D, 0xB6, 0x23, 0x06, 0x00
    };

    qm.start(main_height, seed_height, seed_hash, input, 0, boundary, 10);

    int hash_count = 0;
    while (!qm.solutionAvailable()) {
      ThreadedQRandomX qrx;

      uint64_t main_height = 10;
      uint64_t seed_height = qrx.getSeedHeight(main_height);

      std::vector<uint8_t> seed_hash {
              0x2a, 0x1c, 0x4a, 0x94, 0x33, 0xf1, 0xde, 0x36,
              0xf8, 0xb9, 0x9c, 0x7c, 0x5a, 0xce, 0xb7, 0xbd,
              0x2e, 0xb3, 0x9e, 0x1e, 0xad, 0x64, 0x8e, 0xa5,
              0x82, 0x27, 0xd3, 0x99, 0xad, 0x84, 0xc7, 0x24
      };

      std::vector<uint8_t> internal_input(10000);

      std::vector<uint8_t> output_expected {
              0x5e, 0x09, 0x82, 0xc6, 0x96, 0xa8, 0xa7, 0x8f,
              0xab, 0xdd, 0x59, 0x1b, 0xaa, 0xcb, 0xfb, 0x6a,
              0x2d, 0x98, 0x9d, 0xe0, 0xc9, 0x62, 0x29, 0x4f,
              0x8d, 0x95, 0xc2, 0x00, 0x4b, 0x28, 0x51, 0xbd
      };

      auto output = qrx.hash(main_height, seed_height, seed_hash, internal_input, 0);
      EXPECT_EQ(output_expected, output);
      hash_count++;
    }

    std::cout << hash_count << std::endl;

    ASSERT_TRUE(qm.solutionAvailable());
    // Due to multiple threads running, possible nonce solution may vary
    // following are the most triggered possible nonce values
    EXPECT_TRUE(qm.solutionNonce() == 7424 || qm.solutionNonce() == 7475);
    ASSERT_GE(_mm_getcsr(), MINEXPECTEDMXCSR);
    ASSERT_LE(_mm_getcsr(), MAXEXPECTEDMXCSR);
  }

  TEST(QRXMiner, Run1Thread_bigblob)
  {
    QRXMiner qm;
    ThreadedQRandomX qrx;

    uint64_t main_height = 10;
    uint64_t seed_height = qrx.getSeedHeight(main_height);

    std::vector<uint8_t> seed_hash {
            0x2a, 0x1c, 0x4a, 0x94, 0x33, 0xf1, 0xde, 0x36,
            0xf8, 0xb9, 0x9c, 0x7c, 0x5a, 0xce, 0xb7, 0xbd,
            0x2e, 0xb3, 0x9e, 0x1e, 0xad, 0x64, 0x8e, 0xa5,
            0x82, 0x27, 0xd3, 0x99, 0xad, 0x84, 0xc7, 0x24
    };

    std::vector<uint8_t> input(1000);

    std::vector<uint8_t> boundary = {
            0x0F, 0xFF, 0xFF, 0xE1, 0xAC, 0xF3, 0x55, 0x92,
            0x66, 0xD8, 0x43, 0x89, 0xCE, 0xDE, 0x99, 0x33,
            0xC6, 0x8F, 0xC5, 0x1E, 0xD0, 0xA6, 0xC7, 0x91,
            0xF8, 0xF9, 0xE8, 0x9D, 0xB6, 0x23, 0xF0, 0x0F
    };

    qm.start(main_height, seed_height, seed_hash, input, 0, boundary);

    std::cout << "miner started" << std::endl;
    qm.waitForAnswer(60);

    ASSERT_TRUE(qm.solutionAvailable());
    EXPECT_EQ(2, qm.solutionNonce());
    ASSERT_GE(_mm_getcsr(), MINEXPECTEDMXCSR);
    ASSERT_LE(_mm_getcsr(), MAXEXPECTEDMXCSR);
  }

  TEST(QRXMiner, RunAndRestart)
  {
    QRXMiner qm;
    ThreadedQRandomX qrx;

    uint64_t main_height = 10;
    uint64_t seed_height = qrx.getSeedHeight(main_height);

    std::vector<uint8_t> seed_hash {
            0x2a, 0x1c, 0x4a, 0x94, 0x33, 0xf1, 0xde, 0x36,
            0xf8, 0xb9, 0x9c, 0x7c, 0x5a, 0xce, 0xb7, 0xbd,
            0x2e, 0xb3, 0x9e, 0x1e, 0xad, 0x64, 0x8e, 0xa5,
            0x82, 0x27, 0xd3, 0x99, 0xad, 0x84, 0xc7, 0x24
    };

    std::vector<uint8_t> input {
            0x00, 0x9d, 0x6b, 0x07, 0x38, 0x20, 0xb0, 0x0a,
            0x5b, 0x60, 0xc7, 0x64, 0x6f, 0x3d, 0x30, 0x46,
            0xa3, 0x49, 0x49, 0xcd, 0x93, 0x9b, 0x2b, 0x52,
            0xb9, 0x9f, 0x1b, 0xb2, 0xde, 0x1f, 0x4c, 0x92,
            0x12, 0x7a, 0xe9, 0xe4, 0xc3, 0xe3, 0x9e, 0xe0,
            0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x58,
            0x22, 0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x54, 0xdb, 0x8c, 0x8e, 0xa6, 0x6f, 0xc0, 0x5b,
            0x1d, 0x12, 0x24, 0x25, 0x5a, 0x24, 0x70, 0x8e,
            0x5d, 0x67, 0x33, 0xdc
    };

    std::vector<uint8_t> boundary = {
            0x0F, 0xFF, 0xFF, 0xE1, 0xAC, 0xF3, 0x55, 0x92,
            0x66, 0xD8, 0x43, 0x89, 0xCE, 0xDE, 0x99, 0x33,
            0xC6, 0x8F, 0xC5, 0x1E, 0xD0, 0xA6, 0xC7, 0x91,
            0xF8, 0xF9, 0xE8, 0x9D, 0xB6, 0x23, 0xF0, 0x0F
    };

    qm.start(main_height, seed_height, seed_hash, input, 0, boundary);
    qm.waitForAnswer(60);

    ASSERT_TRUE(qm.solutionAvailable());
    EXPECT_EQ(72, qm.solutionNonce());

    std::vector<uint8_t> expected_winner{
            0x00, 0x00, 0x00, 0x48, 0x38, 0x20, 0xb0, 0x0a,
            0x5b, 0x60, 0xc7, 0x64, 0x6f, 0x3d, 0x30, 0x46,
            0xa3, 0x49, 0x49, 0xcd, 0x93, 0x9b, 0x2b, 0x52,
            0xb9, 0x9f, 0x1b, 0xb2, 0xde, 0x1f, 0x4c, 0x92,
            0x12, 0x7a, 0xe9, 0xe4, 0xc3, 0xe3, 0x9e, 0xe0,
            0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x58,
            0x22, 0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x54, 0xdb, 0x8c, 0x8e, 0xa6, 0x6f, 0xc0, 0x5b,
            0x1d, 0x12, 0x24, 0x25, 0x5a, 0x24, 0x70, 0x8e,
            0x5d, 0x67, 0x33, 0xdc
    };

    EXPECT_EQ(expected_winner, qm.solutionInput());
    ASSERT_GE(_mm_getcsr(), MINEXPECTEDMXCSR);
    ASSERT_LE(_mm_getcsr(), MAXEXPECTEDMXCSR);
  }

  TEST(QRXMiner, MeasureHashRate)
  {
    QRXMiner qm;
    ThreadedQRandomX qrx;

    uint64_t main_height = 10;
    uint64_t seed_height = qrx.getSeedHeight(main_height);

    std::vector<uint8_t> seed_hash {
            0x2a, 0x1c, 0x4a, 0x94, 0x33, 0xf1, 0xde, 0x36,
            0xf8, 0xb9, 0x9c, 0x7c, 0x5a, 0xce, 0xb7, 0xbd,
            0x2e, 0xb3, 0x9e, 0x1e, 0xad, 0x64, 0x8e, 0xa5,
            0x82, 0x27, 0xd3, 0x99, 0xad, 0x84, 0xc7, 0x24
    };

    std::vector<uint8_t> input {
            0x00, 0x9d, 0x6b, 0x07, 0x38, 0x20, 0xb0, 0x0a,
            0x5b, 0x60, 0xc7, 0x64, 0x6f, 0x3d, 0x30, 0x46,
            0xa3, 0x49, 0x49, 0xcd, 0x93, 0x9b, 0x2b, 0x52,
            0xb9, 0x9f, 0x1b, 0xb2, 0xde, 0x1f, 0x4c, 0x92,
            0x12, 0x7a, 0xe9, 0xe4, 0xc3, 0xe3, 0x9e, 0xe0,
            0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x58,
            0x22, 0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x54, 0xdb, 0x8c, 0x8e, 0xa6, 0x6f, 0xc0, 0x5b,
            0x1d, 0x12, 0x24, 0x25, 0x5a, 0x24, 0x70, 0x8e,
            0x5d, 0x67, 0x33, 0xdc
    };

    std::vector<uint8_t> target{
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x66, 0xD8, 0x43, 0x89, 0xCE, 0xDE, 0x99, 0x33,
            0xC6, 0x8F, 0xC5, 0x1E, 0xD0, 0xA6, 0xC7, 0x91,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    qm.start(main_height, seed_height, seed_hash, input, 0, target, 4);
    qm.waitForAnswer(3);
    std::cout << std::endl << "hashes/sec: " << qm.hashRate() << std::endl;

    EXPECT_FALSE(qm.solutionAvailable());
    qm.cancel();
    ASSERT_FALSE(qm.isRunning());
    ASSERT_GE(_mm_getcsr(), MINEXPECTEDMXCSR);
    ASSERT_LE(_mm_getcsr(), MAXEXPECTEDMXCSR);
  }

  TEST(QRXMiner, RunAndCancel)
  {
    QRXMiner qm;
    ThreadedQRandomX qrx;

    uint64_t main_height = 10;
    uint64_t seed_height = qrx.getSeedHeight(main_height);

    std::vector<uint8_t> seed_hash {
            0x2a, 0x1c, 0x4a, 0x94, 0x33, 0xf1, 0xde, 0x36,
            0xf8, 0xb9, 0x9c, 0x7c, 0x5a, 0xce, 0xb7, 0xbd,
            0x2e, 0xb3, 0x9e, 0x1e, 0xad, 0x64, 0x8e, 0xa5,
            0x82, 0x27, 0xd3, 0x99, 0xad, 0x84, 0xc7, 0x24
    };

    std::vector<uint8_t> input {
            0x00, 0x9d, 0x6b, 0x07, 0x38, 0x20, 0xb0, 0x0a,
            0x5b, 0x60, 0xc7, 0x64, 0x6f, 0x3d, 0x30, 0x46,
            0xa3, 0x49, 0x49, 0xcd, 0x93, 0x9b, 0x2b, 0x52,
            0xb9, 0x9f, 0x1b, 0xb2, 0xde, 0x1f, 0x4c, 0x92,
            0x12, 0x7a, 0xe9, 0xe4, 0xc3, 0xe3, 0x9e, 0xe0,
            0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x58,
            0x22, 0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x54, 0xdb, 0x8c, 0x8e, 0xa6, 0x6f, 0xc0, 0x5b,
            0x1d, 0x12, 0x24, 0x25, 0x5a, 0x24, 0x70, 0x8e,
            0x5d, 0x67, 0x33, 0xdc
    };

    std::vector<uint8_t> boundary = {
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x66, 0xD8, 0x43, 0x89, 0xCE, 0xDE, 0x99, 0x33,
            0xC6, 0x8F, 0xC5, 0x1E, 0xD0, 0xA6, 0xC7, 0x91,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    qm.start(main_height, seed_height, seed_hash, input, 0, boundary);
    qm.waitForAnswer(1);
    qm.cancel();
    ASSERT_FALSE(qm.isRunning());

    ASSERT_FALSE(qm.solutionAvailable());
    ASSERT_GE(_mm_getcsr(), MINEXPECTEDMXCSR);
    ASSERT_LE(_mm_getcsr(), MAXEXPECTEDMXCSR);
  }

  TEST(QRXMiner, RunCancelSafety)
  {
    QRXMiner qm;
    ThreadedQRandomX qrx;

    uint64_t main_height = 10;
    uint64_t seed_height = qrx.getSeedHeight(main_height);

    std::vector<uint8_t> seed_hash {
            0x2a, 0x1c, 0x4a, 0x94, 0x33, 0xf1, 0xde, 0x36,
            0xf8, 0xb9, 0x9c, 0x7c, 0x5a, 0xce, 0xb7, 0xbd,
            0x2e, 0xb3, 0x9e, 0x1e, 0xad, 0x64, 0x8e, 0xa5,
            0x82, 0x27, 0xd3, 0x99, 0xad, 0x84, 0xc7, 0x24
    };

    std::vector<uint8_t> input {
            0x00, 0x9d, 0x6b, 0x07, 0x38, 0x20, 0xb0, 0x0a,
            0x5b, 0x60, 0xc7, 0x64, 0x6f, 0x3d, 0x30, 0x46,
            0xa3, 0x49, 0x49, 0xcd, 0x93, 0x9b, 0x2b, 0x52,
            0xb9, 0x9f, 0x1b, 0xb2, 0xde, 0x1f, 0x4c, 0x92,
            0x12, 0x7a, 0xe9, 0xe4, 0xc3, 0xe3, 0x9e, 0xe0,
            0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x58,
            0x22, 0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x54, 0xdb, 0x8c, 0x8e, 0xa6, 0x6f, 0xc0, 0x5b,
            0x1d, 0x12, 0x24, 0x25, 0x5a, 0x24, 0x70, 0x8e,
            0x5d, 0x67, 0x33, 0xdc
    };

    int miners = 0;

    std::vector<uint8_t> boundary = {
            0x9F, 0xFF, 0xFF, 0xE1, 0xAC, 0xF3, 0x55, 0x92,
            0x66, 0xD8, 0x43, 0x89, 0xCE, 0xDE, 0x99, 0x33,
            0xC6, 0x8F, 0xC5, 0x1E, 0xD0, 0xA6, 0xC7, 0x91,
            0xF8, 0xF9, 0xE8, 0x9D, 0xB6, 0x23, 0xF0, 0xFF
    };

    for (int i = 0; i<20; i++) {
      using namespace std::chrono_literals;
      boundary[0] -= 10;
      qm.start(main_height, seed_height, seed_hash, input, 0, boundary);
      std::this_thread::sleep_for(500ms);
      qm.cancel();
    }

    ASSERT_FALSE(qm.isRunning());
    ASSERT_GE(_mm_getcsr(), MINEXPECTEDMXCSR);
    ASSERT_LE(_mm_getcsr(), MAXEXPECTEDMXCSR);
  }

}