# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['unipressed',
 'unipressed.dataset',
 'unipressed.dataset.generated_types',
 'unipressed.id_mapping']

package_data = \
{'': ['*']}

install_requires = \
['requests>=2.28.1,<3.0.0', 'typing-extensions>=4.3.0,<5.0.0']

setup_kwargs = {
    'name': 'unipressed',
    'version': '1.1.0',
    'description': 'Comprehensive Python client for the Uniprot REST API',
    'long_description': '# Unipressed\n\n**Please visit the [project website](https://multimeric.github.io/Unipressed/) for more comprehensive documentation.**\n\n## Introduction\n\nUnipressed (Uniprot REST) is an API client for the protein database [Uniprot](https://www.uniprot.org/).\nIt provides thoroughly typed and documented code to ensure your use of the library is easy, fast, and correct!\n\n### Example\nLet\'s say we\'re interested in very long proteins that are encoded within a chloroplast, in any organism:\n\n```python\nfrom unipressed import UniprotkbClient\n\nfor record in UniprotkbClient.search(\n    query={\n        "and_": [\n            {"organelle": "chloroplast"},\n            {"length": (5000, "*")}\n        ]\n    },\n    fields=["length", "gene_names"]\n).each_record():\n    display(record)\n```\n\n\n<pre style="white-space:pre;overflow-x:auto;line-height:normal;font-family:Menlo,\'DejaVu Sans Mono\',consolas,\'Courier New\',monospace">\n<span style="font-weight: bold">{</span>\n    <span style="color: #008000; text-decoration-color: #008000">\'primaryAccession\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'A0A088CK67\'</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'genes\'</span>: <span style="font-weight: bold">[</span>\n        <span style="font-weight: bold">{</span>\n            <span style="color: #008000; text-decoration-color: #008000">\'geneName\'</span>: <span style="font-weight: bold">{</span>\n                <span style="color: #008000; text-decoration-color: #008000">\'evidences\'</span>: <span style="font-weight: bold">[{</span><span style="color: #008000; text-decoration-color: #008000">\'evidenceCode\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'ECO:0000313\'</span>, <span style="color: #008000; text-decoration-color: #008000">\'source\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'EMBL\'</span>, <span style="color: #008000; text-decoration-color: #008000">\'id\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'AID67672.1\'</span><span style="font-weight: bold">}]</span>,\n                <span style="color: #008000; text-decoration-color: #008000">\'value\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'ftsH\'</span>\n            <span style="font-weight: bold">}</span>\n        <span style="font-weight: bold">}</span>\n    <span style="font-weight: bold">]</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'sequence\'</span>: <span style="font-weight: bold">{</span><span style="color: #008000; text-decoration-color: #008000">\'length\'</span>: <span style="color: #008080; text-decoration-color: #008080; font-weight: bold">5242</span><span style="font-weight: bold">}</span>\n<span style="font-weight: bold">}</span>\n</pre>\n\n\n\n### Advantages\n\n* Detailed type hints for autocompleting queries as you type\n* Autocompletion for return fields\n* Documentation for each field\n* Automatic results parsing, for `json`, `tsv`, `list`, and `xml`\n* Built-in pagination, so you don\'t have to handle any of that yourself!\n* Most of the API is automatically generated, ensuring very rapid updates whenever the API changes\n* Thoroughly tested, with 41 unit tests and counting!\n\n## Usage\n\n### Installation\n\nIf you\'re using poetry:\n```bash\npoetry add unipressed\n```\n\nOtherwise:\n```bash\npip install unipressed\n```\n\n### Dataset Clients\n\nThe `unipressed` module exports a client object for each UniProt dataset:\n\n```python\nfrom unipressed import UniprotkbClient, UniparcClient\n```\n\nWith one of these clients, you can search the dataset:\n\n```python\nrecords = UniprotkbClient.search({\n    "length": (5000, 6000)\n}).each_record()\n\n# Show the first record\nnext(records)\n```\n\n\n<pre style="white-space:pre;overflow-x:auto;line-height:normal;font-family:Menlo,\'DejaVu Sans Mono\',consolas,\'Courier New\',monospace">\n<span style="font-weight: bold">{</span>\n    <span style="color: #008000; text-decoration-color: #008000">\'entryType\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'UniProtKB reviewed (Swiss-Prot)\'</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'primaryAccession\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'Q96RW7\'</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'secondaryAccessions\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'uniProtkbId\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'HMCN1_HUMAN\'</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'entryAudit\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'annotationScore\'</span>: <span style="color: #008080; text-decoration-color: #008080; font-weight: bold">5.0</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'organism\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'proteinExistence\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'1: Evidence at protein level\'</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'proteinDescription\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'genes\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'comments\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'features\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'keywords\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'references\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'uniProtKBCrossReferences\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'sequence\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'extraAttributes\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>\n<span style="font-weight: bold">}</span>\n</pre>\n\n\n\nYou can request a single record by ID:\n\n```python\nUniprotkbClient.fetch_one("Q96RW7")\n```\n\n\n<pre style="white-space:pre;overflow-x:auto;line-height:normal;font-family:Menlo,\'DejaVu Sans Mono\',consolas,\'Courier New\',monospace">\n<span style="font-weight: bold">{</span>\n    <span style="color: #008000; text-decoration-color: #008000">\'entryType\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'UniProtKB reviewed (Swiss-Prot)\'</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'primaryAccession\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'Q96RW7\'</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'secondaryAccessions\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'uniProtkbId\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'HMCN1_HUMAN\'</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'entryAudit\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'annotationScore\'</span>: <span style="color: #008080; text-decoration-color: #008080; font-weight: bold">5.0</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'organism\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'proteinExistence\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'1: Evidence at protein level\'</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'proteinDescription\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'genes\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'comments\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'features\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'keywords\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'references\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'uniProtKBCrossReferences\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'sequence\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n    <span style="color: #008000; text-decoration-color: #008000">\'extraAttributes\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>\n<span style="font-weight: bold">}</span>\n</pre>\n\n\n\nYou can also request multiple records:\n\n```python\nUniprotkbClient.fetch_many(["A0A0C5B5G6", "A0A1B0GTW7"])\n```\n\n\n<pre style="white-space:pre;overflow-x:auto;line-height:normal;font-family:Menlo,\'DejaVu Sans Mono\',consolas,\'Courier New\',monospace">\n<span style="font-weight: bold">[</span>\n    <span style="font-weight: bold">{</span>\n        <span style="color: #008000; text-decoration-color: #008000">\'entryType\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'UniProtKB reviewed (Swiss-Prot)\'</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'primaryAccession\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'A0A0C5B5G6\'</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'uniProtkbId\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'MOTSC_HUMAN\'</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'entryAudit\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'annotationScore\'</span>: <span style="color: #008080; text-decoration-color: #008080; font-weight: bold">5.0</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'organism\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'proteinExistence\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'1: Evidence at protein level\'</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'proteinDescription\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'genes\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'comments\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'features\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'geneLocations\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'keywords\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'references\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'uniProtKBCrossReferences\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'sequence\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'extraAttributes\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>\n    <span style="font-weight: bold">}</span>,\n    <span style="font-weight: bold">{</span>\n        <span style="color: #008000; text-decoration-color: #008000">\'entryType\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'UniProtKB reviewed (Swiss-Prot)\'</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'primaryAccession\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'A0A1B0GTW7\'</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'secondaryAccessions\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'uniProtkbId\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'CIROP_HUMAN\'</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'entryAudit\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'annotationScore\'</span>: <span style="color: #008080; text-decoration-color: #008080; font-weight: bold">5.0</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'organism\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'proteinExistence\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'1: Evidence at protein level\'</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'proteinDescription\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'genes\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'comments\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'features\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'keywords\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'references\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'uniProtKBCrossReferences\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'sequence\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>,\n        <span style="color: #008000; text-decoration-color: #008000">\'extraAttributes\'</span>: <span style="color: #808000; text-decoration-color: #808000">...</span>\n    <span style="font-weight: bold">}</span>\n<span style="font-weight: bold">]</span>\n</pre>\n\n\n\n### ID Mapping\n\nUnipressed also provides one other unique client, which is designed for mapping identifiers. You provide the source and destination database (both of which will autocomplete in VS Code), and a list of identifiers for the source database.\n\n```python\nfrom unipressed import IdMappingClient\nrequest = IdMappingClient.submit(\n    source="UniProtKB_AC-ID", dest="Gene_Name", ids={"A1L190", "A0JP26", "A0PK11"}\n)\nlist(request.each_result())\n```\n\n\n<pre style="white-space:pre;overflow-x:auto;line-height:normal;font-family:Menlo,\'DejaVu Sans Mono\',consolas,\'Courier New\',monospace">\n<span style="font-weight: bold">[</span>\n    <span style="font-weight: bold">{</span><span style="color: #008000; text-decoration-color: #008000">\'from\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'A1L190\'</span>, <span style="color: #008000; text-decoration-color: #008000">\'to\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'SYCE3\'</span><span style="font-weight: bold">}</span>,\n    <span style="font-weight: bold">{</span><span style="color: #008000; text-decoration-color: #008000">\'from\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'A0PK11\'</span>, <span style="color: #008000; text-decoration-color: #008000">\'to\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'CLRN2\'</span><span style="font-weight: bold">}</span>,\n    <span style="font-weight: bold">{</span><span style="color: #008000; text-decoration-color: #008000">\'from\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'A0JP26\'</span>, <span style="color: #008000; text-decoration-color: #008000">\'to\'</span>: <span style="color: #008000; text-decoration-color: #008000">\'POTEB3\'</span><span style="font-weight: bold">}</span>\n<span style="font-weight: bold">]</span>\n</pre>\n\n\n\nNote that, if you submit a large number of IDs, you might need to add a `sleep()` call between submitting the request and retrieving the results.\n\n### Query Syntax\n\nThe query syntax refers to the values you pass in to the `query` argument of the `search()` method.\n\nIn general, you can\'t go wrong by following the type hints.\nI strongly recommend using something like [`pylance`](https://marketplace.visualstudio.com/items?itemName=ms-python.vscode-pylance) for [Visual Studio Code](https://code.visualstudio.com/), which will provide automatic completions and warn you when you have used the wrong syntax.\n\nIf you already know how to use the Uniprot query language, you can always just input your queries as strings:\n\n```python\nUniprotkbClient.search(query="(gene:BRCA*) AND (organism_id:10090)")\n```\n\nHowever, if you want some built-in query validation and code completion using Python\'s type system, then you can instead use a dictionary.\nThe simplest query is a dictionary with a single key: \n\n```python\nUniprotkbClient.search(query={"family": "kinase"})\n```\n\nYou can compile more complex queries using the `and_`, `or_` and `not_` keys.\nThese first two operators take a list of query dictionaries: \n\n```python\nUniprotkbClient.search(query={\n    "and_": [\n        {"family": "kinase"},\n        {"organism_id": "9606"},\n    ]\n})\n```\n\nMost "leaf" nodes of the query tree (ie those that aren\'t operators like `and_`) are strings, integers or floats, which you input as normal Python literals as you can see above.\nFor string fields, you also have access to wildcards, namely the `*` character. \nFor example, if you want every human protein belonging to a gene whose name starts with `PRO`, you could use:\n\n```python\nUniprotkbClient.search(query={\n    "and_": [\n        {"gene": "PRO*"},\n        {"organism_id": "9606"},\n    ]\n})\n```\n\nA few query fields are *ranges*, which you input using a tuple with two elements, indicating the start and end of the range.\nIf you use the literal `"*"` then you can leave the range open at one end. \nFor example, this query returns any protein that is in the range $[5000, \\infty)$\n\n```python\nUniprotkbClient.search(query={"length": (5000, "*")})\n```\n\nFinally, a few query fields take dates.\nThese you input as a Python `datetime.date` object.\nFor example, to find proteins added to UniProt since July 2022, we would do:\n\n```python\nfrom datetime import date\n\nUniprotkbClient.search(query={"date_created": (date(2022, 7, 1), "*")})\n```\n\n### Use with Visual Studio Code\nTo get VS Code to offer suggestions, press the `Trigger Suggest` shortcut which is usually bound to `Ctrl + Space`.\nIn particular, code completion generally won\'t work *until* you open a string literal using a quotation mark.\n\nSecondly, to get live access to the documentation, you can either use the `Show Hover` shortcut, which is usually bound to `Ctrl + K, Ctrl + I`, or you can install the [`docs-view`](https://marketplace.visualstudio.com/items?itemName=bierner.docs-view) extension, which lets you view the docstrings in the sidebar without interfering with your code.\n',
    'author': 'Michael Milton',
    'author_email': 'michael.r.milton@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://multimeric.github.io/Unipressed',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
