"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EndpointConfig = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const os_1 = require("os");
const cdk = require("aws-cdk-lib");
const instance_type_1 = require("./instance-type");
const util_1 = require("./private/util");
const aws_sagemaker_1 = require("aws-cdk-lib/aws-sagemaker");
/**
 * Defines a SageMaker EndpointConfig.
 */
class EndpointConfig extends cdk.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.endpointConfigName,
        });
        this.instanceProductionVariantsByName = {};
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_sagemaker_alpha_EndpointConfigProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, EndpointConfig);
            }
            throw error;
        }
        (props.instanceProductionVariants || []).map(p => this.addInstanceProductionVariant(p));
        // create the endpoint configuration resource
        const endpointConfig = new aws_sagemaker_1.CfnEndpointConfig(this, 'EndpointConfig', {
            kmsKeyId: (props.encryptionKey) ? props.encryptionKey.keyArn : undefined,
            endpointConfigName: this.physicalName,
            productionVariants: cdk.Lazy.any({ produce: () => this.renderInstanceProductionVariants() }),
        });
        this.endpointConfigName = this.getResourceNameAttribute(endpointConfig.attrEndpointConfigName);
        this.endpointConfigArn = this.getResourceArnAttribute(endpointConfig.ref, {
            service: 'sagemaker',
            resource: 'endpoint-config',
            resourceName: this.physicalName,
        });
    }
    /**
     * Imports an EndpointConfig defined either outside the CDK or in a different CDK stack.
     * @param scope the Construct scope.
     * @param id the resource id.
     * @param endpointConfigArn the ARN of the endpoint configuration.
     */
    static fromEndpointConfigArn(scope, id, endpointConfigArn) {
        const endpointConfigName = cdk.Stack.of(scope).splitArn(endpointConfigArn, cdk.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.endpointConfigArn = endpointConfigArn;
                this.endpointConfigName = endpointConfigName;
            }
        }
        return new Import(scope, id, {
            environmentFromArn: endpointConfigArn,
        });
    }
    /**
     * Imports an EndpointConfig defined either outside the CDK or in a different CDK stack.
     * @param scope the Construct scope.
     * @param id the resource id.
     * @param endpointConfigName the name of the endpoint configuration.
     */
    static fromEndpointConfigName(scope, id, endpointConfigName) {
        const endpointConfigArn = cdk.Stack.of(scope).formatArn({
            service: 'sagemaker',
            resource: 'endpoint-config',
            resourceName: endpointConfigName,
        });
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.endpointConfigArn = endpointConfigArn;
                this.endpointConfigName = endpointConfigName;
            }
        }
        return new Import(scope, id, {
            environmentFromArn: endpointConfigArn,
        });
    }
    /**
     * Add production variant to the endpoint configuration.
     *
     * @param props The properties of a production variant to add.
     */
    addInstanceProductionVariant(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_sagemaker_alpha_InstanceProductionVariantProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addInstanceProductionVariant);
            }
            throw error;
        }
        if (props.variantName in this.instanceProductionVariantsByName) {
            throw new Error(`There is already a Production Variant with name '${props.variantName}'`);
        }
        this.validateInstanceProductionVariantProps(props);
        this.instanceProductionVariantsByName[props.variantName] = {
            acceleratorType: props.acceleratorType,
            initialInstanceCount: props.initialInstanceCount || 1,
            initialVariantWeight: props.initialVariantWeight || 1.0,
            instanceType: props.instanceType || instance_type_1.InstanceType.T2_MEDIUM,
            modelName: props.model.modelName,
            variantName: props.variantName,
        };
    }
    /**
     * Get instance production variants associated with endpoint configuration.
     *
     * @internal
     */
    get _instanceProductionVariants() {
        return Object.values(this.instanceProductionVariantsByName);
    }
    /**
     * Find instance production variant based on variant name
     * @param name Variant name from production variant
     *
     * @internal
     */
    _findInstanceProductionVariant(name) {
        const ret = this.instanceProductionVariantsByName[name];
        if (!ret) {
            throw new Error(`No variant with name: '${name}'`);
        }
        return ret;
    }
    validateProductionVariants() {
        // validate number of production variants
        if (this._instanceProductionVariants.length < 1) {
            throw new Error('Must configure at least 1 production variant');
        }
        else if (this._instanceProductionVariants.length > 10) {
            throw new Error('Can\'t have more than 10 production variants');
        }
    }
    validateInstanceProductionVariantProps(props) {
        const errors = [];
        // check instance count is greater than zero
        if (props.initialInstanceCount !== undefined && props.initialInstanceCount < 1) {
            errors.push('Must have at least one instance');
        }
        // check variant weight is not negative
        if (props.initialVariantWeight && props.initialVariantWeight < 0) {
            errors.push('Cannot have negative variant weight');
        }
        // check environment compatibility with model
        const model = props.model;
        if (!util_1.sameEnv(model.env.account, this.env.account)) {
            errors.push(`Cannot use model in account ${model.env.account} for endpoint configuration in account ${this.env.account}`);
        }
        else if (!util_1.sameEnv(model.env.region, this.env.region)) {
            errors.push(`Cannot use model in region ${model.env.region} for endpoint configuration in region ${this.env.region}`);
        }
        if (errors.length > 0) {
            throw new Error(`Invalid Production Variant Props: ${errors.join(os_1.EOL)}`);
        }
    }
    /**
     * Render the list of instance production variants.
     */
    renderInstanceProductionVariants() {
        this.validateProductionVariants();
        return this._instanceProductionVariants.map(v => ({
            acceleratorType: v.acceleratorType?.toString(),
            initialInstanceCount: v.initialInstanceCount,
            initialVariantWeight: v.initialVariantWeight,
            instanceType: v.instanceType.toString(),
            modelName: v.modelName,
            variantName: v.variantName,
        }));
    }
}
exports.EndpointConfig = EndpointConfig;
_a = JSII_RTTI_SYMBOL_1;
EndpointConfig[_a] = { fqn: "@aws-cdk/aws-sagemaker-alpha.EndpointConfig", version: "2.56.1-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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