"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodeHttpHandler = void 0;
const protocol_http_1 = require("@aws-sdk/protocol-http");
const querystring_builder_1 = require("@aws-sdk/querystring-builder");
const http_1 = require("http");
const https_1 = require("https");
const constants_1 = require("./constants");
const get_transformed_headers_1 = require("./get-transformed-headers");
const set_connection_timeout_1 = require("./set-connection-timeout");
const set_socket_timeout_1 = require("./set-socket-timeout");
const write_request_body_1 = require("./write-request-body");
class NodeHttpHandler {
    constructor({ connectionTimeout, socketTimeout, httpAgent, httpsAgent } = {}) {
        this.metadata = { handlerProtocol: "http/1.1" };
        this.connectionTimeout = connectionTimeout;
        this.socketTimeout = socketTimeout;
        const keepAlive = true;
        const maxSockets = 50;
        this.httpAgent = httpAgent || new http_1.Agent({ keepAlive, maxSockets });
        this.httpsAgent = httpsAgent || new https_1.Agent({ keepAlive, maxSockets });
    }
    destroy() {
        this.httpAgent.destroy();
        this.httpsAgent.destroy();
    }
    handle(request, { abortSignal } = {}) {
        return new Promise((resolve, reject) => {
            if (abortSignal === null || abortSignal === void 0 ? void 0 : abortSignal.aborted) {
                const abortError = new Error("Request aborted");
                abortError.name = "AbortError";
                reject(abortError);
                return;
            }
            const isSSL = request.protocol === "https:";
            const queryString = querystring_builder_1.buildQueryString(request.query || {});
            const nodeHttpsOptions = {
                headers: request.headers,
                host: request.hostname,
                method: request.method,
                path: queryString ? `${request.path}?${queryString}` : request.path,
                port: request.port,
                agent: isSSL ? this.httpsAgent : this.httpAgent,
            };
            const requestFunc = isSSL ? https_1.request : http_1.request;
            const req = requestFunc(nodeHttpsOptions, (res) => {
                const httpResponse = new protocol_http_1.HttpResponse({
                    statusCode: res.statusCode || -1,
                    headers: get_transformed_headers_1.getTransformedHeaders(res.headers),
                    body: res,
                });
                resolve({ response: httpResponse });
            });
            req.on("error", (err) => {
                if (constants_1.NODEJS_TIMEOUT_ERROR_CODES.includes(err.code)) {
                    reject(Object.assign(err, { name: "TimeoutError" }));
                }
                else {
                    reject(err);
                }
            });
            set_connection_timeout_1.setConnectionTimeout(req, reject, this.connectionTimeout);
            set_socket_timeout_1.setSocketTimeout(req, reject, this.socketTimeout);
            if (abortSignal) {
                abortSignal.onabort = () => {
                    req.abort();
                    const abortError = new Error("Request aborted");
                    abortError.name = "AbortError";
                    reject(abortError);
                };
            }
            write_request_body_1.writeRequestBody(req, request);
        });
    }
}
exports.NodeHttpHandler = NodeHttpHandler;
