import torch
import torch.nn as nn
from .pnasnet_ops import *

class Cell(nn.Module):

    def __init__(self, genotype, C_prev_prev, C_prev, C, reduction, reduction_prev):
        super(Cell, self).__init__()
        #print(C_prev_prev, C_prev, C)
        self.reduction = reduction

        if reduction_prev is None:
            self.preprocess0 = Identity()
        elif reduction_prev is True:
            self.preprocess0 = FactorizedReduce(C_prev_prev, C)
        else:
            self.preprocess0 = ReLUConvBN(C_prev_prev, C, 1, 1, 0)
        self.preprocess1 = ReLUConvBN(C_prev, C, 1, 1, 0)

        if reduction:
            op_names, indices = zip(*genotype.reduce)
            concat = genotype.reduce_concat
        else:
            op_names, indices = zip(*genotype.normal)
            concat = genotype.normal_concat

        assert len(op_names) == len(indices)
        self._steps = len(op_names) // 2
        self._concat = concat
        self.multiplier = len(concat)

        self._ops = nn.ModuleList()
        for name, index in zip(op_names, indices):
            stride = 2 if reduction and index < 2 else 1
            if reduction_prev is None and index == 0:
                op = OPS[name](C_prev_prev, C, stride, True)
            else:
                op = OPS[name](C, C, stride, True)
            self._ops += [op]
        self._indices = indices

    def forward(self, s0, s1, drop_prob):
        s0 = self.preprocess0(s0)
        s1 = self.preprocess1(s1)

        states = [s0, s1]
        for i in range(self._steps):
            h1 = states[self._indices[2*i]]
            h2 = states[self._indices[2*i+1]]
            op1 = self._ops[2*i]
            op2 = self._ops[2*i+1]
            h1 = op1(h1)
            h2 = op2(h2)
            # if self.training and drop_prob > 0.:
            #   if not isinstance(op1, Identity):
            #     h1 = drop_path(h1, drop_prob)
            #   if not isinstance(op2, Identity):
            #     h2 = drop_path(h2, drop_prob)
            s = h1 + h2
            states += [s]
        return torch.cat([states[i] for i in self._concat], dim=1)


class AuxiliaryHeadImageNet(nn.Module):

    def __init__(self, C, num_classes):
        """assuming input size 14x14"""
        super(AuxiliaryHeadImageNet, self).__init__()
        self.features = nn.Sequential(
            nn.ReLU(inplace=True),
            nn.AvgPool2d(5, stride=2, padding=0, count_include_pad=False),
            nn.Conv2d(C, 128, 1, bias=False),
            nn.BatchNorm2d(128),
            nn.ReLU(inplace=True),
            nn.Conv2d(128, 768, 2, bias=False),
            nn.BatchNorm2d(768),
            nn.ReLU(inplace=True)
        )
        self.classifier = nn.Linear(768, num_classes)

    def forward(self, x):
        x = self.features(x)
        x = self.classifier(x.view(x.size(0), -1))
        return x


class PNASNetwork(nn.Module):

    def __init__(self, C, num_classes, layers, auxiliary, genotype):
        super(PNASNetwork, self).__init__()
        self._layers = layers
        self._auxiliary = auxiliary
        self.drop_path_prob = 0.0

        self.conv0 = nn.Conv2d(3, 96, kernel_size=3,
                               stride=2, padding=0, bias=False)
        self.conv0_bn = nn.BatchNorm2d(96, eps=1e-3)
        self.stem1 = Cell(genotype, 96, 96, C // 4, True, None)
        self.stem2 = Cell(genotype, 96, C *
                          self.stem1.multiplier // 4, C // 2, True, True)

        C_prev_prev, C_prev, C_curr = C * \
            self.stem1.multiplier // 4, C * self.stem2.multiplier // 2, C

        self.cells = nn.ModuleList()
        reduction_prev = True
        for i in range(layers):
            if i in [layers // 3, 2 * layers // 3]:
                C_curr *= 2
                reduction = True
            else:
                reduction = False
            cell = Cell(genotype, C_prev_prev, C_prev,
                        C_curr, reduction, reduction_prev)
            reduction_prev = reduction
            self.cells += [cell]
            C_prev_prev, C_prev = C_prev, cell.multiplier * C_curr
            if i == 2 * layers // 3:
                C_to_auxiliary = C_prev

        if auxiliary:
            self.auxiliary_head = AuxiliaryHeadImageNet(
                C_to_auxiliary, num_classes)
        self.relu = nn.ReLU(inplace=False)
        self.global_pooling = nn.AdaptiveAvgPool2d(1)
        self.num_features = C_prev
        self.classifier = nn.Linear(C_prev, num_classes)

    def forward(self, input):
        logits_aux = None
        s0 = self.conv0(input)
        s0 = self.conv0_bn(s0)
        s1 = self.stem1(s0, s0, self.drop_path_prob)
        s0, s1 = s1, self.stem2(s0, s1, self.drop_path_prob)
        for i, cell in enumerate(self.cells):
            s0, s1 = s1, cell(s0, s1, self.drop_path_prob)
            if i == 2 * self._layers // 3:
                if self._auxiliary and self.training:
                    logits_aux = self.auxiliary_head(s1)
        s1 = self.relu(s1)
        out = self.global_pooling(s1)
        logits = self.classifier(out.view(out.size(0), -1))
        return logits

    def reset_classifier(self, num_classes, dropout=0.0):
        self.num_classes = num_classes
        self.drop_rate = dropout

        del self.classifier

        if self._auxiliary:
            del self.auxiliary_head.classifier
        if self.num_classes:
            self.classifier = nn.Linear(self.num_features, self.num_classes)
            if self._auxiliary:
                self.auxiliary_head.classifier = nn.Linear(768, self.num_classes)
        else:
            self.classifier = None