// Copyright (c) 2021 Marcus Valtonen Örnhag
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.

#include <Eigen/Dense>
#include <cmath>  // max
#include <vector>
#include <complex>
#include "get_valtonenornhag_arxiv_2021.hpp"
#include "quartic.hpp"
#include "relpose.hpp"
#include "normalize2dpts.hpp"

namespace DronePoseLib {
namespace ValtonenOrnhagArxiv2021 {
    inline Eigen::Matrix<double, 5, 1> calculate_coefficients(const Eigen::Matrix<double, 27, 1> &x);
    inline Eigen::Matrix3d create_M(const double w, const Eigen::Matrix<double, 27, 1> &x);

    std::vector<RelPose> get_fEf(
        const Eigen::MatrixXd &p1,
        const Eigen::MatrixXd &p2,
        const Eigen::Matrix3d &R1,
        const Eigen::Matrix3d &R2
    ) {
        // This is a 3-point method
        int nbr_pts = 3;

        // We expect inhomogenous input data, i.e. p1 and p2 are 2x3 matrices
        assert(p1.rows() == 2);
        assert(p2.rows() == 2);
        assert(p1.cols() == nbr_pts);
        assert(p2.cols() == nbr_pts);

        // Compute normalization matrix
        double scale1 = normalize2dpts(p1);
        double scale2 = normalize2dpts(p2);
        double scale = std::max(scale1, scale2);
        Eigen::Vector3d s;
        s << scale, scale, 1.0;
        Eigen::DiagonalMatrix<double, 3> S = s.asDiagonal();

        // Normalize data
        Eigen::Matrix3d x1;
        Eigen::Matrix3d x2;
        x1 = p1.colwise().homogeneous();
        x2 = p2.colwise().homogeneous();
        x1 = S * x1;
        x2 = S * x2;

        // Compute relative rotation
        Eigen::Matrix3d R = R2 * R1.transpose();

        Eigen::Matrix<double, 27, 1> x;
        x << x1.col(0), x1.col(1), x1.col(2),
             x2.col(0), x2.col(1), x2.col(2),
             R.col(0), R.col(1), R.col(2);

        Eigen::Matrix<double, 5, 1> coeffs = calculate_coefficients(x);

        Eigen::Vector4cd w  = solve_quartic(
            coeffs(1) / coeffs(0),
            coeffs(2) / coeffs(0),
            coeffs(3) / coeffs(0),
            coeffs(4) / coeffs(0));

        // Construct putative output
        double thresh = 1e-12;
        Eigen::Matrix3d M;
        std::vector<RelPose> output;
        RelPose relpose;
        Eigen::Matrix3d Q;
        Eigen::Vector3d kinv;
        Eigen::DiagonalMatrix<double, 3> Kinv;
        Eigen::Matrix3d skew_t;

        // Householder is just 500 ns slower than analytic, but is significantly more stable
        for (int i=0; i < w.size(); i++) {
            if (std::abs(std::imag(w[i])) < thresh) {
                M = create_M(std::real(w[i]), x);
                auto qr = M.transpose().colPivHouseholderQr();
                Q = qr.householderQ();
                relpose.t = Q.col(2);
                relpose.f = std::real(w[i]) / scale;

                // Compute fundamental matrix
                kinv << 1.0 / relpose.f, 1.0 / relpose.f, 1.0;
                Kinv = kinv.asDiagonal();
                skew_t << 0, -relpose.t(2), relpose.t(1),
                          relpose.t(2), 0, -relpose.t(0),
                         -relpose.t(1), relpose.t(0), 0;
                relpose.F = Kinv * skew_t * R * Kinv;

                output.push_back(relpose);
            }
        }

        return output;
    }

    inline Eigen::Matrix<double, 5, 1> calculate_coefficients(const Eigen::Matrix<double, 27, 1> &x) {
    Eigen::Matrix<double, 5, 1> coeffs;
    coeffs << x[0]*x[5]*x[8]*x[11]*x[12]*x[17]*x[18]*std::pow(x[25], 2) - x[0]*x[5]*x[8]*x[11]*x[12]*x[17]*x[19]*x[24]*x[25] - x[0]*x[5]*x[8]*x[11]*x[13]*x[17]*x[18]*x[24]*x[25] + x[0]*x[5]*x[8]*x[11]*x[13]*x[17]*x[19]*std::pow(x[24], 2) - x[0]*x[5]*x[8]*x[11]*x[14]*x[15]*x[18]*std::pow(x[25], 2) + x[0]*x[5]*x[8]*x[11]*x[14]*x[15]*x[19]*x[24]*x[25] + x[0]*x[5]*x[8]*x[11]*x[14]*x[16]*x[18]*x[24]*x[25] - x[0]*x[5]*x[8]*x[11]*x[14]*x[16]*x[19]*std::pow(x[24], 2) + x[1]*x[5]*x[8]*x[11]*x[12]*x[17]*x[21]*std::pow(x[25], 2) - x[1]*x[5]*x[8]*x[11]*x[12]*x[17]*x[22]*x[24]*x[25] - x[1]*x[5]*x[8]*x[11]*x[13]*x[17]*x[21]*x[24]*x[25] + x[1]*x[5]*x[8]*x[11]*x[13]*x[17]*x[22]*std::pow(x[24], 2) - x[1]*x[5]*x[8]*x[11]*x[14]*x[15]*x[21]*std::pow(x[25], 2) + x[1]*x[5]*x[8]*x[11]*x[14]*x[15]*x[22]*x[24]*x[25] + x[1]*x[5]*x[8]*x[11]*x[14]*x[16]*x[21]*x[24]*x[25] - x[1]*x[5]*x[8]*x[11]*x[14]*x[16]*x[22]*std::pow(x[24], 2) - x[2]*x[3]*x[8]*x[9]*x[14]*x[17]*x[18]*std::pow(x[25], 2) + x[2]*x[3]*x[8]*x[9]*x[14]*x[17]*x[19]*x[24]*x[25] + x[2]*x[3]*x[8]*x[10]*x[14]*x[17]*x[18]*x[24]*x[25] - x[2]*x[3]*x[8]*x[10]*x[14]*x[17]*x[19]*std::pow(x[24], 2) + x[2]*x[3]*x[8]*x[11]*x[14]*x[15]*x[18]*std::pow(x[25], 2) - x[2]*x[3]*x[8]*x[11]*x[14]*x[15]*x[19]*x[24]*x[25] - x[2]*x[3]*x[8]*x[11]*x[14]*x[16]*x[18]*x[24]*x[25] + x[2]*x[3]*x[8]*x[11]*x[14]*x[16]*x[19]*std::pow(x[24], 2) - x[2]*x[4]*x[8]*x[9]*x[14]*x[17]*x[21]*std::pow(x[25], 2) + x[2]*x[4]*x[8]*x[9]*x[14]*x[17]*x[22]*x[24]*x[25] + x[2]*x[4]*x[8]*x[10]*x[14]*x[17]*x[21]*x[24]*x[25] - x[2]*x[4]*x[8]*x[10]*x[14]*x[17]*x[22]*std::pow(x[24], 2) + x[2]*x[4]*x[8]*x[11]*x[14]*x[15]*x[21]*std::pow(x[25], 2) - x[2]*x[4]*x[8]*x[11]*x[14]*x[15]*x[22]*x[24]*x[25] - x[2]*x[4]*x[8]*x[11]*x[14]*x[16]*x[21]*x[24]*x[25] + x[2]*x[4]*x[8]*x[11]*x[14]*x[16]*x[22]*std::pow(x[24], 2) + x[2]*x[5]*x[6]*x[9]*x[14]*x[17]*x[18]*std::pow(x[25], 2) - x[2]*x[5]*x[6]*x[9]*x[14]*x[17]*x[19]*x[24]*x[25] - x[2]*x[5]*x[6]*x[10]*x[14]*x[17]*x[18]*x[24]*x[25] + x[2]*x[5]*x[6]*x[10]*x[14]*x[17]*x[19]*std::pow(x[24], 2) - x[2]*x[5]*x[6]*x[11]*x[12]*x[17]*x[18]*std::pow(x[25], 2) + x[2]*x[5]*x[6]*x[11]*x[12]*x[17]*x[19]*x[24]*x[25] + x[2]*x[5]*x[6]*x[11]*x[13]*x[17]*x[18]*x[24]*x[25] - x[2]*x[5]*x[6]*x[11]*x[13]*x[17]*x[19]*std::pow(x[24], 2) + x[2]*x[5]*x[7]*x[9]*x[14]*x[17]*x[21]*std::pow(x[25], 2) - x[2]*x[5]*x[7]*x[9]*x[14]*x[17]*x[22]*x[24]*x[25] - x[2]*x[5]*x[7]*x[10]*x[14]*x[17]*x[21]*x[24]*x[25] + x[2]*x[5]*x[7]*x[10]*x[14]*x[17]*x[22]*std::pow(x[24], 2) - x[2]*x[5]*x[7]*x[11]*x[12]*x[17]*x[21]*std::pow(x[25], 2) + x[2]*x[5]*x[7]*x[11]*x[12]*x[17]*x[22]*x[24]*x[25] + x[2]*x[5]*x[7]*x[11]*x[13]*x[17]*x[21]*x[24]*x[25] - x[2]*x[5]*x[7]*x[11]*x[13]*x[17]*x[22]*std::pow(x[24], 2),  // NOLINT [whitespace/line_length]
     -x[0]*x[3]*x[8]*x[9]*x[14]*x[17]*x[18]*x[19]*x[25] + x[0]*x[3]*x[8]*x[9]*x[14]*x[17]*std::pow(x[19], 2)*x[24] + x[0]*x[3]*x[8]*x[10]*x[14]*x[17]*std::pow(x[18], 2)*x[25] - x[0]*x[3]*x[8]*x[10]*x[14]*x[17]*x[18]*x[19]*x[24] + x[0]*x[3]*x[8]*x[11]*x[12]*x[17]*x[18]*x[19]*x[25] - x[0]*x[3]*x[8]*x[11]*x[12]*x[17]*std::pow(x[19], 2)*x[24] - x[0]*x[3]*x[8]*x[11]*x[13]*x[17]*std::pow(x[18], 2)*x[25] + x[0]*x[3]*x[8]*x[11]*x[13]*x[17]*x[18]*x[19]*x[24] - x[0]*x[4]*x[8]*x[9]*x[14]*x[17]*x[19]*x[21]*x[25] + x[0]*x[4]*x[8]*x[9]*x[14]*x[17]*x[19]*x[22]*x[24] + x[0]*x[4]*x[8]*x[10]*x[14]*x[17]*x[18]*x[21]*x[25] - x[0]*x[4]*x[8]*x[10]*x[14]*x[17]*x[18]*x[22]*x[24] + x[0]*x[4]*x[8]*x[11]*x[12]*x[17]*x[18]*x[22]*x[25] - x[0]*x[4]*x[8]*x[11]*x[12]*x[17]*x[19]*x[22]*x[24] - x[0]*x[4]*x[8]*x[11]*x[13]*x[17]*x[18]*x[21]*x[25] + x[0]*x[4]*x[8]*x[11]*x[13]*x[17]*x[19]*x[21]*x[24] - x[0]*x[4]*x[8]*x[11]*x[14]*x[15]*x[18]*x[22]*x[25] + x[0]*x[4]*x[8]*x[11]*x[14]*x[15]*x[19]*x[21]*x[25] + x[0]*x[4]*x[8]*x[11]*x[14]*x[16]*x[18]*x[22]*x[24] - x[0]*x[4]*x[8]*x[11]*x[14]*x[16]*x[19]*x[21]*x[24] + x[0]*x[5]*x[6]*x[9]*x[14]*x[17]*x[18]*x[19]*x[25] - x[0]*x[5]*x[6]*x[9]*x[14]*x[17]*std::pow(x[19], 2)*x[24] - x[0]*x[5]*x[6]*x[10]*x[14]*x[17]*std::pow(x[18], 2)*x[25] + x[0]*x[5]*x[6]*x[10]*x[14]*x[17]*x[18]*x[19]*x[24] - x[0]*x[5]*x[6]*x[11]*x[14]*x[15]*x[18]*x[19]*x[25] + x[0]*x[5]*x[6]*x[11]*x[14]*x[15]*std::pow(x[19], 2)*x[24] + x[0]*x[5]*x[6]*x[11]*x[14]*x[16]*std::pow(x[18], 2)*x[25] - x[0]*x[5]*x[6]*x[11]*x[14]*x[16]*x[18]*x[19]*x[24] + x[0]*x[5]*x[7]*x[9]*x[14]*x[17]*x[19]*x[21]*x[25] - x[0]*x[5]*x[7]*x[9]*x[14]*x[17]*x[19]*x[22]*x[24] - x[0]*x[5]*x[7]*x[10]*x[14]*x[17]*x[18]*x[21]*x[25] + x[0]*x[5]*x[7]*x[10]*x[14]*x[17]*x[18]*x[22]*x[24] + x[0]*x[5]*x[7]*x[11]*x[12]*x[17]*x[18]*x[22]*x[25] - x[0]*x[5]*x[7]*x[11]*x[12]*x[17]*x[19]*x[21]*x[25] - x[0]*x[5]*x[7]*x[11]*x[13]*x[17]*x[18]*x[22]*x[24] + x[0]*x[5]*x[7]*x[11]*x[13]*x[17]*x[19]*x[21]*x[24] - x[0]*x[5]*x[7]*x[11]*x[14]*x[15]*x[18]*x[22]*x[25] + x[0]*x[5]*x[7]*x[11]*x[14]*x[15]*x[19]*x[22]*x[24] + x[0]*x[5]*x[7]*x[11]*x[14]*x[16]*x[18]*x[21]*x[25] - x[0]*x[5]*x[7]*x[11]*x[14]*x[16]*x[19]*x[21]*x[24] + x[0]*x[5]*x[8]*x[9]*x[12]*x[17]*x[19]*x[25]*x[26] - x[0]*x[5]*x[8]*x[9]*x[12]*x[17]*x[20]*std::pow(x[25], 2) - x[0]*x[5]*x[8]*x[9]*x[13]*x[17]*x[19]*x[24]*x[26] + x[0]*x[5]*x[8]*x[9]*x[13]*x[17]*x[20]*x[24]*x[25] - x[0]*x[5]*x[8]*x[9]*x[14]*x[15]*x[19]*x[25]*x[26] + x[0]*x[5]*x[8]*x[9]*x[14]*x[15]*x[20]*std::pow(x[25], 2) + x[0]*x[5]*x[8]*x[9]*x[14]*x[16]*x[19]*x[24]*x[26] - x[0]*x[5]*x[8]*x[9]*x[14]*x[16]*x[20]*x[24]*x[25] - x[0]*x[5]*x[8]*x[10]*x[12]*x[17]*x[18]*x[25]*x[26] + x[0]*x[5]*x[8]*x[10]*x[12]*x[17]*x[20]*x[24]*x[25] + x[0]*x[5]*x[8]*x[10]*x[13]*x[17]*x[18]*x[24]*x[26] - x[0]*x[5]*x[8]*x[10]*x[13]*x[17]*x[20]*std::pow(x[24], 2) + x[0]*x[5]*x[8]*x[10]*x[14]*x[15]*x[18]*x[25]*x[26] - x[0]*x[5]*x[8]*x[10]*x[14]*x[15]*x[20]*x[24]*x[25] - x[0]*x[5]*x[8]*x[10]*x[14]*x[16]*x[18]*x[24]*x[26] + x[0]*x[5]*x[8]*x[10]*x[14]*x[16]*x[20]*std::pow(x[24], 2) - x[0]*x[5]*x[8]*x[11]*x[12]*x[16]*x[18]*x[25]*x[26] + x[0]*x[5]*x[8]*x[11]*x[12]*x[16]*x[19]*x[24]*x[26] + x[0]*x[5]*x[8]*x[11]*x[13]*x[15]*x[18]*x[25]*x[26] - x[0]*x[5]*x[8]*x[11]*x[13]*x[15]*x[19]*x[24]*x[26] - x[1]*x[3]*x[8]*x[9]*x[14]*x[17]*x[18]*x[22]*x[25] + x[1]*x[3]*x[8]*x[9]*x[14]*x[17]*x[19]*x[22]*x[24] + x[1]*x[3]*x[8]*x[10]*x[14]*x[17]*x[18]*x[21]*x[25] - x[1]*x[3]*x[8]*x[10]*x[14]*x[17]*x[19]*x[21]*x[24] + x[1]*x[3]*x[8]*x[11]*x[12]*x[17]*x[19]*x[21]*x[25] - x[1]*x[3]*x[8]*x[11]*x[12]*x[17]*x[19]*x[22]*x[24] - x[1]*x[3]*x[8]*x[11]*x[13]*x[17]*x[18]*x[21]*x[25] + x[1]*x[3]*x[8]*x[11]*x[13]*x[17]*x[18]*x[22]*x[24] + x[1]*x[3]*x[8]*x[11]*x[14]*x[15]*x[18]*x[22]*x[25] - x[1]*x[3]*x[8]*x[11]*x[14]*x[15]*x[19]*x[21]*x[25] - x[1]*x[3]*x[8]*x[11]*x[14]*x[16]*x[18]*x[22]*x[24] + x[1]*x[3]*x[8]*x[11]*x[14]*x[16]*x[19]*x[21]*x[24] - x[1]*x[4]*x[8]*x[9]*x[14]*x[17]*x[21]*x[22]*x[25] + x[1]*x[4]*x[8]*x[9]*x[14]*x[17]*std::pow(x[22], 2)*x[24] + x[1]*x[4]*x[8]*x[10]*x[14]*x[17]*std::pow(x[21], 2)*x[25] - x[1]*x[4]*x[8]*x[10]*x[14]*x[17]*x[21]*x[22]*x[24] + x[1]*x[4]*x[8]*x[11]*x[12]*x[17]*x[21]*x[22]*x[25] - x[1]*x[4]*x[8]*x[11]*x[12]*x[17]*std::pow(x[22], 2)*x[24] - x[1]*x[4]*x[8]*x[11]*x[13]*x[17]*std::pow(x[21], 2)*x[25] + x[1]*x[4]*x[8]*x[11]*x[13]*x[17]*x[21]*x[22]*x[24] + x[1]*x[5]*x[6]*x[9]*x[14]*x[17]*x[18]*x[22]*x[25] - x[1]*x[5]*x[6]*x[9]*x[14]*x[17]*x[19]*x[22]*x[24] - x[1]*x[5]*x[6]*x[10]*x[14]*x[17]*x[18]*x[21]*x[25] + x[1]*x[5]*x[6]*x[10]*x[14]*x[17]*x[19]*x[21]*x[24] - x[1]*x[5]*x[6]*x[11]*x[12]*x[17]*x[18]*x[22]*x[25] + x[1]*x[5]*x[6]*x[11]*x[12]*x[17]*x[19]*x[21]*x[25] + x[1]*x[5]*x[6]*x[11]*x[13]*x[17]*x[18]*x[22]*x[24] - x[1]*x[5]*x[6]*x[11]*x[13]*x[17]*x[19]*x[21]*x[24] - x[1]*x[5]*x[6]*x[11]*x[14]*x[15]*x[19]*x[21]*x[25] + x[1]*x[5]*x[6]*x[11]*x[14]*x[15]*x[19]*x[22]*x[24] + x[1]*x[5]*x[6]*x[11]*x[14]*x[16]*x[18]*x[21]*x[25] - x[1]*x[5]*x[6]*x[11]*x[14]*x[16]*x[18]*x[22]*x[24] + x[1]*x[5]*x[7]*x[9]*x[14]*x[17]*x[21]*x[22]*x[25] - x[1]*x[5]*x[7]*x[9]*x[14]*x[17]*std::pow(x[22], 2)*x[24] - x[1]*x[5]*x[7]*x[10]*x[14]*x[17]*std::pow(x[21], 2)*x[25] + x[1]*x[5]*x[7]*x[10]*x[14]*x[17]*x[21]*x[22]*x[24] - x[1]*x[5]*x[7]*x[11]*x[14]*x[15]*x[21]*x[22]*x[25] + x[1]*x[5]*x[7]*x[11]*x[14]*x[15]*std::pow(x[22], 2)*x[24] + x[1]*x[5]*x[7]*x[11]*x[14]*x[16]*std::pow(x[21], 2)*x[25] - x[1]*x[5]*x[7]*x[11]*x[14]*x[16]*x[21]*x[22]*x[24] + x[1]*x[5]*x[8]*x[9]*x[12]*x[17]*x[22]*x[25]*x[26] - x[1]*x[5]*x[8]*x[9]*x[12]*x[17]*x[23]*std::pow(x[25], 2) - x[1]*x[5]*x[8]*x[9]*x[13]*x[17]*x[22]*x[24]*x[26] + x[1]*x[5]*x[8]*x[9]*x[13]*x[17]*x[23]*x[24]*x[25] - x[1]*x[5]*x[8]*x[9]*x[14]*x[15]*x[22]*x[25]*x[26] + x[1]*x[5]*x[8]*x[9]*x[14]*x[15]*x[23]*std::pow(x[25], 2) + x[1]*x[5]*x[8]*x[9]*x[14]*x[16]*x[22]*x[24]*x[26] - x[1]*x[5]*x[8]*x[9]*x[14]*x[16]*x[23]*x[24]*x[25] - x[1]*x[5]*x[8]*x[10]*x[12]*x[17]*x[21]*x[25]*x[26] + x[1]*x[5]*x[8]*x[10]*x[12]*x[17]*x[23]*x[24]*x[25] + x[1]*x[5]*x[8]*x[10]*x[13]*x[17]*x[21]*x[24]*x[26] - x[1]*x[5]*x[8]*x[10]*x[13]*x[17]*x[23]*std::pow(x[24], 2) + x[1]*x[5]*x[8]*x[10]*x[14]*x[15]*x[21]*x[25]*x[26] - x[1]*x[5]*x[8]*x[10]*x[14]*x[15]*x[23]*x[24]*x[25] - x[1]*x[5]*x[8]*x[10]*x[14]*x[16]*x[21]*x[24]*x[26] + x[1]*x[5]*x[8]*x[10]*x[14]*x[16]*x[23]*std::pow(x[24], 2) - x[1]*x[5]*x[8]*x[11]*x[12]*x[16]*x[21]*x[25]*x[26] + x[1]*x[5]*x[8]*x[11]*x[12]*x[16]*x[22]*x[24]*x[26] + x[1]*x[5]*x[8]*x[11]*x[13]*x[15]*x[21]*x[25]*x[26] - x[1]*x[5]*x[8]*x[11]*x[13]*x[15]*x[22]*x[24]*x[26] - x[2]*x[3]*x[6]*x[11]*x[12]*x[17]*x[18]*x[19]*x[25] + x[2]*x[3]*x[6]*x[11]*x[12]*x[17]*std::pow(x[19], 2)*x[24] + x[2]*x[3]*x[6]*x[11]*x[13]*x[17]*std::pow(x[18], 2)*x[25] - x[2]*x[3]*x[6]*x[11]*x[13]*x[17]*x[18]*x[19]*x[24] + x[2]*x[3]*x[6]*x[11]*x[14]*x[15]*x[18]*x[19]*x[25] - x[2]*x[3]*x[6]*x[11]*x[14]*x[15]*std::pow(x[19], 2)*x[24] - x[2]*x[3]*x[6]*x[11]*x[14]*x[16]*std::pow(x[18], 2)*x[25] + x[2]*x[3]*x[6]*x[11]*x[14]*x[16]*x[18]*x[19]*x[24] - x[2]*x[3]*x[7]*x[9]*x[14]*x[17]*x[18]*x[22]*x[25] + x[2]*x[3]*x[7]*x[9]*x[14]*x[17]*x[19]*x[21]*x[25] + x[2]*x[3]*x[7]*x[10]*x[14]*x[17]*x[18]*x[22]*x[24] - x[2]*x[3]*x[7]*x[10]*x[14]*x[17]*x[19]*x[21]*x[24] - x[2]*x[3]*x[7]*x[11]*x[12]*x[17]*x[19]*x[21]*x[25] + x[2]*x[3]*x[7]*x[11]*x[12]*x[17]*x[19]*x[22]*x[24] + x[2]*x[3]*x[7]*x[11]*x[13]*x[17]*x[18]*x[21]*x[25] - x[2]*x[3]*x[7]*x[11]*x[13]*x[17]*x[18]*x[22]*x[24] + x[2]*x[3]*x[7]*x[11]*x[14]*x[15]*x[18]*x[22]*x[25] - x[2]*x[3]*x[7]*x[11]*x[14]*x[15]*x[19]*x[22]*x[24] - x[2]*x[3]*x[7]*x[11]*x[14]*x[16]*x[18]*x[21]*x[25] + x[2]*x[3]*x[7]*x[11]*x[14]*x[16]*x[19]*x[21]*x[24] - x[2]*x[3]*x[8]*x[9]*x[12]*x[17]*x[19]*x[25]*x[26] + x[2]*x[3]*x[8]*x[9]*x[12]*x[17]*x[20]*std::pow(x[25], 2) + x[2]*x[3]*x[8]*x[9]*x[13]*x[17]*x[18]*x[25]*x[26] - x[2]*x[3]*x[8]*x[9]*x[13]*x[17]*x[20]*x[24]*x[25] + x[2]*x[3]*x[8]*x[9]*x[14]*x[16]*x[18]*x[25]*x[26] - x[2]*x[3]*x[8]*x[9]*x[14]*x[16]*x[19]*x[24]*x[26] + x[2]*x[3]*x[8]*x[10]*x[12]*x[17]*x[19]*x[24]*x[26] - x[2]*x[3]*x[8]*x[10]*x[12]*x[17]*x[20]*x[24]*x[25] - x[2]*x[3]*x[8]*x[10]*x[13]*x[17]*x[18]*x[24]*x[26] + x[2]*x[3]*x[8]*x[10]*x[13]*x[17]*x[20]*std::pow(x[24], 2) - x[2]*x[3]*x[8]*x[10]*x[14]*x[15]*x[18]*x[25]*x[26] + x[2]*x[3]*x[8]*x[10]*x[14]*x[15]*x[19]*x[24]*x[26] + x[2]*x[3]*x[8]*x[11]*x[12]*x[15]*x[19]*x[25]*x[26] - x[2]*x[3]*x[8]*x[11]*x[12]*x[15]*x[20]*std::pow(x[25], 2) - x[2]*x[3]*x[8]*x[11]*x[12]*x[16]*x[19]*x[24]*x[26] + x[2]*x[3]*x[8]*x[11]*x[12]*x[16]*x[20]*x[24]*x[25] - x[2]*x[3]*x[8]*x[11]*x[13]*x[15]*x[18]*x[25]*x[26] + x[2]*x[3]*x[8]*x[11]*x[13]*x[15]*x[20]*x[24]*x[25] + x[2]*x[3]*x[8]*x[11]*x[13]*x[16]*x[18]*x[24]*x[26] - x[2]*x[3]*x[8]*x[11]*x[13]*x[16]*x[20]*std::pow(x[24], 2) + x[2]*x[4]*x[6]*x[9]*x[14]*x[17]*x[18]*x[22]*x[25] - x[2]*x[4]*x[6]*x[9]*x[14]*x[17]*x[19]*x[21]*x[25] - x[2]*x[4]*x[6]*x[10]*x[14]*x[17]*x[18]*x[22]*x[24] + x[2]*x[4]*x[6]*x[10]*x[14]*x[17]*x[19]*x[21]*x[24] - x[2]*x[4]*x[6]*x[11]*x[12]*x[17]*x[18]*x[22]*x[25] + x[2]*x[4]*x[6]*x[11]*x[12]*x[17]*x[19]*x[22]*x[24] + x[2]*x[4]*x[6]*x[11]*x[13]*x[17]*x[18]*x[21]*x[25] - x[2]*x[4]*x[6]*x[11]*x[13]*x[17]*x[19]*x[21]*x[24] + x[2]*x[4]*x[6]*x[11]*x[14]*x[15]*x[19]*x[21]*x[25] - x[2]*x[4]*x[6]*x[11]*x[14]*x[15]*x[19]*x[22]*x[24] - x[2]*x[4]*x[6]*x[11]*x[14]*x[16]*x[18]*x[21]*x[25] + x[2]*x[4]*x[6]*x[11]*x[14]*x[16]*x[18]*x[22]*x[24] - x[2]*x[4]*x[7]*x[11]*x[12]*x[17]*x[21]*x[22]*x[25] + x[2]*x[4]*x[7]*x[11]*x[12]*x[17]*std::pow(x[22], 2)*x[24] + x[2]*x[4]*x[7]*x[11]*x[13]*x[17]*std::pow(x[21], 2)*x[25] - x[2]*x[4]*x[7]*x[11]*x[13]*x[17]*x[21]*x[22]*x[24] + x[2]*x[4]*x[7]*x[11]*x[14]*x[15]*x[21]*x[22]*x[25] - x[2]*x[4]*x[7]*x[11]*x[14]*x[15]*std::pow(x[22], 2)*x[24] - x[2]*x[4]*x[7]*x[11]*x[14]*x[16]*std::pow(x[21], 2)*x[25] + x[2]*x[4]*x[7]*x[11]*x[14]*x[16]*x[21]*x[22]*x[24] - x[2]*x[4]*x[8]*x[9]*x[12]*x[17]*x[22]*x[25]*x[26] + x[2]*x[4]*x[8]*x[9]*x[12]*x[17]*x[23]*std::pow(x[25], 2) + x[2]*x[4]*x[8]*x[9]*x[13]*x[17]*x[21]*x[25]*x[26] - x[2]*x[4]*x[8]*x[9]*x[13]*x[17]*x[23]*x[24]*x[25] + x[2]*x[4]*x[8]*x[9]*x[14]*x[16]*x[21]*x[25]*x[26] - x[2]*x[4]*x[8]*x[9]*x[14]*x[16]*x[22]*x[24]*x[26] + x[2]*x[4]*x[8]*x[10]*x[12]*x[17]*x[22]*x[24]*x[26] - x[2]*x[4]*x[8]*x[10]*x[12]*x[17]*x[23]*x[24]*x[25] - x[2]*x[4]*x[8]*x[10]*x[13]*x[17]*x[21]*x[24]*x[26] + x[2]*x[4]*x[8]*x[10]*x[13]*x[17]*x[23]*std::pow(x[24], 2) - x[2]*x[4]*x[8]*x[10]*x[14]*x[15]*x[21]*x[25]*x[26] + x[2]*x[4]*x[8]*x[10]*x[14]*x[15]*x[22]*x[24]*x[26] + x[2]*x[4]*x[8]*x[11]*x[12]*x[15]*x[22]*x[25]*x[26] - x[2]*x[4]*x[8]*x[11]*x[12]*x[15]*x[23]*std::pow(x[25], 2) - x[2]*x[4]*x[8]*x[11]*x[12]*x[16]*x[22]*x[24]*x[26] + x[2]*x[4]*x[8]*x[11]*x[12]*x[16]*x[23]*x[24]*x[25] - x[2]*x[4]*x[8]*x[11]*x[13]*x[15]*x[21]*x[25]*x[26] + x[2]*x[4]*x[8]*x[11]*x[13]*x[15]*x[23]*x[24]*x[25] + x[2]*x[4]*x[8]*x[11]*x[13]*x[16]*x[21]*x[24]*x[26] - x[2]*x[4]*x[8]*x[11]*x[13]*x[16]*x[23]*std::pow(x[24], 2) - x[2]*x[5]*x[6]*x[9]*x[13]*x[17]*x[18]*x[25]*x[26] + x[2]*x[5]*x[6]*x[9]*x[13]*x[17]*x[19]*x[24]*x[26] + x[2]*x[5]*x[6]*x[9]*x[14]*x[15]*x[19]*x[25]*x[26] - x[2]*x[5]*x[6]*x[9]*x[14]*x[15]*x[20]*std::pow(x[25], 2) - x[2]*x[5]*x[6]*x[9]*x[14]*x[16]*x[18]*x[25]*x[26] + x[2]*x[5]*x[6]*x[9]*x[14]*x[16]*x[20]*x[24]*x[25] + x[2]*x[5]*x[6]*x[10]*x[12]*x[17]*x[18]*x[25]*x[26] - x[2]*x[5]*x[6]*x[10]*x[12]*x[17]*x[19]*x[24]*x[26] - x[2]*x[5]*x[6]*x[10]*x[14]*x[15]*x[19]*x[24]*x[26] + x[2]*x[5]*x[6]*x[10]*x[14]*x[15]*x[20]*x[24]*x[25] + x[2]*x[5]*x[6]*x[10]*x[14]*x[16]*x[18]*x[24]*x[26] - x[2]*x[5]*x[6]*x[10]*x[14]*x[16]*x[20]*std::pow(x[24], 2) - x[2]*x[5]*x[6]*x[11]*x[12]*x[15]*x[19]*x[25]*x[26] + x[2]*x[5]*x[6]*x[11]*x[12]*x[15]*x[20]*std::pow(x[25], 2) + x[2]*x[5]*x[6]*x[11]*x[12]*x[16]*x[18]*x[25]*x[26] - x[2]*x[5]*x[6]*x[11]*x[12]*x[16]*x[20]*x[24]*x[25] + x[2]*x[5]*x[6]*x[11]*x[13]*x[15]*x[19]*x[24]*x[26] - x[2]*x[5]*x[6]*x[11]*x[13]*x[15]*x[20]*x[24]*x[25] - x[2]*x[5]*x[6]*x[11]*x[13]*x[16]*x[18]*x[24]*x[26] + x[2]*x[5]*x[6]*x[11]*x[13]*x[16]*x[20]*std::pow(x[24], 2) - x[2]*x[5]*x[7]*x[9]*x[13]*x[17]*x[21]*x[25]*x[26] + x[2]*x[5]*x[7]*x[9]*x[13]*x[17]*x[22]*x[24]*x[26] + x[2]*x[5]*x[7]*x[9]*x[14]*x[15]*x[22]*x[25]*x[26] - x[2]*x[5]*x[7]*x[9]*x[14]*x[15]*x[23]*std::pow(x[25], 2) - x[2]*x[5]*x[7]*x[9]*x[14]*x[16]*x[21]*x[25]*x[26] + x[2]*x[5]*x[7]*x[9]*x[14]*x[16]*x[23]*x[24]*x[25] + x[2]*x[5]*x[7]*x[10]*x[12]*x[17]*x[21]*x[25]*x[26] - x[2]*x[5]*x[7]*x[10]*x[12]*x[17]*x[22]*x[24]*x[26] - x[2]*x[5]*x[7]*x[10]*x[14]*x[15]*x[22]*x[24]*x[26] + x[2]*x[5]*x[7]*x[10]*x[14]*x[15]*x[23]*x[24]*x[25] + x[2]*x[5]*x[7]*x[10]*x[14]*x[16]*x[21]*x[24]*x[26] - x[2]*x[5]*x[7]*x[10]*x[14]*x[16]*x[23]*std::pow(x[24], 2) - x[2]*x[5]*x[7]*x[11]*x[12]*x[15]*x[22]*x[25]*x[26] + x[2]*x[5]*x[7]*x[11]*x[12]*x[15]*x[23]*std::pow(x[25], 2) + x[2]*x[5]*x[7]*x[11]*x[12]*x[16]*x[21]*x[25]*x[26] - x[2]*x[5]*x[7]*x[11]*x[12]*x[16]*x[23]*x[24]*x[25] + x[2]*x[5]*x[7]*x[11]*x[13]*x[15]*x[22]*x[24]*x[26] - x[2]*x[5]*x[7]*x[11]*x[13]*x[15]*x[23]*x[24]*x[25] - x[2]*x[5]*x[7]*x[11]*x[13]*x[16]*x[21]*x[24]*x[26] + x[2]*x[5]*x[7]*x[11]*x[13]*x[16]*x[23]*std::pow(x[24], 2),  // NOLINT [whitespace/line_length]
     -x[0]*x[3]*x[7]*x[9]*x[14]*x[17]*x[18]*x[19]*x[22] + x[0]*x[3]*x[7]*x[9]*x[14]*x[17]*std::pow(x[19], 2)*x[21] + x[0]*x[3]*x[7]*x[10]*x[14]*x[17]*std::pow(x[18], 2)*x[22] - x[0]*x[3]*x[7]*x[10]*x[14]*x[17]*x[18]*x[19]*x[21] + x[0]*x[3]*x[7]*x[11]*x[12]*x[17]*x[18]*x[19]*x[22] - x[0]*x[3]*x[7]*x[11]*x[12]*x[17]*std::pow(x[19], 2)*x[21] - x[0]*x[3]*x[7]*x[11]*x[13]*x[17]*std::pow(x[18], 2)*x[22] + x[0]*x[3]*x[7]*x[11]*x[13]*x[17]*x[18]*x[19]*x[21] + x[0]*x[3]*x[8]*x[9]*x[13]*x[17]*x[18]*x[20]*x[25] - x[0]*x[3]*x[8]*x[9]*x[13]*x[17]*x[19]*x[20]*x[24] - x[0]*x[3]*x[8]*x[9]*x[14]*x[15]*std::pow(x[19], 2)*x[26] + x[0]*x[3]*x[8]*x[9]*x[14]*x[15]*x[19]*x[20]*x[25] + x[0]*x[3]*x[8]*x[9]*x[14]*x[16]*x[18]*x[19]*x[26] - x[0]*x[3]*x[8]*x[9]*x[14]*x[16]*x[19]*x[20]*x[24] - x[0]*x[3]*x[8]*x[10]*x[12]*x[17]*x[18]*x[20]*x[25] + x[0]*x[3]*x[8]*x[10]*x[12]*x[17]*x[19]*x[20]*x[24] + x[0]*x[3]*x[8]*x[10]*x[14]*x[15]*x[18]*x[19]*x[26] - x[0]*x[3]*x[8]*x[10]*x[14]*x[15]*x[18]*x[20]*x[25] - x[0]*x[3]*x[8]*x[10]*x[14]*x[16]*std::pow(x[18], 2)*x[26] + x[0]*x[3]*x[8]*x[10]*x[14]*x[16]*x[18]*x[20]*x[24] + x[0]*x[3]*x[8]*x[11]*x[12]*x[15]*std::pow(x[19], 2)*x[26] - x[0]*x[3]*x[8]*x[11]*x[12]*x[15]*x[19]*x[20]*x[25] - x[0]*x[3]*x[8]*x[11]*x[12]*x[16]*x[18]*x[19]*x[26] + x[0]*x[3]*x[8]*x[11]*x[12]*x[16]*x[19]*x[20]*x[24] - x[0]*x[3]*x[8]*x[11]*x[13]*x[15]*x[18]*x[19]*x[26] + x[0]*x[3]*x[8]*x[11]*x[13]*x[15]*x[18]*x[20]*x[25] + x[0]*x[3]*x[8]*x[11]*x[13]*x[16]*std::pow(x[18], 2)*x[26] - x[0]*x[3]*x[8]*x[11]*x[13]*x[16]*x[18]*x[20]*x[24] + x[0]*x[4]*x[6]*x[9]*x[14]*x[17]*x[18]*x[19]*x[22] - x[0]*x[4]*x[6]*x[9]*x[14]*x[17]*std::pow(x[19], 2)*x[21] - x[0]*x[4]*x[6]*x[10]*x[14]*x[17]*std::pow(x[18], 2)*x[22] + x[0]*x[4]*x[6]*x[10]*x[14]*x[17]*x[18]*x[19]*x[21] - x[0]*x[4]*x[6]*x[11]*x[14]*x[15]*x[18]*x[19]*x[22] + x[0]*x[4]*x[6]*x[11]*x[14]*x[15]*std::pow(x[19], 2)*x[21] + x[0]*x[4]*x[6]*x[11]*x[14]*x[16]*std::pow(x[18], 2)*x[22] - x[0]*x[4]*x[6]*x[11]*x[14]*x[16]*x[18]*x[19]*x[21] + x[0]*x[4]*x[7]*x[11]*x[12]*x[17]*x[18]*std::pow(x[22], 2) - x[0]*x[4]*x[7]*x[11]*x[12]*x[17]*x[19]*x[21]*x[22] - x[0]*x[4]*x[7]*x[11]*x[13]*x[17]*x[18]*x[21]*x[22] + x[0]*x[4]*x[7]*x[11]*x[13]*x[17]*x[19]*std::pow(x[21], 2) - x[0]*x[4]*x[7]*x[11]*x[14]*x[15]*x[18]*std::pow(x[22], 2) + x[0]*x[4]*x[7]*x[11]*x[14]*x[15]*x[19]*x[21]*x[22] + x[0]*x[4]*x[7]*x[11]*x[14]*x[16]*x[18]*x[21]*x[22] - x[0]*x[4]*x[7]*x[11]*x[14]*x[16]*x[19]*std::pow(x[21], 2) + x[0]*x[4]*x[8]*x[9]*x[12]*x[17]*x[19]*x[23]*x[25] - x[0]*x[4]*x[8]*x[9]*x[12]*x[17]*x[20]*x[22]*x[25] - x[0]*x[4]*x[8]*x[9]*x[13]*x[17]*x[19]*x[23]*x[24] + x[0]*x[4]*x[8]*x[9]*x[13]*x[17]*x[20]*x[21]*x[25] - x[0]*x[4]*x[8]*x[9]*x[14]*x[15]*x[19]*x[22]*x[26] + x[0]*x[4]*x[8]*x[9]*x[14]*x[15]*x[20]*x[22]*x[25] + x[0]*x[4]*x[8]*x[9]*x[14]*x[16]*x[19]*x[21]*x[26] - x[0]*x[4]*x[8]*x[9]*x[14]*x[16]*x[20]*x[22]*x[24] - x[0]*x[4]*x[8]*x[10]*x[12]*x[17]*x[18]*x[23]*x[25] + x[0]*x[4]*x[8]*x[10]*x[12]*x[17]*x[20]*x[22]*x[24] + x[0]*x[4]*x[8]*x[10]*x[13]*x[17]*x[18]*x[23]*x[24] - x[0]*x[4]*x[8]*x[10]*x[13]*x[17]*x[20]*x[21]*x[24] + x[0]*x[4]*x[8]*x[10]*x[14]*x[15]*x[18]*x[22]*x[26] - x[0]*x[4]*x[8]*x[10]*x[14]*x[15]*x[20]*x[21]*x[25] - x[0]*x[4]*x[8]*x[10]*x[14]*x[16]*x[18]*x[21]*x[26] + x[0]*x[4]*x[8]*x[10]*x[14]*x[16]*x[20]*x[21]*x[24] + x[0]*x[4]*x[8]*x[11]*x[12]*x[15]*x[19]*x[22]*x[26] - x[0]*x[4]*x[8]*x[11]*x[12]*x[15]*x[19]*x[23]*x[25] - x[0]*x[4]*x[8]*x[11]*x[12]*x[16]*x[18]*x[22]*x[26] + x[0]*x[4]*x[8]*x[11]*x[12]*x[16]*x[19]*x[23]*x[24] + x[0]*x[4]*x[8]*x[11]*x[13]*x[15]*x[18]*x[23]*x[25] - x[0]*x[4]*x[8]*x[11]*x[13]*x[15]*x[19]*x[21]*x[26] + x[0]*x[4]*x[8]*x[11]*x[13]*x[16]*x[18]*x[21]*x[26] - x[0]*x[4]*x[8]*x[11]*x[13]*x[16]*x[18]*x[23]*x[24] + x[0]*x[5]*x[6]*x[9]*x[12]*x[17]*std::pow(x[19], 2)*x[26] - x[0]*x[5]*x[6]*x[9]*x[12]*x[17]*x[19]*x[20]*x[25] - x[0]*x[5]*x[6]*x[9]*x[13]*x[17]*x[18]*x[19]*x[26] + x[0]*x[5]*x[6]*x[9]*x[13]*x[17]*x[19]*x[20]*x[24] - x[0]*x[5]*x[6]*x[9]*x[14]*x[16]*x[18]*x[20]*x[25] + x[0]*x[5]*x[6]*x[9]*x[14]*x[16]*x[19]*x[20]*x[24] - x[0]*x[5]*x[6]*x[10]*x[12]*x[17]*x[18]*x[19]*x[26] + x[0]*x[5]*x[6]*x[10]*x[12]*x[17]*x[18]*x[20]*x[25] + x[0]*x[5]*x[6]*x[10]*x[13]*x[17]*std::pow(x[18], 2)*x[26] - x[0]*x[5]*x[6]*x[10]*x[13]*x[17]*x[18]*x[20]*x[24] + x[0]*x[5]*x[6]*x[10]*x[14]*x[15]*x[18]*x[20]*x[25] - x[0]*x[5]*x[6]*x[10]*x[14]*x[15]*x[19]*x[20]*x[24] - x[0]*x[5]*x[6]*x[11]*x[12]*x[15]*std::pow(x[19], 2)*x[26] + x[0]*x[5]*x[6]*x[11]*x[12]*x[15]*x[19]*x[20]*x[25] + x[0]*x[5]*x[6]*x[11]*x[12]*x[16]*x[18]*x[19]*x[26] - x[0]*x[5]*x[6]*x[11]*x[12]*x[16]*x[18]*x[20]*x[25] + x[0]*x[5]*x[6]*x[11]*x[13]*x[15]*x[18]*x[19]*x[26] - x[0]*x[5]*x[6]*x[11]*x[13]*x[15]*x[19]*x[20]*x[24] - x[0]*x[5]*x[6]*x[11]*x[13]*x[16]*std::pow(x[18], 2)*x[26] + x[0]*x[5]*x[6]*x[11]*x[13]*x[16]*x[18]*x[20]*x[24] + x[0]*x[5]*x[7]*x[9]*x[12]*x[17]*x[19]*x[22]*x[26] - x[0]*x[5]*x[7]*x[9]*x[12]*x[17]*x[20]*x[22]*x[25] - x[0]*x[5]*x[7]*x[9]*x[13]*x[17]*x[19]*x[21]*x[26] + x[0]*x[5]*x[7]*x[9]*x[13]*x[17]*x[20]*x[22]*x[24] - x[0]*x[5]*x[7]*x[9]*x[14]*x[15]*x[19]*x[23]*x[25] + x[0]*x[5]*x[7]*x[9]*x[14]*x[15]*x[20]*x[22]*x[25] + x[0]*x[5]*x[7]*x[9]*x[14]*x[16]*x[19]*x[23]*x[24] - x[0]*x[5]*x[7]*x[9]*x[14]*x[16]*x[20]*x[21]*x[25] - x[0]*x[5]*x[7]*x[10]*x[12]*x[17]*x[18]*x[22]*x[26] + x[0]*x[5]*x[7]*x[10]*x[12]*x[17]*x[20]*x[21]*x[25] + x[0]*x[5]*x[7]*x[10]*x[13]*x[17]*x[18]*x[21]*x[26] - x[0]*x[5]*x[7]*x[10]*x[13]*x[17]*x[20]*x[21]*x[24] + x[0]*x[5]*x[7]*x[10]*x[14]*x[15]*x[18]*x[23]*x[25] - x[0]*x[5]*x[7]*x[10]*x[14]*x[15]*x[20]*x[22]*x[24] - x[0]*x[5]*x[7]*x[10]*x[14]*x[16]*x[18]*x[23]*x[24] + x[0]*x[5]*x[7]*x[10]*x[14]*x[16]*x[20]*x[21]*x[24] - x[0]*x[5]*x[7]*x[11]*x[12]*x[15]*x[19]*x[22]*x[26] + x[0]*x[5]*x[7]*x[11]*x[12]*x[15]*x[19]*x[23]*x[25] - x[0]*x[5]*x[7]*x[11]*x[12]*x[16]*x[18]*x[23]*x[25] + x[0]*x[5]*x[7]*x[11]*x[12]*x[16]*x[19]*x[21]*x[26] + x[0]*x[5]*x[7]*x[11]*x[13]*x[15]*x[18]*x[22]*x[26] - x[0]*x[5]*x[7]*x[11]*x[13]*x[15]*x[19]*x[23]*x[24] - x[0]*x[5]*x[7]*x[11]*x[13]*x[16]*x[18]*x[21]*x[26] + x[0]*x[5]*x[7]*x[11]*x[13]*x[16]*x[18]*x[23]*x[24] - x[0]*x[5]*x[8]*x[9]*x[12]*x[16]*x[19]*std::pow(x[26], 2) + x[0]*x[5]*x[8]*x[9]*x[12]*x[16]*x[20]*x[25]*x[26] + x[0]*x[5]*x[8]*x[9]*x[13]*x[15]*x[19]*std::pow(x[26], 2) - x[0]*x[5]*x[8]*x[9]*x[13]*x[15]*x[20]*x[25]*x[26] + x[0]*x[5]*x[8]*x[10]*x[12]*x[16]*x[18]*std::pow(x[26], 2) - x[0]*x[5]*x[8]*x[10]*x[12]*x[16]*x[20]*x[24]*x[26] - x[0]*x[5]*x[8]*x[10]*x[13]*x[15]*x[18]*std::pow(x[26], 2) + x[0]*x[5]*x[8]*x[10]*x[13]*x[15]*x[20]*x[24]*x[26] - x[1]*x[3]*x[6]*x[11]*x[12]*x[17]*x[18]*x[19]*x[22] + x[1]*x[3]*x[6]*x[11]*x[12]*x[17]*std::pow(x[19], 2)*x[21] + x[1]*x[3]*x[6]*x[11]*x[13]*x[17]*std::pow(x[18], 2)*x[22] - x[1]*x[3]*x[6]*x[11]*x[13]*x[17]*x[18]*x[19]*x[21] + x[1]*x[3]*x[6]*x[11]*x[14]*x[15]*x[18]*x[19]*x[22] - x[1]*x[3]*x[6]*x[11]*x[14]*x[15]*std::pow(x[19], 2)*x[21] - x[1]*x[3]*x[6]*x[11]*x[14]*x[16]*std::pow(x[18], 2)*x[22] + x[1]*x[3]*x[6]*x[11]*x[14]*x[16]*x[18]*x[19]*x[21] - x[1]*x[3]*x[7]*x[9]*x[14]*x[17]*x[18]*std::pow(x[22], 2) + x[1]*x[3]*x[7]*x[9]*x[14]*x[17]*x[19]*x[21]*x[22] + x[1]*x[3]*x[7]*x[10]*x[14]*x[17]*x[18]*x[21]*x[22] - x[1]*x[3]*x[7]*x[10]*x[14]*x[17]*x[19]*std::pow(x[21], 2) + x[1]*x[3]*x[7]*x[11]*x[14]*x[15]*x[18]*std::pow(x[22], 2) - x[1]*x[3]*x[7]*x[11]*x[14]*x[15]*x[19]*x[21]*x[22] - x[1]*x[3]*x[7]*x[11]*x[14]*x[16]*x[18]*x[21]*x[22] + x[1]*x[3]*x[7]*x[11]*x[14]*x[16]*x[19]*std::pow(x[21], 2) - x[1]*x[3]*x[8]*x[9]*x[12]*x[17]*x[19]*x[23]*x[25] + x[1]*x[3]*x[8]*x[9]*x[12]*x[17]*x[20]*x[22]*x[25] + x[1]*x[3]*x[8]*x[9]*x[13]*x[17]*x[18]*x[23]*x[25] - x[1]*x[3]*x[8]*x[9]*x[13]*x[17]*x[20]*x[22]*x[24] - x[1]*x[3]*x[8]*x[9]*x[14]*x[15]*x[19]*x[22]*x[26] + x[1]*x[3]*x[8]*x[9]*x[14]*x[15]*x[19]*x[23]*x[25] + x[1]*x[3]*x[8]*x[9]*x[14]*x[16]*x[18]*x[22]*x[26] - x[1]*x[3]*x[8]*x[9]*x[14]*x[16]*x[19]*x[23]*x[24] + x[1]*x[3]*x[8]*x[10]*x[12]*x[17]*x[19]*x[23]*x[24] - x[1]*x[3]*x[8]*x[10]*x[12]*x[17]*x[20]*x[21]*x[25] - x[1]*x[3]*x[8]*x[10]*x[13]*x[17]*x[18]*x[23]*x[24] + x[1]*x[3]*x[8]*x[10]*x[13]*x[17]*x[20]*x[21]*x[24] - x[1]*x[3]*x[8]*x[10]*x[14]*x[15]*x[18]*x[23]*x[25] + x[1]*x[3]*x[8]*x[10]*x[14]*x[15]*x[19]*x[21]*x[26] - x[1]*x[3]*x[8]*x[10]*x[14]*x[16]*x[18]*x[21]*x[26] + x[1]*x[3]*x[8]*x[10]*x[14]*x[16]*x[18]*x[23]*x[24] + x[1]*x[3]*x[8]*x[11]*x[12]*x[15]*x[19]*x[22]*x[26] - x[1]*x[3]*x[8]*x[11]*x[12]*x[15]*x[20]*x[22]*x[25] - x[1]*x[3]*x[8]*x[11]*x[12]*x[16]*x[19]*x[21]*x[26] + x[1]*x[3]*x[8]*x[11]*x[12]*x[16]*x[20]*x[22]*x[24] - x[1]*x[3]*x[8]*x[11]*x[13]*x[15]*x[18]*x[22]*x[26] + x[1]*x[3]*x[8]*x[11]*x[13]*x[15]*x[20]*x[21]*x[25] + x[1]*x[3]*x[8]*x[11]*x[13]*x[16]*x[18]*x[21]*x[26] - x[1]*x[3]*x[8]*x[11]*x[13]*x[16]*x[20]*x[21]*x[24] + x[1]*x[4]*x[6]*x[9]*x[14]*x[17]*x[18]*std::pow(x[22], 2) - x[1]*x[4]*x[6]*x[9]*x[14]*x[17]*x[19]*x[21]*x[22] - x[1]*x[4]*x[6]*x[10]*x[14]*x[17]*x[18]*x[21]*x[22] + x[1]*x[4]*x[6]*x[10]*x[14]*x[17]*x[19]*std::pow(x[21], 2) - x[1]*x[4]*x[6]*x[11]*x[12]*x[17]*x[18]*std::pow(x[22], 2) + x[1]*x[4]*x[6]*x[11]*x[12]*x[17]*x[19]*x[21]*x[22] + x[1]*x[4]*x[6]*x[11]*x[13]*x[17]*x[18]*x[21]*x[22] - x[1]*x[4]*x[6]*x[11]*x[13]*x[17]*x[19]*std::pow(x[21], 2) + x[1]*x[4]*x[8]*x[9]*x[13]*x[17]*x[21]*x[23]*x[25] - x[1]*x[4]*x[8]*x[9]*x[13]*x[17]*x[22]*x[23]*x[24] - x[1]*x[4]*x[8]*x[9]*x[14]*x[15]*std::pow(x[22], 2)*x[26] + x[1]*x[4]*x[8]*x[9]*x[14]*x[15]*x[22]*x[23]*x[25] + x[1]*x[4]*x[8]*x[9]*x[14]*x[16]*x[21]*x[22]*x[26] - x[1]*x[4]*x[8]*x[9]*x[14]*x[16]*x[22]*x[23]*x[24] - x[1]*x[4]*x[8]*x[10]*x[12]*x[17]*x[21]*x[23]*x[25] + x[1]*x[4]*x[8]*x[10]*x[12]*x[17]*x[22]*x[23]*x[24] + x[1]*x[4]*x[8]*x[10]*x[14]*x[15]*x[21]*x[22]*x[26] - x[1]*x[4]*x[8]*x[10]*x[14]*x[15]*x[21]*x[23]*x[25] - x[1]*x[4]*x[8]*x[10]*x[14]*x[16]*std::pow(x[21], 2)*x[26] + x[1]*x[4]*x[8]*x[10]*x[14]*x[16]*x[21]*x[23]*x[24] + x[1]*x[4]*x[8]*x[11]*x[12]*x[15]*std::pow(x[22], 2)*x[26] - x[1]*x[4]*x[8]*x[11]*x[12]*x[15]*x[22]*x[23]*x[25] - x[1]*x[4]*x[8]*x[11]*x[12]*x[16]*x[21]*x[22]*x[26] + x[1]*x[4]*x[8]*x[11]*x[12]*x[16]*x[22]*x[23]*x[24] - x[1]*x[4]*x[8]*x[11]*x[13]*x[15]*x[21]*x[22]*x[26] + x[1]*x[4]*x[8]*x[11]*x[13]*x[15]*x[21]*x[23]*x[25] + x[1]*x[4]*x[8]*x[11]*x[13]*x[16]*std::pow(x[21], 2)*x[26] - x[1]*x[4]*x[8]*x[11]*x[13]*x[16]*x[21]*x[23]*x[24] + x[1]*x[5]*x[6]*x[9]*x[12]*x[17]*x[19]*x[22]*x[26] - x[1]*x[5]*x[6]*x[9]*x[12]*x[17]*x[19]*x[23]*x[25] - x[1]*x[5]*x[6]*x[9]*x[13]*x[17]*x[18]*x[22]*x[26] + x[1]*x[5]*x[6]*x[9]*x[13]*x[17]*x[19]*x[23]*x[24] + x[1]*x[5]*x[6]*x[9]*x[14]*x[15]*x[19]*x[23]*x[25] - x[1]*x[5]*x[6]*x[9]*x[14]*x[15]*x[20]*x[22]*x[25] - x[1]*x[5]*x[6]*x[9]*x[14]*x[16]*x[18]*x[23]*x[25] + x[1]*x[5]*x[6]*x[9]*x[14]*x[16]*x[20]*x[22]*x[24] + x[1]*x[5]*x[6]*x[10]*x[12]*x[17]*x[18]*x[23]*x[25] - x[1]*x[5]*x[6]*x[10]*x[12]*x[17]*x[19]*x[21]*x[26] + x[1]*x[5]*x[6]*x[10]*x[13]*x[17]*x[18]*x[21]*x[26] - x[1]*x[5]*x[6]*x[10]*x[13]*x[17]*x[18]*x[23]*x[24] - x[1]*x[5]*x[6]*x[10]*x[14]*x[15]*x[19]*x[23]*x[24] + x[1]*x[5]*x[6]*x[10]*x[14]*x[15]*x[20]*x[21]*x[25] + x[1]*x[5]*x[6]*x[10]*x[14]*x[16]*x[18]*x[23]*x[24] - x[1]*x[5]*x[6]*x[10]*x[14]*x[16]*x[20]*x[21]*x[24] - x[1]*x[5]*x[6]*x[11]*x[12]*x[15]*x[19]*x[22]*x[26] + x[1]*x[5]*x[6]*x[11]*x[12]*x[15]*x[20]*x[22]*x[25] + x[1]*x[5]*x[6]*x[11]*x[12]*x[16]*x[18]*x[22]*x[26] - x[1]*x[5]*x[6]*x[11]*x[12]*x[16]*x[20]*x[21]*x[25] + x[1]*x[5]*x[6]*x[11]*x[13]*x[15]*x[19]*x[21]*x[26] - x[1]*x[5]*x[6]*x[11]*x[13]*x[15]*x[20]*x[22]*x[24] - x[1]*x[5]*x[6]*x[11]*x[13]*x[16]*x[18]*x[21]*x[26] + x[1]*x[5]*x[6]*x[11]*x[13]*x[16]*x[20]*x[21]*x[24] + x[1]*x[5]*x[7]*x[9]*x[12]*x[17]*std::pow(x[22], 2)*x[26] - x[1]*x[5]*x[7]*x[9]*x[12]*x[17]*x[22]*x[23]*x[25] - x[1]*x[5]*x[7]*x[9]*x[13]*x[17]*x[21]*x[22]*x[26] + x[1]*x[5]*x[7]*x[9]*x[13]*x[17]*x[22]*x[23]*x[24] - x[1]*x[5]*x[7]*x[9]*x[14]*x[16]*x[21]*x[23]*x[25] + x[1]*x[5]*x[7]*x[9]*x[14]*x[16]*x[22]*x[23]*x[24] - x[1]*x[5]*x[7]*x[10]*x[12]*x[17]*x[21]*x[22]*x[26] + x[1]*x[5]*x[7]*x[10]*x[12]*x[17]*x[21]*x[23]*x[25] + x[1]*x[5]*x[7]*x[10]*x[13]*x[17]*std::pow(x[21], 2)*x[26] - x[1]*x[5]*x[7]*x[10]*x[13]*x[17]*x[21]*x[23]*x[24] + x[1]*x[5]*x[7]*x[10]*x[14]*x[15]*x[21]*x[23]*x[25] - x[1]*x[5]*x[7]*x[10]*x[14]*x[15]*x[22]*x[23]*x[24] - x[1]*x[5]*x[7]*x[11]*x[12]*x[15]*std::pow(x[22], 2)*x[26] + x[1]*x[5]*x[7]*x[11]*x[12]*x[15]*x[22]*x[23]*x[25] + x[1]*x[5]*x[7]*x[11]*x[12]*x[16]*x[21]*x[22]*x[26] - x[1]*x[5]*x[7]*x[11]*x[12]*x[16]*x[21]*x[23]*x[25] + x[1]*x[5]*x[7]*x[11]*x[13]*x[15]*x[21]*x[22]*x[26] - x[1]*x[5]*x[7]*x[11]*x[13]*x[15]*x[22]*x[23]*x[24] - x[1]*x[5]*x[7]*x[11]*x[13]*x[16]*std::pow(x[21], 2)*x[26] + x[1]*x[5]*x[7]*x[11]*x[13]*x[16]*x[21]*x[23]*x[24] - x[1]*x[5]*x[8]*x[9]*x[12]*x[16]*x[22]*std::pow(x[26], 2) + x[1]*x[5]*x[8]*x[9]*x[12]*x[16]*x[23]*x[25]*x[26] + x[1]*x[5]*x[8]*x[9]*x[13]*x[15]*x[22]*std::pow(x[26], 2) - x[1]*x[5]*x[8]*x[9]*x[13]*x[15]*x[23]*x[25]*x[26] + x[1]*x[5]*x[8]*x[10]*x[12]*x[16]*x[21]*std::pow(x[26], 2) - x[1]*x[5]*x[8]*x[10]*x[12]*x[16]*x[23]*x[24]*x[26] - x[1]*x[5]*x[8]*x[10]*x[13]*x[15]*x[21]*std::pow(x[26], 2) + x[1]*x[5]*x[8]*x[10]*x[13]*x[15]*x[23]*x[24]*x[26] - x[2]*x[3]*x[6]*x[9]*x[12]*x[17]*std::pow(x[19], 2)*x[26] + x[2]*x[3]*x[6]*x[9]*x[12]*x[17]*x[19]*x[20]*x[25] + x[2]*x[3]*x[6]*x[9]*x[13]*x[17]*x[18]*x[19]*x[26] - x[2]*x[3]*x[6]*x[9]*x[13]*x[17]*x[18]*x[20]*x[25] + x[2]*x[3]*x[6]*x[9]*x[14]*x[15]*std::pow(x[19], 2)*x[26] - x[2]*x[3]*x[6]*x[9]*x[14]*x[15]*x[19]*x[20]*x[25] - x[2]*x[3]*x[6]*x[9]*x[14]*x[16]*x[18]*x[19]*x[26] + x[2]*x[3]*x[6]*x[9]*x[14]*x[16]*x[18]*x[20]*x[25] + x[2]*x[3]*x[6]*x[10]*x[12]*x[17]*x[18]*x[19]*x[26] - x[2]*x[3]*x[6]*x[10]*x[12]*x[17]*x[19]*x[20]*x[24] - x[2]*x[3]*x[6]*x[10]*x[13]*x[17]*std::pow(x[18], 2)*x[26] + x[2]*x[3]*x[6]*x[10]*x[13]*x[17]*x[18]*x[20]*x[24] - x[2]*x[3]*x[6]*x[10]*x[14]*x[15]*x[18]*x[19]*x[26] + x[2]*x[3]*x[6]*x[10]*x[14]*x[15]*x[19]*x[20]*x[24] + x[2]*x[3]*x[6]*x[10]*x[14]*x[16]*std::pow(x[18], 2)*x[26] - x[2]*x[3]*x[6]*x[10]*x[14]*x[16]*x[18]*x[20]*x[24] + x[2]*x[3]*x[6]*x[11]*x[12]*x[16]*x[18]*x[20]*x[25] - x[2]*x[3]*x[6]*x[11]*x[12]*x[16]*x[19]*x[20]*x[24] - x[2]*x[3]*x[6]*x[11]*x[13]*x[15]*x[18]*x[20]*x[25] + x[2]*x[3]*x[6]*x[11]*x[13]*x[15]*x[19]*x[20]*x[24] - x[2]*x[3]*x[7]*x[9]*x[12]*x[17]*x[19]*x[22]*x[26] + x[2]*x[3]*x[7]*x[9]*x[12]*x[17]*x[20]*x[22]*x[25] + x[2]*x[3]*x[7]*x[9]*x[13]*x[17]*x[18]*x[22]*x[26] - x[2]*x[3]*x[7]*x[9]*x[13]*x[17]*x[20]*x[21]*x[25] + x[2]*x[3]*x[7]*x[9]*x[14]*x[15]*x[19]*x[22]*x[26] - x[2]*x[3]*x[7]*x[9]*x[14]*x[15]*x[19]*x[23]*x[25] + x[2]*x[3]*x[7]*x[9]*x[14]*x[16]*x[18]*x[23]*x[25] - x[2]*x[3]*x[7]*x[9]*x[14]*x[16]*x[19]*x[21]*x[26] + x[2]*x[3]*x[7]*x[10]*x[12]*x[17]*x[19]*x[21]*x[26] - x[2]*x[3]*x[7]*x[10]*x[12]*x[17]*x[20]*x[22]*x[24] - x[2]*x[3]*x[7]*x[10]*x[13]*x[17]*x[18]*x[21]*x[26] + x[2]*x[3]*x[7]*x[10]*x[13]*x[17]*x[20]*x[21]*x[24] - x[2]*x[3]*x[7]*x[10]*x[14]*x[15]*x[18]*x[22]*x[26] + x[2]*x[3]*x[7]*x[10]*x[14]*x[15]*x[19]*x[23]*x[24] + x[2]*x[3]*x[7]*x[10]*x[14]*x[16]*x[18]*x[21]*x[26] - x[2]*x[3]*x[7]*x[10]*x[14]*x[16]*x[18]*x[23]*x[24] + x[2]*x[3]*x[7]*x[11]*x[12]*x[15]*x[19]*x[23]*x[25] - x[2]*x[3]*x[7]*x[11]*x[12]*x[15]*x[20]*x[22]*x[25] - x[2]*x[3]*x[7]*x[11]*x[12]*x[16]*x[19]*x[23]*x[24] + x[2]*x[3]*x[7]*x[11]*x[12]*x[16]*x[20]*x[21]*x[25] - x[2]*x[3]*x[7]*x[11]*x[13]*x[15]*x[18]*x[23]*x[25] + x[2]*x[3]*x[7]*x[11]*x[13]*x[15]*x[20]*x[22]*x[24] + x[2]*x[3]*x[7]*x[11]*x[13]*x[16]*x[18]*x[23]*x[24] - x[2]*x[3]*x[7]*x[11]*x[13]*x[16]*x[20]*x[21]*x[24] + x[2]*x[3]*x[8]*x[9]*x[12]*x[16]*x[19]*std::pow(x[26], 2) - x[2]*x[3]*x[8]*x[9]*x[12]*x[16]*x[20]*x[25]*x[26] - x[2]*x[3]*x[8]*x[9]*x[13]*x[16]*x[18]*std::pow(x[26], 2) + x[2]*x[3]*x[8]*x[9]*x[13]*x[16]*x[20]*x[24]*x[26] - x[2]*x[3]*x[8]*x[10]*x[12]*x[15]*x[19]*std::pow(x[26], 2) + x[2]*x[3]*x[8]*x[10]*x[12]*x[15]*x[20]*x[25]*x[26] + x[2]*x[3]*x[8]*x[10]*x[13]*x[15]*x[18]*std::pow(x[26], 2) - x[2]*x[3]*x[8]*x[10]*x[13]*x[15]*x[20]*x[24]*x[26] - x[2]*x[4]*x[6]*x[9]*x[12]*x[17]*x[19]*x[22]*x[26] + x[2]*x[4]*x[6]*x[9]*x[12]*x[17]*x[19]*x[23]*x[25] - x[2]*x[4]*x[6]*x[9]*x[13]*x[17]*x[18]*x[23]*x[25] + x[2]*x[4]*x[6]*x[9]*x[13]*x[17]*x[19]*x[21]*x[26] + x[2]*x[4]*x[6]*x[9]*x[14]*x[15]*x[19]*x[22]*x[26] - x[2]*x[4]*x[6]*x[9]*x[14]*x[15]*x[20]*x[22]*x[25] - x[2]*x[4]*x[6]*x[9]*x[14]*x[16]*x[18]*x[22]*x[26] + x[2]*x[4]*x[6]*x[9]*x[14]*x[16]*x[20]*x[21]*x[25] + x[2]*x[4]*x[6]*x[10]*x[12]*x[17]*x[18]*x[22]*x[26] - x[2]*x[4]*x[6]*x[10]*x[12]*x[17]*x[19]*x[23]*x[24] - x[2]*x[4]*x[6]*x[10]*x[13]*x[17]*x[18]*x[21]*x[26] + x[2]*x[4]*x[6]*x[10]*x[13]*x[17]*x[18]*x[23]*x[24] - x[2]*x[4]*x[6]*x[10]*x[14]*x[15]*x[19]*x[21]*x[26] + x[2]*x[4]*x[6]*x[10]*x[14]*x[15]*x[20]*x[22]*x[24] + x[2]*x[4]*x[6]*x[10]*x[14]*x[16]*x[18]*x[21]*x[26] - x[2]*x[4]*x[6]*x[10]*x[14]*x[16]*x[20]*x[21]*x[24] - x[2]*x[4]*x[6]*x[11]*x[12]*x[15]*x[19]*x[23]*x[25] + x[2]*x[4]*x[6]*x[11]*x[12]*x[15]*x[20]*x[22]*x[25] + x[2]*x[4]*x[6]*x[11]*x[12]*x[16]*x[18]*x[23]*x[25] - x[2]*x[4]*x[6]*x[11]*x[12]*x[16]*x[20]*x[22]*x[24] + x[2]*x[4]*x[6]*x[11]*x[13]*x[15]*x[19]*x[23]*x[24] - x[2]*x[4]*x[6]*x[11]*x[13]*x[15]*x[20]*x[21]*x[25] - x[2]*x[4]*x[6]*x[11]*x[13]*x[16]*x[18]*x[23]*x[24] + x[2]*x[4]*x[6]*x[11]*x[13]*x[16]*x[20]*x[21]*x[24] - x[2]*x[4]*x[7]*x[9]*x[12]*x[17]*std::pow(x[22], 2)*x[26] + x[2]*x[4]*x[7]*x[9]*x[12]*x[17]*x[22]*x[23]*x[25] + x[2]*x[4]*x[7]*x[9]*x[13]*x[17]*x[21]*x[22]*x[26] - x[2]*x[4]*x[7]*x[9]*x[13]*x[17]*x[21]*x[23]*x[25] + x[2]*x[4]*x[7]*x[9]*x[14]*x[15]*std::pow(x[22], 2)*x[26] - x[2]*x[4]*x[7]*x[9]*x[14]*x[15]*x[22]*x[23]*x[25] - x[2]*x[4]*x[7]*x[9]*x[14]*x[16]*x[21]*x[22]*x[26] + x[2]*x[4]*x[7]*x[9]*x[14]*x[16]*x[21]*x[23]*x[25] + x[2]*x[4]*x[7]*x[10]*x[12]*x[17]*x[21]*x[22]*x[26] - x[2]*x[4]*x[7]*x[10]*x[12]*x[17]*x[22]*x[23]*x[24] - x[2]*x[4]*x[7]*x[10]*x[13]*x[17]*std::pow(x[21], 2)*x[26] + x[2]*x[4]*x[7]*x[10]*x[13]*x[17]*x[21]*x[23]*x[24] - x[2]*x[4]*x[7]*x[10]*x[14]*x[15]*x[21]*x[22]*x[26] + x[2]*x[4]*x[7]*x[10]*x[14]*x[15]*x[22]*x[23]*x[24] + x[2]*x[4]*x[7]*x[10]*x[14]*x[16]*std::pow(x[21], 2)*x[26] - x[2]*x[4]*x[7]*x[10]*x[14]*x[16]*x[21]*x[23]*x[24] + x[2]*x[4]*x[7]*x[11]*x[12]*x[16]*x[21]*x[23]*x[25] - x[2]*x[4]*x[7]*x[11]*x[12]*x[16]*x[22]*x[23]*x[24] - x[2]*x[4]*x[7]*x[11]*x[13]*x[15]*x[21]*x[23]*x[25] + x[2]*x[4]*x[7]*x[11]*x[13]*x[15]*x[22]*x[23]*x[24] + x[2]*x[4]*x[8]*x[9]*x[12]*x[16]*x[22]*std::pow(x[26], 2) - x[2]*x[4]*x[8]*x[9]*x[12]*x[16]*x[23]*x[25]*x[26] - x[2]*x[4]*x[8]*x[9]*x[13]*x[16]*x[21]*std::pow(x[26], 2) + x[2]*x[4]*x[8]*x[9]*x[13]*x[16]*x[23]*x[24]*x[26] - x[2]*x[4]*x[8]*x[10]*x[12]*x[15]*x[22]*std::pow(x[26], 2) + x[2]*x[4]*x[8]*x[10]*x[12]*x[15]*x[23]*x[25]*x[26] + x[2]*x[4]*x[8]*x[10]*x[13]*x[15]*x[21]*std::pow(x[26], 2) - x[2]*x[4]*x[8]*x[10]*x[13]*x[15]*x[23]*x[24]*x[26] - x[2]*x[5]*x[6]*x[9]*x[13]*x[15]*x[19]*std::pow(x[26], 2) + x[2]*x[5]*x[6]*x[9]*x[13]*x[15]*x[20]*x[25]*x[26] + x[2]*x[5]*x[6]*x[9]*x[13]*x[16]*x[18]*std::pow(x[26], 2) - x[2]*x[5]*x[6]*x[9]*x[13]*x[16]*x[20]*x[24]*x[26] + x[2]*x[5]*x[6]*x[10]*x[12]*x[15]*x[19]*std::pow(x[26], 2) - x[2]*x[5]*x[6]*x[10]*x[12]*x[15]*x[20]*x[25]*x[26] - x[2]*x[5]*x[6]*x[10]*x[12]*x[16]*x[18]*std::pow(x[26], 2) + x[2]*x[5]*x[6]*x[10]*x[12]*x[16]*x[20]*x[24]*x[26] - x[2]*x[5]*x[7]*x[9]*x[13]*x[15]*x[22]*std::pow(x[26], 2) + x[2]*x[5]*x[7]*x[9]*x[13]*x[15]*x[23]*x[25]*x[26] + x[2]*x[5]*x[7]*x[9]*x[13]*x[16]*x[21]*std::pow(x[26], 2) - x[2]*x[5]*x[7]*x[9]*x[13]*x[16]*x[23]*x[24]*x[26] + x[2]*x[5]*x[7]*x[10]*x[12]*x[15]*x[22]*std::pow(x[26], 2) - x[2]*x[5]*x[7]*x[10]*x[12]*x[15]*x[23]*x[25]*x[26] - x[2]*x[5]*x[7]*x[10]*x[12]*x[16]*x[21]*std::pow(x[26], 2) + x[2]*x[5]*x[7]*x[10]*x[12]*x[16]*x[23]*x[24]*x[26],  // NOLINT [whitespace/line_length]
     x[0]*x[3]*x[7]*x[9]*x[13]*x[17]*x[18]*x[20]*x[22] - x[0]*x[3]*x[7]*x[9]*x[13]*x[17]*x[19]*x[20]*x[21] - x[0]*x[3]*x[7]*x[9]*x[14]*x[15]*std::pow(x[19], 2)*x[23] + x[0]*x[3]*x[7]*x[9]*x[14]*x[15]*x[19]*x[20]*x[22] + x[0]*x[3]*x[7]*x[9]*x[14]*x[16]*x[18]*x[19]*x[23] - x[0]*x[3]*x[7]*x[9]*x[14]*x[16]*x[19]*x[20]*x[21] - x[0]*x[3]*x[7]*x[10]*x[12]*x[17]*x[18]*x[20]*x[22] + x[0]*x[3]*x[7]*x[10]*x[12]*x[17]*x[19]*x[20]*x[21] + x[0]*x[3]*x[7]*x[10]*x[14]*x[15]*x[18]*x[19]*x[23] - x[0]*x[3]*x[7]*x[10]*x[14]*x[15]*x[18]*x[20]*x[22] - x[0]*x[3]*x[7]*x[10]*x[14]*x[16]*std::pow(x[18], 2)*x[23] + x[0]*x[3]*x[7]*x[10]*x[14]*x[16]*x[18]*x[20]*x[21] + x[0]*x[3]*x[7]*x[11]*x[12]*x[15]*std::pow(x[19], 2)*x[23] - x[0]*x[3]*x[7]*x[11]*x[12]*x[15]*x[19]*x[20]*x[22] - x[0]*x[3]*x[7]*x[11]*x[12]*x[16]*x[18]*x[19]*x[23] + x[0]*x[3]*x[7]*x[11]*x[12]*x[16]*x[19]*x[20]*x[21] - x[0]*x[3]*x[7]*x[11]*x[13]*x[15]*x[18]*x[19]*x[23] + x[0]*x[3]*x[7]*x[11]*x[13]*x[15]*x[18]*x[20]*x[22] + x[0]*x[3]*x[7]*x[11]*x[13]*x[16]*std::pow(x[18], 2)*x[23] - x[0]*x[3]*x[7]*x[11]*x[13]*x[16]*x[18]*x[20]*x[21] + x[0]*x[3]*x[8]*x[9]*x[13]*x[15]*x[19]*x[20]*x[26] - x[0]*x[3]*x[8]*x[9]*x[13]*x[15]*std::pow(x[20], 2)*x[25] - x[0]*x[3]*x[8]*x[9]*x[13]*x[16]*x[18]*x[20]*x[26] + x[0]*x[3]*x[8]*x[9]*x[13]*x[16]*std::pow(x[20], 2)*x[24] - x[0]*x[3]*x[8]*x[10]*x[12]*x[15]*x[19]*x[20]*x[26] + x[0]*x[3]*x[8]*x[10]*x[12]*x[15]*std::pow(x[20], 2)*x[25] + x[0]*x[3]*x[8]*x[10]*x[12]*x[16]*x[18]*x[20]*x[26] - x[0]*x[3]*x[8]*x[10]*x[12]*x[16]*std::pow(x[20], 2)*x[24] + x[0]*x[4]*x[6]*x[9]*x[12]*x[17]*std::pow(x[19], 2)*x[23] - x[0]*x[4]*x[6]*x[9]*x[12]*x[17]*x[19]*x[20]*x[22] - x[0]*x[4]*x[6]*x[9]*x[13]*x[17]*x[18]*x[19]*x[23] + x[0]*x[4]*x[6]*x[9]*x[13]*x[17]*x[19]*x[20]*x[21] - x[0]*x[4]*x[6]*x[9]*x[14]*x[16]*x[18]*x[20]*x[22] + x[0]*x[4]*x[6]*x[9]*x[14]*x[16]*x[19]*x[20]*x[21] - x[0]*x[4]*x[6]*x[10]*x[12]*x[17]*x[18]*x[19]*x[23] + x[0]*x[4]*x[6]*x[10]*x[12]*x[17]*x[18]*x[20]*x[22] + x[0]*x[4]*x[6]*x[10]*x[13]*x[17]*std::pow(x[18], 2)*x[23] - x[0]*x[4]*x[6]*x[10]*x[13]*x[17]*x[18]*x[20]*x[21] + x[0]*x[4]*x[6]*x[10]*x[14]*x[15]*x[18]*x[20]*x[22] - x[0]*x[4]*x[6]*x[10]*x[14]*x[15]*x[19]*x[20]*x[21] - x[0]*x[4]*x[6]*x[11]*x[12]*x[15]*std::pow(x[19], 2)*x[23] + x[0]*x[4]*x[6]*x[11]*x[12]*x[15]*x[19]*x[20]*x[22] + x[0]*x[4]*x[6]*x[11]*x[12]*x[16]*x[18]*x[19]*x[23] - x[0]*x[4]*x[6]*x[11]*x[12]*x[16]*x[18]*x[20]*x[22] + x[0]*x[4]*x[6]*x[11]*x[13]*x[15]*x[18]*x[19]*x[23] - x[0]*x[4]*x[6]*x[11]*x[13]*x[15]*x[19]*x[20]*x[21] - x[0]*x[4]*x[6]*x[11]*x[13]*x[16]*std::pow(x[18], 2)*x[23] + x[0]*x[4]*x[6]*x[11]*x[13]*x[16]*x[18]*x[20]*x[21] + x[0]*x[4]*x[7]*x[9]*x[12]*x[17]*x[19]*x[22]*x[23] - x[0]*x[4]*x[7]*x[9]*x[12]*x[17]*x[20]*std::pow(x[22], 2) - x[0]*x[4]*x[7]*x[9]*x[13]*x[17]*x[19]*x[21]*x[23] + x[0]*x[4]*x[7]*x[9]*x[13]*x[17]*x[20]*x[21]*x[22] - x[0]*x[4]*x[7]*x[9]*x[14]*x[15]*x[19]*x[22]*x[23] + x[0]*x[4]*x[7]*x[9]*x[14]*x[15]*x[20]*std::pow(x[22], 2) + x[0]*x[4]*x[7]*x[9]*x[14]*x[16]*x[19]*x[21]*x[23] - x[0]*x[4]*x[7]*x[9]*x[14]*x[16]*x[20]*x[21]*x[22] - x[0]*x[4]*x[7]*x[10]*x[12]*x[17]*x[18]*x[22]*x[23] + x[0]*x[4]*x[7]*x[10]*x[12]*x[17]*x[20]*x[21]*x[22] + x[0]*x[4]*x[7]*x[10]*x[13]*x[17]*x[18]*x[21]*x[23] - x[0]*x[4]*x[7]*x[10]*x[13]*x[17]*x[20]*std::pow(x[21], 2) + x[0]*x[4]*x[7]*x[10]*x[14]*x[15]*x[18]*x[22]*x[23] - x[0]*x[4]*x[7]*x[10]*x[14]*x[15]*x[20]*x[21]*x[22] - x[0]*x[4]*x[7]*x[10]*x[14]*x[16]*x[18]*x[21]*x[23] + x[0]*x[4]*x[7]*x[10]*x[14]*x[16]*x[20]*std::pow(x[21], 2) - x[0]*x[4]*x[7]*x[11]*x[12]*x[16]*x[18]*x[22]*x[23] + x[0]*x[4]*x[7]*x[11]*x[12]*x[16]*x[19]*x[21]*x[23] + x[0]*x[4]*x[7]*x[11]*x[13]*x[15]*x[18]*x[22]*x[23] - x[0]*x[4]*x[7]*x[11]*x[13]*x[15]*x[19]*x[21]*x[23] - x[0]*x[4]*x[8]*x[9]*x[12]*x[16]*x[19]*x[23]*x[26] + x[0]*x[4]*x[8]*x[9]*x[12]*x[16]*x[20]*x[22]*x[26] + x[0]*x[4]*x[8]*x[9]*x[13]*x[15]*x[19]*x[23]*x[26] - x[0]*x[4]*x[8]*x[9]*x[13]*x[15]*x[20]*x[23]*x[25] - x[0]*x[4]*x[8]*x[9]*x[13]*x[16]*x[20]*x[21]*x[26] + x[0]*x[4]*x[8]*x[9]*x[13]*x[16]*x[20]*x[23]*x[24] - x[0]*x[4]*x[8]*x[10]*x[12]*x[15]*x[20]*x[22]*x[26] + x[0]*x[4]*x[8]*x[10]*x[12]*x[15]*x[20]*x[23]*x[25] + x[0]*x[4]*x[8]*x[10]*x[12]*x[16]*x[18]*x[23]*x[26] - x[0]*x[4]*x[8]*x[10]*x[12]*x[16]*x[20]*x[23]*x[24] - x[0]*x[4]*x[8]*x[10]*x[13]*x[15]*x[18]*x[23]*x[26] + x[0]*x[4]*x[8]*x[10]*x[13]*x[15]*x[20]*x[21]*x[26] - x[0]*x[5]*x[6]*x[9]*x[12]*x[16]*x[19]*x[20]*x[26] + x[0]*x[5]*x[6]*x[9]*x[12]*x[16]*std::pow(x[20], 2)*x[25] + x[0]*x[5]*x[6]*x[9]*x[13]*x[16]*x[18]*x[20]*x[26] - x[0]*x[5]*x[6]*x[9]*x[13]*x[16]*std::pow(x[20], 2)*x[24] + x[0]*x[5]*x[6]*x[10]*x[12]*x[15]*x[19]*x[20]*x[26] - x[0]*x[5]*x[6]*x[10]*x[12]*x[15]*std::pow(x[20], 2)*x[25] - x[0]*x[5]*x[6]*x[10]*x[13]*x[15]*x[18]*x[20]*x[26] + x[0]*x[5]*x[6]*x[10]*x[13]*x[15]*std::pow(x[20], 2)*x[24] - x[0]*x[5]*x[7]*x[9]*x[12]*x[16]*x[19]*x[23]*x[26] + x[0]*x[5]*x[7]*x[9]*x[12]*x[16]*x[20]*x[23]*x[25] + x[0]*x[5]*x[7]*x[9]*x[13]*x[15]*x[19]*x[23]*x[26] - x[0]*x[5]*x[7]*x[9]*x[13]*x[15]*x[20]*x[22]*x[26] + x[0]*x[5]*x[7]*x[9]*x[13]*x[16]*x[20]*x[21]*x[26] - x[0]*x[5]*x[7]*x[9]*x[13]*x[16]*x[20]*x[23]*x[24] + x[0]*x[5]*x[7]*x[10]*x[12]*x[15]*x[20]*x[22]*x[26] - x[0]*x[5]*x[7]*x[10]*x[12]*x[15]*x[20]*x[23]*x[25] + x[0]*x[5]*x[7]*x[10]*x[12]*x[16]*x[18]*x[23]*x[26] - x[0]*x[5]*x[7]*x[10]*x[12]*x[16]*x[20]*x[21]*x[26] - x[0]*x[5]*x[7]*x[10]*x[13]*x[15]*x[18]*x[23]*x[26] + x[0]*x[5]*x[7]*x[10]*x[13]*x[15]*x[20]*x[23]*x[24] - x[1]*x[3]*x[6]*x[9]*x[12]*x[17]*std::pow(x[19], 2)*x[23] + x[1]*x[3]*x[6]*x[9]*x[12]*x[17]*x[19]*x[20]*x[22] + x[1]*x[3]*x[6]*x[9]*x[13]*x[17]*x[18]*x[19]*x[23] - x[1]*x[3]*x[6]*x[9]*x[13]*x[17]*x[18]*x[20]*x[22] + x[1]*x[3]*x[6]*x[9]*x[14]*x[15]*std::pow(x[19], 2)*x[23] - x[1]*x[3]*x[6]*x[9]*x[14]*x[15]*x[19]*x[20]*x[22] - x[1]*x[3]*x[6]*x[9]*x[14]*x[16]*x[18]*x[19]*x[23] + x[1]*x[3]*x[6]*x[9]*x[14]*x[16]*x[18]*x[20]*x[22] + x[1]*x[3]*x[6]*x[10]*x[12]*x[17]*x[18]*x[19]*x[23] - x[1]*x[3]*x[6]*x[10]*x[12]*x[17]*x[19]*x[20]*x[21] - x[1]*x[3]*x[6]*x[10]*x[13]*x[17]*std::pow(x[18], 2)*x[23] + x[1]*x[3]*x[6]*x[10]*x[13]*x[17]*x[18]*x[20]*x[21] - x[1]*x[3]*x[6]*x[10]*x[14]*x[15]*x[18]*x[19]*x[23] + x[1]*x[3]*x[6]*x[10]*x[14]*x[15]*x[19]*x[20]*x[21] + x[1]*x[3]*x[6]*x[10]*x[14]*x[16]*std::pow(x[18], 2)*x[23] - x[1]*x[3]*x[6]*x[10]*x[14]*x[16]*x[18]*x[20]*x[21] + x[1]*x[3]*x[6]*x[11]*x[12]*x[16]*x[18]*x[20]*x[22] - x[1]*x[3]*x[6]*x[11]*x[12]*x[16]*x[19]*x[20]*x[21] - x[1]*x[3]*x[6]*x[11]*x[13]*x[15]*x[18]*x[20]*x[22] + x[1]*x[3]*x[6]*x[11]*x[13]*x[15]*x[19]*x[20]*x[21] - x[1]*x[3]*x[7]*x[9]*x[12]*x[17]*x[19]*x[22]*x[23] + x[1]*x[3]*x[7]*x[9]*x[12]*x[17]*x[20]*std::pow(x[22], 2) + x[1]*x[3]*x[7]*x[9]*x[13]*x[17]*x[18]*x[22]*x[23] - x[1]*x[3]*x[7]*x[9]*x[13]*x[17]*x[20]*x[21]*x[22] + x[1]*x[3]*x[7]*x[9]*x[14]*x[16]*x[18]*x[22]*x[23] - x[1]*x[3]*x[7]*x[9]*x[14]*x[16]*x[19]*x[21]*x[23] + x[1]*x[3]*x[7]*x[10]*x[12]*x[17]*x[19]*x[21]*x[23] - x[1]*x[3]*x[7]*x[10]*x[12]*x[17]*x[20]*x[21]*x[22] - x[1]*x[3]*x[7]*x[10]*x[13]*x[17]*x[18]*x[21]*x[23] + x[1]*x[3]*x[7]*x[10]*x[13]*x[17]*x[20]*std::pow(x[21], 2) - x[1]*x[3]*x[7]*x[10]*x[14]*x[15]*x[18]*x[22]*x[23] + x[1]*x[3]*x[7]*x[10]*x[14]*x[15]*x[19]*x[21]*x[23] + x[1]*x[3]*x[7]*x[11]*x[12]*x[15]*x[19]*x[22]*x[23] - x[1]*x[3]*x[7]*x[11]*x[12]*x[15]*x[20]*std::pow(x[22], 2) - x[1]*x[3]*x[7]*x[11]*x[12]*x[16]*x[19]*x[21]*x[23] + x[1]*x[3]*x[7]*x[11]*x[12]*x[16]*x[20]*x[21]*x[22] - x[1]*x[3]*x[7]*x[11]*x[13]*x[15]*x[18]*x[22]*x[23] + x[1]*x[3]*x[7]*x[11]*x[13]*x[15]*x[20]*x[21]*x[22] + x[1]*x[3]*x[7]*x[11]*x[13]*x[16]*x[18]*x[21]*x[23] - x[1]*x[3]*x[7]*x[11]*x[13]*x[16]*x[20]*std::pow(x[21], 2) + x[1]*x[3]*x[8]*x[9]*x[12]*x[16]*x[19]*x[23]*x[26] - x[1]*x[3]*x[8]*x[9]*x[12]*x[16]*x[20]*x[22]*x[26] + x[1]*x[3]*x[8]*x[9]*x[13]*x[15]*x[20]*x[22]*x[26] - x[1]*x[3]*x[8]*x[9]*x[13]*x[15]*x[20]*x[23]*x[25] - x[1]*x[3]*x[8]*x[9]*x[13]*x[16]*x[18]*x[23]*x[26] + x[1]*x[3]*x[8]*x[9]*x[13]*x[16]*x[20]*x[23]*x[24] - x[1]*x[3]*x[8]*x[10]*x[12]*x[15]*x[19]*x[23]*x[26] + x[1]*x[3]*x[8]*x[10]*x[12]*x[15]*x[20]*x[23]*x[25] + x[1]*x[3]*x[8]*x[10]*x[12]*x[16]*x[20]*x[21]*x[26] - x[1]*x[3]*x[8]*x[10]*x[12]*x[16]*x[20]*x[23]*x[24] + x[1]*x[3]*x[8]*x[10]*x[13]*x[15]*x[18]*x[23]*x[26] - x[1]*x[3]*x[8]*x[10]*x[13]*x[15]*x[20]*x[21]*x[26] - x[1]*x[4]*x[6]*x[9]*x[13]*x[17]*x[18]*x[22]*x[23] + x[1]*x[4]*x[6]*x[9]*x[13]*x[17]*x[19]*x[21]*x[23] + x[1]*x[4]*x[6]*x[9]*x[14]*x[15]*x[19]*x[22]*x[23] - x[1]*x[4]*x[6]*x[9]*x[14]*x[15]*x[20]*std::pow(x[22], 2) - x[1]*x[4]*x[6]*x[9]*x[14]*x[16]*x[18]*x[22]*x[23] + x[1]*x[4]*x[6]*x[9]*x[14]*x[16]*x[20]*x[21]*x[22] + x[1]*x[4]*x[6]*x[10]*x[12]*x[17]*x[18]*x[22]*x[23] - x[1]*x[4]*x[6]*x[10]*x[12]*x[17]*x[19]*x[21]*x[23] - x[1]*x[4]*x[6]*x[10]*x[14]*x[15]*x[19]*x[21]*x[23] + x[1]*x[4]*x[6]*x[10]*x[14]*x[15]*x[20]*x[21]*x[22] + x[1]*x[4]*x[6]*x[10]*x[14]*x[16]*x[18]*x[21]*x[23] - x[1]*x[4]*x[6]*x[10]*x[14]*x[16]*x[20]*std::pow(x[21], 2) - x[1]*x[4]*x[6]*x[11]*x[12]*x[15]*x[19]*x[22]*x[23] + x[1]*x[4]*x[6]*x[11]*x[12]*x[15]*x[20]*std::pow(x[22], 2) + x[1]*x[4]*x[6]*x[11]*x[12]*x[16]*x[18]*x[22]*x[23] - x[1]*x[4]*x[6]*x[11]*x[12]*x[16]*x[20]*x[21]*x[22] + x[1]*x[4]*x[6]*x[11]*x[13]*x[15]*x[19]*x[21]*x[23] - x[1]*x[4]*x[6]*x[11]*x[13]*x[15]*x[20]*x[21]*x[22] - x[1]*x[4]*x[6]*x[11]*x[13]*x[16]*x[18]*x[21]*x[23] + x[1]*x[4]*x[6]*x[11]*x[13]*x[16]*x[20]*std::pow(x[21], 2) + x[1]*x[4]*x[8]*x[9]*x[13]*x[15]*x[22]*x[23]*x[26] - x[1]*x[4]*x[8]*x[9]*x[13]*x[15]*std::pow(x[23], 2)*x[25] - x[1]*x[4]*x[8]*x[9]*x[13]*x[16]*x[21]*x[23]*x[26] + x[1]*x[4]*x[8]*x[9]*x[13]*x[16]*std::pow(x[23], 2)*x[24] - x[1]*x[4]*x[8]*x[10]*x[12]*x[15]*x[22]*x[23]*x[26] + x[1]*x[4]*x[8]*x[10]*x[12]*x[15]*std::pow(x[23], 2)*x[25] + x[1]*x[4]*x[8]*x[10]*x[12]*x[16]*x[21]*x[23]*x[26] - x[1]*x[4]*x[8]*x[10]*x[12]*x[16]*std::pow(x[23], 2)*x[24] - x[1]*x[5]*x[6]*x[9]*x[12]*x[16]*x[20]*x[22]*x[26] + x[1]*x[5]*x[6]*x[9]*x[12]*x[16]*x[20]*x[23]*x[25] - x[1]*x[5]*x[6]*x[9]*x[13]*x[15]*x[19]*x[23]*x[26] + x[1]*x[5]*x[6]*x[9]*x[13]*x[15]*x[20]*x[22]*x[26] + x[1]*x[5]*x[6]*x[9]*x[13]*x[16]*x[18]*x[23]*x[26] - x[1]*x[5]*x[6]*x[9]*x[13]*x[16]*x[20]*x[23]*x[24] + x[1]*x[5]*x[6]*x[10]*x[12]*x[15]*x[19]*x[23]*x[26] - x[1]*x[5]*x[6]*x[10]*x[12]*x[15]*x[20]*x[23]*x[25] - x[1]*x[5]*x[6]*x[10]*x[12]*x[16]*x[18]*x[23]*x[26] + x[1]*x[5]*x[6]*x[10]*x[12]*x[16]*x[20]*x[21]*x[26] - x[1]*x[5]*x[6]*x[10]*x[13]*x[15]*x[20]*x[21]*x[26] + x[1]*x[5]*x[6]*x[10]*x[13]*x[15]*x[20]*x[23]*x[24] - x[1]*x[5]*x[7]*x[9]*x[12]*x[16]*x[22]*x[23]*x[26] + x[1]*x[5]*x[7]*x[9]*x[12]*x[16]*std::pow(x[23], 2)*x[25] + x[1]*x[5]*x[7]*x[9]*x[13]*x[16]*x[21]*x[23]*x[26] - x[1]*x[5]*x[7]*x[9]*x[13]*x[16]*std::pow(x[23], 2)*x[24] + x[1]*x[5]*x[7]*x[10]*x[12]*x[15]*x[22]*x[23]*x[26] - x[1]*x[5]*x[7]*x[10]*x[12]*x[15]*std::pow(x[23], 2)*x[25] - x[1]*x[5]*x[7]*x[10]*x[13]*x[15]*x[21]*x[23]*x[26] + x[1]*x[5]*x[7]*x[10]*x[13]*x[15]*std::pow(x[23], 2)*x[24] + x[2]*x[3]*x[6]*x[9]*x[12]*x[16]*x[19]*x[20]*x[26] - x[2]*x[3]*x[6]*x[9]*x[12]*x[16]*std::pow(x[20], 2)*x[25] - x[2]*x[3]*x[6]*x[9]*x[13]*x[15]*x[19]*x[20]*x[26] + x[2]*x[3]*x[6]*x[9]*x[13]*x[15]*std::pow(x[20], 2)*x[25] - x[2]*x[3]*x[6]*x[10]*x[12]*x[16]*x[18]*x[20]*x[26] + x[2]*x[3]*x[6]*x[10]*x[12]*x[16]*std::pow(x[20], 2)*x[24] + x[2]*x[3]*x[6]*x[10]*x[13]*x[15]*x[18]*x[20]*x[26] - x[2]*x[3]*x[6]*x[10]*x[13]*x[15]*std::pow(x[20], 2)*x[24] + x[2]*x[3]*x[7]*x[9]*x[12]*x[16]*x[19]*x[23]*x[26] - x[2]*x[3]*x[7]*x[9]*x[12]*x[16]*x[20]*x[23]*x[25] - x[2]*x[3]*x[7]*x[9]*x[13]*x[15]*x[20]*x[22]*x[26] + x[2]*x[3]*x[7]*x[9]*x[13]*x[15]*x[20]*x[23]*x[25] - x[2]*x[3]*x[7]*x[9]*x[13]*x[16]*x[18]*x[23]*x[26] + x[2]*x[3]*x[7]*x[9]*x[13]*x[16]*x[20]*x[21]*x[26] - x[2]*x[3]*x[7]*x[10]*x[12]*x[15]*x[19]*x[23]*x[26] + x[2]*x[3]*x[7]*x[10]*x[12]*x[15]*x[20]*x[22]*x[26] - x[2]*x[3]*x[7]*x[10]*x[12]*x[16]*x[20]*x[21]*x[26] + x[2]*x[3]*x[7]*x[10]*x[12]*x[16]*x[20]*x[23]*x[24] + x[2]*x[3]*x[7]*x[10]*x[13]*x[15]*x[18]*x[23]*x[26] - x[2]*x[3]*x[7]*x[10]*x[13]*x[15]*x[20]*x[23]*x[24] + x[2]*x[4]*x[6]*x[9]*x[12]*x[16]*x[20]*x[22]*x[26] - x[2]*x[4]*x[6]*x[9]*x[12]*x[16]*x[20]*x[23]*x[25] - x[2]*x[4]*x[6]*x[9]*x[13]*x[15]*x[19]*x[23]*x[26] + x[2]*x[4]*x[6]*x[9]*x[13]*x[15]*x[20]*x[23]*x[25] + x[2]*x[4]*x[6]*x[9]*x[13]*x[16]*x[18]*x[23]*x[26] - x[2]*x[4]*x[6]*x[9]*x[13]*x[16]*x[20]*x[21]*x[26] + x[2]*x[4]*x[6]*x[10]*x[12]*x[15]*x[19]*x[23]*x[26] - x[2]*x[4]*x[6]*x[10]*x[12]*x[15]*x[20]*x[22]*x[26] - x[2]*x[4]*x[6]*x[10]*x[12]*x[16]*x[18]*x[23]*x[26] + x[2]*x[4]*x[6]*x[10]*x[12]*x[16]*x[20]*x[23]*x[24] + x[2]*x[4]*x[6]*x[10]*x[13]*x[15]*x[20]*x[21]*x[26] - x[2]*x[4]*x[6]*x[10]*x[13]*x[15]*x[20]*x[23]*x[24] + x[2]*x[4]*x[7]*x[9]*x[12]*x[16]*x[22]*x[23]*x[26] - x[2]*x[4]*x[7]*x[9]*x[12]*x[16]*std::pow(x[23], 2)*x[25] - x[2]*x[4]*x[7]*x[9]*x[13]*x[15]*x[22]*x[23]*x[26] + x[2]*x[4]*x[7]*x[9]*x[13]*x[15]*std::pow(x[23], 2)*x[25] - x[2]*x[4]*x[7]*x[10]*x[12]*x[16]*x[21]*x[23]*x[26] + x[2]*x[4]*x[7]*x[10]*x[12]*x[16]*std::pow(x[23], 2)*x[24] + x[2]*x[4]*x[7]*x[10]*x[13]*x[15]*x[21]*x[23]*x[26] - x[2]*x[4]*x[7]*x[10]*x[13]*x[15]*std::pow(x[23], 2)*x[24],  // NOLINT [whitespace/line_length]
     x[0]*x[3]*x[7]*x[9]*x[13]*x[15]*x[19]*x[20]*x[23] - x[0]*x[3]*x[7]*x[9]*x[13]*x[15]*std::pow(x[20], 2)*x[22] - x[0]*x[3]*x[7]*x[9]*x[13]*x[16]*x[18]*x[20]*x[23] + x[0]*x[3]*x[7]*x[9]*x[13]*x[16]*std::pow(x[20], 2)*x[21] - x[0]*x[3]*x[7]*x[10]*x[12]*x[15]*x[19]*x[20]*x[23] + x[0]*x[3]*x[7]*x[10]*x[12]*x[15]*std::pow(x[20], 2)*x[22] + x[0]*x[3]*x[7]*x[10]*x[12]*x[16]*x[18]*x[20]*x[23] - x[0]*x[3]*x[7]*x[10]*x[12]*x[16]*std::pow(x[20], 2)*x[21] - x[0]*x[4]*x[6]*x[9]*x[12]*x[16]*x[19]*x[20]*x[23] + x[0]*x[4]*x[6]*x[9]*x[12]*x[16]*std::pow(x[20], 2)*x[22] + x[0]*x[4]*x[6]*x[9]*x[13]*x[16]*x[18]*x[20]*x[23] - x[0]*x[4]*x[6]*x[9]*x[13]*x[16]*std::pow(x[20], 2)*x[21] + x[0]*x[4]*x[6]*x[10]*x[12]*x[15]*x[19]*x[20]*x[23] - x[0]*x[4]*x[6]*x[10]*x[12]*x[15]*std::pow(x[20], 2)*x[22] - x[0]*x[4]*x[6]*x[10]*x[13]*x[15]*x[18]*x[20]*x[23] + x[0]*x[4]*x[6]*x[10]*x[13]*x[15]*std::pow(x[20], 2)*x[21] - x[0]*x[4]*x[7]*x[9]*x[12]*x[16]*x[19]*std::pow(x[23], 2) + x[0]*x[4]*x[7]*x[9]*x[12]*x[16]*x[20]*x[22]*x[23] + x[0]*x[4]*x[7]*x[9]*x[13]*x[15]*x[19]*std::pow(x[23], 2) - x[0]*x[4]*x[7]*x[9]*x[13]*x[15]*x[20]*x[22]*x[23] + x[0]*x[4]*x[7]*x[10]*x[12]*x[16]*x[18]*std::pow(x[23], 2) - x[0]*x[4]*x[7]*x[10]*x[12]*x[16]*x[20]*x[21]*x[23] - x[0]*x[4]*x[7]*x[10]*x[13]*x[15]*x[18]*std::pow(x[23], 2) + x[0]*x[4]*x[7]*x[10]*x[13]*x[15]*x[20]*x[21]*x[23] + x[1]*x[3]*x[6]*x[9]*x[12]*x[16]*x[19]*x[20]*x[23] - x[1]*x[3]*x[6]*x[9]*x[12]*x[16]*std::pow(x[20], 2)*x[22] - x[1]*x[3]*x[6]*x[9]*x[13]*x[15]*x[19]*x[20]*x[23] + x[1]*x[3]*x[6]*x[9]*x[13]*x[15]*std::pow(x[20], 2)*x[22] - x[1]*x[3]*x[6]*x[10]*x[12]*x[16]*x[18]*x[20]*x[23] + x[1]*x[3]*x[6]*x[10]*x[12]*x[16]*std::pow(x[20], 2)*x[21] + x[1]*x[3]*x[6]*x[10]*x[13]*x[15]*x[18]*x[20]*x[23] - x[1]*x[3]*x[6]*x[10]*x[13]*x[15]*std::pow(x[20], 2)*x[21] + x[1]*x[3]*x[7]*x[9]*x[12]*x[16]*x[19]*std::pow(x[23], 2) - x[1]*x[3]*x[7]*x[9]*x[12]*x[16]*x[20]*x[22]*x[23] - x[1]*x[3]*x[7]*x[9]*x[13]*x[16]*x[18]*std::pow(x[23], 2) + x[1]*x[3]*x[7]*x[9]*x[13]*x[16]*x[20]*x[21]*x[23] - x[1]*x[3]*x[7]*x[10]*x[12]*x[15]*x[19]*std::pow(x[23], 2) + x[1]*x[3]*x[7]*x[10]*x[12]*x[15]*x[20]*x[22]*x[23] + x[1]*x[3]*x[7]*x[10]*x[13]*x[15]*x[18]*std::pow(x[23], 2) - x[1]*x[3]*x[7]*x[10]*x[13]*x[15]*x[20]*x[21]*x[23] - x[1]*x[4]*x[6]*x[9]*x[13]*x[15]*x[19]*std::pow(x[23], 2) + x[1]*x[4]*x[6]*x[9]*x[13]*x[15]*x[20]*x[22]*x[23] + x[1]*x[4]*x[6]*x[9]*x[13]*x[16]*x[18]*std::pow(x[23], 2) - x[1]*x[4]*x[6]*x[9]*x[13]*x[16]*x[20]*x[21]*x[23] + x[1]*x[4]*x[6]*x[10]*x[12]*x[15]*x[19]*std::pow(x[23], 2) - x[1]*x[4]*x[6]*x[10]*x[12]*x[15]*x[20]*x[22]*x[23] - x[1]*x[4]*x[6]*x[10]*x[12]*x[16]*x[18]*std::pow(x[23], 2) + x[1]*x[4]*x[6]*x[10]*x[12]*x[16]*x[20]*x[21]*x[23];  // NOLINT [whitespace/line_length]

        return coeffs;
    }

    inline Eigen::Matrix3d create_M(const double w, const Eigen::Matrix<double, 27, 1> &x) {
        Eigen::Matrix3d M;
        M << std::pow(w, 2)*x[2]*x[11]*x[25] + w*x[0]*x[11]*x[19] + w*x[1]*x[11]*x[22] - w*x[2]*x[10]*x[26] - x[0]*x[10]*x[20] - x[1]*x[10]*x[23],  // NOLINT [whitespace/line_length]
            -std::pow(w, 2)*x[2]*x[11]*x[24] - w*x[0]*x[11]*x[18] - w*x[1]*x[11]*x[21] + w*x[2]*x[9]*x[26] + x[0]*x[9]*x[20] + x[1]*x[9]*x[23],  // NOLINT [whitespace/line_length]
            -w*x[2]*x[9]*x[25] + w*x[2]*x[10]*x[24] - x[0]*x[9]*x[19] + x[0]*x[10]*x[18] - x[1]*x[9]*x[22] + x[1]*x[10]*x[21],  // NOLINT [whitespace/line_length]
             std::pow(w, 2)*x[5]*x[14]*x[25] + w*x[3]*x[14]*x[19] + w*x[4]*x[14]*x[22] - w*x[5]*x[13]*x[26] - x[3]*x[13]*x[20] - x[4]*x[13]*x[23],  // NOLINT [whitespace/line_length]
            -std::pow(w, 2)*x[5]*x[14]*x[24] - w*x[3]*x[14]*x[18] - w*x[4]*x[14]*x[21] + w*x[5]*x[12]*x[26] + x[3]*x[12]*x[20] + x[4]*x[12]*x[23],  // NOLINT [whitespace/line_length]
            -w*x[5]*x[12]*x[25] + w*x[5]*x[13]*x[24] - x[3]*x[12]*x[19] + x[3]*x[13]*x[18] - x[4]*x[12]*x[22] + x[4]*x[13]*x[21],  // NOLINT [whitespace/line_length]
             std::pow(w, 2)*x[8]*x[17]*x[25] + w*x[6]*x[17]*x[19] + w*x[7]*x[17]*x[22] - w*x[8]*x[16]*x[26] - x[6]*x[16]*x[20] - x[7]*x[16]*x[23],  // NOLINT [whitespace/line_length]
            -std::pow(w, 2)*x[8]*x[17]*x[24] - w*x[6]*x[17]*x[18] - w*x[7]*x[17]*x[21] + w*x[8]*x[15]*x[26] + x[6]*x[15]*x[20] + x[7]*x[15]*x[23],  // NOLINT [whitespace/line_length]
            -w*x[8]*x[15]*x[25] + w*x[8]*x[16]*x[24] - x[6]*x[15]*x[19] + x[6]*x[16]*x[18] - x[7]*x[15]*x[22] + x[7]*x[16]*x[21];  // NOLINT [whitespace/line_length]
        return M;
    }
}  // namespace ValtonenOrnhagArxiv2021
}  // namespace DronePoseLib
