"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KubectlProvider = void 0;
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const lambda_layer_awscli_1 = require("@aws-cdk/lambda-layer-awscli");
const lambda_layer_kubectl_1 = require("@aws-cdk/lambda-layer-kubectl");
const cluster_1 = require("./cluster");
class KubectlProvider extends core_1.NestedStack {
    constructor(scope, id, props) {
        super(scope, id);
        const cluster = props.cluster;
        if (!cluster.kubectlRole) {
            throw new Error('"kubectlRole" is not defined, cannot issue kubectl commands against this cluster');
        }
        if (cluster.kubectlPrivateSubnets && !cluster.kubectlSecurityGroup) {
            throw new Error('"kubectlSecurityGroup" is required if "kubectlSubnets" is specified');
        }
        const memorySize = cluster.kubectlMemory ? cluster.kubectlMemory.toMebibytes() : 1024;
        const handler = new lambda.Function(this, 'Handler', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'kubectl-handler')),
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'index.handler',
            timeout: core_1.Duration.minutes(15),
            description: 'onEvent handler for EKS kubectl resource provider',
            memorySize,
            environment: cluster.kubectlEnvironment,
            // defined only when using private access
            vpc: cluster.kubectlPrivateSubnets ? cluster.vpc : undefined,
            securityGroups: cluster.kubectlSecurityGroup ? [cluster.kubectlSecurityGroup] : undefined,
            vpcSubnets: cluster.kubectlPrivateSubnets ? { subnets: cluster.kubectlPrivateSubnets } : undefined,
        });
        // allow user to customize the layer
        if (!props.cluster.kubectlLayer) {
            handler.addLayers(new lambda_layer_awscli_1.AwsCliLayer(this, 'AwsCliLayer'));
            handler.addLayers(new lambda_layer_kubectl_1.KubectlLayer(this, 'KubectlLayer'));
        }
        else {
            handler.addLayers(props.cluster.kubectlLayer);
        }
        this.handlerRole = handler.role;
        this.handlerRole.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['eks:DescribeCluster'],
            resources: [cluster.clusterArn],
        }));
        // allow this handler to assume the kubectl role
        cluster.kubectlRole.grant(this.handlerRole, 'sts:AssumeRole');
        const provider = new cr.Provider(this, 'Provider', {
            onEventHandler: handler,
            vpc: cluster.kubectlPrivateSubnets ? cluster.vpc : undefined,
            vpcSubnets: cluster.kubectlPrivateSubnets ? { subnets: cluster.kubectlPrivateSubnets } : undefined,
            securityGroups: cluster.kubectlSecurityGroup ? [cluster.kubectlSecurityGroup] : undefined,
        });
        this.serviceToken = provider.serviceToken;
        this.roleArn = cluster.kubectlRole.roleArn;
    }
    static getOrCreate(scope, cluster) {
        // if this is an "owned" cluster, it has a provider associated with it
        if (cluster instanceof cluster_1.Cluster) {
            return cluster._attachKubectlResourceScope(scope);
        }
        // if this is an imported cluster, we need to provision a custom resource provider in this stack
        // we will define one per stack for each cluster based on the cluster uniqueid
        const uid = `${core_1.Names.nodeUniqueId(cluster.node)}-KubectlProvider`;
        const stack = core_1.Stack.of(scope);
        let provider = stack.node.tryFindChild(uid);
        if (!provider) {
            provider = new KubectlProvider(stack, uid, { cluster });
        }
        return provider;
    }
}
exports.KubectlProvider = KubectlProvider;
//# sourceMappingURL=data:application/json;base64,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